# Copyright 2018-2025 Jérôme Dumonteil
# Copyright (c) 2009-2010 Ars Aperta, Itaapy, Pierlis, Talend.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# Authors (odfdo project): jerome.dumonteil@gmail.com
# The odfdo project is a derivative work of the lpod-python project:
# https://github.com/lpod/lpod-python
# Authors: Hervé Cauwelier <herve@itaapy.com>
#          Romain Gauthier <romain@itaapy.com>
#          Jerome Dumonteil <jerome.dumonteil@itaapy.com>
"""Annotation class for "office:annotation" tag."""

from __future__ import annotations

from datetime import datetime
from typing import TYPE_CHECKING, Any

from .element import Element, PropDef, register_element_class
from .elements_between import elements_between
from .mixin_dc_creator import DcCreatorMixin
from .mixin_dc_date import DcDateMixin
from .mixin_md import MDTail

if TYPE_CHECKING:
    from .body import Body


def get_unique_office_name(element: Element | None = None) -> str:
    """Provide an autogenerated unique "office:name" for the document."""
    if element is not None:
        body = element.document_body
    else:
        body = None
    if body:
        used = set(body.get_office_names())
    else:
        used = set()
    # unplugged current paragraph:
    if element is not None:
        used.update(element.get_office_names())
    indice = 1
    while True:
        name = f"__Fieldmark__lpod_{indice}"
        if name in used:
            indice += 1
            continue
        break
    return name


class Annotation(MDTail, Element, DcCreatorMixin, DcDateMixin):
    """An annotation (private note), "text:annotation"."""

    _tag = "office:annotation"
    _properties = (
        PropDef("name", "office:name"),
        PropDef("note_id", "text:id"),
    )

    def __init__(
        self,
        text_or_element: Element | str | None = None,
        creator: str | None = None,
        date: datetime | None = None,
        name: str | None = None,
        parent: Element | None = None,
        **kwargs: Any,
    ) -> None:
        """An annotation (private note), "text:annotation".

        Annotation element credited to the given creator with the
        given text, optionally dated (current date by default).
        If name not provided and some parent is provided, the name is
        autogenerated.

        Args:

            text -- str or odf_element

            creator -- str

            date -- datetime

            name -- str

            parent -- Element
        """
        # fixme : use offset
        # TODO allow paragraph and text styles
        super().__init__(**kwargs)

        if self._do_init:
            self.note_body = text_or_element
            if creator:
                self.creator = creator
            if date is None:
                date = datetime.now()
            self.date = date
            if not name:
                name = get_unique_office_name(parent)
            self.name = name

    @property
    def dc_creator(self) -> str | None:
        """Alias for self.creator property."""
        return self.creator

    @dc_creator.setter
    def dc_creator(self, creator: str) -> None:
        self.creator = creator

    @property
    def dc_date(self) -> datetime | None:
        """Alias for self.date property."""
        return self.date

    @dc_date.setter
    def dc_date(self, dtdate: datetime) -> None:
        self.date = dtdate

    @property
    def note_body(self) -> str:
        return self.text_content

    @note_body.setter
    def note_body(self, text_or_element: Element | str | None) -> None:
        if text_or_element is None:
            self.text_content = ""
        elif isinstance(text_or_element, str):
            self.text_content = text_or_element
        elif isinstance(text_or_element, Element):
            self.clear()
            self.append(text_or_element)
        else:
            raise TypeError(f'Unexpected type for body: "{type(text_or_element)}"')

    @property
    def start(self) -> Annotation:
        """Return self."""
        return self

    @property
    def end(self) -> AnnotationEnd | None:
        """Return the corresponding annotation-end tag or None."""
        name = self.name
        parent = self.parent
        if parent is None:  # pragma: nocover
            raise ValueError("Can't find end tag: no parent available")
        body: Body | Element = self.document_body or parent
        return body.get_annotation_end(name=name)

    def get_annotated(
        self,
        as_text: bool = False,
        no_header: bool = True,
        clean: bool = True,
    ) -> Element | list | str | None:
        """Returns the annotated content from an annotation.

        If no content exists (single position annotation or annotation-end not
        found), returns [] (or "" if text flag is True).
        If as_text is True: returns the text content.
        If clean is True: suppress unwanted tags (deletions marks, ...)
        If no_header is True: existing text:h are changed in text:p
        By default: returns a list of odf_element, cleaned and without headers.

        Args:

            as_text -- boolean

            clean -- boolean

            no_header -- boolean

        Returns: list or Element or text or None
        """
        end = self.end
        if end is None:
            if as_text:
                return ""
            return None
        body: Body | Element = self.document_body or self.root
        return elements_between(
            body, self, end, as_text=as_text, no_header=no_header, clean=clean
        )

    def delete(self, child: Element | None = None, keep_tail: bool = True) -> None:
        """Delete the given element from the XML tree. If no element is given,
        "self" is deleted. The XML library may allow to continue to use an
        element now "orphan" as long as you have a reference to it.

        For Annotation : delete the annotation-end tag if exists.

        Args:

            child -- Element or None
        """
        if child is not None:  # act like normal delete
            super().delete(child)
            return
        end = self.end
        if end:
            end.delete()
        # act like normal delete
        super().delete()

    def check_validity(self) -> None:
        if not self.note_body:
            raise ValueError("Annotation must have a body")
        if not self.dc_creator:
            raise ValueError("Annotation must have a creator")
        if not self.dc_date:
            self.dc_date = datetime.now()

    def __str__(self) -> str:
        return f"{self.note_body}\n{self.dc_creator} {self.dc_date}"


Annotation._define_attribut_property()


class AnnotationEnd(MDTail, Element):
    """End of annotation marker, "office:annotation-end".

    AnnotationEnd: the "office:annotation-end" element may be used to
    define the end of a text range of document content that spans element
    boundaries. In that case, an "office:annotation" element shall precede
    the "office:annotation-end" element. Both elements shall have the same
    value for their office:name attribute. The "office:annotation-end" element
    shall be preceded by an "office:annotation" element that has the same
    value for its office:name attribute as the "office:annotation-end"
    element. An "office:annotation-end" element without a preceding
    "office:annotation" element that has the same name assigned is ignored.
    """

    _tag = "office:annotation-end"
    _properties = (PropDef("name", "office:name"),)

    def __init__(
        self,
        annotation: Annotation | None = None,
        name: str | None = None,
        **kwargs: Any,
    ) -> None:
        """An annotation (private note), "text:annotation".

        Annotation element credited to the given creator with the
        given text, optionally dated (current date by default).
        If name not provided and some parent is provided, the name is
        autogenerated.

        Args:

            text -- str or odf_element

            creator -- str

            date -- datetime

            name -- str

            parent -- Element
        """
        # fixme : use offset
        # TODO allow paragraph and text styles
        super().__init__(**kwargs)
        if self._do_init:
            if annotation:
                name = annotation.name
            if not name:
                raise ValueError("Annotation-end must have a name")
            self.name = name

    @property
    def start(self) -> Annotation | None:
        """Return the corresponding annotation starting tag or None."""
        name = self.name
        parent = self.parent
        if parent is None:
            raise ValueError(
                "Can't find start tag: no parent available"
            )  # pragma:nocover
        body: Body | Element = self.document_body or parent
        return body.get_annotation(name=name)

    @property
    def end(self) -> AnnotationEnd:
        """Return self."""
        return self


AnnotationEnd._define_attribut_property()

register_element_class(Annotation)
register_element_class(AnnotationEnd)
