#!/usr/bin/env python
"""Test suite for :py:mod:`plastid.readers.gff`

Notes
-----
-  :py:func:`GTF2_to_Transcripts` and :py:func:`GFF3_to_Transcripts` are
    tested in :py:mod:`plastid.test.unit.genomics.test_roitools`
"""

import unittest
from plastid.util.services.mini2to3 import cStringIO
from nose.plugins.attrib import attr
from random import shuffle
from plastid.genomics.roitools import GenomicSegment, Transcript, SegmentChain
from plastid.readers.gff import GTF2_Reader,\
                             GFF3_Reader,\
                             GTF2_TranscriptAssembler,\
                             GFF3_TranscriptAssembler,\
                             StopFeature
from plastid.util.services.decorators import skip_if_abstract

from plastid.test.common import sup_data


#===============================================================================
# INDEX: test data
#===============================================================================

_GFF3_FEATURES = [
    SegmentChain(
        GenomicSegment('chrI', 0, 230218, '.'),
        ID='chrI',
        Name='chrI',
        dbxref=['NCBI:NC_001133'],
        phase='.',
        score='.',
        source='SGD',
        type='chromosome'
    ),
    SegmentChain(
        GenomicSegment('chrI', 0, 62, '-'),
        ID='TEL01L-TR',
        Name='TEL01L-TR',
        Note=['Terminal telomeric repeats on the left arm of Chromosome I'],
        dbxref=['SGD:S000028864'],
        display='Terminal telomeric repeats on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='telomeric_repeat'
    ),
    SegmentChain(
        GenomicSegment('chrI', 0, 801, '-'),
        ID='TEL01L',
        Name='TEL01L',
        Note=[
            'Telomeric region on the left arm of Chromosome I; composed of an X element core sequence, X element combinatorial repeats, and a short terminal stretch of telomeric repeats'
        ],
        dbxref=['SGD:S000028862'],
        display='Telomeric region on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='telomere'
    ),
    SegmentChain(
        GenomicSegment('chrI', 62, 336, '-'),
        ID='TEL01L-XR',
        Name='TEL01L-XR',
        Note=[
            'Telomeric X element combinatorial repeat on the left arm of Chr I; contains repeats of the D, C, B and A types, as well as Tbf1p binding sites; formerly called SubTelomeric Repeats'
        ],
        dbxref=['SGD:S000028866'],
        display='Telomeric X element combinatorial repeat on the left arm of Chr I',
        phase='.',
        score='.',
        source='SGD',
        type='X_element_combinatorial_repeat'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W',
        Name='YAL069W',
        Note=[
            'Dubious open reading frame; unlikely to encode a functional protein, based on available experimental and comparative sequence data'
        ],
        Ontology_term=['GO:0003674', 'GO:0005575', 'GO:0008150'],
        dbxref=['SGD:S000002143'],
        display='Dubious open reading frame',
        orf_classification='Dubious',
        phase='.',
        score='.',
        source='SGD',
        type='gene'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        Name='YAL069W_CDS',
        Parent=['YAL069W_mRNA'],
        orf_classification='Dubious',
        phase='0',
        score='.',
        source='SGD',
        type='CDS'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W_mRNA',
        Name='YAL069W_mRNA',
        Parent=['YAL069W'],
        phase='.',
        score='.',
        source='SGD',
        type='mRNA'
    ),
    SegmentChain(
        GenomicSegment('chrI', 336, 801, '-'),
        ID='TEL01L-XC',
        Name='TEL01L-XC',
        Note=[
            'Telomeric X element Core sequence on the left arm of Chromosome I; contains an ARS consensus sequence, an Abf1p binding site consensus sequence and two small overlapping ORFs (YAL068W-A and YAL069W)'
        ],
        dbxref=['SGD:S000028865'],
        display='Telomeric X element Core sequence on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='X_element'
    ),
    SegmentChain(
        GenomicSegment('chrII', 752, 763, '-'),
        ID='TEL02L-XC_nucleotide_match',
        Name='TEL02L-XC_nucleotide_match',
        dbxref=['SGD:S000028865'],
        phase='.',
        score='.',
        source='SGD',
        type='nucleotide_match'
    ),
    SegmentChain(
        GenomicSegment('chrII', 531, 544, '-'),
        ID='TEL02L-XC_binding_site',
        Name='TEL02L-XC_binding_site',
        dbxref=['SGD:S000028865'],
        phase='.',
        score='.',
        source='SGD',
        type='binding_site'
    ),
]

_GFF3_STOP_FEATURES = [
    SegmentChain(
        GenomicSegment('chrI', 0, 230218, '.'),
        ID='chrI',
        Name='chrI',
        dbxref=['NCBI:NC_001133'],
        phase='.',
        score='.',
        source='SGD',
        type='chromosome'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 0, 62, '-'),
        ID='TEL01L-TR',
        Name='TEL01L-TR',
        Note=['Terminal telomeric repeats on the left arm of Chromosome I'],
        dbxref=['SGD:S000028864'],
        display='Terminal telomeric repeats on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='telomeric_repeat'
    ),
    SegmentChain(
        GenomicSegment('chrI', 0, 801, '-'),
        ID='TEL01L',
        Name='TEL01L',
        Note=[
            'Telomeric region on the left arm of Chromosome I; composed of an X element core sequence, X element combinatorial repeats, and a short terminal stretch of telomeric repeats'
        ],
        dbxref=['SGD:S000028862'],
        display='Telomeric region on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='telomere'
    ),
    SegmentChain(
        GenomicSegment('chrI', 62, 336, '-'),
        ID='TEL01L-XR',
        Name='TEL01L-XR',
        Note=[
            'Telomeric X element combinatorial repeat on the left arm of Chr I; contains repeats of the D, C, B and A types, as well as Tbf1p binding sites; formerly called SubTelomeric Repeats'
        ],
        dbxref=['SGD:S000028866'],
        display='Telomeric X element combinatorial repeat on the left arm of Chr I',
        phase='.',
        score='.',
        source='SGD',
        type='X_element_combinatorial_repeat'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W',
        Name='YAL069W',
        Note=[
            'Dubious open reading frame; unlikely to encode a functional protein, based on available experimental and comparative sequence data'
        ],
        Ontology_term=['GO:0003674', 'GO:0005575', 'GO:0008150'],
        dbxref=['SGD:S000002143'],
        display='Dubious open reading frame',
        orf_classification='Dubious',
        phase='.',
        score='.',
        source='SGD',
        type='gene'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        Name='YAL069W_CDS',
        Parent=['YAL069W_mRNA'],
        orf_classification='Dubious',
        phase='0',
        score='.',
        source='SGD',
        type='CDS'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W_mRNA',
        Name='YAL069W_mRNA',
        Parent=['YAL069W'],
        phase='.',
        score='.',
        source='SGD',
        type='mRNA'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 336, 801, '-'),
        ID='TEL01L-XC',
        Name='TEL01L-XC',
        Note=[
            'Telomeric X element Core sequence on the left arm of Chromosome I; contains an ARS consensus sequence, an Abf1p binding site consensus sequence and two small overlapping ORFs (YAL068W-A and YAL069W)'
        ],
        dbxref=['SGD:S000028865'],
        display='Telomeric X element Core sequence on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='X_element'
    ),
    SegmentChain(
        GenomicSegment('chrII', 752, 763, '-'),
        ID='TEL02L-XC_nucleotide_match',
        Name='TEL02L-XC_nucleotide_match',
        dbxref=['SGD:S000028865'],
        phase='.',
        score='.',
        source='SGD',
        type='nucleotide_match'
    ),
    SegmentChain(
        GenomicSegment('chrII', 531, 544, '-'),
        ID='TEL02L-XC_binding_site',
        Name='TEL02L-XC_binding_site',
        dbxref=['SGD:S000028865'],
        phase='.',
        score='.',
        source='SGD',
        type='binding_site'
    ),
    StopFeature,
]

_GFF3_STOP_CHROM_FEATURES = [
    SegmentChain(
        GenomicSegment('chrI', 0, 230218, '.'),
        ID='chrI',
        Name='chrI',
        dbxref=['NCBI:NC_001133'],
        phase='.',
        score='.',
        source='SGD',
        type='chromosome'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 0, 62, '-'),
        ID='TEL01L-TR',
        Name='TEL01L-TR',
        Note=['Terminal telomeric repeats on the left arm of Chromosome I'],
        dbxref=['SGD:S000028864'],
        display='Terminal telomeric repeats on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='telomeric_repeat'
    ),
    SegmentChain(
        GenomicSegment('chrI', 0, 801, '-'),
        ID='TEL01L',
        Name='TEL01L',
        Note=[
            'Telomeric region on the left arm of Chromosome I; composed of an X element core sequence, X element combinatorial repeats, and a short terminal stretch of telomeric repeats'
        ],
        dbxref=['SGD:S000028862'],
        display='Telomeric region on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='telomere'
    ),
    SegmentChain(
        GenomicSegment('chrI', 62, 336, '-'),
        ID='TEL01L-XR',
        Name='TEL01L-XR',
        Note=[
            'Telomeric X element combinatorial repeat on the left arm of Chr I; contains repeats of the D, C, B and A types, as well as Tbf1p binding sites; formerly called SubTelomeric Repeats'
        ],
        dbxref=['SGD:S000028866'],
        display='Telomeric X element combinatorial repeat on the left arm of Chr I',
        phase='.',
        score='.',
        source='SGD',
        type='X_element_combinatorial_repeat'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W',
        Name='YAL069W',
        Note=[
            'Dubious open reading frame; unlikely to encode a functional protein, based on available experimental and comparative sequence data'
        ],
        Ontology_term=['GO:0003674', 'GO:0005575', 'GO:0008150'],
        dbxref=['SGD:S000002143'],
        display='Dubious open reading frame',
        orf_classification='Dubious',
        phase='.',
        score='.',
        source='SGD',
        type='gene'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        Name='YAL069W_CDS',
        Parent=['YAL069W_mRNA'],
        orf_classification='Dubious',
        phase='0',
        score='.',
        source='SGD',
        type='CDS'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W_mRNA',
        Name='YAL069W_mRNA',
        Parent=['YAL069W'],
        phase='.',
        score='.',
        source='SGD',
        type='mRNA'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 336, 801, '-'),
        ID='TEL01L-XC',
        Name='TEL01L-XC',
        Note=[
            'Telomeric X element Core sequence on the left arm of Chromosome I; contains an ARS consensus sequence, an Abf1p binding site consensus sequence and two small overlapping ORFs (YAL068W-A and YAL069W)'
        ],
        dbxref=['SGD:S000028865'],
        display='Telomeric X element Core sequence on the left arm of Chromosome I',
        phase='.',
        score='.',
        source='SGD',
        type='X_element'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrII', 752, 763, '-'),
        ID='TEL02L-XC_nucleotide_match',
        Name='TEL02L-XC_nucleotide_match',
        dbxref=['SGD:S000028865'],
        phase='.',
        score='.',
        source='SGD',
        type='nucleotide_match'
    ),
    SegmentChain(
        GenomicSegment('chrII', 531, 544, '-'),
        ID='TEL02L-XC_binding_site',
        Name='TEL02L-XC_binding_site',
        dbxref=['SGD:S000028865'],
        phase='.',
        score='.',
        source='SGD',
        type='binding_site'
    ),
    StopFeature,
]

# This data taken from SGD on 2014-11-16 at www.yeastgenome.org
_GFF3_FEATURE_TEXT = """##gff-version 3
#date Sun Nov 16 19:50:04 2014
#
# Saccharomyces cerevisiae S288C genome (version=R64-1-1)
#
# Features from the 16 nuclear chromosomes labeled chrI to chrXIV,
# plus the mitochondrial genome labeled chrmt.
#
# Created by Saccharomyces Genome Database (http://www.yeastgenome.org/)
#
# Weekly updates of this file are available for download from:
# http://downloads.yeastgenome.org/curation/chromosomal_feature/saccharomyces_cerevisiae.gff
#
# Please send comments and suggestions to sgd-helpdesk@lists.stanford.edu
#
# SGD is funded as a National Human Genome Research Institute Biomedical Informatics Resource from
# the U. S. National Institutes of Health to Stanford University.
#
chrI    SGD    chromosome    1    230218    .    .    .    ID=chrI;dbxref=NCBI:NC_001133;Name=chrI
chrI    SGD    telomeric_repeat    1    62    .    -    .    ID=TEL01L-TR;Name=TEL01L-TR;Note=Terminal telomeric repeats on the left arm of Chromosome I;display=Terminal telomeric repeats on the left arm of Chromosome I;dbxref=SGD:S000028864
chrI    SGD    telomere    1    801    .    -    .    ID=TEL01L;Name=TEL01L;Note=Telomeric region on the left arm of Chromosome I%3B composed of an X element core sequence%2C X element combinatorial repeats%2C and a short terminal stretch of telomeric repeats;display=Telomeric region on the left arm of Chromosome I;dbxref=SGD:S000028862
chrI    SGD    X_element_combinatorial_repeat    63    336    .    -    .    ID=TEL01L-XR;Name=TEL01L-XR;Note=Telomeric X element combinatorial repeat on the left arm of Chr I%3B contains repeats of the D%2C C%2C B and A types%2C as well as Tbf1p binding sites%3B formerly called SubTelomeric Repeats;display=Telomeric X element combinatorial repeat on the left arm of Chr I;dbxref=SGD:S000028866
chrI    SGD    gene    335    649    .    +    .    ID=YAL069W;Name=YAL069W;Ontology_term=GO:0003674,GO:0005575,GO:0008150;Note=Dubious open reading frame%3B unlikely to encode a functional protein%2C based on available experimental and comparative sequence data;display=Dubious open reading frame;dbxref=SGD:S000002143;orf_classification=Dubious
chrI    SGD    CDS    335    649    .    +    0    Parent=YAL069W_mRNA;Name=YAL069W_CDS;orf_classification=Dubious
chrI    SGD    mRNA    335    649    .    +    .    ID=YAL069W_mRNA;Name=YAL069W_mRNA;Parent=YAL069W
chrI    SGD    X_element    337    801    .    -    .    ID=TEL01L-XC;Name=TEL01L-XC;Note=Telomeric X element Core sequence on the left arm of Chromosome I%3B contains an ARS consensus sequence%2C an Abf1p binding site consensus sequence and two small overlapping ORFs (YAL068W-A and YAL069W);display=Telomeric X element Core sequence on the left arm of Chromosome I;dbxref=SGD:S000028865
chrII    SGD    nucleotide_match    753    763    .    -    .    ID=TEL02L-XC_nucleotide_match;Name=TEL02L-XC_nucleotide_match;dbxref=SGD:S000028865
chrII    SGD    binding_site    532    544    .    -    .    ID=TEL02L-XC_binding_site;Name=TEL02L-XC_binding_site;dbxref=SGD:S000028865""".replace(
    "    ", "\t"
)

# This data taken from SGD on 2014-11-16 at www.yeastgenome.org
_GFF3_STOP_FEATURE_TEXT = """##gff-version 3
#date Sun Nov 16 19:50:04 2014
#
# Saccharomyces cerevisiae S288C genome (version=R64-1-1)
#
# Features from the 16 nuclear chromosomes labeled chrI to chrXIV,
# plus the mitochondrial genome labeled chrmt.
#
# Created by Saccharomyces Genome Database (http://www.yeastgenome.org/)
#
# Weekly updates of this file are available for download from:
# http://downloads.yeastgenome.org/curation/chromosomal_feature/saccharomyces_cerevisiae.gff
#
# Please send comments and suggestions to sgd-helpdesk@lists.stanford.edu
#
# SGD is funded as a National Human Genome Research Institute Biomedical Informatics Resource from
# the U. S. National Institutes of Health to Stanford University.
#
chrI    SGD    chromosome    1    230218    .    .    .    ID=chrI;dbxref=NCBI:NC_001133;Name=chrI
###
chrI    SGD    telomeric_repeat    1    62    .    -    .    ID=TEL01L-TR;Name=TEL01L-TR;Note=Terminal telomeric repeats on the left arm of Chromosome I;display=Terminal telomeric repeats on the left arm of Chromosome I;dbxref=SGD:S000028864
chrI    SGD    telomere    1    801    .    -    .    ID=TEL01L;Name=TEL01L;Note=Telomeric region on the left arm of Chromosome I%3B composed of an X element core sequence%2C X element combinatorial repeats%2C and a short terminal stretch of telomeric repeats;display=Telomeric region on the left arm of Chromosome I;dbxref=SGD:S000028862
chrI    SGD    X_element_combinatorial_repeat    63    336    .    -    .    ID=TEL01L-XR;Name=TEL01L-XR;Note=Telomeric X element combinatorial repeat on the left arm of Chr I%3B contains repeats of the D%2C C%2C B and A types%2C as well as Tbf1p binding sites%3B formerly called SubTelomeric Repeats;display=Telomeric X element combinatorial repeat on the left arm of Chr I;dbxref=SGD:S000028866
chrI    SGD    gene    335    649    .    +    .    ID=YAL069W;Name=YAL069W;Ontology_term=GO:0003674,GO:0005575,GO:0008150;Note=Dubious open reading frame%3B unlikely to encode a functional protein%2C based on available experimental and comparative sequence data;display=Dubious open reading frame;dbxref=SGD:S000002143;orf_classification=Dubious
chrI    SGD    CDS    335    649    .    +    0    Parent=YAL069W_mRNA;Name=YAL069W_CDS;orf_classification=Dubious
chrI    SGD    mRNA    335    649    .    +    .    ID=YAL069W_mRNA;Name=YAL069W_mRNA;Parent=YAL069W
###
chrI    SGD    X_element    337    801    .    -    .    ID=TEL01L-XC;Name=TEL01L-XC;Note=Telomeric X element Core sequence on the left arm of Chromosome I%3B contains an ARS consensus sequence%2C an Abf1p binding site consensus sequence and two small overlapping ORFs (YAL068W-A and YAL069W);display=Telomeric X element Core sequence on the left arm of Chromosome I;dbxref=SGD:S000028865
chrII    SGD    nucleotide_match    753    763    .    -    .    ID=TEL02L-XC_nucleotide_match;Name=TEL02L-XC_nucleotide_match;dbxref=SGD:S000028865
chrII    SGD    binding_site    532    544    .    -    .    ID=TEL02L-XC_binding_site;Name=TEL02L-XC_binding_site;dbxref=SGD:S000028865
###""".replace("    ", "\t")

_GTF2_FEATURES = [
    SegmentChain(
        GenomicSegment('chrI', 223, 790, '+'),
        ID='YAL069W_mRNA',
        gene_aliases='YAL069W',
        gene_id='YAL069W',
        name='YAL069W',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL069W_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W_mRNA',
        gene_aliases='YAL069W',
        gene_id='YAL069W',
        name='YAL069W',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL069W_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 426, 933, '+'),
        ID='YAL068W-A_mRNA',
        gene_aliases='YAL068W-A',
        gene_id='YAL068W-A',
        name='YAL068W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068W-A_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 537, 792, '+'),
        ID='YAL068W-A_mRNA',
        gene_aliases='YAL068W-A',
        gene_id='YAL068W-A',
        name='YAL068W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068W-A_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 1665, 2280, '-'),
        ID='YAL068C_mRNA',
        gene_aliases='PAU8,seripauperin PAU8',
        gene_id='YAL068C',
        name='PAU8',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068C_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 1806, 2169, '-'),
        ID='YAL068C_mRNA',
        gene_aliases='PAU8,seripauperin PAU8',
        gene_id='YAL068C',
        name='PAU8',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068C_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 2368, 2848, '+'),
        ID='YAL067W-A_mRNA',
        gene_aliases='YAL067W-A',
        gene_id='YAL067W-A',
        name='YAL067W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL067W-A_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 2479, 2707, '+'),
        ID='YAL067W-A_mRNA',
        gene_aliases='YAL067W-A',
        gene_id='YAL067W-A',
        name='YAL067W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL067W-A_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrII', 7093, 9127, '-'),
        ID='YBL067C_mRNA',
        gene_aliases='SEO1,putative permease SEO1',
        gene_id='YBL067C',
        name='SEO1',
        phase='.',
        score='.',
        source='.',
        transcript_id='YBL067C_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrII', 7234, 9016, '-'),
        ID='YBL067C_mRNA',
        gene_aliases='SEO1,putative permease SEO1',
        gene_id='YBL067C',
        name='SEO1',
        phase='.',
        score='.',
        source='.',
        transcript_id='YBL067C_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
]

_GTF2_STOP_FEATURES = [
    SegmentChain(
        GenomicSegment('chrI', 223, 790, '+'),
        ID='YAL069W_mRNA',
        gene_aliases='YAL069W',
        gene_id='YAL069W',
        name='YAL069W',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL069W_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W_mRNA',
        gene_aliases='YAL069W',
        gene_id='YAL069W',
        name='YAL069W',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL069W_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 426, 933, '+'),
        ID='YAL068W-A_mRNA',
        gene_aliases='YAL068W-A',
        gene_id='YAL068W-A',
        name='YAL068W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068W-A_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 537, 792, '+'),
        ID='YAL068W-A_mRNA',
        gene_aliases='YAL068W-A',
        gene_id='YAL068W-A',
        name='YAL068W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068W-A_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 1665, 2280, '-'),
        ID='YAL068C_mRNA',
        gene_aliases='PAU8,seripauperin PAU8',
        gene_id='YAL068C',
        name='PAU8',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068C_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 1806, 2169, '-'),
        ID='YAL068C_mRNA',
        gene_aliases='PAU8,seripauperin PAU8',
        gene_id='YAL068C',
        name='PAU8',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068C_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 2368, 2848, '+'),
        ID='YAL067W-A_mRNA',
        gene_aliases='YAL067W-A',
        gene_id='YAL067W-A',
        name='YAL067W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL067W-A_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 2479, 2707, '+'),
        ID='YAL067W-A_mRNA',
        gene_aliases='YAL067W-A',
        gene_id='YAL067W-A',
        name='YAL067W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL067W-A_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrII', 7093, 9127, '-'),
        ID='YBL067C_mRNA',
        gene_aliases='SEO1,putative permease SEO1',
        gene_id='YBL067C',
        name='SEO1',
        phase='.',
        score='.',
        source='.',
        transcript_id='YBL067C_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrII', 7234, 9016, '-'),
        ID='YBL067C_mRNA',
        gene_aliases='SEO1,putative permease SEO1',
        gene_id='YBL067C',
        name='SEO1',
        phase='.',
        score='.',
        source='.',
        transcript_id='YBL067C_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    StopFeature,
]

_GTF2_STOP_CHROM_FEATURES = [
    SegmentChain(
        GenomicSegment('chrI', 223, 790, '+'),
        ID='YAL069W_mRNA',
        gene_aliases='YAL069W',
        gene_id='YAL069W',
        name='YAL069W',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL069W_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 334, 649, '+'),
        ID='YAL069W_mRNA',
        gene_aliases='YAL069W',
        gene_id='YAL069W',
        name='YAL069W',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL069W_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 426, 933, '+'),
        ID='YAL068W-A_mRNA',
        gene_aliases='YAL068W-A',
        gene_id='YAL068W-A',
        name='YAL068W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068W-A_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 537, 792, '+'),
        ID='YAL068W-A_mRNA',
        gene_aliases='YAL068W-A',
        gene_id='YAL068W-A',
        name='YAL068W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068W-A_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 1665, 2280, '-'),
        ID='YAL068C_mRNA',
        gene_aliases='PAU8,seripauperin PAU8',
        gene_id='YAL068C',
        name='PAU8',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068C_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 1806, 2169, '-'),
        ID='YAL068C_mRNA',
        gene_aliases='PAU8,seripauperin PAU8',
        gene_id='YAL068C',
        name='PAU8',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL068C_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrI', 2368, 2848, '+'),
        ID='YAL067W-A_mRNA',
        gene_aliases='YAL067W-A',
        gene_id='YAL067W-A',
        name='YAL067W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL067W-A_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrI', 2479, 2707, '+'),
        ID='YAL067W-A_mRNA',
        gene_aliases='YAL067W-A',
        gene_id='YAL067W-A',
        name='YAL067W-A',
        phase='.',
        score='.',
        source='.',
        transcript_id='YAL067W-A_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    StopFeature,
    SegmentChain(
        GenomicSegment('chrII', 7093, 9127, '-'),
        ID='YBL067C_mRNA',
        gene_aliases='SEO1,putative permease SEO1',
        gene_id='YBL067C',
        name='SEO1',
        phase='.',
        score='.',
        source='.',
        transcript_id='YBL067C_mRNA',
        type='exon',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    SegmentChain(
        GenomicSegment('chrII', 7234, 9016, '-'),
        ID='YBL067C_mRNA',
        gene_aliases='SEO1,putative permease SEO1',
        gene_id='YBL067C',
        name='SEO1',
        phase='.',
        score='.',
        source='.',
        transcript_id='YBL067C_mRNA',
        type='CDS',
        utr3_source='estimated',
        utr5_source='estimated'
    ),
    StopFeature,
]

# This data adapted from SGD at www.yeastgenome.org
_GTF2_FEATURE_TEXT = """chrI    .    exon    224    790    .    +    .    gene_id "YAL069W"; transcript_id "YAL069W_mRNA"; name "YAL069W"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL069W"; ID "YAL069W_mRNA"; 
chrI    .    CDS    335    649    .    +    .    gene_id "YAL069W"; transcript_id "YAL069W_mRNA"; name "YAL069W"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL069W"; ID "YAL069W_mRNA"; 
chrI    .    exon    427    933    .    +    .    gene_id "YAL068W-A"; transcript_id "YAL068W-A_mRNA"; name "YAL068W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL068W-A"; ID "YAL068W-A_mRNA"; 
chrI    .    CDS    538    792    .    +    .    gene_id "YAL068W-A"; transcript_id "YAL068W-A_mRNA"; name "YAL068W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL068W-A"; ID "YAL068W-A_mRNA"; 
chrI    .    exon    1666    2280    .    -    .    gene_id "YAL068C"; transcript_id "YAL068C_mRNA"; name "PAU8"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "PAU8%2Cseripauperin PAU8"; ID "YAL068C_mRNA"; 
chrI    .    CDS    1807    2169    .    -    .    gene_id "YAL068C"; transcript_id "YAL068C_mRNA"; name "PAU8"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "PAU8%2Cseripauperin PAU8"; ID "YAL068C_mRNA"; 
chrI    .    exon    2369    2848    .    +    .    gene_id "YAL067W-A"; transcript_id "YAL067W-A_mRNA"; name "YAL067W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL067W-A"; ID "YAL067W-A_mRNA"; 
chrI    .    CDS    2480    2707    .    +    .    gene_id "YAL067W-A"; transcript_id "YAL067W-A_mRNA"; name "YAL067W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL067W-A"; ID "YAL067W-A_mRNA"; 

chrII    .    exon    7094    9127    .    -    .    gene_id "YBL067C"; transcript_id "YBL067C_mRNA"; name "SEO1"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "SEO1%2Cputative permease SEO1"; ID "YBL067C_mRNA"; 
chrII    .    CDS    7235    9016    .    -    .    gene_id "YBL067C"; transcript_id "YBL067C_mRNA"; name "SEO1"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "SEO1%2Cputative permease SEO1"; ID "YBL067C_mRNA"; """.replace(
    "    ", "\t"
)

_GTF2_STOP_FEATURE_TEXT = """chrI    .    exon    224    790    .    +    .    gene_id "YBL069W"; transcript_id "YBL069W_mRNA"; name "YBL069W"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YBL069W"; ID "YBL069W_mRNA"; 
chrI    .    CDS    335    649    .    +    .    gene_id "YAL069W"; transcript_id "YAL069W_mRNA"; name "YAL069W"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL069W"; ID "YAL069W_mRNA"; 
###
chrI    .    exon    427    933    .    +    .    gene_id "YAL068W-A"; transcript_id "YAL068W-A_mRNA"; name "YAL068W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL068W-A"; ID "YAL068W-A_mRNA"; 
chrI    .    CDS    538    792    .    +    .    gene_id "YAL068W-A"; transcript_id "YAL068W-A_mRNA"; name "YAL068W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL068W-A"; ID "YAL068W-A_mRNA"; 
chrI    .    exon    1666    2280    .    -    .    gene_id "YAL068C"; transcript_id "YAL068C_mRNA"; name "PAU8"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "PAU8%2Cseripauperin PAU8"; ID "YAL068C_mRNA"; 
chrI    .    CDS    1807    2169    .    -    .    gene_id "YAL068C"; transcript_id "YAL068C_mRNA"; name "PAU8"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "PAU8%2Cseripauperin PAU8"; ID "YAL068C_mRNA"; 
###
chrI    .    exon    2369    2848    .    +    .    gene_id "YAL067W-A"; transcript_id "YAL067W-A_mRNA"; name "YAL067W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL067W-A"; ID "YAL067W-A_mRNA"; 
chrI    .    CDS    2480    2707    .    +    .    gene_id "YAL067W-A"; transcript_id "YAL067W-A_mRNA"; name "YAL067W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL067W-A"; ID "YAL067W-A_mRNA"; 
chrII    .    exon    7094    9127    .    -    .    gene_id "YBL067C"; transcript_id "YBL067C_mRNA"; name "SEO1"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "SEO1%2Cputative permease SEO1"; ID "YBL067C_mRNA"; 
chrII    .    CDS    7235    9016    .    -    .    gene_id "YBL067C"; transcript_id "YBL067C_mRNA"; name "SEO1"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "SEO1%2Cputative permease SEO1"; ID "YBL067C_mRNA";
###""".replace("    ", "\t")

_GTF2_FEATURE_TEXT_ADD_3 = """chrI    .    exon    224    790    .    +    .    gene_id "YAL069W"; transcript_id "YAL069W_mRNA"; name "YAL069W"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL069W"; ID "YAL069W_mRNA"; 
chrI    .    CDS    335    646    .    +    .    gene_id "YAL069W"; transcript_id "YAL069W_mRNA"; name "YAL069W"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL069W"; ID "YAL069W_mRNA"; 
chrI    .    exon    427    933    .    +    .    gene_id "YAL068W-A"; transcript_id "YAL068W-A_mRNA"; name "YAL068W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL068W-A"; ID "YAL068W-A_mRNA"; 
chrI    .    CDS    538    789    .    +    .    gene_id "YAL068W-A"; transcript_id "YAL068W-A_mRNA"; name "YAL068W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL068W-A"; ID "YAL068W-A_mRNA"; 
chrI    .    exon    1666    2280    .    -    .    gene_id "YAL068C"; transcript_id "YAL068C_mRNA"; name "PAU8"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "PAU8%2Cseripauperin PAU8"; ID "YAL068C_mRNA"; 
chrI    .    CDS    1804    2169    .    -    .    gene_id "YAL068C"; transcript_id "YAL068C_mRNA"; name "PAU8"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "PAU8%2Cseripauperin PAU8"; ID "YAL068C_mRNA"; 
chrI    .    exon    2369    2848    .    +    .    gene_id "YAL067W-A"; transcript_id "YAL067W-A_mRNA"; name "YAL067W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL067W-A"; ID "YAL067W-A_mRNA"; 
chrI    .    CDS    2480    2704    .    +    .    gene_id "YAL067W-A"; transcript_id "YAL067W-A_mRNA"; name "YAL067W-A"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "YAL067W-A"; ID "YAL067W-A_mRNA"; 

chrII    .    exon    7094    9127    .    -    .    gene_id "YBL067C"; transcript_id "YBL067C_mRNA"; name "SEO1"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "SEO1%2Cputative permease SEO1"; ID "YBL067C_mRNA"; 
chrII    .    CDS    7232    9016    .    -    .    gene_id "YBL067C"; transcript_id "YBL067C_mRNA"; name "SEO1"; utr5_source "estimated"; utr3_source "estimated"; gene_aliases "SEO1%2Cputative permease SEO1"; ID "YBL067C_mRNA"; """.replace(
    "    ", "\t"
)

REJECTED_NAMES = ["TestBadTranscript1", "TestBadTranscript2"]

EXPECTED_IVCS = [
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3221892, 3222171, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247586, 3247710, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111092',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3231783, 3232074, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241320, 3241441, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3254309, 3254471, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0089016',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3231783, 3232074, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241320, 3241441, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0089018',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234511, 3234799, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3243821, 3243945, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252522, 3252684, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111049',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3228449, 3228737, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241781, 3241905, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252236, 3252398, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111053',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3226145, 3226433, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3239203, 3239327, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252236, 3252398, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111056',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3226907, 3227195, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244249, 3244373, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111060',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3227287, 3227575, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3237878, 3238002, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3254309, 3254471, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111085',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3227659, 3227947, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3239203, 3239327, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111096',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232168, 3232459, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3237105, 3237226, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306783',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3226532, 3226820, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246726, 3246850, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306784',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3223220, 3223508, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247171, 3247295, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306787',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3228832, 3229123, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3242386, 3242510, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252236, 3252398, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111052',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3233380, 3233671, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246946, 3247070, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111057',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3231783, 3232074, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244031, 3244155, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111061',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230836, 3231127, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246520, 3246644, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3254309, 3254471, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111083',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3227287, 3227575, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241070, 3241197, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3254309, 3254471, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111084',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232168, 3232459, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3242795, 3242919, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111088',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232168, 3232459, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246946, 3247070, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252522, 3252684, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111100',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3225754, 3226042, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3236897, 3237018, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111066',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234146, 3234434, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3240403, 3240527, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251385, 3251553, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111067',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3224998, 3225286, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3236897, 3237018, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251099, 3251270, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111071',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3231783, 3232074, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241320, 3241441, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0089019',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3225376, 3225664, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238085, 3238209, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251099, 3251270, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111070',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3233773, 3234061, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241781, 3241905, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252816, 3252987, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111097',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3224035, 3224323, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241981, 3242105, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251099, 3251270, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306776',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3226145, 3226433, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238462, 3238586, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306779',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230836, 3231127, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238286, 3238410, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251385, 3251553, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306780',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230087, 3230386, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3242190, 3242314, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306768',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230087, 3230386, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3245095, 3245219, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306771',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3228832, 3229123, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246946, 3247070, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111064',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3231783, 3232074, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244477, 3244601, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250528, 3250699, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111074',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234896, 3235202, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238678, 3238802, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0089020',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3214121, 3214289, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3224998, 3225286, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247379, 3247512, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111063',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3222438, 3222723, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241561, 3241682, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250836, 3250995, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306777',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230468, 3230753, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246726, 3246850, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111058',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3222438, 3222723, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3245293, 3245420, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250836, 3250995, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111075',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3222438, 3222723, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3248030, 3248154, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111076',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3231222, 3231513, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3245933, 3246051, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306773',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3227659, 3227947, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3242591, 3242715, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252522, 3252684, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111050',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3228049, 3228337, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244249, 3244373, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252236, 3252398, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111051',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232565, 3232856, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3240749, 3240870, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252236, 3252398, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111054',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3223220, 3223508, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246311, 3246435, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111077',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234146, 3234434, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246108, 3246232, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3254309, 3254471, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111078',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234146, 3234434, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3245720, 3245844, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252522, 3252684, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111079',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3229608, 3229896, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247795, 3247919, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3254309, 3254471, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111081',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3223220, 3223508, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244477, 3244601, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111087',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3223220, 3223508, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3239203, 3239327, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111089',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3221892, 3222171, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238085, 3238209, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252816, 3252987, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111098',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3228832, 3229123, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3239420, 3239544, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252236, 3252398, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111055',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3233380, 3233671, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244477, 3244601, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111059',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232168, 3232459, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238085, 3238209, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111062',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3233380, 3233671, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247171, 3247295, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3254309, 3254471, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111082',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232565, 3232856, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3243405, 3243529, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111094',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232565, 3232856, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3237489, 3237613, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306782',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3221191, 3221485, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244687, 3244811, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252522, 3252684, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306774',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3225376, 3225664, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3241781, 3241905, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111065',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234511, 3234799, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244904, 3245028, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251099, 3251270, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111068',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3229608, 3229896, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3239203, 3239327, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251099, 3251270, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111069',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3229212, 3229500, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3244249, 3244373, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111101',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3224593, 3224881, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247795, 3247919, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251682, 3251853, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306785',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230087, 3230386, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238678, 3238802, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306769',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230087, 3230386, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247586, 3247710, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306770',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230087, 3230386, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3243190, 3243314, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253447, 3253609, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306772',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3228832, 3229123, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3246726, 3246850, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250528, 3250699, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111072',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3228832, 3229123, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3245095, 3245219, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250528, 3250699, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111073',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232989, 3233280, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3243611, 3243735, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250528, 3250699, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111080',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3232168, 3232459, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3237292, 3237416, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252816, 3252987, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111099',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3223220, 3223508, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3245507, 3245634, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3252816, 3252987, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306775',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3223621, 3223915, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3238965, 3239089, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250528, 3250699, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306778',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3223621, 3223915, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3242591, 3242715, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3250528, 3250699, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306786',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234896, 3235202, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3243190, 3243314, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3251960, 3252122, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0111102',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3234896, 3235202, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3237662, 3237786, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306781',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('2R', 3207058, 3208269, '-'),
        GenomicSegment('2R', 3209087, 3209154, '-'),
        GenomicSegment('2R', 3210064, 3210223, '-'),
        GenomicSegment('2R', 3210280, 3210461, '-'),
        GenomicSegment('2R', 3210526, 3210837, '-'),
        GenomicSegment('2R', 3211323, 3211473, '-'),
        GenomicSegment('2R', 3212086, 3212242, '-'),
        GenomicSegment('2R', 3213656, 3213869, '-'),
        GenomicSegment('2R', 3216793, 3216960, '-'),
        GenomicSegment('2R', 3217017, 3217194, '-'),
        GenomicSegment('2R', 3217247, 3217356, '-'),
        GenomicSegment('2R', 3217421, 3218105, '-'),
        GenomicSegment('2R', 3218163, 3218319, '-'),
        GenomicSegment('2R', 3218480, 3219707, '-'),
        GenomicSegment('2R', 3219765, 3219885, '-'),
        GenomicSegment('2R', 3230087, 3230386, '-'),
        GenomicSegment('2R', 3235353, 3235515, '-'),
        GenomicSegment('2R', 3235635, 3236480, '-'),
        GenomicSegment('2R', 3247379, 3247512, '-'),
        GenomicSegment('2R', 3249222, 3249372, '-'),
        GenomicSegment('2R', 3253970, 3254132, '-'),
        GenomicSegment('2R', 3255892, 3256237, '-'),
        GenomicSegment('2R', 3265038, 3265219, '-'),
        GenomicSegment('2R', 3268975, 3269374, '-'),
        ID='FBtr0306767',
        cds_genome_start=3208187,
        cds_genome_end=3265099
    ),
    Transcript(
        GenomicSegment('3R', 14982961, 14983694, '+'),
        GenomicSegment('3R', 14984237, 14984707, '+'),
        GenomicSegment('3R', 14984780, 14985924, '+'),
        ID='FBtr0083732',
        cds_genome_start=14983222,
        cds_genome_end=14985044
    ),
]

# from dmel-all-no-analysis-no-fasta-r5.43.gff
# Dscam1 examples
GFF3_TO_TRANSCRIPTS_SORTED = """2R    FlyBase    gene    3207059    3269374    .    -    .    ID=FBgn0033159;Name=Dscam;fullname=Down syndrome cell adhesion molecule;Alias=FBgn0004124,FBgn0010604,FBgn0033159,Neu1,Dscam1,p270,CT39257,l(2)43Bc,43Bc,lethal(2)43Bc,l(2)05518,CG17800,DScam,Drosophila Down syndrome cell adhesion molecule,Down's syndrome Cell Adhesion Molecule,DSCAM,Down syndrome cell-adhesion molecule,Down Syndrome Cell Adhesion Molecule,dscam,Down Syndrome cell adhesion molecule,dScam,DmDscam;Ontology_term=GO:0016319,SO:0000010,SO:0000087,GO:0007411,GO:0005887,GO:0008046,GO:0006909,GO:0007422,GO:0048666,GO:0007413,GO:0043025,GO:0030425,GO:0030424,GO:0021551,GO:0042803,GO:0048846,GO:0070593,GO:0051635,GO:0042802;Dbxref=FlyBase:FBan0017800,FlyBase_Annotation_IDs:CG17800,GB_protein:ABI31027,GB_protein:ABI31036,GB_protein:ABI31050,GB_protein:ABI31041,GB_protein:ABI31040,GB_protein:ABI31048,GB_protein:ABI31031,GB_protein:ABI31033,GB_protein:ABI31055,GB_protein:ABI31053,GB_protein:ABI31037,GB_protein:AAM68883,GB_protein:ABI31035,GB_protein:ABI31043,GB_protein:ABI31039,GB_protein:ABI31067,GB_protein:ABI31075,GB_protein:ABI31054,GB_protein:ABI31073,GB_protein:ABI31030,GB_protein:ABI31061,GB_protein:ABI31057,GB_protein:ABI31056,GB_protein:ABI31079,GB_protein:ABI31029,GB_protein:ABI31064,GB_protein:ABI31069,GB_protein:ABI31078,GB_protein:ABI31080,GB_protein:ABI31077,GB_protein:ABI31028,GB_protein:ABI31066,GB_protein:ABI31052,GB_protein:ABI31032,GB_protein:ABI31068,GB_protein:ABI31042,GB_protein:ABI31046,GB_protein:ABI31072,GB_protein:AAS64901,GB_protein:ABI31034,GB_protein:ABI31062,GB_protein:ABI31051,GB_protein:ABI31070,GB_protein:ABI31044,GB_protein:ABI31058,GB_protein:ABI31047,GB_protein:ABI31049,GB_protein:AAM68884,GB_protein:ABI31059,GB_protein:ABI31060,GB_protein:ABI31071,GB_protein:ABI31081,GB_protein:ABI31074,GB_protein:ABI31038,GB_protein:AAF59271,GB_protein:ABI31045,GB_protein:ABI31063,GB_protein:ABI31076,GB_protein:ABI31065,GB:AA697364,GB:AF260530,GB_protein:AAF71926,GB:AI542894,GB:AQ034165,GB:AW941417,GB:AW944596,GB:AY058789,GB_protein:AAL14018,GB:AY061009,GB_protein:AAL28557,GB:BG640148,GB:BH256401,GB:BH759233,GB:BH759234,GB:BT001877,GB_protein:AAN71649,GB:BZ748857,GB:BZ748858,GB:CL528133,GB:CL528281,GB:CZ466627,GB:CZ477334,GB:CZ482393,GB:CZ482394,GB:CZ482629,GB:CZ489596,UniProt/TrEMBL:Q8IGA5,UniProt/TrEMBL:Q95S10,UniProt/TrEMBL:Q95TG0,UniProt/TrEMBL:Q9NBA1,INTERPRO:IPR003598,INTERPRO:IPR003599,INTERPRO:IPR003961,INTERPRO:IPR007110,UniProt/TrEMBL:Q0E9H9,UniProt/TrEMBL:A1Z6X3,UniProt/TrEMBL:A1Z6X2,UniProt/TrEMBL:A1Z6X1,INTERPRO:IPR013783,EntrezGene:35652,UniProt/TrEMBL:Q0E9J9,UniProt/TrEMBL:Q0E9K0,UniProt/TrEMBL:Q0E9J1,UniProt/TrEMBL:Q0E9K3,UniProt/TrEMBL:Q0E9L9,UniProt/TrEMBL:Q0E9K1,UniProt/TrEMBL:Q0E9H0,UniProt/TrEMBL:Q0E9M2,UniProt/TrEMBL:Q0E9G9,UniProt/TrEMBL:Q0E9I8,UniProt/TrEMBL:Q0E9L5,UniProt/TrEMBL:Q0E9L2,UniProt/TrEMBL:Q0E9H5,UniProt/TrEMBL:Q0E9K2,UniProt/TrEMBL:Q0E9J6,UniProt/TrEMBL:Q0E9L7,UniProt/TrEMBL:Q0E9I3,UniProt/TrEMBL:Q0E9K4,UniProt/TrEMBL:Q0E9I6,UniProt/TrEMBL:Q0E9J4,UniProt/TrEMBL:Q0E9H4,UniProt/TrEMBL:Q0E9H8,UniProt/TrEMBL:Q0E9K6,UniProt/TrEMBL:Q0E9I4,UniProt/TrEMBL:Q0E9H6,UniProt/TrEMBL:Q0E9I9,UniProt/TrEMBL:Q0E9J3,UniProt/TrEMBL:Q0E9J5,UniProt/TrEMBL:Q0E9L3,UniProt/TrEMBL:Q0E9M0,UniProt/TrEMBL:Q0E9I2,UniProt/TrEMBL:Q0E9K9,UniProt/TrEMBL:Q0E9J8,UniProt/TrEMBL:Q0E9J0,UniProt/TrEMBL:Q0E9L6,UniProt/TrEMBL:Q0E9L4,UniProt/TrEMBL:Q0E9I0,UniProt/TrEMBL:Q0E9I7,UniProt/TrEMBL:Q0E9J7,UniProt/TrEMBL:Q0E9K5,UniProt/TrEMBL:Q0E9H7,UniProt/TrEMBL:Q0E9L0,UniProt/TrEMBL:Q0E9M4,UniProt/TrEMBL:Q0E9L8,UniProt/TrEMBL:Q0E9H3,UniProt/TrEMBL:Q0E9M1,UniProt/TrEMBL:Q0E9H1,UniProt/TrEMBL:Q0E9M3,UniProt/TrEMBL:Q0E9I5,UniProt/TrEMBL:Q0E9L1,UniProt/TrEMBL:Q0E9K8,UniProt/TrEMBL:Q0E9K7,UniProt/TrEMBL:Q0E9J2,UniProt/TrEMBL:Q0E9H2,UniProt/TrEMBL:Q0E9I1,INTERPRO:IPR021012,INTERPRO:IPR013098,InterologFinder:35652,BIOGRID:61533,FlyAtlas:HDC04985%3BCG17800-RA%3BCG15693-RA,GenomeRNAi:35652,INTERACTIVEFLY:/hjmuller/dscam1.htm;gbunit=AE013599;derived_computed_cyto=43A4-43B1
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089016;Name=Dscam-RA;Parent=FBgn0033159;Alias=CG17800-RA;Dbxref=FlyBase_Annotation_IDs:CG17800-RA,REFSEQ:NM_165514;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089018;Name=Dscam-RC;Parent=FBgn0033159;Alias=CG17800-RC;Dbxref=FlyBase_Annotation_IDs:CG17800-RC,REFSEQ:NM_165513;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089019;Name=Dscam-RD;Parent=FBgn0033159;Alias=CG17800-RD;Dbxref=FlyBase_Annotation_IDs:CG17800-RD,REFSEQ:NM_078925;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089020;Name=Dscam-RE;Parent=FBgn0033159;Alias=CG17800-RE;Dbxref=FlyBase_Annotation_IDs:CG17800-RE,REFSEQ:NM_206047;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111049;Name=Dscam-RAA;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAA,REFSEQ:NM_001043036;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111050;Name=Dscam-RAB;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAB,REFSEQ:NM_001043050;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111051;Name=Dscam-RAC;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAC,REFSEQ:NM_001043012;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111052;Name=Dscam-RAD;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAD,REFSEQ:NM_001043061;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111053;Name=Dscam-RAE;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAE,REFSEQ:NM_001043064;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111054;Name=Dscam-RAF;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAF,REFSEQ:NM_001043062;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111055;Name=Dscam-RAG;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAG,REFSEQ:NM_001043053;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111056;Name=Dscam-RAH;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAH,REFSEQ:NM_001043048;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111057;Name=Dscam-RAI;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAI,REFSEQ:NM_001043063;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111058;Name=Dscam-RAJ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAJ,REFSEQ:NM_001043040;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111059;Name=Dscam-RAK;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAK,REFSEQ:NM_001043045;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111060;Name=Dscam-RAL;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAL,REFSEQ:NM_001043014;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111061;Name=Dscam-RAM;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAM,REFSEQ:NM_001043057;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111062;Name=Dscam-RAO;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAO,REFSEQ:NM_001043059;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111063;Name=Dscam-RAP;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAP,REFSEQ:NM_001043051;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111064;Name=Dscam-RAQ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAQ,REFSEQ:NM_001043023;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111065;Name=Dscam-RAS;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAS,REFSEQ:NM_001043019;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111066;Name=Dscam-RAT;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAT,REFSEQ:NM_001043021;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111067;Name=Dscam-RAU;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAU,REFSEQ:NM_001043037;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111068;Name=Dscam-RAV;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAV,REFSEQ:NM_001043039;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111069;Name=Dscam-RAW;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAW,REFSEQ:NM_001043017;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111070;Name=Dscam-RAX;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAX,REFSEQ:NM_001043015;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111071;Name=Dscam-RAY;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAY,REFSEQ:NM_001043032;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111072;Name=Dscam-RAZ;Parent=FBgn0033159;Alias=Dscam-RAY;Dbxref=FlyBase_Annotation_IDs:CG17800-RAZ,REFSEQ:NM_001043025;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111073;Name=Dscam-RBA;Parent=FBgn0033159;Alias=Dscam-RAZ;Dbxref=FlyBase_Annotation_IDs:CG17800-RBA,REFSEQ:NM_001043034;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111074;Name=Dscam-RBB;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBB,REFSEQ:NM_001043020;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111075;Name=Dscam-RBC;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBC,REFSEQ:NM_001043024;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111076;Name=Dscam-RBD;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBD,REFSEQ:NM_001043013;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111077;Name=Dscam-RBE;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBE,REFSEQ:NM_001043041;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111078;Name=Dscam-RBF;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBF,REFSEQ:NM_001043029;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111079;Name=Dscam-RBG;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBG,REFSEQ:NM_001043016;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111080;Name=Dscam-RBH;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBH,REFSEQ:NM_001043011;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111081;Name=Dscam-RF;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RF,REFSEQ:NM_001043049;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111082;Name=Dscam-RG;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RG,REFSEQ:NM_001043060;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111083;Name=Dscam-RH;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RH,REFSEQ:NM_001043047;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111084;Name=Dscam-RI;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RI,REFSEQ:NM_001043022;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111085;Name=Dscam-RJ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RJ,REFSEQ:NM_001043058;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111087;Name=Dscam-RL;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RL,REFSEQ:NM_001043055;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111088;Name=Dscam-RM;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RM,REFSEQ:NM_001043044;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111089;Name=Dscam-RO;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RO,REFSEQ:NM_001043033;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111092;Name=Dscam-RR;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RR,REFSEQ:NM_001043028;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111094;Name=Dscam-RT;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RT,REFSEQ:NM_001043035;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111096;Name=Dscam-RV;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RV,REFSEQ:NM_001043018;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111097;Name=Dscam-RW;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RW,REFSEQ:NM_001043056;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111098;Name=Dscam-RX;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RX,REFSEQ:NM_001043030;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111099;Name=Dscam-RY;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RY,REFSEQ:NM_001043026;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111100;Name=Dscam-RZ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RZ,REFSEQ:NM_001043052;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111101;Name=Dscam-RAR;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAR,REFSEQ:NM_001043027;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111102;Name=Dscam-RAN;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAN,REFSEQ:NM_001043038;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306767;Name=Dscam-RBX;Parent=FBgn0033159;Alias=FBtr0111086,CG17800-RK;Dbxref=FlyBase_Annotation_IDs:CG17800-RBX;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306768;Name=Dscam-RBY;Parent=FBgn0033159;Alias=FBtr0111103,CG17800-RN;Dbxref=FlyBase_Annotation_IDs:CG17800-RBY;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306769;Name=Dscam-RBZ;Parent=FBgn0033159;Alias=FBtr0111090,CG17800-RP;Dbxref=FlyBase_Annotation_IDs:CG17800-RBZ;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306770;Name=Dscam-RCA;Parent=FBgn0033159;Alias=FBtr0111091,CG17800-RQ;Dbxref=FlyBase_Annotation_IDs:CG17800-RCA;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306771;Name=Dscam-RCB;Parent=FBgn0033159;Alias=FBtr0111093,CG17800-RS;Dbxref=FlyBase_Annotation_IDs:CG17800-RCB;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306772;Name=Dscam-RCC;Parent=FBgn0033159;Alias=FBtr0111095,CG17800-RU;Dbxref=FlyBase_Annotation_IDs:CG17800-RCC;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306773;Name=Dscam-RBI;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBI;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306774;Name=Dscam-RBJ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBJ;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306775;Name=Dscam-RBK;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBK;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306776;Name=Dscam-RBL;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBL;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306777;Name=Dscam-RBM;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBM;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306778;Name=Dscam-RBN;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBN;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306779;Name=Dscam-RBO;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBO;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306780;Name=Dscam-RBP;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBP;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306781;Name=Dscam-RBQ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBQ;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306782;Name=Dscam-RBR;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBR;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306783;Name=Dscam-RBS;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBS;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306784;Name=Dscam-RBT;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBT;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306785;Name=Dscam-RBU;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBU;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306786;Name=Dscam-RBV;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBV;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306787;Name=Dscam-RBW;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBW;score_text=Strongly Supported;score=11
2R    FlyBase    exon    3207059    3208269    .    -    .    ID=FBgn0033159:1;Name=Dscam:1;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_547;Name=Dscam-u3;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_577;Name=Dscam-u3;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_583;Name=Dscam-u3;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_584;Name=Dscam-u3;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_698;Name=Dscam-u3;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_704;Name=Dscam-u3;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    1    ID=CDS_FBgn0033159:1_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    1    ID=CDS_FBgn0033159:1_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    1    ID=CDS_FBgn0033159:1_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    2    ID=CDS_FBgn0033159:1_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    2    ID=CDS_FBgn0033159:1_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    2    ID=CDS_FBgn0033159:1_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3208270    3209087    .    -    .    ID=intron_FBgn0033159:2_FBgn0033159:1;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    0    ID=CDS_FBgn0033159:2_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    0    ID=CDS_FBgn0033159:2_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    0    ID=CDS_FBgn0033159:2_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    2    ID=CDS_FBgn0033159:2_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    2    ID=CDS_FBgn0033159:2_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    2    ID=CDS_FBgn0033159:2_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3209088    3209154    .    -    .    ID=FBgn0033159:2;Name=Dscam:2;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3209155    3210064    .    -    .    ID=intron_FBgn0033159:3_FBgn0033159:2;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    0    ID=CDS_FBgn0033159:3_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    0    ID=CDS_FBgn0033159:3_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    0    ID=CDS_FBgn0033159:3_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    2    ID=CDS_FBgn0033159:3_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    2    ID=CDS_FBgn0033159:3_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    2    ID=CDS_FBgn0033159:3_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3210065    3210223    .    -    .    ID=FBgn0033159:3;Name=Dscam:3;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3210224    3210280    .    -    .    ID=intron_FBgn0033159:4_FBgn0033159:3;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    0    ID=CDS_FBgn0033159:4_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    0    ID=CDS_FBgn0033159:4_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    0    ID=CDS_FBgn0033159:4_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    1    ID=CDS_FBgn0033159:4_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    1    ID=CDS_FBgn0033159:4_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    1    ID=CDS_FBgn0033159:4_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3210281    3210461    .    -    .    ID=FBgn0033159:4;Name=Dscam:4;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3210462    3210526    .    -    .    ID=intron_FBgn0033159:5_FBgn0033159:4;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    0    ID=CDS_FBgn0033159:5_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    0    ID=CDS_FBgn0033159:5_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    0    ID=CDS_FBgn0033159:5_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    2    ID=CDS_FBgn0033159:5_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    2    ID=CDS_FBgn0033159:5_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    2    ID=CDS_FBgn0033159:5_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3210527    3210837    .    -    .    ID=FBgn0033159:5;Name=Dscam:5;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3210838    3211323    .    -    .    ID=intron_FBgn0033159:6_FBgn0033159:5;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    0    ID=CDS_FBgn0033159:6_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    0    ID=CDS_FBgn0033159:6_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    0    ID=CDS_FBgn0033159:6_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    2    ID=CDS_FBgn0033159:6_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    2    ID=CDS_FBgn0033159:6_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    2    ID=CDS_FBgn0033159:6_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3211324    3211473    .    -    .    ID=FBgn0033159:6;Name=Dscam:6;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3211474    3212086    .    -    .    ID=intron_FBgn0033159:7_FBgn0033159:6;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    0    ID=CDS_FBgn0033159:7_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    0    ID=CDS_FBgn0033159:7_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    0    ID=CDS_FBgn0033159:7_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    2    ID=CDS_FBgn0033159:7_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    2    ID=CDS_FBgn0033159:7_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    2    ID=CDS_FBgn0033159:7_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3212087    3212242    .    -    .    ID=FBgn0033159:7;Name=Dscam:7;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3212243    3214121    .    -    .    ID=intron_FBgn0033159:9_FBgn0033159:7;Name=Dscam-in;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063,FBtr0306768,FBtr0306771,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3212243    3213656    .    -    .    ID=intron_FBgn0033159:8_FBgn0033159:7;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059,FBtr0306767,FBtr0306769,FBtr0306770,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3213657    3213869    .    -    0    ID=CDS_FBgn0033159:8_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3213657    3213869    .    -    0    ID=CDS_FBgn0033159:8_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3213657    3213869    .    -    2    ID=CDS_FBgn0033159:8_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3213657    3213869    .    -    .    ID=FBgn0033159:8;Name=Dscam:8;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059,FBtr0306767,FBtr0306769,FBtr0306770,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3213870    3216793    .    -    .    ID=intron_FBgn0033159:10_FBgn0033159:8;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059,FBtr0306767,FBtr0306769,FBtr0306770,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    0    ID=CDS_FBgn0033159:9_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    0    ID=CDS_FBgn0033159:9_704;Name=Dscam-cds;Parent=FBtr0306768,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    2    ID=CDS_FBgn0033159:9_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    2    ID=CDS_FBgn0033159:9_583;Name=Dscam-cds;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    2    ID=CDS_FBgn0033159:9_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3214122    3214289    .    -    .    ID=FBgn0033159:9;Name=Dscam:9;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063,FBtr0306768,FBtr0306771,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3214290    3216793    .    -    .    ID=intron_FBgn0033159:10_FBgn0033159:9;Name=Dscam-in;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063,FBtr0306768,FBtr0306771,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    1    ID=CDS_FBgn0033159:10_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    1    ID=CDS_FBgn0033159:10_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    1    ID=CDS_FBgn0033159:10_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    2    ID=CDS_FBgn0033159:10_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    2    ID=CDS_FBgn0033159:10_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    2    ID=CDS_FBgn0033159:10_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3216794    3216960    .    -    .    ID=FBgn0033159:10;Name=Dscam:10;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3216961    3217017    .    -    .    ID=intron_FBgn0033159:11_FBgn0033159:10;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    1    ID=CDS_FBgn0033159:11_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    1    ID=CDS_FBgn0033159:11_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    1    ID=CDS_FBgn0033159:11_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    2    ID=CDS_FBgn0033159:11_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    2    ID=CDS_FBgn0033159:11_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    2    ID=CDS_FBgn0033159:11_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3217018    3217194    .    -    .    ID=FBgn0033159:11;Name=Dscam:11;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3217195    3217247    .    -    .    ID=intron_FBgn0033159:12_FBgn0033159:11;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    0    ID=CDS_FBgn0033159:12_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    0    ID=CDS_FBgn0033159:12_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    0    ID=CDS_FBgn0033159:12_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    2    ID=CDS_FBgn0033159:12_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    2    ID=CDS_FBgn0033159:12_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    2    ID=CDS_FBgn0033159:12_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3217248    3217356    .    -    .    ID=FBgn0033159:12;Name=Dscam:12;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3217357    3217421    .    -    .    ID=intron_FBgn0033159:13_FBgn0033159:12;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    0    ID=CDS_FBgn0033159:13_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    0    ID=CDS_FBgn0033159:13_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    0    ID=CDS_FBgn0033159:13_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    2    ID=CDS_FBgn0033159:13_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    2    ID=CDS_FBgn0033159:13_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    2    ID=CDS_FBgn0033159:13_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3217422    3218105    .    -    .    ID=FBgn0033159:13;Name=Dscam:13;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3218106    3218163    .    -    .    ID=intron_FBgn0033159:14_FBgn0033159:13;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    0    ID=CDS_FBgn0033159:14_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    0    ID=CDS_FBgn0033159:14_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    0    ID=CDS_FBgn0033159:14_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    2    ID=CDS_FBgn0033159:14_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    2    ID=CDS_FBgn0033159:14_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    2    ID=CDS_FBgn0033159:14_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3218164    3218319    .    -    .    ID=FBgn0033159:14;Name=Dscam:14;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3218320    3218480    .    -    .    ID=intron_FBgn0033159:15_FBgn0033159:14;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    0    ID=CDS_FBgn0033159:15_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    0    ID=CDS_FBgn0033159:15_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    0    ID=CDS_FBgn0033159:15_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    2    ID=CDS_FBgn0033159:15_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    2    ID=CDS_FBgn0033159:15_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    2    ID=CDS_FBgn0033159:15_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3218481    3219707    .    -    .    ID=FBgn0033159:15;Name=Dscam:15;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3219708    3219765    .    -    .    ID=intron_FBgn0033159:16_FBgn0033159:15;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    0    ID=CDS_FBgn0033159:16_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    0    ID=CDS_FBgn0033159:16_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    0    ID=CDS_FBgn0033159:16_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    2    ID=CDS_FBgn0033159:16_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    2    ID=CDS_FBgn0033159:16_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    2    ID=CDS_FBgn0033159:16_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3219766    3219885    .    -    .    ID=FBgn0033159:16;Name=Dscam:16;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3219886    3234896    .    -    .    ID=intron_FBgn0033159:49_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0089020,FBtr0111102,FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3219886    3234511    .    -    .    ID=intron_FBgn0033159:48_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3219886    3234146    .    -    .    ID=intron_FBgn0033159:47_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3219886    3233773    .    -    .    ID=intron_FBgn0033159:46_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    intron    3219886    3233380    .    -    .    ID=intron_FBgn0033159:45_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3219886    3232989    .    -    .    ID=intron_FBgn0033159:44_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3219886    3232565    .    -    .    ID=intron_FBgn0033159:43_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111094,FBtr0111054,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3219886    3232168    .    -    .    ID=intron_FBgn0033159:42_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100,FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3219886    3231783    .    -    .    ID=intron_FBgn0033159:41_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111074,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3219886    3231222    .    -    .    ID=intron_FBgn0033159:40_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3219886    3230836    .    -    .    ID=intron_FBgn0033159:39_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111083,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3219886    3230468    .    -    .    ID=intron_FBgn0033159:38_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    intron    3219886    3230087    .    -    .    ID=intron_FBgn0033159:37_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3219886    3229608    .    -    .    ID=intron_FBgn0033159:36_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    intron    3219886    3229212    .    -    .    ID=intron_FBgn0033159:35_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    intron    3219886    3228832    .    -    .    ID=intron_FBgn0033159:34_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3219886    3228449    .    -    .    ID=intron_FBgn0033159:33_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3219886    3228049    .    -    .    ID=intron_FBgn0033159:32_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3219886    3227659    .    -    .    ID=intron_FBgn0033159:31_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    intron    3219886    3227287    .    -    .    ID=intron_FBgn0033159:30_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3219886    3226907    .    -    .    ID=intron_FBgn0033159:29_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    intron    3219886    3226532    .    -    .    ID=intron_FBgn0033159:28_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3219886    3226145    .    -    .    ID=intron_FBgn0033159:27_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111056,FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3219886    3225754    .    -    .    ID=intron_FBgn0033159:26_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    intron    3219886    3225376    .    -    .    ID=intron_FBgn0033159:25_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    intron    3219886    3224998    .    -    .    ID=intron_FBgn0033159:24_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    intron    3219886    3224593    .    -    .    ID=intron_FBgn0033159:23_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3219886    3224035    .    -    .    ID=intron_FBgn0033159:22_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3219886    3223621    .    -    .    ID=intron_FBgn0033159:21_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3219886    3223220    .    -    .    ID=intron_FBgn0033159:20_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111077,FBtr0111087,FBtr0111089,FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3219886    3222438    .    -    .    ID=intron_FBgn0033159:19_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111075,FBtr0111076,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3219886    3221892    .    -    .    ID=intron_FBgn0033159:18_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3219886    3221191    .    -    .    ID=intron_FBgn0033159:17_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3221192    3221485    .    -    2    ID=CDS_FBgn0033159:17_704;Name=Dscam-cds;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    exon    3221192    3221485    .    -    .    ID=FBgn0033159:17;Name=Dscam:17;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3221486    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:17;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3221893    3222171    .    -    2    ID=CDS_FBgn0033159:18_583;Name=Dscam-cds;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    exon    3221893    3222171    .    -    .    ID=FBgn0033159:18;Name=Dscam:18;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3222172    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:18;Name=Dscam-in;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    CDS    3222439    3222723    .    -    2    ID=CDS_FBgn0033159:19_583;Name=Dscam-cds;Parent=FBtr0111075,FBtr0111076;parent_type=mRNA
2R    FlyBase    CDS    3222439    3222723    .    -    2    ID=CDS_FBgn0033159:19_704;Name=Dscam-cds;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    exon    3222439    3222723    .    -    .    ID=FBgn0033159:19;Name=Dscam:19;Parent=FBtr0111075,FBtr0111076,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3222724    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:19;Name=Dscam-in;Parent=FBtr0111075,FBtr0111076,FBtr0306777;parent_type=mRNA
2R    FlyBase    CDS    3223221    3223508    .    -    2    ID=CDS_FBgn0033159:20_583;Name=Dscam-cds;Parent=FBtr0111077,FBtr0111087,FBtr0111089;parent_type=mRNA
2R    FlyBase    CDS    3223221    3223508    .    -    2    ID=CDS_FBgn0033159:20_704;Name=Dscam-cds;Parent=FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3223221    3223508    .    -    .    ID=FBgn0033159:20;Name=Dscam:20;Parent=FBtr0111077,FBtr0111087,FBtr0111089,FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3223509    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:20;Name=Dscam-in;Parent=FBtr0111077,FBtr0111087,FBtr0111089,FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3223622    3223915    .    -    2    ID=CDS_FBgn0033159:21_704;Name=Dscam-cds;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    exon    3223622    3223915    .    -    .    ID=FBgn0033159:21;Name=Dscam:21;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3223916    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:21;Name=Dscam-in;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3224036    3224323    .    -    2    ID=CDS_FBgn0033159:22_704;Name=Dscam-cds;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    exon    3224036    3224323    .    -    .    ID=FBgn0033159:22;Name=Dscam:22;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3224324    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:22;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    CDS    3224594    3224881    .    -    2    ID=CDS_FBgn0033159:23_704;Name=Dscam-cds;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    exon    3224594    3224881    .    -    .    ID=FBgn0033159:23;Name=Dscam:23;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3224882    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:23;Name=Dscam-in;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    CDS    3224999    3225286    .    -    2    ID=CDS_FBgn0033159:24_583;Name=Dscam-cds;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    exon    3224999    3225286    .    -    .    ID=FBgn0033159:24;Name=Dscam:24;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    intron    3225287    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:24;Name=Dscam-in;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3225377    3225664    .    -    2    ID=CDS_FBgn0033159:25_583;Name=Dscam-cds;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    exon    3225377    3225664    .    -    .    ID=FBgn0033159:25;Name=Dscam:25;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    intron    3225665    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:25;Name=Dscam-in;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    CDS    3225755    3226042    .    -    2    ID=CDS_FBgn0033159:26_583;Name=Dscam-cds;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    exon    3225755    3226042    .    -    .    ID=FBgn0033159:26;Name=Dscam:26;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    intron    3226043    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:26;Name=Dscam-in;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    CDS    3226146    3226433    .    -    2    ID=CDS_FBgn0033159:27_583;Name=Dscam-cds;Parent=FBtr0111056;parent_type=mRNA
2R    FlyBase    CDS    3226146    3226433    .    -    2    ID=CDS_FBgn0033159:27_704;Name=Dscam-cds;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    exon    3226146    3226433    .    -    .    ID=FBgn0033159:27;Name=Dscam:27;Parent=FBtr0111056,FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3226434    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:27;Name=Dscam-in;Parent=FBtr0111056,FBtr0306779;parent_type=mRNA
2R    FlyBase    CDS    3226533    3226820    .    -    2    ID=CDS_FBgn0033159:28_704;Name=Dscam-cds;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    exon    3226533    3226820    .    -    .    ID=FBgn0033159:28;Name=Dscam:28;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3226821    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:28;Name=Dscam-in;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    CDS    3226908    3227195    .    -    2    ID=CDS_FBgn0033159:29_583;Name=Dscam-cds;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    exon    3226908    3227195    .    -    .    ID=FBgn0033159:29;Name=Dscam:29;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    intron    3227196    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:29;Name=Dscam-in;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    CDS    3227288    3227575    .    -    2    ID=CDS_FBgn0033159:30_583;Name=Dscam-cds;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    exon    3227288    3227575    .    -    .    ID=FBgn0033159:30;Name=Dscam:30;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3227576    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:30;Name=Dscam-in;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    CDS    3227660    3227947    .    -    2    ID=CDS_FBgn0033159:31_583;Name=Dscam-cds;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    exon    3227660    3227947    .    -    .    ID=FBgn0033159:31;Name=Dscam:31;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    intron    3227948    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:31;Name=Dscam-in;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    CDS    3228050    3228337    .    -    2    ID=CDS_FBgn0033159:32_583;Name=Dscam-cds;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    exon    3228050    3228337    .    -    .    ID=FBgn0033159:32;Name=Dscam:32;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3228338    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:32;Name=Dscam-in;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    CDS    3228450    3228737    .    -    2    ID=CDS_FBgn0033159:33_583;Name=Dscam-cds;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    exon    3228450    3228737    .    -    .    ID=FBgn0033159:33;Name=Dscam:33;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3228738    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:33;Name=Dscam-in;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    CDS    3228833    3229123    .    -    2    ID=CDS_FBgn0033159:34_583;Name=Dscam-cds;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    exon    3228833    3229123    .    -    .    ID=FBgn0033159:34;Name=Dscam:34;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3229124    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:34;Name=Dscam-in;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    CDS    3229213    3229500    .    -    2    ID=CDS_FBgn0033159:35_583;Name=Dscam-cds;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    exon    3229213    3229500    .    -    .    ID=FBgn0033159:35;Name=Dscam:35;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    intron    3229501    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:35;Name=Dscam-in;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    CDS    3229609    3229896    .    -    2    ID=CDS_FBgn0033159:36_583;Name=Dscam-cds;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    exon    3229609    3229896    .    -    .    ID=FBgn0033159:36;Name=Dscam:36;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    intron    3229897    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:36;Name=Dscam-in;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    CDS    3230088    3230386    .    -    2    ID=CDS_FBgn0033159:37_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3230088    3230386    .    -    2    ID=CDS_FBgn0033159:37_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3230088    3230386    .    -    2    ID=CDS_FBgn0033159:37_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772;parent_type=mRNA
2R    FlyBase    exon    3230088    3230386    .    -    .    ID=FBgn0033159:37;Name=Dscam:37;Parent=FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3230387    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:37;Name=Dscam-in;Parent=FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772;parent_type=mRNA
2R    FlyBase    CDS    3230469    3230753    .    -    2    ID=CDS_FBgn0033159:38_583;Name=Dscam-cds;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    exon    3230469    3230753    .    -    .    ID=FBgn0033159:38;Name=Dscam:38;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    intron    3230754    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:38;Name=Dscam-in;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    CDS    3230837    3231127    .    -    2    ID=CDS_FBgn0033159:39_583;Name=Dscam-cds;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    CDS    3230837    3231127    .    -    2    ID=CDS_FBgn0033159:39_704;Name=Dscam-cds;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    exon    3230837    3231127    .    -    .    ID=FBgn0033159:39;Name=Dscam:39;Parent=FBtr0111083,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3231128    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:39;Name=Dscam-in;Parent=FBtr0111083,FBtr0306780;parent_type=mRNA
2R    FlyBase    CDS    3231223    3231513    .    -    2    ID=CDS_FBgn0033159:40_704;Name=Dscam-cds;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    exon    3231223    3231513    .    -    .    ID=FBgn0033159:40;Name=Dscam:40;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3231514    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:40;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    CDS    3231784    3232074    .    -    2    ID=CDS_FBgn0033159:41_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3231784    3232074    .    -    2    ID=CDS_FBgn0033159:41_583;Name=Dscam-cds;Parent=FBtr0111074,FBtr0111061;parent_type=mRNA
2R    FlyBase    exon    3231784    3232074    .    -    .    ID=FBgn0033159:41;Name=Dscam:41;Parent=FBtr0111074,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3232075    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:41;Name=Dscam-in;Parent=FBtr0111074,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0111061;parent_type=mRNA
2R    FlyBase    CDS    3232169    3232459    .    -    2    ID=CDS_FBgn0033159:42_583;Name=Dscam-cds;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100;parent_type=mRNA
2R    FlyBase    CDS    3232169    3232459    .    -    2    ID=CDS_FBgn0033159:42_704;Name=Dscam-cds;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    exon    3232169    3232459    .    -    .    ID=FBgn0033159:42;Name=Dscam:42;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100,FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3232460    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:42;Name=Dscam-in;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100,FBtr0306783;parent_type=mRNA
2R    FlyBase    CDS    3232566    3232856    .    -    2    ID=CDS_FBgn0033159:43_583;Name=Dscam-cds;Parent=FBtr0111094,FBtr0111054;parent_type=mRNA
2R    FlyBase    CDS    3232566    3232856    .    -    2    ID=CDS_FBgn0033159:43_704;Name=Dscam-cds;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    exon    3232566    3232856    .    -    .    ID=FBgn0033159:43;Name=Dscam:43;Parent=FBtr0111094,FBtr0111054,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3232857    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:43;Name=Dscam-in;Parent=FBtr0111094,FBtr0111054,FBtr0306782;parent_type=mRNA
2R    FlyBase    CDS    3232990    3233280    .    -    2    ID=CDS_FBgn0033159:44_583;Name=Dscam-cds;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    exon    3232990    3233280    .    -    .    ID=FBgn0033159:44;Name=Dscam:44;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3233281    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:44;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    CDS    3233381    3233671    .    -    2    ID=CDS_FBgn0033159:45_583;Name=Dscam-cds;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3233381    3233671    .    -    .    ID=FBgn0033159:45;Name=Dscam:45;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3233672    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:45;Name=Dscam-in;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    CDS    3233774    3234061    .    -    2    ID=CDS_FBgn0033159:46_583;Name=Dscam-cds;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    exon    3233774    3234061    .    -    .    ID=FBgn0033159:46;Name=Dscam:46;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    intron    3234062    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:46;Name=Dscam-in;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    CDS    3234147    3234434    .    -    2    ID=CDS_FBgn0033159:47_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    exon    3234147    3234434    .    -    .    ID=FBgn0033159:47;Name=Dscam:47;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3234435    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:47;Name=Dscam-in;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    CDS    3234512    3234799    .    -    2    ID=CDS_FBgn0033159:48_583;Name=Dscam-cds;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    exon    3234512    3234799    .    -    .    ID=FBgn0033159:48;Name=Dscam:48;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3234800    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:48;Name=Dscam-in;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    CDS    3234897    3235202    .    -    2    ID=CDS_FBgn0033159:49_583;Name=Dscam-cds;Parent=FBtr0111102;parent_type=mRNA
2R    FlyBase    CDS    3234897    3235202    .    -    2    ID=CDS_FBgn0033159:49_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3234897    3235202    .    -    2    ID=CDS_FBgn0033159:49_704;Name=Dscam-cds;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    exon    3234897    3235202    .    -    .    ID=FBgn0033159:49;Name=Dscam:49;Parent=FBtr0089020,FBtr0111102,FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3235203    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:49;Name=Dscam-in;Parent=FBtr0089020,FBtr0111102,FBtr0306781;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3235354    3235515    .    -    .    ID=FBgn0033159:50;Name=Dscam:50;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3235516    3235635    .    -    .    ID=intron_FBgn0033159:51_FBgn0033159:50;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3235636    3236480    .    -    .    ID=FBgn0033159:51;Name=Dscam:51;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3236481    3248030    .    -    .    ID=intron_FBgn0033159:99_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    intron    3236481    3247795    .    -    .    ID=intron_FBgn0033159:98_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111081,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3236481    3247586    .    -    .    ID=intron_FBgn0033159:97_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111092,FBtr0306770;parent_type=mRNA
2R    FlyBase    intron    3236481    3247379    .    -    .    ID=intron_FBgn0033159:96_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111063,FBtr0306767;parent_type=mRNA
2R    FlyBase    intron    3236481    3247171    .    -    .    ID=intron_FBgn0033159:95_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111082,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3236481    3246946    .    -    .    ID=intron_FBgn0033159:94_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    intron    3236481    3246726    .    -    .    ID=intron_FBgn0033159:93_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111072,FBtr0111058,FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3236481    3246520    .    -    .    ID=intron_FBgn0033159:92_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    intron    3236481    3246311    .    -    .    ID=intron_FBgn0033159:91_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    intron    3236481    3246108    .    -    .    ID=intron_FBgn0033159:90_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    intron    3236481    3245933    .    -    .    ID=intron_FBgn0033159:89_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3236481    3245720    .    -    .    ID=intron_FBgn0033159:88_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    intron    3236481    3245507    .    -    .    ID=intron_FBgn0033159:87_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3236481    3245293    .    -    .    ID=intron_FBgn0033159:86_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    intron    3236481    3245095    .    -    .    ID=intron_FBgn0033159:85_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111073,FBtr0306771;parent_type=mRNA
2R    FlyBase    intron    3236481    3244904    .    -    .    ID=intron_FBgn0033159:84_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    intron    3236481    3244687    .    -    .    ID=intron_FBgn0033159:83_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3236481    3244477    .    -    .    ID=intron_FBgn0033159:82_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3236481    3244249    .    -    .    ID=intron_FBgn0033159:81_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3236481    3244031    .    -    .    ID=intron_FBgn0033159:80_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3236481    3243821    .    -    .    ID=intron_FBgn0033159:79_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3236481    3243611    .    -    .    ID=intron_FBgn0033159:78_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3236481    3243405    .    -    .    ID=intron_FBgn0033159:77_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    intron    3236481    3243190    .    -    .    ID=intron_FBgn0033159:76_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111102,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3236481    3242795    .    -    .    ID=intron_FBgn0033159:75_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    intron    3236481    3242591    .    -    .    ID=intron_FBgn0033159:74_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111050,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3236481    3242386    .    -    .    ID=intron_FBgn0033159:73_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    intron    3236481    3242190    .    -    .    ID=intron_FBgn0033159:72_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    intron    3236481    3241981    .    -    .    ID=intron_FBgn0033159:71_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3236481    3241781    .    -    .    ID=intron_FBgn0033159:70_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3236481    3241561    .    -    .    ID=intron_FBgn0033159:69_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3236481    3241320    .    -    .    ID=intron_FBgn0033159:68_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    intron    3236481    3241070    .    -    .    ID=intron_FBgn0033159:67_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    intron    3236481    3240749    .    -    .    ID=intron_FBgn0033159:66_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    intron    3236481    3240403    .    -    .    ID=intron_FBgn0033159:65_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3236481    3239420    .    -    .    ID=intron_FBgn0033159:64_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3236481    3239203    .    -    .    ID=intron_FBgn0033159:63_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3236481    3238965    .    -    .    ID=intron_FBgn0033159:62_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    intron    3236481    3238678    .    -    .    ID=intron_FBgn0033159:61_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0089020,FBtr0306769;parent_type=mRNA
2R    FlyBase    intron    3236481    3238462    .    -    .    ID=intron_FBgn0033159:60_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3236481    3238286    .    -    .    ID=intron_FBgn0033159:59_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3236481    3238085    .    -    .    ID=intron_FBgn0033159:58_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3236481    3237878    .    -    .    ID=intron_FBgn0033159:57_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3236481    3237662    .    -    .    ID=intron_FBgn0033159:56_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3236481    3237489    .    -    .    ID=intron_FBgn0033159:55_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3236481    3237292    .    -    .    ID=intron_FBgn0033159:54_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    intron    3236481    3237105    .    -    .    ID=intron_FBgn0033159:53_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3236481    3236897    .    -    .    ID=intron_FBgn0033159:52_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    CDS    3236898    3237018    .    -    2    ID=CDS_FBgn0033159:52_583;Name=Dscam-cds;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    exon    3236898    3237018    .    -    .    ID=FBgn0033159:52;Name=Dscam:52;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    intron    3237019    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:52;Name=Dscam-in;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    CDS    3237106    3237226    .    -    2    ID=CDS_FBgn0033159:53_704;Name=Dscam-cds;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    exon    3237106    3237226    .    -    .    ID=FBgn0033159:53;Name=Dscam:53;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3237227    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:53;Name=Dscam-in;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    CDS    3237293    3237416    .    -    2    ID=CDS_FBgn0033159:54_583;Name=Dscam-cds;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    exon    3237293    3237416    .    -    .    ID=FBgn0033159:54;Name=Dscam:54;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    intron    3237417    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:54;Name=Dscam-in;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    CDS    3237490    3237613    .    -    2    ID=CDS_FBgn0033159:55_704;Name=Dscam-cds;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    exon    3237490    3237613    .    -    .    ID=FBgn0033159:55;Name=Dscam:55;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3237614    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:55;Name=Dscam-in;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    CDS    3237663    3237786    .    -    2    ID=CDS_FBgn0033159:56_704;Name=Dscam-cds;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    exon    3237663    3237786    .    -    .    ID=FBgn0033159:56;Name=Dscam:56;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3237787    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:56;Name=Dscam-in;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    CDS    3237879    3238002    .    -    2    ID=CDS_FBgn0033159:57_583;Name=Dscam-cds;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    exon    3237879    3238002    .    -    .    ID=FBgn0033159:57;Name=Dscam:57;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3238003    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:57;Name=Dscam-in;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    CDS    3238086    3238209    .    -    2    ID=CDS_FBgn0033159:58_583;Name=Dscam-cds;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    exon    3238086    3238209    .    -    .    ID=FBgn0033159:58;Name=Dscam:58;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3238210    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:58;Name=Dscam-in;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    CDS    3238287    3238410    .    -    2    ID=CDS_FBgn0033159:59_704;Name=Dscam-cds;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    exon    3238287    3238410    .    -    .    ID=FBgn0033159:59;Name=Dscam:59;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3238411    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:59;Name=Dscam-in;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    CDS    3238463    3238586    .    -    2    ID=CDS_FBgn0033159:60_704;Name=Dscam-cds;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    exon    3238463    3238586    .    -    .    ID=FBgn0033159:60;Name=Dscam:60;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3238587    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:60;Name=Dscam-in;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    CDS    3238679    3238802    .    -    2    ID=CDS_FBgn0033159:61_547;Name=Dscam-cds;Parent=FBtr0306769;parent_type=mRNA
2R    FlyBase    CDS    3238679    3238802    .    -    2    ID=CDS_FBgn0033159:61_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3238679    3238802    .    -    .    ID=FBgn0033159:61;Name=Dscam:61;Parent=FBtr0089020,FBtr0306769;parent_type=mRNA
2R    FlyBase    intron    3238803    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:61;Name=Dscam-in;Parent=FBtr0089020,FBtr0306769;parent_type=mRNA
2R    FlyBase    CDS    3238966    3239089    .    -    2    ID=CDS_FBgn0033159:62_704;Name=Dscam-cds;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    exon    3238966    3239089    .    -    .    ID=FBgn0033159:62;Name=Dscam:62;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    intron    3239090    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:62;Name=Dscam-in;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    CDS    3239204    3239327    .    -    2    ID=CDS_FBgn0033159:63_583;Name=Dscam-cds;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    exon    3239204    3239327    .    -    .    ID=FBgn0033159:63;Name=Dscam:63;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3239328    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:63;Name=Dscam-in;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    CDS    3239421    3239544    .    -    2    ID=CDS_FBgn0033159:64_583;Name=Dscam-cds;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    exon    3239421    3239544    .    -    .    ID=FBgn0033159:64;Name=Dscam:64;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3239545    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:64;Name=Dscam-in;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    CDS    3240404    3240527    .    -    2    ID=CDS_FBgn0033159:65_583;Name=Dscam-cds;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    exon    3240404    3240527    .    -    .    ID=FBgn0033159:65;Name=Dscam:65;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3240528    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:65;Name=Dscam-in;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    CDS    3240750    3240870    .    -    2    ID=CDS_FBgn0033159:66_583;Name=Dscam-cds;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    exon    3240750    3240870    .    -    .    ID=FBgn0033159:66;Name=Dscam:66;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    intron    3240871    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:66;Name=Dscam-in;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    CDS    3241071    3241197    .    -    2    ID=CDS_FBgn0033159:67_583;Name=Dscam-cds;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    exon    3241071    3241197    .    -    .    ID=FBgn0033159:67;Name=Dscam:67;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    intron    3241198    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:67;Name=Dscam-in;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    CDS    3241321    3241441    .    -    2    ID=CDS_FBgn0033159:68_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    exon    3241321    3241441    .    -    .    ID=FBgn0033159:68;Name=Dscam:68;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    intron    3241442    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:68;Name=Dscam-in;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3241562    3241682    .    -    2    ID=CDS_FBgn0033159:69_704;Name=Dscam-cds;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    exon    3241562    3241682    .    -    .    ID=FBgn0033159:69;Name=Dscam:69;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3241683    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:69;Name=Dscam-in;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    CDS    3241782    3241905    .    -    2    ID=CDS_FBgn0033159:70_583;Name=Dscam-cds;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    exon    3241782    3241905    .    -    .    ID=FBgn0033159:70;Name=Dscam:70;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3241906    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:70;Name=Dscam-in;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    CDS    3241982    3242105    .    -    2    ID=CDS_FBgn0033159:71_704;Name=Dscam-cds;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    exon    3241982    3242105    .    -    .    ID=FBgn0033159:71;Name=Dscam:71;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3242106    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:71;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    CDS    3242191    3242314    .    -    2    ID=CDS_FBgn0033159:72_704;Name=Dscam-cds;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    exon    3242191    3242314    .    -    .    ID=FBgn0033159:72;Name=Dscam:72;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    intron    3242315    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:72;Name=Dscam-in;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    CDS    3242387    3242510    .    -    2    ID=CDS_FBgn0033159:73_583;Name=Dscam-cds;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    exon    3242387    3242510    .    -    .    ID=FBgn0033159:73;Name=Dscam:73;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    intron    3242511    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:73;Name=Dscam-in;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    CDS    3242592    3242715    .    -    2    ID=CDS_FBgn0033159:74_583;Name=Dscam-cds;Parent=FBtr0111050;parent_type=mRNA
2R    FlyBase    CDS    3242592    3242715    .    -    2    ID=CDS_FBgn0033159:74_704;Name=Dscam-cds;Parent=FBtr0306786;parent_type=mRNA
2R    FlyBase    exon    3242592    3242715    .    -    .    ID=FBgn0033159:74;Name=Dscam:74;Parent=FBtr0111050,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3242716    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:74;Name=Dscam-in;Parent=FBtr0111050,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3242796    3242919    .    -    2    ID=CDS_FBgn0033159:75_583;Name=Dscam-cds;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    exon    3242796    3242919    .    -    .    ID=FBgn0033159:75;Name=Dscam:75;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    intron    3242920    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:75;Name=Dscam-in;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    CDS    3243191    3243314    .    -    2    ID=CDS_FBgn0033159:76_583;Name=Dscam-cds;Parent=FBtr0111102;parent_type=mRNA
2R    FlyBase    CDS    3243191    3243314    .    -    2    ID=CDS_FBgn0033159:76_704;Name=Dscam-cds;Parent=FBtr0306772;parent_type=mRNA
2R    FlyBase    exon    3243191    3243314    .    -    .    ID=FBgn0033159:76;Name=Dscam:76;Parent=FBtr0111102,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3243315    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:76;Name=Dscam-in;Parent=FBtr0111102,FBtr0306772;parent_type=mRNA
2R    FlyBase    CDS    3243406    3243529    .    -    2    ID=CDS_FBgn0033159:77_583;Name=Dscam-cds;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    exon    3243406    3243529    .    -    .    ID=FBgn0033159:77;Name=Dscam:77;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    intron    3243530    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:77;Name=Dscam-in;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    CDS    3243612    3243735    .    -    2    ID=CDS_FBgn0033159:78_583;Name=Dscam-cds;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    exon    3243612    3243735    .    -    .    ID=FBgn0033159:78;Name=Dscam:78;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3243736    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:78;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    CDS    3243822    3243945    .    -    2    ID=CDS_FBgn0033159:79_583;Name=Dscam-cds;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    exon    3243822    3243945    .    -    .    ID=FBgn0033159:79;Name=Dscam:79;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3243946    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:79;Name=Dscam-in;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    CDS    3244032    3244155    .    -    2    ID=CDS_FBgn0033159:80_583;Name=Dscam-cds;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    exon    3244032    3244155    .    -    .    ID=FBgn0033159:80;Name=Dscam:80;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3244156    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:80;Name=Dscam-in;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    CDS    3244250    3244373    .    -    2    ID=CDS_FBgn0033159:81_583;Name=Dscam-cds;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    exon    3244250    3244373    .    -    .    ID=FBgn0033159:81;Name=Dscam:81;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3244374    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:81;Name=Dscam-in;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    CDS    3244478    3244601    .    -    2    ID=CDS_FBgn0033159:82_583;Name=Dscam-cds;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3244478    3244601    .    -    .    ID=FBgn0033159:82;Name=Dscam:82;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3244602    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:82;Name=Dscam-in;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    CDS    3244688    3244811    .    -    2    ID=CDS_FBgn0033159:83_704;Name=Dscam-cds;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    exon    3244688    3244811    .    -    .    ID=FBgn0033159:83;Name=Dscam:83;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3244812    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:83;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3244905    3245028    .    -    2    ID=CDS_FBgn0033159:84_583;Name=Dscam-cds;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    exon    3244905    3245028    .    -    .    ID=FBgn0033159:84;Name=Dscam:84;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    intron    3245029    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:84;Name=Dscam-in;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    CDS    3245096    3245219    .    -    2    ID=CDS_FBgn0033159:85_583;Name=Dscam-cds;Parent=FBtr0111073;parent_type=mRNA
2R    FlyBase    CDS    3245096    3245219    .    -    2    ID=CDS_FBgn0033159:85_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    exon    3245096    3245219    .    -    .    ID=FBgn0033159:85;Name=Dscam:85;Parent=FBtr0111073,FBtr0306771;parent_type=mRNA
2R    FlyBase    intron    3245220    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:85;Name=Dscam-in;Parent=FBtr0111073,FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3245294    3245420    .    -    2    ID=CDS_FBgn0033159:86_583;Name=Dscam-cds;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    exon    3245294    3245420    .    -    .    ID=FBgn0033159:86;Name=Dscam:86;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    intron    3245421    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:86;Name=Dscam-in;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    CDS    3245508    3245634    .    -    2    ID=CDS_FBgn0033159:87_704;Name=Dscam-cds;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    exon    3245508    3245634    .    -    .    ID=FBgn0033159:87;Name=Dscam:87;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3245635    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:87;Name=Dscam-in;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    CDS    3245721    3245844    .    -    2    ID=CDS_FBgn0033159:88_583;Name=Dscam-cds;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    exon    3245721    3245844    .    -    .    ID=FBgn0033159:88;Name=Dscam:88;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    intron    3245845    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:88;Name=Dscam-in;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    CDS    3245934    3246051    .    -    2    ID=CDS_FBgn0033159:89_704;Name=Dscam-cds;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    exon    3245934    3246051    .    -    .    ID=FBgn0033159:89;Name=Dscam:89;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3246052    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:89;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    CDS    3246109    3246232    .    -    2    ID=CDS_FBgn0033159:90_583;Name=Dscam-cds;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    exon    3246109    3246232    .    -    .    ID=FBgn0033159:90;Name=Dscam:90;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    intron    3246233    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:90;Name=Dscam-in;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    CDS    3246312    3246435    .    -    2    ID=CDS_FBgn0033159:91_583;Name=Dscam-cds;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    exon    3246312    3246435    .    -    .    ID=FBgn0033159:91;Name=Dscam:91;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    intron    3246436    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:91;Name=Dscam-in;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    CDS    3246521    3246644    .    -    2    ID=CDS_FBgn0033159:92_583;Name=Dscam-cds;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    exon    3246521    3246644    .    -    .    ID=FBgn0033159:92;Name=Dscam:92;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    intron    3246645    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:92;Name=Dscam-in;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    CDS    3246727    3246850    .    -    2    ID=CDS_FBgn0033159:93_583;Name=Dscam-cds;Parent=FBtr0111072,FBtr0111058;parent_type=mRNA
2R    FlyBase    CDS    3246727    3246850    .    -    2    ID=CDS_FBgn0033159:93_704;Name=Dscam-cds;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    exon    3246727    3246850    .    -    .    ID=FBgn0033159:93;Name=Dscam:93;Parent=FBtr0111072,FBtr0111058,FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3246851    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:93;Name=Dscam-in;Parent=FBtr0111072,FBtr0111058,FBtr0306784;parent_type=mRNA
2R    FlyBase    CDS    3246947    3247070    .    -    2    ID=CDS_FBgn0033159:94_583;Name=Dscam-cds;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    exon    3246947    3247070    .    -    .    ID=FBgn0033159:94;Name=Dscam:94;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    intron    3247071    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:94;Name=Dscam-in;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    CDS    3247172    3247295    .    -    2    ID=CDS_FBgn0033159:95_583;Name=Dscam-cds;Parent=FBtr0111082;parent_type=mRNA
2R    FlyBase    CDS    3247172    3247295    .    -    2    ID=CDS_FBgn0033159:95_704;Name=Dscam-cds;Parent=FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3247172    3247295    .    -    .    ID=FBgn0033159:95;Name=Dscam:95;Parent=FBtr0111082,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3247296    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:95;Name=Dscam-in;Parent=FBtr0111082,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3247380    3247512    .    -    2    ID=CDS_FBgn0033159:96_583;Name=Dscam-cds;Parent=FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3247380    3247512    .    -    2    ID=CDS_FBgn0033159:96_704;Name=Dscam-cds;Parent=FBtr0306767;parent_type=mRNA
2R    FlyBase    exon    3247380    3247512    .    -    .    ID=FBgn0033159:96;Name=Dscam:96;Parent=FBtr0111063,FBtr0306767;parent_type=mRNA
2R    FlyBase    intron    3247513    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:96;Name=Dscam-in;Parent=FBtr0111063,FBtr0306767;parent_type=mRNA
2R    FlyBase    CDS    3247587    3247710    .    -    2    ID=CDS_FBgn0033159:97_547;Name=Dscam-cds;Parent=FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3247587    3247710    .    -    2    ID=CDS_FBgn0033159:97_583;Name=Dscam-cds;Parent=FBtr0111092;parent_type=mRNA
2R    FlyBase    exon    3247587    3247710    .    -    .    ID=FBgn0033159:97;Name=Dscam:97;Parent=FBtr0111092,FBtr0306770;parent_type=mRNA
2R    FlyBase    intron    3247711    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:97;Name=Dscam-in;Parent=FBtr0111092,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3247796    3247919    .    -    2    ID=CDS_FBgn0033159:98_583;Name=Dscam-cds;Parent=FBtr0111081;parent_type=mRNA
2R    FlyBase    CDS    3247796    3247919    .    -    2    ID=CDS_FBgn0033159:98_704;Name=Dscam-cds;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    exon    3247796    3247919    .    -    .    ID=FBgn0033159:98;Name=Dscam:98;Parent=FBtr0111081,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3247920    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:98;Name=Dscam-in;Parent=FBtr0111081,FBtr0306785;parent_type=mRNA
2R    FlyBase    CDS    3248031    3248154    .    -    2    ID=CDS_FBgn0033159:99_583;Name=Dscam-cds;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    exon    3248031    3248154    .    -    .    ID=FBgn0033159:99;Name=Dscam:99;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    intron    3248155    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:99;Name=Dscam-in;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3249223    3249372    .    -    .    ID=FBgn0033159:100;Name=Dscam:100;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3249373    3254309    .    -    .    ID=intron_FBgn0033159:112_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111078,FBtr0089016,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3249373    3253970    .    -    .    ID=intron_FBgn0033159:111_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0089018,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3249373    3253447    .    -    .    ID=intron_FBgn0033159:110_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0089020,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3249373    3252816    .    -    .    ID=intron_FBgn0033159:109_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111097,FBtr0111098,FBtr0111099,FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3249373    3252522    .    -    .    ID=intron_FBgn0033159:108_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3249373    3252236    .    -    .    ID=intron_FBgn0033159:107_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3249373    3251960    .    -    .    ID=intron_FBgn0033159:106_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3249373    3251682    .    -    .    ID=intron_FBgn0033159:105_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0089019,FBtr0111063,FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3249373    3251385    .    -    .    ID=intron_FBgn0033159:104_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111067,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3249373    3251099    .    -    .    ID=intron_FBgn0033159:103_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068,FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3249373    3250836    .    -    .    ID=intron_FBgn0033159:102_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111075,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3249373    3250528    .    -    .    ID=intron_FBgn0033159:101_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073,FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3250529    3250699    .    -    2    ID=CDS_FBgn0033159:101_583;Name=Dscam-cds;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073;parent_type=mRNA
2R    FlyBase    CDS    3250529    3250699    .    -    2    ID=CDS_FBgn0033159:101_704;Name=Dscam-cds;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    exon    3250529    3250699    .    -    .    ID=FBgn0033159:101;Name=Dscam:101;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073,FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3250700    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:101;Name=Dscam-in;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073,FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3250837    3250995    .    -    2    ID=CDS_FBgn0033159:102_583;Name=Dscam-cds;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    CDS    3250837    3250995    .    -    2    ID=CDS_FBgn0033159:102_704;Name=Dscam-cds;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    exon    3250837    3250995    .    -    .    ID=FBgn0033159:102;Name=Dscam:102;Parent=FBtr0111075,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3250996    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:102;Name=Dscam-in;Parent=FBtr0111075,FBtr0306777;parent_type=mRNA
2R    FlyBase    CDS    3251100    3251270    .    -    2    ID=CDS_FBgn0033159:103_583;Name=Dscam-cds;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068;parent_type=mRNA
2R    FlyBase    CDS    3251100    3251270    .    -    2    ID=CDS_FBgn0033159:103_704;Name=Dscam-cds;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    exon    3251100    3251270    .    -    .    ID=FBgn0033159:103;Name=Dscam:103;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068,FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3251271    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:103;Name=Dscam-in;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068,FBtr0306776;parent_type=mRNA
2R    FlyBase    CDS    3251386    3251553    .    -    2    ID=CDS_FBgn0033159:104_583;Name=Dscam-cds;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    CDS    3251386    3251553    .    -    2    ID=CDS_FBgn0033159:104_704;Name=Dscam-cds;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    exon    3251386    3251553    .    -    .    ID=FBgn0033159:104;Name=Dscam:104;Parent=FBtr0111067,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3251554    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:104;Name=Dscam-in;Parent=FBtr0111067,FBtr0306780;parent_type=mRNA
2R    FlyBase    CDS    3251683    3251853    .    -    2    ID=CDS_FBgn0033159:105_577;Name=Dscam-cds;Parent=FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3251683    3251853    .    -    2    ID=CDS_FBgn0033159:105_583;Name=Dscam-cds;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3251683    3251853    .    -    2    ID=CDS_FBgn0033159:105_704;Name=Dscam-cds;Parent=FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    exon    3251683    3251853    .    -    .    ID=FBgn0033159:105;Name=Dscam:105;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0089019,FBtr0111063,FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3251854    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:105;Name=Dscam-in;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0089019,FBtr0111063,FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    CDS    3251961    3252122    .    -    2    ID=CDS_FBgn0033159:106_583;Name=Dscam-cds;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3251961    3252122    .    -    .    ID=FBgn0033159:106;Name=Dscam:106;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3252123    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:106;Name=Dscam-in;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    CDS    3252237    3252398    .    -    2    ID=CDS_FBgn0033159:107_583;Name=Dscam-cds;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    exon    3252237    3252398    .    -    .    ID=FBgn0033159:107;Name=Dscam:107;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3252399    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:107;Name=Dscam-in;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    CDS    3252523    3252684    .    -    2    ID=CDS_FBgn0033159:108_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050;parent_type=mRNA
2R    FlyBase    CDS    3252523    3252684    .    -    2    ID=CDS_FBgn0033159:108_704;Name=Dscam-cds;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    exon    3252523    3252684    .    -    .    ID=FBgn0033159:108;Name=Dscam:108;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3252685    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:108;Name=Dscam-in;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3252817    3252987    .    -    2    ID=CDS_FBgn0033159:109_583;Name=Dscam-cds;Parent=FBtr0111097,FBtr0111098,FBtr0111099;parent_type=mRNA
2R    FlyBase    CDS    3252817    3252987    .    -    2    ID=CDS_FBgn0033159:109_704;Name=Dscam-cds;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    exon    3252817    3252987    .    -    .    ID=FBgn0033159:109;Name=Dscam:109;Parent=FBtr0111097,FBtr0111098,FBtr0111099,FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3252988    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:109;Name=Dscam-in;Parent=FBtr0111097,FBtr0111098,FBtr0111099,FBtr0306775;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_547;Name=Dscam-cds;Parent=FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_583;Name=Dscam-cds;Parent=FBtr0111092,FBtr0111094,FBtr0111096;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_704;Name=Dscam-cds;Parent=FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    exon    3253448    3253609    .    -    .    ID=FBgn0033159:110;Name=Dscam:110;Parent=FBtr0089020,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3253610    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:110;Name=Dscam-in;Parent=FBtr0089020,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_547;Name=Dscam-cds;Parent=FBtr0306769;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_577;Name=Dscam-cds;Parent=FBtr0089018;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_583;Name=Dscam-cds;Parent=FBtr0111087,FBtr0111088,FBtr0111089;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3253971    3254132    .    -    .    ID=FBgn0033159:111;Name=Dscam:111;Parent=FBtr0089018,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3254133    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:111;Name=Dscam-in;Parent=FBtr0089018,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3254310    3254471    .    -    2    ID=CDS_FBgn0033159:112_577;Name=Dscam-cds;Parent=FBtr0089016;parent_type=mRNA
2R    FlyBase    CDS    3254310    3254471    .    -    2    ID=CDS_FBgn0033159:112_583;Name=Dscam-cds;Parent=FBtr0111078,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    exon    3254310    3254471    .    -    .    ID=FBgn0033159:112;Name=Dscam:112;Parent=FBtr0111078,FBtr0089016,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3254472    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:112;Name=Dscam-in;Parent=FBtr0111078,FBtr0089016,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3255893    3256237    .    -    .    ID=FBgn0033159:113;Name=Dscam:113;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3256238    3265038    .    -    .    ID=intron_FBgn0033159:114_FBgn0033159:113;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3265039    3265219    .    -    .    ID=FBgn0033159:114;Name=Dscam:114;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_547;Name=Dscam-u5;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_577;Name=Dscam-u5;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_583;Name=Dscam-u5;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_584;Name=Dscam-u5;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_698;Name=Dscam-u5;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_704;Name=Dscam-u5;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3265220    3268975    .    -    .    ID=intron_FBgn0033159:115_FBgn0033159:114;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3268976    3269374    .    -    .    ID=FBgn0033159:115;Name=Dscam:115;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_547;Name=Dscam-u5;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_577;Name=Dscam-u5;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_583;Name=Dscam-u5;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_584;Name=Dscam-u5;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_698;Name=Dscam-u5;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_704;Name=Dscam-u5;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
3R    FlyBase    gene    14982962    14985924    .    +    .    ID=FBgn0002962;Name=nos;fullname=nanos;Alias=FBgn0010916,FBgn0011347,FBgn0011364,FBgn0026836,DRONANOS,NANOS,l(3)j3B6,nNOS,l(3)j4B6,l(3)07117,CG5637,NOS,Nos,Dronanos,Nanos,nanos;Ontology_term=SO:0000010,SO:0000087,GO:0007283,GO:0005515,GO:0048813,GO:0007281,GO:0008354,GO:0045495,GO:0008595,GO:0007314,GO:0042078,GO:0001709,GO:0048134,GO:0016321,GO:0006378,GO:0045892,GO:0007280,GO:0017148,GO:0048477,GO:0043066,GO:0030718,GO:0003723,GO:0008270;Dbxref=FlyBase:FBan0005637,FlyBase_Annotation_IDs:CG5637,GB_protein:AAF55641,GB:AA978726,GB:AC006091,GB:AF067873,GB:AQ034086,GB:AW943146,GB:AY075406,GB_protein:AAL68233,GB:CL705877,GB:CZ467207,GB:CZ467208,GB:G00548,GB:M72421,GB_protein:AAA28715,Rfam:RF00161,UniProt/Swiss-Prot:P25724,UniProt/TrEMBL:Q8SY38,INTERPRO:IPR008705,INTERPRO:IPR001878,INTERPRO:IPR024161,EntrezGene:42297,InterologFinder:42297,BIOGRID:67295,FlyAtlas:CG5637-RA,GenomeRNAi:42297,INTERACTIVEFLY:/torstoll/nanos1.htm;gbunit=AE014297;derived_computed_cyto=91F4-91F4
3R    FlyBase    mRNA    14982962    14985924    .    +    .    ID=FBtr0083732;Name=nos-RA;Parent=FBgn0002962;Dbxref=FlyBase_Annotation_IDs:CG5637-RA,REFSEQ:NM_057310;score_text=Strongly Supported;score=15
3R    FlyBase    exon    14982962    14983694    .    +    .    ID=FBgn0002962:1;Name=nos:1;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    five_prime_UTR    14982962    14983222    .    +    .    ID=five_prime_UTR_FBgn0002962:1_14;Name=nos-u5;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    CDS    14983223    14983694    .    +    0    ID=CDS_FBgn0002962:1_14;Name=nos-cds;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    intron    14983695    14984237    .    +    .    ID=intron_FBgn0002962:1_FBgn0002962:2;Name=nos-in;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    CDS    14984238    14984707    .    +    2    ID=CDS_FBgn0002962:2_14;Name=nos-cds;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    exon    14984238    14984707    .    +    .    ID=FBgn0002962:2;Name=nos:2;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    intron    14984708    14984780    .    +    .    ID=intron_FBgn0002962:2_FBgn0002962:3;Name=nos-in;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    exon    14984781    14985924    .    +    .    ID=FBgn0002962:3;Name=nos:3;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    CDS    14984781    14985044    .    +    0    ID=CDS_FBgn0002962:3_14;Name=nos-cds;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    three_prime_UTR    14985045    14985924    .    +    .    ID=three_prime_UTR_FBgn0002962:3_14;Name=nos-u3;Parent=FBtr0083732;parent_type=mRNA
4    TestData    gene    5000    6010    .    +    0    ID=TestBadGene1
4    TestData    mRNA    5000    6010    .    +    0    ID=TestBadTranscript1;Parent=TestBadGene1
4    TestData    exon    5000    5010    .    +    0    ID=exon_test11;Parent=TestBadTranscript1;parent_type=mRNA
4    TestData    exon    5500    6010    .    -    0    ID=exon_test12;Parent=TestBadTranscript1;parent_type=mRNA
4    TestData    gene    5000    6010    .    +    0    ID=TestBadGene2
4    TestData    mRNA    5000    6010    .    +    0    ID=TestBadTranscript2;Parent=TestBadGene2
4    TestData    exon    5000    5010    .    +    0    ID=exon_test21;Parent=TestBadTranscript2;parent_type=mRNA
2L    TestData    exon    5500    6010    .    +    0    ID=exon_test22;Parent=TestBadTranscript2;parent_type=mRNA""".replace(
    "    ", "\t"
)

GFF3_TO_TRANSCRIPTS_STOP_FEATURE_SORTED = """2R    FlyBase    gene    3207059    3269374    .    -    .    ID=FBgn0033159;Name=Dscam;fullname=Down syndrome cell adhesion molecule;Alias=FBgn0004124,FBgn0010604,FBgn0033159,Neu1,Dscam1,p270,CT39257,l(2)43Bc,43Bc,lethal(2)43Bc,l(2)05518,CG17800,DScam,Drosophila Down syndrome cell adhesion molecule,Down's syndrome Cell Adhesion Molecule,DSCAM,Down syndrome cell-adhesion molecule,Down Syndrome Cell Adhesion Molecule,dscam,Down Syndrome cell adhesion molecule,dScam,DmDscam;Ontology_term=GO:0016319,SO:0000010,SO:0000087,GO:0007411,GO:0005887,GO:0008046,GO:0006909,GO:0007422,GO:0048666,GO:0007413,GO:0043025,GO:0030425,GO:0030424,GO:0021551,GO:0042803,GO:0048846,GO:0070593,GO:0051635,GO:0042802;Dbxref=FlyBase:FBan0017800,FlyBase_Annotation_IDs:CG17800,GB_protein:ABI31027,GB_protein:ABI31036,GB_protein:ABI31050,GB_protein:ABI31041,GB_protein:ABI31040,GB_protein:ABI31048,GB_protein:ABI31031,GB_protein:ABI31033,GB_protein:ABI31055,GB_protein:ABI31053,GB_protein:ABI31037,GB_protein:AAM68883,GB_protein:ABI31035,GB_protein:ABI31043,GB_protein:ABI31039,GB_protein:ABI31067,GB_protein:ABI31075,GB_protein:ABI31054,GB_protein:ABI31073,GB_protein:ABI31030,GB_protein:ABI31061,GB_protein:ABI31057,GB_protein:ABI31056,GB_protein:ABI31079,GB_protein:ABI31029,GB_protein:ABI31064,GB_protein:ABI31069,GB_protein:ABI31078,GB_protein:ABI31080,GB_protein:ABI31077,GB_protein:ABI31028,GB_protein:ABI31066,GB_protein:ABI31052,GB_protein:ABI31032,GB_protein:ABI31068,GB_protein:ABI31042,GB_protein:ABI31046,GB_protein:ABI31072,GB_protein:AAS64901,GB_protein:ABI31034,GB_protein:ABI31062,GB_protein:ABI31051,GB_protein:ABI31070,GB_protein:ABI31044,GB_protein:ABI31058,GB_protein:ABI31047,GB_protein:ABI31049,GB_protein:AAM68884,GB_protein:ABI31059,GB_protein:ABI31060,GB_protein:ABI31071,GB_protein:ABI31081,GB_protein:ABI31074,GB_protein:ABI31038,GB_protein:AAF59271,GB_protein:ABI31045,GB_protein:ABI31063,GB_protein:ABI31076,GB_protein:ABI31065,GB:AA697364,GB:AF260530,GB_protein:AAF71926,GB:AI542894,GB:AQ034165,GB:AW941417,GB:AW944596,GB:AY058789,GB_protein:AAL14018,GB:AY061009,GB_protein:AAL28557,GB:BG640148,GB:BH256401,GB:BH759233,GB:BH759234,GB:BT001877,GB_protein:AAN71649,GB:BZ748857,GB:BZ748858,GB:CL528133,GB:CL528281,GB:CZ466627,GB:CZ477334,GB:CZ482393,GB:CZ482394,GB:CZ482629,GB:CZ489596,UniProt/TrEMBL:Q8IGA5,UniProt/TrEMBL:Q95S10,UniProt/TrEMBL:Q95TG0,UniProt/TrEMBL:Q9NBA1,INTERPRO:IPR003598,INTERPRO:IPR003599,INTERPRO:IPR003961,INTERPRO:IPR007110,UniProt/TrEMBL:Q0E9H9,UniProt/TrEMBL:A1Z6X3,UniProt/TrEMBL:A1Z6X2,UniProt/TrEMBL:A1Z6X1,INTERPRO:IPR013783,EntrezGene:35652,UniProt/TrEMBL:Q0E9J9,UniProt/TrEMBL:Q0E9K0,UniProt/TrEMBL:Q0E9J1,UniProt/TrEMBL:Q0E9K3,UniProt/TrEMBL:Q0E9L9,UniProt/TrEMBL:Q0E9K1,UniProt/TrEMBL:Q0E9H0,UniProt/TrEMBL:Q0E9M2,UniProt/TrEMBL:Q0E9G9,UniProt/TrEMBL:Q0E9I8,UniProt/TrEMBL:Q0E9L5,UniProt/TrEMBL:Q0E9L2,UniProt/TrEMBL:Q0E9H5,UniProt/TrEMBL:Q0E9K2,UniProt/TrEMBL:Q0E9J6,UniProt/TrEMBL:Q0E9L7,UniProt/TrEMBL:Q0E9I3,UniProt/TrEMBL:Q0E9K4,UniProt/TrEMBL:Q0E9I6,UniProt/TrEMBL:Q0E9J4,UniProt/TrEMBL:Q0E9H4,UniProt/TrEMBL:Q0E9H8,UniProt/TrEMBL:Q0E9K6,UniProt/TrEMBL:Q0E9I4,UniProt/TrEMBL:Q0E9H6,UniProt/TrEMBL:Q0E9I9,UniProt/TrEMBL:Q0E9J3,UniProt/TrEMBL:Q0E9J5,UniProt/TrEMBL:Q0E9L3,UniProt/TrEMBL:Q0E9M0,UniProt/TrEMBL:Q0E9I2,UniProt/TrEMBL:Q0E9K9,UniProt/TrEMBL:Q0E9J8,UniProt/TrEMBL:Q0E9J0,UniProt/TrEMBL:Q0E9L6,UniProt/TrEMBL:Q0E9L4,UniProt/TrEMBL:Q0E9I0,UniProt/TrEMBL:Q0E9I7,UniProt/TrEMBL:Q0E9J7,UniProt/TrEMBL:Q0E9K5,UniProt/TrEMBL:Q0E9H7,UniProt/TrEMBL:Q0E9L0,UniProt/TrEMBL:Q0E9M4,UniProt/TrEMBL:Q0E9L8,UniProt/TrEMBL:Q0E9H3,UniProt/TrEMBL:Q0E9M1,UniProt/TrEMBL:Q0E9H1,UniProt/TrEMBL:Q0E9M3,UniProt/TrEMBL:Q0E9I5,UniProt/TrEMBL:Q0E9L1,UniProt/TrEMBL:Q0E9K8,UniProt/TrEMBL:Q0E9K7,UniProt/TrEMBL:Q0E9J2,UniProt/TrEMBL:Q0E9H2,UniProt/TrEMBL:Q0E9I1,INTERPRO:IPR021012,INTERPRO:IPR013098,InterologFinder:35652,BIOGRID:61533,FlyAtlas:HDC04985%3BCG17800-RA%3BCG15693-RA,GenomeRNAi:35652,INTERACTIVEFLY:/hjmuller/dscam1.htm;gbunit=AE013599;derived_computed_cyto=43A4-43B1
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089016;Name=Dscam-RA;Parent=FBgn0033159;Alias=CG17800-RA;Dbxref=FlyBase_Annotation_IDs:CG17800-RA,REFSEQ:NM_165514;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089018;Name=Dscam-RC;Parent=FBgn0033159;Alias=CG17800-RC;Dbxref=FlyBase_Annotation_IDs:CG17800-RC,REFSEQ:NM_165513;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089019;Name=Dscam-RD;Parent=FBgn0033159;Alias=CG17800-RD;Dbxref=FlyBase_Annotation_IDs:CG17800-RD,REFSEQ:NM_078925;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0089020;Name=Dscam-RE;Parent=FBgn0033159;Alias=CG17800-RE;Dbxref=FlyBase_Annotation_IDs:CG17800-RE,REFSEQ:NM_206047;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111049;Name=Dscam-RAA;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAA,REFSEQ:NM_001043036;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111050;Name=Dscam-RAB;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAB,REFSEQ:NM_001043050;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111051;Name=Dscam-RAC;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAC,REFSEQ:NM_001043012;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111052;Name=Dscam-RAD;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAD,REFSEQ:NM_001043061;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111053;Name=Dscam-RAE;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAE,REFSEQ:NM_001043064;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111054;Name=Dscam-RAF;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAF,REFSEQ:NM_001043062;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111055;Name=Dscam-RAG;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAG,REFSEQ:NM_001043053;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111056;Name=Dscam-RAH;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAH,REFSEQ:NM_001043048;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111057;Name=Dscam-RAI;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAI,REFSEQ:NM_001043063;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111058;Name=Dscam-RAJ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAJ,REFSEQ:NM_001043040;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111059;Name=Dscam-RAK;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAK,REFSEQ:NM_001043045;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111060;Name=Dscam-RAL;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAL,REFSEQ:NM_001043014;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111061;Name=Dscam-RAM;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAM,REFSEQ:NM_001043057;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111062;Name=Dscam-RAO;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAO,REFSEQ:NM_001043059;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111063;Name=Dscam-RAP;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAP,REFSEQ:NM_001043051;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111064;Name=Dscam-RAQ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAQ,REFSEQ:NM_001043023;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111065;Name=Dscam-RAS;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAS,REFSEQ:NM_001043019;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111066;Name=Dscam-RAT;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAT,REFSEQ:NM_001043021;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111067;Name=Dscam-RAU;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAU,REFSEQ:NM_001043037;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111068;Name=Dscam-RAV;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAV,REFSEQ:NM_001043039;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111069;Name=Dscam-RAW;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAW,REFSEQ:NM_001043017;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111070;Name=Dscam-RAX;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAX,REFSEQ:NM_001043015;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111071;Name=Dscam-RAY;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAY,REFSEQ:NM_001043032;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111072;Name=Dscam-RAZ;Parent=FBgn0033159;Alias=Dscam-RAY;Dbxref=FlyBase_Annotation_IDs:CG17800-RAZ,REFSEQ:NM_001043025;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111073;Name=Dscam-RBA;Parent=FBgn0033159;Alias=Dscam-RAZ;Dbxref=FlyBase_Annotation_IDs:CG17800-RBA,REFSEQ:NM_001043034;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111074;Name=Dscam-RBB;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBB,REFSEQ:NM_001043020;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111075;Name=Dscam-RBC;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBC,REFSEQ:NM_001043024;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111076;Name=Dscam-RBD;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBD,REFSEQ:NM_001043013;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111077;Name=Dscam-RBE;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBE,REFSEQ:NM_001043041;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111078;Name=Dscam-RBF;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBF,REFSEQ:NM_001043029;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111079;Name=Dscam-RBG;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBG,REFSEQ:NM_001043016;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111080;Name=Dscam-RBH;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBH,REFSEQ:NM_001043011;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111081;Name=Dscam-RF;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RF,REFSEQ:NM_001043049;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111082;Name=Dscam-RG;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RG,REFSEQ:NM_001043060;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111083;Name=Dscam-RH;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RH,REFSEQ:NM_001043047;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111084;Name=Dscam-RI;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RI,REFSEQ:NM_001043022;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111085;Name=Dscam-RJ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RJ,REFSEQ:NM_001043058;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111087;Name=Dscam-RL;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RL,REFSEQ:NM_001043055;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111088;Name=Dscam-RM;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RM,REFSEQ:NM_001043044;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111089;Name=Dscam-RO;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RO,REFSEQ:NM_001043033;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111092;Name=Dscam-RR;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RR,REFSEQ:NM_001043028;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111094;Name=Dscam-RT;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RT,REFSEQ:NM_001043035;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111096;Name=Dscam-RV;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RV,REFSEQ:NM_001043018;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111097;Name=Dscam-RW;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RW,REFSEQ:NM_001043056;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111098;Name=Dscam-RX;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RX,REFSEQ:NM_001043030;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111099;Name=Dscam-RY;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RY,REFSEQ:NM_001043026;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111100;Name=Dscam-RZ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RZ,REFSEQ:NM_001043052;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111101;Name=Dscam-RAR;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAR,REFSEQ:NM_001043027;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0111102;Name=Dscam-RAN;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RAN,REFSEQ:NM_001043038;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306767;Name=Dscam-RBX;Parent=FBgn0033159;Alias=FBtr0111086,CG17800-RK;Dbxref=FlyBase_Annotation_IDs:CG17800-RBX;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306768;Name=Dscam-RBY;Parent=FBgn0033159;Alias=FBtr0111103,CG17800-RN;Dbxref=FlyBase_Annotation_IDs:CG17800-RBY;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306769;Name=Dscam-RBZ;Parent=FBgn0033159;Alias=FBtr0111090,CG17800-RP;Dbxref=FlyBase_Annotation_IDs:CG17800-RBZ;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306770;Name=Dscam-RCA;Parent=FBgn0033159;Alias=FBtr0111091,CG17800-RQ;Dbxref=FlyBase_Annotation_IDs:CG17800-RCA;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306771;Name=Dscam-RCB;Parent=FBgn0033159;Alias=FBtr0111093,CG17800-RS;Dbxref=FlyBase_Annotation_IDs:CG17800-RCB;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306772;Name=Dscam-RCC;Parent=FBgn0033159;Alias=FBtr0111095,CG17800-RU;Dbxref=FlyBase_Annotation_IDs:CG17800-RCC;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306773;Name=Dscam-RBI;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBI;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306774;Name=Dscam-RBJ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBJ;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306775;Name=Dscam-RBK;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBK;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306776;Name=Dscam-RBL;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBL;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306777;Name=Dscam-RBM;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBM;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306778;Name=Dscam-RBN;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBN;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306779;Name=Dscam-RBO;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBO;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306780;Name=Dscam-RBP;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBP;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306781;Name=Dscam-RBQ;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBQ;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306782;Name=Dscam-RBR;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBR;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306783;Name=Dscam-RBS;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBS;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306784;Name=Dscam-RBT;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBT;score_text=Strongly Supported;score=11
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306785;Name=Dscam-RBU;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBU;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306786;Name=Dscam-RBV;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBV;score_text=Weakly Supported;score=3
2R    FlyBase    mRNA    3207059    3269374    .    -    .    ID=FBtr0306787;Name=Dscam-RBW;Parent=FBgn0033159;Dbxref=FlyBase_Annotation_IDs:CG17800-RBW;score_text=Strongly Supported;score=11
2R    FlyBase    exon    3207059    3208269    .    -    .    ID=FBgn0033159:1;Name=Dscam:1;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_547;Name=Dscam-u3;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_577;Name=Dscam-u3;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_583;Name=Dscam-u3;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_584;Name=Dscam-u3;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_698;Name=Dscam-u3;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    three_prime_UTR    3207059    3208187    .    -    .    ID=three_prime_UTR_FBgn0033159:1_704;Name=Dscam-u3;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    1    ID=CDS_FBgn0033159:1_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    1    ID=CDS_FBgn0033159:1_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    1    ID=CDS_FBgn0033159:1_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    2    ID=CDS_FBgn0033159:1_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    2    ID=CDS_FBgn0033159:1_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3208188    3208269    .    -    2    ID=CDS_FBgn0033159:1_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3208270    3209087    .    -    .    ID=intron_FBgn0033159:2_FBgn0033159:1;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    0    ID=CDS_FBgn0033159:2_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    0    ID=CDS_FBgn0033159:2_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    0    ID=CDS_FBgn0033159:2_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    2    ID=CDS_FBgn0033159:2_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    2    ID=CDS_FBgn0033159:2_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3209088    3209154    .    -    2    ID=CDS_FBgn0033159:2_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3209088    3209154    .    -    .    ID=FBgn0033159:2;Name=Dscam:2;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3209155    3210064    .    -    .    ID=intron_FBgn0033159:3_FBgn0033159:2;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    0    ID=CDS_FBgn0033159:3_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    0    ID=CDS_FBgn0033159:3_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    0    ID=CDS_FBgn0033159:3_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    2    ID=CDS_FBgn0033159:3_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    2    ID=CDS_FBgn0033159:3_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3210065    3210223    .    -    2    ID=CDS_FBgn0033159:3_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3210065    3210223    .    -    .    ID=FBgn0033159:3;Name=Dscam:3;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3210224    3210280    .    -    .    ID=intron_FBgn0033159:4_FBgn0033159:3;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    0    ID=CDS_FBgn0033159:4_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    0    ID=CDS_FBgn0033159:4_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    0    ID=CDS_FBgn0033159:4_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    1    ID=CDS_FBgn0033159:4_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    1    ID=CDS_FBgn0033159:4_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3210281    3210461    .    -    1    ID=CDS_FBgn0033159:4_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3210281    3210461    .    -    .    ID=FBgn0033159:4;Name=Dscam:4;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3210462    3210526    .    -    .    ID=intron_FBgn0033159:5_FBgn0033159:4;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    0    ID=CDS_FBgn0033159:5_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    0    ID=CDS_FBgn0033159:5_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    0    ID=CDS_FBgn0033159:5_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    2    ID=CDS_FBgn0033159:5_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    2    ID=CDS_FBgn0033159:5_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3210527    3210837    .    -    2    ID=CDS_FBgn0033159:5_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3210527    3210837    .    -    .    ID=FBgn0033159:5;Name=Dscam:5;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3210838    3211323    .    -    .    ID=intron_FBgn0033159:6_FBgn0033159:5;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    0    ID=CDS_FBgn0033159:6_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    0    ID=CDS_FBgn0033159:6_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    0    ID=CDS_FBgn0033159:6_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    2    ID=CDS_FBgn0033159:6_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    2    ID=CDS_FBgn0033159:6_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3211324    3211473    .    -    2    ID=CDS_FBgn0033159:6_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3211324    3211473    .    -    .    ID=FBgn0033159:6;Name=Dscam:6;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3211474    3212086    .    -    .    ID=intron_FBgn0033159:7_FBgn0033159:6;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    0    ID=CDS_FBgn0033159:7_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    0    ID=CDS_FBgn0033159:7_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    0    ID=CDS_FBgn0033159:7_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    2    ID=CDS_FBgn0033159:7_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    2    ID=CDS_FBgn0033159:7_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3212087    3212242    .    -    2    ID=CDS_FBgn0033159:7_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3212087    3212242    .    -    .    ID=FBgn0033159:7;Name=Dscam:7;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3212243    3214121    .    -    .    ID=intron_FBgn0033159:9_FBgn0033159:7;Name=Dscam-in;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063,FBtr0306768,FBtr0306771,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3212243    3213656    .    -    .    ID=intron_FBgn0033159:8_FBgn0033159:7;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059,FBtr0306767,FBtr0306769,FBtr0306770,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3213657    3213869    .    -    0    ID=CDS_FBgn0033159:8_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3213657    3213869    .    -    0    ID=CDS_FBgn0033159:8_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3213657    3213869    .    -    2    ID=CDS_FBgn0033159:8_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3213657    3213869    .    -    .    ID=FBgn0033159:8;Name=Dscam:8;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059,FBtr0306767,FBtr0306769,FBtr0306770,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3213870    3216793    .    -    .    ID=intron_FBgn0033159:10_FBgn0033159:8;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111072,FBtr0111073,FBtr0111101,FBtr0111065,FBtr0111068,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111087,FBtr0111089,FBtr0111094,FBtr0111098,FBtr0111099,FBtr0111050,FBtr0111051,FBtr0111054,FBtr0111055,FBtr0111058,FBtr0111059,FBtr0306767,FBtr0306769,FBtr0306770,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306777,FBtr0306778,FBtr0306781,FBtr0306782,FBtr0306785,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    0    ID=CDS_FBgn0033159:9_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    0    ID=CDS_FBgn0033159:9_704;Name=Dscam-cds;Parent=FBtr0306768,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    2    ID=CDS_FBgn0033159:9_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    2    ID=CDS_FBgn0033159:9_583;Name=Dscam-cds;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3214122    3214289    .    -    2    ID=CDS_FBgn0033159:9_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3214122    3214289    .    -    .    ID=FBgn0033159:9;Name=Dscam:9;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063,FBtr0306768,FBtr0306771,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3214290    3216793    .    -    .    ID=intron_FBgn0033159:10_FBgn0033159:9;Name=Dscam-in;Parent=FBtr0111074,FBtr0111070,FBtr0111071,FBtr0111064,FBtr0111066,FBtr0111067,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111088,FBtr0111092,FBtr0111096,FBtr0111097,FBtr0111100,FBtr0111049,FBtr0111052,FBtr0111053,FBtr0111056,FBtr0111057,FBtr0111063,FBtr0306768,FBtr0306771,FBtr0306776,FBtr0306779,FBtr0306780,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    1    ID=CDS_FBgn0033159:10_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    1    ID=CDS_FBgn0033159:10_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    1    ID=CDS_FBgn0033159:10_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    2    ID=CDS_FBgn0033159:10_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    2    ID=CDS_FBgn0033159:10_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3216794    3216960    .    -    2    ID=CDS_FBgn0033159:10_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3216794    3216960    .    -    .    ID=FBgn0033159:10;Name=Dscam:10;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3216961    3217017    .    -    .    ID=intron_FBgn0033159:11_FBgn0033159:10;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    1    ID=CDS_FBgn0033159:11_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    1    ID=CDS_FBgn0033159:11_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    1    ID=CDS_FBgn0033159:11_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    2    ID=CDS_FBgn0033159:11_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    2    ID=CDS_FBgn0033159:11_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3217018    3217194    .    -    2    ID=CDS_FBgn0033159:11_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3217018    3217194    .    -    .    ID=FBgn0033159:11;Name=Dscam:11;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3217195    3217247    .    -    .    ID=intron_FBgn0033159:12_FBgn0033159:11;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    0    ID=CDS_FBgn0033159:12_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    0    ID=CDS_FBgn0033159:12_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    0    ID=CDS_FBgn0033159:12_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    2    ID=CDS_FBgn0033159:12_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    2    ID=CDS_FBgn0033159:12_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3217248    3217356    .    -    2    ID=CDS_FBgn0033159:12_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3217248    3217356    .    -    .    ID=FBgn0033159:12;Name=Dscam:12;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3217357    3217421    .    -    .    ID=intron_FBgn0033159:13_FBgn0033159:12;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    0    ID=CDS_FBgn0033159:13_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    0    ID=CDS_FBgn0033159:13_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    0    ID=CDS_FBgn0033159:13_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    2    ID=CDS_FBgn0033159:13_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    2    ID=CDS_FBgn0033159:13_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3217422    3218105    .    -    2    ID=CDS_FBgn0033159:13_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3217422    3218105    .    -    .    ID=FBgn0033159:13;Name=Dscam:13;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3218106    3218163    .    -    .    ID=intron_FBgn0033159:14_FBgn0033159:13;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    0    ID=CDS_FBgn0033159:14_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    0    ID=CDS_FBgn0033159:14_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    0    ID=CDS_FBgn0033159:14_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    2    ID=CDS_FBgn0033159:14_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    2    ID=CDS_FBgn0033159:14_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3218164    3218319    .    -    2    ID=CDS_FBgn0033159:14_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3218164    3218319    .    -    .    ID=FBgn0033159:14;Name=Dscam:14;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3218320    3218480    .    -    .    ID=intron_FBgn0033159:15_FBgn0033159:14;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    0    ID=CDS_FBgn0033159:15_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    0    ID=CDS_FBgn0033159:15_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    0    ID=CDS_FBgn0033159:15_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    2    ID=CDS_FBgn0033159:15_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    2    ID=CDS_FBgn0033159:15_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3218481    3219707    .    -    2    ID=CDS_FBgn0033159:15_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3218481    3219707    .    -    .    ID=FBgn0033159:15;Name=Dscam:15;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3219708    3219765    .    -    .    ID=intron_FBgn0033159:16_FBgn0033159:15;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    0    ID=CDS_FBgn0033159:16_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    0    ID=CDS_FBgn0033159:16_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    0    ID=CDS_FBgn0033159:16_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    2    ID=CDS_FBgn0033159:16_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    2    ID=CDS_FBgn0033159:16_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3219766    3219885    .    -    2    ID=CDS_FBgn0033159:16_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3219766    3219885    .    -    .    ID=FBgn0033159:16;Name=Dscam:16;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3219886    3234896    .    -    .    ID=intron_FBgn0033159:49_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0089020,FBtr0111102,FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3219886    3234511    .    -    .    ID=intron_FBgn0033159:48_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3219886    3234146    .    -    .    ID=intron_FBgn0033159:47_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3219886    3233773    .    -    .    ID=intron_FBgn0033159:46_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    intron    3219886    3233380    .    -    .    ID=intron_FBgn0033159:45_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3219886    3232989    .    -    .    ID=intron_FBgn0033159:44_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3219886    3232565    .    -    .    ID=intron_FBgn0033159:43_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111094,FBtr0111054,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3219886    3232168    .    -    .    ID=intron_FBgn0033159:42_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100,FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3219886    3231783    .    -    .    ID=intron_FBgn0033159:41_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111074,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3219886    3231222    .    -    .    ID=intron_FBgn0033159:40_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3219886    3230836    .    -    .    ID=intron_FBgn0033159:39_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111083,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3219886    3230468    .    -    .    ID=intron_FBgn0033159:38_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    intron    3219886    3230087    .    -    .    ID=intron_FBgn0033159:37_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3219886    3229608    .    -    .    ID=intron_FBgn0033159:36_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    intron    3219886    3229212    .    -    .    ID=intron_FBgn0033159:35_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    intron    3219886    3228832    .    -    .    ID=intron_FBgn0033159:34_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3219886    3228449    .    -    .    ID=intron_FBgn0033159:33_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3219886    3228049    .    -    .    ID=intron_FBgn0033159:32_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3219886    3227659    .    -    .    ID=intron_FBgn0033159:31_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    intron    3219886    3227287    .    -    .    ID=intron_FBgn0033159:30_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3219886    3226907    .    -    .    ID=intron_FBgn0033159:29_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    intron    3219886    3226532    .    -    .    ID=intron_FBgn0033159:28_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3219886    3226145    .    -    .    ID=intron_FBgn0033159:27_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111056,FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3219886    3225754    .    -    .    ID=intron_FBgn0033159:26_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    intron    3219886    3225376    .    -    .    ID=intron_FBgn0033159:25_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    intron    3219886    3224998    .    -    .    ID=intron_FBgn0033159:24_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    intron    3219886    3224593    .    -    .    ID=intron_FBgn0033159:23_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3219886    3224035    .    -    .    ID=intron_FBgn0033159:22_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3219886    3223621    .    -    .    ID=intron_FBgn0033159:21_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3219886    3223220    .    -    .    ID=intron_FBgn0033159:20_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111077,FBtr0111087,FBtr0111089,FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3219886    3222438    .    -    .    ID=intron_FBgn0033159:19_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111075,FBtr0111076,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3219886    3221892    .    -    .    ID=intron_FBgn0033159:18_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3219886    3221191    .    -    .    ID=intron_FBgn0033159:17_FBgn0033159:16;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3221192    3221485    .    -    2    ID=CDS_FBgn0033159:17_704;Name=Dscam-cds;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    exon    3221192    3221485    .    -    .    ID=FBgn0033159:17;Name=Dscam:17;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3221486    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:17;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3221893    3222171    .    -    2    ID=CDS_FBgn0033159:18_583;Name=Dscam-cds;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    exon    3221893    3222171    .    -    .    ID=FBgn0033159:18;Name=Dscam:18;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3222172    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:18;Name=Dscam-in;Parent=FBtr0111092,FBtr0111098;parent_type=mRNA
2R    FlyBase    CDS    3222439    3222723    .    -    2    ID=CDS_FBgn0033159:19_583;Name=Dscam-cds;Parent=FBtr0111075,FBtr0111076;parent_type=mRNA
2R    FlyBase    CDS    3222439    3222723    .    -    2    ID=CDS_FBgn0033159:19_704;Name=Dscam-cds;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    exon    3222439    3222723    .    -    .    ID=FBgn0033159:19;Name=Dscam:19;Parent=FBtr0111075,FBtr0111076,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3222724    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:19;Name=Dscam-in;Parent=FBtr0111075,FBtr0111076,FBtr0306777;parent_type=mRNA
2R    FlyBase    CDS    3223221    3223508    .    -    2    ID=CDS_FBgn0033159:20_583;Name=Dscam-cds;Parent=FBtr0111077,FBtr0111087,FBtr0111089;parent_type=mRNA
2R    FlyBase    CDS    3223221    3223508    .    -    2    ID=CDS_FBgn0033159:20_704;Name=Dscam-cds;Parent=FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3223221    3223508    .    -    .    ID=FBgn0033159:20;Name=Dscam:20;Parent=FBtr0111077,FBtr0111087,FBtr0111089,FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3223509    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:20;Name=Dscam-in;Parent=FBtr0111077,FBtr0111087,FBtr0111089,FBtr0306775,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3223622    3223915    .    -    2    ID=CDS_FBgn0033159:21_704;Name=Dscam-cds;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    exon    3223622    3223915    .    -    .    ID=FBgn0033159:21;Name=Dscam:21;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3223916    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:21;Name=Dscam-in;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3224036    3224323    .    -    2    ID=CDS_FBgn0033159:22_704;Name=Dscam-cds;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    exon    3224036    3224323    .    -    .    ID=FBgn0033159:22;Name=Dscam:22;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3224324    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:22;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    CDS    3224594    3224881    .    -    2    ID=CDS_FBgn0033159:23_704;Name=Dscam-cds;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    exon    3224594    3224881    .    -    .    ID=FBgn0033159:23;Name=Dscam:23;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3224882    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:23;Name=Dscam-in;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    CDS    3224999    3225286    .    -    2    ID=CDS_FBgn0033159:24_583;Name=Dscam-cds;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    exon    3224999    3225286    .    -    .    ID=FBgn0033159:24;Name=Dscam:24;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    intron    3225287    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:24;Name=Dscam-in;Parent=FBtr0111071,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3225377    3225664    .    -    2    ID=CDS_FBgn0033159:25_583;Name=Dscam-cds;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    exon    3225377    3225664    .    -    .    ID=FBgn0033159:25;Name=Dscam:25;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    intron    3225665    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:25;Name=Dscam-in;Parent=FBtr0111070,FBtr0111065;parent_type=mRNA
2R    FlyBase    CDS    3225755    3226042    .    -    2    ID=CDS_FBgn0033159:26_583;Name=Dscam-cds;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    exon    3225755    3226042    .    -    .    ID=FBgn0033159:26;Name=Dscam:26;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    intron    3226043    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:26;Name=Dscam-in;Parent=FBtr0111066;parent_type=mRNA
2R    FlyBase    CDS    3226146    3226433    .    -    2    ID=CDS_FBgn0033159:27_583;Name=Dscam-cds;Parent=FBtr0111056;parent_type=mRNA
2R    FlyBase    CDS    3226146    3226433    .    -    2    ID=CDS_FBgn0033159:27_704;Name=Dscam-cds;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    exon    3226146    3226433    .    -    .    ID=FBgn0033159:27;Name=Dscam:27;Parent=FBtr0111056,FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3226434    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:27;Name=Dscam-in;Parent=FBtr0111056,FBtr0306779;parent_type=mRNA
2R    FlyBase    CDS    3226533    3226820    .    -    2    ID=CDS_FBgn0033159:28_704;Name=Dscam-cds;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    exon    3226533    3226820    .    -    .    ID=FBgn0033159:28;Name=Dscam:28;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3226821    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:28;Name=Dscam-in;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    CDS    3226908    3227195    .    -    2    ID=CDS_FBgn0033159:29_583;Name=Dscam-cds;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    exon    3226908    3227195    .    -    .    ID=FBgn0033159:29;Name=Dscam:29;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    intron    3227196    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:29;Name=Dscam-in;Parent=FBtr0111060;parent_type=mRNA
2R    FlyBase    CDS    3227288    3227575    .    -    2    ID=CDS_FBgn0033159:30_583;Name=Dscam-cds;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    exon    3227288    3227575    .    -    .    ID=FBgn0033159:30;Name=Dscam:30;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3227576    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:30;Name=Dscam-in;Parent=FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    CDS    3227660    3227947    .    -    2    ID=CDS_FBgn0033159:31_583;Name=Dscam-cds;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    exon    3227660    3227947    .    -    .    ID=FBgn0033159:31;Name=Dscam:31;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    intron    3227948    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:31;Name=Dscam-in;Parent=FBtr0111096,FBtr0111050;parent_type=mRNA
2R    FlyBase    CDS    3228050    3228337    .    -    2    ID=CDS_FBgn0033159:32_583;Name=Dscam-cds;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    exon    3228050    3228337    .    -    .    ID=FBgn0033159:32;Name=Dscam:32;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3228338    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:32;Name=Dscam-in;Parent=FBtr0111051;parent_type=mRNA
2R    FlyBase    CDS    3228450    3228737    .    -    2    ID=CDS_FBgn0033159:33_583;Name=Dscam-cds;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    exon    3228450    3228737    .    -    .    ID=FBgn0033159:33;Name=Dscam:33;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3228738    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:33;Name=Dscam-in;Parent=FBtr0111053;parent_type=mRNA
2R    FlyBase    CDS    3228833    3229123    .    -    2    ID=CDS_FBgn0033159:34_583;Name=Dscam-cds;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    exon    3228833    3229123    .    -    .    ID=FBgn0033159:34;Name=Dscam:34;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3229124    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:34;Name=Dscam-in;Parent=FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111052,FBtr0111055;parent_type=mRNA
2R    FlyBase    CDS    3229213    3229500    .    -    2    ID=CDS_FBgn0033159:35_583;Name=Dscam-cds;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    exon    3229213    3229500    .    -    .    ID=FBgn0033159:35;Name=Dscam:35;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    intron    3229501    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:35;Name=Dscam-in;Parent=FBtr0111101;parent_type=mRNA
2R    FlyBase    CDS    3229609    3229896    .    -    2    ID=CDS_FBgn0033159:36_583;Name=Dscam-cds;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    exon    3229609    3229896    .    -    .    ID=FBgn0033159:36;Name=Dscam:36;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    intron    3229897    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:36;Name=Dscam-in;Parent=FBtr0111069,FBtr0111081;parent_type=mRNA
2R    FlyBase    CDS    3230088    3230386    .    -    2    ID=CDS_FBgn0033159:37_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3230088    3230386    .    -    2    ID=CDS_FBgn0033159:37_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3230088    3230386    .    -    2    ID=CDS_FBgn0033159:37_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772;parent_type=mRNA
2R    FlyBase    exon    3230088    3230386    .    -    .    ID=FBgn0033159:37;Name=Dscam:37;Parent=FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3230387    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:37;Name=Dscam-in;Parent=FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772;parent_type=mRNA
2R    FlyBase    CDS    3230469    3230753    .    -    2    ID=CDS_FBgn0033159:38_583;Name=Dscam-cds;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    exon    3230469    3230753    .    -    .    ID=FBgn0033159:38;Name=Dscam:38;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    intron    3230754    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:38;Name=Dscam-in;Parent=FBtr0111058;parent_type=mRNA
2R    FlyBase    CDS    3230837    3231127    .    -    2    ID=CDS_FBgn0033159:39_583;Name=Dscam-cds;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    CDS    3230837    3231127    .    -    2    ID=CDS_FBgn0033159:39_704;Name=Dscam-cds;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    exon    3230837    3231127    .    -    .    ID=FBgn0033159:39;Name=Dscam:39;Parent=FBtr0111083,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3231128    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:39;Name=Dscam-in;Parent=FBtr0111083,FBtr0306780;parent_type=mRNA
2R    FlyBase    CDS    3231223    3231513    .    -    2    ID=CDS_FBgn0033159:40_704;Name=Dscam-cds;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    exon    3231223    3231513    .    -    .    ID=FBgn0033159:40;Name=Dscam:40;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3231514    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:40;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    CDS    3231784    3232074    .    -    2    ID=CDS_FBgn0033159:41_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3231784    3232074    .    -    2    ID=CDS_FBgn0033159:41_583;Name=Dscam-cds;Parent=FBtr0111074,FBtr0111061;parent_type=mRNA
2R    FlyBase    exon    3231784    3232074    .    -    .    ID=FBgn0033159:41;Name=Dscam:41;Parent=FBtr0111074,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3232075    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:41;Name=Dscam-in;Parent=FBtr0111074,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0111061;parent_type=mRNA
2R    FlyBase    CDS    3232169    3232459    .    -    2    ID=CDS_FBgn0033159:42_583;Name=Dscam-cds;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100;parent_type=mRNA
2R    FlyBase    CDS    3232169    3232459    .    -    2    ID=CDS_FBgn0033159:42_704;Name=Dscam-cds;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    exon    3232169    3232459    .    -    .    ID=FBgn0033159:42;Name=Dscam:42;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100,FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3232460    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:42;Name=Dscam-in;Parent=FBtr0111062,FBtr0111088,FBtr0111099,FBtr0111100,FBtr0306783;parent_type=mRNA
2R    FlyBase    CDS    3232566    3232856    .    -    2    ID=CDS_FBgn0033159:43_583;Name=Dscam-cds;Parent=FBtr0111094,FBtr0111054;parent_type=mRNA
2R    FlyBase    CDS    3232566    3232856    .    -    2    ID=CDS_FBgn0033159:43_704;Name=Dscam-cds;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    exon    3232566    3232856    .    -    .    ID=FBgn0033159:43;Name=Dscam:43;Parent=FBtr0111094,FBtr0111054,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3232857    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:43;Name=Dscam-in;Parent=FBtr0111094,FBtr0111054,FBtr0306782;parent_type=mRNA
2R    FlyBase    CDS    3232990    3233280    .    -    2    ID=CDS_FBgn0033159:44_583;Name=Dscam-cds;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    exon    3232990    3233280    .    -    .    ID=FBgn0033159:44;Name=Dscam:44;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3233281    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:44;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    CDS    3233381    3233671    .    -    2    ID=CDS_FBgn0033159:45_583;Name=Dscam-cds;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3233381    3233671    .    -    .    ID=FBgn0033159:45;Name=Dscam:45;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3233672    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:45;Name=Dscam-in;Parent=FBtr0111082,FBtr0111057,FBtr0111059;parent_type=mRNA
2R    FlyBase    CDS    3233774    3234061    .    -    2    ID=CDS_FBgn0033159:46_583;Name=Dscam-cds;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    exon    3233774    3234061    .    -    .    ID=FBgn0033159:46;Name=Dscam:46;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    intron    3234062    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:46;Name=Dscam-in;Parent=FBtr0111097;parent_type=mRNA
2R    FlyBase    CDS    3234147    3234434    .    -    2    ID=CDS_FBgn0033159:47_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    exon    3234147    3234434    .    -    .    ID=FBgn0033159:47;Name=Dscam:47;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3234435    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:47;Name=Dscam-in;Parent=FBtr0111079,FBtr0111078,FBtr0111067;parent_type=mRNA
2R    FlyBase    CDS    3234512    3234799    .    -    2    ID=CDS_FBgn0033159:48_583;Name=Dscam-cds;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    exon    3234512    3234799    .    -    .    ID=FBgn0033159:48;Name=Dscam:48;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3234800    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:48;Name=Dscam-in;Parent=FBtr0111068,FBtr0111049;parent_type=mRNA
2R    FlyBase    CDS    3234897    3235202    .    -    2    ID=CDS_FBgn0033159:49_583;Name=Dscam-cds;Parent=FBtr0111102;parent_type=mRNA
2R    FlyBase    CDS    3234897    3235202    .    -    2    ID=CDS_FBgn0033159:49_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3234897    3235202    .    -    2    ID=CDS_FBgn0033159:49_704;Name=Dscam-cds;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    exon    3234897    3235202    .    -    .    ID=FBgn0033159:49;Name=Dscam:49;Parent=FBtr0089020,FBtr0111102,FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3235203    3235353    .    -    .    ID=intron_FBgn0033159:50_FBgn0033159:49;Name=Dscam-in;Parent=FBtr0089020,FBtr0111102,FBtr0306781;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3235354    3235515    .    -    2    ID=CDS_FBgn0033159:50_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3235354    3235515    .    -    .    ID=FBgn0033159:50;Name=Dscam:50;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3235516    3235635    .    -    .    ID=intron_FBgn0033159:51_FBgn0033159:50;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3235636    3236480    .    -    1    ID=CDS_FBgn0033159:51_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3235636    3236480    .    -    .    ID=FBgn0033159:51;Name=Dscam:51;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3236481    3248030    .    -    .    ID=intron_FBgn0033159:99_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    intron    3236481    3247795    .    -    .    ID=intron_FBgn0033159:98_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111081,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3236481    3247586    .    -    .    ID=intron_FBgn0033159:97_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111092,FBtr0306770;parent_type=mRNA
2R    FlyBase    intron    3236481    3247379    .    -    .    ID=intron_FBgn0033159:96_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111063,FBtr0306767;parent_type=mRNA
2R    FlyBase    intron    3236481    3247171    .    -    .    ID=intron_FBgn0033159:95_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111082,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3236481    3246946    .    -    .    ID=intron_FBgn0033159:94_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    intron    3236481    3246726    .    -    .    ID=intron_FBgn0033159:93_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111072,FBtr0111058,FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3236481    3246520    .    -    .    ID=intron_FBgn0033159:92_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    intron    3236481    3246311    .    -    .    ID=intron_FBgn0033159:91_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    intron    3236481    3246108    .    -    .    ID=intron_FBgn0033159:90_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    intron    3236481    3245933    .    -    .    ID=intron_FBgn0033159:89_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3236481    3245720    .    -    .    ID=intron_FBgn0033159:88_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    intron    3236481    3245507    .    -    .    ID=intron_FBgn0033159:87_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3236481    3245293    .    -    .    ID=intron_FBgn0033159:86_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    intron    3236481    3245095    .    -    .    ID=intron_FBgn0033159:85_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111073,FBtr0306771;parent_type=mRNA
2R    FlyBase    intron    3236481    3244904    .    -    .    ID=intron_FBgn0033159:84_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    intron    3236481    3244687    .    -    .    ID=intron_FBgn0033159:83_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3236481    3244477    .    -    .    ID=intron_FBgn0033159:82_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3236481    3244249    .    -    .    ID=intron_FBgn0033159:81_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3236481    3244031    .    -    .    ID=intron_FBgn0033159:80_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3236481    3243821    .    -    .    ID=intron_FBgn0033159:79_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3236481    3243611    .    -    .    ID=intron_FBgn0033159:78_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3236481    3243405    .    -    .    ID=intron_FBgn0033159:77_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    intron    3236481    3243190    .    -    .    ID=intron_FBgn0033159:76_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111102,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3236481    3242795    .    -    .    ID=intron_FBgn0033159:75_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    intron    3236481    3242591    .    -    .    ID=intron_FBgn0033159:74_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111050,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3236481    3242386    .    -    .    ID=intron_FBgn0033159:73_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    intron    3236481    3242190    .    -    .    ID=intron_FBgn0033159:72_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    intron    3236481    3241981    .    -    .    ID=intron_FBgn0033159:71_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3236481    3241781    .    -    .    ID=intron_FBgn0033159:70_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3236481    3241561    .    -    .    ID=intron_FBgn0033159:69_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3236481    3241320    .    -    .    ID=intron_FBgn0033159:68_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    intron    3236481    3241070    .    -    .    ID=intron_FBgn0033159:67_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    intron    3236481    3240749    .    -    .    ID=intron_FBgn0033159:66_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    intron    3236481    3240403    .    -    .    ID=intron_FBgn0033159:65_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3236481    3239420    .    -    .    ID=intron_FBgn0033159:64_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3236481    3239203    .    -    .    ID=intron_FBgn0033159:63_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3236481    3238965    .    -    .    ID=intron_FBgn0033159:62_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    intron    3236481    3238678    .    -    .    ID=intron_FBgn0033159:61_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0089020,FBtr0306769;parent_type=mRNA
2R    FlyBase    intron    3236481    3238462    .    -    .    ID=intron_FBgn0033159:60_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3236481    3238286    .    -    .    ID=intron_FBgn0033159:59_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3236481    3238085    .    -    .    ID=intron_FBgn0033159:58_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3236481    3237878    .    -    .    ID=intron_FBgn0033159:57_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3236481    3237662    .    -    .    ID=intron_FBgn0033159:56_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3236481    3237489    .    -    .    ID=intron_FBgn0033159:55_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3236481    3237292    .    -    .    ID=intron_FBgn0033159:54_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    intron    3236481    3237105    .    -    .    ID=intron_FBgn0033159:53_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3236481    3236897    .    -    .    ID=intron_FBgn0033159:52_FBgn0033159:51;Name=Dscam-in;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    CDS    3236898    3237018    .    -    2    ID=CDS_FBgn0033159:52_583;Name=Dscam-cds;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    exon    3236898    3237018    .    -    .    ID=FBgn0033159:52;Name=Dscam:52;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    intron    3237019    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:52;Name=Dscam-in;Parent=FBtr0111071,FBtr0111066;parent_type=mRNA
2R    FlyBase    CDS    3237106    3237226    .    -    2    ID=CDS_FBgn0033159:53_704;Name=Dscam-cds;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    exon    3237106    3237226    .    -    .    ID=FBgn0033159:53;Name=Dscam:53;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    intron    3237227    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:53;Name=Dscam-in;Parent=FBtr0306783;parent_type=mRNA
2R    FlyBase    CDS    3237293    3237416    .    -    2    ID=CDS_FBgn0033159:54_583;Name=Dscam-cds;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    exon    3237293    3237416    .    -    .    ID=FBgn0033159:54;Name=Dscam:54;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    intron    3237417    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:54;Name=Dscam-in;Parent=FBtr0111099;parent_type=mRNA
2R    FlyBase    CDS    3237490    3237613    .    -    2    ID=CDS_FBgn0033159:55_704;Name=Dscam-cds;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    exon    3237490    3237613    .    -    .    ID=FBgn0033159:55;Name=Dscam:55;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3237614    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:55;Name=Dscam-in;Parent=FBtr0306782;parent_type=mRNA
2R    FlyBase    CDS    3237663    3237786    .    -    2    ID=CDS_FBgn0033159:56_704;Name=Dscam-cds;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    exon    3237663    3237786    .    -    .    ID=FBgn0033159:56;Name=Dscam:56;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    intron    3237787    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:56;Name=Dscam-in;Parent=FBtr0306781;parent_type=mRNA
2R    FlyBase    CDS    3237879    3238002    .    -    2    ID=CDS_FBgn0033159:57_583;Name=Dscam-cds;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    exon    3237879    3238002    .    -    .    ID=FBgn0033159:57;Name=Dscam:57;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3238003    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:57;Name=Dscam-in;Parent=FBtr0111085;parent_type=mRNA
2R    FlyBase    CDS    3238086    3238209    .    -    2    ID=CDS_FBgn0033159:58_583;Name=Dscam-cds;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    exon    3238086    3238209    .    -    .    ID=FBgn0033159:58;Name=Dscam:58;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    intron    3238210    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:58;Name=Dscam-in;Parent=FBtr0111070,FBtr0111062,FBtr0111098;parent_type=mRNA
2R    FlyBase    CDS    3238287    3238410    .    -    2    ID=CDS_FBgn0033159:59_704;Name=Dscam-cds;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    exon    3238287    3238410    .    -    .    ID=FBgn0033159:59;Name=Dscam:59;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3238411    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:59;Name=Dscam-in;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    CDS    3238463    3238586    .    -    2    ID=CDS_FBgn0033159:60_704;Name=Dscam-cds;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    exon    3238463    3238586    .    -    .    ID=FBgn0033159:60;Name=Dscam:60;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    intron    3238587    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:60;Name=Dscam-in;Parent=FBtr0306779;parent_type=mRNA
2R    FlyBase    CDS    3238679    3238802    .    -    2    ID=CDS_FBgn0033159:61_547;Name=Dscam-cds;Parent=FBtr0306769;parent_type=mRNA
2R    FlyBase    CDS    3238679    3238802    .    -    2    ID=CDS_FBgn0033159:61_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    exon    3238679    3238802    .    -    .    ID=FBgn0033159:61;Name=Dscam:61;Parent=FBtr0089020,FBtr0306769;parent_type=mRNA
2R    FlyBase    intron    3238803    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:61;Name=Dscam-in;Parent=FBtr0089020,FBtr0306769;parent_type=mRNA
2R    FlyBase    CDS    3238966    3239089    .    -    2    ID=CDS_FBgn0033159:62_704;Name=Dscam-cds;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    exon    3238966    3239089    .    -    .    ID=FBgn0033159:62;Name=Dscam:62;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    intron    3239090    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:62;Name=Dscam-in;Parent=FBtr0306778;parent_type=mRNA
2R    FlyBase    CDS    3239204    3239327    .    -    2    ID=CDS_FBgn0033159:63_583;Name=Dscam-cds;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    exon    3239204    3239327    .    -    .    ID=FBgn0033159:63;Name=Dscam:63;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3239328    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:63;Name=Dscam-in;Parent=FBtr0111069,FBtr0111089,FBtr0111096,FBtr0111056;parent_type=mRNA
2R    FlyBase    CDS    3239421    3239544    .    -    2    ID=CDS_FBgn0033159:64_583;Name=Dscam-cds;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    exon    3239421    3239544    .    -    .    ID=FBgn0033159:64;Name=Dscam:64;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    intron    3239545    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:64;Name=Dscam-in;Parent=FBtr0111055;parent_type=mRNA
2R    FlyBase    CDS    3240404    3240527    .    -    2    ID=CDS_FBgn0033159:65_583;Name=Dscam-cds;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    exon    3240404    3240527    .    -    .    ID=FBgn0033159:65;Name=Dscam:65;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    intron    3240528    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:65;Name=Dscam-in;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    CDS    3240750    3240870    .    -    2    ID=CDS_FBgn0033159:66_583;Name=Dscam-cds;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    exon    3240750    3240870    .    -    .    ID=FBgn0033159:66;Name=Dscam:66;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    intron    3240871    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:66;Name=Dscam-in;Parent=FBtr0111054;parent_type=mRNA
2R    FlyBase    CDS    3241071    3241197    .    -    2    ID=CDS_FBgn0033159:67_583;Name=Dscam-cds;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    exon    3241071    3241197    .    -    .    ID=FBgn0033159:67;Name=Dscam:67;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    intron    3241198    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:67;Name=Dscam-in;Parent=FBtr0111084;parent_type=mRNA
2R    FlyBase    CDS    3241321    3241441    .    -    2    ID=CDS_FBgn0033159:68_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    exon    3241321    3241441    .    -    .    ID=FBgn0033159:68;Name=Dscam:68;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    intron    3241442    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:68;Name=Dscam-in;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3241562    3241682    .    -    2    ID=CDS_FBgn0033159:69_704;Name=Dscam-cds;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    exon    3241562    3241682    .    -    .    ID=FBgn0033159:69;Name=Dscam:69;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3241683    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:69;Name=Dscam-in;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    CDS    3241782    3241905    .    -    2    ID=CDS_FBgn0033159:70_583;Name=Dscam-cds;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    exon    3241782    3241905    .    -    .    ID=FBgn0033159:70;Name=Dscam:70;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    intron    3241906    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:70;Name=Dscam-in;Parent=FBtr0111065,FBtr0111097,FBtr0111053;parent_type=mRNA
2R    FlyBase    CDS    3241982    3242105    .    -    2    ID=CDS_FBgn0033159:71_704;Name=Dscam-cds;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    exon    3241982    3242105    .    -    .    ID=FBgn0033159:71;Name=Dscam:71;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3242106    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:71;Name=Dscam-in;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    CDS    3242191    3242314    .    -    2    ID=CDS_FBgn0033159:72_704;Name=Dscam-cds;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    exon    3242191    3242314    .    -    .    ID=FBgn0033159:72;Name=Dscam:72;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    intron    3242315    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:72;Name=Dscam-in;Parent=FBtr0306768;parent_type=mRNA
2R    FlyBase    CDS    3242387    3242510    .    -    2    ID=CDS_FBgn0033159:73_583;Name=Dscam-cds;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    exon    3242387    3242510    .    -    .    ID=FBgn0033159:73;Name=Dscam:73;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    intron    3242511    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:73;Name=Dscam-in;Parent=FBtr0111052;parent_type=mRNA
2R    FlyBase    CDS    3242592    3242715    .    -    2    ID=CDS_FBgn0033159:74_583;Name=Dscam-cds;Parent=FBtr0111050;parent_type=mRNA
2R    FlyBase    CDS    3242592    3242715    .    -    2    ID=CDS_FBgn0033159:74_704;Name=Dscam-cds;Parent=FBtr0306786;parent_type=mRNA
2R    FlyBase    exon    3242592    3242715    .    -    .    ID=FBgn0033159:74;Name=Dscam:74;Parent=FBtr0111050,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3242716    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:74;Name=Dscam-in;Parent=FBtr0111050,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3242796    3242919    .    -    2    ID=CDS_FBgn0033159:75_583;Name=Dscam-cds;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    exon    3242796    3242919    .    -    .    ID=FBgn0033159:75;Name=Dscam:75;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    intron    3242920    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:75;Name=Dscam-in;Parent=FBtr0111088;parent_type=mRNA
2R    FlyBase    CDS    3243191    3243314    .    -    2    ID=CDS_FBgn0033159:76_583;Name=Dscam-cds;Parent=FBtr0111102;parent_type=mRNA
2R    FlyBase    CDS    3243191    3243314    .    -    2    ID=CDS_FBgn0033159:76_704;Name=Dscam-cds;Parent=FBtr0306772;parent_type=mRNA
2R    FlyBase    exon    3243191    3243314    .    -    .    ID=FBgn0033159:76;Name=Dscam:76;Parent=FBtr0111102,FBtr0306772;parent_type=mRNA
2R    FlyBase    intron    3243315    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:76;Name=Dscam-in;Parent=FBtr0111102,FBtr0306772;parent_type=mRNA
2R    FlyBase    CDS    3243406    3243529    .    -    2    ID=CDS_FBgn0033159:77_583;Name=Dscam-cds;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    exon    3243406    3243529    .    -    .    ID=FBgn0033159:77;Name=Dscam:77;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    intron    3243530    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:77;Name=Dscam-in;Parent=FBtr0111094;parent_type=mRNA
2R    FlyBase    CDS    3243612    3243735    .    -    2    ID=CDS_FBgn0033159:78_583;Name=Dscam-cds;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    exon    3243612    3243735    .    -    .    ID=FBgn0033159:78;Name=Dscam:78;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    intron    3243736    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:78;Name=Dscam-in;Parent=FBtr0111080;parent_type=mRNA
2R    FlyBase    CDS    3243822    3243945    .    -    2    ID=CDS_FBgn0033159:79_583;Name=Dscam-cds;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    exon    3243822    3243945    .    -    .    ID=FBgn0033159:79;Name=Dscam:79;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    intron    3243946    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:79;Name=Dscam-in;Parent=FBtr0111049;parent_type=mRNA
2R    FlyBase    CDS    3244032    3244155    .    -    2    ID=CDS_FBgn0033159:80_583;Name=Dscam-cds;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    exon    3244032    3244155    .    -    .    ID=FBgn0033159:80;Name=Dscam:80;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    intron    3244156    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:80;Name=Dscam-in;Parent=FBtr0111061;parent_type=mRNA
2R    FlyBase    CDS    3244250    3244373    .    -    2    ID=CDS_FBgn0033159:81_583;Name=Dscam-cds;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    exon    3244250    3244373    .    -    .    ID=FBgn0033159:81;Name=Dscam:81;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    intron    3244374    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:81;Name=Dscam-in;Parent=FBtr0111101,FBtr0111060,FBtr0111051;parent_type=mRNA
2R    FlyBase    CDS    3244478    3244601    .    -    2    ID=CDS_FBgn0033159:82_583;Name=Dscam-cds;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3244478    3244601    .    -    .    ID=FBgn0033159:82;Name=Dscam:82;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3244602    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:82;Name=Dscam-in;Parent=FBtr0111074,FBtr0111087,FBtr0111059;parent_type=mRNA
2R    FlyBase    CDS    3244688    3244811    .    -    2    ID=CDS_FBgn0033159:83_704;Name=Dscam-cds;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    exon    3244688    3244811    .    -    .    ID=FBgn0033159:83;Name=Dscam:83;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3244812    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:83;Name=Dscam-in;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3244905    3245028    .    -    2    ID=CDS_FBgn0033159:84_583;Name=Dscam-cds;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    exon    3244905    3245028    .    -    .    ID=FBgn0033159:84;Name=Dscam:84;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    intron    3245029    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:84;Name=Dscam-in;Parent=FBtr0111068;parent_type=mRNA
2R    FlyBase    CDS    3245096    3245219    .    -    2    ID=CDS_FBgn0033159:85_583;Name=Dscam-cds;Parent=FBtr0111073;parent_type=mRNA
2R    FlyBase    CDS    3245096    3245219    .    -    2    ID=CDS_FBgn0033159:85_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    exon    3245096    3245219    .    -    .    ID=FBgn0033159:85;Name=Dscam:85;Parent=FBtr0111073,FBtr0306771;parent_type=mRNA
2R    FlyBase    intron    3245220    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:85;Name=Dscam-in;Parent=FBtr0111073,FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3245294    3245420    .    -    2    ID=CDS_FBgn0033159:86_583;Name=Dscam-cds;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    exon    3245294    3245420    .    -    .    ID=FBgn0033159:86;Name=Dscam:86;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    intron    3245421    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:86;Name=Dscam-in;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    CDS    3245508    3245634    .    -    2    ID=CDS_FBgn0033159:87_704;Name=Dscam-cds;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    exon    3245508    3245634    .    -    .    ID=FBgn0033159:87;Name=Dscam:87;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3245635    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:87;Name=Dscam-in;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    CDS    3245721    3245844    .    -    2    ID=CDS_FBgn0033159:88_583;Name=Dscam-cds;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    exon    3245721    3245844    .    -    .    ID=FBgn0033159:88;Name=Dscam:88;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    intron    3245845    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:88;Name=Dscam-in;Parent=FBtr0111079;parent_type=mRNA
2R    FlyBase    CDS    3245934    3246051    .    -    2    ID=CDS_FBgn0033159:89_704;Name=Dscam-cds;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    exon    3245934    3246051    .    -    .    ID=FBgn0033159:89;Name=Dscam:89;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    intron    3246052    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:89;Name=Dscam-in;Parent=FBtr0306773;parent_type=mRNA
2R    FlyBase    CDS    3246109    3246232    .    -    2    ID=CDS_FBgn0033159:90_583;Name=Dscam-cds;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    exon    3246109    3246232    .    -    .    ID=FBgn0033159:90;Name=Dscam:90;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    intron    3246233    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:90;Name=Dscam-in;Parent=FBtr0111078;parent_type=mRNA
2R    FlyBase    CDS    3246312    3246435    .    -    2    ID=CDS_FBgn0033159:91_583;Name=Dscam-cds;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    exon    3246312    3246435    .    -    .    ID=FBgn0033159:91;Name=Dscam:91;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    intron    3246436    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:91;Name=Dscam-in;Parent=FBtr0111077;parent_type=mRNA
2R    FlyBase    CDS    3246521    3246644    .    -    2    ID=CDS_FBgn0033159:92_583;Name=Dscam-cds;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    exon    3246521    3246644    .    -    .    ID=FBgn0033159:92;Name=Dscam:92;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    intron    3246645    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:92;Name=Dscam-in;Parent=FBtr0111083;parent_type=mRNA
2R    FlyBase    CDS    3246727    3246850    .    -    2    ID=CDS_FBgn0033159:93_583;Name=Dscam-cds;Parent=FBtr0111072,FBtr0111058;parent_type=mRNA
2R    FlyBase    CDS    3246727    3246850    .    -    2    ID=CDS_FBgn0033159:93_704;Name=Dscam-cds;Parent=FBtr0306784;parent_type=mRNA
2R    FlyBase    exon    3246727    3246850    .    -    .    ID=FBgn0033159:93;Name=Dscam:93;Parent=FBtr0111072,FBtr0111058,FBtr0306784;parent_type=mRNA
2R    FlyBase    intron    3246851    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:93;Name=Dscam-in;Parent=FBtr0111072,FBtr0111058,FBtr0306784;parent_type=mRNA
2R    FlyBase    CDS    3246947    3247070    .    -    2    ID=CDS_FBgn0033159:94_583;Name=Dscam-cds;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    exon    3246947    3247070    .    -    .    ID=FBgn0033159:94;Name=Dscam:94;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    intron    3247071    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:94;Name=Dscam-in;Parent=FBtr0111064,FBtr0111100,FBtr0111057;parent_type=mRNA
2R    FlyBase    CDS    3247172    3247295    .    -    2    ID=CDS_FBgn0033159:95_583;Name=Dscam-cds;Parent=FBtr0111082;parent_type=mRNA
2R    FlyBase    CDS    3247172    3247295    .    -    2    ID=CDS_FBgn0033159:95_704;Name=Dscam-cds;Parent=FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3247172    3247295    .    -    .    ID=FBgn0033159:95;Name=Dscam:95;Parent=FBtr0111082,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3247296    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:95;Name=Dscam-in;Parent=FBtr0111082,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3247380    3247512    .    -    2    ID=CDS_FBgn0033159:96_583;Name=Dscam-cds;Parent=FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3247380    3247512    .    -    2    ID=CDS_FBgn0033159:96_704;Name=Dscam-cds;Parent=FBtr0306767;parent_type=mRNA
2R    FlyBase    exon    3247380    3247512    .    -    .    ID=FBgn0033159:96;Name=Dscam:96;Parent=FBtr0111063,FBtr0306767;parent_type=mRNA
2R    FlyBase    intron    3247513    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:96;Name=Dscam-in;Parent=FBtr0111063,FBtr0306767;parent_type=mRNA
2R    FlyBase    CDS    3247587    3247710    .    -    2    ID=CDS_FBgn0033159:97_547;Name=Dscam-cds;Parent=FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3247587    3247710    .    -    2    ID=CDS_FBgn0033159:97_583;Name=Dscam-cds;Parent=FBtr0111092;parent_type=mRNA
2R    FlyBase    exon    3247587    3247710    .    -    .    ID=FBgn0033159:97;Name=Dscam:97;Parent=FBtr0111092,FBtr0306770;parent_type=mRNA
2R    FlyBase    intron    3247711    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:97;Name=Dscam-in;Parent=FBtr0111092,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3247796    3247919    .    -    2    ID=CDS_FBgn0033159:98_583;Name=Dscam-cds;Parent=FBtr0111081;parent_type=mRNA
2R    FlyBase    CDS    3247796    3247919    .    -    2    ID=CDS_FBgn0033159:98_704;Name=Dscam-cds;Parent=FBtr0306785;parent_type=mRNA
2R    FlyBase    exon    3247796    3247919    .    -    .    ID=FBgn0033159:98;Name=Dscam:98;Parent=FBtr0111081,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3247920    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:98;Name=Dscam-in;Parent=FBtr0111081,FBtr0306785;parent_type=mRNA
2R    FlyBase    CDS    3248031    3248154    .    -    2    ID=CDS_FBgn0033159:99_583;Name=Dscam-cds;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    exon    3248031    3248154    .    -    .    ID=FBgn0033159:99;Name=Dscam:99;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    intron    3248155    3249222    .    -    .    ID=intron_FBgn0033159:100_FBgn0033159:99;Name=Dscam-in;Parent=FBtr0111076;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3249223    3249372    .    -    2    ID=CDS_FBgn0033159:100_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3249223    3249372    .    -    .    ID=FBgn0033159:100;Name=Dscam:100;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3249373    3254309    .    -    .    ID=intron_FBgn0033159:112_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111078,FBtr0089016,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3249373    3253970    .    -    .    ID=intron_FBgn0033159:111_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0089018,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3249373    3253447    .    -    .    ID=intron_FBgn0033159:110_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0089020,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3249373    3252816    .    -    .    ID=intron_FBgn0033159:109_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111097,FBtr0111098,FBtr0111099,FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3249373    3252522    .    -    .    ID=intron_FBgn0033159:108_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3249373    3252236    .    -    .    ID=intron_FBgn0033159:107_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3249373    3251960    .    -    .    ID=intron_FBgn0033159:106_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3249373    3251682    .    -    .    ID=intron_FBgn0033159:105_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0089019,FBtr0111063,FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3249373    3251385    .    -    .    ID=intron_FBgn0033159:104_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111067,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3249373    3251099    .    -    .    ID=intron_FBgn0033159:103_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068,FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3249373    3250836    .    -    .    ID=intron_FBgn0033159:102_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111075,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3249373    3250528    .    -    .    ID=intron_FBgn0033159:101_FBgn0033159:100;Name=Dscam-in;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073,FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3250529    3250699    .    -    2    ID=CDS_FBgn0033159:101_583;Name=Dscam-cds;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073;parent_type=mRNA
2R    FlyBase    CDS    3250529    3250699    .    -    2    ID=CDS_FBgn0033159:101_704;Name=Dscam-cds;Parent=FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    exon    3250529    3250699    .    -    .    ID=FBgn0033159:101;Name=Dscam:101;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073,FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    intron    3250700    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:101;Name=Dscam-in;Parent=FBtr0111080,FBtr0111074,FBtr0111072,FBtr0111073,FBtr0306778,FBtr0306786;parent_type=mRNA
2R    FlyBase    CDS    3250837    3250995    .    -    2    ID=CDS_FBgn0033159:102_583;Name=Dscam-cds;Parent=FBtr0111075;parent_type=mRNA
2R    FlyBase    CDS    3250837    3250995    .    -    2    ID=CDS_FBgn0033159:102_704;Name=Dscam-cds;Parent=FBtr0306777;parent_type=mRNA
2R    FlyBase    exon    3250837    3250995    .    -    .    ID=FBgn0033159:102;Name=Dscam:102;Parent=FBtr0111075,FBtr0306777;parent_type=mRNA
2R    FlyBase    intron    3250996    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:102;Name=Dscam-in;Parent=FBtr0111075,FBtr0306777;parent_type=mRNA
2R    FlyBase    CDS    3251100    3251270    .    -    2    ID=CDS_FBgn0033159:103_583;Name=Dscam-cds;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068;parent_type=mRNA
2R    FlyBase    CDS    3251100    3251270    .    -    2    ID=CDS_FBgn0033159:103_704;Name=Dscam-cds;Parent=FBtr0306776;parent_type=mRNA
2R    FlyBase    exon    3251100    3251270    .    -    .    ID=FBgn0033159:103;Name=Dscam:103;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068,FBtr0306776;parent_type=mRNA
2R    FlyBase    intron    3251271    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:103;Name=Dscam-in;Parent=FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111068,FBtr0306776;parent_type=mRNA
2R    FlyBase    CDS    3251386    3251553    .    -    2    ID=CDS_FBgn0033159:104_583;Name=Dscam-cds;Parent=FBtr0111067;parent_type=mRNA
2R    FlyBase    CDS    3251386    3251553    .    -    2    ID=CDS_FBgn0033159:104_704;Name=Dscam-cds;Parent=FBtr0306780;parent_type=mRNA
2R    FlyBase    exon    3251386    3251553    .    -    .    ID=FBgn0033159:104;Name=Dscam:104;Parent=FBtr0111067,FBtr0306780;parent_type=mRNA
2R    FlyBase    intron    3251554    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:104;Name=Dscam-in;Parent=FBtr0111067,FBtr0306780;parent_type=mRNA
2R    FlyBase    CDS    3251683    3251853    .    -    2    ID=CDS_FBgn0033159:105_577;Name=Dscam-cds;Parent=FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3251683    3251853    .    -    2    ID=CDS_FBgn0033159:105_583;Name=Dscam-cds;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3251683    3251853    .    -    2    ID=CDS_FBgn0033159:105_704;Name=Dscam-cds;Parent=FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    exon    3251683    3251853    .    -    .    ID=FBgn0033159:105;Name=Dscam:105;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0089019,FBtr0111063,FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    intron    3251854    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:105;Name=Dscam-in;Parent=FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0089019,FBtr0111063,FBtr0306779,FBtr0306785;parent_type=mRNA
2R    FlyBase    CDS    3251961    3252122    .    -    2    ID=CDS_FBgn0033159:106_583;Name=Dscam-cds;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    exon    3251961    3252122    .    -    .    ID=FBgn0033159:106;Name=Dscam:106;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    intron    3252123    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:106;Name=Dscam-in;Parent=FBtr0111076,FBtr0111077,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111057,FBtr0111058,FBtr0111059;parent_type=mRNA
2R    FlyBase    CDS    3252237    3252398    .    -    2    ID=CDS_FBgn0033159:107_583;Name=Dscam-cds;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    exon    3252237    3252398    .    -    .    ID=FBgn0033159:107;Name=Dscam:107;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    intron    3252399    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:107;Name=Dscam-in;Parent=FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056;parent_type=mRNA
2R    FlyBase    CDS    3252523    3252684    .    -    2    ID=CDS_FBgn0033159:108_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050;parent_type=mRNA
2R    FlyBase    CDS    3252523    3252684    .    -    2    ID=CDS_FBgn0033159:108_704;Name=Dscam-cds;Parent=FBtr0306774;parent_type=mRNA
2R    FlyBase    exon    3252523    3252684    .    -    .    ID=FBgn0033159:108;Name=Dscam:108;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0306774;parent_type=mRNA
2R    FlyBase    intron    3252685    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:108;Name=Dscam-in;Parent=FBtr0111079,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0306774;parent_type=mRNA
2R    FlyBase    CDS    3252817    3252987    .    -    2    ID=CDS_FBgn0033159:109_583;Name=Dscam-cds;Parent=FBtr0111097,FBtr0111098,FBtr0111099;parent_type=mRNA
2R    FlyBase    CDS    3252817    3252987    .    -    2    ID=CDS_FBgn0033159:109_704;Name=Dscam-cds;Parent=FBtr0306775;parent_type=mRNA
2R    FlyBase    exon    3252817    3252987    .    -    .    ID=FBgn0033159:109;Name=Dscam:109;Parent=FBtr0111097,FBtr0111098,FBtr0111099,FBtr0306775;parent_type=mRNA
2R    FlyBase    intron    3252988    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:109;Name=Dscam-in;Parent=FBtr0111097,FBtr0111098,FBtr0111099,FBtr0306775;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_547;Name=Dscam-cds;Parent=FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_583;Name=Dscam-cds;Parent=FBtr0111092,FBtr0111094,FBtr0111096;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3253448    3253609    .    -    2    ID=CDS_FBgn0033159:110_704;Name=Dscam-cds;Parent=FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    exon    3253448    3253609    .    -    .    ID=FBgn0033159:110;Name=Dscam:110;Parent=FBtr0089020,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    intron    3253610    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:110;Name=Dscam-in;Parent=FBtr0089020,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306782;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_547;Name=Dscam-cds;Parent=FBtr0306769;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_577;Name=Dscam-cds;Parent=FBtr0089018;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_583;Name=Dscam-cds;Parent=FBtr0111087,FBtr0111088,FBtr0111089;parent_type=mRNA
2R    FlyBase    CDS    3253971    3254132    .    -    2    ID=CDS_FBgn0033159:111_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3253971    3254132    .    -    .    ID=FBgn0033159:111;Name=Dscam:111;Parent=FBtr0089018,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3254133    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:111;Name=Dscam-in;Parent=FBtr0089018,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306773,FBtr0306781,FBtr0306783,FBtr0306784,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3254310    3254471    .    -    2    ID=CDS_FBgn0033159:112_577;Name=Dscam-cds;Parent=FBtr0089016;parent_type=mRNA
2R    FlyBase    CDS    3254310    3254471    .    -    2    ID=CDS_FBgn0033159:112_583;Name=Dscam-cds;Parent=FBtr0111078,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    exon    3254310    3254471    .    -    .    ID=FBgn0033159:112;Name=Dscam:112;Parent=FBtr0111078,FBtr0089016,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    intron    3254472    3255892    .    -    .    ID=intron_FBgn0033159:113_FBgn0033159:112;Name=Dscam-in;Parent=FBtr0111078,FBtr0089016,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3255893    3256237    .    -    2    ID=CDS_FBgn0033159:113_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3255893    3256237    .    -    .    ID=FBgn0033159:113;Name=Dscam:113;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3256238    3265038    .    -    .    ID=intron_FBgn0033159:114_FBgn0033159:113;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3265039    3265219    .    -    .    ID=FBgn0033159:114;Name=Dscam:114;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_547;Name=Dscam-cds;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_577;Name=Dscam-cds;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_583;Name=Dscam-cds;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_584;Name=Dscam-cds;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_698;Name=Dscam-cds;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    CDS    3265039    3265099    .    -    0    ID=CDS_FBgn0033159:114_704;Name=Dscam-cds;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_547;Name=Dscam-u5;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_577;Name=Dscam-u5;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_583;Name=Dscam-u5;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_584;Name=Dscam-u5;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_698;Name=Dscam-u5;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3265100    3265219    .    -    .    ID=five_prime_UTR_FBgn0033159:114_704;Name=Dscam-u5;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    intron    3265220    3268975    .    -    .    ID=intron_FBgn0033159:115_FBgn0033159:114;Name=Dscam-in;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    exon    3268976    3269374    .    -    .    ID=FBgn0033159:115;Name=Dscam:115;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0089016,FBtr0089018,FBtr0089019,FBtr0089020,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063,FBtr0306767,FBtr0306768,FBtr0306769,FBtr0306770,FBtr0306771,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_547;Name=Dscam-u5;Parent=FBtr0306769,FBtr0306770;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_577;Name=Dscam-u5;Parent=FBtr0089016,FBtr0089018,FBtr0089019;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_583;Name=Dscam-u5;Parent=FBtr0111079,FBtr0111080,FBtr0111074,FBtr0111075,FBtr0111076,FBtr0111077,FBtr0111078,FBtr0111069,FBtr0111070,FBtr0111071,FBtr0111072,FBtr0111073,FBtr0111064,FBtr0111101,FBtr0111065,FBtr0111066,FBtr0111067,FBtr0111068,FBtr0111060,FBtr0111061,FBtr0111102,FBtr0111062,FBtr0111081,FBtr0111082,FBtr0111083,FBtr0111084,FBtr0111085,FBtr0111087,FBtr0111088,FBtr0111089,FBtr0111092,FBtr0111094,FBtr0111096,FBtr0111097,FBtr0111098,FBtr0111099,FBtr0111100,FBtr0111049,FBtr0111050,FBtr0111051,FBtr0111052,FBtr0111053,FBtr0111054,FBtr0111055,FBtr0111056,FBtr0111057,FBtr0111058,FBtr0111059,FBtr0111063;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_584;Name=Dscam-u5;Parent=FBtr0089020;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_698;Name=Dscam-u5;Parent=FBtr0306771;parent_type=mRNA
2R    FlyBase    five_prime_UTR    3268976    3269374    .    -    .    ID=five_prime_UTR_FBgn0033159:115_704;Name=Dscam-u5;Parent=FBtr0306767,FBtr0306768,FBtr0306772,FBtr0306773,FBtr0306774,FBtr0306775,FBtr0306776,FBtr0306777,FBtr0306778,FBtr0306779,FBtr0306780,FBtr0306781,FBtr0306782,FBtr0306783,FBtr0306784,FBtr0306785,FBtr0306786,FBtr0306787;parent_type=mRNA
###
3R    FlyBase    gene    14982962    14985924    .    +    .    ID=FBgn0002962;Name=nos;fullname=nanos;Alias=FBgn0010916,FBgn0011347,FBgn0011364,FBgn0026836,DRONANOS,NANOS,l(3)j3B6,nNOS,l(3)j4B6,l(3)07117,CG5637,NOS,Nos,Dronanos,Nanos,nanos;Ontology_term=SO:0000010,SO:0000087,GO:0007283,GO:0005515,GO:0048813,GO:0007281,GO:0008354,GO:0045495,GO:0008595,GO:0007314,GO:0042078,GO:0001709,GO:0048134,GO:0016321,GO:0006378,GO:0045892,GO:0007280,GO:0017148,GO:0048477,GO:0043066,GO:0030718,GO:0003723,GO:0008270;Dbxref=FlyBase:FBan0005637,FlyBase_Annotation_IDs:CG5637,GB_protein:AAF55641,GB:AA978726,GB:AC006091,GB:AF067873,GB:AQ034086,GB:AW943146,GB:AY075406,GB_protein:AAL68233,GB:CL705877,GB:CZ467207,GB:CZ467208,GB:G00548,GB:M72421,GB_protein:AAA28715,Rfam:RF00161,UniProt/Swiss-Prot:P25724,UniProt/TrEMBL:Q8SY38,INTERPRO:IPR008705,INTERPRO:IPR001878,INTERPRO:IPR024161,EntrezGene:42297,InterologFinder:42297,BIOGRID:67295,FlyAtlas:CG5637-RA,GenomeRNAi:42297,INTERACTIVEFLY:/torstoll/nanos1.htm;gbunit=AE014297;derived_computed_cyto=91F4-91F4
3R    FlyBase    mRNA    14982962    14985924    .    +    .    ID=FBtr0083732;Name=nos-RA;Parent=FBgn0002962;Dbxref=FlyBase_Annotation_IDs:CG5637-RA,REFSEQ:NM_057310;score_text=Strongly Supported;score=15
3R    FlyBase    exon    14982962    14983694    .    +    .    ID=FBgn0002962:1;Name=nos:1;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    five_prime_UTR    14982962    14983222    .    +    .    ID=five_prime_UTR_FBgn0002962:1_14;Name=nos-u5;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    CDS    14983223    14983694    .    +    0    ID=CDS_FBgn0002962:1_14;Name=nos-cds;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    intron    14983695    14984237    .    +    .    ID=intron_FBgn0002962:1_FBgn0002962:2;Name=nos-in;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    CDS    14984238    14984707    .    +    2    ID=CDS_FBgn0002962:2_14;Name=nos-cds;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    exon    14984238    14984707    .    +    .    ID=FBgn0002962:2;Name=nos:2;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    intron    14984708    14984780    .    +    .    ID=intron_FBgn0002962:2_FBgn0002962:3;Name=nos-in;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    exon    14984781    14985924    .    +    .    ID=FBgn0002962:3;Name=nos:3;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    CDS    14984781    14985044    .    +    0    ID=CDS_FBgn0002962:3_14;Name=nos-cds;Parent=FBtr0083732;parent_type=mRNA
3R    FlyBase    three_prime_UTR    14985045    14985924    .    +    .    ID=three_prime_UTR_FBgn0002962:3_14;Name=nos-u3;Parent=FBtr0083732;parent_type=mRNA
4    TestData    gene    5000    6010    .    +    0    ID=TestBadGene1
4    TestData    mRNA    5000    6010    .    +    0    ID=TestBadTranscript1;Parent=TestBadGene1
4    TestData    exon    5000    5010    .    +    0    ID=exon_test11;Parent=TestBadTranscript1;parent_type=mRNA
4    TestData    exon    5500    6010    .    -    0    ID=exon_test12;Parent=TestBadTranscript1;parent_type=mRNA""".replace(
    "    ", "\t"
)

# cat dmel-all-no-analysis-r5.43.gff | grep FBgn0033159 |\
# grep -v "RNAi_reagent" | grep -v "region" |\
# grep -v "mE_Transcription_Start_Sites" | grep -v "regulatory_region" |\
# grep -v "transposable_element_insertion_site" | grep -v "orthologous_to" |\
# cut -f9 | awk 'BEGIN { FS = ";" }
#                { print $1 }' | sed -e "s/ID=//g" >dscam_ids.txt
# # { for id in $(cat dscam_ids.txt); do grep "Parent=${id}" dmel-all-no-analysis-no-fasta-r5.54.gff; done } >dscam_exon_cds_intron.txt

tmp = GFF3_TO_TRANSCRIPTS_SORTED.split("\n")
shuffle(tmp)
GFF3_TO_TRANSCRIPTS_UNSORTED = "\n".join(tmp)

GTF2_TO_TRANSCRIPTS_SORTED = """2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3221893    3222171    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3247587    3247710    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3221893    3222171    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3247587    3247710    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111092"; Name "Dscam-RR"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RR%2CREFSEQ%3ANM_001043028"; ID "FBtr0111092";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3241321    3241441    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; Alias "CG17800-RA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3241321    3241441    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089016"; Name "Dscam-RA"; score_text "Strongly Supported"; Alias "CG17800-RA"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RA%2CREFSEQ%3ANM_165514"; ID "FBtr0089016";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3241321    3241441    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; Alias "CG17800-RC"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3241321    3241441    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089018"; Name "Dscam-RC"; score_text "Strongly Supported"; Alias "CG17800-RC"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RC%2CREFSEQ%3ANM_165513"; ID "FBtr0089018";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3234512    3234799    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3243822    3243945    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3234512    3234799    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3243822    3243945    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111049"; Name "Dscam-RAA"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAA%2CREFSEQ%3ANM_001043036"; ID "FBtr0111049";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3228450    3228737    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3241782    3241905    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3228450    3228737    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3241782    3241905    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111053"; Name "Dscam-RAE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAE%2CREFSEQ%3ANM_001043064"; ID "FBtr0111053";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3226146    3226433    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3239204    3239327    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3226146    3226433    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3239204    3239327    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111056"; Name "Dscam-RAH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAH%2CREFSEQ%3ANM_001043048"; ID "FBtr0111056";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3226908    3227195    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3244250    3244373    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3226908    3227195    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3244250    3244373    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111060"; Name "Dscam-RAL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAL%2CREFSEQ%3ANM_001043014"; ID "FBtr0111060";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3227288    3227575    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3237879    3238002    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3227288    3227575    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3237879    3238002    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111085"; Name "Dscam-RJ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RJ%2CREFSEQ%3ANM_001043058"; ID "FBtr0111085";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3227660    3227947    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3239204    3239327    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3227660    3227947    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3239204    3239327    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111096"; Name "Dscam-RV"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RV%2CREFSEQ%3ANM_001043018"; ID "FBtr0111096";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3232169    3232459    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3237106    3237226    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3232169    3232459    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3237106    3237226    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306783"; Name "Dscam-RBS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBS"; ID "FBtr0306783";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3226533    3226820    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3246727    3246850    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3226533    3226820    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3246727    3246850    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306784"; Name "Dscam-RBT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBT"; ID "FBtr0306784";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3223221    3223508    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3247172    3247295    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3223221    3223508    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3247172    3247295    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306787"; Name "Dscam-RBW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBW"; ID "FBtr0306787";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3228833    3229123    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3242387    3242510    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3228833    3229123    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3242387    3242510    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111052"; Name "Dscam-RAD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAD%2CREFSEQ%3ANM_001043061"; ID "FBtr0111052";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3233381    3233671    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3246947    3247070    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3233381    3233671    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3246947    3247070    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111057"; Name "Dscam-RAI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAI%2CREFSEQ%3ANM_001043063"; ID "FBtr0111057";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3244032    3244155    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3244032    3244155    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111061"; Name "Dscam-RAM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAM%2CREFSEQ%3ANM_001043057"; ID "FBtr0111061";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3230837    3231127    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3246521    3246644    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3230837    3231127    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3246521    3246644    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111083"; Name "Dscam-RH"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RH%2CREFSEQ%3ANM_001043047"; ID "FBtr0111083";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3227288    3227575    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3241071    3241197    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3227288    3227575    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3241071    3241197    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111084"; Name "Dscam-RI"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RI%2CREFSEQ%3ANM_001043022"; ID "FBtr0111084";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3232169    3232459    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3242796    3242919    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3232169    3232459    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3242796    3242919    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111088"; Name "Dscam-RM"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RM%2CREFSEQ%3ANM_001043044"; ID "FBtr0111088";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3232169    3232459    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3246947    3247070    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3232169    3232459    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3246947    3247070    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111100"; Name "Dscam-RZ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RZ%2CREFSEQ%3ANM_001043052"; ID "FBtr0111100";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3225755    3226042    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3236898    3237018    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3225755    3226042    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3236898    3237018    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111066"; Name "Dscam-RAT"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAT%2CREFSEQ%3ANM_001043021"; ID "FBtr0111066";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3234147    3234434    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3240404    3240527    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3251386    3251553    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3234147    3234434    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3240404    3240527    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3251386    3251553    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111067"; Name "Dscam-RAU"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAU%2CREFSEQ%3ANM_001043037"; ID "FBtr0111067";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3224999    3225286    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3236898    3237018    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3251100    3251270    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3224999    3225286    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3236898    3237018    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3251100    3251270    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111071"; Name "Dscam-RAY"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAY%2CREFSEQ%3ANM_001043032"; ID "FBtr0111071";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3241321    3241441    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; Alias "CG17800-RD"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3241321    3241441    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089019"; Name "Dscam-RD"; score_text "Strongly Supported"; Alias "CG17800-RD"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RD%2CREFSEQ%3ANM_078925"; ID "FBtr0089019";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3225377    3225664    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3238086    3238209    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3251100    3251270    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3225377    3225664    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3238086    3238209    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3251100    3251270    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111070"; Name "Dscam-RAX"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAX%2CREFSEQ%3ANM_001043015"; ID "FBtr0111070";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3233774    3234061    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3241782    3241905    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3252817    3252987    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3233774    3234061    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3241782    3241905    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3252817    3252987    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111097"; Name "Dscam-RW"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RW%2CREFSEQ%3ANM_001043056"; ID "FBtr0111097";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3224036    3224323    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3241982    3242105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3251100    3251270    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3224036    3224323    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3241982    3242105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3251100    3251270    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306776"; Name "Dscam-RBL"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBL"; ID "FBtr0306776";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3226146    3226433    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3238463    3238586    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3226146    3226433    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3238463    3238586    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306779"; Name "Dscam-RBO"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBO"; ID "FBtr0306779";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3230837    3231127    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3238287    3238410    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3251386    3251553    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3230837    3231127    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3238287    3238410    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3251386    3251553    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306780"; Name "Dscam-RBP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBP"; ID "FBtr0306780";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3230088    3230386    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3242191    3242314    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; Alias "FBtr0111103%2CCG17800-RN"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3230088    3230386    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3242191    3242314    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306768"; Name "Dscam-RBY"; score_text "Strongly Supported"; Alias "FBtr0111103%2CCG17800-RN"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBY"; ID "FBtr0306768";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3230088    3230386    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3245096    3245219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; Alias "FBtr0111093%2CCG17800-RS"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3230088    3230386    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3245096    3245219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306771"; Name "Dscam-RCB"; score_text "Strongly Supported"; Alias "FBtr0111093%2CCG17800-RS"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RCB"; ID "FBtr0306771";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3228833    3229123    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3246947    3247070    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3228833    3229123    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3246947    3247070    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111064"; Name "Dscam-RAQ"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAQ%2CREFSEQ%3ANM_001043023"; ID "FBtr0111064";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3244478    3244601    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3250529    3250699    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3231784    3232074    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3244478    3244601    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3250529    3250699    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111074"; Name "Dscam-RBB"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBB%2CREFSEQ%3ANM_001043020"; ID "FBtr0111074";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3234897    3235202    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3238679    3238802    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; Alias "CG17800-RE"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3234897    3235202    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3238679    3238802    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3253448    3253609    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0089020"; Name "Dscam-RE"; score_text "Strongly Supported"; Alias "CG17800-RE"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RE%2CREFSEQ%3ANM_206047"; ID "FBtr0089020";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3224999    3225286    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3247380    3247512    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3214122    3214289    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3224999    3225286    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3247380    3247512    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3251683    3251853    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111063"; Name "Dscam-RAP"; score_text "Strongly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAP%2CREFSEQ%3ANM_001043051"; ID "FBtr0111063";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3222439    3222723    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3241562    3241682    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3250837    3250995    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3222439    3222723    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3241562    3241682    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3250837    3250995    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306777"; Name "Dscam-RBM"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBM"; ID "FBtr0306777";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3230469    3230753    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3246727    3246850    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3230469    3230753    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3246727    3246850    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111058"; Name "Dscam-RAJ"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAJ%2CREFSEQ%3ANM_001043040"; ID "FBtr0111058";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3222439    3222723    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3245294    3245420    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3250837    3250995    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3222439    3222723    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3245294    3245420    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3250837    3250995    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111075"; Name "Dscam-RBC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBC%2CREFSEQ%3ANM_001043024"; ID "FBtr0111075";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3222439    3222723    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3248031    3248154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3222439    3222723    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3248031    3248154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111076"; Name "Dscam-RBD"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBD%2CREFSEQ%3ANM_001043013"; ID "FBtr0111076";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3231223    3231513    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3245934    3246051    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3231223    3231513    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3245934    3246051    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3253971    3254132    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0306773"; Name "Dscam-RBI"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBI"; ID "FBtr0306773";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3227660    3227947    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3242592    3242715    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3227660    3227947    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3242592    3242715    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111050"; Name "Dscam-RAB"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAB%2CREFSEQ%3ANM_001043050"; ID "FBtr0111050";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3228050    3228337    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3244250    3244373    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3228050    3228337    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3244250    3244373    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111051"; Name "Dscam-RAC"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAC%2CREFSEQ%3ANM_001043012"; ID "FBtr0111051";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3232566    3232856    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3240750    3240870    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3232566    3232856    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3240750    3240870    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3252237    3252398    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111054"; Name "Dscam-RAF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RAF%2CREFSEQ%3ANM_001043062"; ID "FBtr0111054";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3223221    3223508    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3246312    3246435    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3223221    3223508    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3246312    3246435    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3251961    3252122    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111077"; Name "Dscam-RBE"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBE%2CREFSEQ%3ANM_001043041"; ID "FBtr0111077";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3234147    3234434    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3246109    3246232    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3234147    3234434    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3246109    3246232    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3254310    3254471    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111078"; Name "Dscam-RBF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBF%2CREFSEQ%3ANM_001043029"; ID "FBtr0111078";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3234147    3234434    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3245721    3245844    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3265039    3265219    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3268976    3269374    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3208191    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3210281    3210461    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3210527    3210837    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3211324    3211473    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3212087    3212242    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3213657    3213869    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3216794    3216960    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3217018    3217194    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3217248    3217356    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3217422    3218105    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3218164    3218319    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3218481    3219707    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3219766    3219885    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3234147    3234434    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3235354    3235515    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3235636    3236480    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3245721    3245844    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3249223    3249372    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3252523    3252684    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3255893    3256237    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    CDS    3265039    3265099    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111079"; Name "Dscam-RBG"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RBG%2CREFSEQ%3ANM_001043016"; ID "FBtr0111079";
2R    FlyBase    exon    3207059    3208269    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111081"; Name "Dscam-RF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RF%2CREFSEQ%3ANM_001043049"; ID "FBtr0111081";
2R    FlyBase    exon    3209088    3209154    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111081"; Name "Dscam-RF"; score_text "Weakly Supported"; Dbxref "FlyBase_Annotation_IDs%3ACG17800-RF%2CREFSEQ%3ANM_001043049"; ID "FBtr0111081";
2R    FlyBase    exon    3210065    3210223    .    -    .    gene_id "FBgn0033159"; transcript_id "FBtr0111081"; Name "Dscam-RF"; score_text "Weakly Supported";