"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exec = exec;
exports.cli = cli;
/* eslint-disable @typescript-eslint/no-shadow */ // yargs
const cxapi = require("@aws-cdk/cx-api");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const cdk_toolkit_1 = require("./cdk-toolkit");
const ci_systems_1 = require("./ci-systems");
const display_version_1 = require("./display-version");
const io_host_1 = require("./io-host");
const parse_command_line_arguments_1 = require("./parse-command-line-arguments");
const platform_warnings_1 = require("./platform-warnings");
const pretty_print_error_1 = require("./pretty-print-error");
const singleton_plugin_host_1 = require("./singleton-plugin-host");
const user_configuration_1 = require("./user-configuration");
const api_private_1 = require("../../lib/api-private");
const api_1 = require("../api");
const aws_auth_1 = require("../api/aws-auth");
const bootstrap_1 = require("../api/bootstrap");
const deployments_1 = require("../api/deployments");
const hotswap_1 = require("../api/hotswap");
const context_1 = require("../commands/context");
const docs_1 = require("../commands/docs");
const doctor_1 = require("../commands/doctor");
const flags_1 = require("../commands/flags/flags");
const init_1 = require("../commands/init");
const migrate_1 = require("../commands/migrate");
const cxapp_1 = require("../cxapp");
const proxy_agent_1 = require("./proxy-agent");
const error_1 = require("./telemetry/error");
const ci_1 = require("./util/ci");
const version_1 = require("./version");
const language_1 = require("../commands/language");
if (!process.stdout.isTTY) {
    // Disable chalk color highlighting
    process.env.FORCE_COLOR = '0';
}
async function exec(args, synthesizer) {
    const argv = await (0, parse_command_line_arguments_1.parseCommandLineArguments)(args);
    argv.language = (0, language_1.getLanguageFromAlias)(argv.language) ?? argv.language;
    const cmd = argv._[0];
    // if one -v, log at a DEBUG level
    // if 2 -v, log at a TRACE level
    let ioMessageLevel = 'info';
    if (argv.verbose) {
        switch (argv.verbose) {
            case 1:
                ioMessageLevel = 'debug';
                break;
            case 2:
            default:
                ioMessageLevel = 'trace';
                break;
        }
    }
    const ioHost = io_host_1.CliIoHost.instance({
        logLevel: ioMessageLevel,
        isTTY: process.stdout.isTTY,
        isCI: Boolean(argv.ci),
        currentAction: cmd,
        stackProgress: argv.progress,
        autoRespond: argv.yes,
    }, true);
    const ioHelper = (0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction);
    // Debug should always imply tracing
    (0, aws_auth_1.setSdkTracing)(argv.debug || argv.verbose > 2);
    try {
        await (0, platform_warnings_1.checkForPlatformWarnings)(ioHelper);
    }
    catch (e) {
        await ioHost.defaults.debug(`Error while checking for platform warnings: ${e}`);
    }
    await ioHost.defaults.debug('CDK Toolkit CLI version:', (0, version_1.versionWithBuild)());
    await ioHost.defaults.debug('Command line arguments:', argv);
    const configuration = await user_configuration_1.Configuration.fromArgsAndFiles(ioHelper, {
        commandLineArguments: {
            ...argv,
            _: argv._, // TypeScript at its best
        },
    });
    // Always create and use ProxyAgent to support configuration via env vars
    const proxyAgent = await new proxy_agent_1.ProxyAgentProvider(ioHelper).create({
        proxyAddress: configuration.settings.get(['proxy']),
        caBundlePath: configuration.settings.get(['caBundlePath']),
    });
    try {
        await ioHost.startTelemetry(argv, configuration.context);
    }
    catch (e) {
        await ioHost.asIoHelper().defaults.trace(`Telemetry instantiation failed: ${e.message}`);
    }
    /**
     * The default value for displaying (and refreshing) notices on all commands.
     *
     * If the user didn't supply either `--notices` or `--no-notices`, we do
     * autodetection. The autodetection currently is: do write notices if we are
     * not on CI, or are on a CI system where we know that writing to stderr is
     * safe. We fail "closed"; that is, we decide to NOT print for unknown CI
     * systems, even though technically we maybe could.
     */
    const isSafeToWriteNotices = !(0, ci_1.isCI)() || Boolean((0, ci_systems_1.ciSystemIsStdErrSafe)());
    // Determine if notices should be displayed based on CLI args and configuration
    let shouldDisplayNotices;
    if (argv.notices !== undefined) {
        // CLI argument takes precedence
        shouldDisplayNotices = argv.notices;
    }
    else {
        // Fall back to configuration file setting, then autodetection
        const configNotices = configuration.settings.get(['notices']);
        if (configNotices !== undefined) {
            // Consider string "false" to be falsy in this context
            shouldDisplayNotices = configNotices !== 'false' && Boolean(configNotices);
        }
        else {
            // Default autodetection behavior
            shouldDisplayNotices = isSafeToWriteNotices;
        }
    }
    // Notices either go to stderr, or nowhere
    ioHost.noticesDestination = shouldDisplayNotices ? 'stderr' : 'drop';
    const notices = api_1.Notices.create({
        ioHost,
        context: configuration.context,
        output: configuration.settings.get(['outdir']),
        httpOptions: { agent: proxyAgent },
        cliVersion: (0, version_1.versionNumber)(),
    });
    const refreshNotices = (async () => {
        // the cdk notices command has it's own refresh
        if (shouldDisplayNotices && cmd !== 'notices') {
            try {
                return await notices.refresh();
            }
            catch (e) {
                await ioHelper.defaults.debug(`Could not refresh notices: ${e}`);
            }
        }
    })();
    const sdkProvider = await aws_auth_1.SdkProvider.withAwsCliCompatibleDefaults({
        ioHelper,
        requestHandler: (0, aws_auth_1.sdkRequestHandler)(proxyAgent),
        logger: new aws_auth_1.IoHostSdkLogger((0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction)),
        pluginHost: singleton_plugin_host_1.GLOBAL_PLUGIN_HOST,
    }, configuration.settings.get(['profile']));
    try {
        await ioHost.telemetry?.attachRegion(sdkProvider.defaultRegion);
    }
    catch (e) {
        await ioHost.asIoHelper().defaults.trace(`Telemetry attach region failed: ${e.message}`);
    }
    let outDirLock;
    const cloudExecutable = new cxapp_1.CloudExecutable({
        configuration,
        sdkProvider,
        synthesizer: synthesizer ??
            (async (aws, config) => {
                // Invoke 'execProgram', and copy the lock for the directory in the global
                // variable here. It will be released when the CLI exits. Locks are not re-entrant
                // so release it if we have to synthesize more than once (because of context lookups).
                await outDirLock?.release();
                const { assembly, lock } = await (0, cxapp_1.execProgram)(aws, ioHost.asIoHelper(), config);
                outDirLock = lock;
                return assembly;
            }),
        ioHelper: ioHost.asIoHelper(),
    });
    /** Function to load plug-ins, using configurations additively. */
    async function loadPlugins(...settings) {
        for (const source of settings) {
            const plugins = source.get(['plugin']) || [];
            for (const plugin of plugins) {
                await singleton_plugin_host_1.GLOBAL_PLUGIN_HOST.load(plugin, ioHost);
            }
        }
    }
    await loadPlugins(configuration.settings);
    if ((typeof cmd) !== 'string') {
        throw new toolkit_lib_1.ToolkitError(`First argument should be a string. Got: ${cmd} (${typeof cmd})`);
    }
    try {
        return await main(cmd, argv);
    }
    finally {
        // If we locked the 'cdk.out' directory, release it here.
        await outDirLock?.release();
        // Do PSAs here
        await (0, display_version_1.displayVersionMessage)(ioHelper);
        await refreshNotices;
        if (cmd === 'notices') {
            await notices.refresh({ force: true });
            await notices.display({
                includeAcknowledged: !argv.unacknowledged,
                showTotal: argv.unacknowledged,
            });
        }
        else if (shouldDisplayNotices && cmd !== 'version') {
            await notices.display();
        }
    }
    async function main(command, args) {
        ioHost.currentAction = command;
        const toolkitStackName = api_1.ToolkitInfo.determineName(configuration.settings.get(['toolkitStackName']));
        await ioHost.defaults.debug(`Toolkit stack: ${chalk.bold(toolkitStackName)}`);
        const cloudFormation = new deployments_1.Deployments({
            sdkProvider,
            toolkitStackName,
            ioHelper: (0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction),
        });
        if (args.all && args.STACKS) {
            throw new toolkit_lib_1.ToolkitError('You must either specify a list of Stacks or the `--all` argument');
        }
        args.STACKS = args.STACKS ?? (args.STACK ? [args.STACK] : []);
        args.ENVIRONMENTS = args.ENVIRONMENTS ?? [];
        const selector = {
            allTopLevel: args.all,
            patterns: args.STACKS,
        };
        const cli = new cdk_toolkit_1.CdkToolkit({
            ioHost,
            cloudExecutable,
            toolkitStackName,
            deployments: cloudFormation,
            verbose: argv.trace || argv.verbose > 0,
            ignoreErrors: argv['ignore-errors'],
            strict: argv.strict,
            configuration,
            sdkProvider,
        });
        switch (command) {
            case 'context':
                ioHost.currentAction = 'context';
                return (0, context_1.contextHandler)({
                    ioHelper,
                    context: configuration.context,
                    clear: argv.clear,
                    json: argv.json,
                    force: argv.force,
                    reset: argv.reset,
                });
            case 'docs':
            case 'doc':
                ioHost.currentAction = 'docs';
                return (0, docs_1.docs)({
                    ioHelper,
                    browser: configuration.settings.get(['browser']),
                });
            case 'doctor':
                ioHost.currentAction = 'doctor';
                return (0, doctor_1.doctor)({
                    ioHelper,
                });
            case 'ls':
            case 'list':
                ioHost.currentAction = 'list';
                return cli.list(args.STACKS, {
                    long: args.long,
                    json: argv.json,
                    showDeps: args.showDependencies,
                });
            case 'diff':
                ioHost.currentAction = 'diff';
                const enableDiffNoFail = isFeatureEnabled(configuration, cxapi.ENABLE_DIFF_NO_FAIL_CONTEXT);
                return cli.diff({
                    stackNames: args.STACKS,
                    exclusively: args.exclusively,
                    templatePath: args.template,
                    strict: args.strict,
                    contextLines: args.contextLines,
                    securityOnly: args.securityOnly,
                    fail: args.fail != null ? args.fail : !enableDiffNoFail,
                    compareAgainstProcessedTemplate: args.processed,
                    quiet: args.quiet,
                    changeSet: args['change-set'],
                    toolkitStackName: toolkitStackName,
                    importExistingResources: args.importExistingResources,
                    includeMoves: args['include-moves'],
                });
            case 'drift':
                ioHost.currentAction = 'drift';
                return cli.drift({
                    selector,
                    fail: args.fail,
                });
            case 'refactor':
                if (!configuration.settings.get(['unstable']).includes('refactor')) {
                    throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'refactor\' is unstable. It must be opted in via \'--unstable\', e.g. \'cdk refactor --unstable=refactor\'');
                }
                ioHost.currentAction = 'refactor';
                return cli.refactor({
                    dryRun: args.dryRun,
                    overrideFile: args.overrideFile,
                    revert: args.revert,
                    stacks: selector,
                    additionalStackNames: arrayFromYargs(args.additionalStackName ?? []),
                    force: args.force ?? false,
                    roleArn: args.roleArn,
                });
            case 'bootstrap':
                ioHost.currentAction = 'bootstrap';
                const source = await determineBootstrapVersion(ioHost, args);
                if (args.showTemplate) {
                    const bootstrapper = new bootstrap_1.Bootstrapper(source, (0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction));
                    return bootstrapper.showTemplate(args.json);
                }
                return cli.bootstrap(args.ENVIRONMENTS, {
                    source,
                    roleArn: args.roleArn,
                    forceDeployment: argv.force,
                    toolkitStackName: toolkitStackName,
                    execute: args.execute,
                    tags: configuration.settings.get(['tags']),
                    terminationProtection: args.terminationProtection,
                    usePreviousParameters: args['previous-parameters'],
                    parameters: {
                        bucketName: configuration.settings.get(['toolkitBucket', 'bucketName']),
                        kmsKeyId: configuration.settings.get(['toolkitBucket', 'kmsKeyId']),
                        createCustomerMasterKey: args.bootstrapCustomerKey,
                        qualifier: args.qualifier ?? configuration.context.get('@aws-cdk/core:bootstrapQualifier'),
                        publicAccessBlockConfiguration: args.publicAccessBlockConfiguration,
                        examplePermissionsBoundary: argv.examplePermissionsBoundary,
                        customPermissionsBoundary: argv.customPermissionsBoundary,
                        trustedAccounts: arrayFromYargs(args.trust),
                        trustedAccountsForLookup: arrayFromYargs(args.trustForLookup),
                        untrustedAccounts: arrayFromYargs(args.untrust),
                        cloudFormationExecutionPolicies: arrayFromYargs(args.cloudformationExecutionPolicies),
                        denyExternalId: args.denyExternalId,
                    },
                });
            case 'deploy':
                ioHost.currentAction = 'deploy';
                const parameterMap = {};
                for (const parameter of args.parameters) {
                    if (typeof parameter === 'string') {
                        const keyValue = parameter.split('=');
                        parameterMap[keyValue[0]] = keyValue.slice(1).join('=');
                    }
                }
                if (args.execute !== undefined && args.method !== undefined) {
                    throw new toolkit_lib_1.ToolkitError('Can not supply both --[no-]execute and --method at the same time');
                }
                return cli.deploy({
                    selector,
                    exclusively: args.exclusively,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    notificationArns: args.notificationArns,
                    requireApproval: configuration.settings.get(['requireApproval']),
                    reuseAssets: args['build-exclude'],
                    tags: configuration.settings.get(['tags']),
                    deploymentMethod: determineDeploymentMethod(args, configuration),
                    force: args.force,
                    parameters: parameterMap,
                    usePreviousParameters: args['previous-parameters'],
                    outputsFile: configuration.settings.get(['outputsFile']),
                    progress: configuration.settings.get(['progress']),
                    ci: args.ci,
                    rollback: configuration.settings.get(['rollback']),
                    watch: args.watch,
                    traceLogs: args.logs,
                    concurrency: args.concurrency,
                    assetParallelism: configuration.settings.get(['assetParallelism']),
                    assetBuildTime: configuration.settings.get(['assetPrebuild'])
                        ? cdk_toolkit_1.AssetBuildTime.ALL_BEFORE_DEPLOY
                        : cdk_toolkit_1.AssetBuildTime.JUST_IN_TIME,
                    ignoreNoStacks: args.ignoreNoStacks,
                });
            case 'rollback':
                ioHost.currentAction = 'rollback';
                return cli.rollback({
                    selector,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    force: args.force,
                    validateBootstrapStackVersion: args['validate-bootstrap-version'],
                    orphanLogicalIds: args.orphan,
                });
            case 'import':
                ioHost.currentAction = 'import';
                return cli.import({
                    selector,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    deploymentMethod: {
                        method: 'change-set',
                        execute: args.execute,
                        changeSetName: args.changeSetName,
                    },
                    progress: configuration.settings.get(['progress']),
                    rollback: configuration.settings.get(['rollback']),
                    recordResourceMapping: args['record-resource-mapping'],
                    resourceMappingFile: args['resource-mapping'],
                    force: args.force,
                });
            case 'watch':
                ioHost.currentAction = 'watch';
                await cli.watch({
                    selector,
                    exclusively: args.exclusively,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    reuseAssets: args['build-exclude'],
                    deploymentMethod: determineDeploymentMethod(args, configuration, true),
                    force: args.force,
                    progress: configuration.settings.get(['progress']),
                    rollback: configuration.settings.get(['rollback']),
                    traceLogs: args.logs,
                    concurrency: args.concurrency,
                });
                return;
            case 'destroy':
                ioHost.currentAction = 'destroy';
                return cli.destroy({
                    selector,
                    exclusively: args.exclusively,
                    force: args.force,
                    roleArn: args.roleArn,
                });
            case 'gc':
                ioHost.currentAction = 'gc';
                if (!configuration.settings.get(['unstable']).includes('gc')) {
                    throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'gc\' is unstable. It must be opted in via \'--unstable\', e.g. \'cdk gc --unstable=gc\'');
                }
                if (args.bootstrapStackName) {
                    await ioHost.defaults.warn('--bootstrap-stack-name is deprecated and will be removed when gc is GA. Use --toolkit-stack-name.');
                }
                // roleArn is defined for when cloudformation is invoked
                // This conflicts with direct sdk calls existing in the gc command to s3 and ecr
                if (args.roleArn) {
                    await ioHost.defaults.warn('The --role-arn option is not supported for the gc command and will be ignored.');
                }
                return cli.garbageCollect(args.ENVIRONMENTS, {
                    action: args.action,
                    type: args.type,
                    rollbackBufferDays: args['rollback-buffer-days'],
                    createdBufferDays: args['created-buffer-days'],
                    bootstrapStackName: args.toolkitStackName ?? args.bootstrapStackName,
                    confirm: args.confirm,
                });
            case 'flags':
                ioHost.currentAction = 'flags';
                if (!configuration.settings.get(['unstable']).includes('flags')) {
                    throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'flags\' is unstable. It must be opted in via \'--unstable\', e.g. \'cdk flags --unstable=flags\'');
                }
                const toolkit = new toolkit_lib_1.Toolkit({
                    ioHost,
                    toolkitStackName,
                    unstableFeatures: configuration.settings.get(['unstable']),
                });
                const flagsData = await toolkit.flags(cloudExecutable);
                const handler = new flags_1.FlagCommandHandler(flagsData, ioHelper, args, toolkit, configuration.context.all);
                return handler.processFlagsCommand();
            case 'synthesize':
            case 'synth':
                ioHost.currentAction = 'synth';
                const quiet = configuration.settings.get(['quiet']) ?? args.quiet;
                if (args.exclusively) {
                    return cli.synth(args.STACKS, args.exclusively, quiet, args.validation, argv.json);
                }
                else {
                    return cli.synth(args.STACKS, true, quiet, args.validation, argv.json);
                }
            case 'notices':
                ioHost.currentAction = 'notices';
                // If the user explicitly asks for notices, they are now the primary output
                // of the command and they should go to stdout.
                ioHost.noticesDestination = 'stdout';
                // This is a valid command, but we're postponing its execution because displaying
                // notices automatically happens after every command.
                return;
            case 'metadata':
                ioHost.currentAction = 'metadata';
                return cli.metadata(args.STACK, argv.json);
            case 'acknowledge':
            case 'ack':
                ioHost.currentAction = 'notices';
                return cli.acknowledge(args.ID);
            case 'cli-telemetry':
                ioHost.currentAction = 'cli-telemetry';
                if (args.enable === undefined && args.disable === undefined && args.status === undefined) {
                    throw new toolkit_lib_1.ToolkitError('Must specify \'--enable\', \'--disable\', or \'--status\'');
                }
                if (args.status) {
                    return cli.cliTelemetryStatus(args);
                }
                else {
                    const enable = args.enable ?? !args.disable;
                    return cli.cliTelemetry(enable);
                }
            case 'init':
                ioHost.currentAction = 'init';
                const language = configuration.settings.get(['language']);
                if (args.list) {
                    return (0, init_1.printAvailableTemplates)(ioHelper, language);
                }
                else {
                    // Gate custom template support with unstable flag
                    if (args['from-path'] && !configuration.settings.get(['unstable']).includes('init')) {
                        throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'init\' with custom templates is unstable. It must be opted in via \'--unstable\', e.g. \'cdk init --from-path=./my-template --unstable=init\'');
                    }
                    return (0, init_1.cliInit)({
                        ioHelper,
                        type: args.TEMPLATE,
                        language,
                        canUseNetwork: undefined,
                        generateOnly: args.generateOnly,
                        libVersion: args.libVersion,
                        fromPath: args['from-path'],
                        templatePath: args['template-path'],
                        packageManager: args['package-manager'],
                        projectName: args.name,
                    });
                }
            case 'migrate':
                ioHost.currentAction = 'migrate';
                return cli.migrate({
                    stackName: args['stack-name'],
                    fromPath: args['from-path'],
                    fromStack: args['from-stack'],
                    language: args.language,
                    outputPath: args['output-path'],
                    fromScan: (0, migrate_1.getMigrateScanType)(args['from-scan']),
                    filter: args.filter,
                    account: args.account,
                    region: args.region,
                    compress: args.compress,
                });
            case 'version':
                ioHost.currentAction = 'version';
                return ioHost.defaults.result((0, version_1.versionWithBuild)());
            default:
                throw new toolkit_lib_1.ToolkitError('Unknown command: ' + command);
        }
    }
}
/**
 * Determine which version of bootstrapping
 */
async function determineBootstrapVersion(ioHost, args) {
    let source;
    if (args.template) {
        await ioHost.defaults.info(`Using bootstrapping template from ${args.template}`);
        source = { source: 'custom', templateFile: args.template };
    }
    else if (process.env.CDK_LEGACY_BOOTSTRAP) {
        await ioHost.defaults.info('CDK_LEGACY_BOOTSTRAP set, using legacy-style bootstrapping');
        source = { source: 'legacy' };
    }
    else {
        // in V2, the "new" bootstrapping is the default
        source = { source: 'default' };
    }
    return source;
}
function isFeatureEnabled(configuration, featureFlag) {
    return configuration.context.get(featureFlag) ?? cxapi.futureFlagDefault(featureFlag);
}
/**
 * Translate a Yargs input array to something that makes more sense in a programming language
 * model (telling the difference between absence and an empty array)
 *
 * - An empty array is the default case, meaning the user didn't pass any arguments. We return
 *   undefined.
 * - If the user passed a single empty string, they did something like `--array=`, which we'll
 *   take to mean they passed an empty array.
 */
function arrayFromYargs(xs) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.filter((x) => x !== '');
}
function determineDeploymentMethod(args, configuration, watch) {
    let deploymentMethod;
    switch (args.method) {
        case 'direct':
            if (args.changeSetName) {
                throw new toolkit_lib_1.ToolkitError('--change-set-name cannot be used with method=direct');
            }
            if (args.importExistingResources) {
                throw new toolkit_lib_1.ToolkitError('--import-existing-resources cannot be enabled with method=direct');
            }
            deploymentMethod = { method: 'direct' };
            break;
        case 'change-set':
            deploymentMethod = {
                method: 'change-set',
                execute: true,
                changeSetName: args.changeSetName,
                importExistingResources: args.importExistingResources,
            };
            break;
        case 'prepare-change-set':
            deploymentMethod = {
                method: 'change-set',
                execute: false,
                changeSetName: args.changeSetName,
                importExistingResources: args.importExistingResources,
            };
            break;
        case undefined:
        default:
            deploymentMethod = {
                method: 'change-set',
                execute: watch ? true : args.execute ?? true,
                changeSetName: args.changeSetName,
                importExistingResources: args.importExistingResources,
            };
            break;
    }
    const hotswapMode = determineHotswapMode(args.hotswap, args.hotswapFallback, watch);
    const hotswapProperties = configuration.settings.get(['hotswap']) || {};
    switch (hotswapMode) {
        case hotswap_1.HotswapMode.FALL_BACK:
            return {
                method: 'hotswap',
                properties: hotswapProperties,
                fallback: deploymentMethod,
            };
        case hotswap_1.HotswapMode.HOTSWAP_ONLY:
            return {
                method: 'hotswap',
                properties: hotswapProperties,
            };
        default:
        case hotswap_1.HotswapMode.FULL_DEPLOYMENT:
            return deploymentMethod;
    }
}
function determineHotswapMode(hotswap, hotswapFallback, watch) {
    if (hotswap && hotswapFallback) {
        throw new toolkit_lib_1.ToolkitError('Can not supply both --hotswap and --hotswap-fallback at the same time');
    }
    else if (!hotswap && !hotswapFallback) {
        if (hotswap === undefined && hotswapFallback === undefined) {
            return watch ? hotswap_1.HotswapMode.HOTSWAP_ONLY : hotswap_1.HotswapMode.FULL_DEPLOYMENT;
        }
        else if (hotswap === false || hotswapFallback === false) {
            return hotswap_1.HotswapMode.FULL_DEPLOYMENT;
        }
    }
    let hotswapMode;
    if (hotswap) {
        hotswapMode = hotswap_1.HotswapMode.HOTSWAP_ONLY;
        /* if (hotswapFallback)*/
    }
    else {
        hotswapMode = hotswap_1.HotswapMode.FALL_BACK;
    }
    return hotswapMode;
}
/* c8 ignore start */ // we never call this in unit tests
function cli(args = process.argv.slice(2)) {
    let error;
    exec(args)
        .then(async (value) => {
        if (typeof value === 'number') {
            process.exitCode = value;
        }
    })
        .catch(async (err) => {
        // Log the stack trace if we're on a developer workstation. Otherwise this will be into a minified
        // file and the printed code line and stack trace are huge and useless.
        (0, pretty_print_error_1.prettyPrintError)(err, (0, version_1.isDeveloperBuildVersion)());
        error = {
            name: (0, error_1.cdkCliErrorName)(err.name),
        };
        process.exitCode = 1;
    })
        .finally(async () => {
        try {
            await io_host_1.CliIoHost.get()?.telemetry?.end(error);
        }
        catch (e) {
            await io_host_1.CliIoHost.get()?.asIoHelper().defaults.trace(`Ending Telemetry failed: ${e.message}`);
        }
    });
}
/* c8 ignore stop */
//# sourceMappingURL=data:application/json;base64,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