"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointTelemetrySink = void 0;
const https_1 = require("https");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const network_detector_1 = require("../../../api/network-detector");
const api_private_1 = require("../../../api-private");
const REQUEST_ATTEMPT_TIMEOUT_MS = 500;
/**
 * The telemetry client that hits an external endpoint.
 */
class EndpointTelemetrySink {
    constructor(props) {
        this.events = [];
        this.endpoint = new URL(props.endpoint);
        if (!this.endpoint.hostname || !this.endpoint.pathname) {
            throw new toolkit_lib_1.ToolkitError(`Telemetry Endpoint malformed. Received hostname: ${this.endpoint.hostname}, pathname: ${this.endpoint.pathname}`);
        }
        this.ioHelper = api_private_1.IoHelper.fromActionAwareIoHost(props.ioHost);
        this.agent = props.agent;
        // Batch events every 30 seconds
        setInterval(() => this.flush(), 30000).unref();
    }
    /**
     * Add an event to the collection.
     */
    async emit(event) {
        try {
            this.events.push(event);
        }
        catch (e) {
            // Never throw errors, just log them via ioHost
            await this.ioHelper.defaults.trace(`Failed to add telemetry event: ${e.message}`);
        }
    }
    async flush() {
        try {
            if (this.events.length === 0) {
                return;
            }
            const res = await this.https(this.endpoint, { events: this.events });
            // Clear the events array after successful output
            if (res) {
                this.events = [];
            }
        }
        catch (e) {
            // Never throw errors, just log them via ioHost
            await this.ioHelper.defaults.trace(`Failed to send telemetry event: ${e.message}`);
        }
    }
    /**
     * Returns true if telemetry successfully posted, false otherwise.
     */
    async https(url, body) {
        // Check connectivity before attempting network request
        const hasConnectivity = await network_detector_1.NetworkDetector.hasConnectivity(this.agent);
        if (!hasConnectivity) {
            await this.ioHelper.defaults.trace('No internet connectivity detected, skipping telemetry');
            return false;
        }
        try {
            const res = await doRequest(url, body, this.agent);
            // Successfully posted
            if (res.statusCode && res.statusCode >= 200 && res.statusCode < 300) {
                await this.ioHelper.defaults.trace('Telemetry Sent Successfully');
                return true;
            }
            await this.ioHelper.defaults.trace(`Telemetry Unsuccessful: POST ${url.hostname}${url.pathname}: ${res.statusCode}:${res.statusMessage}`);
            return false;
        }
        catch (e) {
            await this.ioHelper.defaults.trace(`Telemetry Error: POST ${url.hostname}${url.pathname}: ${JSON.stringify(e)}`);
            return false;
        }
    }
}
exports.EndpointTelemetrySink = EndpointTelemetrySink;
/**
 * A Promisified version of `https.request()`
 */
function doRequest(url, data, agent) {
    return new Promise((ok, ko) => {
        const payload = JSON.stringify(data);
        const req = (0, https_1.request)({
            hostname: url.hostname,
            port: url.port || null,
            path: url.pathname,
            method: 'POST',
            headers: {
                'content-type': 'application/json',
                'content-length': payload.length,
            },
            agent,
            timeout: REQUEST_ATTEMPT_TIMEOUT_MS,
        }, ok);
        req.on('error', ko);
        req.on('timeout', () => {
            const error = new toolkit_lib_1.ToolkitError(`Timeout after ${REQUEST_ATTEMPT_TIMEOUT_MS}ms, aborting request`);
            req.destroy(error);
        });
        req.end(payload);
    });
}
//# sourceMappingURL=data:application/json;base64,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