"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FromScan = exports.CfnTemplateGeneratorProvider = exports.FilterType = exports.ScanStatus = exports.TemplateSourceOptions = void 0;
exports.generateCdkApp = generateCdkApp;
exports.generateStack = generateStack;
exports.readFromPath = readFromPath;
exports.readFromStack = readFromStack;
exports.generateTemplate = generateTemplate;
exports.chunks = chunks;
exports.setEnvironment = setEnvironment;
exports.parseSourceOptions = parseSourceOptions;
exports.printBar = printBar;
exports.printDots = printDots;
exports.rewriteLine = rewriteLine;
exports.writeMigrateJsonFile = writeMigrateJsonFile;
exports.getMigrateScanType = getMigrateScanType;
exports.isThereAWarning = isThereAWarning;
exports.buildGeneratedTemplateOutput = buildGeneratedTemplateOutput;
exports.buildCfnClient = buildCfnClient;
exports.appendWarningsToReadme = appendWarningsToReadme;
/* eslint-disable @typescript-eslint/no-require-imports */
/* eslint-disable @typescript-eslint/no-var-requires */
const fs = require("fs");
const path = require("path");
const cloud_assembly_api_1 = require("@aws-cdk/cloud-assembly-api");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const cdk_from_cfn = require("cdk-from-cfn");
const chalk = require("chalk");
const init_1 = require("./init");
const cloudformation_1 = require("../api/cloudformation");
const plugin_1 = require("../api/plugin");
const util_1 = require("../util");
const camelCase = require('camelcase');
const decamelize = require('decamelize');
/** The list of languages supported by the built-in noctilucent binary. */
const MIGRATE_SUPPORTED_LANGUAGES = cdk_from_cfn.supported_languages();
/**
 * Generates a CDK app from a yaml or json template.
 *
 * @param stackName - The name to assign to the stack in the generated app
 * @param stack - The yaml or json template for the stack
 * @param language - The language to generate the CDK app in
 * @param outputPath - The path at which to generate the CDK app
 */
async function generateCdkApp(ioHelper, stackName, stack, language, outputPath, compress) {
    const resolvedOutputPath = path.join(outputPath ?? process.cwd(), stackName);
    const formattedStackName = decamelize(stackName);
    try {
        fs.rmSync(resolvedOutputPath, { recursive: true, force: true });
        fs.mkdirSync(resolvedOutputPath, { recursive: true });
        const generateOnly = compress;
        await (0, init_1.cliInit)({
            ioHelper,
            type: 'app',
            language,
            canUseNetwork: true,
            generateOnly,
            workDir: resolvedOutputPath,
            stackName,
            migrate: true,
        });
        let stackFileName;
        switch (language) {
            case 'typescript':
                stackFileName = `${resolvedOutputPath}/lib/${formattedStackName}-stack.ts`;
                break;
            case 'java':
                stackFileName = `${resolvedOutputPath}/src/main/java/com/myorg/${camelCase(formattedStackName, { pascalCase: true })}Stack.java`;
                break;
            case 'python':
                stackFileName = `${resolvedOutputPath}/${formattedStackName.replace(/-/g, '_')}/${formattedStackName.replace(/-/g, '_')}_stack.py`;
                break;
            case 'csharp':
                stackFileName = `${resolvedOutputPath}/src/${camelCase(formattedStackName, { pascalCase: true })}/${camelCase(formattedStackName, { pascalCase: true })}Stack.cs`;
                break;
            case 'go':
                stackFileName = `${resolvedOutputPath}/${formattedStackName}.go`;
                break;
            default:
                throw new toolkit_lib_1.ToolkitError(`${language} is not supported by CDK Migrate. Please choose from: ${MIGRATE_SUPPORTED_LANGUAGES.join(', ')}`);
        }
        fs.writeFileSync(stackFileName, stack);
        if (compress) {
            await (0, util_1.zipDirectory)(resolvedOutputPath, `${resolvedOutputPath}.zip`);
            fs.rmSync(resolvedOutputPath, { recursive: true, force: true });
        }
    }
    catch (error) {
        fs.rmSync(resolvedOutputPath, { recursive: true, force: true });
        throw error;
    }
}
/**
 * Generates a CDK stack file.
 * @param template - The template to translate into a CDK stack
 * @param stackName - The name to assign to the stack
 * @param language - The language to generate the stack in
 * @returns A string representation of a CDK stack file
 */
function generateStack(template, stackName, language) {
    const formattedStackName = `${camelCase(decamelize(stackName), { pascalCase: true })}Stack`;
    try {
        return cdk_from_cfn.transmute(template, language, formattedStackName);
    }
    catch (e) {
        throw new toolkit_lib_1.ToolkitError(`${formattedStackName} could not be generated because ${e.message}`);
    }
}
/**
 * Reads and returns a stack template from a local path.
 *
 * @param inputPath - The location of the template
 * @returns A string representation of the template if present, otherwise undefined
 */
function readFromPath(inputPath) {
    let readFile;
    try {
        readFile = fs.readFileSync(inputPath, 'utf8');
    }
    catch (e) {
        throw new toolkit_lib_1.ToolkitError(`'${inputPath}' is not a valid path.`);
    }
    if (readFile == '') {
        throw new toolkit_lib_1.ToolkitError(`Cloudformation template filepath: '${inputPath}' is an empty file.`);
    }
    return readFile;
}
/**
 * Reads and returns a stack template from a deployed CloudFormation stack.
 *
 * @param stackName - The name of the stack
 * @param sdkProvider - The sdk provider for making CloudFormation calls
 * @param environment - The account and region where the stack is deployed
 * @returns A string representation of the template if present, otherwise undefined
 */
async function readFromStack(stackName, sdkProvider, environment) {
    const cloudFormation = (await sdkProvider.forEnvironment(environment, plugin_1.Mode.ForReading)).sdk.cloudFormation();
    const stack = await cloudformation_1.CloudFormationStack.lookup(cloudFormation, stackName, true);
    if (stack.stackStatus.isDeploySuccess || stack.stackStatus.isRollbackSuccess) {
        return JSON.stringify(await stack.template());
    }
    else {
        throw new toolkit_lib_1.ToolkitError(`Stack '${stackName}' in account ${environment.account} and region ${environment.region} has a status of '${stack.stackStatus.name}' due to '${stack.stackStatus.reason}'. The stack cannot be migrated until it is in a healthy state.`);
    }
}
/**
 * Takes in a stack name and account and region and returns a generated cloudformation template using the cloudformation
 * template generator.
 *
 * @param GenerateTemplateOptions - An object containing the stack name, filters, sdkProvider, environment, and newScan flag
 * @returns a generated cloudformation template
 */
async function generateTemplate(options) {
    const cfn = new CfnTemplateGeneratorProvider(await buildCfnClient(options.sdkProvider, options.environment), options.ioHelper);
    const ioHelper = options.ioHelper;
    const scanId = await findLastSuccessfulScan(cfn, options);
    // if a customer accidentally ctrl-c's out of the command and runs it again, this will continue the progress bar where it left off
    const curScan = await cfn.describeResourceScan(scanId);
    if (curScan.Status == ScanStatus.IN_PROGRESS) {
        await ioHelper.defaults.info('Resource scan in progress. Please wait, this can take 10 minutes or longer.');
        await scanProgressBar(ioHelper, scanId, cfn);
    }
    await displayTimeDiff(ioHelper, new Date(), new Date(curScan.StartTime));
    let resources = await cfn.listResourceScanResources(scanId, options.filters);
    await ioHelper.defaults.info('finding related resources.');
    let relatedResources = await cfn.getResourceScanRelatedResources(scanId, resources);
    await ioHelper.defaults.info(`Found ${relatedResources.length} resources.`);
    await ioHelper.defaults.info('Generating CFN template from scanned resources.');
    const templateArn = (await cfn.createGeneratedTemplate(options.stackName, relatedResources)).GeneratedTemplateId;
    let generatedTemplate = await cfn.describeGeneratedTemplate(templateArn);
    await ioHelper.defaults.info('Please wait, template creation in progress. This may take a couple minutes.');
    while (generatedTemplate.Status !== ScanStatus.COMPLETE && generatedTemplate.Status !== ScanStatus.FAILED) {
        await printDots(`[${generatedTemplate.Status}] Template Creation in Progress`, 400);
        generatedTemplate = await cfn.describeGeneratedTemplate(templateArn);
    }
    await ioHelper.defaults.info('\nTemplate successfully generated!');
    return buildGeneratedTemplateOutput(generatedTemplate, (await cfn.getGeneratedTemplate(templateArn)).TemplateBody, templateArn);
}
async function findLastSuccessfulScan(cfn, options) {
    const ioHelper = options.ioHelper;
    let resourceScanSummaries = [];
    const clientRequestToken = `cdk-migrate-${options.environment.account}-${options.environment.region}`;
    if (options.fromScan === FromScan.NEW) {
        await ioHelper.defaults.info(`Starting new scan for account ${options.environment.account} in region ${options.environment.region}`);
        try {
            await cfn.startResourceScan(clientRequestToken);
            resourceScanSummaries = (await cfn.listResourceScans()).ResourceScanSummaries;
        }
        catch (e) {
            // continuing here because if the scan fails on a new-scan it is very likely because there is either already a scan in progress
            // or the customer hit a rate limit. In either case we want to continue with the most recent scan.
            // If this happens to fail for a credential error then that will be caught immediately after anyway.
            await ioHelper.defaults.info(`Scan failed to start due to error '${e.message}', defaulting to latest scan.`);
        }
    }
    else {
        resourceScanSummaries = (await cfn.listResourceScans()).ResourceScanSummaries;
        await cfn.checkForResourceScan(resourceScanSummaries, options, clientRequestToken);
    }
    // get the latest scan, which we know will exist
    resourceScanSummaries = (await cfn.listResourceScans()).ResourceScanSummaries;
    let scanId = resourceScanSummaries[0].ResourceScanId;
    // find the most recent scan that isn't in a failed state in case we didn't start a new one
    for (const summary of resourceScanSummaries) {
        if (summary.Status !== ScanStatus.FAILED) {
            scanId = summary.ResourceScanId;
            break;
        }
    }
    return scanId;
}
/**
 * Takes a string of filters in the format of key1=value1,key2=value2 and returns a map of the filters.
 *
 * @param filters - a string of filters in the format of key1=value1,key2=value2
 * @returns a map of the filters
 */
function parseFilters(filters) {
    if (!filters) {
        return {
            'resource-identifier': undefined,
            'resource-type-prefix': undefined,
            'tag-key': undefined,
            'tag-value': undefined,
        };
    }
    const filterShorthands = {
        'identifier': FilterType.RESOURCE_IDENTIFIER,
        'id': FilterType.RESOURCE_IDENTIFIER,
        'type': FilterType.RESOURCE_TYPE_PREFIX,
        'type-prefix': FilterType.RESOURCE_TYPE_PREFIX,
    };
    const filterList = filters.split(',');
    let filterMap = {
        [FilterType.RESOURCE_IDENTIFIER]: undefined,
        [FilterType.RESOURCE_TYPE_PREFIX]: undefined,
        [FilterType.TAG_KEY]: undefined,
        [FilterType.TAG_VALUE]: undefined,
    };
    for (const fil of filterList) {
        const filter = fil.split('=');
        let filterKey = filter[0];
        const filterValue = filter[1];
        // if the key is a shorthand, replace it with the full name
        if (filterKey in filterShorthands) {
            filterKey = filterShorthands[filterKey];
        }
        if (Object.values(FilterType).includes(filterKey)) {
            filterMap[filterKey] = filterValue;
        }
        else {
            throw new toolkit_lib_1.ToolkitError(`Invalid filter: ${filterKey}`);
        }
    }
    return filterMap;
}
/**
 * Takes a list of any type and breaks it up into chunks of a specified size.
 *
 * @param list - The list to break up
 * @param chunkSize - The size of each chunk
 * @returns A list of lists of the specified size
 */
function chunks(list, chunkSize) {
    const chunkedList = [];
    for (let i = 0; i < list.length; i += chunkSize) {
        chunkedList.push(list.slice(i, i + chunkSize));
    }
    return chunkedList;
}
/**
 * Sets the account and region for making CloudFormation calls.
 * @param account - The account to use
 * @param region - The region to use
 * @returns The environment object
 */
function setEnvironment(account, region) {
    return {
        account: account ?? cloud_assembly_api_1.UNKNOWN_ACCOUNT,
        region: region ?? cloud_assembly_api_1.UNKNOWN_REGION,
        name: 'cdk-migrate-env',
    };
}
/**
 * Enum for the source options for the template
 */
var TemplateSourceOptions;
(function (TemplateSourceOptions) {
    TemplateSourceOptions["PATH"] = "path";
    TemplateSourceOptions["STACK"] = "stack";
    TemplateSourceOptions["SCAN"] = "scan";
})(TemplateSourceOptions || (exports.TemplateSourceOptions = TemplateSourceOptions = {}));
/**
 * Enum for the status of a resource scan
 */
var ScanStatus;
(function (ScanStatus) {
    ScanStatus["IN_PROGRESS"] = "IN_PROGRESS";
    ScanStatus["COMPLETE"] = "COMPLETE";
    ScanStatus["FAILED"] = "FAILED";
})(ScanStatus || (exports.ScanStatus = ScanStatus = {}));
var FilterType;
(function (FilterType) {
    FilterType["RESOURCE_IDENTIFIER"] = "resource-identifier";
    FilterType["RESOURCE_TYPE_PREFIX"] = "resource-type-prefix";
    FilterType["TAG_KEY"] = "tag-key";
    FilterType["TAG_VALUE"] = "tag-value";
})(FilterType || (exports.FilterType = FilterType = {}));
/**
 * Validates that exactly one source option has been provided.
 * @param fromPath - The content of the flag `--from-path`
 * @param fromStack - the content of the flag `--from-stack`
 */
function parseSourceOptions(fromPath, fromStack, stackName) {
    if (fromPath && fromStack) {
        throw new toolkit_lib_1.ToolkitError('Only one of `--from-path` or `--from-stack` may be provided.');
    }
    if (!stackName) {
        throw new toolkit_lib_1.ToolkitError('`--stack-name` is a required field.');
    }
    if (!fromPath && !fromStack) {
        return { source: TemplateSourceOptions.SCAN };
    }
    if (fromPath) {
        return { source: TemplateSourceOptions.PATH, templatePath: fromPath };
    }
    return { source: TemplateSourceOptions.STACK, stackName: stackName };
}
/**
 * Takes a set of resources and removes any with the managedbystack flag set to true.
 *
 * @param resourceList - the list of resources provided by the list scanned resources calls
 * @returns a list of resources not managed by cfn stacks
 */
function excludeManaged(resourceList) {
    return resourceList
        .filter((r) => !r.ManagedByStack)
        .map((r) => ({
        ResourceType: r.ResourceType,
        ResourceIdentifier: r.ResourceIdentifier,
    }));
}
/**
 * Transforms a list of resources into a list of resource identifiers by removing the ManagedByStack flag.
 * Setting the value of the field to undefined effectively removes it from the object.
 *
 * @param resourceList - the list of resources provided by the list scanned resources calls
 * @returns a list of ScannedResourceIdentifier[]
 */
function resourceIdentifiers(resourceList) {
    const identifiers = [];
    resourceList.forEach((r) => {
        const identifier = {
            ResourceType: r.ResourceType,
            ResourceIdentifier: r.ResourceIdentifier,
        };
        identifiers.push(identifier);
    });
    return identifiers;
}
/**
 * Takes a scan id and maintains a progress bar to display the progress of a scan to the user.
 *
 * @param scanId - A string representing the scan id
 * @param cloudFormation - The CloudFormation sdk client to use
 */
async function scanProgressBar(ioHelper, scanId, cfn) {
    let curProgress = 0.5;
    // we know it's in progress initially since we wouldn't have gotten here if it wasn't
    let curScan = {
        Status: ScanStatus.IN_PROGRESS,
        $metadata: {},
    };
    while (curScan.Status == ScanStatus.IN_PROGRESS) {
        curScan = await cfn.describeResourceScan(scanId);
        curProgress = curScan.PercentageCompleted ?? curProgress;
        printBar(30, curProgress);
        await new Promise((resolve) => setTimeout(resolve, 2000));
    }
    await ioHelper.defaults.info('\n✅ Scan Complete!');
}
/**
 * Prints a progress bar to the console. To be used in a while loop to show progress of a long running task.
 * The progress bar deletes the current line on the console and rewrites it with the progress amount.
 *
 * @param width - The width of the progress bar
 * @param progress - The current progress to display as a percentage of 100
 */
function printBar(width, progress) {
    if (!process.env.MIGRATE_INTEG_TEST) {
        const FULL_BLOCK = '█';
        const PARTIAL_BLOCK = ['', '▏', '▎', '▍', '▌', '▋', '▊', '▉'];
        const fraction = Math.min(progress / 100, 1);
        const innerWidth = Math.max(1, width - 2);
        const chars = innerWidth * fraction;
        const remainder = chars - Math.floor(chars);
        const fullChars = FULL_BLOCK.repeat(Math.floor(chars));
        const partialChar = PARTIAL_BLOCK[Math.floor(remainder * PARTIAL_BLOCK.length)];
        const filler = '·'.repeat(innerWidth - Math.floor(chars) - (partialChar ? 1 : 0));
        const color = chalk.green;
        rewriteLine('[' + color(fullChars + partialChar) + filler + `] (${progress}%)`);
    }
}
/**
 * Prints a message to the console with a series periods appended to it. To be used in a while loop to show progress of a long running task.
 * The message deletes the current line and rewrites it several times to display 1-3 periods to show the user that the task is still running.
 *
 * @param message - The message to display
 * @param timeoutx4 - The amount of time to wait before printing the next period
 */
async function printDots(message, timeoutx4) {
    if (!process.env.MIGRATE_INTEG_TEST) {
        rewriteLine(message + ' .');
        await new Promise((resolve) => setTimeout(resolve, timeoutx4));
        rewriteLine(message + ' ..');
        await new Promise((resolve) => setTimeout(resolve, timeoutx4));
        rewriteLine(message + ' ...');
        await new Promise((resolve) => setTimeout(resolve, timeoutx4));
        rewriteLine(message);
        await new Promise((resolve) => setTimeout(resolve, timeoutx4));
    }
}
/**
 * Rewrites the current line on the console and writes a new message to it.
 * This is a helper funciton for printDots and printBar.
 *
 * @param message - The message to display
 */
function rewriteLine(message) {
    process.stdout.clearLine(0);
    process.stdout.cursorTo(0);
    process.stdout.write(message);
}
/**
 * Prints the time difference between two dates in days, hours, and minutes.
 *
 * @param time1 - The first date to compare
 * @param time2 - The second date to compare
 */
async function displayTimeDiff(ioHelper, time1, time2) {
    const diff = Math.abs(time1.getTime() - time2.getTime());
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
    await ioHelper.defaults.info(`Using the latest successful scan which is ${days} days, ${hours} hours, and ${minutes} minutes old.`);
}
/**
 * Writes a migrate.json file to the output directory.
 *
 * @param outputPath - The path to write the migrate.json file to
 * @param stackName - The name of the stack
 * @param generatedOutput - The output of the template generator
 */
function writeMigrateJsonFile(outputPath, stackName, migrateJson) {
    const outputToJson = {
        '//': 'This file is generated by cdk migrate. It will be automatically deleted after the first successful deployment of this app to the environment of the original resources.',
        'Source': migrateJson.source,
        'Resources': migrateJson.resources,
    };
    fs.writeFileSync(`${path.join(outputPath ?? process.cwd(), stackName)}/migrate.json`, JSON.stringify(outputToJson, null, 2));
}
/**
 * Takes a string representing the from-scan flag and returns a FromScan enum value.
 *
 * @param scanType - A string representing the from-scan flag
 * @returns A FromScan enum value
 */
function getMigrateScanType(scanType) {
    switch (scanType) {
        case 'new':
            return FromScan.NEW;
        case 'most-recent':
            return FromScan.MOST_RECENT;
        case '':
            return FromScan.DEFAULT;
        case undefined:
            return FromScan.DEFAULT;
        default:
            throw new toolkit_lib_1.ToolkitError(`Unknown scan type: ${scanType}`);
    }
}
/**
 * Takes a generatedTemplateOutput objct and returns a boolean representing whether there are any warnings on any rescources.
 *
 * @param generatedTemplateOutput - A GenerateTemplateOutput object
 * @returns A boolean representing whether there are any warnings on any rescources
 */
function isThereAWarning(generatedTemplateOutput) {
    if (generatedTemplateOutput.resources) {
        for (const resource of generatedTemplateOutput.resources) {
            if (resource.Warnings && resource.Warnings.length > 0) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Builds the GenerateTemplateOutput object from the DescribeGeneratedTemplateOutput and the template body.
 *
 * @param generatedTemplateSummary - The output of the describe generated template call
 * @param templateBody - The body of the generated template
 * @returns A GenerateTemplateOutput object
 */
function buildGeneratedTemplateOutput(generatedTemplateSummary, templateBody, source) {
    const resources = generatedTemplateSummary.Resources;
    const migrateJson = {
        templateBody: templateBody,
        source: source,
        resources: generatedTemplateSummary.Resources.map((r) => ({
            ResourceType: r.ResourceType,
            LogicalResourceId: r.LogicalResourceId,
            ResourceIdentifier: r.ResourceIdentifier,
        })),
    };
    const templateId = generatedTemplateSummary.GeneratedTemplateId;
    return {
        migrateJson: migrateJson,
        resources: resources,
        templateId: templateId,
    };
}
/**
 * Builds a CloudFormation sdk client for making requests with the CFN template generator.
 *
 * @param sdkProvider - The sdk provider for making CloudFormation calls
 * @param environment - The account and region where the stack is deployed
 * @returns A CloudFormation sdk client
 */
async function buildCfnClient(sdkProvider, environment) {
    const sdk = (await sdkProvider.forEnvironment(environment, plugin_1.Mode.ForReading)).sdk;
    sdk.appendCustomUserAgent('cdk-migrate');
    return sdk.cloudFormation();
}
/**
 * Appends a list of warnings to a readme file.
 *
 * @param filepath - The path to the readme file
 * @param resources - A list of resources to append warnings for
 */
function appendWarningsToReadme(filepath, resources) {
    const readme = fs.readFileSync(filepath, 'utf8');
    const lines = readme.split('\n');
    const index = lines.findIndex((line) => line.trim() === 'Enjoy!');
    let linesToAdd = ['\n## Warnings'];
    linesToAdd.push('### Write-only properties');
    linesToAdd.push("Write-only properties are resource property values that can be written to but can't be read by AWS CloudFormation or CDK Migrate. For more information, see [IaC generator and write-only properties](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/generate-IaC-write-only-properties.html).");
    linesToAdd.push('\n');
    linesToAdd.push('Write-only properties discovered during migration are organized here by resource ID and categorized by write-only property type. Resolve write-only properties by providing property values in your CDK app. For guidance, see [Resolve write-only properties](https://docs.aws.amazon.com/cdk/v2/guide/migrate.html#migrate-resources-writeonly).');
    for (const resource of resources) {
        if (resource.Warnings && resource.Warnings.length > 0) {
            linesToAdd.push(`### ${resource.LogicalResourceId}`);
            for (const warning of resource.Warnings) {
                linesToAdd.push(`- **${warning.Type}**: `);
                for (const property of warning.Properties) {
                    linesToAdd.push(`  - ${property.PropertyPath}: ${property.Description}`);
                }
            }
        }
    }
    lines.splice(index, 0, ...linesToAdd);
    fs.writeFileSync(filepath, lines.join('\n'));
}
/**
 * takes a list of resources and returns a list of unique resources based on the resource type and logical resource id.
 *
 * @param resources - A list of resources to deduplicate
 * @returns A list of unique resources
 */
function deduplicateResources(resources) {
    let uniqueResources = {};
    for (const resource of resources) {
        const key = Object.keys(resource.ResourceIdentifier)[0];
        // Creating our unique identifier using the resource type, the key, and the value of the resource identifier
        // The resource identifier is a combination of a key value pair defined by a resource's schema, and the resource type of the resource.
        const uniqueIdentifer = `${resource.ResourceType}:${key}:${resource.ResourceIdentifier[key]}`;
        uniqueResources[uniqueIdentifer] = resource;
    }
    return Object.values(uniqueResources);
}
/**
 * Class for making CloudFormation template generator calls
 */
class CfnTemplateGeneratorProvider {
    constructor(cfn, ioHelper) {
        this.cfn = cfn;
        this.ioHelper = ioHelper;
    }
    async checkForResourceScan(resourceScanSummaries, options, clientRequestToken) {
        if (!resourceScanSummaries || resourceScanSummaries.length === 0) {
            if (options.fromScan === FromScan.MOST_RECENT) {
                throw new toolkit_lib_1.ToolkitError('No scans found. Please either start a new scan with the `--from-scan` new or do not specify a `--from-scan` option.');
            }
            else {
                await this.ioHelper.defaults.info('No scans found. Initiating a new resource scan.');
                await this.startResourceScan(clientRequestToken);
            }
        }
    }
    /**
     * Retrieves a tokenized list of resources and their associated scan. If a token is present the function
     * will loop through all pages and combine them into a single list of ScannedRelatedResources
     *
     * @param scanId - scan id for the to list resources for
     * @param resources - A list of resources to find related resources for
     */
    async getResourceScanRelatedResources(scanId, resources) {
        let relatedResourceList = resources;
        // break the list of resources into chunks of 100 to avoid hitting the 100 resource limit
        for (const chunk of chunks(resources, 100)) {
            // get the first page of related resources
            const res = await this.cfn.listResourceScanRelatedResources({
                ResourceScanId: scanId,
                Resources: chunk,
            });
            // add the first page to the list
            relatedResourceList.push(...(res.RelatedResources ?? []));
            let nextToken = res.NextToken;
            // if there are more pages, cycle through them and add them to the list before moving on to the next chunk
            while (nextToken) {
                const nextRelatedResources = await this.cfn.listResourceScanRelatedResources({
                    ResourceScanId: scanId,
                    Resources: resourceIdentifiers(resources),
                    NextToken: nextToken,
                });
                nextToken = nextRelatedResources.NextToken;
                relatedResourceList.push(...(nextRelatedResources.RelatedResources ?? []));
            }
        }
        relatedResourceList = deduplicateResources(relatedResourceList);
        // prune the managedbystack flag off of them again.
        return process.env.MIGRATE_INTEG_TEST
            ? resourceIdentifiers(relatedResourceList)
            : resourceIdentifiers(excludeManaged(relatedResourceList));
    }
    /**
     * Kicks off a scan of a customers account, returning the scan id. A scan can take
     * 10 minutes or longer to complete. However this will return a scan id as soon as
     * the scan has begun.
     *
     * @returns A string representing the scan id
     */
    async startResourceScan(requestToken) {
        return (await this.cfn.startResourceScan({
            ClientRequestToken: requestToken,
        })).ResourceScanId;
    }
    /**
     * Gets the most recent scans a customer has completed
     *
     * @returns a list of resource scan summaries
     */
    async listResourceScans() {
        return this.cfn.listResourceScans();
    }
    /**
     * Retrieves a tokenized list of resources from a resource scan. If a token is present, this function
     * will loop through all pages and combine them into a single list of ScannedResource[].
     * Additionally will apply any filters provided by the customer.
     *
     * @param scanId - scan id for the to list resources for
     * @param filters - a string of filters in the format of key1=value1,key2=value2
     * @returns a combined list of all resources from the scan
     */
    async listResourceScanResources(scanId, filters = []) {
        let resourceList = [];
        let resourceScanInputs;
        if (filters.length > 0) {
            await this.ioHelper.defaults.info('Applying filters to resource scan.');
            for (const filter of filters) {
                const filterList = parseFilters(filter);
                resourceScanInputs = {
                    ResourceScanId: scanId,
                    ResourceIdentifier: filterList[FilterType.RESOURCE_IDENTIFIER],
                    ResourceTypePrefix: filterList[FilterType.RESOURCE_TYPE_PREFIX],
                    TagKey: filterList[FilterType.TAG_KEY],
                    TagValue: filterList[FilterType.TAG_VALUE],
                };
                const resources = await this.cfn.listResourceScanResources(resourceScanInputs);
                resourceList = resourceList.concat(resources.Resources ?? []);
                let nextToken = resources.NextToken;
                // cycle through the pages adding all resources to the list until we run out of pages
                while (nextToken) {
                    resourceScanInputs.NextToken = nextToken;
                    const nextResources = await this.cfn.listResourceScanResources(resourceScanInputs);
                    nextToken = nextResources.NextToken;
                    resourceList = resourceList.concat(nextResources.Resources ?? []);
                }
            }
        }
        else {
            await this.ioHelper.defaults.info('No filters provided. Retrieving all resources from scan.');
            resourceScanInputs = {
                ResourceScanId: scanId,
            };
            const resources = await this.cfn.listResourceScanResources(resourceScanInputs);
            resourceList = resourceList.concat(resources.Resources ?? []);
            let nextToken = resources.NextToken;
            // cycle through the pages adding all resources to the list until we run out of pages
            while (nextToken) {
                resourceScanInputs.NextToken = nextToken;
                const nextResources = await this.cfn.listResourceScanResources(resourceScanInputs);
                nextToken = nextResources.NextToken;
                resourceList = resourceList.concat(nextResources.Resources ?? []);
            }
        }
        if (resourceList.length === 0) {
            throw new toolkit_lib_1.ToolkitError(`No resources found with filters ${filters.join(' ')}. Please try again with different filters.`);
        }
        resourceList = deduplicateResources(resourceList);
        return process.env.MIGRATE_INTEG_TEST
            ? resourceIdentifiers(resourceList)
            : resourceIdentifiers(excludeManaged(resourceList));
    }
    /**
     * Retrieves information about a resource scan.
     *
     * @param scanId - scan id for the to list resources for
     * @returns information about the scan
     */
    async describeResourceScan(scanId) {
        return this.cfn.describeResourceScan({
            ResourceScanId: scanId,
        });
    }
    /**
     * Describes the current status of the template being generated.
     *
     * @param templateId - A string representing the template id
     * @returns DescribeGeneratedTemplateOutput an object containing the template status and results
     */
    async describeGeneratedTemplate(templateId) {
        const generatedTemplate = await this.cfn.describeGeneratedTemplate({
            GeneratedTemplateName: templateId,
        });
        if (generatedTemplate.Status == ScanStatus.FAILED) {
            throw new toolkit_lib_1.ToolkitError(generatedTemplate.StatusReason);
        }
        return generatedTemplate;
    }
    /**
     * Retrieves a completed generated cloudformation template from the template generator.
     *
     * @param templateId - A string representing the template id
     * @param cloudFormation - The CloudFormation sdk client to use
     * @returns DescribeGeneratedTemplateOutput an object containing the template status and body
     */
    async getGeneratedTemplate(templateId) {
        return this.cfn.getGeneratedTemplate({
            GeneratedTemplateName: templateId,
        });
    }
    /**
     * Kicks off a template generation for a set of resources.
     *
     * @param stackName - The name of the stack
     * @param resources - A list of resources to generate the template from
     * @returns CreateGeneratedTemplateOutput an object containing the template arn to query on later
     */
    async createGeneratedTemplate(stackName, resources) {
        const createTemplateOutput = await this.cfn.createGeneratedTemplate({
            Resources: resources,
            GeneratedTemplateName: stackName,
        });
        if (createTemplateOutput.GeneratedTemplateId === undefined) {
            throw new toolkit_lib_1.ToolkitError('CreateGeneratedTemplate failed to return an Arn.');
        }
        return createTemplateOutput;
    }
    /**
     * Deletes a generated template from the template generator.
     *
     * @param templateArn - The arn of the template to delete
     * @returns A promise that resolves when the template has been deleted
     */
    async deleteGeneratedTemplate(templateArn) {
        await this.cfn.deleteGeneratedTemplate({
            GeneratedTemplateName: templateArn,
        });
    }
}
exports.CfnTemplateGeneratorProvider = CfnTemplateGeneratorProvider;
/**
 * The possible ways to choose a scan to generate a CDK application from
 */
var FromScan;
(function (FromScan) {
    /**
     * Initiate a new resource scan to build the CDK application from.
     */
    FromScan[FromScan["NEW"] = 0] = "NEW";
    /**
     * Use the last successful scan to build the CDK application from. Will fail if no scan is found.
     */
    FromScan[FromScan["MOST_RECENT"] = 1] = "MOST_RECENT";
    /**
     * Starts a scan if none exists, otherwise uses the most recent successful scan to build the CDK application from.
     */
    FromScan[FromScan["DEFAULT"] = 2] = "DEFAULT";
})(FromScan || (exports.FromScan = FromScan = {}));
//# sourceMappingURL=data:application/json;base64,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