"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.execProgram = execProgram;
exports.createAssembly = createAssembly;
const childProcess = require("child_process");
const util_1 = require("util");
const cloud_assembly_api_1 = require("@aws-cdk/cloud-assembly-api");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const fs = require("fs-extra");
const api_1 = require("../api");
const user_configuration_1 = require("../cli/user-configuration");
const version_1 = require("../cli/version");
/** Invokes the cloud executable and returns JSON output */
async function execProgram(aws, ioHelper, config) {
    const debugFn = (msg) => ioHelper.defaults.debug(msg);
    const params = (0, api_1.synthParametersFromSettings)(config.settings);
    const context = {
        ...config.context.all,
        ...params.context,
    };
    await debugFn((0, util_1.format)('context:', context));
    const env = noUndefined({
        // Versioning, outdir, default account and region
        ...await (0, api_1.prepareDefaultEnvironment)(aws, debugFn),
        // Environment variables derived from settings
        ...params.env,
    });
    const build = config.settings.get(['build']);
    if (build) {
        await exec(build);
    }
    let app = config.settings.get(['app']);
    if (!app) {
        throw new toolkit_lib_1.ToolkitError(`--app is required either in command-line, in ${user_configuration_1.PROJECT_CONFIG} or in ${user_configuration_1.USER_DEFAULTS}`);
    }
    // bypass "synth" if app points to a cloud assembly
    if (await fs.pathExists(app) && (await fs.stat(app)).isDirectory()) {
        await debugFn('--app points to a cloud assembly, so we bypass synth');
        // Acquire a read lock on this directory
        const lock = await new api_1.RWLock(app).acquireRead();
        return { assembly: createAssembly(app), lock };
    }
    // Traditionally it has been possible, though not widely advertised, to put a string[] into `cdk.json`.
    // However, we would just quickly join this array back up to string with spaces (unquoted even!) and proceed as usual,
    // thereby losing all the benefits of a pre-segmented command line. This coercion is just here for backwards
    // compatibility with existing configurations. An upcoming PR might retain the benefit of the string[].
    if (Array.isArray(app)) {
        app = app.join(' ');
    }
    const commandLine = await (0, api_1.guessExecutable)(app, debugFn);
    const outdir = config.settings.get(['output']);
    if (!outdir) {
        throw new toolkit_lib_1.ToolkitError('unexpected: --output is required');
    }
    if (typeof outdir !== 'string') {
        throw new toolkit_lib_1.ToolkitError(`--output takes a string, got ${JSON.stringify(outdir)}`);
    }
    try {
        await fs.mkdirp(outdir);
    }
    catch (error) {
        throw new toolkit_lib_1.ToolkitError(`Could not create output directory ${outdir} (${error.message})`);
    }
    await debugFn(`outdir: ${outdir}`);
    env[cxapi.OUTDIR_ENV] = outdir;
    // Acquire a lock on the output directory
    const writerLock = await new api_1.RWLock(outdir).acquireWrite();
    // Send version information
    env[cxapi.CLI_ASM_VERSION_ENV] = cxschema.Manifest.version();
    env[cxapi.CLI_VERSION_ENV] = (0, version_1.versionNumber)();
    await debugFn((0, util_1.format)('env:', env));
    const cleanupTemp = (0, api_1.writeContextToEnv)(env, context, 'add-process-env-later');
    try {
        await exec(commandLine);
        const assembly = createAssembly(outdir);
        return { assembly, lock: await writerLock.convertToReaderLock() };
    }
    catch (e) {
        await writerLock.release();
        throw e;
    }
    finally {
        await cleanupTemp();
    }
    async function exec(commandAndArgs) {
        try {
            await new Promise((ok, fail) => {
                // We use a slightly lower-level interface to:
                //
                // - Pass arguments in an array instead of a string, to get around a
                //   number of quoting issues introduced by the intermediate shell layer
                //   (which would be different between Linux and Windows).
                //
                // - Inherit stderr from controlling terminal. We don't use the captured value
                //   anyway, and if the subprocess is printing to it for debugging purposes the
                //   user gets to see it sooner. Plus, capturing doesn't interact nicely with some
                //   processes like Maven.
                const proc = childProcess.spawn(commandAndArgs, {
                    stdio: ['ignore', 'inherit', 'inherit'],
                    detached: false,
                    shell: true,
                    env: {
                        ...process.env,
                        ...env,
                    },
                });
                proc.on('error', fail);
                proc.on('exit', code => {
                    if (code === 0) {
                        return ok();
                    }
                    else {
                        return fail(new toolkit_lib_1.ToolkitError(`${commandAndArgs}: Subprocess exited with error ${code}`));
                    }
                });
            });
        }
        catch (e) {
            await debugFn(`failed command: ${commandAndArgs}`);
            throw e;
        }
    }
}
/**
 * Creates an assembly with error handling
 */
function createAssembly(appDir) {
    try {
        return new cloud_assembly_api_1.CloudAssembly(appDir, {
            // We sort as we deploy
            topoSort: false,
        });
    }
    catch (error) {
        if (error.message.includes(cxschema.VERSION_MISMATCH)) {
            // this means the CLI version is too old.
            // we instruct the user to upgrade.
            throw new toolkit_lib_1.ToolkitError(`This CDK CLI is not compatible with the CDK library used by your application. Please upgrade the CLI to the latest version.\n(${error.message})`);
        }
        throw error;
    }
}
function noUndefined(xs) {
    return Object.fromEntries(Object.entries(xs).filter(([_, v]) => v !== undefined));
}
//# sourceMappingURL=data:application/json;base64,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