\ gamma-aph.4th
\
\ Evaluate the real gamma function.
\
\ Posted to comp.lang.forth by Andrew P. Haley, 03 Nov 2010,
\
\ Message ID:  <4qqdnVv1nNXWFkzRnZ2dnUVZ8uKdnZ2d@supernews.com>
\ with subsequent revisions to the Chebyshev coefficients,
\ and the algorithm.
\
\ Minor revs. by K. Myneni to use with FSL and ttester.
\ Requires fsl-util.4th and ttester.4th for automated tests.
\
\ Revisions:
\   2010-11-21  km  use coefficients 0--16 as default.
\   2010-11-25  aph recoded chebev and updated coefficients to
\                   support quad precision.

BEGIN-MODULE

BASE @
DECIMAL

Public:

\ Single to float
[undefined] s>f [IF] : s>f ( n - F: n)   s>d d>f ; [THEN]

\ The fractional part of x
: frac ( F: x - y)   fdup floor f- ;

\
\ Evaluate a series in Chebyshev polynomials of the first kind.
\
\ Clenshaw's recurrence is:
\
\    b[r] = 2x*b[r+1] - b[r+2] + a[r]
\
\    f[x] = 1/2 (b[0] - b[2])
\
\ http://en.wikipedia.org/wiki/Clenshaw_algorithm
\
\ a0 is address of the zeroth coefficient, an the last.  The zeroth
\ coefficient in the table should be divided by two.

Private:

fvariable 2x
0 ptr a0
0 ptr aN

Public:

: fpstack? ( - n)   false ( s" FLOATING-STACK" environment?) ;

[defined] fpick fpstack? and [if]

: chebev ( a0 an) ( F: x - y)
    2.0e0 f*

    0.0e0 fdup
    begin
        ( b[r+2] b[r+1] )
        fdup  3 fpick f*
        frot f-
        dup f@ f+
        -1 floats +
    2dup = until  drop

    frot 2.0e0 f/ f*  fswap f-  f@ f+ ;

[else]

: chebev ( x a0 aN -- y)   \  ( a0 aN -- ) ( F: x - y)
    to aN  to a0
    2.0e0 f* 2x f!
    0.0e fdup
    a0 float+  aN do
	( b[r+2] b[r+1] )
	fdup  2x f@ f*
	frot f-
	i f@ f+
    -1 floats +loop
    2x f@ 2.0e0 f/ f*  fswap f-  a0 f@ f+
;

[then]

Private:

\ Coefficients for gamma(1+x) in T* shifted Chebyshev series
\
\ 0 <= x <= 1
\
\ The last few coefficients for gamma are removed because they don't
\ contribute much useful accuracy.  If you're using IEEE double
\ precision, which has at best 16.25 decimal digits, you can remove
\ the last few coefficients without significant loss of accuracy.
42 FLOAT ARRAY gamma_coeffs{
+2.82756448404890450051147575148574014913448e+00 2.0e f/ ( 0)      \ 0x2.d3db441a5237716df4464481cbp+0
+4.91728480089007251437848650729437235806210e-01  ( 1)      \ 0x7.de1eaec7e7b96b59874eea918ep-4
+8.53245417060960622435346727280971878890994e-02  ( 2)      \ 0x1.5d7d4442c82c9229c90a614588p-4
+8.21455815467368275020891289112478419820733e-03  ( 3)      \ 0x2.1a596aa01b1aab88ed74c4a801p-8
+8.87927809469078194809786472275984936188918e-04  ( 4)      \ 0x3.a30f4e7230d7614381ad94fb5fp-12
+5.67656791518861034088548510483648348342876e-05  ( 5)      \ 0x3.b85ebc493de0d2ff072316f891p-16
+5.26407619053145198095117058863840983754024e-06  ( 6)      \ 0x5.85108fb22099d43e847aa44fd3p-20
+1.96127848487387649899838984788127109359382e-07  ( 7)      \ 0x3.4a5cd995d61f7b7a751663d06ap-24
+2.42546591996493612690517937698275537675106e-08  ( 8)      \ 0x6.82c47a222fa09c06dcbf76be62p-28
-4.34058439984653663468699186336508486505555e-11  ( 9)      \ -0x2.fb9a8af8b0579cfe0ef866c235p-36
+1.28135422415311174338901362300399798746431e-10  ( 10)      \ 0x8.ce2ea401268c9db9c960c52651p-36
-7.18515874473529316329038294823398090558499e-12  ( 11)      \ -0x7.e671407e3c559c3973789300b9p-40
+9.87886216808062770462185955399768043336321e-13  ( 12)      \ 0x1.1610b43724aebafd2b00d82c4ep-40
-8.93524894238542347524693617836112813583418e-14  ( 13)      \ -0x1.9268681375728c2e118e7f8138p-44
+9.47267335939156879210938251128555644553008e-15  ( 14)      \ 0x2.aa93fb2cf1b1862b9b9916ec23p-48
-9.42668601023366544621327069601938349076763e-16  ( 15)      \ -0x4.3ed2a9b1dfd7e7a9859dadcbbfp-52
+9.59100989075118037123320356436415670738778e-17  ( 16)      \ 0x6.e93aa2ee5b82fdc3a821ef8ee5p-56
\ -9.67795804241081532201738774143729099152318e-18  ( 17)      \ -0xb.286dd5bce5b0a7d95f42459e4ap-60
\ +9.78898946842480322953862400084787630138202e-19  ( 18)      \ 0x1.20eb8362d7a6dcdda36bf169d7p-60
\ -9.89123297962332627184134484307121688582680e-20  ( 19)      \ -0x1.d319ab5bc2aecc09d8d1766a9cp-64
\ +9.99586299858095279824592084660529081833290e-21  ( 20)      \ 0x2.f3441c21f3467439fff0d64c45p-68
\ -1.00999275547836639649039291079858805944760e-21  ( 21)      \ -0x4.c5019e2b5541938d51f0e9d248p-72
\ +1.02046466435189386855738673903030736662282e-22  ( 22)      \ 0x7.b5dda06d8037a37d22352311b5p-76
\ -1.03099230412467302852517806984987341074311e-23  ( 23)      \ -0xc.76c4421acd1da7bf2aeaceae15p-80
\ +1.04159663867088944758443911074701016536367e-24  ( 24)      \ 0x1.425bc9aad4cfc3b73b6dda617fp-80
\ -1.05228550247433352685959139821327395088141e-25  ( 25)      \ -0x2.091108ee0bf86aba5e26867b48p-84
\ +1.06306684189467999446771427963040372148865e-26  ( 26)      \ 0x3.4a3f923c5ee343c1be66f2aae2p-88
\ -1.07394617488341205516060831323799475539940e-27  ( 27)      \ -0x5.51637599f03fc4ac95e8766647p-92
\ +1.08492791668711431067963156594397981902526e-28  ( 28)      \ 0x8.987ec806e3afa3a1ae8db1b1f0p-96
\ -1.09601553475266520955375213415567449013670e-29  ( 29)      \ -0xd.e4c61537a40b84f3848a5a1ffcp-100
\ +1.10721185683698024620697223872768637718484e-30  ( 30)      \ 0x1.674f8ff53d17682771954483a6p-100
\ -1.11851924553547838306504367512940146612185e-31  ( 31)      \ -0x2.44c4b5a1b9f1041590699314c9p-104
\ +1.12993973471618042142813515265585046208301e-32  ( 32)      \ 0x3.aab79ac90021db81b2a52932dep-108
\ -1.14147512511271298858093925814203264575333e-33  ( 33)      \ -0x5.ed47db7b99774ad2ca8a293b50p-112
\ +1.15312705358799424261916415518891744536435e-34  ( 34)      \ 0x9.946e15c5c1d577494c484131afp-116
\ -1.16489704280678397064251419355865996468066e-35  ( 35)      \ -0xf.7bf07157706ab288a8f1b9b5f8p-120
\ +1.17678653697439469465640285333346639642148e-36  ( 36)      \ 0x1.907090b6d10a7d2caf6f18d22ap-120
\ -1.18879692753517842285833897304042646035042e-37  ( 37)      \ -0x2.873e1bcc4a93760dc5bcac421ep-124
\ +1.20092957166535824918253962523072699681715e-38  ( 38)      \ 0x4.16283e691488a4063f908f4dc4p-128
\ -1.21318580558955068312574262395848928513713e-39  ( 39)      \ -0x6.9aef29141781908e62dfd2bd5ap-132
\ +1.22556695418035705069636891309793082621927e-40  ( 40)      \ 0xa.ad1b0b04f26a172e97b3561f86p-136
\ -1.23807433788902465916457019322480229984253e-41  ( 41)      \ -0x1.14198cb83922e48fc519652bacp-136
17 constant ncoeffs
ncoeffs gamma_coeffs{ }fput
gamma_coeffs{  0 }  ptr  gamma_coeffs
gamma_coeffs{ ncoeffs 1- }  ptr  /gamma_coeff

Public:

\ gamma(x+2), 0 <= x <= 1
: gamma(x+2) ( F: x - y)
   2.0e0 f*  1.0e0 f-
   gamma_coeffs /gamma_coeff chebev ;

\ gamma(x), valid for all real x
\ The method basically follows Hart, p133
\ Computer Approximations, John F. Hart et al., Wiley 1968
\
\   X = frac(x)
\   N = x - X - 2
\
\   gamma(x) = prod(k=0..N-1) (X+2+k).gamma(X+2)   N > 0
\   gamma(x) = prod(k=1..-N) gamma(X+2)/(X+2-k)  N < 0
\

\ prod(k=0..n-1) (x+2+k).gamma(x+2)
: gamma1 \ ( n F: x gamma(x+2) - y)
    >r
    fswap  2.0e0 f+  fswap
    r> 0 do  fover f*  fswap 1.0e0 f+ fswap  loop
    fswap fdrop ;

\ prod(k=1..-n) gamma(x+2)/(x+2-k)
: gamma2 \ ( n F: x gamma(x+2) - y)
    >r
    fswap  1.0e0
    r> negate 1- -1 do  fover i s>f f-  f*  loop
    fswap fdrop  f/ ;

\ gamma(x), -1 <= x <= 0
: gamma3 ( F: x - y)
   fdup  1.0e0 f+  fdup gamma(x+2) -2 gamma2
   fswap f/ ;

\ gamma(x)
: gamma ( F: x - y)
   \ Avoid loss of significance for small -ve x
   fdup f0< if
      fdup -1.0e0 f> if  gamma3  exit  then then

   fdup frac  fswap fover
   f- f>d drop 2-  >r    \ R: int(x)-2
   fdup gamma(x+2)       \ F: frac(x)  gamma(x+2)

   r@ 0 > if  r@ gamma1
      else r@ 0< if  r@ gamma2
         else  fswap fdrop        
       then then
   r> drop ; 


BASE !
END-MODULE

TEST-CODE? [IF] \ -----------------------------------------------
[UNDEFINED] T{ [IF] s" ttester" included [THEN]
BASE @
DECIMAL

0e ABS-NEAR F!
6e-16 REL-NEAR F!
SET-NEAR

cr
TESTING GAMMA
t{ -5.50000000000000000000e0   gamma ->  1.091265478190986298673234429377905644050e-2 r}t
t{ -1.50000000000000000000e0   gamma ->  2.363271801207354703064223311121526910397e0  r}t
t{ -5.00000000000000000000e-1  gamma -> -3.544907701811032054596334966682290365595e0  r}t
t{ -9.96093750000000000000e-2  gamma -> -1.072503239536597514958102042245354802713e1  r}t
t{ -9.76562500000000000000e-3  gamma -> -1.029869618821304653530065997279497010271e2  r}t
t{ -9.76562500000000000000e-4  gamma -> -1.024578182406251657399893334832135395808e3  r}t
t{ -9.15527343750000000000e-5  gamma -> -1.092324397288995617696983265948265923149e4  r}t
t{ -9.53674316406250000000e-6  gamma -> -1.048581772250973570707723687558376661871e5  r}t
t{ -9.53674316406250000000e-7  gamma -> -1.048576577216608139658440777112175594556e6  r}t
t{ -8.94069671630859375000e-8  gamma -> -1.118481124388241999670367804518639747770e7  r}t
t{ -9.31322574615478515625e-9  gamma -> -1.073741829772156741128346993949966485455e8  r}t
t{ -9.31322574615478515625e-10 gamma -> -1.073741824577215665822663037401346809860e9  r}t
t{  9.31322574615478515625e-10 gamma ->  1.073741823422784336019597314614097372460e9  r}t
t{  9.31322574615478515625e-9  gamma ->  1.073741818227843443097688207594467448551e8  r}t
t{  8.94069671630859375000e-8  gamma ->  1.118481008945109019362344877219153706402e7  r}t
t{  9.53674316406250000000e-7  gamma ->  1.048575422785278334942024740755654594258e6  r}t
t{  9.53674316406250000000e-6  gamma ->  1.048570227937673889355688064142857546799e5  r}t
t{  9.15527343750000000000e-5  gamma ->  1.092208954154494030761882107122382678852e4  r}t
t{  9.76562500000000000000e-4  gamma ->  1.023423749345567830369498718239930270889e3  r}t
t{  9.76562500000000000000e-3  gamma ->  1.018323574465657785492625352485558252969e2  r}t
t{  9.96093750000000000000e-2  gamma ->  9.552397687164723022417629289987808228824e0  r}t
t{  1.99707031250000000000e-1  gamma ->  4.597968034013318226695456341842933379706e0  r}t
t{  2.99804687500000000000e-1  gamma ->  2.993616880053459168661436986823960210170e0  r}t
t{  3.99902343750000000000e-1  gamma ->  2.218714529939859677739331379384482879923e0  r}t
t{  5.00000000000000000000e-1  gamma ->  1.772453850905516027298167483341145182798e0  r}t
t{  5.99609375000000000000e-1  gamma ->  1.490089134484006114176588874284823585028e0  r}t
t{  6.99707031250000000000e-1  gamma ->  1.298519535868670737632132738963089866089e0  r}t
t{  7.99804687500000000000e-1  gamma ->  1.164449217446022582886308246575328249578e0  r}t
t{  8.99902343750000000000e-1  gamma ->  1.068707497692992543225759611255141289229e0  r}t
t{  1.00000000000000000000e0   gamma ->  1.000000000000000000000000000000000000000e0  r}t
t{  1.50000000000000000000e0   gamma ->  8.862269254527580136490837416705725913988e-1 r}t
t{  2.00000000000000000000e0   gamma ->  1.000000000000000000000000000000000000000e0  r}t
t{  2.50000000000000000000e0   gamma ->  1.329340388179137020473625612505858887098e0  r}t
t{  3.00000000000000000000e0   gamma ->  2.000000000000000000000000000000000000000e0  r}t
t{  5.50000000000000000000e0   gamma ->  5.234277778455352018114900849241819367949e1  r}t
t{  2.00000000000000000000e1   gamma ->  1.216451004088320000000000000000000000000e17 r}t
t{  1.00000000000000000000e2   gamma ->  9.332621544394415268169923885626670049072e155 r}t

BASE !

[THEN]

