/*
 * main.c
 * Main entry point for the kirc IRC client
 * Author: Michael Czigler
 * License: MIT
 */

#include "dcc.h"
#include "editor.h"
#include "helper.h"
#include "network.h"
#include "protocol.h"
#include "terminal.h"
#include "transport.h"

static void kirc_parse_channels(kirc_t *ctx,
        char *value)
{
    char *tok = NULL;
    size_t siz = 0, idx = 0;

    for (tok = strtok(value, ",|"); tok != NULL && idx < KIRC_CHANNEL_LIMIT; tok = strtok(NULL, ",|")) {
        siz = sizeof(ctx->channels[idx]) - 1;
        safecpy(ctx->channels[idx], tok, siz);
        idx++;
    }
}

static void kirc_parse_mechanism(kirc_t *ctx,
        char *value)
{
    char *mechanism = strtok(value, ":");
    char *token = strtok(NULL, ":");
    size_t siz = 0;

    if (strcmp(mechanism, "EXTERNAL") == 0) {
        ctx->mechanism = SASL_EXTERNAL;
    } else if (strcmp(mechanism, "PLAIN") == 0) {
        ctx->mechanism = SASL_PLAIN;
        siz = sizeof(ctx->auth) - 1;
        safecpy(ctx->auth, token, siz);
    }
}

static int kirc_init(kirc_t *ctx)
{
    memset(ctx, 0, sizeof(*ctx));
    
    size_t siz = 0;

    siz = sizeof(ctx->server) - 1;
    safecpy(ctx->server, KIRC_DEFAULT_SERVER, siz);
    
    siz = sizeof(ctx->port) - 1;
    safecpy(ctx->port, KIRC_DEFAULT_PORT, siz);

    ctx->mechanism = SASL_NONE;

    char *env;

    env = getenv("KIRC_SERVER");
    if (env && *env) {
        siz = sizeof(ctx->server) - 1;
        safecpy(ctx->server, env, siz);
    }

    env = getenv("KIRC_PORT");
    if (env && *env) {
        siz = sizeof(ctx->port) - 1;
        safecpy(ctx->port, env, siz);
    }

    env = getenv("KIRC_CHANNELS");
    if (env && *env) {
        kirc_parse_channels(ctx, env);
    }

    env = getenv("KIRC_REALNAME");
    if (env && *env) {
        siz = sizeof(ctx->realname) - 1;
        safecpy(ctx->realname, env, siz);
    }

    env = getenv("KIRC_USERNAME");
    if (env && *env) {
        siz = sizeof(ctx->username) - 1;
        safecpy(ctx->username, env, siz);
    }

    env = getenv("KIRC_PASSWORD");
    if (env && *env) {
        siz = sizeof(ctx->password) - 1;
        safecpy(ctx->password, env, siz);
    }

    env = getenv("KIRC_AUTH");
    if (env && *env) {
        kirc_parse_mechanism(ctx, env);
    }

    return 0;
}

static int kirc_free(kirc_t *ctx)
{
    size_t siz = -1;

    siz = sizeof(ctx->auth);

    if (secure_zero(ctx->auth, siz) < 0) {
        fprintf(stderr, "auth token value not safely cleared\n");
        return -1;
    }

    siz = sizeof(ctx->password);

    if (secure_zero(ctx->password, siz) < 0) {
        fprintf(stderr, "password value not safely cleared\n");
        return -1;
    }

    return 0;
}

static int kirc_args(kirc_t *ctx, int argc, char *argv[])
{
    if (argc < 2) {
        fprintf(stderr, "%s: no arguments\n", argv[0]);
        return -1;
    }

    int opt;
    size_t siz = 0;

    while ((opt = getopt(argc, argv, "s:p:r:u:k:c:a:")) > 0) {
        switch (opt) {
        case 's':  /* server */
            siz = sizeof(ctx->server) - 1;
            safecpy(ctx->server, optarg, siz);
            break;

        case 'p':  /* port */
            siz = sizeof(ctx->port) - 1;
            safecpy(ctx->port, optarg, siz);
            break;

        case 'r':  /* realname */
            siz = sizeof(ctx->realname) - 1;
            safecpy(ctx->realname, optarg, siz);
            break;

        case 'u':  /* username */
            siz = sizeof(ctx->username) - 1;
            safecpy(ctx->username, optarg, siz);
            break;

        case 'k':  /* password */
            siz = sizeof(ctx->password) - 1;
            safecpy(ctx->password, optarg, siz);
            break;

        case 'c':  /* channel(s) */
            kirc_parse_channels(ctx, optarg);
            break;

        case 'a':  /* SASL authentication */
            kirc_parse_mechanism(ctx, optarg);
            break;

        case ':':
            fprintf(stderr, "%s: missing -%c value\n",
                argv[0], opt);
            return -1;

        case '?':
            fprintf(stderr, "%s: unknown argument\n",
                argv[0]);
            return -1;

        default:
            return -1;
        }
    }

    if (optind >= argc) {
        fprintf(stderr, "nickname not specified\n");
        return -1;
    }

    size_t nickname_n = sizeof(ctx->nickname) - 1;
    safecpy(ctx->nickname, argv[optind], nickname_n);

    return 0;
}

static int kirc_run(kirc_t *ctx)
{
    editor_t editor;

    if (editor_init(&editor, ctx) < 0) {
        fprintf(stderr, "editor_init failed\n");
        return -1;
    }

    transport_t transport;

    if (transport_init(&transport, ctx) < 0) {
        fprintf(stderr, "transport_init failed\n");
        return -1;
    }

    network_t network;

    if (network_init(&network, &transport, ctx) < 0) {
        fprintf(stderr, "network_init failed\n");
        return -1;
    }

    dcc_t dcc;

    if (dcc_init(&dcc, ctx) < 0) {
        fprintf(stderr, "dcc_init failed\n");
        network_free(&network);
        return -1;
    }

    if (network_connect(&network) < 0) {
        fprintf(stderr, "network_connect failed\n");
        dcc_free(&dcc);
        network_free(&network);
        return -1;
    }

    if (ctx->mechanism != SASL_NONE) {
        network_send(&network, "CAP REQ :sasl\r\n");
    }

    network_send(&network, "NICK %s\r\n", ctx->nickname);
    
    char *username, *realname;
    
    if (ctx->username[0] != '\0') {
        username = ctx->username;
    } else {
        username = ctx->nickname;
    }

    if (ctx->realname[0] != '\0') {
        realname = ctx->realname;
    } else {
        realname = ctx->nickname;
    }

    network_send(&network, "USER %s - - :%s\r\n",
        username, realname);

    if (ctx->mechanism != SASL_NONE) {
        network_send(&network, "AUTHENTICATE %s\r\n",
            (ctx->mechanism == SASL_EXTERNAL) ? "EXTERNAL" : "PLAIN");
    }

    if (ctx->password[0] != '\0') {
        network_send(&network, "PASS %s\r\n",
            ctx->password);
    }

    size_t siz = sizeof(ctx->target) - 1;
    safecpy(ctx->target, ctx->channels[0], siz);

    terminal_t terminal;

    if (terminal_init(&terminal, ctx) < 0) {
        fprintf(stderr, "terminal_init failed\n");
        dcc_free(&dcc);
        network_free(&network);
        return -1;
    }

    if (terminal_enable_raw(&terminal) < 0) {
        fprintf(stderr, "terminal_enable_raw failed\n");
        terminal_disable_raw(&terminal);
        dcc_free(&dcc);
        network_free(&network);
        return -1;
    }

    struct pollfd fds[2] = {
        { .fd = STDIN_FILENO, .events = POLLIN },
        { .fd = network.transport->fd, .events = POLLIN }
    };

    for (;;) {
        int rc = poll(fds, 2, -1);

        if (rc == -1) {
            if (errno == EINTR) {
                continue;
            }

            fprintf(stderr, "poll error: %s\n",
                strerror(errno));
            break;
        }

        if (rc == 0) {
            continue;
        }

        if (fds[0].revents & (POLLERR | POLLHUP | POLLNVAL)) {
            fprintf(stderr, "stdin error or hangup\n");
            break;
        }

        if (fds[1].revents & (POLLERR | POLLHUP | POLLNVAL)) {
            fprintf(stderr, "network connection error or closed\n");
            break;
        }

        if (fds[0].revents & POLLIN) {
            editor_process_key(&editor);

            if (editor.event == EDITOR_EVENT_TERMINATE)
                break;

            if (editor.event == EDITOR_EVENT_SEND) {
                char *msg = editor_last_entry(&editor);
                network_command_handler(&network, msg);
            }

            editor_handle(&editor);
        }

        if (fds[1].revents & POLLIN) {
            int recv = network_receive(&network);

            if (recv < 0) {
                fprintf(stderr, "network_receive error\n");
                break;
            }

            if (recv > 0) {
                char *msg = network.buffer;

                for (;;) {
                    char *eol = strstr(msg, "\r\n");
                    if (!eol) break; 

                    *eol = '\0';

                    protocol_t protocol;
                    protocol_init(&protocol, ctx);
                    protocol_parse(&protocol, msg);

                    switch(protocol.event) {
                    case PROTOCOL_EVENT_CTCP_CLIENTINFO:
                        if (strcmp(protocol.command, "PRIVMSG") == 0) {
                            network_send(&network,
                                "NOTICE %s :\001PING ACTION CLIENTINFO "
                                "DCC PING TIME VERSION\001\r\n",
                                protocol.nickname);
                        }
                        break;

                    case PROTOCOL_EVENT_CTCP_DCC:
                        if (strcmp(protocol.command, "PRIVMSG") == 0) {
                            dcc_request(&dcc, protocol.nickname,
                                protocol.message);
                        }
                        break;

                    case PROTOCOL_EVENT_CTCP_PING:
                        if (strcmp(protocol.command, "PRIVMSG") == 0) {
                            if (protocol.message[0] != '\0') {
                                network_send(&network,
                                    "NOTICE %s :\001PING %s\001\r\n",
                                    protocol.nickname, protocol.message);
                            } else {
                                network_send(&network,
                                    "NOTICE %s :\001PING\001\r\n",
                                    protocol.nickname);
                            }
                        }
                        break;

                    case PROTOCOL_EVENT_CTCP_TIME:
                        if (strcmp(protocol.command, "PRIVMSG") == 0) {
                            char tbuf[128];
                            time_t now;
                            time(&now);
                            struct tm *info = localtime(&now);
                            strftime(tbuf, sizeof(tbuf), "%c", info);
                            network_send(&network,
                                "NOTICE %s :\001TIME %s\001\r\n",
                                protocol.nickname, tbuf);
                        }
                        break;

                    case PROTOCOL_EVENT_CTCP_VERSION:
                        if (strcmp(protocol.command, "PRIVMSG") == 0) {
                            network_send(&network,
                                "NOTICE %s :\001VERSION kirc %s\001\r\n",
                                protocol.nickname, KIRC_VERSION_MAJOR "."
                                KIRC_VERSION_MINOR "." KIRC_VERSION_PATCH);
                        }
                        break; 

                    case PROTOCOL_EVENT_PING:
                        network_send(&network, "PONG :%s\r\n",
                            protocol.message);
                        break;

                    case PROTOCOL_EVENT_EXT_AUTHENTICATE:
                        network_authenticate(&network);
                        break;

                    case PROTOCOL_EVENT_001_RPL_WELCOME:
                        network_join_channels(&network);
                        break;

                    default:
                        break; 
                    }

                    protocol_handle(&protocol);

                    msg = eol + 2;
                }

                size_t remain = network.buffer
                    + network.len - msg;
                memmove(network.buffer, msg, remain);
                network.len = remain;

                editor_handle(&editor);
            }
            
        }

        dcc_process(&dcc);
    }

    terminal_disable_raw(&terminal);
    dcc_free(&dcc);
    network_free(&network);

    return 0;
}

int main(int argc, char *argv[])
{
    kirc_t ctx;

    if (kirc_init(&ctx) < 0) {
        return EXIT_FAILURE;
    }

    if (kirc_args(&ctx, argc, argv) < 0) {
        kirc_free(&ctx);
        return EXIT_FAILURE;
    }

    if (kirc_run(&ctx) < 0) {
        kirc_free(&ctx);
        return EXIT_FAILURE;
    }

    kirc_free(&ctx);

    return EXIT_SUCCESS;
}
