//----------------------------------------------------------------------------
//
// TSDuck sample C++ application using tsp memory plugins.
//
// This sample application uses the input and output memory plugins at the
// same time. Real applications may use one of them only.
//
//----------------------------------------------------------------------------

#include "tsduck.h"

// Enforce COM and network init on Windows, transparent elsewhere.
TS_MAIN(MainCode);


//----------------------------------------------------------------------------
// An event handler for memory input plugin.
// It is invoked by the "memory" input plugin each time TS packets are needed.
//----------------------------------------------------------------------------

// For the purpose of demonstration, this class sends a predefined list of
// transport stream packets one by one. A real application should get its
// input packets from somewhere else and return them in larger chunks.

class InputHandler : public ts::PluginEventHandlerInterface
{
public:
    // Constructors.
    InputHandler() = delete;
    InputHandler(ts::Report& report);

    // Event handling (from ts::PluginEventHandlerInterface).
    virtual void handlePluginEvent(const ts::PluginEventContext& context) override;

private:
    size_t _next_packet;
    ts::Report& _report;
    static const ts::TSPacketVector _packets;
};

// A predefined list of transport stream packets used as input.
const ts::TSPacketVector InputHandler::_packets{
    {{0x47, 0x00, 0x64, 0x10, 0x27, 0xCD, 0x88, 0x8B, 0x72, 0x99, 0xB9, 0x7E, 0x9F, 0xC8, 0x11, 0xD8,
      0x34, 0xEC, 0xFE, 0x57, 0xFF, 0xF9, 0x80, 0x2F, 0xF5, 0xCD, 0x0B, 0x05, 0xD0, 0x9F, 0x21, 0x23,
      0x60, 0x47, 0x28, 0x80, 0x4E, 0xE3, 0xAD, 0x89, 0xFF, 0x92, 0x85, 0x93, 0x5E, 0x41, 0xB8, 0xCF,
      0xED, 0xA8, 0x41, 0xEC, 0x13, 0x22, 0x7B, 0x5C, 0x37, 0xD2, 0xDB, 0x06, 0xB1, 0xB5, 0xF5, 0x62,
      0x9D, 0x3C, 0x48, 0x99, 0x5B, 0xDD, 0xF9, 0x66, 0x1A, 0xD1, 0xE9, 0xE1, 0x43, 0x6F, 0xDF, 0x17,
      0x0E, 0x85, 0xD1, 0x98, 0x97, 0x64, 0xFB, 0x18, 0xF9, 0x91, 0xB6, 0x7D, 0x5A, 0xBF, 0x2D, 0xAD,
      0xCD, 0x28, 0xC2, 0xDE, 0xB2, 0x47, 0x92, 0x57, 0xF5, 0x5B, 0x1C, 0x71, 0xC8, 0xF0, 0xC9, 0x21,
      0x3E, 0x0D, 0xC3, 0xF8, 0x32, 0x15, 0xEF, 0x55, 0x81, 0xF6, 0x05, 0xA8, 0x0D, 0x83, 0xD2, 0x8D,
      0x34, 0x29, 0x1C, 0x2A, 0x01, 0x07, 0xE4, 0x50, 0x48, 0x55, 0xD4, 0xA8, 0xDF, 0x94, 0x47, 0x4C,
      0x2A, 0x0A, 0x84, 0xB5, 0x6F, 0xD4, 0xF3, 0x87, 0x8C, 0xB1, 0x2B, 0x47, 0x85, 0x0A, 0x2D, 0x1D,
      0x9A, 0x2B, 0x18, 0x45, 0x99, 0x01, 0xC1, 0x93, 0x79, 0xFB, 0x94, 0xF3, 0x52, 0x24, 0x1D, 0xB4,
      0xD9, 0x2F, 0x2D, 0x95, 0xD6, 0xFA, 0xA8, 0xA5, 0x46, 0x62, 0x54, 0x01}},

    {{0x47, 0x00, 0x64, 0x10, 0x9E, 0x9E, 0xD4, 0x70, 0x06, 0xDB, 0x11, 0xD4, 0x7A, 0xA5, 0x97, 0x11,
      0x2E, 0xBC, 0x7D, 0x62, 0x7C, 0xA6, 0x90, 0xB8, 0x49, 0x72, 0x1A, 0xBD, 0xA0, 0x04, 0x0D, 0x59,
      0x42, 0xDD, 0x91, 0x29, 0xA4, 0xD4, 0x21, 0xD1, 0x63, 0xD8, 0x36, 0xC6, 0xAE, 0xD2, 0x65, 0xE0,
      0xA5, 0x85, 0xE9, 0xF9, 0xD9, 0x7E, 0x44, 0xF1, 0x7B, 0xF2, 0x49, 0x19, 0x22, 0xEC, 0x8D, 0x9B,
      0xBC, 0xAB, 0xE3, 0xC9, 0x0A, 0x98, 0x3F, 0xF3, 0xAF, 0x3B, 0x1B, 0xE9, 0x52, 0xCF, 0x70, 0x8A,
      0x2C, 0x22, 0xF9, 0xF8, 0x93, 0x7B, 0xAF, 0xD8, 0x99, 0x97, 0x4F, 0x54, 0x59, 0x3F, 0x69, 0xC4,
      0xFB, 0xD6, 0x93, 0xC8, 0x00, 0x80, 0x08, 0x8D, 0x1B, 0x97, 0xB2, 0xDD, 0x9D, 0x39, 0x2B, 0x55,
      0xA8, 0xAA, 0x2D, 0x15, 0xDB, 0xF4, 0x3E, 0xC6, 0x4C, 0x38, 0x44, 0x3E, 0x04, 0x05, 0x83, 0xAA,
      0x52, 0x8E, 0x81, 0xD6, 0x69, 0x20, 0x33, 0xAA, 0xB6, 0x5C, 0xFF, 0xB8, 0xCB, 0xE5, 0xCC, 0xAB,
      0xB8, 0x73, 0xE8, 0xC3, 0xDE, 0x25, 0x38, 0xCA, 0x8F, 0x38, 0xF4, 0x33, 0xEE, 0xA4, 0xA6, 0x57,
      0x17, 0x2F, 0xC2, 0xCB, 0xE9, 0xD7, 0xCC, 0x42, 0xCA, 0x27, 0xFF, 0xDA, 0xF6, 0x5D, 0x53, 0x86,
      0xBB, 0x7E, 0x0A, 0x62, 0xA9, 0x83, 0xD0, 0x54, 0x2D, 0x50, 0x1F, 0xBC}},

    {{0x47, 0x00, 0x64, 0x10, 0x37, 0x71, 0x69, 0xA4, 0x09, 0xF0, 0xE4, 0x7F, 0xCC, 0x4C, 0xDA, 0x9F,
      0x21, 0x90, 0x12, 0x7A, 0xEC, 0x04, 0x48, 0xC8, 0x6C, 0x9D, 0x0C, 0xF2, 0x3E, 0x3E, 0x5C, 0xCA,
      0xF3, 0x98, 0xF3, 0xCC, 0x09, 0x44, 0x61, 0x30, 0x91, 0x1B, 0xAA, 0x66, 0x8E, 0x8A, 0x2D, 0xDC,
      0x23, 0x9B, 0xF9, 0xE0, 0x8F, 0x67, 0x79, 0xDD, 0x7C, 0x6C, 0xF8, 0x35, 0x82, 0x13, 0x8B, 0x23,
      0xFF, 0x1C, 0xE4, 0x5B, 0x53, 0x17, 0x95, 0xD0, 0x08, 0x40, 0xAE, 0x6B, 0xFF, 0xF9, 0x80, 0x27,
      0x6F, 0xE1, 0xF2, 0x82, 0x85, 0xB6, 0xF5, 0x00, 0x5F, 0x49, 0xF4, 0x67, 0x45, 0x84, 0xE8, 0x76,
      0xE8, 0xB9, 0xE0, 0x06, 0x12, 0x1A, 0x6A, 0x5F, 0x16, 0x7D, 0x59, 0x5A, 0x4D, 0xA2, 0x7F, 0x4D,
      0x93, 0x85, 0x93, 0xEA, 0x79, 0x2D, 0x8C, 0xAB, 0xA1, 0x6C, 0xE4, 0x04, 0x80, 0xB1, 0x87, 0x40,
      0xCB, 0x07, 0x93, 0xED, 0xAA, 0x8C, 0x82, 0xDD, 0xA6, 0x4A, 0xFB, 0x1F, 0x95, 0x6A, 0x2C, 0xEF,
      0x61, 0x09, 0x73, 0x07, 0x9C, 0xDA, 0x03, 0xE0, 0x49, 0x94, 0x19, 0x4C, 0xE1, 0xCF, 0x99, 0xE9,
      0x42, 0xDE, 0xCD, 0x4F, 0xEC, 0x79, 0xD8, 0xF6, 0xE2, 0xE2, 0xEB, 0x2A, 0x6C, 0x66, 0x2E, 0xBE,
      0x44, 0x5C, 0xC7, 0x97, 0xE2, 0x3A, 0xC9, 0xB5, 0x52, 0xA8, 0xEF, 0x3F}}
};

// Constructor.
InputHandler::InputHandler(ts::Report& report) :
    _next_packet(0),
    _report(report)
{
}

// This event handler is called each time the memory plugin needs input packets.
void InputHandler::handlePluginEvent(const ts::PluginEventContext& context)
{
    ts::PluginEventData* data = dynamic_cast<ts::PluginEventData*>(context.pluginData());
    if (data != nullptr) {
        if (_next_packet < _packets.size() && data->maxSize() >= ts::PKT_SIZE) {
            _report.info(u"returning input packet #%d", {_next_packet});
            data->append(&_packets[_next_packet++], ts::PKT_SIZE);
        }
        else {
            _report.info(u"end of input");
        }
    }
}


//----------------------------------------------------------------------------
// An event handler for memory output plugin.
// It is invoked by the "memory" output plugin each time TS packets are sent.
//----------------------------------------------------------------------------

class OutputHandler : public ts::PluginEventHandlerInterface
{
public:
    // Constructors.
    OutputHandler() = delete;
    OutputHandler(ts::Report& report);

    // Event handling (from ts::PluginEventHandlerInterface).
    virtual void handlePluginEvent(const ts::PluginEventContext& context) override;

private:
    ts::Report& _report;
};

// Constructor.
OutputHandler::OutputHandler(ts::Report& report) :
    _report(report)
{
}

// This event handler is called each time the memory plugin sends output packets.
void OutputHandler::handlePluginEvent(const ts::PluginEventContext& context)
{
    ts::PluginEventData* data = dynamic_cast<ts::PluginEventData*>(context.pluginData());
    if (data != nullptr) {
        const ts::TSPacket* packets = reinterpret_cast<const ts::TSPacket*>(data->data());
        const size_t packets_count = data->size() / ts::PKT_SIZE;
        _report.info(u"received %d output packets", {packets_count});
        for (size_t i = 0; i < packets_count; ++i) {
            _report.info(u"packet #%d: %s", {i, ts::UString::Dump(packets + i, ts::PKT_SIZE, ts::UString::COMPACT)});
        }
    }
}


//----------------------------------------------------------------------------
// Application entry point.
//----------------------------------------------------------------------------

int MainCode(int argc, char* argv[])
{
    // Create a thread-safe asynchronous report.
    ts::AsyncReport report;

    // Create our event handlers for the memory plugins.
    InputHandler input(report);
    OutputHandler output(report);

    // Create a transport stream processor and register our event handlers.
    ts::TSProcessor tsp(report);
    tsp.registerEventHandler(&input, ts::PluginType::INPUT);
    tsp.registerEventHandler(&output, ts::PluginType::OUTPUT);

    // Build tsp options.
    ts::TSProcessorArgs opt;
    opt.input = {u"memory", {}};
    opt.plugins = {
        {u"count", {}},
    };
    opt.output = {u"memory", {}};

    // Start the TS processing.
    if (!tsp.start(opt)) {
        return EXIT_FAILURE;
    }

    // And wait for TS processing termination.
    tsp.waitForTermination();
    return EXIT_SUCCESS;
}
