/*  aptsignal.c
 *
 *  APT signal processing functions of wxapt application
 */

/*
 *  wxapt: An application to decode APT signals from
 *  weather satellites and produce an image of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "aptsignal.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  Detect_NOAA_Sync()
 *
 *  Detects the Channel A sync of a NOAA APT signal.
 *  This is 7 cycles at 1040 Hz on the 2.4 KHz sub-carrier.
 */

  void
Detect_NOAA_Sync( int search_limit )
{
  /* FIFO Buffer for ch A sync detection */
  static int sync_fifo[NOAA_SYNC_FIFO_SIZE];

  /* An amplitude sample of 2.4kHz sub-carrier */
  short	sample_a;

  int
	sync_idx,    /* Counter used in detecting sync train position  */
	carrier_val, /* Instantaneous value of 2.4 Khz sub-carrier     */
	sync_val,    /* Value produced by sync detection algorithm     */
	sync_max,    /* Max value produced by sync detection algorithm */
	sync_ave,    /* Average value of carrier samples in sync fifo  */
	sync_pos,    /* Position in sample buffer where sync o/p peaks */
	peak_idx,    /* Index used in searching for peak in sync value */
	fifo_idx,    /* Index for sync detection fifo */
	sync_cnt,	 /* Counter of sync pulses processed */
	idx;         /* Index for various loops etc.  */


  /* Move through buffer up to search_limit, looking   */
  /* for sync detector peak. On exit, the buffer index */
  /* should be at the end of 'A' sync pulse sequence.  */

  sync_max = fifo_idx = 0;
  sync_pos = gbl_line_idx;
  for( peak_idx = 0; peak_idx < search_limit; peak_idx++ )
  {
	/* Read a carrier sample */
	sample_a = gbl_line_buffer[gbl_line_idx++];

	/* This is the instantaneous magnitude of the 2.4 Khz  */
	/* subcarrier, calculated as the scalar magnitude of   */
	/* two consecutive signal samples at fi deg phase angle */
	Carrier_Amplitude( sample_a, &carrier_val );

	/* Store carrier amplitude samples in fifo in cyclic manner */
	if( ++fifo_idx >= NOAA_SYNC_FIFO_SIZE ) fifo_idx = 0;
	sync_fifo[fifo_idx] = carrier_val;

	/* Evaluate the average of the samples in fifo, to */
	/* be used as a zero reference in summation below. */
	sync_val = 0;
	for( idx = 0; idx < NOAA_SYNC_FIFO_SIZE; idx++ )
	  sync_val += sync_fifo[idx];
	sync_ave = sync_val / NOAA_SYNC_FIFO_SIZE;

	/* Subtract carrier values from accumulator
	 * for the leading zero amplitude period */
	sync_val = 0;
	sync_idx = fifo_idx;
	for( idx = 0; idx < NOAA_SYNC_LEAD; idx++ )
	{
	  sync_val -= sync_fifo[sync_idx] - sync_ave;
	  if( ++sync_idx >= NOAA_SYNC_FIFO_SIZE )
		sync_idx = 0;
	}

	/* Add and subtract NOAA_SYNC_SAMPLES values to the
	 * accumulator, for each half cycle of the sync train */
	for( sync_cnt = 0; sync_cnt < NOAA_SYNC_PULSES; sync_cnt++ )
	{
	  /* Add carrier values into accumulator */
	  for( idx = 0; idx < NOAA_SYNC_SAMPLES; idx++ )
	  {
		sync_val += sync_fifo[sync_idx] - sync_ave;
		if( ++sync_idx >= NOAA_SYNC_FIFO_SIZE )
		  sync_idx = 0;
	  }

	  /* Subtract carrier values from accumulator */
	  for( idx = 0; idx < NOAA_SYNC_SAMPLES; idx++ )
	  {
		sync_val -= sync_fifo[sync_idx] - sync_ave;
		if( ++sync_idx >= NOAA_SYNC_FIFO_SIZE )
		  sync_idx = 0;
	  }

	} /* for( sync_cnt = 0; sync_cnt < NOAA_SYNC_PULSES; sync_cnt++ ) */

	/* Subtract carrier values for the tail end of the sync train */
	for( sync_cnt = 0; sync_cnt < NOAA_SYNC_TAIL; sync_cnt++ )
	{
	  sync_val -= sync_fifo[sync_idx] - sync_ave;
	  if( ++sync_idx >= NOAA_SYNC_FIFO_SIZE )
		sync_idx = 0;
	}

	/* Record position in buffer where sync detector peaks */
	if( sync_val > sync_max )
	{
	  sync_max = sync_val;
	  sync_pos = gbl_line_idx;
	}

  } /* End of for( peak_idx = 0; peak_idx < search_limit; peak_idx++ ) */

  /* Decoding from DSP */
  if( isFlagSet(ACTION_PROCESS_DSP) ||
	  isFlagSet(ACTION_RECORD_APT) )
  {
	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > NOAA_SYNC_LOWER) &&
		(sync_max < NOAA_SYNC_UPPER) )
	{
  	  gbl_line_idx = sync_pos;
	  gbl_sync_err = sync_pos - gbl_sync_ref;
	}
	else
	{
	  gbl_line_idx = 0;
	}
  } /* if( isFlagClear(ACTION_PROCESS_DSP) ) */
  else if( isFlagSet(ACTION_PROCESS_FILE) )
  {
	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > NOAA_SYNC_LOWER) &&
		(sync_max < NOAA_SYNC_UPPER) )
	  gbl_line_idx = sync_pos;
	else
	  gbl_line_idx = 0;
  }
  else gbl_line_idx = 0;

} /* End of Detect_NOAA_Sync() */

/*------------------------------------------------------------------------*/

/*  Detect_Meteor_Sync()
 *
 *  Detects the sync train of a Meteor type APT signal.
 *  This is 5 cycles at 1200 Hz on the 2.4 KHz sub-carrier.
 */

  void
Detect_Meteor_Sync( int search_limit )
{
  /* FIFO Buffer for sync detection */
  static int sync_fifo[METEOR_SYNC_FIFO_SIZE];

  /* An amplitude sample of the 2.4kHz sub-carrier */
  short sample_a;

  int
	sync_idx,      /* Counter used in detecting sync train position  */
	carrier_val,   /* Instantaneous value of 2.4 Khz sub-carrier     */
	sync_val,      /* Value produced by sync detection algorithm     */
	sync_max,      /* Max value produced by sync detection algorithm */
	sync_ave,      /* Average value of carrier samples in sync fifo  */
	sync_pos,      /* Position in sample buffer where sync o/p peaks */
	sync_cnt,      /* Counter of sync pulses processed */
	peak_idx,      /* Index used in searching for peak in sync value */
	fifo_idx,      /* Index for sync detection fifo */
	idx;           /* Index for various loops etc.  */


  /* Move through buffer until sync detector  */
  /* peaks. On exit, the buffer index should  */
  /* be at the end of the sync pulse sequence */

  sync_max = fifo_idx = 0;
  sync_pos = gbl_line_idx;
  for( peak_idx = 0; peak_idx < search_limit; peak_idx++ )
  {
	/* Read a carrier sample */
	sample_a = gbl_line_buffer[gbl_line_idx++];

	/* This is the instantaneous magnitude of the 2.4 Khz  */
	/* subcarrier, calculated as the scalar magnitude of   */
	/* two consecutive signal samples at 90 deg phase angle */
	Carrier_Amplitude( sample_a, &carrier_val );

	/* Store samples in fifo in cyclic manner */
	if( ++fifo_idx == METEOR_SYNC_FIFO_SIZE ) fifo_idx = 0;
	sync_fifo[fifo_idx] = carrier_val;

	/* Evaluate the average of the samples in fifo, to */
	/* be used as a zero reference in summation below. */
	sync_val = 0;
	for( idx = 0; idx < METEOR_SYNC_FIFO_SIZE; idx++ )
	  sync_val += sync_fifo[idx];
	sync_ave = sync_val/METEOR_SYNC_FIFO_SIZE;

	/* Subtract 5x2-sample groups and add 5x4-sample groups */
	/* to the sync accumulator alternately, simulating      */
	/* synchromous detection. The sync train appears to be  */
	/* 5 cycles at 1200 Hz with a 2/1 mark/space ratio.     */
	/* Their average value is used as a zero reference.     */
	sync_val = 0;
	sync_idx = fifo_idx;
	for( sync_cnt = 0; sync_cnt < METEOR_SYNC_PULSES; sync_cnt++ )
	{
	  /* Subtract carrier samples from accumulator */
	  for( idx = 0; idx < METEOR_SYNC_MARK; idx++ )
	  {
		sync_val -= sync_fifo[fifo_idx] - sync_ave;
		if( ++sync_idx >= METEOR_SYNC_FIFO_SIZE )
		  sync_idx = 0;
	  }

	  /* Add carrier samples into accumulator */
	  for( idx = 0; idx < METEOR_SYNC_SPACE; idx++ )
	  {
		sync_val += sync_fifo[fifo_idx] - sync_ave;
		if( ++sync_idx >= METEOR_SYNC_FIFO_SIZE )
		  sync_idx = 0;
	  }

	} /* for( sync_cnt = 0; sync_cnt < METEOR_SYNC_PULSES; sync_cnt++ ) */

	/* Record position in buffer where sync detector peaks */
	if( sync_val > sync_max)
	{
	  sync_max = sync_val;
	  sync_pos = gbl_line_idx;
	}
  } /* End of for( peak_idx = 0; peak_idx < search_limit-1; peak_idx++ ) */

  /* Decoding from DSP */
  if( isFlagClear(ACTION_PROCESS_FILE) )
  {
	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > METEOR_SYNC_LOWER) &&
		(sync_max < METEOR_SYNC_UPPER) )
	{
	  /* Display sync error */
	  gbl_line_idx = sync_pos;
	  gbl_sync_err = sync_pos - gbl_sync_ref;
	}
	else /* If sync level is out of range, show 'Unlock' */
	{
	  gbl_line_idx = 0;
	} /* if( (sync_max > METEOR_SYNC_LOWER) && */

  } /* if( isFlagClear(ACTION_PROCESS_FILE) ) */
  else
  {
	/* Setup buffer index to max sync position */
	/* if sync maximum is within the threshold */
	if( (sync_max > METEOR_SYNC_LOWER) &&
		(sync_max < METEOR_SYNC_UPPER) )
	  gbl_line_idx = sync_pos;
	else
	  gbl_line_idx = 0;
  }

} /* End of Detect_Meteor_Sync() */

/*------------------------------------------------------------------------*/

/*  Detect_Sync_Train()
 *
 *  Detects sync train position in line buffer
 *  depending on the type of satellite processed
 */

  int
Detect_Sync_Train( int search_limit )
{
  gbl_line_idx = 0;
  switch( gbl_sat_type )
  {
	case NOAA_15: case NOAA_18: case NOAA_19:
	  Detect_NOAA_Sync( search_limit );
	  break;

	case METEOR:
	  Detect_Meteor_Sync( search_limit );
  }

  return( gbl_line_idx );
} /* End of Detect_Sync_Train() */

/*------------------------------------------------------------------------*/

/* Init_Reception()
 *
 * Initialize Reception of signal from Satellite
 */
  void
Init_Reception( void )
{
  /* Setup sound card or rtlsdr device */
  if( isFlagSet(USE_RTLSDR_RX) )
  {
	/* Initialize RTLSDR device */
	Rtlsdr_Init();
  }
  else
  {
	/* Set up Sound Card */
	Open_Capture();
  }

} /* Init_Reception() */

/*------------------------------------------------------------------------*/

/*  Setup_Signal_Level()
 *
 *  Displays a simple graphical aid for setting
 *  up the signal level from the APT receiver FIXME
 */

  void
Setup_Signal_Level( void )
{
  int
	carrier_max, /* Max value of carrier in  buffer */
	carrier_val, /* Sampled value of 2.4kHz carrier */
	idx;         /* Index used in loops etc */

  /* Initialize Reception, aborts on error */
  Init_Reception();

  /* Fill in the samples buffer from DSP, abort on error */
  while( isFlagSet(ACTION_SETUP_AUDIO) )
  {
	Read_Device_Buffer(gbl_line_buffer, BLOCK_SIZE);

	/*** Find max value of 2.4kHz subcarrier in buffer ***/
	carrier_max = 0;
	for( idx = 0; idx < BLOCK_SIZE; idx++ )
	{
	  /* Calculate carrier amplitude
	   * from two consecutive samples */
	  Carrier_Amplitude( gbl_line_buffer[idx], &carrier_val );

	  /* Find maximum carrier level */
	  if( carrier_val > carrier_max )
		carrier_max = carrier_val;
	} /* for( idx = 0; idx < BLOCK_SIZE; idx++ ) */
	printf( "\nSignal Level: %6d  (Maximum 32000)", carrier_max );

  } /* while( isFlagSet(ACTION_SETUP_AUDIO) ) */

  puts( "\nExiting Signal Level Setup" );
  exit( 0 );

} /* Setup_Signal_Level() */

/*------------------------------------------------------------------------*/

/*  Record_APT_Signal()
 *
 *  Records samples of the 2.4 Khz subcarrier in a file
 */

  void
Record_APT_Signal( void )
{
  /* Number of APT image lines processed */
  int line_cnt;

  /* Temp buffer to write line buffer to file */
  unsigned char buff[2];
  int idx;

  /* Samples written to file */
  size_t sample_cnt;


  /* Initialize Reception, abort on error */
  Init_Reception();

  /* Process file name */
  File_Name( gbl_samples_file );

  /* Add extension to file name */
  size_t s = sizeof( gbl_samples_file );
  Strlcat( gbl_samples_file, "-", s );
  Strlcat( gbl_samples_file, gbl_sat_names[gbl_sat_type], s );
  Strlcat( gbl_samples_file, ".bin", s );

  /* Create a file for writing samples */
  gbl_samples_fp = fopen( gbl_samples_file, "w+" );
  if( gbl_samples_fp == NULL )
  {
	perror( gbl_samples_file );
	exit( -1 );
  }

  /* Read samples from DSP, stop on user interrupt */
  /* or at time limit in number of APT lines read. */
  /* gbl_duration is in seconds of recording time. */
  line_cnt = 0;

  /* Write satellite type to samples file, abort on error */
  sample_cnt = fwrite(&gbl_sat_type, 4, 1, gbl_samples_fp);
  if( sample_cnt < 1 )
  {
	perror( "wxapt: fwrite()" );
	fprintf( stderr, "Failed to write to samples file\n" );
	exit( -1 );
  }
  if( sample_cnt != 1 )
  {
	fprintf( stderr, "Failed to write to samples file\n" );
	exit( -1 );
  }

  puts( "Recording APT Signal Samples to File...");

  /* Stop if line count reaches time */
  /* limit or on user stop request   */
  while( isFlagSet(ACTION_RECORD_APT) &&
	    (++line_cnt / 2 != gbl_duration) )
  {
	/* Fill samples buffer from DSP */
	Fill_Samples_Buffer();

	/* Write carrier sample values to samples file, abort on error */
	for( idx = 0; idx < BLOCK_SIZE; idx++ )
	{
	  buff[0] = gbl_line_buffer[idx] & 0xff;
	  buff[1] = (gbl_line_buffer[idx] >> 8) & 0xff;

	  sample_cnt = fwrite(buff, 1, 2, gbl_samples_fp);
	  if( sample_cnt < 1 )
	  {
		perror( "wxapt: fwrite()" );
		fprintf( stderr, "Failed to write to samples file\n" );
	  }

	  if( (sample_cnt != 2) )
	  {
		fprintf( stderr, "Failed to write to samples file\n" );
		exit( -1 );
	  }

	} /* for( idx = 0; idx < BLOCK_SIZE; idx++ ) */
  } /* while( isFlagSet(ACTION_RECORD_APT) && */

  if( isFlagClear(ACTION_RECORD_APT) )
	puts("\nAborting Recording due to User Interrupt");
  else
	puts( "Finished Recording APT Signal Samples to File");

} /* End of Record_APT_Signal() */

/*------------------------------------------------------------------------*/

/*  File_Fill_Buffer()
 *
 *  Fills the signal sample buffer from source file
 */

  int
File_Fill_Buffer( void )
{
  /* For reading processed dsp data from file */
  size_t sample_cnt;

  /* Temp buffer to write line buffer to file */
  unsigned char buff[2];
  int idx;


  /* Fill in the samples buffer from file, return on error */
  gbl_line_idx = 0;
  for( idx = 0; idx < BLOCK_SIZE; idx++ )
  {
	sample_cnt = fread(buff, 1, 2, gbl_samples_fp);

	if( feof(gbl_samples_fp) ) return( FALSE );

	if( sample_cnt != 2 )
	{
	  perror( "wxapt: fread()" );
	  fprintf( stderr, "Failed to read from samples file\n" );
	  exit( -1 );
	}

	gbl_line_buffer[idx]  = buff[0];
	gbl_line_buffer[idx] |= buff[1] << 8;
	if( gbl_line_buffer[idx] & 0x8000 )
	  gbl_line_buffer[idx] -= 0x10000;

  } /* for( idx = 0; idx < BLOCK_SIZE; idx++ ) */

  /* Set buffer index just after sync train */
  /* Search for sync up to 1.5* sync refer. */
  Detect_Sync_Train((3 * gbl_sync_refs[gbl_sat_type]) / 2);

  /* If sync detection is far out, keep */
  /* buffer index in default position */
  if( abs(gbl_line_idx - gbl_sync_ref) > SYNC_ACCEPT )
	gbl_line_idx = gbl_sync_ref;

  return( TRUE );
} /* End of File_Fill_Buffer() */

/*------------------------------------------------------------------------*/

/* Read_Device_Buffer()
 *
 * Reads the samples buffer of the RTLSDR device
 * or the samples buffer of the Sound Card DSP
 */
  void
Read_Device_Buffer( short *buffer, int buff_size )
{
  if( isFlagSet(USE_RTLSDR_RX) )
  {
	/* Fill in the line buffer from RTL device */
	Read_RTL_Buffer(buffer, buff_size);
  }
  else
  {
	/* Fill in the line buffer from Sound DSP */
	Read_SND_Buffer(buffer, buff_size);
  } /* if( isFlagSet(USE_RTLSDR_RX) ) */

} /* Read_Device_Buffer() */

/*------------------------------------------------------------------------*/

/*  Fill_Samples_Buffer()
 *
 *  Fills samples buffer from Sound DSP or RTLSDR device.
 *  Buffer filling is synchronized with APT sync train.
 */

  void
Fill_Samples_Buffer( void )
{
  static int block_size = BLOCK_SIZE;


  /* Search for sync up to 1.5 * sync ref, return if sync fails */
  Read_Device_Buffer( gbl_line_buffer, block_size + dsp_rate_err );
  Detect_Sync_Train((5 * gbl_sync_ref) / 4);

  /* Try to keep APT sync train at beginning of buffer. This  */
  /* done done by increasing or decreasing the data block     */
  /* size argument passed to read(), by adding or subtracting */
  /* half the difference between detected sync position and a */
  /* reference value. Error is limited to +-2 to avoid coarse */
  /* corrections to sync position due to noise interference.  */
  if( gbl_sync_err > SYNC_ACCEPT )
	gbl_sync_err = SYNC_ACCEPT;
  else if( gbl_sync_err < -SYNC_ACCEPT )
	gbl_sync_err = -SYNC_ACCEPT;
  gbl_line_idx += gbl_sync_err;

  /* Vary the data block size argument to read() to
   * take up any drift in sync position due to errors
   * in the DSP sampling rate and satellite movement.
   * Only half the sync error is used to slow and
   * stabilize sync correction. */
  block_size = BLOCK_SIZE + gbl_sync_err/2;

} /* End of Fill_Samples_Buffer() */

/*------------------------------------------------------------------------*/

/*  Synchronize_Buffer()
 *
 *  Synchronizes samples buffer with the APT sync train
 */

  void
Synchronize_Buffer( void )
{
  /* Data block size used as read() argument  */
  int block_size;

  int
	last_line_idx,	/* Saved value of samples buffer index */
	rate_ok_cnt;	/* Count number of similar rate measurements */


  /* Initialize Reception, abort on error */
  Init_Reception();

  /* Load appropriate sync reference */
  gbl_sync_ref = gbl_sync_refs[ gbl_sat_type ];
  dsp_rate_err  = 0;
  last_line_idx = 0;
  rate_ok_cnt   = 0;

  /* Measure ADC sample rate error
   * compared to satellite's timing */
  SetFlag(ACTION_RATE_ERROR);

  puts( "Synchronizing with APT Sync Train..." );

  /*** Find DSP rate error ***/
  /* Locate sync train (sets gbl_line_idx just after it) */
  while( isFlagSet(ACTION_SYNC_BUFFER) &&
	  (!gbl_line_idx || abs(gbl_sync_err) > SYNC_ACCEPT) )
  {
	if( isFlagSet(ACTION_RATE_ERROR) )
	{
	  /* Fill in the line buffer from Device DSP */
	  Read_Device_Buffer(gbl_line_buffer, BLOCK_SIZE);

	  /* Locate sync train (sets gbl_line_idx just after it) */
	  Detect_Sync_Train(BLOCK_SIZE);

	  /* DSP rate error is change in sync train position */
	  if( gbl_line_idx )
	  {
		dsp_rate_err  = gbl_line_idx - last_line_idx;
		last_line_idx = gbl_line_idx;

		/* If measured dsp sampling rate error is
		 * within limits, signal end of measurement */
		if( abs(dsp_rate_err) > SYNC_ACCEPT )
		  rate_ok_cnt = 0;
		else
		  rate_ok_cnt++;
	  }

	  /* Finish with measuring dsp rate error */
	  if( rate_ok_cnt >= RATE_OK_COUNT )
		ClearFlag(ACTION_RATE_ERROR);
	  else continue;

	  /* Dummy-read samples to align buffer with sync ref. */
	  block_size = gbl_line_idx - gbl_sync_ref;
	  if( block_size < 0 )
		block_size += BLOCK_SIZE;
	  block_size += (dsp_rate_err * block_size)/BLOCK_SIZE;
	  Read_Device_Buffer(gbl_line_buffer, block_size);
	  gbl_sync_err = 0;

	} /* if( isFlagSet(ACTION_RATE_ERROR) ) */

	/* Abort on user interrupt */
	if( isFlagClear(ACTION_SYNC_BUFFER) )
	{
	  puts( "\nAborting due to User Interrupt" );
	  exit( 0 );
	}

	/*** Refine buffer alignment with sync reference ***/
	/* Dummy-read samples to align buffer with sync reference */
	block_size = BLOCK_SIZE + dsp_rate_err + gbl_sync_err;
	Read_Device_Buffer(gbl_line_buffer, block_size);

	/* Locate sync train (sets gbl_line_idx just after it) */
	Detect_Sync_Train(BLOCK_SIZE);

  } /* while( isFlagSet(ACTION_SYNC_BUFFER)... */

  if( isFlagSet(ACTION_SYNC_BUFFER) )
	puts( "Synchronized with APT Sync Train" );
  else
  {
	puts( "\nAborting due to User Interrupt" );
	exit( 0 );
  }

} /* End of Synchronize_Buffer() */

/*------------------------------------------------------------------------*/

/* Carrier_Amplitude()
 *
 * The instantaneous amplitude a of the 2.4 KHz sub-carrier
 * is calculated from an incoming sample (the I sample) and
 * one that was delayed by 90 deg in a circular buffer
 * the Q sample). The amplitude is A = SQRT( I^2 + Q^2 ).
 */
  void
Carrier_Amplitude( short sample, int *amplitude )
{
  /* Circular samples buffer, length is 1/4 of the
   * samples/cycle of the 2.4 kHz sub-carrier */
  static short *buffer = NULL;

  static int
	samples_cycle = 0,	/* Amplitude samples per sub-carrier cycle */
	samples_sum   = 0,  /* Summation of sub-carrier samples over 1 cycle */
	samples_cnt   = 0,	/* Count of Amplitude samples summated */
	dc_offset = 0,	/* DC offset of the demodulated signal */
	buf_size  = 0,	/* Circular buffer size */
	buf_idx   = 0;	/* Buffer index */

  /* The value as float of the i and q
   * samples and the sum of their squares */
  double
	id = 0.0,
	qd = 0.0;


  /* Initialize on first call */
  if( buffer == NULL )
  {
	/* Number of amplitude samples per sub-carrier cycle */
	samples_cycle = SND_DSP_RATE / CARRIER_FREQ;

	/* Allocate circular buffer. Size is 1/4 of sub-carrier cycle */
	buf_size = SND_DSP_RATE / CARRIER_FREQ / 4;
	mem_alloc( (void *)&buffer, buf_size * sizeof(short) );
	bzero( buffer, buf_size * sizeof(short) );
  } /* if( buffer == NULL ) */

  /* Summate sub-carrier samples */
  samples_sum += sample;
  samples_cnt++;

  /* Calculate DC offset per cycle */
  if( samples_cnt >= samples_cycle )
  {
	samples_cnt = 0;
	dc_offset = samples_sum / samples_cycle;
	samples_sum = 0;
  }

  /* Sum of squares of I and Q samples.
   * The calculated DC offset is removed */
  sample -= dc_offset;
  id = (double)sample;
  qd = (double)buffer[buf_idx];

  /* The instantaneous amplitude of the sub-carrier */
  *amplitude = (int)hypot( id, qd );
  if( *amplitude > MAX_SUBCARRIER_AMPL )
	*amplitude = MAX_SUBCARRIER_AMPL;

  /* Store I sample and advance
   * index of I and Q samples */
  buffer[buf_idx] = sample;
  buf_idx++;
  if( buf_idx >= buf_size )
	buf_idx = 0;

  return;
} /* End of Carrier_Amplitude() */

/*------------------------------------------------------------------------*/

