/*
 *  wxapt: An application to decode APT signals from weather
 *  satellites and produce an image of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */


#ifndef COMMON_H
#define COMMON_H	1

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <rtl-sdr.h>
#include <pthread.h>
#include "sound.h"

/* Color map data */
typedef struct
{
  /* Pseudo-colorization maps */
  unsigned char
	gray_from[6],	gray_to[6],
	red_from[6],	red_to[6],
	green_from[6],	green_to[6],
	blue_from[6],	blue_to[6];

  int
	white_thld,  /* White (clouds) threshold */
	num_ranges;  /* Number of grayscale ranges */

} color_map_t;

/* Runtime config data */
typedef struct
{
  uint32_t
	rtlsdr_dev_index,	/* librtlsdr Device index  */
	rtlsdr_center_freq;	/* Center Frequency for SDR Tuner */

  int
	rtlsdr_lpf_bw,		/* Bandwidth in Hz of the Butterworth LPF */
	rtlsdr_freq_corr;	/* Correction factor in ppm for synthesizer */

  char
	pcm_dev[32],  /* PCM device name */
	cap_src[32],  /* Capture source  */
	cap_vol[32],  /* Capture volume  */
	mst_vol[32],  /* Master volume control */
	pcm_vol[32];  /* Playback volume control */

  int
	channel,	/* ALSA "channel" for use in mixer setup */
	cap_lev;	/* Recording/Capture level */

  int
	use_chn,		/* Channel in use (frontleft=0, frontright=1 etc) */
	snd_buf_size,	/* Size of sound card DSP buffer */
	num_chn;		/* Number of signal channels (2=stereo, 1=mono) */

  /* wxapt working directory */
  char wxapt_dir[64];

  /* Default duration in
   * sec of image decoding */
  int default_dur;

  /* Color map data */
  color_map_t
	noaa_A_map,
	meteor_map;

} rc_data_t;

/* Actions for above */
enum
{
  INIT_BWORTH_LPF = 0,
  RTL_DAGC_ENABLE
};

/* Return values */
#define ERROR		1
#define SUCCESS		0

#define FALSE	0
#define TRUE	1

/* Generel definitions for image processing */
#define MAX_FILE_NAME      128 /* Max length for optional filenames */

/* Size of char arrays (strings) for error messages etc */
#define MESG_SIZE	128

/* Flow control flags */
#define ACTION_PROCESS_DSP		0x000001 /* Process images from dsp       */
#define ACTION_PROCESS_FILE		0x000002 /* Process images from file      */
#define ACTION_RECORD_APT		0x000004 /* Record apt signal to file     */
#define ACTION_SETUP_AUDIO		0x000008 /* Run signal level setup func.  */
#define ACTION_SYNC_BUFFER		0x000010 /* Run buffer synchronizing func */
#define ACTION_RATE_ERROR		0x000020 /* Measure dsp sampling rate err */
#define ACTION_FLAGS_ALL		0x00003F /* All action flags (clearing)   */
#define IMAGE_NORMALIZE			0x000040 /* Histogram normalize wx image  */
#define IMAGE_COLORIZE			0x000080 /* Pseudo Colorize wx image	  */
#define IMAGE_ROTATE			0x000100 /* Rotate wx image 180 degrees   */
#define CAPTURE_SETUP			0x000200 /* Sound Capture has been set-up */
#define MIXER_SETUP				0x000400 /* Sound Mixer has been set-up   */
#define START_STOP_TIMER		0x000800 /* Start-Stop timer is set       */
#define RTLSDR_INIT				0x001000 /* RTLSDR device initialized OK  */
#define USE_RTLSDR_RX			0x002000 /* Use RTLSDR to receive images  */

#define TWOPI	6.283185307
#define PI2		1.570796327

/*---------------------------------------------------------------------*/

/* Function prototypes produced by cproto */
/* aptsignal.c */
void Detect_NOAA_Sync(int search_limit);
void Detect_Meteor_Sync(int search_limit);
int Detect_Sync_Train(int search_limit);
void Init_Reception(void);
void Setup_Signal_Level(void);
void Record_APT_Signal(void);
int File_Fill_Buffer(void);
void Read_Device_Buffer(short *buffer, int buff_size);
void Fill_Samples_Buffer(void);
void Synchronize_Buffer(void);
void Carrier_Amplitude(short sample, int *amplitude);
/* image.c */
void NOAA_Image_Line(unsigned char *image_line, int *max_gray_val);
void Meteor_Image_Line(unsigned char *image_line, int *max_gray_val);
void Dsp_Process_Image(void);
void File_Process_Image(void);
void Normalize(unsigned char *image_buffer, int image_size);
void Rotate(unsigned char *image_buffer, int image_size);
void Colorize(unsigned char *image_buffer_a, unsigned char *image_buffer_b, int image_size, unsigned char *colorized_buffer, int map_type);
/* main.c */
int main(int argc, char *argv[]);
/* rtlsdr.c */
void Rtl_Cb(unsigned char *buf, uint32_t len, void *ctx);
void *RtlSdr_Read_Async(void *pid);
void Rtlsdr_Set_Center_Freq(uint32_t center_freq);
void Set_Tuner_Gain_Mode(int mode);
void Rtlsdr_Set_Agc_Mode(int mode);
void Rtlsdr_Init(void);
void Read_RTL_Buffer(short *buffer, int buff_size);
void Init_Butterworth_2Pole_LPF(void);
void Butterworth_2Pole_LPF(char *buf, int buf_size);
void Close_RTL_Device(void);
/* shared.c */
/* sound.c */
void Open_PCM(snd_pcm_t **handle, snd_pcm_stream_t stream);
void Open_Capture(void);
void Open_Mixer(void);
void Set_Capture_Level(int level);
void Close_Capture(void);
void Read_SND_Buffer(short *buffer, int buff_size);
void Xrun_Recovery(snd_pcm_t *handle, int error);
/* utils.c */
void Load_Config(void);
void Load_Colormap(FILE *wxaptrc, color_map_t *map);
int Load_Line(char *buff, FILE *pfile, char *messg);
void Start_Stop_Time(char *optarg, unsigned int *start_hrs, unsigned int *start_min, unsigned int *sleep_sec, unsigned int *duration);
void File_Name(char *file_name);
char *Fname(char *fpath);
void Usage(void);
void mem_alloc(void **ptr, size_t req);
void mem_realloc(void **ptr, size_t req);
void free_ptr(void **ptr);
void Open_File(FILE **fp, char *fname, const char *mode);
void File_Image(FILE *fp, const char *type, int width, int height, int max_val, unsigned char *buffer);
int isFlagSet(int flag);
int isFlagClear(int flag);
void SetFlag(int flag);
void ClearFlag(int flag);
void ToggleFlag(int flag);
void Strlcpy(char *dest, const char *src, size_t n);
void Strlcat(char *dest, const char *src, size_t n);
void Cleanup(void);

/*-------------------------------------------------------------------*/

#endif /* End of common.h */

