/*  image.c
 *
 *  Image processing functions of wxapt application
 */

/*
 *  wxapt: An application to decode APT signals from
 *  weather satellites and produce an image of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "image.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/*  NOAA_Image_Line()
 *
 *  Fills one line of one channel's image by
 *  reading samples from buffer and processing
 */

  void
NOAA_Image_Line( unsigned char *image_line, int *max_gray_val )
{
  /* An amplitude sample of the 2.4kHz sub-carrier */
  short sample_a;

  int
	pixel_val, /* Un-normalized value of pixel */
	line_idx,  /* Index to image line buffer   */
	idx, tmp;  /* Index for loops etc and temp */


  /* Process samples to make one line of image */
  for( line_idx = 0; line_idx < NOAA_IMAGE_WIDTH; line_idx++ )
  {
	pixel_val = 0;

	/*** Summate NOAA_PIXEL_SAMPLES carrier
	 * samples for each pixel value ***/
	for( idx = 0; idx < NOAA_PIXEL_SAMPLES; idx++ )
	{
	  /* Read a sample of the 2.4kHz sub_carrier */
	  sample_a = gbl_line_buffer[gbl_line_idx++];

	  /* This is the instantaneous magnitude of the 2.4 Khz */
	  /* subcarrier, calculated as the scalar magnitude of */
	  /* two consecutive signal samples at 90 deg phase angle */
	  Carrier_Amplitude( sample_a, &tmp );
	  pixel_val += tmp;

	} /* End of for( idx = 0; idx < NOAA_PIXEL_SAMPLES; idx++ ) */

	/* Normalize pixel values to below 255 */
	tmp = (pixel_val * NOAA_SCALE_FACTOR) / MAX_SUBCARRIER_AMPL;
	if( tmp > MAX_PIXEL_VAL ) tmp = MAX_PIXEL_VAL;
	image_line[line_idx] = (unsigned char)tmp;

	/* Record max gray value of line */
	if( image_line[line_idx] > *max_gray_val )
	  *max_gray_val = image_line[line_idx];

  } /* End of for( line_idx = 0; line_idx < NOAA_IMAGE_WIDTH; line_idx++ ) */

} /* End of NOAA_Image_Line() */

/*------------------------------------------------------------------------*/

/*  Meteor_Image_Line()
 *
 *  Fills one line of a Meteor APT image by
 *  reading samples from buffer and processing
 */

  void
Meteor_Image_Line( unsigned char *image_line, int *max_gray_val )
{
  /* An amplitude sample of the 2.4kHz sub-carrier */
  short sample_a;

  int
	pixel_val, /* Un-normalized value of pixel */
	line_idx,  /* Index to image line buffer   */
	idx, tmp;  /* Index for loops etc and temp */


  /* Process samples to make one line of image */
  for( line_idx = 0; line_idx < METEOR_IMAGE_WIDTH; line_idx++ )
  {
	pixel_val = 0;

	/*** Summate METEOR_NSAMPLES carrier samples for each pixel value ***/
	for( idx = 0; idx < METEOR_PIXEL_SAMPLES; idx++ )
	{
	  /* Read a sample of the 2.4kHz sub_carrier*/
	  sample_a = gbl_line_buffer[gbl_line_idx++];

	  /* This is the instantaneous magnitude of the 2.4 Khz  */
	  /* subcarrier, calculated as the scalar magnitude of   */
	  /* two consecutive signal samples at 90 deg phase angle */
	  Carrier_Amplitude( sample_a, &tmp );
	  pixel_val += tmp;

	} /* End of for( idx = 0; idx < METEOR_PIXEL_SAMPLES; idx++ ) */

	/* Normalize pixel values to below 255 */
	tmp = (pixel_val * METEOR_SCALE_FACTOR) / MAX_SUBCARRIER_AMPL;
	if( tmp > MAX_PIXEL_VAL ) tmp = MAX_PIXEL_VAL;
	image_line[line_idx] = (unsigned char)tmp;

	/* Record max gray value of line */
	if( image_line[line_idx] > *max_gray_val )
	  *max_gray_val = image_line[line_idx];

  } /* End of for( line_idx = 0; line_idx < image_width; line_idx++ ) */

} /* End of Meteor_Image_Line() */

/*------------------------------------------------------------------------*/

/*  Dsp_Process_Image()
 *
 *  Processes APT images direct from DSP signal samples.
 */

  void
Dsp_Process_Image( void )
{
  /* Number of APT image lines processed */
  int line_cnt;

  int
	idx,
	carrier_val, /* Instantaneous value of 2.4Khz sub-carrier */
	carrier_max; /* Maximum value of 2.4KHz sub-carrier       */


  /* Process APT images according to satellite type */
  carrier_max = 0;
  switch( gbl_sat_type )
  {
	case NOAA_15: case NOAA_18: case NOAA_19: /* Decode NOAA APT images */
	  {
		/* Image file names of NOAA ch A and B images */
		char
		  A_image_file[MAX_FILE_NAME+20],
		  B_image_file[MAX_FILE_NAME+20];

		  /* Image file names of NOAA ch A color images */
		  char  A_color_file[MAX_FILE_NAME+20];

		  int
			A_max_gray_val,  /* Maximum gray value in channel A PGM image */
			B_max_gray_val,  /* Maximum gray value in channel B PGM image */
			A_image_height,  /* Height in lines of channel A PGM image    */
			B_image_height,  /* Height in lines of channel B PGM image    */
			A_image_size,    /* Current size in bytes of A image buffer   */
			B_image_size;    /* Current size in bytes of B image buffer   */

		  /* Buffers for the output images */
		  unsigned char *A_image_buffer = NULL;
		  unsigned char *B_image_buffer = NULL;
		  unsigned char *A_color_buffer = NULL;

		  FILE
			*A_image_fp, /* Channel A image file */
			*B_image_fp; /* Channel B image file */

		  /* Process file name */
		  File_Name( gbl_image_file );

		  /* Prepare image file names */
		  size_t s = sizeof( A_image_file );
		  Strlcpy( A_image_file, gbl_image_file, s );
		  Strlcat( A_image_file, "-", s );
		  Strlcat( A_image_file, gbl_sat_names[gbl_sat_type], s );
		  Strlcat( A_image_file, "-A.pgm", s );

		  s = sizeof( B_image_file );
		  Strlcpy( B_image_file, gbl_image_file, s );
		  Strlcat( B_image_file, "-", s );
		  Strlcat( B_image_file, gbl_sat_names[gbl_sat_type], s );
		  Strlcat( B_image_file, "-B.pgm", s );

		  /* Prepare color image file names */
		  s = sizeof( A_color_file );
		  Strlcpy( A_color_file, gbl_image_file, s );
		  Strlcat( A_color_file, "-", s );
		  Strlcat( A_color_file, gbl_sat_names[gbl_sat_type], s );
		  Strlcat( A_color_file, "-A+B-color.ppm", s );

		  /* Clear variables before decoding samples buffer */
		  A_image_height = B_image_height = 0;
		  A_image_size   = B_image_size   = 0;
		  A_max_gray_val = B_max_gray_val = 0;
		  A_image_buffer = B_image_buffer = NULL;
		  line_cnt       = 0;

		  puts( "Decoding Images directly from APT signal..." );

		  /*** Process samples to make an image ***/
		  /* Continue if line count less than limit or action enabled */
		  while( isFlagSet(ACTION_PROCESS_DSP) &&
			  (++line_cnt / 2 < gbl_duration) )
		  {
			/* Fill APT samples buffer */
			Fill_Samples_Buffer();

			/* Bypass sync and space/markers of ch A */
			gbl_line_idx += NOAA_SYNC_CHA_SPACE;

			/* Re-allocate memory to image buffers, abort on error */
			mem_realloc( (void **)&A_image_buffer,
				(size_t)(A_image_size + NOAA_IMAGE_WIDTH) );

			mem_realloc( (void **)&B_image_buffer,
				(size_t)(B_image_size + NOAA_IMAGE_WIDTH) );

			/* Decode one image line */
			NOAA_Image_Line( A_image_buffer + A_image_size, &A_max_gray_val );
			A_image_height++;
			A_image_size = NOAA_IMAGE_WIDTH * A_image_height;

			/* Bypass ch A gray scale/ch B sync/ch B space/markers. */
			/* This is done by searching this zone for carrier max. */
			for( idx = 0; idx < NOAA_CHA_CHB_SPACE; idx++ )
			{
			  Carrier_Amplitude( gbl_line_buffer[gbl_line_idx], &carrier_val );
			  gbl_line_idx++;

			  /* Find maximum carrier level */
			  if( carrier_val > carrier_max )
				carrier_max = carrier_val;
			} /* for( idx = 0; idx < NOAA_CHA_CHB_SPACE; idx++ ) */

			/* Decode an image line */
			NOAA_Image_Line( B_image_buffer + B_image_size, &B_max_gray_val );
			B_image_height++;
			B_image_size = NOAA_IMAGE_WIDTH * B_image_height;

		  } /* while( isFlagSet(ACTION_PROCESS_DSP) && */

		  /* Rotate images 180 degrees if enabled */
		  if( isFlagSet(IMAGE_ROTATE) )
		  {
			Rotate( A_image_buffer, A_image_size );
			Rotate( B_image_buffer, B_image_size );
		  }

		  /* Carry out histogram normalization if enabled */
		  if( isFlagSet(IMAGE_NORMALIZE) )
		  {
			Normalize( A_image_buffer, A_image_size );
			Normalize( B_image_buffer, B_image_size );
			A_max_gray_val = B_max_gray_val = 255;
		  }

		  /* Carry out pseudo-colorization if enabled */
		  if( isFlagSet(IMAGE_COLORIZE) )
		  {
			/* Re-allocate memory to image buffers, abort on error */
			A_color_buffer = NULL;
			mem_realloc( (void **)&A_color_buffer, (size_t)(3 * A_image_size) );

			Colorize( A_image_buffer, B_image_buffer,
				A_image_size, A_color_buffer, NOAA_A_COLORMAP );

			/* Open Channel A colorized image file */
			Open_File( &A_image_fp, A_color_file, "w+" );

			/* Write Ch-A colorized buffer to PPM file */
			File_Image( A_image_fp, "P6", NOAA_IMAGE_WIDTH,
				A_image_height, A_max_gray_val,  A_color_buffer );

			fclose( A_image_fp );
		  } /* if( isFlagSet(IMAGE_COLORIZE) ) */

		  /*** Processing finished, complete image files ***/

		  /* Open Channel A image file */
		  Open_File( &A_image_fp, A_image_file, "w+");

		  /* Open Channel B image file */
		  Open_File( &B_image_fp, B_image_file, "w+" );

		  /* Write Ch-A greyscale buffer to PGM file */
		  File_Image( A_image_fp, "P5", NOAA_IMAGE_WIDTH,
			  A_image_height, A_max_gray_val, A_image_buffer );

		  /* Write Ch-B greyscale buffer to PGM file */
		  File_Image( B_image_fp, "P5", NOAA_IMAGE_WIDTH,
			  B_image_height, B_max_gray_val, B_image_buffer );

		  free( A_image_buffer );
		  free( B_image_buffer );
		  fclose( A_image_fp );
		  fclose( B_image_fp );
		  if( isFlagSet(IMAGE_COLORIZE) && A_color_buffer )
			free( A_color_buffer );

		  puts( "Finished Decoding Images directly from APT signal" );

	  } /* End of case NOAA: */
	  break;

	case METEOR: /* Decode Meteor APT images */
	  {
		/* Image file name of Meteor image */
		char Met_image_file[MAX_FILE_NAME+20];
		char Met_color_file[MAX_FILE_NAME+20];

		int
		  max_gray_val,     /* Maximum gray value in Meteor APT image */
		  Met_image_height, /* Height in lines of Meteor APT image    */
		  Met_image_size;   /* Current size in bytes of Meteor image  */

		/* Buffer for the output image */
		unsigned char *Met_image_buffer = NULL;
		unsigned char *Met_color_buffer = NULL;

		/* Meteor APT image file pointer */
		FILE *Met_image_fp;


		/* Process file name */
		File_Name( gbl_image_file );

		/* Prepare image file name */
		size_t s = sizeof( Met_image_file );
		Strlcpy( Met_image_file, gbl_image_file, s );
		Strlcat( Met_image_file, gbl_sat_names[gbl_sat_type], s );
		Strlcat( Met_image_file, ".pgm", s );

		s = sizeof( Met_color_file );
		Strlcpy( Met_color_file, gbl_image_file, s );
		Strlcat( Met_color_file, gbl_sat_names[gbl_sat_type], s );
		Strlcat( Met_color_file, "-color.ppm", s );

		/* Clear variables before decoding sample buffer */
		Met_image_height = 0;
		max_gray_val     = 0;
		Met_image_size   = 0;
		Met_image_buffer = NULL;
		line_cnt         = 0;

		puts( "Decoding Images directly from APT signal..." );

		/*** Process samples to make an image ***/
		/*** Continue if line count less than limit or action enabled ***/
		while( isFlagSet(ACTION_PROCESS_DSP) &&
			(++line_cnt / 2 < gbl_duration) )
		{
		  /* Fill APT samples buffer */
		  Fill_Samples_Buffer();

		  /* Bypass sync and space/markers of image */
		  /* This is done by searching this zone for carrier max. */
		  for( idx = 0; idx < METEOR_SYNC_IMAGE_SPACE; idx++ )
		  {
			Carrier_Amplitude( gbl_line_buffer[gbl_line_idx], &carrier_val );
			gbl_line_idx++;

			/* Find maximum carrier level */
			if( carrier_val > carrier_max )
			  carrier_max = carrier_val;
		  } /* for( idx = 0; idx < METEOR_SYNC_IMAGE_SPACE; idx++ ) */

		  /* Re-allocate memory to image buffer, abort on error */
		  mem_realloc( (void *)&Met_image_buffer,
			  (size_t)(Met_image_size + METEOR_IMAGE_WIDTH) );

		  /* Decode an image line */
		  Meteor_Image_Line( Met_image_buffer + Met_image_size, &max_gray_val );
		  Met_image_height++;
		  Met_image_size = METEOR_IMAGE_WIDTH * Met_image_height;

		} /* while( (++line_cnt/2 < gbl_duration) && */

		/* Rotate images 180 degrees if requested */
		if( isFlagSet(IMAGE_ROTATE) )
		  Rotate( Met_image_buffer, Met_image_size );

		/* Carry out histogram normalization if not suppressed */
		if( isFlagSet(IMAGE_NORMALIZE) )
		{
		  Normalize( Met_image_buffer, Met_image_size );
		  max_gray_val = 255;
		}

		/* Carry out pseudo-colorization if enabled */
		if( isFlagSet(IMAGE_COLORIZE) )
		{
		  /* Re-allocate memory to image buffers, abort on error */
		  Met_color_buffer = NULL;
		  mem_realloc( (void *)&Met_color_buffer, (size_t)(3 * Met_image_size) );

		  /* The second argument is a dummy */
		  Colorize( Met_image_buffer, Met_image_buffer,
			  Met_image_size, Met_color_buffer, METEOR_COLORMAP );

		  /* Open Meteor colorized image file */
		  Open_File( &Met_image_fp, Met_color_file, "w+" );

		  /* Write Meteor colorized buffer to PPM file */
		  File_Image( Met_image_fp, "P6", METEOR_IMAGE_WIDTH,
			  Met_image_height, max_gray_val,  Met_color_buffer );

		  fclose( Met_image_fp );

		} /* if( isFlagSet(IMAGE_COLORIZE) ) */

		/*** Processing finished, complete image files ***/

		/* Open Meteor image file */
		Open_File( &Met_image_fp, Met_image_file, "w+" );

		/* Write Meteor image buffer to file */
		File_Image( Met_image_fp, "P5",
			METEOR_IMAGE_WIDTH, Met_image_height,
			max_gray_val, Met_image_buffer );

		free( Met_image_buffer );
		if( isFlagSet(IMAGE_COLORIZE) && Met_color_buffer )
		  free( Met_color_buffer );
		fclose( Met_image_fp );

		puts( "Finished Decoding Images directly from APT signal" );

	  } /* End of case METEOR: */

  } /* End of switch( gbl_sat_type ) */

  /* Abort on user interrupt */
  if( isFlagClear(ACTION_PROCESS_DSP) )
  {
	puts( "\nAborting due to User Interrupt" );
	exit( 0 );
  }

} /* End of Dsp_Process_Image() */

/*------------------------------------------------------------------------*/

/*  File_Process_Image()
 *
 *  Processes APT images from signal samples file.
 */

  void
File_Process_Image( void )
{
  /* File suffix to be removed from file name */
  char suffix[20];

  /* Position of suffix in file name */
  char *suffix_pos;


  /* Abort on user interrupt */
  if( isFlagClear(ACTION_PROCESS_FILE) )
  {
	puts( "\nAborting due to User Interrupt" );
	exit( 0 );
  }

  puts( "Decoding Images from File..." );

  /* Read satellite type from file, abort on error */
  Open_File( &gbl_samples_fp, gbl_samples_file, "r" );
  if( fread(&gbl_sat_type, 4, 1, gbl_samples_fp) < 1 )
  {
	perror( "wxapt: fread()" );
	fprintf( stderr, "Failed to read from samples file\n" );
	exit( -1 );
  }

  if( (gbl_sat_type != NOAA_15) &&
	  (gbl_sat_type != NOAA_18) &&
	  (gbl_sat_type != NOAA_19) &&
	  (gbl_sat_type != METEOR) )
  {
	fprintf( stderr, "Unknown Satellite Type\n" );
	exit( -1 );
  }

  /* Read sync position reference */
  gbl_sync_ref = gbl_sync_refs[ gbl_sat_type ];

  /* Create the suffix to be removed from sample file name. */
  /* This is to remove the suffix <sat_name>.bin which is  */
  /* added automatically by wxapt to samples file name.   */
  suffix[0] = '-';
  suffix[1] = '\0';
  size_t s = sizeof(suffix);
  Strlcat( suffix, gbl_sat_names[gbl_sat_type], s );
  Strlcat( suffix, ".bin", s );

  /* Find position of suffix in file name */
  suffix_pos = strstr( gbl_samples_file, suffix );

  /* Cut out suffix if it exists */
  if( suffix_pos != NULL )
	suffix_pos[0] = '\0';

  /* Process APT images according to satellite type */
  switch( gbl_sat_type )
  {
	case NOAA_15: case NOAA_18: case NOAA_19: /* Decode NOAA APT images */
	  {
		/* Image file names of NOAA ch A and B images */
		char
		  A_image_file[MAX_FILE_NAME+20],
		  B_image_file[MAX_FILE_NAME+20];

		  /* Image file names of NOAA ch A color images */
		  char A_color_file[MAX_FILE_NAME+20];

		  int
			line_cnt,        /* Number of APT image lines processed       */
			A_max_gray_val,  /* Maximum gray value in channel A PGM image */
			B_max_gray_val,  /* Maximum gray value in channel B PGM image */
			A_image_height,  /* Height in lines of channel A PGM image    */
			B_image_height,  /* Height in lines of channel B PGM image    */
			A_image_size,    /* Current size in bytes of A image buffer   */
			B_image_size;    /* Current size in bytes of B image buffer   */

		  /* Buffers for the output images */
		  unsigned char *A_image_buffer = NULL;
		  unsigned char *B_image_buffer = NULL;
		  unsigned char *A_color_buffer = NULL;

		  FILE
			*A_image_fp = NULL,   /* Channel A image file  */
			*B_image_fp = NULL;   /* Channel B image file  */

		  /* Process file name for images */
		  s = sizeof( gbl_image_file );
		  Strlcpy( gbl_image_file, rc_data.wxapt_dir, s );
		  Strlcat( gbl_image_file, "images/", s );
		  Strlcat( gbl_image_file, Fname(gbl_samples_file), s );

		  /* Prepare image file names */
		  s = sizeof( A_image_file );
		  Strlcpy( A_image_file, gbl_image_file, s );
		  Strlcat( A_image_file, "-", s );
		  Strlcat( A_image_file, gbl_sat_names[gbl_sat_type], s );
		  Strlcat( A_image_file, "-A.pgm", s );
		  s = sizeof( B_image_file );
		  Strlcpy( B_image_file, gbl_image_file, s );
		  Strlcat( B_image_file, "-", s );
		  Strlcat( B_image_file, gbl_sat_names[gbl_sat_type], s );
		  Strlcat( B_image_file, "-B.pgm", s );

		  /* Prepare color image file names */
		  s = sizeof( A_color_file );
		  Strlcpy( A_color_file, gbl_image_file, s );
		  Strlcat( A_color_file, "-", s );
		  Strlcat( A_color_file, gbl_sat_names[gbl_sat_type], s );
		  Strlcat( A_color_file, "-A+B-color.ppm", s );

		  /* Clear variables before decoding samples buffer */
		  A_image_height = B_image_height = 0;
		  A_image_size   = B_image_size   = 0;
		  A_max_gray_val = B_max_gray_val = 0;
		  A_image_buffer = B_image_buffer = NULL;
		  line_cnt       = 0;

		  /* Process samples from file to make an image */
		  while( File_Fill_Buffer() )
		  {
			/* Bypass A sync and space/markers of ch A. */
			gbl_line_idx += NOAA_SYNC_CHA_SPACE;

			/* Re-allocate memory to image buffers, abort on error */
			mem_realloc( (void *)&A_image_buffer,
				(size_t)(A_image_size + NOAA_IMAGE_WIDTH) );
			mem_realloc( (void *)&B_image_buffer,
				(size_t)(B_image_size + NOAA_IMAGE_WIDTH) );

			/* Decode an image line */
			NOAA_Image_Line( A_image_buffer + A_image_size, &A_max_gray_val );
			A_image_height++;
			A_image_size = NOAA_IMAGE_WIDTH * A_image_height;

			/* Bypass ch A gray scale/ch B sync/ch B space/markers */
			gbl_line_idx += NOAA_CHA_CHB_SPACE;

			/* Decode an image line */
			NOAA_Image_Line( B_image_buffer + B_image_size, &B_max_gray_val );
			B_image_height++;
			B_image_size = NOAA_IMAGE_WIDTH * B_image_height;

			line_cnt++;

		  } /* End of while( File_Fill_Buffer() ) */

		  /* Rotate images 180 degrees if requested */
		  if( isFlagSet(IMAGE_ROTATE ) )
		  {
			Rotate( A_image_buffer, A_image_size );
			Rotate( B_image_buffer, B_image_size );
		  }

		  /* Carry out histogram normalization if not suppressed */
		  if( isFlagSet(IMAGE_NORMALIZE) )
		  {
			Normalize( A_image_buffer, A_image_size );
			Normalize( B_image_buffer, B_image_size );
			A_max_gray_val = B_max_gray_val = 255;
		  }

		  /* Carry out pseudo-colorization if enabled */
		  if( isFlagSet(IMAGE_COLORIZE) )
		  {
			/* Re-allocate memory to image buffers, abort on error */
			mem_realloc( (void *)&A_color_buffer, (size_t)(3 * A_image_size) );

			Colorize( A_image_buffer, B_image_buffer,
				A_image_size, A_color_buffer, NOAA_A_COLORMAP );

			/* Open Channel A colorized image file */
			Open_File( &A_image_fp, A_color_file, "w+" );

			/* Write Ch-A colorized buffer to PPM file */
			File_Image( A_image_fp, "P6", NOAA_IMAGE_WIDTH,
				A_image_height, A_max_gray_val,  A_color_buffer );

			fclose( A_image_fp );

		  } /* if( isFlagSet(IMAGE_COLORIZE) ) */

		  /*** Processing finished, complete image files ***/

		  /* Resize window as required */
		  gbl_duration = line_cnt / 2;

		  /* Open Channel A image file */
		  Open_File( &A_image_fp, A_image_file, "w+" );

		  /* Open Channel B image file */
		  Open_File( &B_image_fp, B_image_file, "w+" );

		  /* Write Ch-A greyscale buffer to PGM file */
		  File_Image( A_image_fp, "P5", NOAA_IMAGE_WIDTH,
			  A_image_height, A_max_gray_val,  A_image_buffer );

		  /* Write Ch-B greyscale buffer to PGM file */
		  File_Image( B_image_fp, "P5", NOAA_IMAGE_WIDTH,
			  B_image_height, B_max_gray_val,  B_image_buffer );

		  free( A_image_buffer );
		  free( B_image_buffer );
		  fclose( A_image_fp );
		  fclose( B_image_fp );
		  if( isFlagSet(IMAGE_COLORIZE) )
			free( A_color_buffer );

		  puts( "Finished Decoding Images from File" );

	  } /* End of case NOAA : */
	  break;

	case METEOR: /* Decode METEOR APT images */
	  {
		/* Meteor image file name */
		char Met_image_file[MAX_FILE_NAME+20];

		int
		  line_cnt,     /* Number of APT image lines processed       */
		  max_gray_val, /* Maximum gray value of Meteor APT image    */
		  Met_image_height, /* Height in lines of Meteor APT image   */
		  Met_image_size;   /* Current size in bytes of Meteor image */

		/* Buffer for the output image */
		unsigned char *Met_image_buffer = NULL;
		unsigned char *Met_color_buffer = NULL;

		/* Meteor image file pointer */
		FILE *Met_image_fp;

		/* Process file name for image */
		s = sizeof( gbl_image_file );
		Strlcpy( gbl_image_file, rc_data.wxapt_dir, s );
		Strlcat( gbl_image_file, "images/", s );
		Strlcat( gbl_image_file, Fname(gbl_samples_file), s );

		/* Prepare image file name */
		s = sizeof( Met_image_file );
		Strlcpy( Met_image_file, gbl_image_file, s );
		Strlcat( Met_image_file, gbl_sat_names[gbl_sat_type], s );
		Strlcat( Met_image_file, "-.pgm", s );

		/* Clear variables before decoding sample buffer */
		Met_image_height = 0;
		max_gray_val     = 0;
		Met_image_size   = 0;
		Met_image_buffer = NULL;
		line_cnt         = 0;

		/* Process samples from file to make an image */
		while( File_Fill_Buffer() )
		{
		  /* Bypass sync and space/markers of image */
		  gbl_line_idx += METEOR_SYNC_IMAGE_SPACE;

		  /* Re-allocate memory to image buffer, abort on error */
		  mem_realloc( (void *)&Met_image_buffer,
			  (size_t)(Met_image_size + METEOR_IMAGE_WIDTH) );

		  /* Decode an image line */
		  Meteor_Image_Line( Met_image_buffer + Met_image_size, &max_gray_val );
		  Met_image_height++;
		  Met_image_size = METEOR_IMAGE_WIDTH * Met_image_height;

		  line_cnt++;

		} /* End of while( File_Fill_Buffer() ) */

		/* Rotate images 180 degrees if requested */
		if( isFlagSet(IMAGE_ROTATE) )
		  Rotate( Met_image_buffer, Met_image_size );

		/* Carry out histogram normalization if not suppressed */
		if( isFlagSet(IMAGE_NORMALIZE) )
		{
		  Normalize( Met_image_buffer, Met_image_size );
		  max_gray_val = 255;
		}

		/* Carry out pseudo-colorization if enabled */
		if( isFlagSet(IMAGE_COLORIZE) )
		{
		  /* Colorized image file name */
		  char Met_color_file[MAX_FILE_NAME+20];

		  /* Re-allocate memory to image buffers, abort on error */
		  Met_color_buffer = NULL;
		  mem_realloc( (void *)&Met_color_buffer, (size_t)(3 * Met_image_size) );

		  /* The second argument is a dummy */
		  Colorize( Met_image_buffer, Met_image_buffer,
			  Met_image_size, Met_color_buffer, METEOR_COLORMAP );

		  /* Prepare image file names */
		  s = sizeof( Met_color_file );
		  Strlcpy( Met_color_file, gbl_image_file, s );
		  Strlcat( Met_color_file, "-Meteor.ppm", s );

		  /* Open Channel A colorized image file */
		  Open_File( &Met_image_fp, Met_color_file, "w+" );

		  /* Write Ch-A colorized buffer to PPM file */
		  File_Image( Met_image_fp, "P6", METEOR_IMAGE_WIDTH,
			  Met_image_height, max_gray_val,  Met_color_buffer );

		  fclose( Met_image_fp );

		} /* if( isFlagSet(IMAGE_COLORIZE) ) */

		/*** Processing finished, complete image files ***/

		/* Resize window as required */
		gbl_duration = line_cnt / 2;

		/* Open Meteor APT image file */
		Open_File( &Met_image_fp, Met_image_file, "w+" );

		/* Write Meteor image buffer to file */
		File_Image(
			Met_image_fp, "P5",
			METEOR_IMAGE_WIDTH, Met_image_height,
			max_gray_val, Met_image_buffer );

		free( Met_image_buffer );
		if( isFlagSet(IMAGE_COLORIZE) )
		  free( Met_color_buffer );
		fclose( Met_image_fp );

		puts( "Finished Decoding Images from File" );

	  } /* End of case METEOR : */

  } /* End of switch( gbl_sat_type ) */

  /* Abort on user interrupt */
  if( isFlagClear(ACTION_PROCESS_FILE) )
  {
	puts( "\nAborting due to User Interrupt" );
	exit( 0 );
  }

} /* End of File_Process_Image() */

/*------------------------------------------------------------------------*/

/*  Normalize()
 *
 *  Does histogram (linear) normalization of a pgm (P5) image file
 */

  void
Normalize( unsigned char *image_buffer, int image_size )
{
  int
	hist[256],  /* Intensity histogram of pgm image file  */
	blk_cutoff, /* Count of pixels for black cutoff value */
	wht_cutoff, /* Count of pixels for white cutoff value */
	pixel_val,  /* Used for calculating normalized pixels */
	pixel_cnt,  /* Total pixels counter for cut-off point */
	idx;        /* Index for loops etc */

  int
	black_val,  /* Black cut-off pixel intensity value */
	white_val,  /* White cut-off pixel intensity value */
	val_range;  /* Range of intensity values in image  */

  if( image_size <= 0 ) return;

  /* Clear histogram */
  for( idx = 0; idx < 256; idx++ )
	hist[ idx ] = 0;

  /* Build image intensity histogram */
  for( idx = 0; idx < image_size; idx++ )
	++hist[ image_buffer[idx] ];

  /* Determine black/white cut-off counts */
  blk_cutoff = (image_size * BLACK_CUT_OFF) / 100;
  wht_cutoff = (image_size * WHITE_CUT_OFF) / 100;

  /* Find black cut-off intensity value */
  pixel_cnt = 0;
  for( black_val = 0; black_val < 256; black_val++ )
  {
	pixel_cnt += hist[ black_val ];
	if( pixel_cnt > blk_cutoff ) break;
  }

  /* Find white cut-off intensity value */
  pixel_cnt = 0;
  for( white_val = 255; white_val >= 0; white_val-- )
  {
	pixel_cnt += hist[ white_val ];
	if( pixel_cnt > wht_cutoff ) break;
  }

  /* Rescale pixels in image for full intensity range */
  val_range = white_val - black_val;
  if( (int) val_range <= 0 ) return;

  /* Perform histogram normalization on images */
  for( pixel_cnt = 0; pixel_cnt < image_size; pixel_cnt++ )
  {
	pixel_val = image_buffer[ pixel_cnt ];
	pixel_val = ( (pixel_val - black_val) * 255 ) / val_range;

	pixel_val = ( pixel_val < 0 ? 0 : pixel_val );
	pixel_val = ( pixel_val > 255 ? 255 : pixel_val );
	image_buffer[ pixel_cnt ] = (unsigned char)pixel_val;
  }

} /* End of Normalize() */

/*------------------------------------------------------------------------*/

/*  Rotate()
 *
 *  Rotates a pgm (P5) image by 180 degrees
 */

  void
Rotate( unsigned char *image_buffer, int image_size )
{
  int idx; /* Index for loops etc */

  unsigned char
	*idx_temp,  /* Buffer location to be saved in temp    */
	*idx_swap;  /* Buffer location to be swaped with temp */

  /* Holds a pixel value temporarily */
  unsigned char temp;

  if( image_size <= 0 ) return;

  for( idx = 0; idx < image_size / 2; idx++ )
  {
	idx_temp = image_buffer + idx;
	idx_swap = image_buffer - 1 + image_size - idx;
	temp = *( idx_temp );
	*( idx_temp ) = *( idx_swap );
	*( idx_swap ) = temp;
  }

} /* End of Rotate() */

/*------------------------------------------------------------------------*/

/*  Colorize()
 *
 *  Pseudo-colorizes a greyscale image using tables
 */

  void
Colorize(
	unsigned char *image_buffer_a,
	unsigned char *image_buffer_b,
	int image_size,
	unsigned char *colorized_buffer,
	int map_type )
{
  int range, idg, idc, delta, slope;
  color_map_t *map;

  /* Select color map */
  switch( map_type )
  {
	case NOAA_A_COLORMAP:
	  map = &rc_data.noaa_A_map;
	  break;

	case METEOR_COLORMAP:
	  map = &rc_data.meteor_map;
	  break;

	default: return;
  }

  /*** Pseudo-colorize greyscale image ***/
  idc = 0;
  for( idg = 0; idg < image_size; idg++ )
  {
	/* Enter clouds image */
	if( image_buffer_b[idg] >= map->white_thld )
	{
	  colorized_buffer[idc++] = image_buffer_b[idg];
	  colorized_buffer[idc++] = image_buffer_b[idg];
	  colorized_buffer[idc++] = image_buffer_b[idg];
	}
	else /* Colorize */
	{
	  /* Find gray scale range of image pixel */
	  for( range = 0; range < map->num_ranges; range++ )
		if( (image_buffer_a[idg] >= map->gray_from[range]) &&
			(image_buffer_a[idg] <= map->gray_to[range]) )
		  break;
	  if( range == map->num_ranges ) return;

	  /* Colorize image */
	  delta = map->gray_to[range] - map->gray_from[range];
	  if( delta <= 0 ) return;

	  slope = ( map->red_to[range] - map->red_from[range] );
	  colorized_buffer[idc++] = (unsigned char)( map->red_from[range] +
		  (slope*(image_buffer_a[idg]-map->gray_from[range]))/delta );

	  slope = ( map->green_to[range] - map->green_from[range] );
	  colorized_buffer[idc++] = (unsigned char)(map->green_from[range] +
		  (slope*(image_buffer_a[idg]-map->gray_from[range]))/delta );

	  slope = ( map->blue_to[range] - map->blue_from[range] );
	  colorized_buffer[idc++] = (unsigned char)(map->blue_from[range] +
		  (slope*(image_buffer_a[idg]-map->gray_from[range]))/delta );

	} /* if( image_buffer_b[idg] >= map->white_thld ) */

  } /* for( idg = 0; idg < image_size; idg++ ) */

} /* End of Colorize() */

/*------------------------------------------------------------------------*/

