/*  main.c
 *
 *  Main body of wxapt application
 */

/*
 *  wxapt: An application to decode APT signals from
 *  weather satellites and produce image(s) of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "main.h"
#include "shared.h"


/* Signal handler */
static void sig_handler( int signal );

/*------------------------------------------------------------------------*/

  int
main (int argc, char *argv[])
{
  /* Command line option returned by getopt() */
  int option;

  /* New and old actions for sigaction() */
  struct sigaction sa_new, sa_old;

  /* Time limit of recording or image processing */
  unsigned int
	duration  = 0,
	start_hrs = 0,
	start_min = 0,
	sleep_sec = 0;

  /* Initialize new actions */
  sa_new.sa_handler = sig_handler;
  sigemptyset( &sa_new.sa_mask );
  sa_new.sa_flags = 0;

  /* Register function to handle signals */
  sigaction( SIGINT,  &sa_new, &sa_old );
  sigaction( SIGSEGV, &sa_new, 0 );
  sigaction( SIGFPE,  &sa_new, 0 );
  sigaction( SIGTERM, &sa_new, 0 );
  sigaction( SIGABRT, &sa_new, 0 );
  sigaction( SIGCONT, &sa_new, 0 );


  /* Load config file first. It is
   * overruled by any options below */
  Load_Config();

  /* Process command line options. Defaults settings below. */
  rc_data.rtlsdr_center_freq = 137100000;
  gbl_sat_type = NOAA_19;
  SetFlag( IMAGE_NORMALIZE | ACTION_PROCESS_DSP );
  gbl_image_file[0]   = '\0';
  gbl_samples_file[0] = '\0';

  /* Process command line options */
  while( (option = getopt(argc, argv, "a:b:f:i:p:r::R::s:t:chlnov") ) != -1 )
	switch( option )
	{
	  int freq, bw;
	  case 'a': /* Specify satellite APT type */
	  if( optarg == NULL ) break;
	  gbl_sat_type = atoi( optarg );
	  if( (gbl_sat_type < NOAA_15) ||
		  (gbl_sat_type > METEOR) )
	  {
		fprintf( stderr, "APT Satellite type not valid\n" );
		exit( -1 );
	  }
	  break;

	  case 'b': /* LP Filter cutoff frequency */
	  if( optarg == NULL ) break;
	  bw = atoi( optarg );
	  if( bw <= 0 )
	  {
		fprintf( stderr, "LPF Cut-off frequency not valid\n" );
		exit( -1 );
	  }
	  rc_data.rtlsdr_lpf_bw = bw;
	  break;
  
	  case 'c': /* Enable pseudo-colorization */
	  SetFlag( IMAGE_COLORIZE );
	  break;

	  case 'f': /* Set Tuner center frequency */
	  if( optarg == NULL ) break;
	  freq = atoi( optarg );
	  if( freq <= 0 )
	  {
		fprintf( stderr, "Tuner Center Frequency not valid\n" );
		exit( -1 );
	  }
	  rc_data.rtlsdr_center_freq = freq;
	  break;

	  case 'h': /* Print usage and exit */
	  Usage();
	  exit( 0 );

	  case 'i': /* Specify image file name */
	  if( optarg == NULL ) break;
	  if( strlen(optarg) >= MAX_FILE_NAME )
	  {
		fprintf( stderr, "Image file name too long\n" );
		exit( -1 );
	  }
	  strcpy( gbl_image_file, optarg );
	  break;

	  case 'l': /* Go into audio level setup loop */
	  SetFlag( ACTION_SETUP_AUDIO );
	  break;

	  case 'n': /* Suppress histogram normalization */
	  ClearFlag( IMAGE_NORMALIZE );
	  break;

	  case 'o': /* Rotate (flip) image over 180 deg */
	  SetFlag( IMAGE_ROTATE );
	  break;

	  case 'p': /* Process a recorded APT signal */
	  if( optarg == NULL ) break;
	  if( strlen(optarg) >= MAX_FILE_NAME )
	  {
		fprintf( stderr, "Samples file name too long\n" );
		exit( -1 );
	  }
	  strcpy( gbl_samples_file, optarg );
	  SetFlag( ACTION_PROCESS_FILE );
	  ClearFlag( ACTION_PROCESS_DSP );
	  break;

	  case 'r': /* Only record APT signal */
	  if( optarg != NULL )
	  {
		if( strlen(optarg) >= MAX_FILE_NAME )
		{
		  fprintf( stderr, "Samples file name too long\n" );
		  exit( -1 );
		}
		strcpy( gbl_samples_file, optarg );
	  }
	  SetFlag( ACTION_RECORD_APT );
	  ClearFlag( ACTION_PROCESS_DSP );
	  break;

	  case 'R': /* Record and then process APT signal */
	  if( optarg != NULL )
	  {
		if( strlen(optarg) > MAX_FILE_NAME )
		{
		  fprintf( stderr, "Samples file name too long\n" );
		  exit( -1 );
		}
		strcpy( gbl_samples_file, optarg );
	  }
	  SetFlag( ACTION_RECORD_APT | ACTION_PROCESS_FILE );
	  ClearFlag( ACTION_PROCESS_DSP );
	  break;

	  /* Decode Start-Stop times and calc sleep time and op duration */
	  case 's':
	  if( optarg == NULL ) break;
	  Start_Stop_Time(
		  optarg, &start_hrs, &start_min, &sleep_sec, &duration );
	  break;

	  case 't': /* Set time limit of record/process */
	  if( optarg == NULL ) break;
	  duration = (unsigned int)atoi( optarg );
	  if( duration <= 0 )
	  {
		fprintf( stderr, "Operation time-out value not valid\n" );
		exit( -1 );
	  }

	  /* If time limit is less than 60 sec, assume its in min */
	  if( duration < 60 ) duration *= 60;
	  gbl_duration = duration;
	  break;

	  case 'v': /* Print version */
	  puts( PACKAGE_STRING );
	  exit( 0 );

	  default: /* Print usage and exit */
	  Usage();
	  exit(-1);

	} /* End of switch( option ) */

  /* Sleep if delay > 0, exit on interrupt */
  if( sleep_sec > 0 )
  {
	fprintf( stderr,
		"wxapt: Going to sleep till %02d:%02d UTC\n", start_hrs, start_min );
	sleep( sleep_sec );
  }

  /* Carry out action according to options */
  if( isFlagSet(ACTION_RECORD_APT) )
  {
	SetFlag( ACTION_SYNC_BUFFER );
	Synchronize_Buffer();
	Record_APT_Signal();
	if( isFlagSet(ACTION_PROCESS_FILE) )
	  File_Process_Image();
  }
  else if( isFlagSet(ACTION_PROCESS_FILE) )
  {
	File_Process_Image();
  }
  else if( isFlagSet(ACTION_SETUP_AUDIO) )
  {
	Setup_Signal_Level();
  }
  else if( isFlagSet(ACTION_PROCESS_DSP) )
  {
	SetFlag( ACTION_SYNC_BUFFER );
	Synchronize_Buffer();
	Dsp_Process_Image();
  }

  Cleanup();

  return 0;
} /* main () */

/*------------------------------------------------------------------------*/

/*  sig_handler()
 *
 *  Signal Action Handler function
 */

static void sig_handler( int signal )
{
  static char xit = FALSE;
  switch( signal )
  {
	case SIGINT:
	  if( xit )
	  {
		fprintf( stderr, "\n%s\n", "wxapt: Exiting via User Interrupt" );
		Cleanup();
		exit(0);
	  }
	  ClearFlag( ACTION_FLAGS_ALL );
	  xit = TRUE;
	  return;

	case SIGSEGV:
	  fprintf( stderr, "\n%s\n", "wxapt: Segmentation Fault" );
	  break;

	case SIGFPE:
	  fprintf( stderr, "\n%s\n", "wxapt: Floating Point Exception" );
	  break;

	case SIGABRT:
	  fprintf( stderr, "\n%s\n", "wxapt: Abort Signal received" );
	  break;

	case SIGTERM:
	  fprintf( stderr, "\n%s\n", "wxapt: Termination Request received" );
	  break;
  }

  Cleanup();
  exit(-1);

} /* End of sig_handler() */

/*------------------------------------------------------------------------*/

