/*  sound.c
 *
 *  Soundcard handling functions of wxapt application
 */

/* wxapt: An application to decode APT signals from
 * weather satellites and produce image(s) of the weather.
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "sound.h"
#include "shared.h"

/* Receive samples buffer */
static short *recv_buffer  = NULL;

static int
  recv_buf_size,	/* Receive DSP signal samples buffer */
  recv_buf_idx;		/* Index to Receive signal samples buffer */

/* ALSA pcm capture and mixer handles */
static snd_pcm_t *capture_handle  = NULL;
static snd_mixer_t *mixer_handle  = NULL;
static snd_pcm_hw_params_t *hw_params = NULL;

/* Simple mixer elements for setting up
 * capture amd playback sources and volume */
static snd_mixer_elem_t *cap_elem = NULL;

/*------------------------------------------------------------------------*/

/* Open_PCM()
 *
 * Opens a pcm device for a given handle
 */
  void
Open_PCM(
	snd_pcm_t **handle,
	snd_pcm_stream_t stream )
{
  int error;

  /* Open pcm */
  error = snd_pcm_open(
	  handle, rc_data.pcm_dev, stream, SND_PCM_ASYNC );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot open sound device %s\n", rc_data.pcm_dev );
	exit( -1 );
  }

  /* Allocate memory to hardware parameters structure */
  error = snd_pcm_hw_params_malloc( &hw_params );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot allocate hw_params struct\n" );
	exit( -1 );
  }

  /* Initialize hardware parameter structure */
  error = snd_pcm_hw_params_any( *handle, hw_params );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot initialize hw_params struct\n" );
	exit( -1 );
  }

  /* Set access type */
  error = snd_pcm_hw_params_set_access(
	  *handle, hw_params, SND_PCM_ACCESS );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot set PCM access type\n" );
	exit( -1 );
  }

  /* Set sample format */
  error = snd_pcm_hw_params_set_format(
	  *handle, hw_params, SND_PCM_FORMAT );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot set sample format\n" );
	exit( -1 );
  }

  /* Set sample rate */
  error = snd_pcm_hw_params_set_rate(
	  *handle, hw_params, (unsigned int)SND_DSP_RATE, EXACT_VAL );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot set sample rate to %d\n", SND_DSP_RATE );
	exit( -1 );
  }

  /* Set channel count */
  error = snd_pcm_hw_params_set_channels(
	  *handle, hw_params, (unsigned int)rc_data.num_chn );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot set channel count to %d\n", rc_data.num_chn );
	exit( -1 );
  }

  /* Set number of periods */
  error = snd_pcm_hw_params_set_periods(
	  *handle, hw_params, NUM_PERIODS, EXACT_VAL );
  if( error < 0)
  {
	fprintf( stderr, "Cannot set number periods to %d\n", NUM_PERIODS );
	exit( -1 );
  }

  /* Set period size */
  error = snd_pcm_hw_params_set_period_size(
	  *handle, hw_params, PERIOD_SIZE, EXACT_VAL );
  if( error < 0)
  {
	fprintf( stderr, "Cannot set period size to %d\n", PERIOD_SIZE );
	exit( -1 );
  }

  /* Set parameters */
  error = snd_pcm_hw_params( *handle, hw_params );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot set capture parameters\n" );
	exit( -1 );
  }
  snd_pcm_hw_params_free( hw_params );
  hw_params = NULL;

  /* Prepare sound interface for use */
  error = snd_pcm_prepare( *handle );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot prepare sound interface\n" );
	exit( -1 );
  }

} /* Open_PCM() */

/*------------------------------------------------------------------------*/

/* Open_Capture()
 *
 * Opens sound card for Capture
 */
  void
Open_Capture( void )
{
  /* Return if Capture is setup */
  if( isFlagSet(CAPTURE_SETUP) ) return;

  /* Open & setup pcm for Capture */
  Open_PCM(	&capture_handle, SND_PCM_STREAM_CAPTURE );

  /* Allocate memory to DSP data buffer */
  rc_data.snd_buf_size = rc_data.num_chn * PERIOD_SIZE;
  if( gbl_snd_buffer == NULL )
  {
	mem_alloc( (void *)&gbl_snd_buffer,
		(size_t)rc_data.snd_buf_size * sizeof(short) );
	memset( gbl_snd_buffer, 0,
		(size_t)rc_data.snd_buf_size * sizeof(short) );
  }

  /* Size of receive samples buffer in 'shorts' */
  recv_buf_size = PERIOD_SIZE * rc_data.num_chn;

  /* Index to recv samples buffer (set to end) */
  recv_buf_idx = recv_buf_size;

  /* Allocate memory to receive samples buffer */
  if( recv_buffer == NULL )
  {
	mem_alloc( (void **)&recv_buffer,
		(size_t)recv_buf_size * sizeof(short) );
	memset( recv_buffer, 0,
		(size_t)recv_buf_size * sizeof(short) );
  }

  /* Open mixer & set playback voulume, abort on failure.
   * Failure to set volume level is not considered fatal */
  Open_Mixer();
  Set_Capture_Level( rc_data.cap_lev );

  SetFlag( CAPTURE_SETUP );

} /* Open_Capture() */

/*------------------------------------------------------------------------*/

/* Open_Mixer()
 *
 * Opens mixer interface
 */
  void
Open_Mixer( void )
{
  snd_mixer_elem_t *elem;
  snd_mixer_selem_id_t *sid;

  /* Abort if mixer already setup */
  if( isFlagSet(MIXER_SETUP) ) return;

  /* Open mixer handle */
  int error = snd_mixer_open( &mixer_handle, 0 );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot open mixer handle\n" );
	exit( -1 );
  }

  /* Attach mixer */
  error = snd_mixer_attach( mixer_handle, rc_data.pcm_dev );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot attach mixer to %s\n", rc_data.pcm_dev );
	exit( -1 );
  }

  /* Register mixer */
  error = snd_mixer_selem_register( mixer_handle, NULL, NULL );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot register mixer\n" );
	exit( -1 );
  }

  /* Load mixer */
  error = snd_mixer_load( mixer_handle );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot load mixer\n" );
	exit( -1 );
  }

  /* Allocate selem_id structure */
  error = snd_mixer_selem_id_malloc( &sid );
  if( error < 0 )
  {
	fprintf( stderr, "Cannot allocate selem_id struct\n" );
	exit( -1 );
  }

  /* Find capture source selem */
  snd_mixer_selem_id_set_index( sid, 0 );
  snd_mixer_selem_id_set_name( sid, rc_data.cap_src );
  elem = snd_mixer_find_selem( mixer_handle, sid );
  if( elem == NULL )
  {
	fprintf( stderr, "Cannot find capture source element %s\n", rc_data.cap_src );
	snd_mixer_selem_id_free(sid);
	exit( -1 );
  }

  /* Set capture switch for capture source */
  if( snd_mixer_selem_has_capture_switch(elem) )
  {
	error = snd_mixer_selem_set_capture_switch(
		elem, rc_data.channel, 1 );
	if( error < 0 )
	{
	  fprintf( stderr, "Cannot set capture device %s\n", rc_data.cap_src );
	  snd_mixer_selem_id_free(sid);
	  exit( -1 );
	}
  }
  else
  {
	fprintf( stderr, "Device %s does not have Capture capability\n", rc_data.cap_src );
	snd_mixer_selem_id_free(sid);
	exit( -1 );
  }

  /* Find capture volume selem if not -- */
  if( strcmp(rc_data.cap_vol, "--") != 0 )
  {
	snd_mixer_selem_id_set_index( sid, 0 );
	snd_mixer_selem_id_set_name( sid, rc_data.cap_vol );
	cap_elem = snd_mixer_find_selem( mixer_handle, sid );
	if( !cap_elem )
	{
	  fprintf( stderr, "Cannot find Capture volume %s\n", rc_data.cap_vol );
	}
  }
  snd_mixer_selem_id_free(sid);

  SetFlag( MIXER_SETUP );
} /* Open_Mixer() */

/*------------------------------------------------------------------------*/

/* Set_Capture_Level()
 *
 * Sets Capture Control level
 */
  void
Set_Capture_Level( int level )
{
  long cmin, cmax;
  int error;

  /* Abort with no error if Mixer not setup */
  if( mixer_handle == NULL ) return;

  /* Set capture volume */
  if( cap_elem != NULL )
  {
	if( snd_mixer_selem_has_capture_volume(cap_elem) )
	{
	  /* Change from % volume to sound card value */
	  long lev;
	  snd_mixer_selem_get_capture_volume_range(
		  cap_elem, &cmin, &cmax );
	  lev = cmin + ((cmax - cmin) * level) / 100;

	  /* Set capture volume */
	  error = snd_mixer_selem_set_capture_volume(
		  cap_elem, rc_data.channel, lev );
	  if( error < 0 )
	  {
		fprintf( stderr, "Cannot set capture volume to %d: Error: %s\n",
			level, snd_strerror(error) );
		exit( -1 );
	  }
	}
  } /* if( cap_elem != NULL ) */

} /* Set_Capture_Level() */

/*------------------------------------------------------------------------*/

/*
 * Close sound card interfaces
 */

  void
Close_Capture( void )
{
  if( capture_handle != NULL )
	snd_pcm_close( capture_handle );
  if( hw_params != NULL )
	snd_pcm_hw_params_free( hw_params );
  if( mixer_handle != NULL )
	snd_mixer_close( mixer_handle );
}

/*------------------------------------------------------------------------*/

/* Read_SND_Buffer()
 *
 * Reads the DSP's buffer and separates Left/Right
 * channel data depending on stereo or mono mode
 */

  void
Read_SND_Buffer( short *buffer, int buff_size )
{
  int idx;
  snd_pcm_sframes_t error;


  /* Transfer data from dsp buffer to buffer to be processed, */
  /* separating left or right channel data if in stereo mode  */
  if( buff_size >= BLOCK_BUFFER_SIZE )
	buff_size -= BLOCK_BUFFER_SIZE;
  for( idx = 0; idx < buff_size; idx++ )
  {
	if( snd_buf_idx >= rc_data.snd_buf_size )
	{
	  /* Start buffer index according to channel */
	  snd_buf_idx = rc_data.use_chn;

	  /* Fill in the samples buffer from DSP, return -1 on error */
	  error = snd_pcm_readi(
		  capture_handle, gbl_snd_buffer, PERIOD_SIZE );
	  if( error != PERIOD_SIZE )
	  {
		char mesg[MESG_SIZE];
		snprintf( mesg, MESG_SIZE,
			"Read from sound interface failed\n"\
			  "Error: %s", snd_strerror((int)error) );
		fprintf( stderr, "%s\n", mesg );

		/* Try to recover from error */
		Xrun_Recovery(capture_handle, (int)error);

	  } /* if( error != NUM_FRAMES ) */
	} /* if( snd_buf_idx >= rc_data.snd_buf_size ) */

	/* Fill buffer from left or right channel. snd_buf_idx
	 * is arranged to point to the correct location */
	buffer[idx] = gbl_snd_buffer[snd_buf_idx];

	/* Simulates a pulsed 2.4kHz signal
	{
	  static double w  = 0.0;
	  static double dw = TWOPI * 2400.0 / 48000.0;
	  static int cnt = 0;

	  if( cnt < 46 )
		buffer[idx] = 0;
	  else if( cnt < 69 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 92 )
		buffer[idx] = 0;
	  else if( cnt < 115 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 138 )
		buffer[idx] = 0;
	  else if( cnt < 161 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 184 )
		buffer[idx] = 0;
	  else if( cnt < 207 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 230 )
		buffer[idx] = 0;
	  else if( cnt < 253 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 276 )
		buffer[idx] = 0;
	  else if( cnt < 299 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 322 )
		buffer[idx] = 0;
	  else if( cnt < 345 )
		buffer[idx] = (short)( 32000.0 * sin(w) );
	  else if( cnt < 448 )
		buffer[idx] = 0;
	  else if( cnt < 1004 )
		buffer[idx] = (short)( 10000.0 * sin(w) );
	  else
	  {
		if( (cnt / 400) & 1 )
		  buffer[idx] = (short)( 32000.0 * sin(w) );
		else buffer[idx] = 0;
	  }

	  w += dw;
	  if( w >= TWOPI ) w -= TWOPI;
	  if( cnt++ == 24000 ) cnt = 0;
	} */

	/* Increment according to mono/stereo mode */
	snd_buf_idx += rc_data.num_chn;

  } /* for( idx = 0; idx < buff_size; idx++ ) */

} /* End of Read_SND_Buffer() */

/*------------------------------------------------------------------------*/

/* Xrun_Recovery()
 *
 * Recover from underrrun (broken pipe) and suspend
 */
  void
Xrun_Recovery( snd_pcm_t *handle, int error )
{
  char mesg[MESG_SIZE];
  if( error == -EPIPE )
  {
	error = snd_pcm_prepare( handle );
	if( error < 0 )
	{
	  snprintf( mesg, sizeof(mesg),
		  "Cannot recover from underrun, prepare failed\n"\
			"Error: %s\n", snd_strerror(error) );
	  fprintf( stderr, "%s\n", mesg );
	  exit( -1 );
	}
  }
  else if( error == -ESTRPIPE )
  {
	while( (error = snd_pcm_resume(handle)) == -EAGAIN )
	  sleep(1);
	if( error < 0 )
	{
	  error = snd_pcm_prepare( handle );
	  if( error < 0 )
	  {
		snprintf( mesg, sizeof(mesg),
			"Cannot recover from suspend, prepare failed\n"\
			  "Error: %s\n", snd_strerror(error) );
		fprintf( stderr, "%s\n", mesg );
		exit( -1 );
	  }
	}
  }

} /* Xrun_Recovery() */

/*------------------------------------------------------------------------*/

