/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm.h"
#include "rfm_modules.h"


void 
rfm_time_out(widgets_t *widgets_p, const gchar *path){
    rfm_threaded_show_text(widgets_p);
    rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-error", g_strconcat(path, ": ", NULL));
    rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", g_strconcat(_(strerror(ETIMEDOUT)), "\n", NULL));
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_threaded_cursor_reset(rfm_global_p->window);
    return;
}

gboolean rfm_entry_available(widgets_t *widgets_p, record_entry_t *en){
    if (!en) return TRUE;
    if (!en->path) return FALSE;
    gboolean is_module = (en->module != NULL);
    gboolean retval = TRUE;
    if (!is_module){
        rfm_global_t *rfm_global_p = rfm_global();
	rfm_threaded_cursor_wait(rfm_global_p->window);
	retval = rfm_g_file_test_with_wait(en->path, G_FILE_TEST_EXISTS);
	rfm_threaded_cursor_reset(rfm_global_p->window);
    }
    if (!retval) {
	rfm_time_out(widgets_p, en->path);
    }
    return retval;
}


static 
gchar *get_mnt_point(record_entry_t *en){
    
    if (!en || !en->path) return FALSE;
    if (!g_path_is_absolute(en->path)) return FALSE;
    gchar *mnt_point;
    if (IS_SDIR(en->type)){
	mnt_point = g_strdup(en->path);
    } else {
	mnt_point = g_path_get_dirname(en->path);
    }
    if (!mnt_point) return FALSE;
    gchar *rpath = realpath(mnt_point, NULL);
    g_free(mnt_point);
    mnt_point = rpath;
    if (!rpath) return FALSE;

   do {	
       if (FSTAB_is_mounted(mnt_point)) {
	   return mnt_point;
       }
       NOOP(stderr, "not mounted: %s\n", mnt_point);
       gchar *path = g_path_get_dirname(mnt_point);
       g_free(mnt_point);
       mnt_point = path;
   } while (mnt_point && strcmp(mnt_point, "/"));


    return mnt_point;
}

static 
gchar *get_mnt_partition(const gchar *mnt_point){
   gchar *partition = FSTAB_get_mnt_fsname(mnt_point);
   if (!partition) return NULL;
   NOOP(stderr, "*** partition (%s) = %s\n", mnt_point, partition);
   return partition;
}

static gboolean set_remote_mnt_type(const gchar *path, gint64 type){
   gboolean retval = FALSE;
    DBHashTable *dbh;
    gchar *dbh_path = g_build_filename(MOUNT_TYPE_DBH_FILE, NULL);
 
    unsigned char keylength = 11;
    TRACE("opening %s...\n",dbh_path); 
    dbh = dbh_new(dbh_path, &keylength, DBH_THREAD_SAFE|DBH_PARALLEL_SAFE);
    if (!dbh) {
        gchar *directory = g_path_get_dirname(dbh_path);
        if (!g_file_test(directory, G_FILE_TEST_IS_DIR)){
            g_mkdir_with_parents(directory, 0700);
        }
        g_free(directory);
	dbh = dbh_new(dbh_path, &keylength, DBH_CREATE|DBH_THREAD_SAFE|DBH_PARALLEL_SAFE);
    }
    TRACE("opened %s.\n",dbh_path); 
    if (!dbh){
	DBG("cannot create DBH file: %s\n", dbh_path);
	g_free(dbh_path);
	return retval;
    }
    dbh_set_parallel_lock_timeout(dbh, 3);
    gchar *rpath = realpath(path, NULL);
    GString *gs = g_string_new (rpath);
    sprintf ((char *)DBH_KEY (dbh), "%10u", g_string_hash (gs));
    NOOP("%s -> %10u\n", rpath, g_string_hash (gs));
    g_string_free (gs, TRUE);
    dbh_set_recordsize(dbh, sizeof(gint64));
    *((gint64 *)(dbh->data)) = type;
    if (dbh_update (dbh) == 0){
	DBG("could not update %s\n", dbh_path);
    } else {
	retval = TRUE;
    }
    dbh_close (dbh);
    g_free(dbh_path);
    g_free(rpath);
    return retval;
}

static gint64 get_remote_mnt_type(const gchar *path){
    gint retval = 0;
    DBHashTable *dbh;
    gchar *dbh_path = g_build_filename(MOUNT_TYPE_DBH_FILE, NULL);
    if (!g_file_test(dbh_path, G_FILE_TEST_EXISTS)){
	NOOP("cannot open DBH file: %s\n", dbh_path);
	g_free(dbh_path);
	return 0;
    }

 
    unsigned char keylength = 11;
    TRACE("opening %s...\n",dbh_path); 
    dbh = dbh_new(dbh_path, &keylength, DBH_THREAD_SAFE|DBH_PARALLEL_SAFE);
    TRACE("opened %s.\n",dbh_path); 
    if (!dbh){
	if (g_file_test(dbh_path, G_FILE_TEST_EXISTS)){
	    NOOP("cannot open DBH file: %s\n", dbh_path);
	}
	g_free(dbh_path);
	return retval;
    }
    dbh_set_parallel_lock_timeout(dbh, 3);
    gchar *rpath = realpath(path, NULL);
    GString *gs = g_string_new (rpath);
    sprintf ((char *)DBH_KEY (dbh), "%10u", g_string_hash (gs));
    NOOP("%s -> %10u\n", rpath, g_string_hash (gs));
    g_string_free (gs, TRUE);
    if (dbh_load(dbh)){
	retval = *((gint64 *)(dbh->data));
    }
    dbh_close (dbh);
    g_free(dbh_path);
    g_free(rpath);
    return retval;
}

static gint
get_mnt_type (record_entry_t *en){
    if (!en || !en->path) {
	NOOP(stderr, "xfdir_mnt_type(%s) = %s\n", in_path, "no");
	return 0;
    }
    
    NOOP("heartbeat records, getting mount point\n");

// 1. Get mount point... (this function will do the "realpath" internally)
// does not block:
   gchar *mnt_point = get_mnt_point(en);

   // No mnt_point, then it is not local...
   if (!mnt_point) {
       NOOP(stderr, "no mount point for %s\n", en->path);
       return 0;
   }
  
   // root directory is always local
   if (strcmp(mnt_point, "/")==0){
	NOOP(stderr, "xfdir_mnt_type(%s) = %s --> %s\n", 
		in_path, "yes", mnt_point);
       g_free(mnt_point);
       return 2;
   }

// Before testing a mountpoint for partition, see if item is
// in the dbh table for monitor or local override type.
//
// Remote files may be considered local on user request basis.
   gint mnt_type = get_remote_mnt_type(mnt_point);
   if (mnt_type) {
	NOOP(stderr, "xfdir_mnt_type(%s) --> %d\n", en->path, mnt_type);
	g_free(mnt_point);
	return mnt_type;
   }


// 2. Get mount point partition.
// Methinks this does not block
   gchar *partition = get_mnt_partition(mnt_point);
   if (!partition) {
	NOOP(stderr, "NO partition for %s\n", en->path);
        g_free(mnt_point);
        return 0;
   }
   gboolean is_local =  g_path_is_absolute(partition) && 
			rfm_g_file_test(partition, G_FILE_TEST_EXISTS);
			;
    if (!is_local) NOOP(stderr, "partition not local %s\n", partition);
    g_free(partition);
    g_free(mnt_point);
   if (is_local){
       return 2;
   }
   return 0;
}

gboolean
rfm_set_monitor_type(const gchar *path) {
    NOOP(stderr, "setting %s as monitor type\n", path);
    return set_remote_mnt_type(path, 1);
}

gboolean
rfm_set_local_type(const gchar *path) {
    return set_remote_mnt_type(path, 2);
} 

gboolean
rfm_set_remote_type(const gchar *path) {
    return set_remote_mnt_type(path, 0);
} 

static void
clear_mimetype(record_entry_t *en){
    if (!en) return;
    g_free(en->encoding);
    g_free(en->filetype);
    g_free(en->mimetype);
    g_free(en->mimemagic);
    en->mimetype = en->mimemagic = NULL;
    en->encoding = en->filetype = NULL;
}

static void
free_dtype_items(record_entry_t *en){
    g_free(en->mimetype);
    g_free(en->mimemagic);
    g_free(en->filetype);
    g_free(en->encoding);
}

void
rfm_set_mime_dtype(record_entry_t *en){
    if (!en) return;
    if (IS_SDIR(en->type)) {
	free_dtype_items(en);
	en->mimetype = g_strdup("inode/directory");
	en->mimemagic = g_strdup("inode/directory");
	en->filetype = g_strdup(_("Directory"));
	en->encoding = g_strdup(_("binary"));
    }
    if (IS_SCHR(en->type)){
	free_dtype_items(en);
	en->mimetype = g_strdup("inode/chardevice");
	en->mimemagic = g_strdup("inode/chardevice");
	en->filetype = g_strdup(_("Character device"));
	en->encoding = g_strdup(_("C"));
    }
    if (IS_SBLK(en->type)) {
	free_dtype_items(en);
	en->mimetype = g_strdup("inode/blockdevice");
	en->mimemagic = g_strdup("inode/blockdevice");	
	en->filetype = g_strdup(_("Block device"));
	en->encoding = g_strdup(_("C"));
    }
    if (IS_SFIFO(en->type)) {
	free_dtype_items(en);
	en->mimetype = g_strdup("inode/fifo");
	en->mimemagic = g_strdup("inode/fifo");	
	en->filetype = g_strdup(_("FIFO"));
	en->encoding = g_strdup(_("C"));
    }
    if (IS_SSOCK(en->type)) {
	free_dtype_items(en);
	en->mimetype = g_strdup("inode/socket");
	en->mimemagic = g_strdup("inode/socket");	
	en->filetype = g_strdup(_("Socket"));
	en->encoding = g_strdup(_("C"));
    }
}




/**
 * rfm_copy_entry:
 * @src_en: a valid record_entry_t pointer
 * Returns: allocated record_entry_t pointer with same information as src_en
 *
 * Allocates memory and copies the input record entry. 
 *
 **/
record_entry_t *
rfm_copy_entry (const record_entry_t * src_en) {
    record_entry_t *en;
    if(!src_en)
        return NULL;
    en = (record_entry_t *) malloc (sizeof (record_entry_t));
    if (!en) g_error("malloc: %s\n", strerror(errno));
    memset (en, 0, sizeof (record_entry_t));
    memcpy (en, src_en, sizeof (record_entry_t));

    if(src_en->st) {
        en->st = (struct stat *)malloc (sizeof (struct stat));
	if (!en->st) g_error("malloc: %s\n", strerror(errno));
        memcpy (en->st, src_en->st, sizeof (struct stat));
    }

    en->type = src_en->type;
    en->module = (src_en->module) ? src_en->module : NULL;
    en->tag = (src_en->tag) ? g_strdup (src_en->tag) : NULL;
    en->path = (src_en->path) ? g_strdup (src_en->path) : NULL;
    en->pseudo_path = (src_en->pseudo_path) ? 
	g_strdup (src_en->pseudo_path) : NULL;

    en->filetype = (src_en->filetype) ? g_strdup (src_en->filetype) : NULL;
    en->mimetype = (src_en->mimetype) ? g_strdup (src_en->mimetype) : NULL;
    en->mimemagic = (src_en->mimemagic) ? g_strdup (src_en->mimemagic) : NULL;
    en->encoding = (src_en->encoding) ? g_strdup (src_en->encoding) : NULL;


    return en;
}

/**
 * rfm_destroy_entry:
 * @en: a valid record_entry_t pointer
 *
 * Frees allocated memory for the record entry.
 *
 **/
void
rfm_destroy_entry (record_entry_t * en) {
    if(!en) return;
    g_free (en->tag);
    g_free (en->path);
    g_free (en->pseudo_path);

    g_free (en->filetype);
    g_free (en->mimetype);
    g_free (en->mimemagic);
    g_free (en->encoding);


    g_free (en->st);
    g_free (en);
    en = NULL;
}

/**
 * rfm_mk_entry:
 * Returns: allocated record_entry_t pointer
 *
 * Allocates memory for an empty record entry inheriting @type characteristics.
 **/
record_entry_t *
rfm_mk_entry (gint dtype) {
    record_entry_t *en;
    en = (record_entry_t *) malloc (sizeof (record_entry_t));
    if (!en) g_error("malloc: %s\n", strerror(errno));
    memset (en, 0, sizeof (record_entry_t));
    en->type = dtype & __DTYPE;
    rfm_set_mime_dtype(en);
   return en;
}

/**
 * rfm_mk_entry_path:
 * @path: path to local file to obtain stat information.
 * @type: Type of record entry to inherit. May be 0.
 * Returns: allocated record_entry_t pointer
 *
 * Allocates memory without filling in stat information for a record entry 
 * pointing to @path. Type may be inherited from @type if different
 * from 0.
 **/

record_entry_t *
rfm_mk_entry_path (const gchar * path, gint dtype) {
    record_entry_t *en;
    en = rfm_mk_entry (dtype);
    en->st = (struct stat *)malloc (sizeof (struct stat));
    if (!en->st) g_error("malloc: %s\n", strerror(errno));
    memset(en->st, 0,sizeof(struct stat));
    
    en->path = g_strdup (path);
    return en;
}

static void
set_mount_status(record_entry_t *en){
    // We requiere previous stat.
    // Directories may change from local to remote on mounts
    if (!IS_SDIR(en->type) && !IS_SBLK(en->type)
	// && !isofs
       ) return;

     gint type = get_mnt_type(en);
     if (type < 0){
	 // blocked
	SET_NOACCESS_TYPE(en->type);
	SET_NOWRITE_TYPE (en->type);
	return;
     }
    if (type == 2) {
	SET_LOCAL_TYPE(en->type);
	SET_MONITOR_TYPE(en->type);
    }
    else {
	UNSET_LOCAL_TYPE(en->type);
	if (type == 1) SET_MONITOR_TYPE(en->type);
	else UNSET_MONITOR_TYPE(en->type);
    }


    // This sets up the greenball update, if necessary.
    // (only if fstab plugin is available, which usually is)
    gint mount_status =  FSTAB_entry_is_mounted(en);
    if (mount_status != 0){
	SET_MOUNTED_TYPE(en->type);
	NOOP(stderr, "set_mount_status(): %s is now mounted\n", en->path); 
    } else if (IS_MOUNTED_TYPE(en->type)){
	UNSET_MOUNTED_TYPE(en->type);
	NOOP(stderr, "set_mount_status(): %s is now unmounted\n", en->path); 
    } 
    return;
}


static gboolean
stat_access_mode(record_entry_t *en){

     // Now we stat, following symlinks.
     //
    if(stat (en->path, en->st) < 0) {
	TRACE ("stat_access_mode(): cannot stat %s (%s)\n",
		    en->path, strerror(errno));
	memset (en->st, 0, sizeof (struct stat));
	if (lstat(en->path, en->st)==0){
            if (IS_SLNK(en->st->st_mode)) SET_BROKEN_LNK (en->type);
            else SET_NOACCESS_TYPE(en->type);
        }
        else DBG("stat_access_mode(): cannot lstat %s (%s)\n",
                en->path, strerror(errno));
	return FALSE;
    }

    // stat type overrides d_type (which is not always available)
    //
    // if d_type not specified or we are dealing with a symlink... 
    // XXX d_type also fails in remote set as local...
    // if (en->type == 0 || IS_SLNK(en->type))
    {
	if (S_ISDIR(en->st->st_mode)) SET_SDIR(en->type);
	else if (S_ISLNK(en->st->st_mode)) SET_SLNK(en->type);
	else if (S_ISSOCK(en->st->st_mode)) SET_SSOCK(en->type);
	else if (S_ISFIFO(en->st->st_mode)) SET_SFIFO(en->type);
	else if (S_ISBLK(en->st->st_mode)) SET_SBLK(en->type);
	else if (S_ISCHR(en->st->st_mode)) SET_SCHR(en->type);
	else if (S_ISREG(en->st->st_mode)) SET_SREG(en->type);
	clear_mimetype(en);
	rfm_set_mime_dtype(en);

    }


    // Stat has been done.

     if(access (en->path, R_OK) < 0) {
	SET_NOACCESS_TYPE (en->type);
     }

    if(access (en->path, W_OK) != 0) {
	if(S_ISDIR (en->st->st_mode)) {
	    SET_NOWRITE_TYPE (en->type);
	    return TRUE;
	} 
	// If you can write to the directory you can
	// write to the file. Let us check... 
	gchar *g = g_path_get_dirname (en->path);
	if(access (g, W_OK) != 0) {
	    SET_NOWRITE_TYPE (en->type);
	}
	g_free (g);
    }
    if(en->st->st_mode & (S_IXUSR|S_IXGRP|S_IXOTH) && S_ISREG(en->st->st_mode)) {
        SET_EXE_TYPE (en->type);
    }
    return TRUE;
}

void
rfm_entry_set_type(record_entry_t *en){
    NOOP(stderr, "rfm_entry_set_type %s\n", en->path);
    stat_access_mode(en);
    set_mount_status(en);
    return;
}


/**
 * rfm_stat_entry:
 * @path: path to local file to obtain stat information.
 * @type: Type of record entry to inherit. May be 0.
 * Returns: allocated record_entry_t pointer
 *
 * Allocates memory and fills in stat information for a record entry 
 * pointing to @path. Type may be inherited from @type if different
 * from 0.
 *
 **/
record_entry_t *
rfm_stat_entry (const gchar * path, gint dtype) {
    NOOP(stderr, "rfm_stat_entry %s\n", path);
    record_entry_t *en = rfm_mk_entry_path (path, dtype);
	
    rfm_entry_set_type(en);
    return en;
}

// hack test
gboolean rfm_is_valid_entry(view_t *view_p, record_entry_t *en){
    if (!rfm_population_read_lock (view_p, "rfm_is_valid_entry")){
        return FALSE;
    }
    gboolean valid=FALSE;
    gint i;
    for (i=0; view_p->population_pp && view_p->population_pp[i]; i++){
	if (view_p->population_pp[i]->en == en){
	    valid=TRUE;
	    break;
	}
    }
    rfm_population_read_unlock (view_p, "rfm_is_valid_entry");
    if (!valid) {
	NOOP("rfm_is_valid_entry(): dropping invalid entry\n");
    }
    return valid;
}
