/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#define RFM_PRIMARY_C

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm.h"
#include "rfm_modules.h"

// static functions:
#include "primary.i"
#include "primary-ls.i"

gchar *
rfm_default_url_mount_point(const gchar *url){
    gchar *computer=NULL;
    gchar *remote_path=NULL;
    gchar *u = NULL;
    if (url){
	u = g_strdup(url);
	gchar *p = strstr(u, "://");
	if (p) {
	    p = p+strlen("://");
	    if (strchr(p,'/')){
		*strchr(p,'/') = 0;
		computer = g_strdup(p);
		remote_path = g_strdup_printf("/%s",p + strlen(p) + 1);
	    }
	}
	g_free(u);
    }

    	gchar *user = g_path_get_basename(g_get_home_dir ());
	gchar *dir;
	gchar *device = NULL;
	if (remote_path) {
	    device = g_path_is_absolute(remote_path)? remote_path+1: remote_path;
	}
	if (computer && remote_path) {
	    dir = g_strdup_printf("%s-%s", computer, device);
	} else {
	    dir = g_strdup((computer)? computer : device);
	}
	gchar *default_value = 
	    g_build_filename (g_get_tmp_dir (), user, "mnt", dir, NULL);
	g_free(user);
	g_free(dir);
	g_free(computer);
	g_free(remote_path);
	return default_value;

}

static GSList *valid_view_list = NULL;
static pthread_rwlock_t valid_view_rwlock;
static pthread_once_t valid_view_once_control = PTHREAD_ONCE_INIT;
static pthread_mutex_t view_list_mutex = PTHREAD_MUTEX_INITIALIZER;

static 
void valid_view_init(void){
    pthread_rwlockattr_t Attr;
    pthread_rwlockattr_init(&Attr);
    pthread_rwlock_init(&valid_view_rwlock, &Attr);
}


#if defined DEBUG_TRACE || defined DEBUG
static gint mutex_count = 0;
#endif

void 
rfm_view_list_unlock(const gchar *source){
    NOOP("- %s -- unlocking view_list now... %d\n",source, --mutex_count);
#if defined DEBUG || defined DEBUG_TRACE
    if (mutex_count < 0) g_error("unlocking non locked RWlock\n");
#endif
    pthread_rwlock_unlock(&valid_view_rwlock);

}

static GSList ** 
view_list_lock(view_t *view_p, gboolean w, const gchar *source){
    NOOP("+++ attempting %s lock for view_list... (%d)\n", 
            (w)?"write":"read", mutex_count);
    
    pthread_once(&valid_view_once_control, valid_view_init);
    if (w) {
        pthread_rwlock_wrlock(&valid_view_rwlock);
    } else {
        pthread_rwlock_rdlock(&valid_view_rwlock);
    }
    NOOP("+ %s ++ locking view_list now (%s)... %d\n", source,
            (w)?"write":"read", ++mutex_count);
    if (!view_p) return &valid_view_list;

    pthread_mutex_lock(&view_list_mutex);
    void *data = g_slist_find(valid_view_list, view_p);
    pthread_mutex_unlock(&view_list_mutex);

    if (data){
	return &valid_view_list;
    }
    NOOP("view %p not found in list\n", view_p);
    rfm_view_list_unlock("view_list_lock");
    return NULL;
}

GSList ** 
rfm_view_list_lock(view_t *view_p, const gchar *source){
    GSList **p=view_list_lock(view_p, FALSE, source);
    return p;
}


void
rfm_add_view(view_t *view_p){
    TRACE("rfm_add_view()...\n");
    // write lock on valid views.
    view_list_lock(NULL, FALSE, "rfm_add_view");
    pthread_mutex_lock(&view_list_mutex);
    void *data = g_slist_find(valid_view_list, view_p);
    pthread_mutex_unlock(&view_list_mutex);
    if (data){
            DBG("view %p already in viewlist\n", view_p);
            rfm_view_list_unlock("rfm_add_view");
            return;
    }
    TRACE("adding view %p to list\n", view_p);
    pthread_mutex_lock(&view_list_mutex);
    valid_view_list = g_slist_prepend(valid_view_list, view_p);
    pthread_mutex_unlock(&view_list_mutex);
    rfm_view_list_unlock("rfm_add_view");
    return;
}

void
rfm_rm_view(view_t *view_p){
    TRACE("rfm_rm_view()...\n");
    // write lock on valid views.
    view_list_lock(NULL, TRUE, "rfm_rm_view");

    pthread_mutex_lock(&view_list_mutex);
    void *data = g_slist_find(valid_view_list, view_p);
    pthread_mutex_unlock(&view_list_mutex);

    if (data){
        pthread_mutex_lock(&view_list_mutex);
        valid_view_list = g_slist_remove(valid_view_list, view_p);
        pthread_mutex_unlock(&view_list_mutex);
        rfm_view_list_unlock("rfm_rm_view");
        return;
    }
    rfm_view_list_unlock("rfm_rm_view");
    DBG("view %p not in viewlist\n", view_p);
    return;
}



extern GThread *main_loop_thread;


static gboolean
main_context_function_f(gpointer data){
    void **arg = data;
    void * (*function)(gpointer) = arg[0];
    gpointer function_data = arg[1];
    GMutex *mutex = arg[2];
    GCond *signal = arg[3];
    void **result_p = arg[4];
    void *result = (*function)(function_data);
    g_mutex_lock(mutex);
    *result_p = result;
    g_cond_signal(signal);
    g_mutex_unlock(mutex);
    return FALSE;
}

void *
rfm_context_function(void * (*function)(gpointer), void * function_data){
    void *arg[5];
    GMutex *mutex = NULL;
    rfm_mutex_init(mutex);
    GCond *signal;
    rfm_cond_init(signal);
    void *result=GINT_TO_POINTER(-1);
    arg[0] = function;
    arg[1] = function_data;
    arg[2] = mutex;
    arg[3] = signal;
    arg[4] = &result;
    
    if (!rfm_get_gtk_thread()){
        g_warning("gtk_main_thread not set. You can expect problems.\n");
    }
    if (rfm_get_gtk_thread()== g_thread_self()){
        NOOP("main call to context function\n");
	main_context_function_f(arg);
    } else {
        NOOP("thread call to context function\n");
	g_main_context_invoke(NULL, main_context_function_f, arg);
	g_mutex_lock(mutex);
	if (result == GINT_TO_POINTER(-1)) g_cond_wait(signal, mutex);
	g_mutex_unlock(mutex);
    }
    rfm_mutex_free(mutex);
    rfm_cond_free(signal);
    return result;
}

gboolean 
rfm_main_context_check(view_t *view_p, gboolean view_entry){
    // Check if window is not in exit mode
    rfm_global_t *rfm_global_p = rfm_global();
    g_mutex_lock(rfm_global_p->status_mutex);
    gint status = rfm_global_p->status;
    g_mutex_unlock(rfm_global_p->status_mutex);
    if(status == STATUS_EXIT) return FALSE;
    if (!view_p) return TRUE;

    // Check if view is still listed
    TRACE("rfm_main_context_check()...\n");
    if (!rfm_view_list_lock(view_p, "rfm_main_context_check")) return FALSE;

    // Check if view is not in exit mode
    status = view_p->flags.status;
    if(status == STATUS_EXIT) goto done;

    // Check if view->en is still valid (move callbacks.i function to rfm lib)
    if (view_entry &&
	    !rfm_entry_available(&(view_p->widgets), view_p->en)){
	goto done;
    }
    rfm_view_list_unlock("rfm_main_context_check1");
    return TRUE;
done:
    rfm_view_list_unlock("rfm_main_context_check2");
    return FALSE;
}

RFM_RW_LOCK_INIT(drag_info_lock);

//#define LOCK_TRACE

#ifdef LOCK_TRACE
/*static GHashTable *thread_hash = NULL;
static GHashTable *lock_ref_hash = NULL;*/
static gint pop_lock_count = 0;

/*const gchar *
get_thread_text(GThread *thread){
    if (rfm_get_gtk_thread() == thread) return "main thread";
    gchar *text = g_hash_table_lookup(thread_hash, thread);
    return (const gchar *)text;
}*/


/*gint thread_has_lock(GThread *thread){
    return GPOINTER_TO_INT(g_hash_table_lookup(lock_ref_hash, thread));
}*/

static void inc_RW(gboolean increment, const gchar *source){
    static pthread_mutex_t lock_ref_mutex=PTHREAD_MUTEX_INITIALIZER;
	pthread_mutex_lock(&lock_ref_mutex);
	/*if (!lock_ref_hash){
	    lock_ref_hash = g_hash_table_new(g_direct_hash, g_direct_equal);
	}*/
	//gint count = GPOINTER_TO_INT(g_hash_table_lookup(lock_ref_hash, g_thread_self()));
        // These two commented out TRACE instructions will really vomit output
        // and turn application into a snail. Use with care.
	if (increment) {
	    //count++;
            pop_lock_count++;
		//if (get_thread_text(g_thread_self()))
    			DBG("lock...%s: %d\n", source, pop_lock_count);
    			//DBG("lock...%s\n",  get_thread_text(g_thread_self()));
	} else {
	    //count--;
            pop_lock_count--;
		//if (get_thread_text(g_thread_self()))
    			DBG("unlock...%s: %d\n", source, pop_lock_count);
    			//DBG("unlock...%s\n", get_thread_text(g_thread_self()));
	}
	//g_hash_table_replace(lock_ref_hash, g_thread_self(), GINT_TO_POINTER(count));
	pthread_mutex_unlock(&lock_ref_mutex);
    }
#endif

//#define FILTER if (!strstr(source, "rect"))
//#define FILTER if (0)
#define FILTER if (0)
void
rfm_population_read_unlock (view_t * view_p, const gchar *source) {
    FILTER DBG("--- rfm_population_read_unlock() from  %s\n", source);
    rfm_rw_lock_reader_unlock(&(view_p->mutexes.population_rwlock));
#ifdef LOCK_TRACE
    inc_RW(FALSE, source);
#endif
}
void
rfm_population_write_unlock (view_t * view_p, const gchar *source) {
    FILTER DBG("--- rfm_population_write_unlock() from  %s\n", source);
    rfm_rw_lock_writer_unlock(&(view_p->mutexes.population_rwlock));
#ifdef LOCK_TRACE
    inc_RW(FALSE, source);
#endif
}


gboolean
rfm_population_write_lock (view_t * view_p, const gchar *source) {

    // This should only return false on serial mismatch.
    // Write lock on exit condition is necessary for view cleanup
    gint population_serial = view_p->flags.population_serial;


    FILTER DBG("... rfm_population_write_lock() request from  %s\n", source);
    rfm_rw_lock_writer_lock(&(view_p->mutexes.population_rwlock));

    FILTER DBG("+++ rfm_population_write_lock() obtained for  %s\n", source);
    gint obtained_serial = view_p->flags.population_serial;
    if (obtained_serial != population_serial) {
	rfm_rw_lock_writer_unlock(&(view_p->mutexes.population_rwlock));
	return FALSE;
    }
#ifdef LOCK_TRACE
    inc_RW(TRUE, source);
#endif
    return TRUE;
}


gboolean
rfm_population_read_lock (view_t * view_p, const gchar *source) {
    gboolean status = view_p->flags.status;
    if (status == STATUS_EXIT){
	TRACE( "rfm_population_read_lock STATUS_EXIT\n");
	return FALSE;
    }
    gint population_serial = view_p->flags.population_serial;

    NOOP("... rfm_population_read_lock() request from  %s\n", source);
    rfm_rw_lock_reader_lock(&(view_p->mutexes.population_rwlock));
    FILTER DBG("ooo rfm_population_reader_lock() obtained for  %s\n", source);
    gint obtained_serial = view_p->flags.population_serial;
    if (obtained_serial != population_serial) {
	DBG( "population serial mismatch: %d != %d read lock cancelled.\n",
	    obtained_serial, population_serial);
	rfm_rw_lock_reader_unlock(&(view_p->mutexes.population_rwlock));
	return FALSE;
    }
#ifdef LOCK_TRACE
    inc_RW(TRUE, source);
#endif

    return TRUE;
}

gboolean
rfm_population_try_read_lock (view_t * view_p, const gchar *source) {
    gboolean status = view_p->flags.status;
    if (status == STATUS_EXIT) return FALSE;
    gint population_serial = view_p->flags.population_serial;

    NOOP("... rfm_population_try_read_lock() request from  %s\n", source);
    gboolean result = rfm_rw_lock_reader_trylock(&(view_p->mutexes.population_rwlock));

    if (!result){
	FILTER DBG("~~~ rfm_population_try_reader_lock() NOT obtained for  %s\n", 
                source);
        return FALSE;
    }

    FILTER DBG("ooo rfm_population_try_reader_lock() obtained for  %s\n", source);

    gint obtained_serial = view_p->flags.population_serial;
    if (obtained_serial != population_serial) {
	rfm_rw_lock_reader_unlock(&(view_p->mutexes.population_rwlock));
	return FALSE;
    }
#ifdef LOCK_TRACE
    inc_RW(TRUE, source);
#endif

    return TRUE;
}
//#undef LOCK_TRACE

static gint thread_count = 0;

static void
inc_dec_view_ref(view_t *view_p, gboolean increment){
    RfmRWLock *lock;
    rfm_global_t *rfm_global_p = rfm_global();
    if (view_p) {
	lock = &(view_p->mutexes.view_lock);
    } else {
	if (!rfm_global_p) {
	    TRACE("increment/decrement view_ref(): invalid view_p (rfm_global_p==NULL)\n");
	    return; 
	}
	NOOP( "*+*+ global window lock\n");
	lock = &(rfm_global_p->window_lock);
    } 
    // XXX since this is main thread business (as well
    //     as child thread), reader lock may block...
    //     but this is very rare condition.
    if (increment){
	if (rfm_get_gtk_thread() == g_thread_self()){
	    while (!rfm_rw_lock_reader_trylock(lock)) gtk_main_iteration();
	} else 	rfm_rw_lock_reader_lock(lock);
    }
    else rfm_rw_lock_reader_unlock(lock);
    static pthread_mutex_t inc_dec_mutex=PTHREAD_MUTEX_INITIALIZER;

    pthread_mutex_lock(&inc_dec_mutex);
    if (increment) {
	if (view_p) view_p->flags.thread_count++;
	thread_count++;
    } else {
 	if (view_p) view_p->flags.thread_count--;
	thread_count--;
    }
    gint view_count = 1;
    if (view_p) view_count = view_p->flags.thread_count;
    if (thread_count == 0 || view_count == 0){
	// Janitor signal (This wakes up the janitor).
	TRACE("thread unreference sending janitor the signal. (%d,%d)\n",
		thread_count, view_count);
	g_cond_signal(rfm_global_p->janitor_signal);
    }
    pthread_mutex_unlock(&inc_dec_mutex);
}

static void *
wait_f(gpointer data){
    void **argv =data;
    view_t *view_p = argv[0];
    GThread *thread = argv[1];
    g_free(argv);
    g_thread_join(thread);
    rfm_thread_unreference(view_p, thread);
    return NULL;
}

#ifdef DEBUG_TRACE
#define DEBUG_THREADS
#endif

#ifdef DEBUG_THREADS
static GSList *view_ref_list = NULL;
static pthread_mutex_t *
get_view_ref_mutex(void){
    static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
    return &mutex;
}

void
rfm_thread_unreference_quiet(view_t *view_p, GThread *thread){
    if (thread) inc_dec_view_ref(view_p, FALSE);
}
#else
void
rfm_thread_unreference_quiet(view_t *view_p, GThread *thread){
    return rfm_thread_unreference(view_p, thread);
}
#endif


void
rfm_thread_unreference(view_t *view_p, GThread *thread){
    if (thread) inc_dec_view_ref(view_p, FALSE);
#ifdef DEBUG_THREADS
    pthread_mutex_t *view_ref_mutex=get_view_ref_mutex();
    pthread_mutex_lock(view_ref_mutex);
    const gchar *removed_text = get_thread_text(thread);

    TRACE( "- view decrement: 0x%x (%s) view ref = %d\n", 
	    GPOINTER_TO_INT(thread), removed_text,
	    g_slist_length(view_ref_list)-1);
	
    view_ref_list = g_slist_remove(view_ref_list, thread);
    g_hash_table_remove(thread_hash, thread);

#ifdef DEBUG_TRACE
    gchar *trace_text = NULL;
    
    GSList *t = view_ref_list;
    if (g_slist_length(view_ref_list)) for(;t && t->data; t=t->next){
	gchar *dbg_text = g_hash_table_lookup(thread_hash, t->data);
	if (dbg_text) {
	    gchar *g =
		g_strdup_printf("%s 0x%x (%s)", 
			trace_text?trace_text:"- decrement pending:",
			GPOINTER_TO_INT(t->data), dbg_text);
	    g_free(trace_text);
	    trace_text = g;
	}
    } else {
	trace_text = g_strdup("- view decrement: no threads pending");
    }
    if (trace_text) {
	TRACE( "%s (%d)\n", trace_text, g_slist_length(view_ref_list));
	g_free(trace_text);
    }
#endif
    pthread_mutex_unlock(view_ref_mutex);

#endif
}

void
rfm_thread_reference(view_t *view_p, GThread *thread, const gchar *dbg_text){
	if (thread) inc_dec_view_ref(view_p, TRUE);
#ifdef DEBUG_THREADS
	if (dbg_text){
	    TRACE( "- view increment: 0x%x:0x%x (%s) view ref = %d\n",
		    GPOINTER_TO_INT(view_p), GPOINTER_TO_INT(thread), dbg_text, g_slist_length(view_ref_list)+1);
	    pthread_mutex_t *view_ref_mutex=get_view_ref_mutex();
	    pthread_mutex_lock(view_ref_mutex);
	    if (!thread_hash){
		thread_hash = g_hash_table_new(g_direct_hash, g_direct_equal);
	    }
	    view_ref_list = g_slist_append(view_ref_list, thread);
	    g_hash_table_replace(thread_hash, thread, (void *)dbg_text);
	    pthread_mutex_unlock(view_ref_mutex);
	}
#endif
}

// If view is NULL, the lock is on the window, not any particular view.
GThread *
rfm_view_thread_create(
	view_t *view_p,
	gpointer(*thread_f)(gpointer), gpointer data, 
	const gchar *dbg_text){
    rfm_global_t *rfm_global_p = rfm_global();

    if (rfm_global_p){
        g_mutex_lock(rfm_global_p->status_mutex);
        gint status = rfm_global_p->status;
        g_mutex_unlock(rfm_global_p->status_mutex);
	if (status == STATUS_EXIT) {
	    DBG("rfm_view_thread_create: rfm_global_p->status == STATUS_EXIT\n");
	    return NULL;
	}
    }
    
    if (view_p){
        TRACE("rfm_view_thread_create()...\n");
        if (!rfm_view_list_lock(view_p, "rfm_view_thread_create")) {
            return NULL;
        }
	gint status = view_p->flags.status;
	if (status == STATUS_EXIT) {
            rfm_view_list_unlock("rfm_view_thread_create");
	    return NULL;	
	}
    }

#ifdef DEBUG_TRACE
    if (dbg_text){
	TRACE("Thread: %s\n", dbg_text);
    }
#endif
    GThread *thread = rfm_thread_create (dbg_text, thread_f, data, TRUE);
    if (thread){
	rfm_thread_reference(view_p, thread, dbg_text);
	void **arg = (void **)malloc(2*sizeof(void *));
	arg[0] = view_p;
	arg[1] = thread;
	rfm_thread_create ("wait_f", wait_f, arg, FALSE);
    }
    if (view_p) rfm_view_list_unlock("rfm_view_thread_create");
    return thread;
}

///////////////////////////////////////////////////////////////////
// Main thread threadpool queue creation

GThreadPool *
rfm_thread_queue_new(void (*signal_pool_f)(void *, void *), void *pool_data, gint max_threads){
    GThreadPool *thread_pool;
    GError *error = NULL;
    thread_pool = g_thread_pool_new (signal_pool_f,
               pool_data, max_threads, TRUE, &error);
    if (error){
	g_error("g_thread_pool_new: %s\n", error->message);
    }
   // g_thread_pool_set_max_idle_time (miliseconds);	
    return thread_pool;
}

gboolean
rfm_threadqueue_push(GThreadPool *thread_pool, gint signal_enum, view_t *view_p, void *data){
    rfm_global_t *rfm_global_p = rfm_global();
    if (!thread_pool) {
	DBG("rfm_global_p->thread_queue not yet initialized.\n");
	return FALSE;
    }
    g_mutex_lock(rfm_global_p->status_mutex);
    gint status = rfm_global_p->status;
    g_mutex_unlock(rfm_global_p->status_mutex);
    if (status == STATUS_EXIT) return FALSE;
    
    void **arg = (void **)malloc(3*sizeof(void *));
    if (!arg) g_error("malloc: %s\n", strerror(errno));
    arg[0] = GINT_TO_POINTER(signal_enum);
    arg[1] = view_p;
    arg[2] = data;
	
    NOOP(stderr, "rfm_threadqueue_push: g_thread_pool_push, signal %d\n", signal_enum);
    g_thread_pool_push(thread_pool, arg, NULL);
    return TRUE;
}

#define PASTE_SHM_NAME paste_shm_name()

static const gchar*
paste_shm_name(void){
    static const gchar *name=NULL;
    if (!name){
	name = g_strdup_printf("/%d-rfm-pasteboard", (gint)geteuid());
    }
    return name;
}

gchar *
rfm_get_hash_key (const gchar * key, gint size) {
    gchar *hash_key = NULL;
    GString *gs = g_string_new (key);
    if (size <=0) {
	hash_key = g_strdup_printf ("%010u", g_string_hash (gs));
    } else {
	gint usize = 999;
	if (size <= 999) usize = size;
	hash_key = g_strdup_printf ("%010u-%d", g_string_hash (gs), usize);
    }
    g_string_free (gs, TRUE);
    NOOP("%s: hashkey=%s\n", key, hash_key);
    return hash_key;
}


void
rfm_set_widget (void *widget, const gchar * name){
    if(!name) g_error ("rfm_set_widget invalid call");
    rfm_global_t *rfm_global_p = rfm_global();
    if (!rfm_global_p || !rfm_global_p->window){
	NOOP("rfm_set_widget invalid parameters\n");
	return;
    }
    g_object_set_data (G_OBJECT (rfm_global_p->window), name, widget);
    NOOP(stderr, "set %s: 0x%x\n", name, GPOINTER_TO_INT(widget));
    return;
}

void *
rfm_get_widget (const gchar * name) {
    if(!name) {
	DBG ("rfm_get_widget: !name\n");
	return NULL;
    }
    rfm_global_t *rfm_global_p = rfm_global();
    if (!rfm_global_p || !rfm_global_p->window){
	g_error("invalid call to rfm_get_widget: %s\n", name);
    }
    void *pointer = g_object_get_data (G_OBJECT (rfm_global_p->window), name);
    if(!pointer) {
	// This may be not set yet, and may not indicate any error.
	NOOP("Cannot find widget (or pointer) associated to \"%s\"\n", name);
	TRACE("Cannot find widget (or pointer) associated to \"%s\"\n", name);
    }
    return pointer;

}

gchar *
rfm_esc_string (const gchar * string) {
    int i,
      j,
      k;
    char *charset = "\\\"\' ()|<>";

    for(j = 0, i = 0; i < strlen (string); i++) {
        for(k = 0; k < strlen (charset); k++) {
            if(string[i] == charset[k])
                j++;
        }
    }
    gchar *estring = (gchar *) malloc (strlen (string) + j + 1);
    memset (estring, 0, strlen (string) + j + 1);
    for(j = 0, i = 0; i < strlen (string); i++, j++) {
        for(k = 0; k < strlen (charset); k++) {
            if(string[i] == charset[k])
                estring[j++] = '\\';
        }
        estring[j] = string[i];
    }
    NOOP ("ESC:estring=%s\n", estring);
    return estring;
}

#define MAX_PATH_LABEL 40
#define MAX_PATH_START_LABEL 18
const gchar *
rfm_chop_excess (gchar * b) {
    // chop excess length...

    const gchar *home = g_get_home_dir();
    gchar *bb;
    if (strncmp(home, b, strlen(home))==0){
        if (strlen(home) == strlen(b)) return b;
        bb = g_strconcat("~/", b + strlen(home)+1, NULL);
    } else {
        bb = g_strdup(b);
    }
    
    int len = strlen (bb);

    if(len < MAX_PATH_LABEL) {
        strcpy(b, bb);
        g_free(bb);
        return (b);
    }
        
    bb[MAX_PATH_START_LABEL - 3] = 0;

    gchar *g = g_strconcat(bb, "...", b + (len - MAX_PATH_LABEL + MAX_PATH_START_LABEL), NULL);
    strcpy (b, g);
    g_free(bb);
    g_free(g);

    return b;
}

/*
 * This function converts a time value specified in seconds since 1970-01-01
 * to a string representation. It shows the date and the time if the point
 * if less then six month in the past and only the date otherwise.
 * The exact format must be provided by a translation string. 
 *
 * The function should be thread-save since there are not used static
 * (or even global) variables if the system provided a localtime_r() function.
 *
 * Arguments:     when:    time in seconds since 1970-01-01
 *                string:  the result char-array in which the string is placed
 *                length:  the length of the string-array
 *                
 * Return value:  string on success, NULL on failure
 */
gchar *
rfm_time_to_string (time_t when) {
    time_t now = time (NULL);
#ifdef HAVE_LOCALTIME_R
    struct tm timestruct;
#endif
    struct tm *timestruct_ptr;
    char *formatstring;
    gchar *s = NULL;
    gchar string[64];

#ifdef HAVE_MEMSET
    memset (string, 0, 64);
#else
    string[0] = 0;
#endif

    formatstring = difftime (now, when) > 24 * 60 * 60 * 30 * 6
        /* strftime format for non-recent files (older than 6 months)  */
        ? _("%b %e  %Y")
        /* strftime format for recent files */
        : _("%b %e %H:%M");

#ifdef HAVE_LOCALTIME_R
    timestruct_ptr = &timestruct;
    localtime_r (&when, timestruct_ptr);
#else
    timestruct_ptr = localtime (&when);
#endif

    if(strftime (string, 64, formatstring, localtime (&when)) == 0) {
        return NULL;
    }
    s = rfm_utf_string (string);
    return s;
}

gchar *
rfm_mode_string (mode_t mode) {
    return mode_string (mode);
}
    
static pthread_mutex_t user_string_mutex=PTHREAD_MUTEX_INITIALIZER;

gchar *
rfm_user_string (struct stat *st) {
    pthread_mutex_lock(&user_string_mutex);
    struct passwd *p;
    gchar *user_string;
    if((p = getpwuid (st->st_uid)) != NULL)
            user_string = g_strdup(p->pw_name);
        else if((gint)st->st_uid < 0)
            user_string = g_strdup("");
        else
            user_string = g_strdup_printf("%d", (gint)st->st_uid);
    pthread_mutex_unlock(&user_string_mutex);
    return user_string;
}

static pthread_mutex_t group_string_mutex=PTHREAD_MUTEX_INITIALIZER;

gchar *
rfm_group_string (struct stat *st) {
    pthread_mutex_lock(&group_string_mutex);
    struct group *g;
    gchar *group_string;
    if((g =  getgrgid(st->st_gid)) != NULL)
            group_string = g_strdup(g->gr_name);
    else
        group_string = g_strdup_printf("%d", (gint)st->st_gid);
    pthread_mutex_unlock(&group_string_mutex);
    return group_string;
}

static pthread_mutex_t date_string_mutex=PTHREAD_MUTEX_INITIALIZER;
gchar *
rfm_date_string (time_t the_time) {
    pthread_mutex_lock(&date_string_mutex);

#ifdef HAVE_LOCALTIME_R
        struct tm t_r;
#endif
        struct tm *t;

#ifdef HAVE_LOCALTIME_R
        t = localtime_r (&the_time, &t_r);
#else
        t = localtime (&the_time);
#endif
        gchar *date_string=
	    g_strdup_printf ("%04d/%02d/%02d  %02d:%02d", t->tm_year + 1900,
                 t->tm_mon + 1, t->tm_mday, t->tm_hour, t->tm_min);
    pthread_mutex_unlock(&date_string_mutex);

    return date_string;
}

gchar *
rfm_utf_string (const gchar * t) {
    if(!t) {
	NOOP("rfm_utf_string(): string == NULL!\n");
        return g_strdup ("");
    }

    if(g_utf8_validate (t, -1, NULL)) {
        return g_strdup (t);
    }
    /* so we got a non-UTF-8 */
    /* but is it a valid locale string? */
    gchar *actual_tag;
    actual_tag = g_locale_to_utf8 (t, -1, NULL, NULL, NULL);
    if(actual_tag)
        return actual_tag;
    /* So it is not even a valid locale string... 
     * Let us get valid utf-8 caracters then: */
    const gchar *p;
    actual_tag = g_strdup ("");
    for(p = t; *p; p++) {
        // short circuit end of string:
        gchar *r = g_locale_to_utf8 (p, -1, NULL, NULL, NULL);
        if(g_utf8_validate (p, -1, NULL)) {
            gchar *qq = g_strconcat (actual_tag, p, NULL);
            g_free (actual_tag);
            actual_tag = qq;
            break;
        } else if(r) {
            gchar *qq = g_strconcat (actual_tag, r, NULL);
            g_free (r);
            g_free (actual_tag);
            actual_tag = qq;
            break;
        }
        // convert caracter to utf-8 valid.
        gunichar gu = g_utf8_get_char_validated (p, 2);
        if(gu == (gunichar) - 1) {
            //DBG("gu=%d\n",(int)gu);
            gu = g_utf8_get_char_validated ("?", -1);
        }
        gchar outbuf[8];
        memset (outbuf, 0, 8);
        gint outbuf_len = g_unichar_to_utf8 (gu, outbuf);
        if(outbuf_len < 0) {
            DBG ("unichar=%d char =%c outbuf_len=%d\n", gu, p[0], outbuf_len);
        }
        gchar *qq = g_strconcat (actual_tag, outbuf, NULL);
        g_free (actual_tag);
        actual_tag = qq;
    }
    return actual_tag;
}

gchar *
rfm_sizetag (off_t tama, gint count) {
    gchar *tag = _("bytes");
    gchar *buf = NULL;
    double utama = tama;

    buf = NULL;
    if(utama > 0) {
        if(utama >= (off_t)1000 * 1000 * 1000) {
            utama /= ((off_t)1000 * 1000 * 1000);
            tag = _("Gigabytes");
        } else if(utama >= 1000 * 1000) {
            utama /= (1000 * 1000);
            tag = _("Megabytes");
        } else if(utama >= 1000) {
            utama /= 1000;
            tag = _("Kilobytes");
        }
        if(count <= 0) {
            /* format for size column of regular files */
            buf = g_strdup_printf ("%.2lf %s", utama, tag);
        } else {
            gchar *plural_text=
                g_strdup_printf (ngettext ("%'u item", "%'u items", 
                            count),count);
	    if (tama < 1000) {
		buf = g_strdup_printf ("%s: %.0lf %s.", plural_text,
                    utama, tag);
	    } else {
		buf = g_strdup_printf ("%s: %.2lf %s.", plural_text,
                    utama, tag);
	    }
            g_free(plural_text);
    
        }
    } else {
        if(count <=0) {
            buf = g_strdup_printf (_("The location is empty."));
        } else {
            buf=
                g_strdup_printf (ngettext ("%'u item", "%'u items", count),
                        count);
        }
    }
    return buf;
}

/**
 * rfm_host_name:
 * @xid: Window X id
 *
 * This gets the hostname of the window client. This is different from
 * #g_get_host_name() which gets the name of the window server. This may be 
 * different when the window is displayed on a remote server.
 *
 **/
gchar *
rfm_host_name (Window xid) {
    char *name = NULL;
    unsigned char *property_data;
    unsigned long items,
      remaining;
    int actual_format;
    Atom atomo,
      actual_atom;
    if (rfm_get_gtk_thread() != g_thread_self())
	g_error("rfm_host_name(): only to be called by gtk thread\n");
    rfm_global_t *rfm_global_p = rfm_global();
    if (!rfm_global_p) return "localhost";
    atomo = XInternAtom (rfm_global_p->Xdisplay, "WM_CLIENT_MACHINE", FALSE);
    if(XGetWindowProperty (rfm_global_p->Xdisplay, 
		xid,
                atomo, 0, 255, FALSE, XA_STRING,
                &actual_atom, &actual_format, &items, 
		&remaining, &property_data) == Success) {
        NOOP ("property_data=%s", ((property_data) ? property_data : (unsigned char *)"null"));
        if(!property_data)
            name = g_strdup (g_get_host_name ());
        else {
            name = g_strdup ((const gchar *)property_data);
            XFree (property_data);
        }
    } else
        name = g_strdup (g_get_host_name ());
    return name;
}

gint
rfm_mkdir (const gchar * dir) {
    if(rfm_g_file_test (dir, G_FILE_TEST_EXISTS)) {
        gchar *message;
        if(!rfm_g_file_test (dir, G_FILE_TEST_IS_DIR)) {
            message = g_strdup_printf ("%s: %s (ENOTDIR: %s)", dir, strerror (EEXIST), strerror (ENOTDIR));
        } else {
            message = g_strdup_printf ("%s: %s", dir, strerror (EEXIST));
        }
        rfm_confirm (NULL, GTK_MESSAGE_ERROR, message, NULL, NULL);
        g_free (message);
        return 0;
    }
    if(g_mkdir_with_parents (dir, 0700) < 0) {
        //DBG("cannot create %s: %s\n", dir, strerror(errno));
    }
    if(rfm_g_file_test (dir, G_FILE_TEST_EXISTS)) {
        if(rfm_g_file_test (dir, G_FILE_TEST_IS_DIR)) {
            return 0;
        }
    }
    NOOP ("!rfm_g_file_test(%s, G_FILE_TEST_IS_DIR\n", dir);
    return -1;
}

void
rfm_save_to_go_history (char *p) {
    gchar *f = g_build_filename (GOTO_DBH_FILE, NULL);
    COMBOBOX_save_to_history (f, p);
    g_free (f);
}

void
rfm_init_env (void) {
    gint i;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
      environ_t *environ_v = rfm_get_environ();
      // XXX: This initialization may now be obsolete:
      //      must check.
      for(i = 0; environ_v[i].env_var; i++) {
        // copy default values from static memory to heap
        if(strcmp (environ_v[i].env_var, "SUDO_ASKPASS") == 0 ||
	    strcmp(environ_v[i].env_var, "SSH_ASKPASS") == 0 ) {
            environ_v[i].env_string = g_find_program_in_path ("rodent-getpass");
	} 
	else if(environ_v[i].env_string) {
            environ_v[i].env_string = g_strdup (environ_v[i].env_string);
            NOOP ("ENV: %s %s\n", environ_v[i].env_var, environ_v[i].env_string);
        }
      }
      g_once_init_leave (&initialized, 1);
    }
    return;
}


void
rfm_setenv (const gchar *name, gchar *value, gboolean verbose) {
    rfm_init_env();
    NOOP(stderr, "setting %s to %s\n", name, value);
    gint which;
    gboolean go_ahead = FALSE;
    environ_t *environ_v = rfm_get_environ();
    for(which = 0; environ_v[which].env_var; which++){
        if(strcmp (name, environ_v[which].env_var) == 0){
            break;
	}
    }
    if(!environ_v[which].env_var) return;
    if(!value || !strlen (value)) {
        g_free (environ_v[which].env_string);
#ifdef HAVE_UNSETENV
        environ_v[which].env_string = NULL;
        unsetenv (name);
#else
        environ_v[which].env_string = g_strconcat (name, "=", NULL);
        putenv (environ_v[which].env_string);
#endif
        /*if(verbose) {
            if(strcmp (name, "SMB_USER") == 0) {
                TRACE("Mcs manager changed rfm environment: %s\n", name);
            } else {
                TRACE("Mcs manager changed rfm environment: %s=%s\n", name, ((value) ? value : " "));
            }
        }*/
        return;
    }
    if(strcmp (name, "RFM_MAX_PREVIEW_SIZE") == 0) {
        if(is_number (value))
            go_ahead = TRUE;
    } else if(strcmp (name, "TERMCMD") == 0) {
        if(value && strlen (value)) {
            gchar *c;
            gchar *t = g_strdup (value);
            t = g_strstrip (t);
            if(strchr (t, ' '))
                t = strtok (t, " ");
            c = g_find_program_in_path (t);
            if(c && access (c, X_OK) == 0) {
                go_ahead = TRUE;
            }
            g_free (c);
            c = NULL;
            g_free (t);
            t = NULL;
        }
    } else
        go_ahead = TRUE;
    if(go_ahead) {
        g_free (environ_v[which].env_string);
	gchar *getpass = NULL;
	gchar *editor = NULL;
	if (strcmp (name, "EDITOR") == 0){
	    editor = rfm_get_text_editor_envar(value);
	    NOOP(stderr, "Setting text editor to %s\n", editor);
	}
	if (editor){
	    value = editor;
	} else {
	    if (strcmp (name, "SUDO_ASKPASS") == 0 ||
		strcmp(name, "SSH_ASKPASS") == 0 ){
		if (!g_file_test(value, G_FILE_TEST_EXISTS)){
		    getpass = g_find_program_in_path ("rodent-getpass");
		}
	    }
	}
	if (getpass) value = getpass;


	NOOP(stderr, "rfm_setenv(): setting %s -> %s \n", name, value);
#ifdef HAVE_SETENV
        environ_v[which].env_string = g_strdup (value);
        setenv (name, environ_v[which].env_string, 1);
#else
        environ_v[which].env_string = g_strconcat (name, "=", value, NULL);
        putenv (environ_v[which].env_string);
#endif
	g_free(editor);
    } else {                    /* not go_ahead */
        DBG ("failed to change rfm environment: %s\n", name);
    }
    return;
}

void
rfm_threadwait (void) {
    struct timespec thread_wait = {
        0, 100000000
    };
    nanosleep (&thread_wait, NULL);
}

#ifdef DEBUG_NOOP
# include <sys/times.h>
static struct tms *last_clock = NULL;

const gchar *
profile (void) {
    struct tms this_clock;
    static gchar *p = NULL;
    g_free (p);
    if(!last_clock) {
        last_clock = (struct tms *)malloc (sizeof (struct tms));
        times (last_clock);
    }
    times (&this_clock);
    p = g_strdup_printf ("\n\tPROFILE*** user=%ld, system=%ld",
                         (long)(this_clock.tms_utime - last_clock->tms_utime),
                         (long)(this_clock.tms_stime - last_clock->tms_stime));
    memcpy (last_clock, &this_clock, sizeof (struct tms));

    return (const gchar *)p;
}
#endif
static gboolean
rect_OK(view_t * view_p, GdkRectangle * rect_p){
    // nope return TRUE;
    if (view_p->flags.type == DESKVIEW_TYPE) return TRUE;
    GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");
    if (!scrolled_window || !GTK_IS_SCROLLED_WINDOW(scrolled_window)) return FALSE;
    if (!GTK_IS_ADJUSTMENT(
	gtk_scrolled_window_get_vadjustment (scrolled_window)
	)) return TRUE;
    g_mutex_lock(view_p->mutexes.status_mutex);
    gint  status = view_p->flags.status;
    g_mutex_unlock(view_p->mutexes.status_mutex);
    if (status == STATUS_EXIT) return FALSE;

    gdouble position=gtk_adjustment_get_value (
	    gtk_scrolled_window_get_vadjustment (scrolled_window));
    gdouble page=gtk_adjustment_get_page_size (
	    gtk_scrolled_window_get_vadjustment (scrolled_window));
    if (rect_p->y >= position && rect_p->y <= position + page + 0.9){
	return TRUE;
    }
    if (rect_p->y + rect_p->height >= position &&
	    rect_p->y + rect_p->height <= position + page + 0.9){
	return TRUE;
    }
    NOOP(stderr, "y=(%d,%d), page=%lf position=%lf\n", 
	    rect_p->y, rect_p->y + rect_p->height,
	    page, position);
    return FALSE;
}


typedef struct expose_rect_t {
    view_t *view_p;
    GdkRectangle rect;
} expose_rect_t;

static void
expose_it(expose_rect_t *expose_rect_p){
     NOOP(stderr, "expose_it %d,%d %d,%d --\n", 
	     expose_rect_p->rect.x, expose_rect_p->rect.y, 
	     expose_rect_p->rect.width, expose_rect_p->rect.height);
    GdkWindow * window;
    window = gtk_widget_get_window(expose_rect_p->view_p->widgets.paper);
    gdk_window_invalidate_rect (window, &(expose_rect_p->rect), TRUE);
}

// main context function
static gboolean
expose_rect_f(gpointer data){
    expose_rect_t *expose_rect_p = data;
    if (rfm_main_context_check(expose_rect_p->view_p, FALSE)){
	expose_it(expose_rect_p);
    }
    g_free(data);
    return FALSE;
}


void
rfm_expose_rect (view_t * view_p, const GdkRectangle *rect_p) {
    TRACE("rfm_expose_rect()...\n");
    if (!rfm_view_list_lock(view_p, "rfm_expose_rect")) return;
    if (rfm_get_gtk_thread() == g_thread_self()) {
	expose_rect_t expose_rect_v;
	expose_rect_v.view_p = view_p;
	memcpy(&(expose_rect_v.rect), rect_p, sizeof(GdkRectangle));
	expose_it(&expose_rect_v);
        rfm_view_list_unlock("rfm_expose_rect");
	return;
    }
    expose_rect_t *expose_rect_p = (expose_rect_t *)malloc(sizeof(expose_rect_t));
    if (!expose_rect_p) g_error("malloc: %s\n", strerror(errno));
    expose_rect_p->view_p = view_p;
    memcpy(&(expose_rect_p->rect), rect_p, sizeof(GdkRectangle));
    // don't wait here
    g_main_context_invoke (NULL, expose_rect_f, expose_rect_p);
    rfm_view_list_unlock("rfm_expose_rect");
    return;
}


// non threaded:

// the  +5 refers to the red outline and the cute shadow and is less
// than the empty space between icon and text (< TEXTSPACING) 
void
rfm_expose_icon (view_t * view_p, const population_t * population_p) {
    NOOP (">> rfm_expose_icon\n");
    GdkRectangle rect;
    if (!population_p) return;
    if (!rfm_get_population_rect(view_p, population_p, &rect))return;
    gint icon_size = ICON_SIZE(view_p);
    if (icon_size >= SMALL_ICON_SIZE) {
	rect.width = CELLWIDTH(view_p);
	rect.height = icon_size + 5;
    } else  if (icon_size >= TINY_ICON_SIZE){
	rect.width = rect.height = icon_size+2;
    }
    else {
	rect.width = rect.height = TINY_ICON_SIZE+2;

    }

    if (!rect_OK(view_p, &rect)){
	NOOP(stderr, "icon for %s is out of sight!\n",
		(population_p->en)?population_p->en->path:"null");
	return;
    }
    rfm_expose_rect(view_p, &rect);
    //gdk_window_invalidate_rect (gtk_widget_get_window(view_p->widgets.paper), &rect, TRUE);
    return;
}

// This should only be for saturated case, otherwise a plain expose item.
void
rfm_expose_label (view_t * view_p, const population_t * population_p) {
    if (!population_p) return;
    NOOP (">> rfm_expose_label (full)\n");
    GdkRectangle rect;

    if (ICON_SIZE_ID(view_p) == 0) {
 	if (!rfm_get_population_rect(view_p, population_p, &rect)){
	    return;
	}

   } else {
	if (!rfm_get_population_label_rect_full(view_p, population_p, &rect)){
	    return;
	}
	if (ICON_SIZE(view_p) >= SMALL_ICON_SIZE){
	    // I wonder where the 2 comes from (required).
	    rect.height = CELLHEIGHT(view_p) - (TEXTSPACING + ICON_SIZE(view_p) + 2);
	}
    }

    // Expose double height to erase non saturated text (hack!)
    // rect.height *= 2;

    if (!rect_OK(view_p, &rect)){
	TRACE( "label for %s is out of sight!\n",
		(population_p->en)?population_p->en->path:"null");
	return;
    }
    NOOP(stderr, "expose label: %s x: %d-%d y: %d-%d\n", 
	    population_p->label,
	    rect.x, rect.x+rect.width, rect.y, rect.y+rect.height);
    rfm_expose_rect(view_p, &rect);
    //gdk_window_invalidate_rect (gtk_widget_get_window(view_p->widgets.paper), &rect, TRUE);
    return;
}

void
rfm_expose_item (view_t * view_p, const population_t * population_p) {
    GdkRectangle rect;
    if (!rfm_get_population_rect(view_p, population_p, &rect)) return;
    NOOP ("1>> rfm_expose_item %d,%d %d,%d\n", rect.x, rect.y, rect.width, rect.height);
    NOOP ("2>> rfm_expose_item %d,%d %d,%d\n", rect.x, rect.y, rect.width, rect.height);
    if (!rect_OK(view_p, &rect)){
	NOOP(stderr, "item for %s is out of sight!\n",
		(population_p->en)?population_p->en->path:"null");
	NOOP (stderr, "2>> rfm_expose_item %d,%d %d,%d\n", rect.x, rect.y, rect.width, rect.height);
	return;
    }
    rfm_expose_rect(view_p, &rect);
    //gdk_window_invalidate_rect (gtk_widget_get_window(view_p->widgets.paper), &rect, TRUE);
    return;
}

// client based pasteboard (with MIT-shm)
// no interclient communication, but then again,
// I really never found any use for it 


void
rfm_clear_paste_buffer(void){
    rfm_rw_lock_writer_lock(&drag_info_lock);
    shm_unlink (PASTE_SHM_NAME);
    rfm_rw_lock_writer_unlock(&drag_info_lock);
}

void
rfm_store_paste_buffer(gchar *buffer, gint len){ 
    rfm_rw_lock_writer_lock(&drag_info_lock);

    // Remove old MIT-shm  pasteboard.
    shm_unlink (PASTE_SHM_NAME);
    
    gint fd = shm_open (PASTE_SHM_NAME, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR);
    if(fd < 0){
        g_error ("rfm_store_paste_buffer(): shm_open(%s): %s", PASTE_SHM_NAME, strerror (errno));
    }
    // Get writelock .

    // Truncate to necessary memory size to allocate.
    if(ftruncate (fd, sizeof(gint)+strlen(buffer)+1) < 0) {
        g_error ("rfm_store_paste_buffer(): ftruncate(%s): %s", PASTE_SHM_NAME, strerror (errno));
    }

    // Get a shared memory pointer.
    void *p = mmap (NULL, sizeof(gint)+strlen(buffer)+1, 
	    PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
   // Close file descriptor
    close(fd);
    // Save size as first sizeof(gint) bytes.
    gint *len_p = p;
    *len_p=sizeof(gint)+strlen(buffer)+1;
    // Save text buffer now.
    gchar *buffer_p = p + sizeof(gint);
    strcpy(buffer_p, buffer);
    // Put in shared memory.
    if(msync (p, sizeof(gint)+strlen(buffer)+1, MS_SYNC) < 0){
        DBG ("rfm_store_paste_buffer(): msync(%s): %s\n", PASTE_SHM_NAME, strerror (errno));
    }
    // Release map so other processes may shm_unlink.
    munmap (p, sizeof(gint)+strlen(buffer)+1);

 
    // release writelock on shm_name file descriptor
    rfm_rw_lock_writer_unlock(&drag_info_lock);
}

static 
gint get_paste_length(){
    gint fd = shm_open (PASTE_SHM_NAME, O_RDONLY, S_IRUSR | S_IWUSR);
    if(fd < 0){
	return 0;
    }
    // Get readlock on shm_name file descriptor.
    rfm_rw_lock_reader_lock(&drag_info_lock);

    // Figure out the size.
    void *p = mmap (NULL, sizeof(gint), 
    //void *p = mmap (NULL, 133, 
	    PROT_READ, MAP_SHARED, fd, 0);
    close(fd);

    gint len = *((gint *)p);
    if(msync (p, sizeof(gint), MS_SYNC) < 0){
        DBG ("msync(%s): %s\n", PASTE_SHM_NAME, strerror (errno));
    }
    munmap (p, sizeof(gint));
    // release writelock on shm_name file descriptor
    rfm_rw_lock_reader_unlock(&drag_info_lock);
    return len;
}


gchar *
rfm_get_paste_buffer (void ) {
    // Get readlock on shm_name file descriptor.
    rfm_rw_lock_reader_lock(&drag_info_lock);
    gint len=get_paste_length();
    if (len==0) {
	rfm_rw_lock_reader_unlock(&drag_info_lock);
	return NULL;
    }
    int fd = shm_open (PASTE_SHM_NAME, O_RDONLY, S_IRUSR | S_IWUSR);
    if(fd < 0){
	rfm_rw_lock_reader_unlock(&drag_info_lock);
	return NULL;
    }


    void *pp = mmap (NULL, len, PROT_READ, MAP_SHARED, fd, 0);
    gchar *buffer_p = pp + sizeof(gint);
    gchar *buffer = g_strdup(buffer_p);
    munmap (pp, len);
    // release writelock on shm_name file descriptor
    close(fd);
    rfm_rw_lock_reader_unlock(&drag_info_lock);
    return buffer;
}


int
rfm_pasteboard_status (view_t * view_p) {
    if(!view_p) return 0;
    // This is cool with client side pasteboard because
    // nobody is going to mess with pasteboard.
    // With server side, this is not so, because somebody
    // might mess it up and rodent will still think the pasteboard
    // is valid...

    // This crashes with multithread server side pasteboard:
    rfm_update_pasteboard (view_p);

    gchar *b = view_p->xbuffer;
    if(!b || !strlen (b)) return 0;
    
    const gchar *cut = "#xfvalid_buffer:cut";
    const gchar *copy = "#xfvalid_buffer:copy";
    if(strncmp (b, copy, strlen (copy)) == 0)
        return 1;
    if(strncmp (b, cut, strlen (cut)) == 0)
        return 2;
    return 0;
}

gchar **
rfm_pasteboard_v(view_t * view_p){
    if (!rfm_pasteboard_status (view_p)) return NULL;
    // this is to skip the valid buffer line:
    gchar *search = strchr (view_p->xbuffer, '\n');
    if(!search)return NULL;
    search++;
    gchar **v = g_strsplit(search, "\n", -1);
    return v;
}


/* returns 0 if not in pasteboard, 1 if in copy pasteboard or 2 if
 * in cut pasteboard */
int
rfm_in_pasteboard (view_t * view_p, record_entry_t * en) {
    if(!en || !en->path) return FALSE;
    if(IS_ROOT_TYPE (en->type) && !IS_SDIR(en->type)) return FALSE;

    gchar **pasteboard_v = rfm_pasteboard_v(view_p);
    gchar **p = pasteboard_v;
    gint retval=0;
    gint status = rfm_pasteboard_status (view_p);
    for(; p && *p; p++){
        if(strcmp (*p, en->path) == 0) {
            retval = status;
            break;
        }
    }
    g_strfreev(pasteboard_v);
    return retval;
#if 0
    obsolete...
    int paste_type = rfm_valid_pasteboard (view_p);
    if(!paste_type) return FALSE;
    gchar *b = g_strdup (view_p->xbuffer);

    // this is the valid buffer line:
    gchar *search = strtok (b, "\n");
    if(!search) {
        g_free (b);
        return 0;
    }
    // now we check line per line for the path.
    search = strtok (NULL, "\n");
    while(search) {
        if(strcmp (search, en->path) == 0) {
            NOOP ("PASTE type =%d %s\n", paste_type, en->path);
            g_free (b);
            return paste_type;
        }
        search = strtok (NULL, "\n");
    }
    g_free (b);
    return 0;
#endif
}

gboolean
rfm_update_pasteboard (view_t * view_p) {
    if(!view_p->xbuffer) view_p->xbuffer = rfm_get_paste_buffer ();
    gchar *current_xbuffer = rfm_get_paste_buffer ();
    if (!current_xbuffer && !view_p->xbuffer) return FALSE;
    if(!view_p->xbuffer && current_xbuffer){
        view_p->xbuffer = current_xbuffer;
        return TRUE;
    }
    if(view_p->xbuffer && !current_xbuffer){
        g_free (view_p->xbuffer);
        view_p->xbuffer = NULL;
        return TRUE;
    }
    // here both pointers are valid
    if(strcmp (current_xbuffer, view_p->xbuffer)) {
        NOOP ("XBUFFER: xbuffer has changed! %s\n", current_xbuffer);
        g_free (view_p->xbuffer);
        view_p->xbuffer = current_xbuffer;
        return TRUE;
    } else {
        NOOP ("XBUFFER: xbuffer OK!\n");
        g_free (current_xbuffer);
        return FALSE;
    }

}

////////////////////////////////////////////////////////////////////////////////////////////
 
gboolean
rfm_write_ok_path(const gchar *target_path){
    if (!target_path) return FALSE;
    if (!g_path_is_absolute(target_path)){
	DBG("rfm_write_ok_path() is FALSE: %s is not absolute!\n", 
		target_path);
	return FALSE;
    }

    gchar *dirname = NULL;
    if (rfm_g_file_test (target_path, G_FILE_TEST_IS_DIR)) {
	dirname =g_strdup(target_path);
    } else {
	dirname = g_path_get_dirname(target_path);
    }
    gboolean result = access(dirname, W_OK);
    g_free(dirname);
    return (result<0)?0:1;
}


gboolean
rfm_read_ok_path(const gchar *target_path){
    if (!target_path) return FALSE;
    if (!g_path_is_absolute(target_path)){
	DBG("rfm_read_ok_path() is FALSE: %s is not absolute!\n", 
		target_path);
	return FALSE;
    }


    gboolean result = access(target_path, R_OK);
    return (result<0)?0:1;

}

static
gchar *
default_shell(void){
    gchar *shell=NULL;
    if(!shell)
        shell = g_find_program_in_path ("bash");
    if(!shell)
        shell = g_find_program_in_path ("zsh");
    if(!shell)
        shell = g_find_program_in_path ("sh");
    if (!shell && rfm_void(PLUGIN_DIR, "ps", "module_active")) {
	shell = g_find_program_in_path ("tcsh");
	if(!shell)
	    shell = g_find_program_in_path ("csh");
    }
    if(!shell)
        shell = g_find_program_in_path ("ksh");
    if(!shell)
        shell = g_find_program_in_path ("sash");
    if(!shell)
        shell = g_find_program_in_path ("ash");
    if(!shell){
	DBG("unable to find a valid shell\n");
    }

    return shell;
}

    // dash is OK now.
    // Only csh/tcsh is broken, since it will not
    // pass on SIGTERM when controler gets SIGUSR1
    // This is only a problem if rodent_ps is not 
    // loadable.
    // gchar *
static gchar *
check_shell(gchar *shell){
    if (!shell) return NULL;
    // This is because the stop process button will not work with csh.
    if (!rfm_void(PLUGIN_DIR, "ps", "module_active") && strstr(shell, "csh")) {
	g_free(shell);
	shell = NULL;
    }
    return shell;
}

gchar *
rfm_shell(void){
    gchar *shell=NULL;
    if(getenv ("SHELL") && strlen (getenv ("SHELL"))) {
        shell = g_find_program_in_path (getenv ("SHELL"));
    }

    if(!shell && getenv ("XTERM_SHELL") && strlen (getenv ("XTERM_SHELL"))) {
        shell = g_find_program_in_path (getenv ("XTERM_SHELL"));
    }
    shell = check_shell(shell);

    if (!shell){
	shell = default_shell();
    }
    return shell;
}

gchar *
rfm_xterm_shell(void){
    gchar *shell=NULL;

    if(!shell && getenv ("XTERM_SHELL") && strlen (getenv ("XTERM_SHELL"))) {
        shell = g_find_program_in_path (getenv ("XTERM_SHELL"));
    }
    if(!shell && getenv ("SHELL") && strlen (getenv ("SHELL"))) {
        shell = g_find_program_in_path (getenv ("SHELL"));
    }
    shell = check_shell(shell);
    if (!shell){
	shell = default_shell();
    }
    return shell;
}


// Quickie test
population_t *
rfm_locate_path(view_t *view_p, const gchar *pathname){
    if (!view_p || ! view_p->en || !view_p->population_pp) return NULL;
    if (!rfm_population_read_lock(view_p, "rfm_locate_path")) return NULL;
    population_t **pp = view_p->population_pp;
    for (;pp && *pp; pp++){
	population_t *p = *pp;
	if (p->en && strcmp(p->en->path, pathname)==0){
	    rfm_population_read_unlock(view_p, "rfm_locate_path");
	    return p;
	}
    }
    rfm_population_read_unlock(view_p, "rfm_locate_path");
    return NULL;
}


/////////////////   timeout functinality  ////////////////////////////////7
//#define DISABLE_FILE_TEST_WITH_TIMEOUT 1
#ifdef DISABLE_FILE_TEST_WITH_TIMEOUT
gboolean
rfm_g_file_test_with_wait(const gchar *path, GFileTest test){
    return rfm_g_file_test(path, test);
}

#else
typedef struct heartbeat_t{
    gboolean condition;
    GMutex *mutex;
    GCond *signal;
    GThread *thread;
    gchar *path;
    GFileTest test;
} heartbeat_t;

static void *
heartbeat_g_file_test(gpointer data){
    heartbeat_t *heartbeat_p = data;

    // This function call may block
    NOOP("heartbeat doing stat %s\n", heartbeat_p->path);
    struct stat st;
    if (lstat(heartbeat_p->path, &st) < 0) return NULL;
    
    // If test is not for symlink, and item is a symlink,
    // then follow the symlink for the test.
    if (S_ISLNK(st.st_mode)){
	if (heartbeat_p->test == G_FILE_TEST_IS_SYMLINK){
	    return GINT_TO_POINTER(TRUE);
	}
	if (stat(heartbeat_p->path, &st) < 0) return NULL;
    }

    gboolean retval = FALSE;
    switch (heartbeat_p->test){
	case G_FILE_TEST_EXISTS: retval = TRUE; break;
	case G_FILE_TEST_IS_REGULAR: retval = S_ISREG(st.st_mode); break;
	case G_FILE_TEST_IS_EXECUTABLE:  
	    retval = ((st.st_mode & (S_IXUSR|S_IXGRP|S_IXOTH)) && S_ISREG(st.st_mode));
	    break;
	case G_FILE_TEST_IS_SYMLINK: retval = S_ISLNK(st.st_mode); break;
	case G_FILE_TEST_IS_DIR: retval = S_ISDIR (st.st_mode); break;

    }
    
    g_mutex_lock(heartbeat_p->mutex);
    heartbeat_p->condition = TRUE;
    g_mutex_unlock(heartbeat_p->mutex);
    NOOP("heartbeat signal %d\n", retval);
    g_cond_signal(heartbeat_p->signal);
    return GINT_TO_POINTER(retval);

}

static 
void *wait_on_thread(gpointer data){
    heartbeat_t *heartbeat_p = data;
    void *value = g_thread_join(heartbeat_p->thread);

    rfm_mutex_free(heartbeat_p->mutex);
    rfm_cond_free(heartbeat_p->signal);
    g_free (heartbeat_p->path);
    g_free (heartbeat_p);
    return value;
}

// g_file_test_with_timeout
gboolean
rfm_g_file_test_with_wait(const gchar *path, GFileTest test){
    if (!path) return FALSE;
    if (!g_path_is_absolute(path)) return FALSE;
    NOOP(stderr, "rfm_g_file_test_with_wait: %s\n", path);

    heartbeat_t *heartbeat_p = (heartbeat_t *)malloc(sizeof(heartbeat_t));
    if (!heartbeat_p) g_error("malloc heartbeat_p: %s\n",strerror(errno));
    memset(heartbeat_p, 0, sizeof(heartbeat_t));

    rfm_mutex_init(heartbeat_p->mutex);
    rfm_cond_init(heartbeat_p->signal);
    heartbeat_p->condition = 0;
    heartbeat_p->path = g_strdup(path);
    heartbeat_p->test = test;

    g_mutex_lock(heartbeat_p->mutex);
    NOOP("Creating wait thread for heartbeat_g_file_test_with_timeout\n");
    // This thread does not affect view nor window.
    heartbeat_p->thread =
	rfm_thread_create ("heartbeat_g_file_test", heartbeat_g_file_test, heartbeat_p, TRUE);
    if (heartbeat_p->thread && !heartbeat_p->condition) {
	if (!rfm_cond_timed_wait(heartbeat_p->signal, heartbeat_p->mutex, 2)) {
	    g_mutex_unlock(heartbeat_p->mutex);
	    NOOP("dead heartbeat: rfm_g_file_test\n");
	    // Dead heartbeat:
	    // Fire off a wait and cleanup thread.
	    rfm_thread_create ("wait_on_thread", wait_on_thread, heartbeat_p, FALSE);
	    return FALSE;
	}
    }
    g_mutex_unlock(heartbeat_p->mutex);
    return (GPOINTER_TO_INT(wait_on_thread(heartbeat_p)));

}
#endif
gboolean
rfm_g_file_test(const gchar *path, GFileTest test){
    if (!path) return FALSE;
    if (!g_path_is_absolute(path)) return FALSE;
    return g_file_test(path, test);
}


