/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent.h"
#include "rfm_modules.h"
#include "rodent_popup.i"

void
rodent_update_menu(GtkWidget *menu){
    if (!menu || !GTK_IS_CONTAINER(menu)) return;
    // First, hide everything in menu.
#if GTK_MAJOR_VERSION==2 && GTK_MINOR_VERSION<24
    // This is deprecated as of 2.24, which forces us
    // to get children and hide each one of them... bleah!
    gtk_widget_hide_all (menu);
#else
    GList *children = gtk_container_get_children(GTK_CONTAINER(menu));
    GList *tmp=children;
    for (; tmp && tmp->data; tmp = tmp->next){
	if(GTK_IS_WIDGET(tmp->data)) gtk_widget_hide (GTK_WIDGET(tmp->data));
    }
    g_list_free(children);
#endif
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;
    
    // Show the file menu
    SHOW_IT("file_menu");
    // hide items in file menu
    HIDE_IT("rename_menuitem");
    HIDE_IT("duplicate_menuitem");
    HIDE_IT("symlink_menuitem");
    HIDE_IT("touch_menuitem");
    HIDE_IT("properties_menuitem");
 
    // Show search item if we have fgr
    gchar *fgr = g_find_program_in_path("rodent-fgr");
    if (fgr) SHOW_IT ( "find2");
    g_free(fgr);

    // Next, show other items on a need to see basis.
    //
    // First test, do we have a valid associated population_p?
    gint items=g_slist_length(view_p->selection_list);
    if (items == 1){
        // Yes, we do. So we show the cp/mv/ln items.
	SHOW_IT("rename_menuitem");
	SHOW_IT("duplicate_menuitem");
	SHOW_IT("symlink_menuitem");
    }
    if (items && view_p->selection_list->data ) {
	HIDE_IT ( "addbookmark_menuitem");
	HIDE_IT ( "removebookmark_menuitem");
	show_remove_item(view_p);
	show_open_items(view_p);
	show_properties_item(view_p);
	if (items==1) {
            // Mimetype specific items are shown here:
	    popup_autostuff (view_p->selection_list->data);

            // Does the item need the mount/umount item?
            // This will only be effective if fstab plugin is loadable.
	    show_mount_item(view_p); 

	    record_entry_t *en = view_p->selection_list->data;
	    if (IS_SDIR(en->type)){
		gboolean on = rodent_path_is_bookmarkable(en->path);
		if (on) setup_bookmark_menuitem( en, "addbookmark_menuitem", on);
		else setup_bookmark_menuitem( en, "removebookmark_menuitem", on);
	    }
	} 	
	if(view_p->module) {
	    rfm_rational (PLUGIN_DIR, view_p->module, widgets_p, 
		    view_p->selection_list->data, "hide_local_menu_items");
	    rfm_rational (PLUGIN_DIR, view_p->module, widgets_p, 
		    view_p->selection_list->data, "show_local_menu_items");
	}
	SHOW_IT ("touch_menuitem");
    } 
    else { 
	//rodent_reset_menu_toggles ();// this is now after popup is mapped
        rodent_reset_menu ();
    }

}

void
rodent_pop_menu (const gchar *menu_id, GdkEventButton * event) {
    gboolean is_thread = (g_thread_self() != rfm_get_gtk_thread());
    if (is_thread){
	g_error("rodent_pop_menu: fixme, this is broken when called from a thread\n");
	return;
    }
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;

    if(view_p->module && !(event->state & GDK_CONTROL_MASK)){
	if (rfm_rational (PLUGIN_DIR, view_p->module,  
		    widgets_p, NULL, 
		"private_popup")) return ;
    }

    GtkMenu *menu = rfm_get_widget(menu_id);
    if (!menu){
	DBG("Menu not ready: be patient\n");
	return;
    }
#if 10
    // hide/show items depending on context:
    rodent_update_menu(GTK_WIDGET(menu));
#endif
    // pop it:
    NOOP(stderr, "gtk_menu_popup \"%s\" now...\n", menu_id);
#if GTK_MAJOR_VERSION<3 
    //gtk_menu_popup (menu, rfm_get_widget("main_popup_menu_menu"), NULL, NULL, NULL, 3, event->time);
    gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 3, event->time);
#else
    GdkDeviceManager *gdm = 
	    gdk_display_get_device_manager (gdk_display_get_default());    
    gtk_menu_popup_for_device (menu,
	    gdk_device_manager_get_client_pointer(gdm), // GdkDevice *device, 
	    NULL, NULL, NULL, NULL,
	    NULL,  //GDestroyNotify destroy, 
	    3, event->time);
#endif
                                                          
#if 10
    // Finally, set toggles to correct setting (with popup mapped).
    if (!g_slist_length(view_p->selection_list) || !view_p->selection_list->data ) {
        rodent_reset_menu_toggles ();
    }
#endif


    NOOP(stderr, "menu popped \"%s\" now...\n", menu_id);
    return ;
}

///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
void
rodent_push_view_go_history (void) {
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    if (!widgets_p) return;
    view_t *view_p = widgets_p->view_p;
    
    record_entry_t *history_en;
    /*if (!view_p->en || !view_p->en->path) return; */
    if(view_p->go_list) {
        GList *last = g_list_last (view_p->go_list);
        record_entry_t *last_en;
        last_en = last->data;
        if(!last_en && !view_p->en)
            return;             /* NULL case */
        if(last_en && view_p->en) {
            if(!last_en->path && !view_p->en->path)
                return;
            if(last_en->path && view_p->en->path) {
                if(strcmp (last_en->path, view_p->en->path) == 0)
                    return;
            }
        }
    }
    history_en = rfm_copy_entry (view_p->en);
    view_p->go_list = g_list_append (view_p->go_list, history_en);
    //if (view_p->f_list){}


    return;
}
////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

/****************/


void
rodent_reset_menu (void) {
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t * view_p = widgets_p->view_p;
    gchar *void_items[]={"goto_menu", "sort_menu", "view_menu", "help_menu", "refresh3", "run2", "navigation_separator",  "file_separator", "exec_separator", "exit2", NULL};

    gchar **p;

    static gsize initialized = 0;
    if (g_once_init_enter(&initialized)){
	// dynamically constructed texts.
	GtkWidget *a = rfm_get_widget ("host2");
        rfm_replace_menu_label(a, g_get_host_name ());
	/*GtkWidget *label = gtk_bin_get_child (GTK_BIN (a));
	gtk_label_set_text ((GtkLabel *) label, g_get_host_name ());*/
	a = rfm_get_widget ("default_iconsize");
	gchar *text = g_strdup_printf("%s (%s)", _("Reset default state"), _("Single Instance"));
        rfm_replace_menu_label(a, text);
/*	label = gtk_bin_get_child (GTK_BIN (a));
	gtk_label_set_text ((GtkLabel *) label, text);*/
	g_free(text);
 
	a = rfm_get_widget ("default_iconsize_all");
	text = g_strdup_printf("%s (%s)", _("Reset default state"), _("All Instances"));
        rfm_replace_menu_label(a, text);
	/*label = gtk_bin_get_child (GTK_BIN (a));
	gtk_label_set_text ((GtkLabel *) label, text);*/
	g_free(text);
	g_once_init_leave(&initialized, 1);
    }

    for (p=void_items; p && *p; p++) SHOW_IT(*p);
    if (view_p->flags.type == DESKVIEW_TYPE) HIDE_IT("newtab");
    if(!view_p->en || !view_p->en->path) HIDE_IT ( "goup_menuitem");
    if(!view_p->go_list) HIDE_IT ("go_back_menuitem");
    if(!view_p->f_list) HIDE_IT("go_forward_menuitem");

    rodent_bookmark_set_menuitems(&(view_p->widgets), "bookmark");
    gint i;
    for(i = 0; i < DEEPEST_BOOK_MENU_LEVELS; i++) {
	gchar *name = g_strdup_printf ("bookmark-%d", i);
	GtkWidget *menuitem = rfm_get_widget(name); 
	 if (!menuitem){
	     DBG("rodent_reset_menu(): widget %s not found\n", name);
	     g_free(name);
	     continue;
	 }
	 gchar *path=g_object_get_data(G_OBJECT(menuitem), "path");
	 if (!path) {
	     HIDE_IT ( name);
	 } else {
	     SHOW_IT ( name);
	 }
	 g_free(name);
    }
   
    for(i = 0; i < DEEPEST_DIR_MENU_LEVELS; i++) {
	char *name = g_strdup_printf ("level-%d", i);
	HIDE_IT ( name);
	g_free (name);
    }
    /* this shows valid directories to goto */
    recursive_dirname (&(view_p->widgets), view_p->deepest_dir, 0);
    
    gchar **w_name,
     *widget_name[] = { "show_hidden_menuitem", "preview_images_menuitem", "newfile_menuitem", "newdirectory_menuitem", NULL };
    for(w_name = widget_name; w_name && *w_name; w_name++) {
	if(view_p->en && view_p->en->path) {
	    if(view_p->module) {
		if(rfm_void (PLUGIN_DIR, view_p->module, *w_name))
		    SHOW_IT ( *w_name);
	    } else if(IS_SDIR(view_p->en->type)) {
		SHOW_IT ( *w_name);
	    }
	}
    }

    if(view_p->flags.preferences & SORT_ASCENDING){
	HIDE_IT ( "ascending1");
        // False positive.
        // coverity[copy_paste_error : FALSE]
	SHOW_IT ( "descending1");
    } else {
	HIDE_IT ( "descending1");
	SHOW_IT ( "ascending1");
    }

    // dotdesktop_menu will not exist if dotdesktop module is not present
    gboolean have_dotdesktop =
	GPOINTER_TO_INT(rfm_void(PLUGIN_DIR, "dotdesktop", "is_root_module"));
    if (have_dotdesktop) {
	GtkWidget *applications_menu = rfm_get_widget("applications_menu");
	if (applications_menu) {
	    GtkWidget *popup_widget=
		rfm_rational(PLUGIN_DIR, "dotdesktop",widgets_p,
			applications_menu, 
			"dotdesktop_nondetached_menu");
	    if (popup_widget) {
		gtk_widget_show_all(applications_menu);
	    }
	}
    }	


#if 0
    if (view_p->flags.type == DESKVIEW_TYPE) {
	if (ICON_SIZE(view_p) < BIG_ICON_SIZE) {
	    SHOW_IT ( "plus_iconsize");
	}
	if (ICON_SIZE(view_p) >= TINY_ICON_SIZE) {
	    SHOW_IT ( "minus_iconsize");
	}
    }
    if (rfm_get_default_size() != ICON_SIZE_ID(view_p)){
	SHOW_IT ( "default_iconsize");
    }
#endif



    if(rfm_pasteboard_status (view_p) && view_p->en && view_p->en->path
	    && 
	      IS_SDIR(view_p->en->type)
	    )
    {
        SHOW_IT ( "paste_menuitem");
    }

    if(view_p->constructor) {
        SHOW_IT ( "newwindow");

    }

    if(getenv ("TERMINAL_CMD") && strlen (getenv ("TERMINAL_CMD"))) {
        gchar *term = g_find_program_in_path (getenv ("TERMINAL_CMD"));
	if (!term){
	    const gchar **p=rfm_get_terminals();
	    for (;p && *p; p++){
		term = g_find_program_in_path (*p);
		if (term) break;
	    }
	}

	if (term) {
	    SHOW_IT ( "terminal2");
	    g_free(term);
	} else {
#ifdef DEBUG
	    const gchar *t=getenv ("TERMINAL_CMD");
	    DBG("Terminal command \"%s\" not found\n", t);
#endif
	}
    }
    gchar *fgr = g_find_program_in_path("rodent-fgr");
    if (fgr) {
	SHOW_IT ( "find2");
	g_free(fgr);
    }
    gchar *rodent_diff = g_find_program_in_path("rodent-diff");
    if (rodent_diff){
	SHOW_IT ( "differences2");
	g_free(rodent_diff);
    }

    if(view_p->en) {
        SHOW_IT ( "select_menu");
        SHOW_IT ( "edit_separator");
    }
    if (rfm_void(RFM_MODULE_DIR, "settings", "module_active")) {
	SHOW_IT ( "settings2");
    }
    HIDE_IT ( "addbookmark_menuitem");
    HIDE_IT ( "removebookmark_menuitem");
    if (view_p->en && view_p->en->path && g_path_is_absolute(view_p->en->path)){
	gboolean on = rodent_path_is_bookmarkable(view_p->en->path);
	if (on){
	    setup_bookmark_menuitem( view_p->en, "addbookmark_menuitem", on);
	} else {
	    setup_bookmark_menuitem( view_p->en, "removebookmark_menuitem", on);
	}
    }
    GtkWidget *inner_label = rfm_get_widget ("sort_menu");
    if(view_p->flags.preferences & SORT_ASCENDING)
        rfm_replace_menu_image(inner_label, "xffm/stock_sort-ascending");
    else
        rfm_replace_menu_image(inner_label, "xffm/stock_sort-descending");
}

void
rodent_reset_menu_toggles (void) {
    NOOP( "***** popup now at rodent_reset_menu_toggles\n");
    widgets_t *widgets_p = rfm_get_widget ("widgets_p");
    view_t *view_p = widgets_p->view_p;
    GtkCheckMenuItem *check;
    switch (GPOINTER_TO_INT (ICON_SIZE_ID(view_p))){
	case LIST_ICON_SIZE:
	    check = rfm_get_widget ("compact_iconsize"); break;
	    break;
	case TINY_ICON_SIZE:
	    check = rfm_get_widget ("tiny_iconsize"); break;
	    break;
	case MEDIUM_ICON_SIZE:
	    check = rfm_get_widget ("big_iconsize"); break;
	    break;
	case BIG_ICON_SIZE:
	    check = rfm_get_widget ("huge_iconsize"); break;
	    break;
	case SMALL_ICON_SIZE:
	default:
	    check = rfm_get_widget ("normal_iconsize"); break;
	    break;
    }
    gtk_check_menu_item_set_active (check, TRUE);
    switch (view_p->flags.sortcolumn){
	case NAME_SORT:
	    check = rfm_get_widget ("namesort"); break;
	case SIZE_SORT:
	    check = rfm_get_widget ("size6"); break;
	case DATE_SORT:
	    check = rfm_get_widget ("date6"); break;
	case OWNER_SORT:
	    check = rfm_get_widget ("owner6"); break;
	case GROUP_SORT:
	    check = rfm_get_widget ("group6"); break;
	case MODE_SORT:
	    check = rfm_get_widget ("mode6"); break;
	case TYPE_SORT:
	default:
	    check = rfm_get_widget ("unsorted6"); break;

    }
    gtk_check_menu_item_set_active (check, TRUE);
    gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
				    rfm_get_widget ("casesort_menuitem"), view_p->flags.preferences & __CASE_INSENSITIVE);

    NOOP ("before setting __SHOW_IMAGES toggle now...%s\n", (view_p->flags.preferences & __SHOW_IMAGES) ? "ON" : "Off");
    gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
				    rfm_get_widget ("preview_images_menuitem"), view_p->flags.preferences & __SHOW_IMAGES);

    NOOP ("after setting __SHOW_IMAGES toggle now...%s\n", (view_p->flags.preferences & __SHOW_IMAGES) ? "ON" : "Off");
    gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
				    rfm_get_widget ("show_hidden_menuitem"), view_p->flags.preferences & __SHOW_HIDDEN);
    gtk_check_menu_item_set_active ((GtkCheckMenuItem *)
				    rfm_get_widget ("show_backup_menuitem"), view_p->flags.preferences & __SHOWS_BACKUP);
    NOOP ("setting __SHOW_HIDDEN toggle now...%s\n", (view_p->flags.preferences & __SHOW_HIDDEN) ? "ON" : "Off");
    return;
}

static pthread_mutex_t flags_mutex=PTHREAD_MUTEX_INITIALIZER;
void
rodent_recover_flags (extra_key_t * extra_key_p) {
    DBHashTable *runflags;
    GString *gs;
    gint64 *flags;

    pthread_mutex_lock(&flags_mutex);
    TRACE("opening %s...\n",extra_key_p->flagfile); 
    runflags = dbh_new (extra_key_p->flagfile, NULL, DBH_READ_ONLY|DBH_PARALLEL_SAFE);
    TRACE("opened %s.\n",extra_key_p->flagfile); 

    if(runflags  == NULL) {
        NOOP ("Cannot open %s\n", extra_key_p->flagfile);
        extra_key_p->flag1 = 0;
        extra_key_p->flag2 = 0;
	pthread_mutex_unlock(&flags_mutex);
        return;
    }
    dbh_set_parallel_lock_timeout(runflags, 3);
    gs = g_string_new (extra_key_p->response);
    sprintf ((char *)DBH_KEY (runflags), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    if (dbh_load (runflags)){
    flags = (gint64 *)runflags->data;
        extra_key_p->flag1 = flags[0];
        extra_key_p->flag2 = flags[1];
    } else {
        extra_key_p->flag1 = extra_key_p->flag2 = 0;
    }
    dbh_close (runflags);
    pthread_mutex_unlock(&flags_mutex);

    NOOP ("rodent_recover_flags(): flags recovered from dbh file for %s, flag1=%d flag2=%d\n",
         extra_key_p->response, extra_key_p->flag1, extra_key_p->flag2);
}

void
rodent_save_flags (extra_key_t * extra_key_p) {
    DBHashTable *runflags;
    GString *gs;
    gint64 *flags;

    pthread_mutex_lock(&flags_mutex);
    TRACE("opening %s...\n",extra_key_p->flagfile); 
    runflags = dbh_new (extra_key_p->flagfile, NULL, DBH_PARALLEL_SAFE);
    TRACE("opened %s.\n",extra_key_p->flagfile); 

    if(runflags == NULL) {
        NOOP ("Creating cache file: %s", extra_key_p->flagfile);
	unsigned char keylength=11;
        gchar *directory = g_path_get_dirname(extra_key_p->flagfile);
        if (!g_file_test(directory, G_FILE_TEST_IS_DIR)){
            g_mkdir_with_parents(directory, 0700);
        }
        g_free(directory);
        TRACE("opening %s...\n",extra_key_p->flagfile); 
	runflags = dbh_new (extra_key_p->flagfile, &keylength, DBH_CREATE|DBH_PARALLEL_SAFE);
        TRACE("opened %s.\n",extra_key_p->flagfile); 
        if(runflags == NULL) {
            DBG ("Cannot create %s\n", extra_key_p->flagfile);
	    pthread_mutex_unlock(&flags_mutex);
	    return;
        }
    }
    dbh_set_parallel_lock_timeout(runflags, 3);
    gs = g_string_new (extra_key_p->response);
    sprintf ((char *)DBH_KEY (runflags), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);
    flags = (gint64 *)runflags->data;
    flags[0] = extra_key_p->flag1;
    flags[1] = extra_key_p->flag2;
    dbh_set_recordsize (runflags, 2 * sizeof (gint64));

    dbh_update (runflags);
    dbh_close (runflags);
    pthread_mutex_unlock(&flags_mutex);
    NOOP ("flags saved in dbh file  %s flag1=%d flag2=%d\n", extra_key_p->flagfile, extra_key_p->flag1, extra_key_p->flag2);

}


gchar *
rodent_get_text_editor(record_entry_t *en){
    if(!g_path_is_absolute(en->path)) {
	return NULL;
    }
    gchar **text_editors = NULL;
    // Default editor command.
    // We magic the file because the freedesktop standard
    // is borked and changes text/ for application/ on a 
    // number of files. Precisely what should not be done.
    //
    if (!en->mimetype) en->mimetype = MIME_type (en->path, en->st);
    if(!en->mimetype || strcmp(en->mimetype, _("unknown"))==0) {
	if (IS_LOCAL_TYPE(en->type) && !en->mimemagic){
	    en->mimemagic = rfm_rational(RFM_MODULE_DIR, "mime", en, "mime_magic", "mime_function");
	    if (!en->mimemagic) en->mimemagic = g_strdup(_("unknown"));
	} else en->mimemagic = g_strdup(_("unknown"));
    }
    if(en->mimemagic && strstr (en->mimemagic, "text/")){
	    text_editors = MIME_apps ("text/plain");
    } else if(en->mimetype && strstr (en->mimetype, "text/")){
	    text_editors = MIME_apps ("text/plain");
    }

    // Even magic mimetype may miss the mark. Take care of that here... 
    if(!en->filetype) {
	if (IS_LOCAL_TYPE(en->type)) {
	    en->filetype = rfm_rational(RFM_MODULE_DIR, "mime", en, "mime_file", "mime_function"); 
	    if (!en->filetype) en->filetype = g_strdup(_("unknown"));
	}
	else en->filetype = g_strdup(_("unknown"));
    }
    if(!text_editors) {
	if(en->filetype && strstr (en->filetype, "text"))
	    text_editors = MIME_apps ("text/plain");
    }
    
    if(!text_editors) {
	return NULL;
    }
    g_strfreev(text_editors);
    /* OK to apply an editor. We will use the default editor */
    gchar *basename=NULL;
    if(getenv ("EDITOR") && strlen (getenv ("EDITOR"))) {
	basename=g_strdup(getenv ("EDITOR"));
    }
    return (basename);
}


