#!/usr/bin/python3
# -*- coding: utf-8 -*-


import os
from pathlib import Path

from urllib3 import PoolManager

from slpkg.repositories import Repositories


class Check:  # pylint: disable=[R0903]
    """Check for repository update."""

    def __init__(self) -> None:
        self.repos = Repositories()
        self.repositories = Repositories().repositories

    def test(self) -> None:
        """Test for ChangLog files."""

        http = PoolManager()  # Initialize PoolManager once, outside the loop

        for name, data in self.repositories.items():
            local_size: int = 0  # Initialize local_size for each repository

            local_chg_txt: Path = Path(data['path'], 'ChangeLog.txt')
            repo_chg_txt: str = f"{data['mirror_changelog']}ChangeLog.txt"

            try:
                repo = http.request('GET', repo_chg_txt)

                repo_size: int = 0  # Default to 0 if Content-Length is not available
                if 'content-length' in repo.headers:  # Check if the header exists (case-insensitive)
                    repo_size = int(repo.headers['content-length'])
                elif 'transfer-encoding' in repo.headers and repo.headers['transfer-encoding'] == 'chunked':
                    # If chunked, Content-Length won't be present. Handle as needed.
                    # For a ChangeLog, you'd likely need to read the whole body to get its size.
                    # For now, we'll just indicate it's chunked and set size to 0 or -1.
                    print(f"Warning: '{name}' ChangeLog.txt uses chunked transfer encoding. Cannot get size from headers.")
                    repo_size = -1  # Or some other indicator
                else:
                    print(f"Warning: '{name}' ChangeLog.txt response missing 'Content-Length' header.")
                    # Fallback: Read the body to get the actual size if Content-Length is missing
                    # This might be slow for very large files.
                    # repo_size = len(repo.data)

                if local_chg_txt.is_file():
                    local_size = int(os.stat(local_chg_txt).st_size)

                print(f'{name=} = {local_size=}, {repo_size=}, {local_size != repo_size}')

            except Exception as e:  # Catch broader exceptions for network issues etc. # pylint: disable=[W0718]
                print(f"Error checking {name} ({repo_chg_txt}): {e}")
                print(f'{name=} = {local_size=}, repo_size={"NA"}, False (Error during check)')


if __name__ == "__main__":
    check = Check()
    check.test()
