/*
 * Copyright (c) 1999-2004
 *	Stelian Pop <stelian@popies.net>, 1999-2004
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <config.h>
#include <compaterr.h>
#include <sys/types.h>
#include <sys/xattr.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <bsdcompat.h>
#include <protocols/dumprestore.h>
#ifdef TRANSSELINUX			/*GAN6May06 SELinux MLS */
# include <selinux/selinux.h>
#endif
#include "restore.h"
#include "extern.h"
#include "pathnames.h"

/*
 * Data structures below taken from the kernel
 */

/* Maximum number of references to one attribute block */
#define EXT2_XATTR_REFCOUNT_MAX		1024

/* Name indexes */
#define EXT2_XATTR_INDEX_MAX			10
#define EXT2_XATTR_INDEX_USER			1
#define EXT2_XATTR_INDEX_POSIX_ACL_ACCESS	2
#define EXT2_XATTR_INDEX_POSIX_ACL_DEFAULT	3
#define EXT2_XATTR_INDEX_TRUSTED		4
#define	EXT2_XATTR_INDEX_LUSTRE			5
#define EXT2_XATTR_INDEX_SECURITY		6
#define EXT2_XATTR_INDEX_SYSTEM			7

struct ext2_xattr_header {
	u_int32_t	h_magic;	/* magic number for identification */
	u_int32_t	h_refcount;	/* reference count */
	u_int32_t	h_blocks;	/* number of disk blocks used */
	u_int32_t	h_hash;		/* hash value of all attributes */
	u_int32_t	h_reserved[4];	/* zero right now */
};

struct ext3_xattr_ibody_header {
	u_int32_t	h_magic;	/* magic number for identification */
};

struct ext2_xattr_entry {
	u_char		e_name_len;	/* length of name */
	u_char		e_name_index;	/* attribute name index */
	u_int16_t	e_value_offs;	/* offset in disk block of value */
	u_int32_t	e_value_block;	/* disk block attribute is stored on (n/i) */
	u_int32_t	e_value_size;	/* size of attribute value */
	u_int32_t	e_hash;		/* hash value of name and value */
	char		e_name[0];	/* attribute name */
};

#define EXT2_XATTR_PAD_BITS		2
#define EXT2_XATTR_PAD		(1<<EXT2_XATTR_PAD_BITS)
#define EXT2_XATTR_ROUND		(EXT2_XATTR_PAD-1)
#ifndef EXT2_XATTR_LEN
#define EXT2_XATTR_LEN(name_len) \
	(((name_len) + EXT2_XATTR_ROUND + \
	sizeof(struct ext2_xattr_entry)) & ~EXT2_XATTR_ROUND)
#endif
#define EXT2_XATTR_NEXT(entry) \
	( (struct ext2_xattr_entry *)( \
	  (char *)(entry) + EXT2_XATTR_LEN((entry)->e_name_len)) )
#define EXT3_XATTR_SIZE(size) \
	(((size) + EXT2_XATTR_ROUND) & ~EXT2_XATTR_ROUND)

#define HDR(buffer) ((struct ext2_xattr_header *)(buffer))
#define ENTRY(ptr) ((struct ext2_xattr_entry *)(ptr))
#define IS_LAST_ENTRY(entry) (*(__u32 *)(entry) == 0)

#define BFIRST(buffer) ENTRY(HDR(buffer)+1)
#define IFIRST(buffer) ENTRY(((struct ext3_xattr_ibody_header *)(buffer))+1)

#define FIRST_ENTRY(buffer) \
	((HDR(buffer)->h_magic == EXT2_XATTR_MAGIC2) ? \
		IFIRST(buffer) : \
		BFIRST(buffer))

/*
 * On-block xattr value offsets start at the beginning of the block, but
 * on-inode xattr value offsets start after the initial header
 * (ext3_xattr_ibody_header).
 */
#define VALUE_OFFSET(buffer, entry) \
	(((HDR(buffer)->h_magic == EXT2_XATTR_MAGIC2) ? \
		(entry)->e_value_offs + sizeof(struct ext3_xattr_ibody_header) : \
		(entry)->e_value_offs))

static int xattr_cb_list (char *, char *, int, int, void *);
static int xattr_cb_set (char *, char *, int, int, void *);
static int xattr_cb_compare (char *, char *, int, int, void *);
static int xattr_verify (char *);
int xattr_count (char *, int *);
static int xattr_walk (char *, int (*)(char *, char *, int, int, void *), void *);

#define POSIX_ACL_XATTR_VERSION 0x0002

#define ACL_UNDEFINED_ID        (-1)

#define ACL_USER_OBJ            (0x01)
#define ACL_USER                (0x02)
#define ACL_GROUP_OBJ           (0x04)
#define ACL_GROUP               (0x08)
#define ACL_MASK                (0x10)
#define ACL_OTHER               (0x20)

typedef struct {
	u_int16_t	e_tag;
	u_int16_t	e_perm;
	u_int32_t	e_id;
} posix_acl_xattr_entry;

typedef struct {
	u_int32_t		a_version;
	posix_acl_xattr_entry	a_entries[0];
} posix_acl_xattr_header;

static inline size_t
posix_acl_xattr_size(int count)
{
	return (sizeof(posix_acl_xattr_header) +
		(count * sizeof(posix_acl_xattr_entry)));
}

struct posix_acl_entry {
	short		e_tag;
	unsigned short	e_perm;
	unsigned int	e_id;
};

struct posix_acl {
	unsigned int		a_count;
	struct posix_acl_entry	a_entries[0];
};

#define EXT3_ACL_VERSION        0x0001

typedef struct {
	u_int16_t	e_tag;
	u_int16_t	e_perm;
	u_int32_t	e_id;
} ext3_acl_entry;

typedef struct {
	u_int16_t	e_tag;
	u_int16_t	e_perm;
} ext3_acl_entry_short;

typedef struct {
	u_int32_t	a_version;
} ext3_acl_header;

static inline int ext3_acl_count(size_t size)
{
	ssize_t s;
	size -= sizeof(ext3_acl_header);
	s = size - 4 * sizeof(ext3_acl_entry_short);
	if (s < 0) {
		if (size % sizeof(ext3_acl_entry_short))
			return -1;
		return size / sizeof(ext3_acl_entry_short);
	} else {
		if (s % sizeof(ext3_acl_entry))
			return -1;
		return s / sizeof(ext3_acl_entry) + 4;
	}
}

int
posix_acl_to_xattr(const struct posix_acl *acl, void *buffer, size_t size) {
	posix_acl_xattr_header *ext_acl = (posix_acl_xattr_header *)buffer;
	posix_acl_xattr_entry *ext_entry = ext_acl->a_entries;
	int real_size, n;

	real_size = posix_acl_xattr_size(acl->a_count);
	if (!buffer)
		return real_size;
	if (real_size > (long)size) {
		fprintf(stderr, "ACL: not enough space to convert (%d %d)\n", real_size, (int)size);
		return -1;
	}

	ext_acl->a_version = POSIX_ACL_XATTR_VERSION;
#if BYTE_ORDER == BIG_ENDIAN
	swabst("1i", (u_char *)ext_acl);
#endif

	for (n=0; n < (long)acl->a_count; n++, ext_entry++) {
		ext_entry->e_tag  = acl->a_entries[n].e_tag;
		ext_entry->e_perm = acl->a_entries[n].e_perm;
		ext_entry->e_id   = acl->a_entries[n].e_id;
#if BYTE_ORDER == BIG_ENDIAN
		swabst("2s1i", (u_char *)ext_entry);
#endif
	}
	return real_size;
}

static struct posix_acl *
ext3_acl_from_disk(const void *value, size_t size)
{
	const char *end = (char *)value + size;
	int n, count;
	struct posix_acl *acl;

	if (!value)
		return NULL;
	if (size < sizeof(ext3_acl_header)) {
		fprintf(stderr, "ACL size too little\n");
		return NULL;
	}
#if BYTE_ORDER == BIG_ENDIAN
	swabst("1i", (u_char *)value);
#endif
	if (((ext3_acl_header *)value)->a_version != EXT3_ACL_VERSION) {
		fprintf(stderr, "ACL version unknown\n");
		return NULL;
	}
	value = (char *)value + sizeof(ext3_acl_header);
	count = ext3_acl_count(size);
	if (count < 0) {
		fprintf(stderr, "ACL bad count\n");
		return NULL;
	}
	if (count == 0)
		return NULL;
	acl = malloc(sizeof(struct posix_acl) + count * sizeof(struct posix_acl_entry));
	if (!acl) {
		fprintf(stderr, "ACL malloc failed\n");
		return NULL;
	}
	acl->a_count = count;

	for (n=0; n < count; n++) {
		ext3_acl_entry *entry = (ext3_acl_entry *)value;
#if BYTE_ORDER == BIG_ENDIAN
		swabst("2s", (u_char *)entry);
#endif
		if ((char *)value + sizeof(ext3_acl_entry_short) > end)
			goto fail;
		acl->a_entries[n].e_tag  = entry->e_tag;
		acl->a_entries[n].e_perm = entry->e_perm;
		switch(acl->a_entries[n].e_tag) {
		case ACL_USER_OBJ:
		case ACL_GROUP_OBJ:
		case ACL_MASK:
		case ACL_OTHER:
			value = (char *)value + sizeof(ext3_acl_entry_short);
			acl->a_entries[n].e_id = ACL_UNDEFINED_ID;
			break;

		case ACL_USER:
		case ACL_GROUP:
#if BYTE_ORDER == BIG_ENDIAN
			swabst("4b1i", (u_char *)entry);
#endif
			value = (char *)value + sizeof(ext3_acl_entry);
			if ((char *)value > end)
				goto fail;
			acl->a_entries[n].e_id = entry->e_id;
			break;

		default:
			goto fail;
		}
	}
	if (value != end)
		goto fail;
	return acl;

fail:
	fprintf(stderr, "ACL bad entry\n");
	free(acl);
	return NULL;
}

/*
 * Dump code starts here :)
 */

static int
xattr_cb_list(char *name, char *value, int valuelen, int isSELinux, UNUSED(void *private))
{
	if (strcmp(name, "system.data") == 0) {
		return GOOD;
	}
	(void)isSELinux;
	value[valuelen] = '\0';
	printf("EA: %s:%s\n", name, value);

	return GOOD;
}

static int
xattr_cb_set(char *name, char *value, int valuelen, int isSELinux, void *private)
{
	char *path = (char *)private;
	int err;

	if (strcmp(name, "system.data") == 0) {
		memcpy(inline_data + INLINE_DATA_MAX_INODE_SIZE, value, valuelen);
		return GOOD;
	}
	if (Nflag)
		return GOOD;

	(void)isSELinux;
#ifdef TRANSSELINUX			/*GAN6May06 SELinux MLS */
	if (isSELinux)
		err = lsetfilecon(path, value);
	else
#endif
		err = lsetxattr(path, name, value, valuelen, 0);

	if (err) {
		warn("%s: EA set %s:%s failed", path, name, value);
		return FAIL;
	}

	return GOOD;
}

/* if value is a printable string then we return it as a NUL-terminated string in buffer, otherwise return a hex representation */
static void xattr_print(char *buffer, int bufsz, const char *value, int valuelen)
{
	int ascii = 1;
	int i;
	int dst;
	for (i=0; ascii && i < valuelen && i < bufsz-1; ++i) {
		buffer[i] = value[i];
		if(!isprint((unsigned char)value[i]))
			ascii = 0;
	}
	if (ascii) {
		buffer[i] = 0;
		return;
	}
	i = 0;
	dst = 0;
	while (i < valuelen) {
		if (dst == bufsz-1) {
			buffer[dst] = 0;
			return;
		}
		if (dst+(i>0?4:3) > bufsz) {
			return;
		}
		sprintf(buffer+dst, "%s%02x", i>0?" ":"", (unsigned char)value[i]);
		dst += (i>0?3:2);
		++i;
	}
}

static int
xattr_cb_compare(char *name, char *value, int valuelen, int isSELinux, void *private)
{
	char *path = (char *)private;
	char valuef[XATTR_MAXSIZE];
	int valuesz;

	if (strcmp(name, "system.data") == 0) {
		memcpy(inline_data + INLINE_DATA_MAX_INODE_SIZE, value, valuelen);
		return GOOD;
	}
	(void)isSELinux;
#ifdef TRANSSELINUX			/*GAN6May06 SELinux MLS */
	if (isSELinux)
	{
		char *con = NULL;

		if (lgetfilecon(path, &con) < 0) {
			warn("%s: EA compare lgetfilecon failed\n", path);
			return FAIL;
		}

		valuesz = strlen(con) + 1;
		valuef[0] = 0;
		strncat(valuef, con, sizeof(valuef) - 1);
		freecon(con);
	}
	else {
#endif
		valuesz = lgetxattr(path, name, valuef, XATTR_MAXSIZE);
		if (valuesz < 0) {
			if(errno == ENODATA) {
				valuef[0] = 0;
			} else {
				warn("%s: EA compare lgetxattr for %s failed\n", path, name);
				return FAIL;
			}
		}
#ifdef TRANSSELINUX			/*GAN6May06 SELinux MLS */
	}
#endif

	if (valuesz != valuelen || memcmp(value, valuef, valuelen)) {
		/* GAN24May06: show name and new value for user to compare */
		char tvalue[49];
		xattr_print(tvalue, 49, value, valuelen);
		if (valuesz < 0) {
			fprintf(stderr, "%s: EA %s with value %s on tape does not exist on disk\n", path, name, tvalue);
		} else {
			char fvalue[49];
			xattr_print(fvalue, 49, valuef, valuesz);
			fprintf(stderr, "%s: EA %s value changed from %s on tape to %s on disk\n", path, name, tvalue, fvalue);
		}
		return FAIL;
	}

	return GOOD;
}

static int
xattr_verify(char *buffer)
{
	struct ext2_xattr_entry *entry;
	char *end;

	end = buffer + XATTR_MAXSIZE;

#if BYTE_ORDER == BIG_ENDIAN
	swabst("4i", (u_char *)buffer);
#endif

	if (HDR(buffer)->h_magic != EXT2_XATTR_MAGIC &&
	    HDR(buffer)->h_magic != EXT2_XATTR_MAGIC2) {
		fprintf(stderr, "error in EA block 1\n");
		fprintf(stderr, "magic = %x\n", HDR(buffer)->h_magic);

		return FAIL;
	}

	/* check the on-disk data structure */
	entry = FIRST_ENTRY(buffer);
#if BYTE_ORDER == BIG_ENDIAN
	swabst("2b1s3i", (u_char *)entry);
#endif
	while (!IS_LAST_ENTRY(entry)) {
		struct ext2_xattr_entry *next = EXT2_XATTR_NEXT(entry);

		if ((char *)next >= end) {
			fprintf(stderr, "error in EA block\n");
			return FAIL;
		}
		entry = next;
#if BYTE_ORDER == BIG_ENDIAN
		swabst("2b1s3i", (u_char *)entry);
#endif
	}
	return GOOD;
}

int
xattr_count(char *buffer, int *count)
{
	struct ext2_xattr_entry *entry;
	int result = 0;

	/* list the attribute names */
	for (entry = FIRST_ENTRY(buffer); !IS_LAST_ENTRY(entry);
	     entry = EXT2_XATTR_NEXT(entry))
		switch (entry->e_name_index) {
			case EXT2_XATTR_INDEX_SYSTEM:
				if (entry->e_name_len == 4 && strncmp(entry->e_name, "data", 4)==0)
					break;
				/* FALLTHROUGH */
			default:
				result++;
				break;
		}
	*count = result;
	return GOOD;
}

/* We do not want to return early from this function as even if one attribute fails to apply for some reason, we want to try the rest. */
static int
xattr_walk(char *buffer, int (*xattr_cb)(char *, char *, int, int, void *), void *private)
{
	struct ext2_xattr_entry *entry;
	int rcode = GOOD;

	/* list the attribute names */
	for (entry = FIRST_ENTRY(buffer); !IS_LAST_ENTRY(entry);
	     entry = EXT2_XATTR_NEXT(entry)) {
		char name[XATTR_MAXSIZE], value[XATTR_MAXSIZE];
		int size;
		int off;
		int convertacl = 0;
		int convertcon = 0;

		switch (entry->e_name_index) {
		case EXT2_XATTR_INDEX_USER:
			strcpy(name, "user.");
			break;
		case EXT2_XATTR_INDEX_POSIX_ACL_ACCESS:
			strcpy(name, "system.posix_acl_access");
			convertacl = 1;
			break;
		case EXT2_XATTR_INDEX_POSIX_ACL_DEFAULT:
			strcpy(name, "system.posix_acl_default");
			convertacl = 1;
			break;
		case EXT2_XATTR_INDEX_TRUSTED:
			strcpy(name, "trusted.");
			break;
		case EXT2_XATTR_INDEX_LUSTRE:
			strcpy(name, "lustre.");
			break;
		case EXT2_XATTR_INDEX_SYSTEM:
			strcpy(name, "system.");
			break;
		case EXT2_XATTR_INDEX_SECURITY:
			strcpy(name, "security.");
#ifdef TRANSSELINUX			/*GAN6May06 SELinux MLS */
			convertcon = transselinuxflag;
#endif
			break;
		default:
			fprintf(stderr, "Unknown EA index\n");
			rcode = FAIL;
			continue;
		}

		off = strlen(name);
		memcpy(name + off, entry->e_name, entry->e_name_len);
		name[off + entry->e_name_len] = '\0';
		size = entry->e_value_size;

		memcpy(value, buffer + VALUE_OFFSET(buffer, entry), size);
		value[size]='\0';

		if (convertacl) {
			struct posix_acl *acl;

			acl = ext3_acl_from_disk(value, size);
			if (!acl) {
				rcode = FAIL;
				continue;
			}
			size = posix_acl_to_xattr(acl, value, XATTR_MAXSIZE);
			if (size < 0) {
				rcode = FAIL;
				continue;
			}
			free(acl);
		}

#ifdef TRANSSELINUX			/*GAN6May06 SELinux MLS */
		if (convertcon  &&  strcmp(name, "security.selinux"))
			convertcon = 0;	/*GAN24May06 only for selinux */

		if (convertcon) {
			char *con = NULL;
			int err;

			if (!transselinuxarg)
				err = security_canonicalize_context(value, &con);
			else {
				strncat(value, transselinuxarg, sizeof(value) - 1);
				err = security_canonicalize_context_raw(value, &con);
			}

			if (err < 0) {
				warn("%s: EA canonicalize failed\n", value);
				rcode = FAIL;
				continue;
			}

			size = strlen(con) + 1;
			value[0] = 0;
			strncat(value, con, sizeof(value) - 1);
			freecon(con);
		}
#endif

		if (xattr_cb(name, value, size, convertcon, private) != GOOD)
			rcode = FAIL;
	}

	return rcode;
}

/* return count of the xattrs on the given file or FAIL if failed */
int xattr_count_file(char *path)
{
	int size, count = 0;
	char *names = NULL, *end_names, *name;

	/* return number of bytes a/v */
	size = llistxattr(path, NULL, 0);
	if (size < 0) {
			warn("%s: llistxattr failed", path);
			return FAIL;
	}

	names = malloc(size + 1);
	if (!names)
	{
			warn("%s: malloc failed", path);
			return FAIL;
	}

	/* return nr of bytes and park data in names array */
	size = llistxattr(path, names, size);
	if (size < 0)
	{
			warn("%s: llistxattr failed", path);
			free(names);
			return FAIL;
	}

	names[size] = '\0';
	end_names = names + size;

	for (name = names; name != end_names; name = strchr(name, '\0') + 1)
	{
			if (!*name)
				continue;
			count++;
	}
	free(names);

	return count;
}

int
xattr_compare(char *path, char *buffer)
{
	/* with ext4 there may be two tape blocks for attrs: one from the actual inode, plus
	   one extra filesystem block. So we cannot check attribute counts here
	   but defer it to the caller. */

	if (buffer) {
		if (xattr_verify(buffer) == FAIL)
			return FAIL;
	}

	return xattr_walk(buffer, xattr_cb_compare, path);
}

int
xattr_extract(char *path, char *buffer)
{
	if (dflag) {
		fprintf(stderr, "xattr_extract(%s)\n", path);
		xattr_walk(buffer, xattr_cb_list, NULL);
	}

	if (xattr_verify(buffer) == FAIL)
		return FAIL;

	return xattr_walk(buffer, xattr_cb_set, path);
}
