# Copyright 2018-2025 Jérôme Dumonteil
# Copyright (c) 2009-2013 Ars Aperta, Itaapy, Pierlis, Talend.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# Authors (odfdo project): jerome.dumonteil@gmail.com
# The odfdo project is a derivative work of the lpod-python project:
# https://github.com/lpod/lpod-python
# Authors: Hervé Cauwelier <herve@itaapy.com>
#          Jerome Dumonteil <jerome.dumonteil@itaapy.com>
"""Reference related classes for "text:reference-..." tags."""

from __future__ import annotations

from typing import TYPE_CHECKING, Any, Union, cast

from .element import Element, PropDef, register_element_class
from .element_strip import strip_elements, strip_tags
from .elements_between import elements_between

if TYPE_CHECKING:
    from .body import Body


class ReferenceMixin(Element):
    """Mixin class for classes containing References.

    Used by the following classes: "text:a", "text:h", "text:meta", "text:meta-field",
    "text:p", "text:ruby-base", "text:span". And with "office:text" for compatibility
    with previous versions.
    """

    def get_reference_marks_single(self) -> list[ReferenceMark]:
        """Return all the reference marks. Search only the tags
        text:reference-mark.
        Consider using : get_reference_marks()

        Returns: list of ReferenceMark
        """
        return cast(
            list[ReferenceMark],
            self._filtered_elements(
                "descendant::text:reference-mark",
            ),
        )

    def get_reference_mark_single(
        self,
        position: int = 0,
        name: str | None = None,
    ) -> ReferenceMark | None:
        """Return the reference mark that matches the criteria. Search only the
        tags text:reference-mark.
        Consider using : get_reference_mark()

        Args:

            position -- int

            name -- str

        Returns: ReferenceMark or None if not found
        """
        return cast(
            Union[None, ReferenceMark],
            self._filtered_element(
                "descendant::text:reference-mark", position, text_name=name
            ),
        )

    def get_reference_mark_starts(self) -> list[ReferenceMarkStart]:
        """Return all the reference mark starts. Search only the tags
        text:reference-mark-start.
        Consider using : get_reference_marks()

        Returns: list of ReferenceMarkStart
        """
        return cast(
            list[ReferenceMarkStart],
            self._filtered_elements(
                "descendant::text:reference-mark-start",
            ),
        )

    def get_reference_mark_start(
        self,
        position: int = 0,
        name: str | None = None,
    ) -> ReferenceMarkStart | None:
        """Return the reference mark start that matches the criteria. Search
        only the tags text:reference-mark-start.
        Consider using : get_reference_mark()

        Args:

            position -- int

            name -- str

        Returns: ReferenceMarkStart or None if not found
        """
        return cast(
            Union[None, ReferenceMarkStart],
            self._filtered_element(
                "descendant::text:reference-mark-start", position, text_name=name
            ),
        )

    def get_reference_mark_ends(self) -> list[ReferenceMarkEnd]:
        """Return all the reference mark ends. Search only the tags
        text:reference-mark-end.
        Consider using : get_reference_marks()

        Returns: list of ReferenceMarkEnd
        """
        return cast(
            list[ReferenceMarkEnd],
            self._filtered_elements(
                "descendant::text:reference-mark-end",
            ),
        )

    def get_reference_mark_end(
        self,
        position: int = 0,
        name: str | None = None,
    ) -> ReferenceMarkEnd | None:
        """Return the reference mark end that matches the criteria. Search only
        the tags text:reference-mark-end.
        Consider using : get_reference_marks()

        Args:

            position -- int

            name -- str

        Returns: ReferenceMarkEnd or None if not found
        """
        return cast(
            Union[None, ReferenceMarkEnd],
            self._filtered_element(
                "descendant::text:reference-mark-end", position, text_name=name
            ),
        )

    def get_reference_marks(self) -> list[ReferenceMark | ReferenceMarkStart]:
        """Return all the reference marks, either single position reference
        (text:reference-mark) or start of range reference (text:reference-mark-
        start).

        Returns: list of ReferenceMark or ReferenceMarkStart
        """
        return cast(
            list[ReferenceMark | ReferenceMarkStart],
            self._filtered_elements(
                "descendant::text:reference-mark-start | descendant::text:reference-mark"
            ),
        )

    def get_reference_mark(
        self,
        position: int = 0,
        name: str | None = None,
    ) -> ReferenceMark | ReferenceMarkStart | None:
        """Return the reference mark that match the criteria. Either single
        position reference mark (text:reference-mark) or start of range
        reference (text:reference-mark-start).

        Args:

            position -- int

            name -- str

        Returns: ReferenceMark or ReferenceMarkStart or None if not found
        """
        if name:
            request = (
                f"descendant::text:reference-mark-start"
                f'[@text:name="{name}"] '
                f"| descendant::text:reference-mark"
                f'[@text:name="{name}"]'
            )
            return self._filtered_element(
                request,
                position=0,
            )  # type: ignore[return-value]
        request = (
            "descendant::text:reference-mark-start | descendant::text:reference-mark"
        )
        return cast(
            Union[None, ReferenceMark, ReferenceMarkStart],
            self._filtered_element(request, position),
        )

    def get_references(self, name: str | None = None) -> list[Reference]:
        """Return all the references (text:reference-ref). If name is provided,
        returns the references of that name.

        Args:

            name -- str or None

        Returns: list of Reference
        """
        if name is None:
            return self._filtered_elements(
                "descendant::text:reference-ref",
            )  # type: ignore[return-value]
        request = f'descendant::text:reference-ref[@text:ref-name="{name}"]'
        return cast(list[Reference], self._filtered_elements(request))


class Reference(Element):
    """A reference to a content marked by a reference mark, "text:reference-
    ref".".

    The odf_reference element ("text:reference-ref") represents a field that
    references a "text:reference-mark-start" or "text:reference-mark" element.
    Its text:reference-format attribute specifies what is displayed from the
    referenced element. Default is 'page'
    Actual content is not updated except for the 'text' format by the
    update() method.


    Creation of references can be tricky, consider using this method:
        odfdo.paragraph.insert_reference()

    Values for text:reference-format :
        The defined values for the text:reference-format attribute supported by
        all reference fields are:
          - 'chapter': displays the number of the chapter in which the
            referenced item appears.
          - 'direction': displays whether the referenced item is above or
            below the reference field.
          - 'page': displays the number of the page on which the referenced
            item appears.
          - 'text': displays the text of the referenced item.
        Additional defined values for the text:reference-format attribute
        supported by references to sequence fields are:
          - 'caption': displays the caption in which the sequence is used.
          - 'category-and-value': displays the name and value of the sequence.
          - 'value': displays the value of the sequence.

        References to bookmarks and other references support additional values,
        which display the list label of the referenced item. If the referenced
        item is contained in a list or a numbered paragraph, the list label is
        the formatted number of the paragraph which contains the referenced
        item. If the referenced item is not contained in a list or numbered
        paragraph, the list label is empty, and the referenced field therefore
        displays nothing. If the referenced bookmark or reference contains more
        than one paragraph, the list label of the paragraph at which the
        bookmark or reference starts is taken.

        Additional defined values for the text:reference-format attribute
        supported by all references to bookmark's or other reference fields
        are:
          - 'number': displays the list label of the referenced item. [...]
          - 'number-all-superior': displays the list label of the referenced
            item and adds the contents of all list labels of superior levels
            in front of it. [...]
          - 'number-no-superior': displays the contents of the list label of
            the referenced item.
    """

    _tag = "text:reference-ref"
    _properties = (PropDef("name", "text:ref-name"),)
    FORMAT_ALLOWED = (
        "chapter",
        "direction",
        "page",
        "text",
        "caption",
        "category-and-value",
        "value",
        "number",
        "number-all-superior",
        "number-no-superior",
    )

    def __init__(self, name: str = "", ref_format: str = "", **kwargs: Any) -> None:
        """Create a reference to a content marked by a reference mark
        "text:reference-ref".

        An actual reference mark with the provided name should exist.

        Consider using: odfdo.paragraph.insert_reference()

        The text:ref-name attribute identifies a "text:reference-mark" or
        "text:referencemark-start" element by the value of that element's
        text:name attribute.
        If ref_format is 'text', the current text content of the reference_mark
        is retrieved.

        Args:

            name -- str : name of the reference mark

            ref_format -- str : format of the field. Default is 'page', allowed
                            values are 'chapter', 'direction', 'page', 'text',
                            'caption', 'category-and-value', 'value', 'number',
                            'number-all-superior', 'number-no-superior'.
        """
        super().__init__(**kwargs)
        if self._do_init:
            self.name = name
            self.ref_format = ref_format

    @property
    def ref_format(self) -> str | None:
        reference = self.get_attribute("text:reference-format")
        if isinstance(reference, str):
            return reference
        return None

    @ref_format.setter
    def ref_format(self, ref_format: str) -> None:
        """Set the text:reference-format attribute.

        Args:

            ref_format -- str
        """
        if not ref_format or ref_format not in self.FORMAT_ALLOWED:
            ref_format = "page"
        self.set_attribute("text:reference-format", ref_format)

    def update(self) -> None:
        """Update the content of the reference text field.

        Currently only 'text' format is implemented. Other values, for example
        the 'page' text field, may need to be refreshed through a visual ODF
        parser.
        """
        ref_format = self.ref_format
        if ref_format != "text":
            # only 'text' is implemented
            return None
        body: Body | Element = self.document_body or self.root
        name = self.name
        if hasattr(body, "get_reference_mark"):
            reference = body.get_reference_mark(name=name)
            if isinstance(reference, ReferenceMarkStart):
                self.text = reference.referenced_text()


Reference._define_attribut_property()


class ReferenceMark(Element):
    """A point reference, "text:reference-mark".

    A point reference marks a position in text and is represented by a single
    "text:reference-mark" element.
    """

    _tag = "text:reference-mark"
    _properties = (PropDef("name", "text:name"),)

    def __init__(self, name: str = "", **kwargs: Any) -> None:
        """A point reference "text:reference-mark".

        A point reference marks a position in text and is
        represented by a single "text:reference-mark" element.
        Consider using the wrapper: odfdo.paragraph.set_reference_mark()

        Args:

            name -- str
        """
        super().__init__(**kwargs)
        if self._do_init:
            self.name = name


ReferenceMark._define_attribut_property()


class ReferenceMarkEnd(Element):
    """End of a range reference, "text:reference-mark-end"."""

    _tag = "text:reference-mark-end"
    _properties = (PropDef("name", "text:name"),)

    def __init__(self, name: str = "", **kwargs: Any) -> None:
        """The "text:reference-mark-end" element represent the end of a range
        reference.
        Consider using the wrappers: odfdo.paragraph.set_reference_mark() and
        odfdo.paragraph.set_reference_mark_end()

        Args:

            name -- str
        """
        super().__init__(**kwargs)
        if self._do_init:
            self.name = name

    def referenced_text(self) -> str:
        """Return the text between reference-mark-start and reference-mark-
        end.
        """
        name = self.name
        request = (
            f"//text()"
            f"[preceding::text:reference-mark-start[@text:name='{name}'] "
            f"and following::text:reference-mark-end[@text:name='{name}']]"
        )
        result = " ".join(str(x) for x in self.xpath(request))
        return result


ReferenceMarkEnd._define_attribut_property()


class ReferenceMarkStart(Element):
    """Start of a range reference, "text:reference-mark-start"."""

    _tag = "text:reference-mark-start"
    _properties = (PropDef("name", "text:name"),)

    def __init__(self, name: str = "", **kwargs: Any) -> None:
        """The "text:reference-mark-start" element represent the start of a range
        reference.
        Consider using the wrapper: odfdo.paragraph.set_reference_mark()

        Args:

            name -- str
        """
        super().__init__(**kwargs)
        if self._do_init:
            self.name = name

    def referenced_text(self) -> str:
        """Return the text between reference-mark-start and reference-mark-
        end.
        """
        name = self.name
        request = (
            f"//text()"
            f"[preceding::text:reference-mark-start[@text:name='{name}'] "
            f"and following::text:reference-mark-end[@text:name='{name}']]"
        )
        result = " ".join(str(x) for x in self.xpath(request))
        return result

    def get_referenced(
        self,
        no_header: bool = False,
        clean: bool = True,
        as_xml: bool = False,
        as_list: bool = False,
    ) -> Element | list | str | None:
        """Return the document content between the start and end tags of the
        reference. The content returned by this method can spread over several
        headers and paragraphs. By default, the content is returned as an
        "office:text" odf element.

        Args:

            no_header -- boolean (default to False), translate existing headers
                         tags "text:h" into paragraphs "text:p".

            clean -- boolean (default to True), suppress unwanted tags. Striped
                     tags are : 'text:change', 'text:change-start',
                     'text:change-end', 'text:reference-mark',
                     'text:reference-mark-start', 'text:reference-mark-end'.

            as_xml -- boolean (default to False), format the returned content as
                      a XML string (serialization).

            as_list -- boolean (default to False), do not embed the returned
                       content in a "office:text'" element, instead simply
                       return a raw list of odf elements.
        """
        if self.parent is None:
            raise ValueError(
                "Reference need some upper document part"
            )  # pragma: nocover
        body: Body | Element = self.document_body or self.parent
        if hasattr(body, "get_reference_mark_end"):
            end = body.get_reference_mark_end(name=self.name)
        else:
            end = None
        if end is None:
            raise ValueError("No reference-end found")
        content_list = elements_between(
            body, self, end, as_text=False, no_header=no_header, clean=clean
        )
        if as_list:
            return content_list
        referenced = Element.from_tag("office:text")
        for chunk in content_list:
            referenced.append(chunk)
        if as_xml:
            return referenced.serialize()
        else:
            return referenced

    def delete(self, child: Element | None = None, keep_tail: bool = True) -> None:
        """Delete the given element from the XML tree. If no element is given,
        "self" is deleted. The XML library may allow to continue to use an
        element now "orphan" as long as you have a reference to it.

        For odf_reference_mark_start : delete the reference-end tag if exists.

        Args:

            child -- Element

            keep_tail -- boolean (default to True), True for most usages.
        """
        if child is not None:  # act like normal delete
            return super().delete(child, keep_tail)  # pragma: nocover
        name = self.name
        if self.parent is None:
            raise ValueError("Can't delete the root element")  # pragma: nocover
        body: Body | Element = self.document_body or self.parent
        if hasattr(body, "get_reference_mark_end"):
            ref_end = body.get_reference_mark_end(name=name)
        else:
            ref_end = None
        if ref_end:  # pragma: nocover
            ref_end.delete()
        # act like normal delete
        return super().delete()


ReferenceMarkStart._define_attribut_property()


def strip_references(element: Element) -> Element | list:
    """Remove all the 'text:reference-ref' tags of the element, keeping inner
    sub elements (for example the referenced value if format is 'text').

    Nota : using the .delete() on the reference mark will delete inner content.
    """
    to_strip = ("text:reference-ref",)
    return strip_tags(element, to_strip)


def remove_all_reference_marks(element: Element) -> Element | list:
    """Remove all the 'text:reference-mark', 'text:reference-mark-start', and
    'text:reference-mark-end' tags of the element, keeping inner sub elements.

    Nota : using the .delete() on the reference mark will delete inner content.
    """
    to_strip = (
        "text:reference-mark",
        "text:reference-mark-start",
        "text:reference-mark-end",
    )
    return strip_tags(element, to_strip)


def remove_reference_mark(
    element: Element,
    position: int = 0,
    name: str | None = None,
) -> None:
    """Remove the 'text:reference-mark', 'text:reference-mark-start', and
    'text:reference-mark-end' tags of the element, identified by name or
    position, keeping inner sub elements.

    Nota : using the .delete() on the reference mark will delete inner content.
    """
    if hasattr(element, "get_reference_mark"):
        start_ref = element.get_reference_mark(position=position, name=name)
    else:
        start_ref = None
    if hasattr(element, "get_reference_mark_end"):
        end_ref = element.get_reference_mark_end(position=position, name=name)
    else:
        end_ref = None
    to_strip: list[ReferenceMark | ReferenceMarkStart | ReferenceMarkEnd] = []
    if start_ref:
        to_strip.append(start_ref)
    if end_ref:
        to_strip.append(end_ref)
    strip_elements(element, to_strip)


register_element_class(Reference)
register_element_class(ReferenceMark)
register_element_class(ReferenceMarkStart)
register_element_class(ReferenceMarkEnd)
