/**
 * "UltimateDailyWallpaper" Copyright (C) 2025 Patrice Coni
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "setwallpaper.h"

#include <QProcess>
#include <QDebug>
#include <QPixmap>
#include <QApplication>
#include <QFile>
#include <QSettings>
#include <QTextStream>
#include <QDir>
#include <QThread>

setWallpaper::setWallpaper()
{
}

void setWallpaper::_set_wallpaper(QString _wallpaperfile, int _Parameter, int _change_lockscreen_wallpaper)
{
    if(_wallpaperfile.contains("\'"))
    {
        _wallpaperfile.replace("\'", "\'\\'\'");
    }

    if(_wallpaperfile.contains("\x22"))
    {
        _wallpaperfile.replace("\x22", "\x5C\x22");
    }

    if(!_wallpaperfile.isEmpty())
    {
        QString command;
        QString content;
        QStringList arguments;

        switch (_Parameter)
        {
#if defined(Q_OS_LINUX)

            case 0:
                // Budgie
                command = "gsettings";
                arguments << "set"
                        << "org.gnome.desktop.background"
                        << "picture-uri"
                        << "file://" + _wallpaperfile;
            break;

            case 1:
                // Cinnamon
                command = "gsettings";
                arguments << "set"
                        << "org.cinnamon.desktop.background"
                        << "picture-uri"
                        << "file://" + _wallpaperfile;
            break;

            case 2:
                // LXDE
                command = "pcmanfm";
                arguments << "--set-wallpaper"
                        << _wallpaperfile;
            break;

            case 3:
                // MATE
                command = "gsettings";
                arguments << "set"
                        << "org.mate.background"
                        << "picture-filename"
                        << _wallpaperfile;
            break;

            case 4:
                // Trinity Desktop Environment (TDE)
                command = "dcop";
                arguments << "kdesktop"
                        << "KBackgroundIface"
                        << "setWallpaper"
                        << _wallpaperfile
                        << "8";
            break;

            case 5:
            {
                // KDE Plasma 5
                command = "plasma-apply-wallpaperimage";
                arguments << _wallpaperfile;

                if (_change_lockscreen_wallpaper == 2)
                {
                    doProcess(command, arguments, _Parameter);

                    command = "kwriteconfig5";
                    arguments.clear();
                    arguments << "--file" << "kscreenlockerrc"
                            << "--group" << "Greeter"
                            << "--group" << "Wallpaper"
                            << "--group" << "org.kde.image"
                            << "--group" << "General"
                            << "--key" << "Image"
                            << _wallpaperfile;
                }
            }
            break;

            case 6:
                // LXQt
                command = "pcmanfm-qt";
                arguments << "--set-wallpaper"
                        << _wallpaperfile
                        << "--wallpaper-mode=stretch";
            break;

            case 7:
                // Deepin Desktop Environment (DDE)
                command = "gsettings";
                arguments << "set"
                        << "com.deepin.wrap.gnome.desktop.background"
                        << "picture-uri"
                        << "file://" + _wallpaperfile;
            break;

            case 8:
            {
                // GNOME 3 / Flashback
                QStringList schemes = { "picture-uri", "picture-uri-dark" };

                for (const QString &scheme : schemes)
                {
                    command = "gsettings";
                    arguments.clear();
                    arguments << "set"
                            << "org.gnome.desktop.background"
                            << scheme
                            << "file://" + _wallpaperfile;

                    doProcess(command, arguments, _Parameter);
                }

                if (_change_lockscreen_wallpaper == 2)
                {
                    for (const QString &scheme : schemes)
                    {
                        command = "gsettings";
                        arguments.clear();
                        arguments << "set"
                                << "org.gnome.desktop.screensaver"
                                << scheme
                                << "file://" + _wallpaperfile;

                        doProcess(command, arguments, _Parameter);
                    }
                }
                return;
            }
            break;

            case 9:
                // KDE Plasma 6
                command = "plasma-apply-wallpaperimage";
                arguments << _wallpaperfile;

                if (_change_lockscreen_wallpaper == 2)
                {
                    doProcess(command, arguments, _Parameter);

                    command = "kwriteconfig6";
                    arguments.clear();
                    arguments << "--file" << "kscreenlockerrc"
                            << "--group" << "Greeter"
                            << "--group" << "Wallpaper"
                            << "--group" << "org.kde.image"
                            << "--group" << "General"
                            << "--key" << "Image"
                            << _wallpaperfile;
                }
            break;
        }

#elif defined(Q_OS_MACOS)
        case 0:
            QString command = "osascript";
            QStringList arguments;
            arguments << "-e" << "tell application \"System Events\" to tell every desktop to set picture to \"" + _wallpaperfile + "\"";

            int exitCode = doProcess(command, arguments, _Parameter);

            if (exitCode != 0)
            {
                qDebug() << tr("Error: Failed to set wallpaper on macOS.");
            }
            return;
        break;
        }
#endif

        if(!(_Parameter==2) && !(_Parameter==8))
        {
            if(doProcess(command, arguments, _Parameter)==255)
            {
                qDebug() << tr("Error - Failed to set wallpaper.");
            }
            command.clear();
        }
    }
}

int setWallpaper::doProcess(QString command, QStringList arguments, int used_desktop)
{
    Q_UNUSED(used_desktop);

    QProcess proc;

    proc.setEnvironment(QProcess::systemEnvironment());
    proc.start(command, arguments);

    if (!proc.waitForStarted()) {
        qWarning() << "Failed to run command!";
        return 255;
    }

    proc.waitForFinished();

    return proc.exitCode();
}
