"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOrCreateInstallationId = getOrCreateInstallationId;
const crypto_1 = require("crypto");
const fs = require("fs");
const path = require("path");
const util_1 = require("../../util");
const INSTALLATION_ID_PATH = path.join((0, util_1.cdkCacheDir)(), 'installation-id.json');
/**
 * Get or create installation id
 */
async function getOrCreateInstallationId(ioHelper) {
    try {
        // Create the cache directory if it doesn't exist
        if (!fs.existsSync(path.dirname(INSTALLATION_ID_PATH))) {
            fs.mkdirSync(path.dirname(INSTALLATION_ID_PATH), { recursive: true });
        }
        // Check if the installation ID file exists
        if (fs.existsSync(INSTALLATION_ID_PATH)) {
            const cachedId = fs.readFileSync(INSTALLATION_ID_PATH, 'utf-8').trim();
            // Validate that the cached ID is a valid UUID
            const UUID_REGEX = /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
            if (UUID_REGEX.test(cachedId)) {
                return cachedId;
            }
            // If invalid, fall through to create a new one
        }
        // Create a new installation ID
        const newId = (0, crypto_1.randomUUID)();
        try {
            fs.writeFileSync(INSTALLATION_ID_PATH, newId);
        }
        catch (e) {
            // If we can't write the file, still return the generated ID
            // but log a trace message about the failure
            await ioHelper.defaults.trace(`Failed to write installation ID to ${INSTALLATION_ID_PATH}: ${e}`);
        }
        return newId;
    }
    catch (e) {
        // If anything goes wrong, generate a temporary ID for this session
        // and log a trace message about the failure
        await ioHelper.defaults.trace(`Error getting installation ID: ${e}`);
        return (0, crypto_1.randomUUID)();
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5zdGFsbGF0aW9uLWlkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW5zdGFsbGF0aW9uLWlkLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBV0EsOERBbUNDO0FBOUNELG1DQUFvQztBQUNwQyx5QkFBeUI7QUFDekIsNkJBQTZCO0FBRTdCLHFDQUF5QztBQUV6QyxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBQSxrQkFBVyxHQUFFLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztBQUU5RTs7R0FFRztBQUNJLEtBQUssVUFBVSx5QkFBeUIsQ0FBQyxRQUFrQjtJQUNoRSxJQUFJLENBQUM7UUFDSCxpREFBaUQ7UUFDakQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUN2RCxFQUFFLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3hFLENBQUM7UUFFRCwyQ0FBMkM7UUFDM0MsSUFBSSxFQUFFLENBQUMsVUFBVSxDQUFDLG9CQUFvQixDQUFDLEVBQUUsQ0FBQztZQUN4QyxNQUFNLFFBQVEsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLG9CQUFvQixFQUFFLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDO1lBRXZFLDhDQUE4QztZQUM5QyxNQUFNLFVBQVUsR0FBRyxpRUFBaUUsQ0FBQztZQUNyRixJQUFJLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQztnQkFDOUIsT0FBTyxRQUFRLENBQUM7WUFDbEIsQ0FBQztZQUNELCtDQUErQztRQUNqRCxDQUFDO1FBRUQsK0JBQStCO1FBQy9CLE1BQU0sS0FBSyxHQUFHLElBQUEsbUJBQVUsR0FBRSxDQUFDO1FBQzNCLElBQUksQ0FBQztZQUNILEVBQUUsQ0FBQyxhQUFhLENBQUMsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDaEQsQ0FBQztRQUFDLE9BQU8sQ0FBTSxFQUFFLENBQUM7WUFDaEIsNERBQTREO1lBQzVELDRDQUE0QztZQUM1QyxNQUFNLFFBQVEsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLHNDQUFzQyxvQkFBb0IsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3BHLENBQUM7UUFDRCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFBQyxPQUFPLENBQU0sRUFBRSxDQUFDO1FBQ2hCLG1FQUFtRTtRQUNuRSw0Q0FBNEM7UUFDNUMsTUFBTSxRQUFRLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUNyRSxPQUFPLElBQUEsbUJBQVUsR0FBRSxDQUFDO0lBQ3RCLENBQUM7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgcmFuZG9tVVVJRCB9IGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHR5cGUgeyBJb0hlbHBlciB9IGZyb20gJy4uLy4uL2FwaS1wcml2YXRlJztcbmltcG9ydCB7IGNka0NhY2hlRGlyIH0gZnJvbSAnLi4vLi4vdXRpbCc7XG5cbmNvbnN0IElOU1RBTExBVElPTl9JRF9QQVRIID0gcGF0aC5qb2luKGNka0NhY2hlRGlyKCksICdpbnN0YWxsYXRpb24taWQuanNvbicpO1xuXG4vKipcbiAqIEdldCBvciBjcmVhdGUgaW5zdGFsbGF0aW9uIGlkXG4gKi9cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBnZXRPckNyZWF0ZUluc3RhbGxhdGlvbklkKGlvSGVscGVyOiBJb0hlbHBlcikge1xuICB0cnkge1xuICAgIC8vIENyZWF0ZSB0aGUgY2FjaGUgZGlyZWN0b3J5IGlmIGl0IGRvZXNuJ3QgZXhpc3RcbiAgICBpZiAoIWZzLmV4aXN0c1N5bmMocGF0aC5kaXJuYW1lKElOU1RBTExBVElPTl9JRF9QQVRIKSkpIHtcbiAgICAgIGZzLm1rZGlyU3luYyhwYXRoLmRpcm5hbWUoSU5TVEFMTEFUSU9OX0lEX1BBVEgpLCB7IHJlY3Vyc2l2ZTogdHJ1ZSB9KTtcbiAgICB9XG5cbiAgICAvLyBDaGVjayBpZiB0aGUgaW5zdGFsbGF0aW9uIElEIGZpbGUgZXhpc3RzXG4gICAgaWYgKGZzLmV4aXN0c1N5bmMoSU5TVEFMTEFUSU9OX0lEX1BBVEgpKSB7XG4gICAgICBjb25zdCBjYWNoZWRJZCA9IGZzLnJlYWRGaWxlU3luYyhJTlNUQUxMQVRJT05fSURfUEFUSCwgJ3V0Zi04JykudHJpbSgpO1xuXG4gICAgICAvLyBWYWxpZGF0ZSB0aGF0IHRoZSBjYWNoZWQgSUQgaXMgYSB2YWxpZCBVVUlEXG4gICAgICBjb25zdCBVVUlEX1JFR0VYID0gL15bMC05YS1mXXs4fS1bMC05YS1mXXs0fS1bMC05YS1mXXs0fS1bMC05YS1mXXs0fS1bMC05YS1mXXsxMn0kL2k7XG4gICAgICBpZiAoVVVJRF9SRUdFWC50ZXN0KGNhY2hlZElkKSkge1xuICAgICAgICByZXR1cm4gY2FjaGVkSWQ7XG4gICAgICB9XG4gICAgICAvLyBJZiBpbnZhbGlkLCBmYWxsIHRocm91Z2ggdG8gY3JlYXRlIGEgbmV3IG9uZVxuICAgIH1cblxuICAgIC8vIENyZWF0ZSBhIG5ldyBpbnN0YWxsYXRpb24gSURcbiAgICBjb25zdCBuZXdJZCA9IHJhbmRvbVVVSUQoKTtcbiAgICB0cnkge1xuICAgICAgZnMud3JpdGVGaWxlU3luYyhJTlNUQUxMQVRJT05fSURfUEFUSCwgbmV3SWQpO1xuICAgIH0gY2F0Y2ggKGU6IGFueSkge1xuICAgICAgLy8gSWYgd2UgY2FuJ3Qgd3JpdGUgdGhlIGZpbGUsIHN0aWxsIHJldHVybiB0aGUgZ2VuZXJhdGVkIElEXG4gICAgICAvLyBidXQgbG9nIGEgdHJhY2UgbWVzc2FnZSBhYm91dCB0aGUgZmFpbHVyZVxuICAgICAgYXdhaXQgaW9IZWxwZXIuZGVmYXVsdHMudHJhY2UoYEZhaWxlZCB0byB3cml0ZSBpbnN0YWxsYXRpb24gSUQgdG8gJHtJTlNUQUxMQVRJT05fSURfUEFUSH06ICR7ZX1gKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ld0lkO1xuICB9IGNhdGNoIChlOiBhbnkpIHtcbiAgICAvLyBJZiBhbnl0aGluZyBnb2VzIHdyb25nLCBnZW5lcmF0ZSBhIHRlbXBvcmFyeSBJRCBmb3IgdGhpcyBzZXNzaW9uXG4gICAgLy8gYW5kIGxvZyBhIHRyYWNlIG1lc3NhZ2UgYWJvdXQgdGhlIGZhaWx1cmVcbiAgICBhd2FpdCBpb0hlbHBlci5kZWZhdWx0cy50cmFjZShgRXJyb3IgZ2V0dGluZyBpbnN0YWxsYXRpb24gSUQ6ICR7ZX1gKTtcbiAgICByZXR1cm4gcmFuZG9tVVVJRCgpO1xuICB9XG59XG4iXX0=