///////////////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2021 Edouard Griffiths, F4EXB <f4exb06@gmail.com>                   //
//                                                                                   //
// This program is free software; you can redistribute it and/or modify              //
// it under the terms of the GNU General Public License as published by              //
// the Free Software Foundation as version 3 of the License, or                      //
// (at your option) any later version.                                               //
//                                                                                   //
// This program is distributed in the hope that it will be useful,                   //
// but WITHOUT ANY WARRANTY; without even the implied warranty of                    //
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                      //
// GNU General Public License V3 for more details.                                   //
//                                                                                   //
// You should have received a copy of the GNU General Public License                 //
// along with this program. If not, see <http://www.gnu.org/licenses/>.              //
///////////////////////////////////////////////////////////////////////////////////////
/*
DVB-T2 LDPC parity bit address tables

Entries below copied from:
https://www.etsi.org/deliver/etsi_en/302700_302799/302755/01.04.01_60/en_302755v010401p.pdf

Copyright 2018 Ahmet Inan <xdsopl@gmail.com>
*/

#ifndef DVB_T2_TABLES_HH
#define DVB_T2_TABLES_HH

namespace ldpctool {

struct DVB_T2_TABLE_A1
{
	static const int M = 360;
	static const int N = 64800;
	static const int K = 32400;
	static const int LINKS_MIN_CN = 6;
	static const int LINKS_MAX_CN = 7;
	static const int LINKS_TOTAL = 226799;
	static const int DEG_MAX = 8;
	static constexpr int DEG[] = {
		8, 3, 0
	};
	static constexpr int LEN[] = {
		36, 54, 0
	};
	static constexpr int POS[] = {
		54,	9318,	14392,	27561,	26909,	10219,	2534,	8597,
		55,	7263,	4635,	2530,	28130,	3033,	23830,	3651,
		56,	24731,	23583,	26036,	17299,	5750,	792,	9169,
		57,	5811,	26154,	18653,	11551,	15447,	13685,	16264,
		58,	12610,	11347,	28768,	2792,	3174,	29371,	12997,
		59,	16789,	16018,	21449,	6165,	21202,	15850,	3186,
		60,	31016,	21449,	17618,	6213,	12166,	8334,	18212,
		61,	22836,	14213,	11327,	5896,	718,	11727,	9308,
		62,	2091,	24941,	29966,	23634,	9013,	15587,	5444,
		63,	22207,	3983,	16904,	28534,	21415,	27524,	25912,
		64,	25687,	4501,	22193,	14665,	14798,	16158,	5491,
		65,	4520,	17094,	23397,	4264,	22370,	16941,	21526,
		66,	10490,	6182,	32370,	9597,	30841,	25954,	2762,
		67,	22120,	22865,	29870,	15147,	13668,	14955,	19235,
		68,	6689,	18408,	18346,	9918,	25746,	5443,	20645,
		69,	29982,	12529,	13858,	4746,	30370,	10023,	24828,
		70,	1262,	28032,	29888,	13063,	24033,	21951,	7863,
		71,	6594,	29642,	31451,	14831,	9509,	9335,	31552,
		72,	1358,	6454,	16633,	20354,	24598,	624,	5265,
		73,	19529,	295,	18011,	3080,	13364,	8032,	15323,
		74,	11981,	1510,	7960,	21462,	9129,	11370,	25741,
		75,	9276,	29656,	4543,	30699,	20646,	21921,	28050,
		76,	15975,	25634,	5520,	31119,	13715,	21949,	19605,
		77,	18688,	4608,	31755,	30165,	13103,	10706,	29224,
		78,	21514,	23117,	12245,	26035,	31656,	25631,	30699,
		79,	9674,	24966,	31285,	29908,	17042,	24588,	31857,
		80,	21856,	27777,	29919,	27000,	14897,	11409,	7122,
		81,	29773,	23310,	263,	4877,	28622,	20545,	22092,
		82,	15605,	5651,	21864,	3967,	14419,	22757,	15896,
		83,	30145,	1759,	10139,	29223,	26086,	10556,	5098,
		84,	18815,	16575,	2936,	24457,	26738,	6030,	505,
		85,	30326,	22298,	27562,	20131,	26390,	6247,	24791,
		86,	928,	29246,	21246,	12400,	15311,	32309,	18608,
		87,	20314,	6025,	26689,	16302,	2296,	3244,	19613,
		88,	6237,	11943,	22851,	15642,	23857,	15112,	20947,
		89,	26403,	25168,	19038,	18384,	8882,	12719,	7093,
		0,	14567,	24965,
		1,	3908,	100,
		2,	10279,	240,
		3,	24102,	764,
		4,	12383,	4173,
		5,	13861,	15918,
		6,	21327,	1046,
		7,	5288,	14579,
		8,	28158,	8069,
		9,	16583,	11098,
		10,	16681,	28363,
		11,	13980,	24725,
		12,	32169,	17989,
		13,	10907,	2767,
		14,	21557,	3818,
		15,	26676,	12422,
		16,	7676,	8754,
		17,	14905,	20232,
		18,	15719,	24646,
		19,	31942,	8589,
		20,	19978,	27197,
		21,	27060,	15071,
		22,	6071,	26649,
		23,	10393,	11176,
		24,	9597,	13370,
		25,	7081,	17677,
		26,	1433,	19513,
		27,	26925,	9014,
		28,	19202,	8900,
		29,	18152,	30647,
		30,	20803,	1737,
		31,	11804,	25221,
		32,	31683,	17783,
		33,	29694,	9345,
		34,	12280,	26611,
		35,	6526,	26122,
		36,	26165,	11241,
		37,	7666,	26962,
		38,	16290,	8480,
		39,	11774,	10120,
		40,	30051,	30426,
		41,	1335,	15424,
		42,	6865,	17742,
		43,	31779,	12489,
		44,	32120,	21001,
		45,	14508,	6996,
		46,	979,	25024,
		47,	4554,	21896,
		48,	7989,	21777,
		49,	4972,	20661,
		50,	6612,	2730,
		51,	12742,	4418,
		52,	29194,	595,
		53,	19267,	20113,
	};
};

struct DVB_T2_TABLE_A2
{
	static const int M = 360;
	static const int N = 64800;
	static const int K = 38880;
	static const int LINKS_MIN_CN = 10;
	static const int LINKS_MAX_CN = 11;
	static const int LINKS_TOTAL = 285119;
	static const int DEG_MAX = 12;
	static constexpr int DEG[] = {
		12, 3, 0
	};
	static constexpr int LEN[] = {
		36, 72, 0
	};
	static constexpr int POS[] = {
		22422,	10282,	11626,	19997,	11161,	2922,	3122,	99,	5625,	17064,	8270,	179,
		25087,	16218,	17015,	828,	20041,	25656,	4186,	11629,	22599,	17305,	22515,	6463,
		11049,	22853,	25706,	14388,	5500,	19245,	8732,	2177,	13555,	11346,	17265,	3069,
		16581,	22225,	12563,	19717,	23577,	11555,	25496,	6853,	25403,	5218,	15925,	21766,
		16529,	14487,	7643,	10715,	17442,	11119,	5679,	14155,	24213,	21000,	1116,	15620,
		5340,	8636,	16693,	1434,	5635,	6516,	9482,	20189,	1066,	15013,	25361,	14243,
		18506,	22236,	20912,	8952,	5421,	15691,	6126,	21595,	500,	6904,	13059,	6802,
		8433,	4694,	5524,	14216,	3685,	19721,	25420,	9937,	23813,	9047,	25651,	16826,
		21500,	24814,	6344,	17382,	7064,	13929,	4004,	16552,	12818,	8720,	5286,	2206,
		22517,	2429,	19065,	2921,	21611,	1873,	7507,	5661,	23006,	23128,	20543,	19777,
		1770,	4636,	20900,	14931,	9247,	12340,	11008,	12966,	4471,	2731,	16445,	791,
		6635,	14556,	18865,	22421,	22124,	12697,	9803,	25485,	7744,	18254,	11313,	9004,
		19982,	23963,	18912,	7206,	12500,	4382,	20067,	6177,	21007,	1195,	23547,	24837,
		756,	11158,	14646,	20534,	3647,	17728,	11676,	11843,	12937,	4402,	8261,	22944,
		9306,	24009,	10012,	11081,	3746,	24325,	8060,	19826,	842,	8836,	2898,	5019,
		7575,	7455,	25244,	4736,	14400,	22981,	5543,	8006,	24203,	13053,	1120,	5128,
		3482,	9270,	13059,	15825,	7453,	23747,	3656,	24585,	16542,	17507,	22462,	14670,
		15627,	15290,	4198,	22748,	5842,	13395,	23918,	16985,	14929,	3726,	25350,	24157,
		24896,	16365,	16423,	13461,	16615,	8107,	24741,	3604,	25904,	8716,	9604,	20365,
		3729,	17245,	18448,	9862,	20831,	25326,	20517,	24618,	13282,	5099,	14183,	8804,
		16455,	17646,	15376,	18194,	25528,	1777,	6066,	21855,	14372,	12517,	4488,	17490,
		1400,	8135,	23375,	20879,	8476,	4084,	12936,	25536,	22309,	16582,	6402,	24360,
		25119,	23586,	128,	4761,	10443,	22536,	8607,	9752,	25446,	15053,	1856,	4040,
		377,	21160,	13474,	5451,	17170,	5938,	10256,	11972,	24210,	17833,	22047,	16108,
		13075,	9648,	24546,	13150,	23867,	7309,	19798,	2988,	16858,	4825,	23950,	15125,
		20526,	3553,	11525,	23366,	2452,	17626,	19265,	20172,	18060,	24593,	13255,	1552,
		18839,	21132,	20119,	15214,	14705,	7096,	10174,	5663,	18651,	19700,	12524,	14033,
		4127,	2971,	17499,	16287,	22368,	21463,	7943,	18880,	5567,	8047,	23363,	6797,
		10651,	24471,	14325,	4081,	7258,	4949,	7044,	1078,	797,	22910,	20474,	4318,
		21374,	13231,	22985,	5056,	3821,	23718,	14178,	9978,	19030,	23594,	8895,	25358,
		6199,	22056,	7749,	13310,	3999,	23697,	16445,	22636,	5225,	22437,	24153,	9442,
		7978,	12177,	2893,	20778,	3175,	8645,	11863,	24623,	10311,	25767,	17057,	3691,
		20473,	11294,	9914,	22815,	2574,	8439,	3699,	5431,	24840,	21908,	16088,	18244,
		8208,	5755,	19059,	8541,	24924,	6454,	11234,	10492,	16406,	10831,	11436,	9649,
		16264,	11275,	24953,	2347,	12667,	19190,	7257,	7174,	24819,	2938,	2522,	11749,
		3627,	5969,	13862,	1538,	23176,	6353,	2855,	17720,	2472,	7428,	573,	15036,
		0,	18539,	18661,
		1,	10502,	3002,
		2,	9368,	10761,
		3,	12299,	7828,
		4,	15048,	13362,
		5,	18444,	24640,
		6,	20775,	19175,
		7,	18970,	10971,
		8,	5329,	19982,
		9,	11296,	18655,
		10,	15046,	20659,
		11,	7300,	22140,
		12,	22029,	14477,
		13,	11129,	742,
		14,	13254,	13813,
		15,	19234,	13273,
		16,	6079,	21122,
		17,	22782,	5828,
		18,	19775,	4247,
		19,	1660,	19413,
		20,	4403,	3649,
		21,	13371,	25851,
		22,	22770,	21784,
		23,	10757,	14131,
		24,	16071,	21617,
		25,	6393,	3725,
		26,	597,	19968,
		27,	5743,	8084,
		28,	6770,	9548,
		29,	4285,	17542,
		30,	13568,	22599,
		31,	1786,	4617,
		32,	23238,	11648,
		33,	19627,	2030,
		34,	13601,	13458,
		35,	13740,	17328,
		36,	25012,	13944,
		37,	22513,	6687,
		38,	4934,	12587,
		39,	21197,	5133,
		40,	22705,	6938,
		41,	7534,	24633,
		42,	24400,	12797,
		43,	21911,	25712,
		44,	12039,	1140,
		45,	24306,	1021,
		46,	14012,	20747,
		47,	11265,	15219,
		48,	4670,	15531,
		49,	9417,	14359,
		50,	2415,	6504,
		51,	24964,	24690,
		52,	14443,	8816,
		53,	6926,	1291,
		54,	6209,	20806,
		55,	13915,	4079,
		56,	24410,	13196,
		57,	13505,	6117,
		58,	9869,	8220,
		59,	1570,	6044,
		60,	25780,	17387,
		61,	20671,	24913,
		62,	24558,	20591,
		63,	12402,	3702,
		64,	8314,	1357,
		65,	20071,	14616,
		66,	17014,	3688,
		67,	19837,	946,
		68,	15195,	12136,
		69,	7758,	22808,
		70,	3564,	2925,
		71,	3434,	7769,
	};
};

struct DVB_T2_TABLE_A3
{
	static const int M = 360;
	static const int N = 64800;
	static const int K = 43200;
	static const int LINKS_MIN_CN = 9;
	static const int LINKS_MAX_CN = 10;
	static const int LINKS_TOTAL = 215999;
	static const int DEG_MAX = 13;
	static constexpr int DEG[] = {
		13, 3, 0
	};
	static constexpr int LEN[] = {
		12, 108, 0
	};
	static constexpr int POS[] = {
		317,	2255,	2324,	2723,	3538,	3576,	6194,	6700,	9101,	10057,	12739,	17407,	21039,
		1958,	2007,	3294,	4394,	12762,	14505,	14593,	14692,	16522,	17737,	19245,	21272,	21379,
		127,	860,	5001,	5633,	8644,	9282,	12690,	14644,	17553,	19511,	19681,	20954,	21002,
		2514,	2822,	5781,	6297,	8063,	9469,	9551,	11407,	11837,	12985,	15710,	20236,	20393,
		1565,	3106,	4659,	4926,	6495,	6872,	7343,	8720,	15785,	16434,	16727,	19884,	21325,
		706,	3220,	8568,	10896,	12486,	13663,	16398,	16599,	19475,	19781,	20625,	20961,	21335,
		4257,	10449,	12406,	14561,	16049,	16522,	17214,	18029,	18033,	18802,	19062,	19526,	20748,
		412,	433,	558,	2614,	2978,	4157,	6584,	9320,	11683,	11819,	13024,	14486,	16860,
		777,	5906,	7403,	8550,	8717,	8770,	11436,	12846,	13629,	14755,	15688,	16392,	16419,
		4093,	5045,	6037,	7248,	8633,	9771,	10260,	10809,	11326,	12072,	17516,	19344,	19938,
		2120,	2648,	3155,	3852,	6888,	12258,	14821,	15359,	16378,	16437,	17791,	20614,	21025,
		1085,	2434,	5816,	7151,	8050,	9422,	10884,	12728,	15353,	17733,	18140,	18729,	20920,
		856,	1690,	12787,
		6532,	7357,	9151,
		4210,	16615,	18152,
		11494,	14036,	17470,
		2474,	10291,	10323,
		1778,	6973,	10739,
		4347,	9570,	18748,
		2189,	11942,	20666,
		3868,	7526,	17706,
		8780,	14796,	18268,
		160,	16232,	17399,
		1285,	2003,	18922,
		4658,	17331,	20361,
		2765,	4862,	5875,
		4565,	5521,	8759,
		3484,	7305,	15829,
		5024,	17730,	17879,
		7031,	12346,	15024,
		179,	6365,	11352,
		2490,	3143,	5098,
		2643,	3101,	21259,
		4315,	4724,	13130,
		594,	17365,	18322,
		5983,	8597,	9627,
		10837,	15102,	20876,
		10448,	20418,	21478,
		3848,	12029,	15228,
		708,	5652,	13146,
		5998,	7534,	16117,
		2098,	13201,	18317,
		9186,	14548,	17776,
		5246,	10398,	18597,
		3083,	4944,	21021,
		13726,	18495,	19921,
		6736,	10811,	17545,
		10084,	12411,	14432,
		1064,	13555,	17033,
		679,	9878,	13547,
		3422,	9910,	20194,
		3640,	3701,	10046,
		5862,	10134,	11498,
		5923,	9580,	15060,
		1073,	3012,	16427,
		5527,	20113,	20883,
		7058,	12924,	15151,
		9764,	12230,	17375,
		772,	7711,	12723,
		555,	13816,	15376,
		10574,	11268,	17932,
		15442,	17266,	20482,
		390,	3371,	8781,
		10512,	12216,	17180,
		4309,	14068,	15783,
		3971,	11673,	20009,
		9259,	14270,	17199,
		2947,	5852,	20101,
		3965,	9722,	15363,
		1429,	5689,	16771,
		6101,	6849,	12781,
		3676,	9347,	18761,
		350,	11659,	18342,
		5961,	14803,	16123,
		2113,	9163,	13443,
		2155,	9808,	12885,
		2861,	7988,	11031,
		7309,	9220,	20745,
		6834,	8742,	11977,
		2133,	12908,	14704,
		10170,	13809,	18153,
		13464,	14787,	14975,
		799,	1107,	3789,
		3571,	8176,	10165,
		5433,	13446,	15481,
		3351,	6767,	12840,
		8950,	8974,	11650,
		1430,	4250,	21332,
		6283,	10628,	15050,
		8632,	14404,	16916,
		6509,	10702,	16278,
		15900,	16395,	17995,
		8031,	18420,	19733,
		3747,	4634,	17087,
		4453,	6297,	16262,
		2792,	3513,	17031,
		14846,	20893,	21563,
		17220,	20436,	21337,
		275,	4107,	10497,
		3536,	7520,	10027,
		14089,	14943,	19455,
		1965,	3931,	21104,
		2439,	11565,	17932,
		154,	15279,	21414,
		10017,	11269,	16546,
		7169,	10161,	16928,
		10284,	16791,	20655,
		36,	3175,	8475,
		2605,	16269,	19290,
		8947,	9178,	15420,
		5687,	9156,	12408,
		8096,	9738,	14711,
		4935,	8093,	19266,
		2667,	10062,	15972,
		6389,	11318,	14417,
		8800,	18137,	18434,
		5824,	5927,	15314,
		6056,	13168,	15179,
		3284,	13138,	18919,
		13115,	17259,	17332,
	};
};

struct DVB_T2_TABLE_A4
{
	static const int M = 360;
	static const int N = 64800;
	static const int K = 48600;
	static const int LINKS_MIN_CN = 13;
	static const int LINKS_MAX_CN = 14;
	static const int LINKS_TOTAL = 226799;
	static const int DEG_MAX = 12;
	static constexpr int DEG[] = {
		12, 3, 0
	};
	static constexpr int LEN[] = {
		15, 120, 0
	};
	static constexpr int POS[] = {
		0,	6385,	7901,	14611,	13389,	11200,	3252,	5243,	2504,	2722,	821,	7374,
		1,	11359,	2698,	357,	13824,	12772,	7244,	6752,	15310,	852,	2001,	11417,
		2,	7862,	7977,	6321,	13612,	12197,	14449,	15137,	13860,	1708,	6399,	13444,
		3,	1560,	11804,	6975,	13292,	3646,	3812,	8772,	7306,	5795,	14327,	7866,
		4,	7626,	11407,	14599,	9689,	1628,	2113,	10809,	9283,	1230,	15241,	4870,
		5,	1610,	5699,	15876,	9446,	12515,	1400,	6303,	5411,	14181,	13925,	7358,
		6,	4059,	8836,	3405,	7853,	7992,	15336,	5970,	10368,	10278,	9675,	4651,
		7,	4441,	3963,	9153,	2109,	12683,	7459,	12030,	12221,	629,	15212,	406,
		8,	6007,	8411,	5771,	3497,	543,	14202,	875,	9186,	6235,	13908,	3563,
		9,	3232,	6625,	4795,	546,	9781,	2071,	7312,	3399,	7250,	4932,	12652,
		10,	8820,	10088,	11090,	7069,	6585,	13134,	10158,	7183,	488,	7455,	9238,
		11,	1903,	10818,	119,	215,	7558,	11046,	10615,	11545,	14784,	7961,	15619,
		12,	3655,	8736,	4917,	15874,	5129,	2134,	15944,	14768,	7150,	2692,	1469,
		13,	8316,	3820,	505,	8923,	6757,	806,	7957,	4216,	15589,	13244,	2622,
		14,	14463,	4852,	15733,	3041,	11193,	12860,	13673,	8152,	6551,	15108,	8758,
		15,	3149,	11981,
		16,	13416,	6906,
		17,	13098,	13352,
		18,	2009,	14460,
		19,	7207,	4314,
		20,	3312,	3945,
		21,	4418,	6248,
		22,	2669,	13975,
		23,	7571,	9023,
		24,	14172,	2967,
		25,	7271,	7138,
		26,	6135,	13670,
		27,	7490,	14559,
		28,	8657,	2466,
		29,	8599,	12834,
		30,	3470,	3152,
		31,	13917,	4365,
		32,	6024,	13730,
		33,	10973,	14182,
		34,	2464,	13167,
		35,	5281,	15049,
		36,	1103,	1849,
		37,	2058,	1069,
		38,	9654,	6095,
		39,	14311,	7667,
		40,	15617,	8146,
		41,	4588,	11218,
		42,	13660,	6243,
		43,	8578,	7874,
		44,	11741,	2686,
		0,	1022,	1264,
		1,	12604,	9965,
		2,	8217,	2707,
		3,	3156,	11793,
		4,	354,	1514,
		5,	6978,	14058,
		6,	7922,	16079,
		7,	15087,	12138,
		8,	5053,	6470,
		9,	12687,	14932,
		10,	15458,	1763,
		11,	8121,	1721,
		12,	12431,	549,
		13,	4129,	7091,
		14,	1426,	8415,
		15,	9783,	7604,
		16,	6295,	11329,
		17,	1409,	12061,
		18,	8065,	9087,
		19,	2918,	8438,
		20,	1293,	14115,
		21,	3922,	13851,
		22,	3851,	4000,
		23,	5865,	1768,
		24,	2655,	14957,
		25,	5565,	6332,
		26,	4303,	12631,
		27,	11653,	12236,
		28,	16025,	7632,
		29,	4655,	14128,
		30,	9584,	13123,
		31,	13987,	9597,
		32,	15409,	12110,
		33,	8754,	15490,
		34,	7416,	15325,
		35,	2909,	15549,
		36,	2995,	8257,
		37,	9406,	4791,
		38,	11111,	4854,
		39,	2812,	8521,
		40,	8476,	14717,
		41,	7820,	15360,
		42,	1179,	7939,
		43,	2357,	8678,
		44,	7703,	6216,
		0,	3477,	7067,
		1,	3931,	13845,
		2,	7675,	12899,
		3,	1754,	8187,
		4,	7785,	1400,
		5,	9213,	5891,
		6,	2494,	7703,
		7,	2576,	7902,
		8,	4821,	15682,
		9,	10426,	11935,
		10,	1810,	904,
		11,	11332,	9264,
		12,	11312,	3570,
		13,	14916,	2650,
		14,	7679,	7842,
		15,	6089,	13084,
		16,	3938,	2751,
		17,	8509,	4648,
		18,	12204,	8917,
		19,	5749,	12443,
		20,	12613,	4431,
		21,	1344,	4014,
		22,	8488,	13850,
		23,	1730,	14896,
		24,	14942,	7126,
		25,	14983,	8863,
		26,	6578,	8564,
		27,	4947,	396,
		28,	297,	12805,
		29,	13878,	6692,
		30,	11857,	11186,
		31,	14395,	11493,
		32,	16145,	12251,
		33,	13462,	7428,
		34,	14526,	13119,
		35,	2535,	11243,
		36,	6465,	12690,
		37,	6872,	9334,
		38,	15371,	14023,
		39,	8101,	10187,
		40,	11963,	4848,
		41,	15125,	6119,
		42,	8051,	14465,
		43,	11139,	5167,
		44,	2883,	14521,
	};
};

struct DVB_T2_TABLE_A5
{
	static const int M = 360;
	static const int N = 64800;
	static const int K = 51840;
	static const int LINKS_MIN_CN = 17;
	static const int LINKS_MAX_CN = 18;
	static const int LINKS_TOTAL = 233279;
	static const int DEG_MAX = 11;
	static constexpr int DEG[] = {
		11, 3, 0
	};
	static constexpr int LEN[] = {
		18, 126, 0
	};
	static constexpr int POS[] = {
		0,	149,	11212,	5575,	6360,	12559,	8108,	8505,	408,	10026,	12828,
		1,	5237,	490,	10677,	4998,	3869,	3734,	3092,	3509,	7703,	10305,
		2,	8742,	5553,	2820,	7085,	12116,	10485,	564,	7795,	2972,	2157,
		3,	2699,	4304,	8350,	712,	2841,	3250,	4731,	10105,	517,	7516,
		4,	12067,	1351,	11992,	12191,	11267,	5161,	537,	6166,	4246,	2363,
		5,	6828,	7107,	2127,	3724,	5743,	11040,	10756,	4073,	1011,	3422,
		6,	11259,	1216,	9526,	1466,	10816,	940,	3744,	2815,	11506,	11573,
		7,	4549,	11507,	1118,	1274,	11751,	5207,	7854,	12803,	4047,	6484,
		8,	8430,	4115,	9440,	413,	4455,	2262,	7915,	12402,	8579,	7052,
		9,	3885,	9126,	5665,	4505,	2343,	253,	4707,	3742,	4166,	1556,
		10,	1704,	8936,	6775,	8639,	8179,	7954,	8234,	7850,	8883,	8713,
		11,	11716,	4344,	9087,	11264,	2274,	8832,	9147,	11930,	6054,	5455,
		12,	7323,	3970,	10329,	2170,	8262,	3854,	2087,	12899,	9497,	11700,
		13,	4418,	1467,	2490,	5841,	817,	11453,	533,	11217,	11962,	5251,
		14,	1541,	4525,	7976,	3457,	9536,	7725,	3788,	2982,	6307,	5997,
		15,	11484,	2739,	4023,	12107,	6516,	551,	2572,	6628,	8150,	9852,
		16,	6070,	1761,	4627,	6534,	7913,	3730,	11866,	1813,	12306,	8249,
		17,	12441,	5489,	8748,	7837,	7660,	2102,	11341,	2936,	6712,	11977,
		18,	10155,	4210,
		19,	1010,	10483,
		20,	8900,	10250,
		21,	10243,	12278,
		22,	7070,	4397,
		23,	12271,	3887,
		24,	11980,	6836,
		25,	9514,	4356,
		26,	7137,	10281,
		27,	11881,	2526,
		28,	1969,	11477,
		29,	3044,	10921,
		30,	2236,	8724,
		31,	9104,	6340,
		32,	7342,	8582,
		33,	11675,	10405,
		34,	6467,	12775,
		35,	3186,	12198,
		0,	9621,	11445,
		1,	7486,	5611,
		2,	4319,	4879,
		3,	2196,	344,
		4,	7527,	6650,
		5,	10693,	2440,
		6,	6755,	2706,
		7,	5144,	5998,
		8,	11043,	8033,
		9,	4846,	4435,
		10,	4157,	9228,
		11,	12270,	6562,
		12,	11954,	7592,
		13,	7420,	2592,
		14,	8810,	9636,
		15,	689,	5430,
		16,	920,	1304,
		17,	1253,	11934,
		18,	9559,	6016,
		19,	312,	7589,
		20,	4439,	4197,
		21,	4002,	9555,
		22,	12232,	7779,
		23,	1494,	8782,
		24,	10749,	3969,
		25,	4368,	3479,
		26,	6316,	5342,
		27,	2455,	3493,
		28,	12157,	7405,
		29,	6598,	11495,
		30,	11805,	4455,
		31,	9625,	2090,
		32,	4731,	2321,
		33,	3578,	2608,
		34,	8504,	1849,
		35,	4027,	1151,
		0,	5647,	4935,
		1,	4219,	1870,
		2,	10968,	8054,
		3,	6970,	5447,
		4,	3217,	5638,
		5,	8972,	669,
		6,	5618,	12472,
		7,	1457,	1280,
		8,	8868,	3883,
		9,	8866,	1224,
		10,	8371,	5972,
		11,	266,	4405,
		12,	3706,	3244,
		13,	6039,	5844,
		14,	7200,	3283,
		15,	1502,	11282,
		16,	12318,	2202,
		17,	4523,	965,
		18,	9587,	7011,
		19,	2552,	2051,
		20,	12045,	10306,
		21,	11070,	5104,
		22,	6627,	6906,
		23,	9889,	2121,
		24,	829,	9701,
		25,	2201,	1819,
		26,	6689,	12925,
		27,	2139,	8757,
		28,	12004,	5948,
		29,	8704,	3191,
		30,	8171,	10933,
		31,	6297,	7116,
		32,	616,	7146,
		33,	5142,	9761,
		34,	10377,	8138,
		35,	7616,	5811,
		0,	7285,	9863,
		1,	7764,	10867,
		2,	12343,	9019,
		3,	4414,	8331,
		4,	3464,	642,
		5,	6960,	2039,
		6,	786,	3021,
		7,	710,	2086,
		8,	7423,	5601,
		9,	8120,	4885,
		10,	12385,	11990,
		11,	9739,	10034,
		12,	424,	10162,
		13,	1347,	7597,
		14,	1450,	112,
		15,	7965,	8478,
		16,	8945,	7397,
		17,	6590,	8316,
		18,	6838,	9011,
		19,	6174,	9410,
		20,	255,	113,
		21,	6197,	5835,
		22,	12902,	3844,
		23,	4377,	3505,
		24,	5478,	8672,
		25,	4453,	2132,
		26,	9724,	1380,
		27,	12131,	11526,
		28,	12323,	9511,
		29,	8231,	1752,
		30,	497,	9022,
		31,	9288,	3080,
		32,	2481,	7515,
		33,	2696,	268,
		34,	4023,	12341,
		35,	7108,	5553,
	};
};

struct DVB_T2_TABLE_A6
{
	static const int M = 360;
	static const int N = 64800;
	static const int K = 54000;
	static const int LINKS_MIN_CN = 21;
	static const int LINKS_MAX_CN = 22;
	static const int LINKS_TOTAL = 237599;
	static const int DEG_MAX = 13;
	static constexpr int DEG[] = {
		13, 3, 0
	};
	static constexpr int LEN[] = {
		15, 135, 0
	};
	static constexpr int POS[] = {
		0,	4362,	416,	8909,	4156,	3216,	3112,	2560,	2912,	6405,	8593,	4969,	6723,
		1,	2479,	1786,	8978,	3011,	4339,	9313,	6397,	2957,	7288,	5484,	6031,	10217,
		2,	10175,	9009,	9889,	3091,	4985,	7267,	4092,	8874,	5671,	2777,	2189,	8716,
		3,	9052,	4795,	3924,	3370,	10058,	1128,	9996,	10165,	9360,	4297,	434,	5138,
		4,	2379,	7834,	4835,	2327,	9843,	804,	329,	8353,	7167,	3070,	1528,	7311,
		5,	3435,	7871,	348,	3693,	1876,	6585,	10340,	7144,	5870,	2084,	4052,	2780,
		6,	3917,	3111,	3476,	1304,	10331,	5939,	5199,	1611,	1991,	699,	8316,	9960,
		7,	6883,	3237,	1717,	10752,	7891,	9764,	4745,	3888,	10009,	4176,	4614,	1567,
		8,	10587,	2195,	1689,	2968,	5420,	2580,	2883,	6496,	111,	6023,	1024,	4449,
		9,	3786,	8593,	2074,	3321,	5057,	1450,	3840,	5444,	6572,	3094,	9892,	1512,
		10,	8548,	1848,	10372,	4585,	7313,	6536,	6379,	1766,	9462,	2456,	5606,	9975,
		11,	8204,	10593,	7935,	3636,	3882,	394,	5968,	8561,	2395,	7289,	9267,	9978,
		12,	7795,	74,	1633,	9542,	6867,	7352,	6417,	7568,	10623,	725,	2531,	9115,
		13,	7151,	2482,	4260,	5003,	10105,	7419,	9203,	6691,	8798,	2092,	8263,	3755,
		14,	3600,	570,	4527,	200,	9718,	6771,	1995,	8902,	5446,	768,	1103,	6520,
		15,	6304,	7621,
		16,	6498,	9209,
		17,	7293,	6786,
		18,	5950,	1708,
		19,	8521,	1793,
		20,	6174,	7854,
		21,	9773,	1190,
		22,	9517,	10268,
		23,	2181,	9349,
		24,	1949,	5560,
		25,	1556,	555,
		26,	8600,	3827,
		27,	5072,	1057,
		28,	7928,	3542,
		29,	3226,	3762,
		0,	7045,	2420,
		1,	9645,	2641,
		2,	2774,	2452,
		3,	5331,	2031,
		4,	9400,	7503,
		5,	1850,	2338,
		6,	10456,	9774,
		7,	1692,	9276,
		8,	10037,	4038,
		9,	3964,	338,
		10,	2640,	5087,
		11,	858,	3473,
		12,	5582,	5683,
		13,	9523,	916,
		14,	4107,	1559,
		15,	4506,	3491,
		16,	8191,	4182,
		17,	10192,	6157,
		18,	5668,	3305,
		19,	3449,	1540,
		20,	4766,	2697,
		21,	4069,	6675,
		22,	1117,	1016,
		23,	5619,	3085,
		24,	8483,	8400,
		25,	8255,	394,
		26,	6338,	5042,
		27,	6174,	5119,
		28,	7203,	1989,
		29,	1781,	5174,
		0,	1464,	3559,
		1,	3376,	4214,
		2,	7238,	67,
		3,	10595,	8831,
		4,	1221,	6513,
		5,	5300,	4652,
		6,	1429,	9749,
		7,	7878,	5131,
		8,	4435,	10284,
		9,	6331,	5507,
		10,	6662,	4941,
		11,	9614,	10238,
		12,	8400,	8025,
		13,	9156,	5630,
		14,	7067,	8878,
		15,	9027,	3415,
		16,	1690,	3866,
		17,	2854,	8469,
		18,	6206,	630,
		19,	363,	5453,
		20,	4125,	7008,
		21,	1612,	6702,
		22,	9069,	9226,
		23,	5767,	4060,
		24,	3743,	9237,
		25,	7018,	5572,
		26,	8892,	4536,
		27,	853,	6064,
		28,	8069,	5893,
		29,	2051,	2885,
		0,	10691,	3153,
		1,	3602,	4055,
		2,	328,	1717,
		3,	2219,	9299,
		4,	1939,	7898,
		5,	617,	206,
		6,	8544,	1374,
		7,	10676,	3240,
		8,	6672,	9489,
		9,	3170,	7457,
		10,	7868,	5731,
		11,	6121,	10732,
		12,	4843,	9132,
		13,	580,	9591,
		14,	6267,	9290,
		15,	3009,	2268,
		16,	195,	2419,
		17,	8016,	1557,
		18,	1516,	9195,
		19,	8062,	9064,
		20,	2095,	8968,
		21,	753,	7326,
		22,	6291,	3833,
		23,	2614,	7844,
		24,	2303,	646,
		25,	2075,	611,
		26,	4687,	362,
		27,	8684,	9940,
		28,	4830,	2065,
		29,	7038,	1363,
		0,	1769,	7837,
		1,	3801,	1689,
		2,	10070,	2359,
		3,	3667,	9918,
		4,	1914,	6920,
		5,	4244,	5669,
		6,	10245,	7821,
		7,	7648,	3944,
		8,	3310,	5488,
		9,	6346,	9666,
		10,	7088,	6122,
		11,	1291,	7827,
		12,	10592,	8945,
		13,	3609,	7120,
		14,	9168,	9112,
		15,	6203,	8052,
		16,	3330,	2895,
		17,	4264,	10563,
		18,	10556,	6496,
		19,	8807,	7645,
		20,	1999,	4530,
		21,	9202,	6818,
		22,	3403,	1734,
		23,	2106,	9023,
		24,	6881,	3883,
		25,	3895,	2171,
		26,	4062,	6424,
		27,	3755,	9536,
		28,	4683,	2131,
		29,	7347,	8027,
	};
};

struct DVB_T2_TABLE_B1
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 3240;
	static const int LINKS_MIN_CN = 3;
	static const int LINKS_MAX_CN = 4;
	static const int LINKS_TOTAL = 48599;
	static const int DEG_MAX = 12;
	static constexpr int DEG[] = {
		12, 3, 0
	};
	static constexpr int LEN[] = {
		4, 5, 0
	};
	static constexpr int POS[] = {
		6295,	9626,	304,	7695,	4839,	4936,	1660,	144,	11203,	5567,	6347,	12557,
		10691,	4988,	3859,	3734,	3071,	3494,	7687,	10313,	5964,	8069,	8296,	11090,
		10774,	3613,	5208,	11177,	7676,	3549,	8746,	6583,	7239,	12265,	2674,	4292,
		11869,	3708,	5981,	8718,	4908,	10650,	6805,	3334,	2627,	10461,	9285,	11120,
		7844,	3079,	10773,
		3385,	10854,	5747,
		1360,	12010,	12202,
		6189,	4241,	2343,
		9840,	12726,	4977,
	};
};

struct DVB_T2_TABLE_B2
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 7200;
	static const int LINKS_MIN_CN = 4;
	static const int LINKS_MAX_CN = 7;
	static const int LINKS_TOTAL = 48599;
	static const int DEG_MAX = 8;
	static constexpr int DEG[] = {
		8, 3, 0
	};
	static constexpr int LEN[] = {
		5, 15, 0
	};
	static constexpr int POS[] = {
		20,	712,	2386,	6354,	4061,	1062,	5045,	5158,
		21,	2543,	5748,	4822,	2348,	3089,	6328,	5876,
		22,	926,	5701,	269,	3693,	2438,	3190,	3507,
		23,	2802,	4520,	3577,	5324,	1091,	4667,	4449,
		24,	5140,	2003,	1263,	4742,	6497,	1185,	6202,
		0,	4046,	6934,
		1,	2855,	66,
		2,	6694,	212,
		3,	3439,	1158,
		4,	3850,	4422,
		5,	5924,	290,
		6,	1467,	4049,
		7,	7820,	2242,
		8,	4606,	3080,
		9,	4633,	7877,
		10,	3884,	6868,
		11,	8935,	4996,
		12,	3028,	764,
		13,	5988,	1057,
		14,	7411,	3450,
	};
};

struct DVB_T2_TABLE_B3
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 9720;
	static const int LINKS_MIN_CN = 8;
	static const int LINKS_MAX_CN = 9;
	static const int LINKS_TOTAL = 58319;
	static const int DEG_MAX = 12;
	static constexpr int DEG[] = {
		12, 3, 0
	};
	static constexpr int LEN[] = {
		5, 22, 0
	};
	static constexpr int POS[] = {
		71,	1478,	1901,	2240,	2649,	2725,	3592,	3708,	3965,	4080,	5733,	6198,
		393,	1384,	1435,	1878,	2773,	3182,	3586,	5465,	6091,	6110,	6114,	6327,
		160,	1149,	1281,	1526,	1566,	2129,	2929,	3095,	3223,	4250,	4276,	4612,
		289,	1446,	1602,	2421,	3559,	3796,	5590,	5750,	5763,	6168,	6271,	6340,
		947,	1227,	2008,	2020,	2266,	3365,	3588,	3867,	4172,	4250,	4865,	6290,
		3324,	3704,	4447,
		1206,	2565,	3089,
		529,	4027,	5891,
		141,	1187,	3206,
		1990,	2972,	5120,
		752,	796,	5976,
		1129,	2377,	4030,
		6077,	6108,	6231,
		61,	1053,	1781,
		2820,	4109,	5307,
		2088,	5834,	5988,
		3725,	3945,	4010,
		1081,	2780,	3389,
		659,	2221,	4822,
		3033,	6060,	6160,
		756,	1489,	2350,
		3350,	3624,	5470,
		357,	1825,	5242,
		585,	3372,	6062,
		561,	1417,	2348,
		971,	3719,	5567,
		1005,	1675,	2062,
	};
};

struct DVB_T2_TABLE_B4
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 10800;
	static const int LINKS_MIN_CN = 9;
	static const int LINKS_MAX_CN = 10;
	static const int LINKS_TOTAL = 53999;
	static const int DEG_MAX = 13;
	static constexpr int DEG[] = {
		13, 3, 0
	};
	static constexpr int LEN[] = {
		3, 27, 0
	};
	static constexpr int POS[] = {
		0,	2084,	1613,	1548,	1286,	1460,	3196,	4297,	2481,	3369,	3451,	4620,	2622,
		1,	122,	1516,	3448,	2880,	1407,	1847,	3799,	3529,	373,	971,	4358,	3108,
		2,	259,	3399,	929,	2650,	864,	3996,	3833,	107,	5287,	164,	3125,	2350,
		3,	342,	3529,
		4,	4198,	2147,
		5,	1880,	4836,
		6,	3864,	4910,
		7,	243,	1542,
		8,	3011,	1436,
		9,	2167,	2512,
		10,	4606,	1003,
		11,	2835,	705,
		12,	3426,	2365,
		13,	3848,	2474,
		14,	1360,	1743,
		0,	163,	2536,
		1,	2583,	1180,
		2,	1542,	509,
		3,	4418,	1005,
		4,	5212,	5117,
		5,	2155,	2922,
		6,	347,	2696,
		7,	226,	4296,
		8,	1560,	487,
		9,	3926,	1640,
		10,	149,	2928,
		11,	2364,	563,
		12,	635,	688,
		13,	231,	1684,
		14,	1129,	3894,
	};
};

struct DVB_T2_TABLE_B5
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 11880;
	static const int LINKS_MIN_CN = 9;
	static const int LINKS_MAX_CN = 13;
	static const int LINKS_TOTAL = 47519;
	static const int DEG_MAX = 12;
	static constexpr int DEG[] = {
		12, 3, 0
	};
	static constexpr int LEN[] = {
		1, 32, 0
	};
	static constexpr int POS[] = {
		3,	3198,	478,	4207,	1481,	1009,	2616,	1924,	3437,	554,	683,	1801,
		4,	2681,	2135,
		5,	3107,	4027,
		6,	2637,	3373,
		7,	3830,	3449,
		8,	4129,	2060,
		9,	4184,	2742,
		10,	3946,	1070,
		11,	2239,	984,
		0,	1458,	3031,
		1,	3003,	1328,
		2,	1137,	1716,
		3,	132,	3725,
		4,	1817,	638,
		5,	1774,	3447,
		6,	3632,	1257,
		7,	542,	3694,
		8,	1015,	1945,
		9,	1948,	412,
		10,	995,	2238,
		11,	4141,	1907,
		0,	2480,	3079,
		1,	3021,	1088,
		2,	713,	1379,
		3,	997,	3903,
		4,	2323,	3361,
		5,	1110,	986,
		6,	2532,	142,
		7,	1690,	2405,
		8,	1298,	1881,
		9,	615,	174,
		10,	1648,	3112,
		11,	1415,	2808,
	};
};

struct DVB_T2_TABLE_B6
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 12600;
	static const int LINKS_MIN_CN = 11;
	static const int LINKS_MAX_CN = 13;
	static const int LINKS_TOTAL = 44999;
	static const int DEG_MAX = 3;
	static constexpr int DEG[] = {
		3, 0
	};
	static constexpr int LEN[] = {
		35, 0
	};
	static constexpr int POS[] = {
		5,	896,	1565,
		6,	2493,	184,
		7,	212,	3210,
		8,	727,	1339,
		9,	3428,	612,
		0,	2663,	1947,
		1,	230,	2695,
		2,	2025,	2794,
		3,	3039,	283,
		4,	862,	2889,
		5,	376,	2110,
		6,	2034,	2286,
		7,	951,	2068,
		8,	3108,	3542,
		9,	307,	1421,
		0,	2272,	1197,
		1,	1800,	3280,
		2,	331,	2308,
		3,	465,	2552,
		4,	1038,	2479,
		5,	1383,	343,
		6,	94,	236,
		7,	2619,	121,
		8,	1497,	2774,
		9,	2116,	1855,
		0,	722,	1584,
		1,	2767,	1881,
		2,	2701,	1610,
		3,	3283,	1732,
		4,	168,	1099,
		5,	3074,	243,
		6,	3460,	945,
		7,	2049,	1746,
		8,	566,	1427,
		9,	3545,	1168,
	};
};

struct DVB_T2_TABLE_B7
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 13320;
	static const int LINKS_MIN_CN = 15;
	static const int LINKS_MAX_CN = 19;
	static const int LINKS_TOTAL = 49319;
	static const int DEG_MAX = 13;
	static constexpr int DEG[] = {
		13, 3, 0
	};
	static constexpr int LEN[] = {
		1, 36, 0
	};
	static constexpr int POS[] = {
		3,	2409,	499,	1481,	908,	559,	716,	1270,	333,	2508,	2264,	1702,	2805,
		4,	2447,	1926,
		5,	414,	1224,
		6,	2114,	842,
		7,	212,	573,
		0,	2383,	2112,
		1,	2286,	2348,
		2,	545,	819,
		3,	1264,	143,
		4,	1701,	2258,
		5,	964,	166,
		6,	114,	2413,
		7,	2243,	81,
		0,	1245,	1581,
		1,	775,	169,
		2,	1696,	1104,
		3,	1914,	2831,
		4,	532,	1450,
		5,	91,	974,
		6,	497,	2228,
		7,	2326,	1579,
		0,	2482,	256,
		1,	1117,	1261,
		2,	1257,	1658,
		3,	1478,	1225,
		4,	2511,	980,
		5,	2320,	2675,
		6,	435,	1278,
		7,	228,	503,
		0,	1885,	2369,
		1,	57,	483,
		2,	838,	1050,
		3,	1231,	1990,
		4,	1738,	68,
		5,	2392,	951,
		6,	163,	645,
		7,	2644,	1704,
	};
};

struct DVB_T2_TABLE_B8
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 5400;
	static const int LINKS_MIN_CN = 4;
	static const int LINKS_MAX_CN = 5;
	static const int LINKS_TOTAL = 53999;
	static const int DEG_MAX = 12;
	static constexpr int DEG[] = {
		12, 3, 0
	};
	static constexpr int LEN[] = {
		5, 10, 0
	};
	static constexpr int POS[] = {
		416,	8909,	4156,	3216,	3112,	2560,	2912,	6405,	8593,	4969,	6723,	6912,
		8978,	3011,	4339,	9312,	6396,	2957,	7288,	5485,	6031,	10218,	2226,	3575,
		3383,	10059,	1114,	10008,	10147,	9384,	4290,	434,	5139,	3536,	1965,	2291,
		2797,	3693,	7615,	7077,	743,	1941,	8716,	6215,	3840,	5140,	4582,	5420,
		6110,	8551,	1515,	7404,	4879,	4946,	5383,	1831,	3441,	9569,	10472,	4306,
		1505,	5682,	7778,
		7172,	6830,	6623,
		7281,	3941,	3505,
		10270,	8669,	914,
		3622,	7563,	9388,
		9930,	5058,	4554,
		4844,	9609,	2707,
		6883,	3237,	1714,
		4768,	3878,	10017,
		10127,	3334,	8267,
	};
};

struct DVB_T2_TABLE_B9
{
	static const int M = 360;
	static const int N = 16200;
	static const int K = 6480;
	static const int LINKS_MIN_CN = 5;
	static const int LINKS_MAX_CN = 6;
	static const int LINKS_TOTAL = 58319;
	static const int DEG_MAX = 12;
	static constexpr int DEG[] = {
		12, 3, 0
	};
	static constexpr int LEN[] = {
		6, 12, 0
	};
	static constexpr int POS[] = {
		5650,	4143,	8750,	583,	6720,	8071,	635,	1767,	1344,	6922,	738,	6658,
		5696,	1685,	3207,	415,	7019,	5023,	5608,	2605,	857,	6915,	1770,	8016,
		3992,	771,	2190,	7258,	8970,	7792,	1802,	1866,	6137,	8841,	886,	1931,
		4108,	3781,	7577,	6810,	9322,	8226,	5396,	5867,	4428,	8827,	7766,	2254,
		4247,	888,	4367,	8821,	9660,	324,	5864,	4774,	227,	7889,	6405,	8963,
		9693,	500,	2520,	2227,	1811,	9330,	1928,	5140,	4030,	4824,	806,	3134,
		1652,	8171,	1435,
		3366,	6543,	3745,
		9286,	8509,	4645,
		7397,	5790,	8972,
		6597,	4422,	1799,
		9276,	4041,	3847,
		8683,	7378,	4946,
		5348,	1993,	9186,
		6724,	9015,	5646,
		4502,	4439,	8474,
		5107,	7342,	9442,
		1387,	8910,	2660,
	};
};

} // namespace ldpctool

#endif
