// Thomas Nagy 2007-2023 GPLV3

#include "canvas_flag.h"
#include "data_item.h"
#include  <QtGlobal>
#include "sem_mediator.h"
#include "canvas_view.h"
  #include <QElapsedTimer>
#include <QPrinter>
#include <QDesktopWidget>
 #include <QPair>  
#include  <QMenu> 
#include <QToolTip>
#include <QSvgGenerator>
#include  <QColorDialog> 
#include<KToolBar> 
#include<KActionCollection>	
#include<KMessageBox>
#include <QDialog>
#include "canvas_ref.h"
#include <QUrl>
#include  <QActionGroup> 
#include "canvas_item.h"
#include "canvas_pic.h"
#include "canvas_link.h"
#include<QCoreApplication>
#include <QSet>
#include <QPrintDialog>
#include "canvas_sort.h"
#include <windef.h>
#include <QRadioButton>
#include "canvas_sort_toggle.h"
#include "semantik.h"
#include <QTextCursor> 
#include  <QApplication> 
#include "canvas_chain.h"
#include <QtDebug>
#include <QScrollBar>
#include  <QGraphicsScene>
#include <QWheelEvent>
#include  <QMatrix>
#include <QPointF>
#include <QFileDialog>
#include <QSpinBox>
#include "export_map_dialog.h"
#include "kurlrequester.h"
#include <QPoint>
 #include "mem_command.h"

 #include <QGraphicsTextItem>

#include <math.h>

#define PIPAD 20

canvas_view::canvas_view(QWidget *i_oWidget, sem_mediator *i_oControl, QMenu* i_oColorMenu) : QGraphicsView(i_oWidget)
{
	m_oSemantikWindow = i_oWidget;
	m_bDisableGradient = false;
	m_oLastPoint = QPointF(0, 0);

	//m_oRubbery = new QRubberBand(QRubberBand::Rectangle, this);
	//m_oRubbery->setGeometry(QRect(0, 0, 0, 0));

	QGraphicsScene *l_oScene = new QGraphicsScene(this);
	l_oScene->setSceneRect(-400, -400, 400, 400);
	setScene(l_oScene);

	//setCacheMode(CacheBackground);
	setRenderHint(QPainter::Antialiasing);

	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);

	setMinimumSize(200, 200);

	m_oMediator = i_oControl;

	m_bPressed = false;
	m_bScroll = false;

	m_oRubberLine = new rubber_line(QRubberBand::Line, this);

	QBrush l_oBrush = QBrush();
	//QColor l_oColor = QColor("#000077");
	//l_oColor.setAlpha(200);

	QColor l_oColor = QColor(0, 0, 150, 100);
	l_oBrush.setColor(l_oColor);
	//m_oRubber->setBrush(l_oBrush);
	m_oColorMenu = i_oColorMenu;

	setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
}

void canvas_view::setupActions(KActionCollection* i_oActionCollection) {
	QAction *l_o = NULL;

	m_oAddItemAction = new QAction(i18n("Add an element"), this);
	connect(m_oAddItemAction, SIGNAL(triggered()), this, SLOT(slot_add_item()));
	i_oActionCollection->addAction(notr("add_element"), m_oAddItemAction);
	i_oActionCollection->setDefaultShortcut(m_oAddItemAction, QKeySequence(notr("Ctrl+Return")));

	m_oDeleteAction = new QAction(i18n("Delete selection"), this);
	connect(m_oDeleteAction, SIGNAL(triggered()), this, SLOT(slot_delete()));
	i_oActionCollection->addAction(notr("delete_element"), m_oDeleteAction);
	i_oActionCollection->setDefaultShortcut(m_oDeleteAction, QKeySequence(notr("Del")));

	m_oInsertSiblingAction = l_o = new QAction(i18n("Insert a sibling"), this);
	connect(l_o, SIGNAL(triggered()), this, SLOT(slot_add_sibling()));
	i_oActionCollection->addAction(notr("add_sibling"), l_o);
	i_oActionCollection->setDefaultShortcut(m_oInsertSiblingAction, QKeySequence(notr("Shift+Return")));

	m_oMoveUpAction = l_o = new QAction(i18n("Move up"), this); l_o->setShortcut(notr("Alt+Up")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_move())); i_oActionCollection->addAction(notr("move_up"), l_o); l_o->setData(QVariant(0));
	m_oMoveDownAction = l_o = new QAction(i18n("Move down"), this); l_o->setShortcut(notr("Alt+Down")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_move())); i_oActionCollection->addAction(notr("move_down"), l_o); l_o->setData(QVariant(1));
	m_oMoveLeftAction = l_o = new QAction(i18n("Move left"), this); l_o->setShortcut(notr("Alt+Left")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_move())); i_oActionCollection->addAction(notr("move_left"), l_o); l_o->setData(QVariant(2));
	m_oMoveRightAction = l_o = new QAction(i18n("Move right"), this); l_o->setShortcut(notr("Alt+Right")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_move())); i_oActionCollection->addAction(notr("move_right"), l_o); l_o->setData(QVariant(3));

	m_oSelectUpAction = l_o = new QAction(i18n("Select up"), this); l_o->setShortcut(notr("Up")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_sel())); i_oActionCollection->addAction(notr("select_up"), l_o); l_o->setData(QVariant(0));
	m_oSelectDownAction = l_o = new QAction(i18n("Select down"), this); l_o->setShortcut(notr("Down")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_sel())); i_oActionCollection->addAction(notr("select_down"), l_o); l_o->setData(QVariant(1));
	m_oSelectLeftAction = l_o = new QAction(i18n("Select left"), this); l_o->setShortcut(notr("Left")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_sel())); i_oActionCollection->addAction(notr("select_left"), l_o); l_o->setData(QVariant(2));
	m_oSelectRightAction = l_o = new QAction(i18n("Select right"), this); l_o->setShortcut(notr("Right")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_sel())); i_oActionCollection->addAction(notr("select_right"), l_o); l_o->setData(QVariant(3));


	m_oSelectSubtreeAction = new QAction(i18n("Select subtree"), this);
	connect(m_oSelectSubtreeAction, SIGNAL(triggered()), this, SLOT(slot_select_subtree()));
	i_oActionCollection->addAction(notr("select_subtree"), m_oSelectSubtreeAction);


	m_oNextRootAction = l_o = new QAction(i18n("Next root"), this); l_o->setShortcut(notr("PgDown")); connect(l_o, SIGNAL(triggered()), this, SLOT(slot_next_root())); i_oActionCollection->addAction(notr("next_root"), l_o); l_o->setData(QVariant(1));


	m_oEditAction = new QAction(i18n("Toggle edit"), this);
	m_oEditAction->setShortcut(notr("Return"));
	i_oActionCollection->addAction(notr("toggle_edit"), m_oEditAction);
	connect(m_oEditAction, SIGNAL(triggered()), this, SLOT(slot_toggle_edit()));

	m_oCancelEditAction = new QAction(i18n("Cancel edit"), this);
	m_oCancelEditAction->setShortcut(notr("Escape"));
	i_oActionCollection->addAction(notr("cancel_edit"), m_oCancelEditAction);
	connect(m_oCancelEditAction, SIGNAL(triggered()), this, SLOT(slot_cancel_edit()));
	m_oCancelEditAction->setEnabled(false);

	m_oMenu = new QMenu(this);
	m_oMenu->addAction(m_oAddItemAction);
	//m_oMenu->addAction(m_oEditAction);
	m_oMenu->addAction(m_oDeleteAction);
	m_oMenu->addAction(m_oSelectSubtreeAction);

	//m_oDataMenu = m_oMenu->addMenu(i18n("Data type"));
	m_oMenu->addMenu(m_oColorMenu);

#define newAction(s, v, dest)  dest = l_o = new QAction(s, this); \
	connect(l_o, SIGNAL(triggered()), this, SLOT(slot_change_data())); \
	addAction(l_o); l_o->setData(QVariant(v)); \
	m_oMenu->addAction(l_o); \
	l_o->setCheckable(true);

	m_oMenu->addSeparator();
	newAction(i18n("Text"), VIEW_TEXT, m_oTextType);
	newAction(i18n("Diagram"), VIEW_DIAG, m_oDiagramType);
	newAction(i18n("Table"), VIEW_TABLE, m_oTableType);
	newAction(i18n("Image"), VIEW_IMG, m_oImageType);

	m_bDeleting = false;
	//connect(scene(), SIGNAL(selectionChanged()), this, SLOT(selection_changed())); // TODO check with m_bPressed

	setDragMode(QGraphicsView::RubberBandDrag);

	scene()->setFont(m_oMediator->m_oFont);

	setBackgroundBrush(m_oMediator->m_oColor);
}

void canvas_view::resizeEvent(QResizeEvent* e)
{
	QGraphicsView::resizeEvent(e);

	QRect l_oRect = viewport()->rect();
	QRectF ar = QRectF(mapToScene(l_oRect.topLeft()), mapToScene(l_oRect.bottomRight()));
	QRectF br = scene()->itemsBoundingRect();
	br = br.united(ar);

	scene()->setSceneRect(br);
}

QRectF canvas_view::visibleRect()
{
	if (scene()->items().size())
	{
		int l_iCount = 0;
		QRectF l_oRect;
		foreach (QGraphicsItem *l_o, scene()->items())
		{
			if (!l_o->isVisible())
			{
				continue;
			}
			if (l_iCount == 0)
			{
				l_oRect = l_o->sceneBoundingRect();
			}
			else
			{
				l_oRect |= l_o->sceneBoundingRect();
			}
			l_iCount += 1;
		}
		if (l_iCount)
		{
			return l_oRect;
		}
	}
	return scene()->itemsBoundingRect();
}

void canvas_view::check_selection()
{
	if (m_bDeleting) return;

	QList<int> lst;
	foreach(canvas_item* k, selection()) {
		lst.append(k->Id());
	}

	// stupid set intersection
	QList<int> unlst;
	foreach(const data_item& x, m_oMediator->m_oItems.values()) {
		if (x.m_bSelected) {
			unlst.append(x.m_iId);
		}
	}

	foreach(int i, lst) {
		if (!unlst.contains(i))
			goto undo;
	}

	foreach (int i, unlst) {
		if (!lst.contains(i))
			goto undo;
	}

	mem_sel *sel;
	goto end;
	undo:
		sel = new mem_sel(m_oMediator);
		sel->sel = lst;
		sel->apply();
		//qDebug()<<"selected"<<lst<<"unselected"<<unlst;
	end:
		;
}

void canvas_view::slot_next_root()
{
	switch (((QAction*) QObject::sender())->data().toInt())
	{
		case 0:
			m_oMediator->prev_root();
			break;
		case 1:
			m_oMediator->next_root();
			break;
		default:
			break;
	}
	QList<canvas_item*> sel = selection();
	if (sel.size()==1) ensureVisible(sel[0]);
}

void canvas_view::edit_off() {
	canvas_item* sel = NULL;
	foreach (QGraphicsItem *tmp, items()) {
		if (tmp->type() == CANVAS_ITEM_T) {
			sel = (canvas_item*) tmp;

			if (sel->textInteractionFlags() & Qt::TextEditorInteraction) {

				sel->setTextInteractionFlags(Qt::NoTextInteraction);
				if (sel->toPlainText() == notr("")) {
					sel->setPlainText(i18n("Empty"));
					sel->update_links();
				}

				if (sel->toPlainText() != (*m_oMediator + sel->Id()).m_sSummary) {
					mem_edit *ed = new mem_edit(m_oMediator);
					ed->newSummary = sel->toPlainText();
					ed->apply();
				}

				m_oAddItemAction->setEnabled(true);
				m_oInsertSiblingAction->setEnabled(true);
				m_oDeleteAction->setEnabled(true);
				m_oNextRootAction->setEnabled(true);

				m_oMoveUpAction->setEnabled(true);
				m_oMoveDownAction->setEnabled(true);
				m_oMoveLeftAction->setEnabled(true);
				m_oMoveRightAction->setEnabled(true);
				m_oSelectUpAction->setEnabled(true);
				m_oSelectDownAction->setEnabled(true);
				m_oSelectLeftAction->setEnabled(true);
				m_oSelectRightAction->setEnabled(true);
				m_oSelectSubtreeAction->setEnabled(true);
				m_oMediator->check_undo(true);
			}
		}
	}
}

void canvas_view::slot_toggle_edit()
{
	if (!hasFocus()) {
		return;
	}

	canvas_item* sel = NULL;
	foreach (QGraphicsItem *tmp, items()) {
		if (tmp->type() == CANVAS_ITEM_T && tmp->isSelected()) {
			if (sel) {
				sel = NULL;
				break;
			} else {
				sel = (canvas_item*) tmp;
			}
		}
	}

	if (sel) {
		if (sel->textInteractionFlags() & Qt::TextEditorInteraction) {
			sel->setTextInteractionFlags(Qt::NoTextInteraction);
			if (sel->toPlainText() == notr("")) {
				sel->setPlainText(i18n("Empty"));
				sel->update(); // seems to update faster
				sel->update_links();

			}
			if (sel->toPlainText() != (*m_oMediator + sel->Id()).m_sSummary) {
				mem_edit *ed = new mem_edit(m_oMediator);
				ed->newSummary = sel->toPlainText();
				ed->apply();
			}
		} else {
			sel->setTextInteractionFlags(Qt::TextEditorInteraction);
			if (sel->toPlainText() == i18n("Empty")) {
				sel->setPlainText("");
				sel->update(); // seems to update faster
				sel->update_links();
			}
			sel->setFocus();

			m_oAddItemAction->setEnabled(false);
			m_oCancelEditAction->setEnabled(true);
			m_oInsertSiblingAction->setEnabled(false);
			m_oDeleteAction->setEnabled(false);
			m_oNextRootAction->setEnabled(false);

			m_oMoveUpAction->setEnabled(false);
			m_oMoveDownAction->setEnabled(false);
			m_oMoveLeftAction->setEnabled(false);
			m_oMoveRightAction->setEnabled(false);
			m_oSelectUpAction->setEnabled(false);
			m_oSelectDownAction->setEnabled(false);
			m_oSelectLeftAction->setEnabled(false);
			m_oSelectRightAction->setEnabled(false);
			m_oSelectSubtreeAction->setEnabled(false);
			m_oMediator->check_undo(false);
			return;
		}
	}

	m_oAddItemAction->setEnabled(true);
	m_oCancelEditAction->setEnabled(false);
	m_oInsertSiblingAction->setEnabled(true);
	m_oDeleteAction->setEnabled(true);
	m_oNextRootAction->setEnabled(true);

	m_oMoveUpAction->setEnabled(true);
	m_oMoveDownAction->setEnabled(true);
	m_oMoveLeftAction->setEnabled(true);
	m_oMoveRightAction->setEnabled(true);
	m_oSelectUpAction->setEnabled(true);
	m_oSelectDownAction->setEnabled(true);
	m_oSelectLeftAction->setEnabled(true);
	m_oSelectRightAction->setEnabled(true);
	m_oSelectSubtreeAction->setEnabled(true);
	m_oMediator->check_undo(true);
}

void canvas_view::slot_cancel_edit()
{
	canvas_item* sel = NULL;
	foreach (QGraphicsItem *tmp, items()) {
		if (tmp->type() == CANVAS_ITEM_T && tmp->isSelected()) {
			if (sel) {
				sel = NULL;
				break;
			} else {
				sel = (canvas_item*) tmp;
			}
		}
	}

	if (sel && sel->textInteractionFlags() & Qt::TextEditorInteraction) {
		sel->setTextInteractionFlags(Qt::NoTextInteraction);
		sel->update_data();
	}

	m_oAddItemAction->setEnabled(true);
	m_oCancelEditAction->setEnabled(false);
	m_oInsertSiblingAction->setEnabled(true);
	m_oDeleteAction->setEnabled(true);
	m_oNextRootAction->setEnabled(true);

	m_oMoveUpAction->setEnabled(true);
	m_oMoveDownAction->setEnabled(true);
	m_oMoveLeftAction->setEnabled(true);
	m_oMoveRightAction->setEnabled(true);
	m_oSelectUpAction->setEnabled(true);
	m_oSelectDownAction->setEnabled(true);
	m_oSelectLeftAction->setEnabled(true);
	m_oSelectRightAction->setEnabled(true);
	m_oSelectSubtreeAction->setEnabled(true);
	m_oMediator->check_undo(true);
}

void canvas_view::slot_move()
{
	QList<canvas_item*> sel = selection();
	if (sel.size() < 1) return;
	switch (((QAction*) QObject::sender())->data().toInt())
	{
		case 0: move_sel(0, -20); break;
		case 1: move_sel(0, 20); break;
		case 2: move_sel(-20, 0); break;
		case 3: move_sel(20, 0); break;
		default: break;
	}
}

void canvas_view::slot_sel()
{
	QList<canvas_item*> sel = selection();
	if (sel.size() != 1) return;
	int l_iId = sel[0]->Id();
	switch (((QAction*) QObject::sender())->data().toInt())
	{
		case 0: m_oMediator->select_item_keyboard(l_iId, 3); break;
		case 1: m_oMediator->select_item_keyboard(l_iId, 4); break;
		case 2: m_oMediator->select_item_keyboard(l_iId, 1); break;
		case 3: m_oMediator->select_item_keyboard(l_iId, 2); break;
		default: break;
	}
	if (sel.size()==1) ensureVisible(sel[0]);
}

void canvas_view::slot_add_item()
{
	QList<canvas_item*> sel = selection();
	int l_iId = NO_ITEM;
	if (sel.size() == 1) {
		l_iId = sel[0]->Id();
	}

	deselect_all();

	mem_add *add = new mem_add(m_oMediator);
	add->parent = l_iId;
	if (m_oMediator->m_oItems.contains(l_iId))
	{
		data_item& p = m_oMediator->m_oItems[l_iId];
		add->item.m_iXX = p.m_iXX + 30;
		add->item.m_iYY = p.m_iYY + 30;
	}
	else
	{
		add->item.m_iXX = m_oLastPoint.x();
		add->item.m_iYY = m_oLastPoint.y();
	}

	add->apply();
	if (m_oMediator->m_iAutoReorg)
	{
		reorganize();
	}
}

void canvas_view::slot_add_sibling()
{
	QList<canvas_item*> sel = selection();
	if (sel.size() != 1) return;
	int l_iId = m_oMediator->parent_of(sel[0]->Id());
	if (l_iId == NO_ITEM) return;
	deselect_all();

	data_item& p = m_oMediator->m_oItems[l_iId];
	mem_add *add = new mem_add(m_oMediator);
	add->parent = l_iId;
	add->item.m_iXX = p.m_iXX + 30;
	add->item.m_iYY = p.m_iYY + 30;
	add->apply();
	if (m_oMediator->m_iAutoReorg)
	{
		reorganize();
	}
}

void canvas_view::slot_delete()
{
	QList<int> l_oLst;
	foreach (canvas_item *l_oItem, selection()) {
		l_oLst.push_back(l_oItem->Id());
	}

	if (l_oLst.isEmpty()) {
		return;
	}

	mem_sel *sel = new mem_sel(m_oMediator);
	sel->unsel = l_oLst;
	sel->apply();

	mem_delete *del = new mem_delete(m_oMediator);
	del->init(l_oLst);
	del->apply();
}

void canvas_view::move_sel(int i_iX, int i_iY)
{
	QList<canvas_item*> sel = selection();
	check_canvas_size();
	foreach (canvas_item *l_oItem, sel) { l_oItem->moveBy(i_iX, i_iY); }
	foreach (canvas_item *l_oItem, sel) { l_oItem->update_links(); ensureVisible(l_oItem); }
}

void canvas_view::zoom_in()
{
	double i_iScaleFactor = 1.05;
	qreal i_rFactor = matrix().scale(i_iScaleFactor, i_iScaleFactor).mapRect(QRectF(0, 0, 1, 1)).width();
	if (i_rFactor < 0.01 || i_rFactor > 1000) return;
	scale(i_iScaleFactor, i_iScaleFactor);
	check_canvas_size();
}

void canvas_view::zoom_out()
{
	double i_iScaleFactor = 0.95;
	qreal i_rFactor = matrix().scale(i_iScaleFactor, i_iScaleFactor).mapRect(QRectF(0, 0, 1, 1)).width();
	if (i_rFactor < 0.01 || i_rFactor > 1000) return;
	scale(i_iScaleFactor, i_iScaleFactor);
	check_canvas_size();
}

void canvas_view::wheelEvent(QWheelEvent *i_oEvent)
{
	bool l_bCtrl = i_oEvent->modifiers() & Qt::ControlModifier;
	if (m_oMediator->m_oWindef->m_bUseTouchpad xor l_bCtrl)
	{
		QGraphicsView::wheelEvent(i_oEvent);
		return;
	}

	QPointF l_o = mapToScene(i_oEvent->pos());
	qreal i_iScaleFactor = pow(2.0, i_oEvent->delta() / 440.0);
	qreal i_rFactor = matrix().scale(i_iScaleFactor, i_iScaleFactor).mapRect(QRectF(0, 0, 1, 1)).width();
	if (i_rFactor < 0.01 || i_rFactor > 1000) return;

	QRectF l_oRect;
	if (scene()->selectedItems().size())
	{
		l_oRect = scene()->selectedItems().at(0)->sceneBoundingRect();
		foreach (QGraphicsItem *l_o, scene()->selectedItems())
		{
			l_oRect |= l_o->sceneBoundingRect();
		}
	}
	else
	{
		l_oRect = visibleRect();
	}

	qreal l_fPad;
	QSize l_oSize = viewport()->size();
	if (l_oSize.width() * l_oRect.height() < l_oSize.height() * l_oRect.width())
	{
		l_fPad = (1.0 * l_oRect.width() * (1.0 * PIPAD / (l_oSize.width() - PIPAD * 1.0)));
	}
	else
	{
		l_fPad = (1.0 * l_oRect.height() * (1.0 * PIPAD / (l_oSize.height() - PIPAD * 1.0)));
	}
	l_oRect.adjust(-l_fPad, -l_fPad, l_fPad, l_fPad);
	QRectF l_oViewRect = viewport()->rect();
	QRectF l_oNewRect = matrix().scale(i_iScaleFactor, i_iScaleFactor).mapRect(l_oRect);

	if (scene()->selectedItems().size())
	{
		if (l_oNewRect.width() > l_oViewRect.width() or l_oNewRect.height() > l_oViewRect.height())
		{
			return;
		}
		scale(i_iScaleFactor, i_iScaleFactor);

		QPointF l_oB = l_oViewRect.center();
		QPointF l_oOrig = mapToScene(l_oB.x(), l_oB.y());
		QPointF l_oCenter = l_oOrig + (1 - 1 / i_iScaleFactor) * (l_o - l_oOrig);
		centerOn(l_oCenter);

		ensureVisible(l_oRect);
	}
	else
	{
		qreal l_oDiff = qMin(l_oViewRect.width() - l_oRect.width(), l_oViewRect.height() - l_oRect.height()) / 2.;
		if (l_oDiff > 0)
		{
			l_oRect.adjust(-l_oDiff, -l_oDiff, l_oDiff, l_oDiff);
			l_oNewRect = matrix().scale(i_iScaleFactor, i_iScaleFactor).mapRect(l_oRect);
		}

		if (i_iScaleFactor < 1 and i_iScaleFactor * l_oNewRect.width() < l_oViewRect.width() and i_iScaleFactor * l_oNewRect.height() < l_oViewRect.height())
		{
			fitInView(l_oRect, Qt::KeepAspectRatio);
			centerOn(l_oRect.center());
		}
		else
		{
			QPointF l_oB = l_oViewRect.center();
			QPointF l_oOrig = mapToScene(l_oB.x(), l_oB.y());
			QPointF l_oCenter = l_oOrig + (1 - 1 / i_iScaleFactor) * (l_o - l_oOrig);

			scale(i_iScaleFactor, i_iScaleFactor);
			centerOn(l_oCenter);
		}
	}

	i_rFactor = matrix().mapRect(QRectF(0, 0, 1, 1)).width();
	emit sig_message(i18n("Zoom: %1%", QString::number(i_rFactor * 100, 'g', 4)), 2000);
}

void canvas_view::notify_open_map() {
	QRect l_oRect = viewport()->rect();
	if (m_oItems.size() < 1) // no rectangle
	{
		scene()->setSceneRect(QRectF(mapToScene(l_oRect.topLeft()), mapToScene(l_oRect.bottomRight())));
		return;
	}
	check_canvas_size();
	fit_zoom();
}

void canvas_view::notify_select(const QList<int>& unsel, const QList<int>& sel) {

	edit_off();
	foreach (int k, sel) {
		Q_ASSERT(m_oItems[k] != NULL);
		if (!m_oItems[k]->isSelected())
		{
			m_oItems[k]->setSelected(true);
		}
	}

	foreach (int k, unsel) {
		Q_ASSERT(m_oItems[k] != NULL);
		if (m_oItems[k]->isSelected())
		{
			m_oItems[k]->setSelected(false);
		}
	}
	check_selected();
}

void canvas_view::notify_pic(int i_iId)
{
	canvas_item *l_o = m_oItems[i_iId];
	l_o->update_data();
	l_o->m_oPic->update();
}

void canvas_view::change_colors(QAction* i_oAct)
{
	if (! hasFocus()) return;

	static QColor l_oColor;

	int l_iIdx = -1;
	semantik_win *l_o = (semantik_win*) m_oSemantikWindow;
	for (int i=1; i<l_o->m_oColorGroup->actions().size(); ++i)
	{
		if (l_o->m_oColorGroup->actions()[i]==i_oAct)
		{
			l_iIdx = i;
			if (i==l_o->m_oColorGroup->actions().size()-1)
			{
				l_oColor = QColorDialog::getColor(l_oColor, this);
				if (!l_oColor.isValid())
				{
					return;
				}
			}
			break;
		}
	}

	mem_color* col = new mem_color(m_oMediator);
	col->newColor = l_iIdx;
	col->m_oNewCustomColor.m_oInnerColor = l_oColor;
	col->apply();
}

void canvas_view::change_flags(QAction* i_oAction)
{
	int l_iIndex = 0;
	semantik_win *l_o = (semantik_win*) m_oSemantikWindow;
	for (int i=0; i<l_o->m_oFlagGroup->actions().size(); ++i)
	{
		QAction *l_oAct = l_o->m_oFlagGroup->actions()[i];
		if (l_oAct == i_oAction) l_iIndex = i;
	}
	QString l_sName = m_oMediator->m_oFlagSchemes[l_iIndex].m_sId;

	/*
	canvas_item *l_oItem = selection()[0];
	data_item *l_oData = *m_oMediator + l_oItem->Id();

	if (i_oAction->isChecked()) l_oData->m_oFlags.push_back(l_sName);
	else l_oData->m_oFlags.removeAll(l_sName);
	l_oItem->update_flags(); */

	mem_flag* tmp = new mem_flag(m_oMediator);
	tmp->flag = l_sName;
	tmp->add = i_oAction->isChecked();
	tmp->apply();
}

void canvas_view::check_selected()
{
	QList <canvas_item*> sel = selection();
	semantik_win *l_o = (semantik_win*) m_oSemantikWindow;
	//foreach(QAction* l_oAct, l_o->m_oFlagGroup->actions())
	for (int i=0; i<l_o->m_oFlagGroup->actions().size(); ++i)
	{
		QAction *l_oAct = l_o->m_oFlagGroup->actions()[i];

		if (sel.size() != 1) l_oAct->setChecked(false);
		else
		{
			QString l_sName = m_oMediator->m_oFlagSchemes[i].m_sId;
			if (m_oMediator->m_oItems.contains(sel[0]->Id()))
			{
				data_item& l_oData = m_oMediator->m_oItems[sel[0]->Id()];
				l_oAct->setChecked(l_oData.m_oFlags.contains(l_sName));
			}
		}
		l_oAct->setEnabled(sel.size());
	}
	foreach(QAction* l_oAct, l_o->m_oColorGroup->actions())
	{
		l_oAct->setEnabled(sel.size() >= 1);
	}
}

void canvas_view::deselect_all(bool i_oSignal)
{
	edit_off();
	mem_sel *sel = new mem_sel(m_oMediator);
	if (sel->unsel.empty())
	{
		delete sel;
	}
	else
	{
		sel->apply();
	}
}

void canvas_view::enable_actions()
{
	foreach (QAction* l_o, actions())
	{
		l_o->setEnabled(true); // TODO we could disable the move actions one by one
	}
	enable_menu_actions();
}

void canvas_view::enable_menu_actions()
{
	QList<canvas_item*> sel = selection();
	m_oAddItemAction->setEnabled(sel.size()<=1);
	m_oDeleteAction->setEnabled(sel.size()>0);
	//m_oEditAction->setEnabled(sel.size()==1);
	m_oColorMenu->setEnabled(sel.size()>=1);
	//m_oDataMenu->setEnabled(sel.size()==1);
	m_oSelectSubtreeAction->setEnabled(sel.size()==1);

	if (sel.size() == 1)
	{
		data_item& l_oData = *m_oMediator + sel[0]->Id();

		#define fafa(v, t) v->setChecked(l_oData.m_iDataType == t);
		fafa(m_oTextType, VIEW_TEXT);
		fafa(m_oDiagramType, VIEW_DIAG);
		fafa(m_oTableType, VIEW_TABLE);
		fafa(m_oImageType, VIEW_IMG);
	}

	m_oTextType->setEnabled(sel.size()==1);
	m_oTableType->setEnabled(sel.size()==1);
	m_oImageType->setEnabled(sel.size()==1);
	m_oDiagramType->setEnabled(sel.size()==1);
	/*
	foreach (QAction* l_o, m_oDataMenu->actions())
	{
		l_o->setEnabled(sel.size()==1);
	}*/
}

void canvas_view::mousePressEvent(QMouseEvent *i_oEv)
{
	m_oLastPressPoint = i_oEv->pos();

	if (i_oEv->button() == Qt::MidButton) {
		m_bScroll = true;
		setDragMode(QGraphicsView::ScrollHandDrag);
		return;
	}

	if (i_oEv->button() == Qt::RightButton)
	{
		// first, we cannot edit an item when right-click is selected
		edit_off();

		// select the item under the cursor if available and show the popup menu
		m_oLastPoint = mapToScene(i_oEv->pos());
		QGraphicsItem *l_oItem = itemAt(i_oEv->pos());
		if (l_oItem && l_oItem->type() == CANVAS_ITEM_T)
		{
			if (!l_oItem->isSelected())
			{
				QList<int> lst;
				lst.append(((canvas_item*) l_oItem)->Id());

				mem_sel *sel = new mem_sel(m_oMediator);
				sel->sel = lst;
				sel->apply();
			}
		}
		else
		{
			deselect_all();
		}
		enable_menu_actions();
		m_oMenu->popup(i_oEv->globalPos());
		i_oEv->accept();
		return;
	}

	QGraphicsItem *l_oItem = itemAt(i_oEv->pos());
	QList<canvas_item*> sel = selection();
	if (sel.size() == 1 && QApplication::keyboardModifiers() & Qt::ShiftModifier)
	{
		// link items on click sequences
		if (l_oItem && l_oItem->type() == CANVAS_ITEM_T) {

			int id1 = sel.at(0)->Id();
			int id2 = ((canvas_item*) l_oItem)->Id();
			m_oMediator->link_items(id1, id2);

			QList<int> unlst;
			unlst.append(id1);
			QList<int> lst;
			lst.append(id2);

			mem_sel *sel = new mem_sel(m_oMediator);
			sel->sel = lst;
			sel->unsel = unlst;
			sel->apply();
			return;
		}
	}

	canvas_chain *kk=NULL;
	if (l_oItem && (kk = dynamic_cast<canvas_chain*>(l_oItem)))
	{
		canvas_item *l_oParent = dynamic_cast<canvas_item*>(kk->parentItem());
		Q_ASSERT(l_oParent);

		QRectF r = l_oParent->boundingRect().translated(l_oParent->pos());
		m_oLastPressPoint.setX(r.x() + r.width() / 2.);
		m_oLastPressPoint.setY(r.y() + r.height() / 2.);
		m_oLastPressPoint = mapFromScene(m_oLastPressPoint);

		QRect l_oSel = QRect(m_oLastPressPoint, i_oEv->pos());
		m_oRubberLine->setGeometry(l_oSel);

		m_oRubberLine->setVisible(true);
		return;
	}

        canvas_sort *l_oSort = dynamic_cast<canvas_sort*>(itemAt(i_oEv->pos()));
	if (l_oSort)
		return;
	canvas_sort_toggle *l_oToggle = dynamic_cast<canvas_sort_toggle*>(l_oItem);
	if (l_oToggle)
		return;
	QGraphicsView::mousePressEvent(i_oEv);
}


void canvas_view::mouseReleaseEvent(QMouseEvent *i_oEv)
{
	if (m_oRubberLine->isVisible())
	{
		m_oRubberLine->setVisible(false);
		canvas_item *l_oR1 = NULL;
		canvas_item *l_oR2 = NULL;

		foreach (QGraphicsItem *l_oI1, scene()->items(mapToScene(m_oLastPressPoint)))
		{
			if (l_oI1->type() == CANVAS_ITEM_T)
			{
				l_oR1 = (canvas_item*) l_oI1;
				break;
			}
		}

		foreach (QGraphicsItem *l_oI1, scene()->items(mapToScene(i_oEv->pos())))
		{
			if (l_oI1->type() == CANVAS_ITEM_T)
			{
				l_oR2 = (canvas_item*) l_oI1;
				break;
			}
		}

		if (l_oR1 && l_oR2 && l_oR1 != l_oR2)
		{
			if (i_oEv->modifiers() & Qt::ControlModifier)
			{
				m_oMediator->ref_items(l_oR1->Id(), l_oR2->Id());
			}
			else
			{
				m_oMediator->link_items(l_oR1->Id(), l_oR2->Id());
			}
			deselect_all(); // TODO
		}
		m_oRubberLine->hide();
		return;
	}

	if (m_bScroll)
	{
		m_bScroll = false;
		setDragMode(QGraphicsView::RubberBandDrag);
	}

	QGraphicsItem *l_oItem = itemAt(i_oEv->pos());
	canvas_sort *l_oSort = dynamic_cast<canvas_sort*>(l_oItem);
	if (l_oSort)
	{
		int l_iId = l_oSort->m_oFrom->Id();
		int l_iParentId = m_oMediator->parent_of(l_iId);

		mem_sort *srt = new mem_sort(m_oMediator);
		srt->init(l_iParentId, l_iId, m_oMediator->m_iSortCursor);
		srt->apply();

		m_oMediator->m_iSortCursor++;
		if (m_oMediator->m_iSortCursor >= m_oMediator->num_children(l_iParentId))
		{
			m_oMediator->m_iSortCursor = 0;
		}

		emit sig_message(i18n("Click to set Item %1", QString::number(m_oMediator->m_iSortCursor+1)), -1);
		return;
	}

	canvas_sort_toggle *l_oToggle = dynamic_cast<canvas_sort_toggle*>(l_oItem);
	if (l_oToggle)
	{
		if (scene()->selectedItems().size() == 1)
		{
			canvas_item *it = static_cast<canvas_item*>(l_oToggle->parentItem());

			QList<int> lst;
			lst.append(it->Id());

			mem_sel *sel = new mem_sel(m_oMediator);
			sel->sel = lst;
			sel->unsel = lst;
			sel->m_iSortSel = it->Id();
			sel->apply();
		}
		return;
	}
	QGraphicsView::mouseReleaseEvent(i_oEv);

	edit_off();

	if (i_oEv->button() == Qt::RightButton) return;
	m_bPressed = false;

	check_selection();

	//qDebug()<<"mouse release event!";
	QList<int> lst;
	foreach (QGraphicsItem* k, scene()->selectedItems()) {
		canvas_item* t = (canvas_item*) k;
		lst.append(t->Id());
	}
	if (lst.size()) {
		data_item& p = m_oMediator->m_oItems[lst[0]];
		canvas_item *v = m_oItems[lst[0]];
		if (qAbs(p.m_iXX - v->pos().x()) + qAbs(p.m_iYY - v->pos().y()) > 0.1) {
			mem_move *mv = new mem_move(m_oMediator);
			mv->sel = lst;
			for (int i = 0; i < lst.size(); ++i) {
				data_item& it = m_oMediator->m_oItems[lst[i]];
				mv->oldPos.append(QPointF(it.m_iXX, it.m_iYY));
				mv->newPos.append(m_oItems[lst[i]]->pos());
			}
			mv->apply();
		} // else { qDebug()<<"move too small"; }
	}
}

void canvas_view::mouseDoubleClickEvent(QMouseEvent* i_oEv)
{
	if (i_oEv->button() != Qt::LeftButton) return;
	m_oLastPoint = mapToScene(i_oEv->pos());
	QGraphicsItem *l_oItem = itemAt(i_oEv->pos());

	if (l_oItem) {
		if (l_oItem->type() == CANVAS_ITEM_T) {
			deselect_all();

			canvas_item *l_oR = (canvas_item*) l_oItem;
			l_oR->setSelected(false);

			mem_add *add = new mem_add(m_oMediator);
			add->item.m_iXX = m_oLastPoint.x();
			add->item.m_iYY = m_oLastPoint.y();
			add->parent = l_oR->Id();
			add->apply();
		} else if (l_oItem->type() == CANVAS_LINK_T) {
			canvas_link *l_oLink = (canvas_link*) l_oItem;
			mem_unlink *link = new mem_unlink(m_oMediator);
			link->child = l_oLink->m_oTo->Id();
			link->parent = l_oLink->m_oFrom->Id();
			link->apply();
		}
		else if (l_oItem->type() == CANVAS_REF_T)
		{
			canvas_ref *l_oRef = (canvas_ref*) l_oItem;
			mem_unref *l_oMem  = new mem_unref(m_oMediator);
			l_oMem->m_iChild = l_oRef->m_oTo->Id();
			l_oMem->m_iParent= l_oRef->m_oFrom->Id();
			l_oMem->apply();
		}
	} else if (i_oEv->modifiers() != Qt::ControlModifier) {
		mem_add *add = new mem_add(m_oMediator);
		add->item.m_iXX = m_oLastPoint.x();
		add->item.m_iYY = m_oLastPoint.y();
		add->apply();
	}
}

void canvas_view::check_canvas_size()
{
	QRectF br = scene()->itemsBoundingRect();
	br.adjust(-GAP, -GAP, GAP, GAP);
	br = br.united(scene()->sceneRect());
	scene()->setSceneRect(br);
}

void canvas_view::fit_zoom()
{
	//QRectF l_o = scene()->sceneRect();
	//fitInView(QRectF(l_o.topLeft()+QPointF(100, 100), l_o.size()+QSizeF(200, -200)), Qt::KeepAspectRatio);
#if 0
	QRectF l_o;
	foreach (QGraphicsItem *it, items())
	{
		l_o |= it->boundingRect();
	}
#endif
	//check_canvas_size();
	QRectF l_oRect;
	if (scene()->selectedItems().size())
	{
		l_oRect = scene()->selectedItems().at(0)->sceneBoundingRect();
		foreach (QGraphicsItem *l_o, scene()->selectedItems())
		{
			l_oRect |= l_o->sceneBoundingRect();
		}
	}
	else
	{
		l_oRect = visibleRect();
	}

	qreal l_fPad;
	QSize l_oSize = viewport()->size();
	if (l_oSize.width() * l_oRect.height() < l_oSize.height() * l_oRect.width())
	{
		l_fPad = (1.0 * l_oRect.width() * (1.0 * PIPAD / (l_oSize.width() - PIPAD * 1.0)));
	}
	else
	{
		l_fPad = (1.0 * l_oRect.height() * (1.0 * PIPAD / (l_oSize.height() - PIPAD * 1.0)));

	}
	fitInView(l_oRect.adjusted(-l_fPad, -l_fPad, l_fPad, l_fPad), Qt::KeepAspectRatio);
}

void canvas_view::slot_change_data()
{
	mem_datatype* t = new mem_datatype(m_oMediator);
	t->newDataType = ((QAction*) QObject::sender())->data().toInt();
	if (t->m_iId != NO_ITEM && t->newDataType != t->oldDataType) t->apply();
	else delete(t);
}

void canvas_view::notify_datatype(int i_iId)
{
        canvas_item *l_o = m_oItems[i_iId];
        l_o->update_data();
        l_o->m_oPic->update();
}

rubber_line::rubber_line(QRubberBand::Shape i, QWidget* j) : QRubberBand(i, j)
{
}

void rubber_line::paintEvent(QPaintEvent *i_oEv)
{
	QPainter painter(this);
	painter.setRenderHint(QPainter::Antialiasing);
	painter.setPen(m_oColor);

	if (_direction > 0)
	{
		painter.drawLine(QPoint(0, 0), QPoint(size().width(), size().height()));
	}
	else
	{
		painter.drawLine(QPoint(size().width(), 0), QPoint(0, size().height()));
	}
}

void rubber_line::setGeometry(const QRect& i_o)
{
	_direction = i_o.width() * i_o.height();
	QRubberBand::setGeometry(i_o.normalized());
}

#define HSPACER 32.
#define WSPACER 32.

double canvas_view::compute_height(QMap<int, double> &map, QMap<int, QList<int> >&children, int id) {
	double size = 0;

	QMap<int, QList<int> >::const_iterator it = children.constFind(id);
	if (it != children.cend()) {
		const QList<int> &tmp = it.value();
		size += (tmp.size() - 1) * HSPACER;
		foreach (int k, tmp) {
			size += compute_height(map, children, k);
		}
	}

	double tmp = m_oItems[id]->reorgHeight();
	if (size < tmp) size = tmp;

	map[id] = size;
	//qDebug()<<"size for"<<id<<" "<<size;
	return size;
}

void canvas_view::compute_width(QMap<int, double> &map, QMap<int, QList<int> >&children, int id, int level) {
	double w = m_oItems[id]->boundingRect().width();
	QMap<int, double>::const_iterator jt = map.constFind(level);
	if (jt != map.cend()) {
		const double &val = jt.value();
		map[level] = val > w ? val : w;
	} else {
		map[level] = w;
	}

	QMap<int, QList<int> >::const_iterator it = children.constFind(id);
	if (it != children.cend()) {
		const QList<int> &tmp = it.value();
		foreach (int sub, tmp) {
			compute_width(map, children, sub, level+1);
		}
	}
}

void add_to_list(const QMap<int, QList<int> > &i_oChildren, QList<int> &i_oTmp, QList<QPair<canvas_item*, canvas_item*>> &i_oPairs, int i_iItem, QMap<int, canvas_item*> &i_oItems) {
	i_oTmp.append(i_iItem);
	QList<int> l_oSubList = i_oChildren[i_iItem];
	for (QList<int>::const_iterator it = l_oSubList.cbegin(); it != l_oSubList.cend(); ++it)
	{
		i_oPairs.append(QPair<canvas_item*, canvas_item*>(i_oItems[i_iItem], i_oItems[*it]));
		add_to_list(i_oChildren, i_oTmp, i_oPairs, *it, i_oItems);
	}
}

void canvas_view::reorganize()
{
	QList<int> l_oRoots = m_oMediator->all_roots();
	if (m_oMediator->m_oWindef->m_iReorgType == 1)
	{
		QMap<int, QList<int> > l_oChildren;
		for (QList<QPoint>::const_iterator it = m_oMediator->m_oLinks.cbegin(); it != m_oMediator->m_oLinks.cend(); ++it)
		{
			const QPoint& l_oP = *it;
			l_oChildren[l_oP.x()] << l_oP.y();
		}

		foreach (int l_iRoot, l_oRoots)
		{
			QList<int> l_oTmp;
			QList<QPair<canvas_item*, canvas_item*>> l_oPairs;
			add_to_list(l_oChildren, l_oTmp, l_oPairs, l_iRoot, m_oItems);

			QList<canvas_item*> l_oCanvasItems;
			foreach (int l_oId, l_oTmp)
			{
				canvas_item* l_oItem = m_oItems[l_oId];
				data_item l_oData = m_oMediator->m_oItems[l_oId];

				if (!l_oItem->m_oPic->isVisible())
				{
					l_oItem->m_dWW2 = l_oData.m_iWW / 2.;
					l_oItem->m_dHH2 = l_oData.m_iHH / 2.;
					l_oItem->m_dPosX = l_oData.m_iXX + l_oItem->m_dWW2;
					l_oItem->m_dPosY = l_oData.m_iYY + l_oItem->m_dHH2;
				}
				else
				{
					l_oItem->m_dWW2 = qMax(l_oData.m_iWW, 100.) / 2.;
					l_oItem->m_dHH2 = l_oData.m_iHH / 2. + (100 + 5) / 2.;
					l_oItem->m_dPosX = l_oData.m_iXX + l_oItem->m_dWW2;
					l_oItem->m_dPosY = l_oData.m_iYY + l_oItem->m_dHH2;
				}
				l_oItem->m_dRadius = sqrt(l_oItem->m_dWW2 * l_oItem->m_dWW2 + l_oItem->m_dHH2 * l_oItem->m_dHH2);
				l_oItem->m_dForceX = l_oItem->m_dForceY = 0;
				l_oItem->m_dSpeedX = l_oItem->m_dSpeedY = 0;

				l_oCanvasItems.append(l_oItem);
			}

			QElapsedTimer l_oT;
			l_oT.start();
			while (!l_oT.hasExpired(m_oMediator->m_iForceTime))
			{
				foreach (canvas_item* l_oItem, l_oCanvasItems)
				{
					l_oItem->m_dForceX = l_oItem->m_dForceY = 0;
				}

				for (int i=0; i<l_oCanvasItems.size(); ++i)
				{
					canvas_item *l_oIt = l_oCanvasItems[i];
					for (int j=i+1; j<l_oCanvasItems.size(); ++j)
					{
						canvas_item *l_oJt = l_oCanvasItems[j];
						double l_dDx = l_oIt->m_dPosX - l_oJt->m_dPosX;
						double l_dDy = l_oIt->m_dPosY - l_oJt->m_dPosY;
						double l_dDist = sqrt(l_dDx * l_dDx + l_dDy * l_dDy);
						double l_dNdx = l_dDx/l_dDist;
						double l_dNdy = l_dDy/l_dDist;

						double l_dForceX = 800 * l_dNdx / (l_dDist * l_dDist);
						double l_dForceY = 800 * l_dNdy / (l_dDist * l_dDist);

						l_oIt->m_dForceX += l_dForceX;
						l_oIt->m_dForceY += l_dForceY;

						l_oJt->m_dForceX -= l_dForceX;
						l_oJt->m_dForceY -= l_dForceY;

						#define XPAD 10
						double l_dAdx = qAbs(l_dDx);
						double l_dAdy = qAbs(l_dDy);
						double l_dDw = XPAD + l_oIt->m_dWW2 + l_oJt->m_dWW2;
						double l_dDh = XPAD + l_oIt->m_dHH2 + l_oJt->m_dHH2;
						if (l_dAdx < l_dDw && l_dAdy < l_dDh)
						{
							double l_dClutX = 3 * l_dNdx * (l_dDw - l_dAdx);
							double l_dClutY = 3 * l_dNdy * (l_dDh - l_dAdy);

							l_oIt->m_dForceX += l_dClutX;
							l_oIt->m_dForceY += l_dClutY;

							l_oJt->m_dForceX -= l_dClutX;
							l_oJt->m_dForceY -= l_dClutY;
						}
					}
				}

				for (QList<QPair<canvas_item*,canvas_item*>>::const_iterator it = l_oPairs.cbegin(); it != l_oPairs.cend(); ++it)
				{
					QPair<canvas_item*,canvas_item*> l_oPair = *it;
					canvas_item *l_oParent = l_oPair.first;
					canvas_item *l_oChild = l_oPair.second;

					double l_bX = l_oParent->m_dPosX - l_oChild->m_dPosX;
					double l_bY = l_oParent->m_dPosY - l_oChild->m_dPosY;
					double l_bDist = sqrt(l_bX * l_bX + l_bY * l_bY);
					if (l_bDist < 0.1)
					{
						l_bDist = 0.1;
					}

					double l_bThreshold = (m_oMediator->m_dForceLen * (l_oParent->m_dRadius + l_oChild->m_dRadius)) + 0.01;
					double l_dFx = 0.1 * (- l_bThreshold + l_bDist) * l_bX / l_bDist;
					double l_dFy = (l_bDist - l_bThreshold) * l_bY / (10 * l_bDist);

					l_oParent->m_dForceX -= l_dFx;
					l_oParent->m_dForceY -= l_dFy;

					l_oChild->m_dForceX += l_dFx;
					l_oChild->m_dForceY += l_dFy;
				}

				double l_dMaxSpeed = 1;
				double l_dMaxForce = 1;
				for (QList<canvas_item*>::const_iterator it = l_oCanvasItems.cbegin(); it != l_oCanvasItems.cend(); ++it)
				{
					canvas_item *l_oIt = *it;
					double l_dForce = sqrt(l_oIt->m_dForceX * l_oIt->m_dForceX + l_oIt->m_dForceY * l_oIt->m_dForceY);
					if (l_dForce > l_dMaxForce)
					{
						l_dMaxForce = l_dForce;
					}
					double l_dSpeed = sqrt(l_oIt->m_dSpeedX * l_oIt->m_dSpeedX + l_oIt->m_dSpeedY * l_oIt->m_dSpeedY);
					if (l_dSpeed > l_dMaxSpeed)
					{
						l_dMaxSpeed = l_dSpeed;
					}
				}
				// limit acceleration
				double tick = 2. / qMax(l_dMaxForce, l_dMaxSpeed);

				for (QList<canvas_item*>::const_iterator it = l_oCanvasItems.cbegin(); it != l_oCanvasItems.cend(); ++it)
				{
					// speed
					canvas_item *l_oIt = *it;

					l_oIt->m_dSpeedX /= 2;
					l_oIt->m_dSpeedY *= 0.5;

					l_oIt->m_dSpeedX += l_oIt->m_dForceX * tick;
					l_oIt->m_dSpeedY += l_oIt->m_dForceY * tick;
				}

				for (QList<canvas_item*>::const_iterator it = l_oCanvasItems.cbegin(); it != l_oCanvasItems.cend(); ++it)
				{
					// position
					canvas_item *l_oIt = *it;
					l_oIt->m_dPosX += l_oIt->m_dSpeedX * tick;
					l_oIt->m_dPosY += l_oIt->m_dSpeedY * tick;
				}
			}

			canvas_item *l_oRoot = m_oItems[l_iRoot];
			for (QList<canvas_item*>::const_iterator it = l_oCanvasItems.cbegin(); it != l_oCanvasItems.cend(); ++it)
			{
				canvas_item *l_oIt = *it;
				if (l_oRoot != l_oIt)
				{
					l_oIt->m_dPosX += l_oRoot->pos().x() - l_oRoot->m_dPosX - m_oMediator->m_oItems[l_oIt->m_iId].m_iWW/2.
	+ m_oMediator->m_oItems[l_oRoot->m_iId].m_iWW/2.;
					l_oIt->m_dPosY += l_oRoot->pos().y() - l_oRoot->m_dPosY - m_oMediator->m_oItems[l_oIt->m_iId].m_iHH/2.
	+ m_oMediator->m_oItems[l_oRoot->m_iId].m_iHH/2.;

					l_oIt->setPos(l_oIt->m_dPosX, l_oIt->m_dPosY);
				}
			}
			l_oRoot->m_dPosX = l_oRoot->pos().x();
			l_oRoot->m_dPosY = l_oRoot->pos().y();
			l_oRoot->setPos(l_oRoot->m_dPosX, l_oRoot->m_dPosY);
		}

		// apply the layout for undo/redo
		mem_move *mv = new mem_move(m_oMediator);
		mv->sel.clear();
		foreach(const data_item& x, m_oMediator->m_oItems.values()) {
			canvas_item *v = m_oItems[x.m_iId];
			QPointF p = v->pos();
			if (p.x() != x.m_iXX || p.y() != x.m_iYY) {
				mv->sel.append(x.m_iId);
				mv->oldPos.append(QPointF(x.m_iXX, x.m_iYY));
				mv->newPos.append(v->pos());
			}
		}

		if (mv->sel.size() > 0) {
			mv->apply();
		} else {
			delete mv;
		}
	}
	else
	{
		QMap<int, double> height;
		QMap<int, QList<int> > children;

		for (QList<QPoint>::const_iterator it = m_oMediator->m_oLinks.cbegin(); it != m_oMediator->m_oLinks.cend(); ++it)
		{
			const QPoint& l_oP = *it;
			children[l_oP.x()] << l_oP.y();
		}

		foreach (int k, l_oRoots) {
			double ref = compute_height(height, children, k);
			QMap<int, QList<int> >::iterator it = children.find(k);
			if (it != children.end())
			{
				QList<int> &tmp = it.value();

				ref -= (tmp.size() - 1) * HSPACER;
				int mid = 0;
				int max = 0;
				int tot = 0;
				double left_height = 0;
				foreach (int sub, tmp) {
					tot += height[sub];
					if (tot * (ref - tot) >= max) {
						max = tot * (ref - tot);
						mid = sub;

						left_height += height[sub] + HSPACER;
					} else {
						break;
					}
				}

				QMap<int, double> width;
				compute_width(width, children, k, 0);

				left_height -= HSPACER;

				int left = 1;
				double acc_height = m_oItems[k]->y() + m_oItems[k]->reorgHeight() / 2 - left_height / 2;
				foreach (int sub, tmp) {

					// put the element in place, then recurse

					double y = acc_height + height[sub] / 2 - m_oItems[sub]->reorgHeight() / 2;
					if (left) {
						double x = m_oItems[k]->x() + m_oItems[k]->boundingRect().width() - width[0] - WSPACER;
						m_oItems[sub]->setPos(x - m_oItems[sub]->boundingRect().width(), y);
					} else {
						double x = m_oItems[k]->x() + width[0] + WSPACER;
						m_oItems[sub]->setPos(x, y);
					}

					acc_height += height[sub] + HSPACER;

					m_oItems[k]->update_links();
					pack(width, height, children, sub, 1, left);

					// now to the right
					if (sub == mid) {
						left = 0;
						acc_height = m_oItems[k]->y() + m_oItems[k]->reorgHeight() / 2 - (height[k] - left_height - HSPACER) / 2;
					}
				}
			}
		}

		// now apply the layout for undo/redo
		mem_move *mv = new mem_move(m_oMediator);
		mv->sel.clear();
		foreach(const data_item& x, m_oMediator->m_oItems.values()) {
			canvas_item *v = m_oItems[x.m_iId];
			QPointF p = v->pos();
			if (p.x() != x.m_iXX || p.y() != x.m_iYY) {
				mv->sel.append(x.m_iId);
				mv->oldPos.append(QPointF(x.m_iXX, x.m_iYY));
				mv->newPos.append(v->pos());
			}
		}

		if (mv->sel.size() > 0) {
			mv->apply();
		} else {
			delete mv;
		}

		QRectF l_oRect = visibleRect().adjusted(-PIPAD, -PIPAD, PIPAD, PIPAD);
		scene()->setSceneRect(l_oRect);
	}
}

void canvas_view::pack(QMap<int, double> &width, QMap<int, double> &height, QMap<int, QList<int> >&children, int id, int level, int left) {
	QMap<int, QList<int> >::iterator it = children.find(id);
	if (it != children.end()) {
		QList<int> &tmp = it.value();
		double acc_height = m_oItems[id]->y() + m_oItems[id]->reorgHeight() / 2 - height[id] / 2;
		foreach (int sub, tmp) {
			double y = acc_height + height[sub] / 2 - m_oItems[sub]->reorgHeight()/2;
			if (left) {
				double x = m_oItems[id]->x() + m_oItems[id]->boundingRect().width() - width[0] - WSPACER;
				m_oItems[sub]->setPos(x - m_oItems[sub]->boundingRect().width(), y);
			} else {
				double x = m_oItems[id]->x() + width[0] + WSPACER;
				m_oItems[sub]->setPos(x, y);
			}
			acc_height += height[sub] + HSPACER;
			pack(width, height, children, sub, level+1, left);
		}
		m_oItems[id]->update_links();
	}
}

void canvas_view::export_map_size()
{
	QRectF l_oRect = visibleRect().adjusted(-PIPAD, -PIPAD, PIPAD, PIPAD);

	export_map_dialog* exp = new export_map_dialog(this);

	exp->kurlrequester->setMode(KFile::File | KFile::LocalOnly);
	exp->kurlrequester->setFilter(i18n("*.png|PNG Files (*.png)\n*.svg|SVG Files (*.svg)\n*.pdf|PDF Files (*.pdf)"));

	exp->kurlrequester->setUrl(QUrl(m_oMediator->m_sExportUrl));
	exp->m_oWidthC->setChecked(m_oMediator->m_bExportIsWidth);
	exp->m_iBaseWidth = l_oRect.width();
	exp->m_iBaseHeight = l_oRect.height();

	if (m_oMediator->m_bExportIsWidth)
	{
		if (m_oMediator->m_iExportWidth > 0)
			exp->m_oWidth->setValue(m_oMediator->m_iExportWidth);
		else
			exp->m_oWidth->setValue(l_oRect.width());
	}
	else
	{
		if (m_oMediator->m_iExportHeight > 0)
			exp->m_oHeight->setValue(m_oMediator->m_iExportHeight);
		else
			exp->m_oHeight->setValue(l_oRect.height());
	}

	if (exp->exec() == QDialog::Accepted)
	{
		if (m_oMediator->m_iExportWidth != exp->m_oWidth->value())
		{
			m_oMediator->m_iExportWidth = exp->m_oWidth->value();
			m_oMediator->set_dirty();
		}

		if (m_oMediator->m_iExportHeight != exp->m_oHeight->value())
		{
			m_oMediator->m_iExportHeight = exp->m_oHeight->value();
			m_oMediator->set_dirty();
		}

		if (!exp->kurlrequester->url().isValid() || exp->kurlrequester->url().isEmpty())
		{
			m_oMediator->notify_message(i18n("No destination file selected"), 5000);
			return;
		}

		if (m_oMediator->m_sExportUrl != exp->kurlrequester->url().url())
		{
			m_oMediator->m_sExportUrl = exp->kurlrequester->url().url();
			m_oMediator->set_dirty();
		}

		QPair<int, int> p;
		if (exp->m_oWidthC->isChecked()) {
			p.first = exp->m_oWidth->value();
		} else {
			p.second = exp->m_oHeight->value();
		}

		// TODO upload remote files?
		QUrl url = exp->kurlrequester->url();
		if (url.isRelative()) {
			url.setPath(QDir::homePath() + notr("/") + url.toLocalFile());
		}
		int status = batch_print_map(url, p);
		if (status == 0)
			m_oMediator->notify_message(i18n("Exported '%1'", url.fileName()), 2000);
		else
			KMessageBox::sorry(this, i18n("Could not save to %1", url.fileName()), i18n("Missing picture"));
	}
}

void canvas_view::notify_export_doc()
{
	QRectF l_oRect = visibleRect().adjusted(-PIPAD, -PIPAD, PIPAD, PIPAD);
	QRectF l_oR(0, 0, l_oRect.width(), l_oRect.height());

	// fill with white
	QImage l_oImage((int) l_oR.width(), (int) l_oR.height(), QImage::Format_RGB32);
	l_oImage.fill(qRgb(255,255,255));

	QPainter l_oP;
	l_oP.begin(&l_oImage);
	l_oP.setRenderHints(QPainter::Antialiasing);
	scene()->render(&l_oP, l_oR, l_oRect);
	l_oP.end();

	l_oImage.save(QString(m_oMediator->m_sTempDir + QString("/wholemap.png")));
}

QList<canvas_item*> canvas_view::selection() {
	// FIXME use scene()->selectedItems()
	QList<canvas_item*> ret;

	foreach (QGraphicsItem *tmp, items()) {
		if (tmp->type() == CANVAS_ITEM_T && tmp->isSelected()) {
			ret.append((canvas_item*) tmp);
		}
	}
	return ret;
}

void canvas_view::mouseMoveEvent(QMouseEvent *i_oEv)
{
	if (m_oRubberLine->isVisible())
	{
		QRect l_oSel = QRect(m_oLastPressPoint, i_oEv->pos());
		m_oRubberLine->setGeometry(l_oSel);
	}

	if (m_bScroll)
	{
		QScrollBar *h = horizontalScrollBar();
		QScrollBar *v = verticalScrollBar();
		QPoint d = i_oEv->pos() - m_oLastPressPoint;
		h->setValue(h->value() - d.x());
		v->setValue(v->value() - d.y());
		m_oLastPressPoint = i_oEv->pos();
		//QGraphicsView::mouseMoveEvent(i_oEv);
		return;
	}


	QGraphicsView::mouseMoveEvent(i_oEv);

	QList<QGraphicsItem*> sel = scene()->selectedItems();
	/*if (sel.size() > 4) { // does not solve the repainting problem
		QSet<canvas_link*> lst;
		foreach (QGraphicsItem*tmp, sel) {
			if (tmp->type() == CANVAS_ITEM_T && tmp->isSelected()) {
				canvas_item* x = (canvas_item*) tmp;
				foreach (canvas_link* l_oLink, x->m_oLinks) {
					lst.insert(l_oLink);
				}
			}
		}
		foreach (canvas_link* tmp, lst) {
			tmp->update_pos();
		}
	} else*/ {
		foreach (QGraphicsItem*tmp, sel) {
			if (tmp->type() == CANVAS_ITEM_T && tmp->isSelected()) {
				((canvas_item*) tmp)->update_links();
			}
		}
	}
}

void canvas_view::notify_add_item(int id) {
	Q_ASSERT(! m_oItems.contains(id));
	canvas_item* l_oR = new canvas_item(this, id);
	m_oItems[id] = l_oR;
	l_oR->update_data();
	// do not call methods that create events here

	check_canvas_size();
}

void canvas_view::notify_change_data(int id)
{
	//canvas_item* l_oR = m_oItems.value(id);
	//qDebug()<<"FIXME data has changed!!!"<<id;
}

void canvas_view::notify_delete_item(int id) {
	m_bDeleting = true;
	canvas_item *l_oR1 = m_oItems.value(id);
	Q_ASSERT(l_oR1!=NULL);
	l_oR1->hide();
	scene()->removeItem(l_oR1);
	m_oItems.remove(id);
	delete l_oR1;
	m_bDeleting = false;

	check_canvas_size();
}

void canvas_view::notify_ref_items(int i_iId1, int i_iId2)
{
	canvas_item *l_oR1 = m_oItems.value(i_iId1);
	canvas_item *l_oR2 = m_oItems.value(i_iId2);
	canvas_ref* l_oRef = new canvas_ref(this, l_oR1, l_oR2);
	l_oRef->update_pos();
	l_oR2->update();
}

void canvas_view::notify_unref_items(int i_iId1, int i_iId2)
{
	canvas_item *    l_oR1 = m_oItems.value(i_iId1);
canvas_item *l_oR2 = m_oItems.value(i_iId2);
	foreach (QGraphicsItem *l_oItem, items())
	{
		if (l_oItem->type() == CANVAS_REF_T)
		{
			canvas_ref* l_oRef = (canvas_ref*) l_oItem;
			if (
				(l_oRef->m_oFrom == l_oR1 && l_oRef->m_oTo == l_oR2)
				||
				(l_oRef->m_oFrom == l_oR2 && l_oRef->m_oTo == l_oR1)
			   )
			{
				l_oRef->hide();
				l_oRef->rm_link();
				delete l_oRef;
				break;
			}
		}
	}
	l_oR1->update();
	l_oR2->update();
}

void canvas_view::notify_link_items(int id1, int id2) {
	canvas_item *l_oR1 = m_oItems.value(id1);
	canvas_item *l_oR2 = m_oItems.value(id2);
	canvas_link * l_oLink = new canvas_link(this, l_oR1, l_oR2);
	l_oLink->update_pos();
	l_oR2->update();
}

void canvas_view::notify_unlink_items(int id1, int id2) {
	canvas_item *l_oR1 = m_oItems.value(id1);
	canvas_item *l_oR2 = m_oItems.value(id2);

	foreach (QGraphicsItem *l_oItem, items())
	{
		if (l_oItem->type() == CANVAS_LINK_T)
		{
			canvas_link *l_oLink = (canvas_link*) l_oItem;
			if (
				(l_oLink->m_oFrom == l_oR1 && l_oLink->m_oTo == l_oR2)
				||
				(l_oLink->m_oFrom == l_oR2 && l_oLink->m_oTo == l_oR1)
			   )
			{
				l_oLink->hide();
				l_oLink->rm_link();
				delete l_oLink;
				break;
			}
		}
	}
	l_oR1->update();
	l_oR2->update();
}

void canvas_view::notify_move(const QList<int>&sel, const QList<QPointF>&pos) {
	for (int i = 0; i < sel.size(); ++i) {
		m_oItems[sel[i]]->setPos(pos[i]);
		m_oItems[sel[i]]->update_links();
	}
	check_canvas_size();
}

void canvas_view::notify_repaint(int id) {
	m_oItems[id]->update();
}

void canvas_view::notify_edit(int id) {
	data_item& sel = *m_oMediator + id;
	if (m_oItems[id]->toPlainText() != sel.m_sSummary) {
		m_oItems[id]->setPlainText(sel.m_sSummary);
		m_oItems[id]->adjustSize();
		m_oItems[id]->update_links();
	}
	m_oItems[id]->update();
}

void canvas_view::notify_flag(int id) {
	m_oItems[id]->update_flags();
}

void canvas_view::notify_sort(int i_iId, bool i_bShow)
{
	int j=0;
	for (int i=0; i<m_oMediator->m_oLinks.size(); i++)
	{
		QPoint l_oP = m_oMediator->m_oLinks.at(i);
		if (l_oP.x() == i_iId)
		{
			++j;
			canvas_item *l_oRect = m_oItems.value(l_oP.y());

			if (j != l_oRect->m_iNum)
			{
				l_oRect->m_iNum = j;
			}
			l_oRect->m_oSort->update();
			l_oRect->m_oSort->setVisible(i_bShow);
		}
	}

}

void canvas_view::notify_focus(void* ptr)
{
	bool cond = ptr == this;
	m_oAddItemAction->setEnabled(cond);
	m_oDeleteAction->setEnabled(cond);
	m_oEditAction->setEnabled(cond);
}

void canvas_view::focusInEvent(QFocusEvent *i_oEv)
{
	QGraphicsView::focusInEvent(i_oEv);
	m_oMediator->notify_focus(this);
}

void canvas_view::focusOutEvent(QFocusEvent *i_oEv)
{
	QGraphicsView::focusOutEvent(i_oEv);
	edit_off();
}

void recurse_add(int id, QList<int>& sel, const QList<QPoint>& m_oLinks)
{
	sel.append(id);
	for (int i=0; i < m_oLinks.size(); i++)
	{
		QPoint l_oP = m_oLinks.at(i);
		if (l_oP.x() == id) recurse_add(l_oP.y(), sel, m_oLinks);
	}
}

void canvas_view::slot_select_subtree()
{
	QList<canvas_item*> seli = selection();
	Q_ASSERT(seli.size() == 1);

	mem_sel *sel = new mem_sel(m_oMediator);
	recurse_add(seli.at(0)->m_iId, sel->sel, m_oMediator->m_oLinks);
	sel->apply();
}

int canvas_view::batch_print_map(const QUrl& i_oUrl, QPair<int, int> & p) {

	QRectF l_oRect = visibleRect().adjusted(-PIPAD, -PIPAD, PIPAD, PIPAD);
	foreach (QGraphicsItem*it, scene()->items())
	{
		it->setCacheMode(QGraphicsItem::NoCache); // the magic happens here
	}

	QRectF l_oR(0, 0, l_oRect.width(), l_oRect.height());

	if (p.first != 0) {
		l_oR.setWidth(p.first);
		if (p.second == 0) {
			l_oR.setHeight((p.first * l_oRect.height()) / (double) l_oRect.width());
		}
	}
	if (p.second != 0) {
		l_oR.setHeight(p.second);
		if (p.first == 0) {
			l_oR.setWidth((p.second * l_oRect.width()) / (double) l_oRect.height());
		}
	}

	Qt::AspectRatioMode rat = (p.first == 0 || p.second == 0) ? Qt::KeepAspectRatio : Qt::IgnoreAspectRatio;

	QList<QGraphicsItem*> selection = scene()->selectedItems();
	foreach(QGraphicsItem *l_o, selection) {
		l_o->setSelected(false);
	}
	int l_iRetVal = 0;

	QString url = i_oUrl.path();
	if (url.endsWith("png")) {
		// fill with white
		QImage l_oImage((int) l_oR.width(), (int) l_oR.height(), QImage::Format_RGB32);
		l_oImage.fill(qRgb(255,255,255));

		QPainter l_oP;
		l_oP.begin(&l_oImage);
		l_oP.setRenderHints(QPainter::Antialiasing);
		scene()->render(&l_oP, l_oR, l_oRect, rat);
		l_oP.end();

		l_oImage.save(url);
	} else if (url.endsWith("pdf")) {
		QPrinter l_oPrinter;
		//l_oPrinter.setResolution(QPrinter::HighResolution);
		l_oPrinter.setOrientation(QPrinter::Portrait);
		l_oPrinter.setOutputFormat(QPrinter::PdfFormat);
		l_oPrinter.setPaperSize(l_oR.size(), QPrinter::DevicePixel);
		l_oPrinter.setPageMargins(0, 0, 0, 0, QPrinter::DevicePixel);
		l_oPrinter.setOutputFileName(url);

		QPainter l_oPdf;
		if (l_oPdf.begin(&l_oPrinter))
		{
			m_bDisableGradient = DISABLE_GRADIENT;
			scene()->render(&l_oPdf, l_oR, l_oRect, rat);
			l_oPdf.end();
			m_bDisableGradient = false;
		}
	} else if (url.endsWith("svg")) {
		QSvgGenerator l_oGenerator;
		l_oGenerator.setFileName(url);
		l_oGenerator.setSize(QSize(l_oR.width(), l_oR.height()));
		l_oGenerator.setViewBox(l_oR);
		l_oGenerator.setTitle(i18n("Semantik map"));
		l_oGenerator.setDescription(notr("Generated by Semantik, if it does not render properly check your system fonts!"));
		l_oGenerator.setResolution(QApplication::desktop()->logicalDpiX());

		QPainter l_oP;
		l_oP.begin(&l_oGenerator);
		l_oP.setRenderHints(QPainter::Antialiasing);
		scene()->render(&l_oP, l_oR, l_oRect, rat);
		l_oP.end();
	} else {
		l_iRetVal = 12;
	}
	foreach(QGraphicsItem *l_o, selection) {
		l_o->setSelected(true);
	}
	return l_iRetVal;

}

// The following looks copy-pasted but it is not. Watch carefully
void canvas_view::slot_print()
{
	QPrinter *l_oP = new QPrinter;

	QRectF l_oRect = visibleRect().adjusted(-PIPAD, -PIPAD, PIPAD, PIPAD);
	foreach (QGraphicsItem*it, scene()->items())
	{
		it->setCacheMode(QGraphicsItem::NoCache); // the magic happens here
	}

	QRectF l_oR(0, 0, l_oRect.width(), l_oRect.height());

	l_oP->setOrientation(QPrinter::Portrait);
	l_oP->setOutputFormat(QPrinter::PdfFormat);
	l_oP->setPaperSize(l_oR.size(), QPrinter::DevicePixel);
	l_oP->setPageMargins(0, 0, 0, 0, QPrinter::DevicePixel);

        QPrintDialog l_oD(l_oP, this);
        if (l_oD.exec() != QDialog::Accepted)
        {
		emit sig_message(i18n("Printing cancelled"), 3000);
		return;
	}

	QPainter l_oPdf;
	if (l_oPdf.begin(l_oP))
	{
		m_bDisableGradient = DISABLE_GRADIENT;
		scene()->render(&l_oPdf, QRectF(), l_oRect, Qt::KeepAspectRatio);
		l_oPdf.end();
		m_bDisableGradient = false;
		emit sig_message(i18n("Printing completed"), 5000);
	}
	else
	{
		emit sig_message(i18n("Problem during printing :-("), 5000);
	}
}

void canvas_view::notify_font()
{
	scene()->setFont(m_oMediator->m_oFont);
}

void canvas_view::slot_background_color()
{
	setBackgroundBrush(m_oMediator->m_oColor);
}

void canvas_view::keyPressEvent(QKeyEvent *i_oEvent)
{
	if (i_oEvent->modifiers() & Qt::ControlModifier)
	{
		m_oRubberLine->m_oColor = m_oMediator->m_oAltArrowColor;
	}
	if (m_oRubberLine->isVisible())
	{
		m_oRubberLine->hide();
		m_oRubberLine->show();
	}
	else
	{
		QGraphicsView::keyPressEvent(i_oEvent);
	}
}

void canvas_view::keyReleaseEvent(QKeyEvent *i_oEvent)
{
	if (!(i_oEvent->modifiers() & Qt::ControlModifier))
	{
		m_oRubberLine->m_oColor = m_oMediator->m_oArrowColor;
	}
	if (m_oRubberLine->isVisible())
	{
		m_oRubberLine->hide();
		m_oRubberLine->show();
	}
	else
	{
		QGraphicsView::keyReleaseEvent(i_oEvent);
	}
}

