// Thomas Nagy 2007-2023 GPLV3

#include <QDebug>
/*
  */ %: include "con.h"  
        #include <stdlib.h> 
 #include "sembind.h"  	
  #include "data_item.h"	
  %: include "sem_mediator.h" 

#include "mem_box.h"

mem_del_box::mem_del_box(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_del_box::init(QList<data_box*> _items, QList<data_link*> _links)
{
	items = _items;
	links = _links;
}


void mem_del_box::undo()
{
	data_item& item = model->m_oItems[m_iId];
	foreach (data_box *k, items) {
		item.m_oBoxes[k->m_iId] = k;
		model->notify_add_box(m_iId, k->m_iId);
	}
	foreach (data_link *k, links) {
		item.m_oLinks.append(k);
		model->notify_link_box(m_iId, k);
	}
	undo_dirty();
}

void mem_del_box::redo()
{
	data_item& item = model->m_oItems[m_iId];
	foreach (data_link *k, links) {
		model->notify_unlink_box(m_iId, k);
		item.m_oLinks.removeAll(k);
	}
	foreach (data_box *k, items) {
		model->notify_del_box(m_iId, k->m_iId);
		item.m_oBoxes.remove(k->m_iId);
	}
	redo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_add_box::mem_add_box(sem_mediator* mod, int id, int boxid) : mem_command(mod),
item(model->m_oItems[id])
{
	m_iId = id;
	box = new data_box(boxid);
	box->m_oCustom.m_oInnerColor = QColor("#cafeba");
}

void mem_add_box::redo()
{
	item.m_oBoxes[box->m_iId] = box;
	model->notify_add_box(item.m_iId, box->m_iId);
	redo_dirty();
}

void mem_add_box::undo()
{
	model->notify_del_box(item.m_iId, box->m_iId);
	item.m_oBoxes.remove(box->m_iId);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_edit_box::mem_edit_box(sem_mediator* mod, int id, int bid) : mem_command(mod),
item(model->m_oItems[id])
{
	box = item.m_oBoxes[bid];
	oldText = box->m_sText;
	newHeight = oldHeight = box->m_iHH;
	newWidth = oldWidth = box->m_iWW;
	m_iNewBoxHeight = m_iOldBoxHeight = box->m_iBoxHeight;
}

void mem_edit_box::redo()
{
	box->m_sText = newText;
	box->m_iHH = newHeight;
	box->m_iWW = newWidth;
	box->m_iLabelPosition = m_iNewLabelPosition;
	box->m_iBoxHeight = m_iNewBoxHeight;
	model->notify_edit_box(item.m_iId, box->m_iId);
	redo_dirty();
}

void mem_edit_box::undo()
{
	box->m_sText = oldText;
	box->m_iHH = oldHeight;
	box->m_iWW = oldWidth;
	box->m_iLabelPosition = m_iOldLabelPosition;
	box->m_iBoxHeight = m_iOldBoxHeight;
	model->notify_edit_box(item.m_iId, box->m_iId);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_link_box::mem_link_box(sem_mediator* mod, int id) : mem_command(mod) {
	m_iId = id;
}

void mem_link_box::init(int parent, int parentPos, int child, int childPos) {
	link = new data_link();
	link->m_iParent = parent;
	link->m_iParentPos = parentPos;
	link->m_iChild = child;
	link->m_iChildPos = childPos;
}

void mem_link_box::redo() {
	//qDebug()<<"redo mem_link_box"<<link;
	model->m_oItems[m_iId].m_oLinks.append(link);
	model->notify_link_box(m_iId, link);
	redo_dirty();
}

void mem_link_box::undo() {
	//qDebug()<<"undo mem_link_box"<<link;
	model->notify_unlink_box(m_iId, link);
	model->m_oItems[m_iId].m_oLinks.removeAll(link);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_unlink_box::mem_unlink_box(sem_mediator* mod, int id) : mem_command(mod) {
	m_iId = id;
}

void mem_unlink_box::redo() {
	//qDebug()<<"redo mem_link_box"<<link;
	model->notify_unlink_box(m_iId, link);
	model->m_oItems[m_iId].m_oLinks.removeAll(link);
	redo_dirty();
}

void mem_unlink_box::undo() {
	//qDebug()<<"undo mem_link_box"<<link;
	model->m_oItems[m_iId].m_oLinks.append(link);
	model->notify_link_box(m_iId, link);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

#define CH_COLOR 1
#define CH_BORDER 2
#define CH_PENST 4

mem_prop_box::mem_prop_box(sem_mediator* mod, int id) : mem_command(mod) {
	m_iId = id;
}

void mem_prop_box::redo() {
	for (int i = prev_values.size(); i < items.size(); ++i)
	{
		diagram_item *cur = items.at(i);
		diagram_item *it = new diagram_item();
		it->m_oCustom = cur->m_oCustom;
		it->m_iColor = cur->m_iColor;
		it->pen_style = cur->pen_style;
		it->border_width = cur->border_width;
		prev_values[cur] = it;
	}
	foreach (diagram_item *cur, items) {
		if (change_type & CH_COLOR) {
			cur->m_oCustom = new_props.m_oCustom;
			cur->m_iColor = new_props.m_iColor;
		}
		if (change_type & CH_BORDER) {
			cur->border_width = new_props.border_width;
		}
		if (change_type & CH_PENST) {
			cur->pen_style = new_props.pen_style;
		}
	}
	model->notify_box_props(m_iId, items);
	redo_dirty();
}

void mem_prop_box::undo() {
	foreach (diagram_item *cur, items) {
		if (change_type & CH_COLOR) {
			cur->m_oCustom = prev_values[cur]->m_oCustom;
			cur->m_iColor = prev_values[cur]->m_iColor;
		}
		if (change_type & CH_BORDER) {
			cur->border_width = prev_values[cur]->border_width;
		}
		if (change_type & CH_PENST) {
			cur->pen_style = prev_values[cur]->pen_style;
		}
	}
	model->notify_box_props(m_iId, items);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_pos_box::mem_pos_box(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_pos_box::redo() {
	foreach (data_box* box, next_values.keys()) {
		box->m_iXX = next_values[box].x();
		box->m_iYY = next_values[box].y();
	}
	model->notify_pos_box(m_iId, next_values.keys());
	redo_dirty();
}

void mem_pos_box::undo() {
	foreach (data_box* box, prev_values.keys()) {
		box->m_iXX = prev_values[box].x();
		box->m_iYY = prev_values[box].y();
	}
	model->notify_pos_box(m_iId, prev_values.keys());
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_change_link_box::mem_change_link_box(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_change_link_box::redo() {
	link->copy_from(next);
	model->notify_change_link_box(m_iId, link);
	redo_dirty();
}

void mem_change_link_box::undo() {
	link->copy_from(prev);
	model->notify_change_link_box(m_iId, link);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_size_box::mem_size_box(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_size_box::redo() {
	foreach (data_box* box, next_values.keys()) {
		box->m_iXX = next_values[box].x();
		box->m_iYY = next_values[box].y();
		if (next_values[box].width()  > 0) box->m_iWW = next_values[box].width();
		if (next_values[box].height() > 0) box->m_iHH = next_values[box].height();
	}
	model->notify_size_box(m_iId, next_values.keys());
	redo_dirty();
}

void mem_size_box::undo() {
	foreach (data_box* box, prev_values.keys()) {
		box->m_iXX = prev_values[box].x();
		box->m_iYY = prev_values[box].y();
		if (prev_values[box].width()  > 0) box->m_iWW = prev_values[box].width();
		if (prev_values[box].height() > 0) box->m_iHH = prev_values[box].height();
	}
	model->notify_size_box(m_iId, prev_values.keys());
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_edit_link::mem_edit_link(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_edit_link::redo() {
	link->copy_from(next);
	QList<diagram_item*> items;
	items.append(link);
	model->m_oLinkState.copy_from(next);
	model->notify_box_props(m_iId, items);
	redo_dirty();
}

void mem_edit_link::undo() {
	link->copy_from(prev);
	QList<diagram_item*> items;
	items.append(link);
	model->m_oLinkState.copy_from(m_oPrevLinkState);
	model->notify_box_props(m_iId, items);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_import_box::mem_import_box(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_import_box::init(QList<data_box*> _items, QList<data_link*> _links)
{
	new_items = _items;
	new_links = _links;

	data_item& l_oItem = model->m_oItems[m_iId];
	old_items.append(l_oItem.m_oBoxes.values());
	old_links.append(l_oItem.m_oLinks);
	m_iOldFont = l_oItem.m_oDiagramFont;

	m_bExportIsWidthOld = l_oItem.m_bExportIsWidth;
	m_iExportWidthOld = l_oItem.m_iExportWidth;
	m_iExportHeightOld = l_oItem.m_iExportHeight;
	m_sExportUrlOld = l_oItem.m_sExportUrl;
}


void mem_import_box::undo()
{
	data_item& l_oItem = model->m_oItems[m_iId];
	l_oItem.m_oDiagramFont = m_iOldFont;
	model->notify_change_properties(NULL);
	foreach (data_link *k, new_links) {
		model->notify_unlink_box(m_iId, k);
		l_oItem.m_oLinks.removeAll(k);
	}
	foreach (data_box *k, new_items) {
		model->notify_del_box(m_iId, k->m_iId);
		l_oItem.m_oBoxes.remove(k->m_iId);
	}
	foreach (data_box *k, old_items) {
		l_oItem.m_oBoxes[k->m_iId] = k;
		model->notify_add_box(m_iId, k->m_iId);
	}
	foreach (data_link *k, old_links) {
		l_oItem.m_oLinks.append(k);
		model->notify_link_box(m_iId, k);
	}
	if (model->m_bIsDiagram)
	{
		model->m_oColorSchemes = m_oOldColorSchemes;
	}
	l_oItem.m_bExportIsWidth = m_bExportIsWidthOld;
	l_oItem.m_iExportWidth = m_iExportWidthOld;
	l_oItem.m_iExportHeight = m_iExportHeightOld;
	l_oItem.m_sExportUrl = m_sExportUrlOld;
	undo_dirty();
}

void mem_import_box::redo()
{
	data_item& l_oItem = model->m_oItems[m_iId];
	l_oItem.m_oDiagramFont = m_iNewFont;
	model->notify_change_properties(NULL);
	foreach (data_link *k, old_links) {
		model->notify_unlink_box(m_iId, k);
		l_oItem.m_oLinks.removeAll(k);
	}
	foreach (data_box *k, old_items) {
		model->notify_del_box(m_iId, k->m_iId);
		l_oItem.m_oBoxes.remove(k->m_iId);
	}
	foreach (data_box *k, new_items) {
		l_oItem.m_oBoxes[k->m_iId] = k;
		model->notify_add_box(m_iId, k->m_iId);
	}
	foreach (data_link *k, new_links) {
		l_oItem.m_oLinks.append(k);
		model->notify_link_box(m_iId, k);
	}
	if (model->m_bIsDiagram)
	{
		model->m_oColorSchemes = m_oNewColorSchemes;
	}
	l_oItem.m_bExportIsWidth = m_bExportIsWidthNew;
	l_oItem.m_iExportWidth = m_iExportWidthNew;
	l_oItem.m_iExportHeight = m_iExportHeightNew;
	l_oItem.m_sExportUrl = m_sExportUrlNew;
	redo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_matrix::mem_matrix(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_matrix::redo() {
	m_oBox->m_oRowSizes.clear();
	foreach (int l_i, m_oNewRowSizes) {
		m_oBox->m_oRowSizes.append(l_i);
	}
	m_oBox->m_oColSizes.clear();
	foreach (int l_i, m_oNewColSizes) {
		m_oBox->m_oColSizes.append(l_i);
	}
	m_oBox->m_iWW = m_iNewWW;
	m_oBox->m_iHH = m_iNewHH;
	QList<data_box*> lst;
	lst.push_back(m_oBox);
	model->notify_size_box(m_iId, lst);
	redo_dirty();
}

void mem_matrix::undo() {
	m_oBox->m_oRowSizes.clear();
	foreach (int l_i, m_oOldRowSizes) {
		m_oBox->m_oRowSizes.append(l_i);
	}
	m_oBox->m_oColSizes.clear();
	foreach (int l_i, m_oOldColSizes) {
		m_oBox->m_oColSizes.append(l_i);
	}
	m_oBox->m_iWW = m_iOldWW;
	m_oBox->m_iHH = m_iOldHH;
	QList<data_box*> lst;
	lst.push_back(m_oBox);
	model->notify_size_box(m_iId, lst);
	undo_dirty();
}

void mem_matrix::init(data_box *i_oBox) {
	m_oBox = i_oBox;
	foreach (int l_i, m_oBox->m_oRowSizes) {
		m_oOldRowSizes.push_back(l_i);
		m_oNewRowSizes.push_back(l_i);
	}
	foreach (int l_i, m_oBox->m_oColSizes) {
		m_oOldColSizes.push_back(l_i);
		m_oNewColSizes.push_back(l_i);
	}
	m_iOldWW = m_iNewWW = m_oBox->m_iWW;
	m_iOldHH = m_iNewHH = m_oBox->m_iHH;
}

///////////////////////////////////////////////////////////////////

mem_class::mem_class(sem_mediator* mod, int id) : mem_command(mod),  m_oOldBox(id), m_oNewBox(id)
{
	m_iId = id;
}

void mem_class::redo() {
	data_item& item = model->m_oItems[m_iId];
	data_box *l_oBox = item.m_oBoxes[m_iBoxId];
	*l_oBox = m_oNewBox;

	QList<data_box*> lst;
	lst.push_back(l_oBox);
	model->notify_size_box(m_iId, lst);
	redo_dirty();
}

void mem_class::undo() {
	data_item& item = model->m_oItems[m_iId];
	data_box *l_oBox = item.m_oBoxes[m_iBoxId];
	*l_oBox = m_oOldBox;

	QList<data_box*> lst;
	lst.push_back(l_oBox);
	model->notify_size_box(m_iId, lst);
	undo_dirty();
}

void mem_class::init(data_box *i_oBox) {
	m_iBoxId = i_oBox->m_iId;
	m_oNewBox = m_oOldBox = *i_oBox;
}

///////////////////////////////////////////////////////////////////

mem_entity::mem_entity(sem_mediator* mod, int id) : mem_command(mod),  m_oOldBox(id), m_oNewBox(id)
{
	m_iId = id;
}

void mem_entity::redo() {
	data_item& item = model->m_oItems[m_iId];
	data_box *l_oBox = item.m_oBoxes[m_iBoxId];
	*l_oBox = m_oNewBox;

	QList<data_box*> lst;
	lst.push_back(l_oBox);
	model->notify_size_box(m_iId, lst);
	redo_dirty();
}

void mem_entity::undo() {
	data_item& item = model->m_oItems[m_iId];
	data_box *l_oBox = item.m_oBoxes[m_iBoxId];
	*l_oBox = m_oOldBox;

	QList<data_box*> lst;
	lst.push_back(l_oBox);
	model->notify_size_box(m_iId, lst);
	undo_dirty();
}

void mem_entity::init(data_box *i_oBox) {
	m_iBoxId = i_oBox->m_iId;
	m_oNewBox = m_oOldBox = *i_oBox;
}

///////////////////////////////////////////////////////////////////

mem_diagram_properties::mem_diagram_properties(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_diagram_properties::redo() {
	data_item& item = model->m_oItems[m_iId];
	item.m_oDiagramFont = m_oNewFont;
	model->notify_change_properties(NULL);
}

void mem_diagram_properties::undo() {
	data_item& item = model->m_oItems[m_iId];
	item.m_oDiagramFont = m_oOldFont;
	model->notify_change_properties(NULL);
}

///////////////////////////////////////////////////////////////////

mem_size_sequence::mem_size_sequence(sem_mediator* i_oMod, int i_iId, data_box* i_oBox) : mem_command(i_oMod), m_oPrevBox(i_iId), m_oNextBox(i_iId)
{
	m_iId = i_iId;
	m_iBoxId = i_oBox->m_iId;
	m_oPrevBox.m_iWW = i_oBox->m_iWW;
	m_oPrevBox.m_iHH = i_oBox->m_iHH;
	m_oPrevBox.m_iBoxHeight = i_oBox->m_iBoxHeight;
}

void mem_size_sequence::redo() {
	data_item& item = model->m_oItems[m_iId];
        data_box *l_oBox = item.m_oBoxes[m_iBoxId];
	l_oBox->m_iWW = m_oNextBox.m_iWW;
	l_oBox->m_iHH = m_oNextBox.m_iHH;
	l_oBox->m_iBoxHeight = m_oNextBox.m_iBoxHeight;
	model->notify_sequence_box(m_iId, m_iBoxId);
	redo_dirty();
}

void mem_size_sequence::undo() {
	data_item& item = model->m_oItems[m_iId];
        data_box *l_oBox = item.m_oBoxes[m_iBoxId];
	l_oBox->m_iWW = m_oPrevBox.m_iWW;
	l_oBox->m_iHH = m_oPrevBox.m_iHH;
	l_oBox->m_iBoxHeight = m_oPrevBox.m_iBoxHeight;
	model->notify_sequence_box(m_iId, m_iBoxId);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_text_align_box::mem_text_align_box(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_text_align_box::redo() {
	data_item& l_oItem = model->m_oItems[m_iId];
	foreach (data_box box, m_oPrevValues)
	{
		l_oItem.m_oBoxes[box.m_iId]->m_iAlign = m_oAlign;
	}
	model->notify_text_align(m_iId, m_oPrevValues);
	redo_dirty();
}

void mem_text_align_box::undo() {
	data_item& l_oItem = model->m_oItems[m_iId];
	foreach (data_box box, m_oPrevValues) {
		 l_oItem.m_oBoxes[box.m_iId]->m_iAlign = box.m_iAlign;
	}
	model->notify_text_align(m_iId, m_oPrevValues);
	undo_dirty();
}

///////////////////////////////////////////////////////////////////

mem_insert_box::mem_insert_box(sem_mediator* mod, int id) : mem_command(mod)
{
	m_iId = id;
}

void mem_insert_box::init_data(sem_mediator*i_oOld, sem_mediator*i_oNew)
{
	data_item& l_oOldItem = i_oOld->m_oItems[m_iId];
	data_item& l_oNewItem = i_oNew->m_oItems[i_oNew->m_oItems.keys().at(0)];

	int l_iDeltaY = 0;
	int l_iDeltaX = 0;
	if (l_oOldItem.m_oBoxes.size() > 0 && l_oNewItem.m_oBoxes.size() > 0)
	{
		double l_dOldMinX = 268435456;
		double l_dOldMaxX = -268435456;
		double l_dOldBottomY = -268435456;
		foreach (data_box *l_oBox, l_oOldItem.m_oBoxes.values())
		{
			l_dOldMinX = qMin((double) l_oBox->m_iXX, l_dOldMinX);
			l_dOldMaxX = qMax((double) l_oBox->m_iXX + l_oBox->m_iWW, l_dOldMaxX);
			l_dOldBottomY = qMax((double) l_oBox->m_iYY + l_oBox->m_iHH, l_dOldBottomY);
		}

		double l_dNewMinX = 268435456;
		double l_dNewMaxX = -268435456;
		double l_dNewTopY = 268435456;
		foreach (data_box *l_oBox, l_oNewItem.m_oBoxes.values())
		{
			l_dNewMinX = qMin((double) l_oBox->m_iXX, l_dNewMinX);
			l_dNewMaxX = qMax((double) l_oBox->m_iXX + l_oBox->m_iWW, l_dNewMaxX);
			l_dNewTopY = qMin((double) l_oBox->m_iYY, l_dNewTopY);
		}
		l_iDeltaY = l_dNewTopY - l_dOldBottomY - 30;
		l_iDeltaX = (l_dNewMaxX + l_dNewMinX - l_dOldMaxX - l_dOldMinX) / 2.;
	}

	QHash<int, int> l_oTranslate;
	foreach (int l_iVal, l_oNewItem.m_oBoxes.keys())
	{
		data_box * l_oBox = l_oNewItem.m_oBoxes[l_iVal];
		l_oBox->m_iId = i_oOld->next_box_seq(m_iId);
		l_oBox->m_iXX -= l_iDeltaX;
		l_oBox->m_iYY -= l_iDeltaY;
		l_oTranslate[l_iVal] = l_oBox->m_iId;
		m_oNewBoxes.append(l_oBox);
	}

	foreach (data_link *l_oLink, l_oNewItem.m_oLinks)
	{
		l_oLink->m_iId = i_oOld->next_box_link_seq(m_iId);
		l_oLink->m_iParent = l_oTranslate[l_oLink->m_iParent];
		l_oLink->m_iChild = l_oTranslate[l_oLink->m_iChild];
		m_oNewLinks.append(l_oLink);
	}
}

void mem_insert_box::undo()
{
	data_item& l_oItem = model->m_oItems[m_iId];
	foreach (data_link *k, m_oNewLinks) {
		model->notify_unlink_box(m_iId, k);
		l_oItem.m_oLinks.removeAll(k);
	}
	foreach (data_box *k, m_oNewBoxes) {
		model->notify_del_box(m_iId, k->m_iId);
		l_oItem.m_oBoxes.remove(k->m_iId);
	}
	undo_dirty();
}

void mem_insert_box::redo()
{
	data_item& l_oItem = model->m_oItems[m_iId];
	foreach (data_box *k, m_oNewBoxes) {
		l_oItem.m_oBoxes[k->m_iId] = k;
		model->notify_add_box(m_iId, k->m_iId);
	}
	foreach (data_link *k, m_oNewLinks) {
		l_oItem.m_oLinks.append(k);
		model->notify_link_box(m_iId, k);
	}
	redo_dirty();
}
