//----------------------------------------------------------------------------------------------------------------------
// foo_midi: A foobar2000 component to play MIDI files (https://github.com/stuerp/foo_midi)
// Copyright (c) 2022-2024 Peter Stuer
//
// This file was written from scratch for QB64-PE
//----------------------------------------------------------------------------------------------------------------------

#include "InstrumentBankManager.h"
#include "../../../data/miniz.h"

InstrumentBankManager &InstrumentBankManager::Instance() {
    static InstrumentBankManager instance;
    return instance;
}

void InstrumentBankManager::SetPath(const char *path) {
    if (path && path[0]) {
        if (filepath_has_extension(path, "ad") || filepath_has_extension(path, "opl") || filepath_has_extension(path, "op2") ||
            filepath_has_extension(path, "tmb") || filepath_has_extension(path, "wopl")) {
            type = Type::Opal;
            location = Location::File;
            data.clear();
            fileName = path;
        } else if (filepath_has_extension(path, "sf2")) {
            type = Type::Primesynth;
            location = Location::File;
            data.clear();
            fileName = path;
        } else if (filepath_has_extension(path, "sf3") || filepath_has_extension(path, "sfo")) {
            type = Type::TinySoundFont;
            location = Location::File;
            data.clear();
            fileName = path;
        }
#ifdef _WIN32
        else if (filepath_has_extension(path, "dll")) {
            type = Type::VSTi;
            location = Location::File;
            data.clear();
            fileName = path;
        }
#endif
    } else {
        SetDefaults();
    }
}

void InstrumentBankManager::SetData(const uint8_t *data, size_t size, Type type) {
    if (data && size && type < Type::VSTi) {
        fileName.clear();
        location = Location::Memory;
        this->data.assign(data, data + size);

        if (type == Type::Primesynth)
            this->type = Type::TinySoundFont; // primesynth cannot load soundfonts from memory
        else
            this->type = type;
    } else {
        SetDefaults();
    }
}

// GENMIDI(XG).wopl (MIT licensed) from DMXOPL3 (https://github.com/sneakernets/DMXOPL)
const uint8_t InstrumentBankManager::defaultBank[] = {
    120, 218, 140, 206, 179, 98,  134, 65,  16,  133, 225, 19,  163, 137, 81,  254, 182, 62,  111, 23,  219, 78,  25,  219, 188, 252, 56,  51,  91,  156, 224,
    173, 159, 193, 214, 210, 242, 188, 87,  30,  29,  89,  156, 67,  3,   208, 143, 6,   252, 163, 209, 221, 171, 243, 216, 226, 117, 37,  102, 192, 107, 21,
    225, 132, 160, 117, 170, 168, 197, 114, 107, 99,  101, 99,  242, 128, 93,  187, 8,   247, 11,  196, 150, 239, 174, 109, 84,  47,  194, 227, 2,   13,  31,
    226, 234, 125, 135, 207, 201, 160, 254, 193, 1,   234, 244, 138, 203, 73,  76,  5,   191, 130, 132, 136, 240, 11,  4,   181, 60,  236, 154, 183, 167, 62,
    215, 132, 62,  104, 195, 248, 22,  190, 3,   90,  94,  132, 99,  64,  235, 134, 92,  9,   64,  27,  209, 43,  53,  208, 114, 186, 35,  4,   109, 84,  132,
    1,   175, 85,  132, 91,  3,   173, 87,  132, 23,  128, 150, 212, 63,  192, 107, 214, 43,  62,  104, 131, 248, 187, 229, 235, 231, 195, 187, 216, 220, 233,
    67,  44,  247, 177, 203, 9,   243, 176, 235, 196, 250, 106, 217, 212, 140, 69,  220, 32,  15,  188, 50,  102, 29,  80,  109, 92,  219, 118, 223, 59,  21,
    81,  69,  49,  54,  105, 51,  96,  59,  144, 138, 36,  151, 239, 172, 87,  40,  198, 207, 233, 24,  228, 242, 58,  131, 152, 160, 177, 133, 70,  72,  34,
    46,  191, 9,   210, 59,  38,  213, 237, 173, 144, 158, 252, 234, 250, 251, 255, 233, 189, 153, 228, 245, 234, 244, 186, 22,  18,  24,  156, 202, 191, 23,
    225, 204, 138, 141, 36,  14,  109, 175, 50,  155, 115, 207, 236, 211, 174, 28,  91,  128, 198, 128, 221, 23,  139, 91,  1,   237, 7,   81,  35,  220, 165,
    181, 90,  70,  216, 198, 137, 70,  33,  200, 178, 39,  47,  57,  46,  250, 126, 53,  92,  67,  61,  174, 228, 56,  60,  111, 236, 173, 1,   118, 96,  71,
    83,  212, 234, 14,  198, 181, 227, 68,  179, 82,  112, 6,   117, 154, 1,   190, 95,  37,  125, 34,  99,  152, 16,  189, 111, 36,  199, 40,  254, 137, 125,
    205, 197, 4,   8,   162, 232, 150, 82,  7,   197, 210, 145, 225, 7,   104, 137, 116, 248, 32,  45,  125, 109, 251, 3,   34,  158, 192, 19,  171, 237, 240,
    166, 173, 231, 198, 217, 239, 12,  39,  224, 230, 130, 224, 114, 185, 213, 195, 227, 162, 231, 214, 207, 99,  112, 231, 49,  228, 189, 249, 243, 152, 136,
    114, 148, 183, 132, 204, 64,  60,  122, 220, 125, 173, 182, 45,  104, 119, 153, 177, 186, 147, 124, 40,  0,   89,  157, 250, 119, 84,  217, 169, 247, 169,
    59,  193, 144, 123, 247, 222, 119, 129, 42,  84,  157, 200, 176, 114, 195, 185, 203, 235, 112, 130, 137, 16,  138, 224, 113, 165, 126, 141, 101, 123, 198,
    159, 163, 238, 226, 228, 49,  232, 167, 143, 62,  7,   124, 7,   223, 89,  109, 68,  35,  49,  43,  16,  180, 163, 93,  89,  35,  33,  203, 238, 194, 195,
    71,  197, 37,  107, 146, 53,  18,  67,  147, 196, 123, 241, 151, 2,   37,  34,  17,  155, 67,  198, 149, 57,  9,   56,  67,  158, 91,  77,  78,  130, 172,
    76,  254, 144, 49,  36,  39,  69,  124, 31,  163, 60,  61,  154, 154, 205, 144, 25,  139, 27,  200, 106, 4,   66,  94,  158, 71,  77,  189, 3,   197, 63,
    113, 29,  56,  66,  37,  67,  18,  174, 97,  95,  57,  108, 106, 124, 106, 28,  224, 145, 44,  149, 159, 74,  225, 251, 195, 99,  15,  98,  72,  25,  99,
    232, 151, 227, 15,  130, 86,  208, 138, 185, 234, 33,  143, 157, 98,  27,  69,  229, 238, 7,   58,  69,  183, 114, 61,  197, 188, 29,  15,  116, 2,   207,
    179, 175, 185, 153, 80,  76,  250, 165, 212, 4,   246, 125, 192, 126, 221, 38,  141, 42,  120, 160, 133, 33,  116, 177, 175, 92,  86,  48,  19,  73,  32,
    241, 238, 36,  230, 7,   38,  38,  32,  38,  126, 163, 160, 232, 199, 227, 15,  73,  208, 161, 111, 216, 26,  178, 35,  65,  59,  108, 102, 138, 195, 167,
    160, 156, 65,  36,  101, 194, 248, 81,  84,  157, 49,  57,  193, 142, 62,  57,  73,  210, 239, 226, 60,  246, 53,  135, 72,  166, 115, 147, 222, 56,  246,
    54,  148, 187, 199, 6,   48,  131, 222, 24,  0,   57,  135, 156, 179, 178, 47,  20,  176, 122, 204, 40,  178, 25,  207, 77,  181, 250, 250, 177, 131, 130,
    167, 109, 123, 137, 232, 32,  220, 140, 155, 87,  70,  141, 205, 157, 70,  84,  187, 60,  218, 109, 132, 51,  69,  194, 5,   65,  145, 140, 196, 225, 83,
    200, 3,   215, 239, 157, 79,  140, 235, 25,  186, 255, 250, 189, 42,  113, 222, 69,  142, 83,  72,  16,  138, 8,   81,  70,  143, 193, 107, 178, 95,  121,
    202, 168, 138, 133, 205, 12,  97,  41,  251, 154, 139, 185, 120, 125, 216, 91,  114, 163, 138, 200, 202, 199, 230, 99,  31,  71,  189, 23,  63,  54,  159,
    72,  215, 72,  215, 52,  7,   251, 162, 129, 96,  166, 67,  56,  113, 40,  204, 247, 38,  252, 229, 196, 51,  84,  47,  146, 122, 142, 124, 67,  173, 34,
    193, 171, 120, 181, 205, 52,  187, 50,  63,  239, 100, 183, 40,  15,  92,  213, 81,  73,  200, 79,  19,  149, 164, 159, 35,  248, 19,  149, 162, 180, 81,
    218, 216, 24,  8,   176, 188, 180, 236, 108, 4,   200, 227, 111, 211, 171, 54,  46,  38,  250, 165, 15,  47,  130, 79,  93,  197, 208, 229, 215, 46,  34,
    164, 143, 244, 205, 85,  213, 5,   170, 174, 118, 86,  161, 250, 174, 209, 207, 197, 234, 82,  134, 206, 217, 51,  37,  17,  124, 5,   217, 111, 132, 187,
    109, 45,  135, 39,  10,  4,   21,  94,  165, 77,  162, 250, 208, 168, 66,  28,  228, 126, 4,   77,  200, 105, 233, 236, 150, 93,  164, 112, 252, 223, 197,
    178, 61,  15,  197, 37,  34,  189, 241, 239, 116, 137, 177, 125, 33,  133, 12,  217, 233, 56,  125, 86,  156, 41,  171, 54,  22,  55,  205, 80,  221, 73,
    21,  70,  134, 91,  73,  189, 67,  124, 59,  146, 239, 74,  238, 252, 212, 59,  146, 239, 23,  201, 119, 9,   156, 90,  237, 48,  108, 52,  182, 109, 171,
    59,  57,  55,  93,  46,  86,  161, 175, 135, 254, 224, 232, 25,  148, 35,  212, 252, 108, 244, 12,  194, 125, 56,  137, 33,  16,  50,  141, 240, 201, 213,
    94,  166, 110, 215, 200, 251, 130, 251, 186, 79,  189, 196, 221, 50,  242, 62,  41,  77,  124, 234, 21,  102, 245, 161, 167, 47,  110, 118, 157, 84,  239,
    5,   181, 200, 115, 237, 216, 81,  120, 238, 126, 103, 28,  40,  252, 236, 24,  33,  152, 28,  21,  120, 191, 184, 252, 74,  51,  218, 21,  181, 174, 52,
    195, 199, 57,  190, 109, 0,   138, 112, 21,  59,  69,  129, 251, 186, 209, 73,  193, 189, 186, 119, 18,  13,  96,  168, 225, 218, 254, 73,  240, 250, 176,
    210, 138, 197, 237, 104, 220, 178, 51,  49,  20,  48,  134, 171, 153, 30,  10,  26,  26,  248, 115, 10,  127, 78,  96,  8,   156, 65,  254, 84,  254, 116,
    230, 49,  214, 120, 122, 236, 176, 21,  136, 157, 228, 1,   183, 98,  30,  201, 113, 215, 222, 11,  229, 15,  132, 27,  39,  229, 241, 139, 57,  106, 102,
    168, 116, 123, 233, 246, 111, 222, 102, 147, 17,  139, 101, 201, 44,  136, 30,  233, 137, 119, 69,  95,  224, 206, 14,  120, 164, 225, 119, 160, 191, 121,
    77,  7,   120, 191, 248, 38,  146, 156, 65,  171, 189, 194, 10,  119, 155, 209, 111, 191, 205, 130, 244, 219, 244, 72,  172, 91,  214, 239, 25,  253, 12,
    40,  254, 183, 207, 148, 74,  140, 142, 42,  184, 16,  23,  158, 192, 16,  177, 2,   155, 234, 102, 203, 11,  23,  245, 94,  157, 252, 53,  244, 215, 30,
    91,  12,  247, 165, 147, 45,  98,  254, 197, 227, 159, 73,  124, 134, 89,  21,  53,  227, 172, 239, 197, 178, 158, 130, 235, 65,  45,  109, 232, 63,  4,
    207, 206, 55,  74,  80,  118, 137, 113, 16,  250, 174, 198, 6,   240, 58,  217, 30,  50,  34,  233, 255, 239, 201, 222, 181, 224, 86,  247, 31,  146, 22,
    252, 236, 142, 247, 168, 94,  120, 251, 36,  116, 229, 243, 205, 224, 85,  206, 97,  240, 230, 100, 216, 119, 136, 84,  84,  222, 249, 30,  245, 22,  62,
    181, 113, 134, 161, 22,  181, 237, 91,  195, 241, 96,  38,  39,  156, 72,  194, 133, 122, 140, 77,  160, 190, 136, 181, 29,  102, 163, 51,  191, 10,  81,
    232, 48,  204, 108, 54,  75,  151, 158, 28,  73,  137, 41,  202, 229, 83,  217, 187, 240, 220, 126, 255, 10,  226, 32,  186, 147, 238, 92,  103, 217, 33,
    43,  156, 187, 70,  185, 84,  67,  50,  22,  139, 218, 157, 145, 69,  180, 81,  49,  22,  67,  223, 17,  89,  68,  48,  132, 33,  206, 96,  32,  155, 57,
    167, 104, 172, 160, 145, 203, 59,  154, 168, 115, 10,  114, 3,   185, 129, 205, 48,  33,  27,  200, 57,  195, 80,  84,  95,  223, 86,  73,  34,  197, 175,
    151, 83,  134, 20,  169, 183, 236, 245, 133, 82,  197, 51,  21,  207, 52,  219, 225, 120, 212, 232, 100, 113, 200, 238, 131, 140, 234, 188, 142, 121, 98,
    239, 224, 81,  93,  114, 124, 144, 222, 146, 222, 242, 71,  205, 30,  59,  100, 107, 237, 241, 40,  19,  52,  83,  101, 171, 17,  15,  4,   87,  248, 234,
    224, 60,  207, 35,  41,  22,  232, 133, 29,  21,  226, 232, 221, 107, 116, 73,  191, 209, 47,  137, 169, 29,  199, 106, 168, 98,  42,  102, 171, 181, 109,
    155, 21,  48,  226, 199, 57,  50,  213, 106, 153, 205, 180, 79,  191, 195, 230, 208, 182, 251, 69,  183, 234, 127, 71,  44,  251, 245, 93,  207, 81,  226,
    34,  174, 203, 89,  207, 51,  99,  236, 32,  33,  158, 225, 145, 140, 185, 89,  84,  196, 38,  243, 20,  124, 247, 29,  169, 129, 199, 197, 144, 119, 97,
    178, 6,   232, 69,  175, 223, 234, 137, 24,  97,  43,  115, 173, 63,  194, 167, 24,  8,   192, 2,   233, 224, 1,   81,  212, 159, 126, 74,  44,  67,  234,
    0,   202, 138, 119, 108, 35,  92,  147, 152, 139, 137, 92,  81,  250, 141, 93,  146, 232, 229, 167, 55,  110, 236, 147, 36,  239, 174, 151, 124, 162, 235,
    128, 235, 64,  250, 244, 90,  75,  56,  102, 246, 116, 134, 204, 217, 179, 227, 75,  8,   162, 203, 184, 166, 69,  20,  125, 63,  30,  164, 68,  99,  136,
    250, 30,  25,  44,  17,  233, 214, 25,  85,  167, 105, 50,  167, 167, 4,   33,  63,  191, 87,  91,  38,  34,  146, 223, 40,  80,  7,   209, 91,  232, 45,
    223, 98,  240, 102, 60,  5,   103, 168, 100, 207, 245, 22,  52,  82,  226, 160, 146, 145, 146, 17,  228, 182, 2,   168, 16,  128, 106, 185, 245, 40,  12,
    235, 94,  9,   186, 202, 80,  135, 47,  1,   8,   154, 160, 33,  187, 57,  243, 3,   116, 201, 80,  160, 181, 30,  88,  65,  141, 242, 222, 66,  116, 110,
    63,  176, 130, 240, 153, 54,  157, 221, 235, 108, 43,  96,  34,  163, 41,  220, 135, 189, 137, 134, 56,  146, 86,  196, 207, 80,  61,  67,  63,  187, 215,
    15,  249, 111, 228, 191, 249, 70,  81,  218, 106, 43,  142, 12,  166, 130, 119, 94,  140, 148, 43,  37,  175, 24,  205, 180, 20,  99,  45,  202, 208, 75,
    70,  179, 132, 255, 196, 127, 250, 163, 125, 61,  17,  51,  158, 179, 62,  20,  21,  232, 98,  159, 2,   253, 142, 120, 57,  106, 31,  250, 23,  21,  70,
    201, 213, 149, 64,  43,  90,  253, 81,  187, 167, 211, 25,  171, 179, 168, 218, 80,  251, 23,  160, 247, 244, 182, 26,  24,  130, 177, 0,   198, 105, 237,
    53,  32,  126, 226, 247, 247, 117, 26,  115, 27,  41,  13,  12,  44,  135, 231, 142, 246, 10,  56,  198, 171, 253, 165, 188, 217, 107, 25,  207, 194, 103,
    160, 130, 180, 36,  162, 136, 87,  74,  23,  156, 238, 159, 79,  156, 250, 48,  211, 113, 194, 108, 34,  93,  109, 71,  195, 89,  243, 2,   208, 213, 232,
    34,  232, 167, 175, 89,  134, 123, 133, 33,  137, 234, 53,  45,  87,  83,  210, 64,  26,  154, 162, 188, 76,  183, 155, 1,   62,  0,   100, 203, 44,  85,
    23,  30,  175, 36,  250, 130, 171, 214, 67,  83,  143, 42,  66,  125, 73,  223, 251, 148, 172, 32,  43,  102, 170, 125, 52,  107, 215, 146, 121, 36,  117,
    53,  185, 92,  208, 239, 76,  46,  3,   71,  208, 239, 236, 95,  6,   232, 144, 29,  134, 44,  93,  75,  77,  51,  24,  119, 67,  223, 101, 60,  4,   7,
    241, 90,  141, 57,  25,  87,  84,  41,  90,  39,  136, 231, 226, 205, 108, 231, 117, 181, 78,  192, 251, 220, 230, 50,  72,  189, 208, 27,  67,  188, 68,
    26,  91,  128, 92,  181, 218, 131, 182, 121, 68,  191, 41,  34,  59,  239, 2,   187, 27,  112, 179, 223, 12,  219, 81,  135, 34,  107, 223, 108, 207, 39,
    218, 173, 247, 228, 195, 97,  40,  59,  11,  114, 147, 17,  181, 226, 76,  147, 156, 36,  27,  131, 202, 24,  2,   167, 16,  237, 186, 68,  62,  113, 24,
    240, 21,  190, 186, 188, 211, 54,  231, 18,  7,   42,  46,  84,  255, 70,  129, 231, 241, 214, 26,  178, 116, 113, 72,  17,  125, 175, 255, 182, 5,   202,
    71,  208, 91,  194, 221, 33,  43,  150, 77,  82,  78,  28,  2,   10,  245, 156, 190, 69,  145, 29,  31,  132, 255, 64,  33,  239, 251, 182, 29,  206, 233,
    3,   232, 146, 132, 113, 10,  122, 95,  27,  92,  0,   223, 227, 12,  69,  174, 25,  92,  64,  89,  36,  191, 211, 28,  98,  129, 8,   155, 241, 28,  239,
    66,  40,  40,  242, 8,   45,  149, 240, 222, 222, 87,  137, 101, 187, 25,  170, 47,  239, 59,  19,  124, 199, 105, 181, 216, 114, 18,  178, 115, 118, 255,
    34,  244, 98,  109, 45,  146, 143, 223, 122, 19,  156, 83,  240, 121, 114, 85,  136, 37,  39,  144, 147, 65,  133, 1,   179, 10,  195, 63,  233, 184, 30,
    223, 202, 77,  185, 205, 228, 251, 145, 25,  205, 173, 40,  29,  241, 51,  81,  253, 35,  179, 146, 58,  12,  216, 136, 141, 173, 70,  88,  203, 233, 7,
    229, 61,  203, 35,  153, 243, 224, 189, 165, 111, 49,  33,  149, 91,  238, 129, 91,  24,  29,  165, 188, 194, 52,  133, 236, 205, 97,  173, 201, 142, 199,
    67,  153, 73,  4,   240, 42,  135, 245, 203, 145, 122, 100, 189, 8,   189, 48,  184, 25,  49,  252, 196, 79,  241, 16,  30,  106, 15,  26,  155, 250, 130,
    70,  32,  104, 229, 216, 253, 229, 97,  116, 255, 148, 46,  175, 93,  238, 151, 152, 50,  122, 196, 94,  205, 183, 80,  38,  207, 144, 103, 214, 7,   173,
    24,  255, 247, 185, 226, 144, 143, 94,  120, 37,  28,  126, 206, 83,  236, 68,  18,  52,  68,  67,  151, 7,   184, 30,  12,  204, 41,  55,  235, 112, 223,
    242, 126, 201, 97,  0,   94,  69,  33,  230, 98,  42,  4,   73,  210, 75,  167, 138, 229, 234, 191, 159, 42,  166, 105, 244, 127, 83,  197, 66,  241, 171,
    144, 47,  49,  141, 46,  62,  205, 198, 140, 205, 113, 219, 142, 7,   235, 50,  118, 189, 122, 249, 241, 160, 160, 253, 223, 19,  187, 132, 122, 57,  169,
    10,  218, 47,  250, 255, 65,  40,  219, 54,  195, 224, 99,  75,  162, 17,  10,  89,  118, 196, 204, 196, 224, 194, 17,  116, 43,  72,  234, 235, 125, 132,
    35,  164, 106, 215, 251, 68,  28,  128, 62,  205, 176, 132, 49,  4,   173, 43,  174, 224, 219, 205, 108, 246, 53,  4,   213, 197, 222, 197, 105, 180, 190,
    122, 245, 74,  58,  140, 235, 206, 133, 126, 198, 191, 47,  145, 200, 57,  51,  62,  44,  229, 12,  70,  148, 45,  241, 117, 153, 85,  189, 4,   201, 179,
    137, 182, 42,  126, 182, 224, 68,  146, 111, 139, 156, 129, 223, 241, 79,  247, 255, 186, 204, 183, 40,  170, 108, 20,  12,  202, 136, 222, 53,  72,  37,
    142, 224, 169, 26,  100, 226, 188, 160, 242, 130, 105, 134, 229, 90,  237, 82,  86,  243, 103, 139, 230, 212, 31,  166, 254, 0,   149, 223, 203, 249, 18,
    100, 45,  93,  216, 255, 100, 11,  245, 37,  12,  142, 134, 90,  104, 209, 39,  69,  159, 76,  51,  252, 149, 86,  203, 255, 158, 205, 203, 77,  221, 201,
    117, 50,  125, 163, 85,  82,  248, 250, 251, 66,  9,   223, 187, 75,  74,  25,  170, 231, 104, 222, 169, 243, 78,  197, 92,  76,  226, 117, 82,  119, 61,
    81,  134, 11,  30,  88,  19,  131, 199, 53,  80,  141, 69,  218, 88,  140, 146, 189, 100, 111, 107, 90,  14,  155, 217, 206, 91,  151, 109, 10,  18,  69,
    67,  109, 150, 224, 121, 211, 45,  224, 8,   71,  255, 250, 91,  183, 64,  203, 207, 131, 222, 154, 150, 67,  196, 14,  109, 141, 241, 6,   56,  43,  141,
    196, 247, 110, 93,  26,  94,  76,  244, 123, 134, 91,  232, 12,  122, 96,  45,  229, 123, 86,  235, 113, 57,  216, 86,  180, 46,  203, 44,  71,  101, 67,
    232, 47,  39,  71,  126, 24,  17,  37,  227, 170, 254, 114, 241, 173, 159, 119, 136, 4,   235, 176, 174, 53,  45,  135, 78,  123, 51,  27,  68,  186, 67,
    60,  148, 179, 111, 7,   56,  130, 54,  63,  174, 125, 216, 240, 165, 209, 67,  15,  175, 241, 17,  50,  64,  6,   90,  211, 114, 232, 49,  227, 70,  168,
    46,  219, 252, 80,  208, 152, 104, 110, 39,  87,  239, 25,  108, 163, 28,  97,  128, 35,  90,  68,  139, 48,  23,  147, 249, 134, 162, 97,  77,  25,  245,
    220, 211, 182, 132, 122, 224, 173, 33,  213, 218, 120, 137, 40,  68,  210, 113, 96,  114, 136, 109, 54,  205, 72,  93,  150, 119, 161, 170, 122, 162, 62,
    64,  58,  119, 232, 109, 100, 26,  25,  28,  201, 139, 228, 69,  171, 54,  240, 79,  199, 162, 134, 21,  174, 203, 234, 131, 162, 200, 13,  199, 22,  192,
    16,  247, 173, 39,  229, 13,  107, 202, 211, 136, 188, 76,  94,  70,  110, 83,  184, 170, 101, 81,  31,  242, 150, 161, 250, 102, 221, 75,  171, 87,  235,
    101, 208, 111, 247, 8,   162, 116, 179, 116, 51,  243, 129, 87,  135, 232, 214, 24,  15,  101, 182, 185, 90,  40,  188, 125, 33,  22,  62,  62,  17,  119,
    16,  248, 68,  202, 24,  152, 28,  140, 120, 143, 29,  139, 4,   205, 232, 236, 233, 249, 21,  103, 240, 228, 29,  30,  33,  67,  183, 62,  62,  34,  120,
    243, 60,  35,  100, 251, 173, 175, 143, 128, 214, 211, 122, 198, 192, 229, 48,  253, 89,  76,  216, 228, 106, 200, 48,  137, 201, 141, 242, 147, 41,  248,
    119, 12,  166, 36,  142, 224, 174, 26,  76,  18,  62,  153, 51,  6,   38,  135, 110, 187, 51,  148, 37,  150, 95,  240, 72,  246, 94,  95,  86,  134, 35,
    85,  190, 54,  120, 110, 97,  40,  80,  85,  211, 38,  229, 205, 203, 155, 199, 24,  88,  117, 48,  153, 170, 181, 174, 168, 29,  153, 221, 135, 60,  62,
    203, 121, 221, 137, 33,  24,  187, 35,  135, 208, 233, 190, 77,  128, 190, 251, 239, 37,  168, 255, 8,   249, 44,  141, 113, 112, 65,  196, 141, 232, 185,
    241, 160, 217, 51,  91,  36,  248, 93,  177, 75,  151, 7,   186, 8,   155, 204, 59,  161, 23,  174, 93,  73,  245, 138, 246, 46,  202, 231, 168, 118, 126,
    161, 133, 28,  70,  121, 102, 233, 234, 192, 184, 164, 207, 235, 221, 143, 105, 68,  7,   123, 247, 83,  126, 211, 219, 100, 132, 55,  218, 200, 105, 124,
    239, 22,  82,  227, 162, 111, 67,  95,  41,  113, 55,  79,  76,  202, 229, 13,  163, 163, 2,   191, 143, 98,  157, 183, 199, 138, 153, 89,  6,   169, 244,
    189, 61,  92,  244, 198, 241, 152, 64,  14,  167, 124, 16,  241, 104, 29,  85,  170, 62,  123, 88,  230, 119, 131, 23,  219, 113, 27,  115, 154, 97,  106,
    212, 163, 239, 200, 250, 41,  99,  111, 73,  78,  181, 135, 14,  253, 98,  35,  100, 245, 116, 26,  185, 79,  1,   162, 94,  243, 43,  156, 122, 255, 228,
    90,  145, 168, 239, 254, 10,  243, 107, 38,  215, 130, 87,  152, 38,  163, 91,  107, 181, 34,  102, 174, 205, 93,  118, 85,  99,  87,  30,  245, 36,  182,
    174, 32,  213, 79,  238, 82,  136, 167, 105, 107, 25,  197, 22,  212, 174, 178, 186, 186, 66,  38,  230, 48,  203, 233, 232, 92,  140, 250, 211, 135, 169,
    211, 179, 192, 53,  201, 34,  25,  54,  172, 185, 196, 193, 147, 216, 62,  143, 120, 74,  59,  202, 78,  152, 6,   253, 86,  120, 83,  200, 212, 154, 204,
    80,  8,   179, 154, 148, 102, 32,  40,  77,  236, 139, 146, 10,  12,  123, 29,  6,   16,  55,  113, 55,  118, 219, 221, 54,  114, 251, 32,  226, 84,  220,
    213, 141, 157, 253, 239, 19,  135, 1,   92,  81,  237, 252, 130, 91,  91,  201, 182, 197, 88,  86,  77,  202, 46,  170, 62,  42,  65,  126, 49,  32,  138,
    68,  125, 72,  66,  217, 26,  191, 8,   62,  21,  175, 183, 237, 174, 206, 144, 29,  216, 148, 195, 135, 2,   84,  96,  242, 30,  148, 223, 155, 58,  44,
    58,  62,  240, 123, 24,  191, 97,  109, 178, 167, 125, 208, 106, 195, 230, 230, 186, 89,  20,  201, 77,  16,  129, 231, 39,  68,  148, 170, 99,  42,  160,
    166, 68,  16,  245, 243, 191, 161, 156, 225, 82,  51,  100, 119, 89,  1,   205, 111, 247, 100, 157, 237, 81,  68,  240, 222, 191, 241, 90,  241, 188, 132,
    134, 99,  199, 80,  12,  62,  211, 126, 135, 125, 101, 53,  199, 7,   85,  46,  193, 216, 251, 24,  196, 177, 249, 82,  201, 243, 71,  223, 71,  9,   38,
    90,  41,  87,  117, 155, 201, 46,  205, 99,  166, 214, 188, 149, 105, 59,  148, 193, 7,   133, 87,  24,  146, 208, 79,  147, 242, 164, 70,  55,  229, 72,
    80,  57,  114, 109, 113, 109, 153, 185, 241, 230, 23,  190, 218, 101, 54,  203, 210, 19,  125, 72,  29,  215, 228, 25,  194, 69,  235, 149, 50,  141, 53,
    60,  56,  122, 8,   147, 112, 83,  212, 52,  226, 65,  231, 225, 12,  198, 118, 156, 94,  101, 201, 18,  140, 12,  47,  23,  129, 87,  186, 39,  80,  192,
    226, 0,   250, 17,  253, 168, 221, 52,  98,  65,  59,  106, 102, 140, 195, 135, 83,  31,  0,   141, 108, 182, 71,  105, 25,  149, 207, 76,  60,  5,   10,
    142, 206, 230, 168, 84,  42,  149, 154, 172, 40,  187, 126, 96,  93,  51,  158, 125, 99,  85,  240, 27,  180, 4,   241, 235, 55,  63,  22,  156, 83,  128,
    214, 209, 58,  191, 25,  50,  167, 63,  157, 212, 218, 102, 189, 91,  156, 250, 96,  234, 19,  128, 79,  65,  121, 171, 255, 181, 92,  214, 154, 126, 119,
    190, 156, 70,  3,   12,  237, 110, 160, 127, 92,  109, 134, 172, 128, 29,  137, 103, 216, 147, 248, 52,  152, 2,   220, 108, 126, 208, 160, 151, 203, 245,
    248, 154, 16,  142, 72,  199, 226, 59,  136, 44,  61,  132, 242, 198, 72,  36,  100, 244, 101, 140, 34,  103, 248, 99,  58,  14,  26,  26,  68,  249, 252,
    196, 85,  160, 14,  98,  62,  84,  255, 160, 47,  172, 181, 7,   237, 120, 238, 10,  3,   76,  188, 47,  123, 224, 175, 167, 223, 222, 187, 211, 31,  214,
    123, 54,  231, 232, 23,  164, 24,  223, 49,  198, 198, 161, 63,  56,  60,  125, 71,  250, 255, 236, 221, 5,   144, 83,  121, 182, 7,   224, 147, 220, 36,
    29,  220, 198, 245, 102, 92,  158, 52,  140, 148, 227, 54,  222, 227, 46,  1,   130, 212, 100, 104, 104, 25,  43,  121, 111, 157, 29,  119, 119, 119, 119,
    215, 146, 117, 119, 183, 146, 53,  156, 117, 246, 134, 48,  218, 182, 190, 76,  239, 119, 98,  127, 236, 43,  234, 32,  109, 191, 123, 206, 154, 24,  183,
    75,  35,  255, 112, 65,  118, 251, 203, 170, 16,  209, 252, 194, 238, 123, 159, 198, 188, 16,  189, 20,  129, 64,  104, 126, 213, 190, 207, 183, 221, 205,
    228, 64,  75,  35,  73,  50,  236, 192, 53,  133, 120, 227, 246, 47,  231, 99,  84,  203, 218, 236, 116, 247, 154, 113, 27,  254, 175, 214, 73,  194, 63,
    86,  152, 93,  61,  171, 186, 40,  123, 155, 49,  240, 71,  7,   185, 226, 187, 243, 15,  185, 9,   39,  53,  242, 15,  35,  98,  196, 95,  144, 127, 24,
    30,  73,  95,  249, 135, 173, 95,  205, 111, 126, 244, 233, 221, 245, 90,  189, 246, 23,  100, 48,  34,  251, 61,  228, 199, 126, 247, 142, 183, 50,  24,
    243, 46,  223, 37,  223, 72,  47,  12,  142, 63,  139, 153, 89,  38,  103, 78,  117, 238, 233, 125, 230, 39,  34,  70,  198, 199, 35,  55,  50,  230, 108,
    251, 234, 79,  99,  231, 242, 39,  239, 142, 92,  121, 249, 186, 152, 178, 229, 15,  214, 54,  147, 69,  31,  192, 62,  16,  8,   4,   2,   129, 64,  32,
    204, 62,  103, 73,  173, 35,  109, 164, 32,  6,   248, 60,  109, 169, 153, 127, 56,  239, 150, 82,  110, 227, 105, 114, 41,  215, 200, 129, 108, 154, 125,
    32,  16,  8,   19,  246, 111, 109, 77,  143, 204, 50,  53,  3,   34,  185, 242, 242, 223, 71,  220, 180, 252, 247, 201, 59,  194, 200, 189, 162, 164, 147,
    4,   2,   129, 64,  32,  16,  8,   4,   66,  179, 8,   51,  186, 179, 164, 94,  45,  139, 171, 183, 205, 239, 49,  235, 228, 61,  217, 164, 230, 236, 133,
    49,  223, 125, 107, 246, 194, 184, 83,  46,  189, 35,  23,  63,  202, 110, 58,  73,  32,  12,  82,  97,  255, 214, 221, 59,  155, 115, 15,  250, 188, 68,
    108, 120, 36,  111, 205, 127, 136, 198, 252, 135, 168, 46,  203, 78,  141, 249, 15,  145, 187, 195, 231, 31,  6,   151, 112, 232, 57,  245, 198, 184, 129,
    174, 255, 105, 31,  120, 254, 195, 154, 183, 243, 15,  95,  205, 178, 7,   167, 92,  62,  47,  31,  165, 127, 220, 223, 7,   66,  115, 254, 83,  51,  252,
    16,  189, 215, 200, 31,  52,  242, 15,  165, 157, 222, 201, 63,  228, 150, 175, 43,  229, 178, 252, 67,  34,  255, 64,  32,  16,  8,   4,   2,   129, 64,
    32,  16,  8,   132, 193, 44,  188, 123, 254, 67,  54,  196, 161, 94,  157, 91,  91,  216, 94,  159, 151, 77,  229, 232, 49,  255, 97,  217, 219, 243, 31,
    150, 189, 123, 254, 131, 78,  18,  8,   4,   2,   161, 89,  4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,
    16,  8,   4,   2,   129, 208, 220, 182, 188, 168, 90,  79,  179, 84,  74,  143, 235, 47,  123, 236, 191, 216, 57,  219, 122, 81,  220, 112, 170, 100, 167,
    92,  242, 173, 228, 91,  155, 126, 31,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  14,  172, 158, 123, 110, 218, 216, 5,   251,
    222, 207, 127, 244, 216, 9,   91,  46,  180, 22,  95,  253, 89,  164, 87,  95,  150, 70,  107, 241, 214, 159, 69,  229, 43,  31,  59,  45,  162, 226, 122,
    28,  2,   97,  211, 221, 225, 85,  61,  99,  78,  45,  219, 174, 223, 213, 145, 237, 25,  236, 236, 125, 63,  95,  49,  146, 194, 240, 202, 136, 37,  155,
    23,  90,  47,  169, 142, 47,  86,  206, 235, 42,  22,  38,  92,  251, 141, 157, 243, 229, 175, 148, 191, 162, 147, 132, 193, 39,  204, 110, 239, 88,  156,
    238, 157, 30,  150, 45,  184, 171, 69,  191, 219, 183, 43,  229, 142, 189, 163, 82,  153, 178, 127, 84,  146, 106, 49,  42,  187, 45,  223, 45,  114, 171,
    115, 171, 117, 178, 167, 64,  32,  204, 172, 119, 119, 213, 210, 25,  103, 47,  249, 159, 126, 133, 164, 60,  178, 90,  172, 109, 19,  149, 147, 78,  91,
    22,  105, 118, 202, 159, 118, 208, 221, 247, 36,  113, 173, 9,   117, 4,   2,   129, 64,  32,  16,  8,   253, 22,  129, 64,  32,  16,  8,   4,   2,   129,
    64,  32,  16,  8,   4,   130, 43,  188, 234, 237, 105, 103, 109, 110, 87,  63,  95,  142, 42,  70,  146, 143, 157, 150, 117, 108, 153, 91,  50,  234, 145,
    205, 242, 217, 169, 165, 184, 116, 220, 35,  187, 20,  227, 171, 49,  66,  39,  9,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,
    4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  252, 123, 133, 99,  106, 245, 246, 247, 14,  127, 232, 227, 58,  247, 161, 149, 243, 174, 104,
    76,  125, 248, 205, 206, 249, 165, 231, 213, 54,  47,  78,  184, 254, 249, 125, 10,  67,  159, 24,  250, 132, 78,  18,  8,   4,   2,   129, 64,  32,  16,
    8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  248, 247, 10,  83,  59,  22,  45,  88,  216, 213, 72,  64,
    108, 92,  111, 147, 182, 117, 180, 239, 217, 51,  255, 48,  114, 100, 229, 188, 163, 178, 252, 195, 21,  89,  254, 161, 122, 94,  119, 177, 56,  225, 242,
    79,  143, 47,  148, 211, 114, 170, 147, 4,   2,   129, 64,  32,  16,  122, 22,  129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   225,
    168, 142, 218, 25,  255, 147, 110, 216, 12,  219, 223, 175, 79,  202, 229, 74,  212, 182, 137, 91,  27,  59,  97,  43,  195, 247, 219, 120, 218, 98,  39,
    59,  97,  9,   132, 193, 43,  204, 236, 168, 117, 213, 107, 157, 157, 233, 212, 106, 246, 180, 199, 220, 246, 37,  231, 164, 243, 219, 59,  210, 121, 181,
    57,  221, 11,  246, 140, 102, 37,  145, 148, 202, 99,  39,  127, 232, 169, 24,  127, 221, 151, 71,  199, 184, 131, 171, 79,  70,  229, 250, 41,  147, 35,
    46,  200, 110, 58,  249, 119, 11,  4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129,
    64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   225, 144, 89,  233, 209, 139, 23,  117, 182, 47,  142, 222, 107, 253, 234, 136, 104, 137,
    36,  162, 181, 244, 202, 194, 36,  125, 249, 213, 235, 147, 214, 210, 138, 114, 146, 222, 244, 161, 251, 147, 113, 231, 186, 246, 97,  176, 11,  4,   2,
    129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  204, 236, 174, 167, 135, 117, 44,  168, 46,  78,  247, 219, 56,  192, 122, 191, 214, 61,  227, 125, 213,
    18,  73,  161, 240, 232, 232, 243, 202, 177, 100, 250, 125, 91,  21,  31,  107, 156, 150, 30,  148, 157, 226, 213, 236, 246, 1,   232, 3,   129, 64,  32,
    16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  68,  108, 74,  2,   161, 173, 187, 222, 89,  155,
    86,  63,  115, 64,  99,  200, 152, 242, 138, 117, 165, 216, 124, 249, 241, 249, 49,  159, 93,  179, 174, 208, 50,  41,  191, 50,  31,  83,  179, 155, 78,
    18,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  244, 81,  4,
    2,   129, 112, 100, 87,  125, 183, 169, 237, 243, 206, 25,  64,  72,  202, 49,  166, 101, 229, 143, 115, 19,  46,  95,  241, 227, 194, 152, 97,  43,  127,
    92,  220, 231, 166, 21,  63,  205, 109, 119, 105, 225, 23,  155, 90,  31,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,
    2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  152, 221, 94,  175, 183, 159, 149,
    238, 113, 220, 172, 61,  251, 19,  146, 136, 74,  225, 214, 150, 216, 105, 207, 229, 145, 175, 12,  201, 78,  149, 189, 50,  224, 134, 201, 113, 129, 78,
    254, 109, 2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,
    129, 64,  32,  16,  8,   4,   2,   97,  70,  189, 54,  55,  109, 171, 117, 204, 237, 35,  206, 186, 254, 11,  235, 87,  71,  68,  18,  195, 182, 138, 117,
    23,  151, 70,  23,  58,  14,  43,  196, 164, 229, 235, 99,  116, 113, 245, 218, 136, 181, 81,  209, 73,  2,   129, 64,  32,  52,  139, 64,  32,  16,  8,
    4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 208, 214, 93,  239, 172, 165, 71,  86,  251, 29,  34,  145, 68,  82,  44,  70,  249, 150, 223, 36,  185,
    161, 183, 252, 166, 229, 157, 211, 13,  147, 227, 228, 193, 221, 73,  2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   211, 22,  214, 106,
    231, 214, 210, 195, 58,  22,  68,  239, 53,  60,  198, 197, 208, 72,  10,  133, 71,  71,  159, 87,  142, 37,  211, 239, 219, 42,  30,  107, 156, 150, 150,
    238, 219, 42,  185, 97,  114, 180, 109, 250, 125, 32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,
    32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   225, 200, 108, 91,  243, 49,  245, 197, 3,   9,   201, 208, 114, 181, 56,  101, 215, 66,  122,
    101, 219, 46,  249, 41,  45,  83,  118, 141, 202, 181, 109, 187, 228, 178, 188, 220, 181, 255, 201, 157, 36,  16,  8,   4,   2,   129, 64,  32,  16,  8,
    4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 48,  179, 218, 217, 117, 68,  173, 115, 106, 244, 85,  195, 179, 91,
    86,  67,  99,  124, 172, 90,  27,  173, 155, 175, 94,  19,  89,  45,  111, 62,  13,  219, 33,  30,  28,  172, 157, 36,  16,  8,   4,   2,   129, 64,  32,
    16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  8,   4,
    2,   129, 112, 196, 226, 121, 179, 234, 237, 115, 79,  143, 62,  106, 253, 234, 245, 171, 35,  31,  201, 240, 24,  91,  122, 125, 101, 76,  186, 117, 213,
    157, 113, 89,  203, 170, 236, 244, 213, 213, 119, 198, 184, 174, 17,  179, 117, 146, 64,  32,  16,  8,   4,   2,   129, 64,  248, 75,  138, 64,  32,  16,
    8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 112, 100, 245, 172, 9,   49,  80,  229, 35,  137, 104, 45,  189, 178, 48,  73,  95,  126, 229, 250,
    164, 181, 180, 162, 156, 164, 55,  253, 255, 253, 201, 13,  147, 227, 24,  157, 252, 247, 10,  4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,
    16,  8,   4,   2,   129, 64,  32,  16,  8,   4,   2,   129, 64,  32,  16,  102, 215, 170, 103, 158, 115, 228, 57,  139, 251, 255, 245, 205, 252, 195, 155,
    141, 252, 195, 155, 231, 103, 249, 135, 53,  123, 100, 249, 135, 143, 110, 200, 63,  236, 17,  161, 147, 4,   2,   129, 64,  32,  16,  8,   4,   2,   97,
    204, 11,  31,  108, 129, 64,  32,  236, 244, 231, 118, 206, 2,   184, 141, 36,  11,  195, 127, 119, 15,  180, 228, 177, 100, 14,  39,  227, 186, 152, 150,
    153, 193, 177, 194, 12,  218, 208, 161, 98,  105, 45,  229, 20,  43,  145, 241, 56,  76,  199, 204, 204, 204, 204, 204, 204, 204, 84,  180, 130, 72,  178,
    178, 161, 158, 81,  96,  156, 213, 68,  83,  21,  58,  232, 247, 170, 166, 94,  98,  251, 211, 63,  221, 175, 7,   255, 18,  12,  0,   197, 153, 160, 141,
    19,  155, 156, 152, 125, 94,  239, 253, 103, 34,  0,   162, 124, 114, 179, 166, 225, 229, 173, 212, 65,  248, 50,  110, 240, 70,  232, 98,  26,  111, 244,
    47,  108, 195, 46,  60,  57,  72,  32,  54,  148, 136, 13,  104, 15,  158, 237, 141, 208, 126, 209, 132, 139, 31,  201, 32,  2,   0,   142, 30,  213, 128,
    227, 199, 169, 3,   179, 195, 43,  97,  46,  56,  116, 139, 192, 144, 205, 42,  14,  194, 86,  120, 137, 86,  76,  159, 203, 156, 132, 32,  138, 69,  170,
    119, 101, 50,  32,  127, 194, 251, 47,  201, 56,  200, 117, 209, 125, 209, 235, 162, 79,  228, 69,  173, 11,  252, 69,  228, 69,  173, 11,  124, 89,  100,
    141, 56,  249, 175, 147, 71,  196, 56,  176, 122, 195, 135, 162, 14,  160, 196, 161, 243, 189, 58,  84,  209, 91,  32,  43,  240, 236, 154, 132, 18,  172,
    105, 96,  129, 128, 15,  239, 126, 55,  235, 152, 243, 203, 95,  32,  104, 76,  76,  32,  136, 55,  101, 128, 55,  97,  53,  60,  69,  43,  83,  12,  96,
    98,  17,  128, 163, 155, 84,  116, 10,  13,  84,  104, 96,  100, 161, 96,  92,  212, 186, 192, 19,  69,  122, 138, 105, 46,  26,  48,  29,  211, 23,  36,
    99,  253, 195, 233, 68,  191, 217, 23,  25,  26,  50,  231, 167, 71,  182, 97,  114, 24,  21,  130, 102, 0,   101, 110, 17,  22,  195, 65,  104, 65,  203,
    186, 100, 34,  26,  51,  215, 13,  39,  220, 190, 72,  211, 16,  137, 251, 197, 72,  210, 131, 19,  227, 26,  221, 89,  58,  64,  69,  53,  1,   138, 98,
    137, 226, 6,   145, 158, 102, 179, 211, 101, 54,  177, 67,  36,  106, 50,  78,  0,   29,  12,  254, 175, 243, 151, 112, 246, 229, 157, 163, 97,  84,  42,
    140, 133, 129, 251, 68,  214, 10,  14,  98,  105, 64,  128, 162, 192, 53,  52,  31,  124, 68,  3,   132, 6,   130, 31,  103, 129, 113, 145, 181, 195, 240,
    35,  200, 136, 161, 240, 111, 232, 176, 55,  62,  253, 189, 27,  64,  89,  158, 80,  139, 16,  74,  166, 134, 98,  81,  115, 113, 98,  32,  126, 253, 226,
    200, 112, 245, 99,  131, 47,  202, 52,  94,  207, 203, 156, 213, 41,  249, 141, 240, 89,  21,  87,  242, 15,  81,  180, 139, 244, 164, 161, 217, 161, 65,
    171, 104, 104, 178, 53,  136, 145, 92,  29,  139, 70,  146, 182, 4,   179, 170, 132, 147, 143, 194, 0,   110, 130, 31,  42,  59,  212, 205, 116, 229, 167,
    97,  10,  32,  3,   123, 227, 101, 54,  1,   24,  45,  8,   50,  24,  109, 248, 166, 14,  24,  15,  145, 211, 26,  96,  105, 16,  93,  189, 106, 123, 108,
    208, 93,  130, 173, 1,   216, 127, 19,  243, 243, 14,  254, 245, 2,   235, 82,  254, 154, 163, 86,  69,  59,  149, 124, 142, 90,  61,  233, 77,  131, 193,
    96,  76,  61,  171, 65,  213, 63,  187, 200, 30,  7,   96,  171, 72,  79,  132, 58,  65,  152, 225, 216, 139, 211, 4,   106, 17,  66,  233, 200, 80,  220,
    12,  61,  109, 219, 22,  49,  160, 55,  187, 172, 11,  237, 126, 198, 57,  193, 145, 91,  25,  71,  246, 197, 148, 160, 112, 186, 162, 119, 208, 59,  188,
    105, 224, 12,  198, 156, 106, 26,  162, 136, 174, 21,  75,  243, 66,  18,  96,  160, 14,  199, 31,  207, 0,   245, 240, 9,   85,  51,  244, 2,   152, 168,
    20,  141, 239, 47,  52,  48,  210, 72,  26,  67,  241, 196, 224, 80,  108, 232, 52,  196, 133, 160, 174, 98,  156, 131, 255, 242, 54,  198, 241, 231, 61,
    20,  252, 231, 183, 49,  159, 85,  145, 217, 100, 182, 173, 161, 47,  150, 76,  194, 53,  12,  96,  33,  56,  166, 34,  159, 103, 179, 149, 236, 159, 29,
    29,  69,  122, 73,  111, 56,  178, 109, 75,  106, 36,  157, 24,  140, 193, 253, 8,   195, 87,  51,  45,  64,  248, 43,  6,   24,  59,  48,  58,  70,  237,
    42,  116, 224, 27,  99,  20,  47,  193, 75,  214, 109, 79,  158, 155, 140, 234, 12,  2,   53,  196, 252, 126, 197, 247, 245, 126, 170, 41,  185, 53,  148,
    250, 62,  190, 142, 29,  80,  115, 243, 40,  126, 142, 159, 135, 82,  99,  91,  106, 236, 131, 1,   220, 3,   3,   4,   165, 87,  67,  69,  249, 97,  230,
    88,  23,  235, 177, 126, 82,  63,  220, 226, 214, 15,  15,  84,  250, 225, 182, 243, 251, 129, 124, 152, 124, 120, 125, 98,  139, 96,  36,  35,  219, 47,
    168, 97,  14,  243, 215, 77,  87,  63,  182, 15,  216, 245, 166, 155, 149, 169, 230, 199, 246, 145, 94,  20,  55,  41,  228, 61,  228, 61,  206, 126, 184,
    197, 173, 31,  78,  88,  253, 160, 84,  235, 7,   133, 40,  246, 178, 236, 75,  13,  14,  164, 204, 238, 149, 177, 177, 158, 199, 16,  52,  0,   232, 97,
    20,  33,  254, 138, 49,  252, 117, 118, 249, 93,  116, 9,   94,  253, 43,  252, 235, 15,  165, 183, 3,   251, 176, 111, 121, 106, 204, 5,   224, 60,  227,
    60,  142, 5,   140, 16,  206, 16,  150, 114, 65,  248, 183, 77,  248, 1,   126, 176, 98,  100, 56,  102, 198, 45,  33,  253, 130, 19,  169, 198, 241, 3,
    179, 152, 97,  248, 197, 39,  171, 255, 230, 165, 119, 161, 66,  232, 17,  4,   235, 24,  117, 238, 8,   19,  170, 78,  224, 149, 189, 8,   24,  125, 150,
    134, 191, 216, 26,  78,  179, 222, 110, 95,  129, 136, 189, 112, 249, 91,  39,  161, 195, 141, 32,  246, 194, 254, 248, 112, 194, 154, 139, 152, 203, 108,
    250, 129, 59,  152, 33,  182, 165, 41,  160, 187, 30,  93,  175, 144, 250, 210, 20,  66,  112, 116, 189, 138, 213, 88,  45,  52,  56,  1,   46,  132, 155,
    30,  75,  16,  85,  133, 96,  107, 152, 55,  144, 26,  72,  193, 141, 160, 1,   43,  172, 158, 228, 165, 184, 166, 249, 78,  12,  48,  82,  95,  126, 21,
    26,  112, 124, 128, 225, 118, 220, 46,  52,  184, 0,   28,  29,  133, 197, 21,  194, 184, 166, 158, 38,  140, 159, 35,  132, 34,  91,  34,  67,  17,  212,
    88,  89,  27,  17,  132, 127, 231, 225, 107, 153, 129, 241, 39,  57,  207, 56,  125, 232, 91,  17,  73,  71,  250, 35,  67,  53,  8,   221, 8,   224, 69,
    95,  218, 119, 140, 5,   249, 83,  94,  5,   7,   225, 119, 248, 221, 186, 120, 42,  61,  108, 110, 136, 39,  134, 134, 147, 49,  87,  66,  39,  128, 131,
    120, 231, 179, 168, 217, 145, 241, 57,  9,   81,  68,  151, 139, 94,  112, 1,   56,  8,   115, 225, 23,  132, 183, 94,  47,  8,   95,  10,  58,  8,   244,
    249, 244, 249, 240, 20,  247, 49,  35,  224, 124, 18,  118, 216, 175, 152, 126, 81,  120, 59,  251, 159, 60,  98,  17,  2,   212, 241, 119, 88,  21,  212,
    77,  113, 93,  173, 91,  132, 80,  50,  50,  26,  27,  170, 53,  23,  75,  192, 209, 132, 194, 56,  90,  231, 79,  252, 24,  147, 143, 114, 118, 71,  109,
    72,  165, 162, 102, 159, 203, 87,  171, 91,  87,  98,  243, 25,  208, 130, 226, 171, 208, 252, 250, 220, 143, 105, 75,  251, 153, 202, 90,  155, 162, 163,
    92,  1,   142, 235, 73,  104, 21,  13,  205, 66,  3,   113, 104, 24,  199, 184, 125, 124, 8,   141, 36,  250, 221, 186, 170, 94,  36,  154, 24,  71,  16,
    7,   238, 6,   244, 77,  175, 18,  213, 107, 95,  7,   125, 86,  62,  127, 238, 10,  196, 3,   193, 128, 138, 129, 123, 20,  180, 47,  123, 157, 168, 198,
    95,  69,  245, 153, 229, 18,  240, 106, 188, 218, 214, 16,  78,  39,  34,  131, 3,   23,  232, 168, 205, 224, 248, 217, 206, 35,  79,  199, 55,  86,  229,
    158, 226, 220, 139, 29,  216, 97,  105, 240, 70,  120, 217, 206, 220, 39,  177, 127, 50,  129, 188, 141, 188, 173, 123, 209, 58,  179, 211, 220, 184, 168,
    199, 92,  23,  143, 60,  53,  150, 174, 74,  160, 189, 44,  16,  240, 243, 245, 91,  153, 122, 224, 173, 93,  212, 199, 199, 19,  108, 193, 129, 175, 46,
    162, 164, 131, 116, 156, 35,  44,  77,  8,   25,  149, 243, 248, 208, 249, 132, 213, 76,  211, 8,   31,  125, 152, 105, 7,   194, 35,  138, 93,  45,  60,
    240, 214, 17,  106, 141, 3,   106, 7,   3,   122, 25,  33,  62,  223, 199, 254, 166, 213, 53,  237, 253, 155, 166, 55,  71,  69,  213, 62,  255, 111, 26,
    125, 18,  125, 18,  60,  69,  8,   1,   40,  40,  111, 131, 175, 241, 173, 119, 194, 177, 66,  60,  93,  13,  158, 44,  3,   183, 48,  14,  210, 251, 173,
    18,  165, 120, 119, 89,  107, 94,  252, 205, 146, 66,  230, 148, 203, 138, 213, 147, 94,  159, 129, 88,  145, 207, 19,  179, 238, 155, 159, 84,  60,  221,
    53,  203, 39,  24,  146, 32,  9,   146, 224, 53,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  250,
    31,  188, 187, 23,  218, 69,  94,  156, 47,  72,  131, 118, 145, 190, 160, 122, 145, 87,  206, 255, 192, 29,  254, 135, 166, 189, 58,  128, 98,  214, 238,
    7,   175, 254, 7,   174, 56,  252, 15,  190, 189, 42,  40,  38,  4,   97,  53,  86,  95,  188, 255, 97,  117, 106, 44,  150, 54,  151, 37,  134, 205, 117,
    131, 145, 116, 172, 234, 155, 218, 22,  92,  75,  2,   32,  40,  20,  208, 180, 51,  159, 209, 27,  240, 98,  85,  107, 48,  50,  32,  120, 137, 200, 255,
    46,  255, 3,   190, 161, 66,  225, 98,  163, 87,  252, 15,  101, 159, 106, 61,  175, 190, 144, 255, 193, 249, 222, 95,  227, 42,  47,  235, 76,  87,  242,
    97,  106, 85,  212, 174, 236, 142, 242, 238, 127, 152, 172, 161, 201, 214, 208, 65,  58,  174, 140, 255, 193, 143, 54,  183, 113, 8,   147, 240, 149, 241,
    63,  248, 49,  213, 77,  67,  156, 196, 189, 17,  102, 48,  130, 170, 132, 103, 146, 103, 122, 247, 63,  16,  158, 11,  51,  142, 236, 75,  232, 185, 74,
    143, 233, 49,  111, 26,  230, 184, 237, 197, 32,  25,  244, 224, 127, 48,  64,  239, 99,  1,   191, 237, 127, 240, 29,  46,  52,  208, 74,  165, 23,  160,
    224, 91,  248, 214, 85,  240, 63,  72,  255, 3,   59,  219, 15,  103, 43,  237, 144, 118, 232, 138, 248, 31,  172, 99,  53,  56,  24,  59,  174, 50,  78,
    142, 55,  57,  231, 226, 58,  114, 157, 244, 63,  72,  255, 131, 244, 63,  72,  255, 131, 244, 63,  72,  255, 131, 244, 63,  72,  255, 195, 197, 17,  36,
    65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  36,  65,  18,  164, 255, 65,  250, 31,  228, 247, 63,
    216, 12,  220, 204, 136, 65,  81,  246, 129, 212, 191, 251, 17,  4,   249, 223, 117, 4,   133, 6,   224, 118, 145, 255, 23,  254, 7,   59,  246, 175, 103,
    60,  16,  224, 19,  28,  16,  174, 7,   162, 138, 138, 66,  201, 134, 237, 158, 252, 31,  243, 63,  184, 143, 2,   128, 245, 196, 48,  186, 155, 78,  182,
    193, 48,  50,  107, 208, 101, 252, 252, 78,  24,  200, 204, 195, 235, 122, 49,  253, 178, 248, 31,  164, 255, 65,  250, 31,  204, 238, 240, 218, 235, 239,
    186, 233, 174, 30,  156, 31,  83,  49,  199, 58,  70,  89,  160, 98,  17,  248, 73,  38,  195, 42,  213, 247, 69,  133, 231, 225, 121, 210, 255, 32,  253,
    15,  210, 255, 32,  253, 15,  210, 255, 32,  253, 15,  210, 255, 32,  253, 15,  242, 9,   134, 36,  72,  130, 36,  72,  130, 36,  72,  130, 36,  72,  130,
    36,  120, 39,  156, 2,   147, 202, 4,   251};

void InstrumentBankManager::SetDefaults() {
    auto size_mz = uLongf(defaultBankSize);
    data.resize(size_mz);
    uncompress(&data[0], &size_mz, defaultBank, sizeof(defaultBank));
    fileName.clear();
    location = Location::Memory;
    type = Type::Opal;
}
