#ifndef NES_NESLIB_H
#define NES_NESLIB_H

#include "nes.h"


/*
 (C) 2015 Alex Semenov (Shiru)
 (C) 2016 Lauri Kasanen
  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.
  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:
  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.
*/

// NES hardware-dependent functions by Shiru (shiru@mail.ru)
// Feel free to do anything you want with this code, consider it Public Domain

// Versions history:
//  280215 - fixed palette glitch caused with the active DMC DMA glitch
//  030914 - minor fixes in the vram update system
//  310814 - added vram_flush_update
//  120414 - removed adr argument from vram_write and vram_read,
//           unrle_vram renamed to vram_unrle, with adr argument removed
//  060414 - many fixes and improvements, including sequental VRAM updates
//  previous versions were created since mid-2011, there were many updates


void nes_game(void);


// set bg and spr palettes, data is 32 bytes array
void pal_all(const char *data);

// set bg palette only, data is 16 bytes array
void pal_bg(const char *data);

// set spr palette only, data is 16 bytes array
void pal_spr(const char *data);

// set a palette entry, index is 0..31
inline void pal_col(unsigned char index, unsigned char color);

// reset palette to $0f
void pal_clear(void);

// set virtual bright both for sprites and background, 0 is black, 4 is normal, 8 is white
void pal_bright(unsigned char bright);

// set virtual bright for sprites only
void pal_spr_bright(unsigned char bright);

// set virtual bright for sprites background only
void pal_bg_bright(unsigned char bright);



// wait actual TV frame, 50hz for PAL, 60hz for NTSC
void ppu_wait_nmi(void);

// wait virtual frame, it is always 50hz, frame-to-frame in PAL, frameskip in NTSC
void ppu_wait_frame(void);

// turn off rendering, nmi still enabled when rendering is disabled
void ppu_off(void);

// turn on bg, spr
void ppu_on_all(void);

// turn on bg only
void ppu_on_bg(void);

// turn on spr only
void ppu_on_spr(void);

// set PPU_MASK directly
inline void ppu_mask(unsigned char mask);

// get current video system, 0 for PAL, not 0 for NTSC
unsigned char ppu_system(void);

// Return an 8-bit counter incremented at each vblank
unsigned char nesclock(void);

// get/set the internal ppu ctrl cache var for manual writing
inline unsigned char get_ppu_ctrl_var(void);
inline void set_ppu_ctrl_var(unsigned char var);


// clear OAM buffer, all the sprites are hidden
void oam_clear(void);

// set sprite display mode, 0 for 8x8 sprites, 1 for 8x16 sprites
inline void oam_size(unsigned char size);

// set sprite in OAM buffer, chrnum is tile, attr is attribute, sprid is offset in OAM in bytes
// returns sprid+4, which is offset for a next sprite
inline unsigned char oam_spr(unsigned char x, unsigned char y,
					unsigned char chrnum, unsigned char attr,
					unsigned char sprid);

// set metasprite in OAM buffer
// meta sprite is a const unsigned char array, it contains four bytes per sprite
// in order x offset, y offset, tile, attribute
// x=128 is end of a meta sprite
// returns sprid+4, which is offset for a next sprite
unsigned char oam_meta_spr(unsigned char x, unsigned char y,
					unsigned char sprid, const unsigned char *data);

// hide all remaining sprites from given offset
void oam_hide_rest(unsigned char sprid);



// play a music in FamiTone format
void music_play(unsigned char song);

// stop music
void music_stop(void);

// pause and unpause music
void music_pause(unsigned char pause);

// play FamiTone sound effect on channel 0..3
void sfx_play(unsigned char sound, unsigned char channel);

// play a DPCM sample, 1..63
void sample_play(unsigned char sample);



// poll controller and return flags like PAD_LEFT etc, input is pad number (0 or 1)
unsigned char pad_poll(unsigned char pad);

// poll controller in trigger mode, a flag is set only on button down, not hold
// if you need to poll the pad in both normal and trigger mode, poll it in the
// trigger mode for first, then use pad_state
unsigned char pad_trigger(unsigned char pad);

// get previous pad state without polling ports
inline unsigned char pad_state(unsigned char pad);


// set scroll, including rhe top bits
// it is always applied at beginning of a TV frame, not at the function call
void scroll(unsigned int x, unsigned int y);

// set scroll after screen split invoked by the sprite 0 hit
// warning: all CPU time between the function call and the actual split point will be wasted!
// warning: the program loop has to fit into the frame time, ppu_wait_frame should not be used
//          otherwise empty frames without split will be inserted, resulting in jumpy screen
// warning: only X scroll could be changed in this version
void split(unsigned int x, unsigned int y);


// select current chr bank for sprites, 0..1
void bank_spr(unsigned char n);

// select current chr bank for background, 0..1
void bank_bg(unsigned char n);



// get random number 0..255 or 0..65535
unsigned char rand8(void);
unsigned int  rand16(void);

// set random seed
void set_rand(unsigned int seed);



// when display is enabled, vram access could only be done with this vram update system
// the function sets a pointer to the update buffer that contains data and addresses
// in a special format. It allows to write non-sequental bytes, as well as horizontal or
// vertical nametable sequences.
// buffer pointer could be changed during rendering, but it only takes effect on a new frame
// number of transferred bytes is limited by vblank time
// to disable updates, call this function with NULL pointer

// the update data format:
//  MSB, LSB, byte for a non-sequental write
//  MSB|NT_UPD_HORZ, LSB, LEN, [bytes] for a horizontal sequence
//  MSB|NT_UPD_VERT, LSB, LEN, [bytes] for a vertical sequence
//  NT_UPD_EOF to mark end of the buffer

// length of this data should be under 256 bytes

inline void set_vram_update(unsigned char *buf);

// all following vram functions only work when display is disabled

// do a series of VRAM writes, the same format as for set_vram_update, but writes done right away
void flush_vram_update(unsigned char *buf);

// set vram pointer to write operations if you need to write some data to vram
inline void vram_adr(unsigned int adr);

// put a byte at current vram address, works only when rendering is turned off
inline void vram_put(unsigned char n);

// fill a block with a byte at current vram address, works only when rendering is turned off
void vram_fill(unsigned char n, unsigned int len);

// set vram autoincrement, 0 for +1 and not 0 for +32
inline void vram_inc(unsigned char n);

// read a block from current address of vram, works only when rendering is turned off
void vram_read(unsigned char *dst, unsigned int size);

// write a block to current address of vram, works only when rendering is turned off
void vram_write(const unsigned char *src, unsigned int size);


// unpack RLE data to current address of vram, mostly used for nametables
void vram_unrle(const unsigned char *data);

// unpack LZ4 data to this address
void vram_unlz4(const unsigned char *in, unsigned char *out,
				const unsigned uncompressed_size);
/*
	Rough speeds for a full 1024 nametable:
	- rle takes 0.5 frames
	- uncompressed takes 1.3 frames
	- lz4 takes 2.8 frames
*/


// like memset, but does not return anything
void memfill(void *dst, unsigned char value, unsigned int len);

// delay for N frames
void delay(unsigned char frames);

// display.sinc functions
void oam_clear_fast(void);
void oam_meta_spr_pal(unsigned char x,unsigned char y,unsigned char pal,const unsigned char *metasprite);
void oam_meta_spr_clip(signed int x,unsigned char y,const unsigned char *metasprite);



#define PAD_A			0x01
#define PAD_B			0x02
#define PAD_SELECT		0x04
#define PAD_START		0x08
#define PAD_UP			0x10
#define PAD_DOWN		0x20
#define PAD_LEFT		0x40
#define PAD_RIGHT		0x80

#define OAM_FLIP_V		0x80
#define OAM_FLIP_H		0x40
#define OAM_BEHIND		0x20

#define MAX(x1,x2)		((x1)<(x2)?(x2):(x1))
#define MIN(x1,x2)		((x1)<(x2)?(x1):(x2))

#define MASK_SPR		0x10
#define MASK_BG			0x08
#define MASK_EDGE_SPR		0x04
#define MASK_EDGE_BG		0x02

#define NAMETABLE_A		0x2000
#define NAMETABLE_B		0x2400
#define NAMETABLE_C		0x2800
#define NAMETABLE_D		0x2c00

#define NT_UPD_HORZ		0x40
#define NT_UPD_VERT		0x80
#define NT_UPD_EOF		0xff

// macro to calculate nametable address from X,Y in compile time

#define NTADR_A(x,y)	 	(NAMETABLE_A|(((y)<<5)|(x)))
#define NTADR_B(x,y) 		(NAMETABLE_B|(((y)<<5)|(x)))
#define NTADR_C(x,y) 		(NAMETABLE_C|(((y)<<5)|(x)))
#define NTADR_D(x,y) 		(NAMETABLE_D|(((y)<<5)|(x)))

// macro to get MSB and LSB

#define MSB(x)			(((x)>>8))
#define LSB(x)			(((x)&0xff))

#pragma compile("neslib.c")

#endif
