/* --8<--8<--8<--8<--
 *
 * Copyright (C) 2006 Smithsonian Astrophysical Observatory
 *
 * This file is part of tracefct
 *
 * tracefct is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * tracefct is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 *       Free Software Foundation, Inc. 
 *       51 Franklin Street, Fifth Floor
 *       Boston, MA  02110-1301, USA
 *
 * -->8-->8-->8-->8-- */

#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>


#include "tracefct.h"

/* forward definitions of private functions */
static struct	node *get_node(void);
static int	height_function_stack(void);
static int	is_stack_empty(void);
static void	print_function_stack(FILE *fp);
static void	print_prefix(FILE *fp);
static void	vexit_print(FILE *fout, const char *format,
			       va_list args);

/* private info */

struct node
{
  const char *function_name;
  struct node *next_ptr;
};

typedef struct node NODE;
typedef NODE *NODEPTR;
typedef NODEPTR STACKPTR;

static STACKPTR stack_function_names = NULL;
static const char *program_name = "UNKNOWN PROGRAM";
static int print_upon_enter_and_exit = 0;
static int stack_level_to_print = -1;

static char prefix[512];
static char outbuf[8192];

static FILE *output = NULL;/* = stderr;*/

#define NUM_BLANK_SPACES "   "

#ifdef __cplusplus
#  define TFDIE   TraceFct::die
#  define TFEXIT  TraceFct::exit
#  define TFLEAVE TraceFct::leave
#  define TFENTER TraceFct::enter
#  define TFOPEN  TraceFct::open
#  define TFCLOSE TraceFct::close
#  define TFINIT  TraceFct::init
#else 
#  define TFDIE   tf_die
#  define TFENTER tf_enter
#  define TFEXIT  tf_exit
#  define TFLEAVE tf_leave
#  define TFOPEN  tf_open
#  define TFCLOSE tf_close
#  define TFINIT  tf_init
#endif

/* public functions */

/*

  PROCEDURE
  tf_init - Initialize the function trace stack.

  DESCRIPTION
  This routine initializes the function stack as well as registering
  the name of the executable.  The calling routine can also set up
  some output options which determine the depth of the function stack
  to print and whether a diagnostic should be printed at every entry
  and return of a function.  @code{tracefct} stores the passed pointer
  to the program name, it does @emph{not} make a copy of the string.
  The calling function mustn't invalidate the pointer.

*/

void
TFINIT(
 const char *name,	/* the name of the program, preferrably
			   argv[0].  path information (if present) is
			   not stripped. */
 int print_it,		/* boolean flag.  if true, a message will be
			   output upon entry and exit of a
			   function. */
 int num_fct_to_print	/* if positive, only the requested number of
		     	   function names will be printed.  if
		     	   non-positive, all of the function names
		     	   will be printed. */
)
{
  /* save the function name */
  program_name = name;

  /* initialize_function name stack pointer */
  stack_function_names = NULL;

  print_upon_enter_and_exit = print_it;
  stack_level_to_print = num_fct_to_print;

  /* initialize output file handle */
  output = stderr;
}

/*

  PROCEDURE
  tf_enter - Register function.

  DESCRIPTION
  This routine is called upon entry into a function.  It registers the
  function's name so that it can be used in messages.  Note that it
  saves the pointer to the function name, it does @emph{not} make a
  copy of the string.

*/
void
TFENTER(
  const char *name		/* the name of the function. */
)
{
  NODEPTR newPtr;

  if ( NULL == ( newPtr = get_node() ) )
    TFEXIT(-1, "Unable to allocate space for node");
  newPtr->function_name = name;
  newPtr->next_ptr = stack_function_names;
  stack_function_names = newPtr;

  if (print_upon_enter_and_exit)
  {
    int i, height = height_function_stack();

    if ( NULL == output )
      output = stderr;
    
    print_prefix(output);

    for (i = 0; i < height - 1; i++)
      fputs(NUM_BLANK_SPACES, output);

    fputs("entering ", output);
    fputs(name, output);
    fputc('\n', output);
  }
}

/*

  PROCEDURE
  tf_die - Exit a program, dumping the function stack.

  DESCRIPTION
  This function prints an error message to @code{stderr} as well as
  the @code{tracefct} output stream (if different from @code{stderr})
  and then exits the program with a value of @code{EXIT_FAILURE}.  The
  error message is passed in the same fashion as the arguments to
  @code{vprintf} or @code{sprintf}.  The message may contain multiple
  output lines (i.e., multiple newline characters).  If a trailing
  newline character is not specified, it will be appended.

  Contrast this with @code{tf_exit}.

*/

void TFDIE
(
  const char *format,	/* a printf style format string. passed to vsprintf */
  ...			/* additional arguments to be passed to vfprintf */
)
{
  va_list args;

  va_start(args, format);

  vexit_print( stderr, format, args );

  va_end(args);

  if ( NULL == output )
    output = stderr;
  
  if ( stderr != output )
  {
    va_start(args, format);
    vexit_print( output, format, args );
    va_end(args);
  }

#ifdef __cplusplus
  ::exit(EXIT_FAILURE);
#else
  exit(EXIT_FAILURE);
#endif
}

/*

  PROCEDURE
  tf_exit - Exit a program, dumping the function stack.

  DESCRIPTION
  This function prints an error message to @code{stderr} as well as
  the @code{tracefct} output stream (if different from @code{stderr})
  and then exits the program with the supplied error code.
  Interesting error codes are predefined in @file{exiterrvals.h}.  The
  error message is passed in the same fashion as the arguments to
  @code{vprintf} or @code{sprintf}.  The message may contain multiple
  output lines (i.e., multiple newline characters).  If a trailing
  newline character is not specified, it will be appended.

*/

void TFEXIT
(
  int exit_code,	/* the exit code to be returned to the system */
  const char *format,		/* a printf style format string. passed to vsprintf */
  ...			/* additional arguments to be passed to vfprintf */
)
{
  va_list args;

  va_start(args, format);

  vexit_print( stderr, format, args );

  va_end(args);

  if ( NULL == output )
    output = stderr;
  
  if ( stderr != output )
  {
    va_start(args, format);
    vexit_print( output, format, args );
    va_end(args);
  }

#ifdef __cplusplus
  ::exit(exit_code);
#else
  exit(exit_code);
#endif
}

static void
vexit_print(
  FILE *fout,		/* where to print */
  const char *format,		/* a printf style format string. passed to vsprintf */
  va_list args		/* objects to print */
)
{
  char *start, *end;

  vsprintf(outbuf, format, args);

  for ( start = outbuf ; (end = strchr( start, '\n' )) ; start = end )
  {
    char save;
    save = *++end;
    *end = '\0';
    print_prefix(fout);
    fputs( start, fout );
    *end = save;
  }

  if ( *start )
  {
    print_prefix( fout );
    fputs(start, fout);
  }

  if (  '\n' != outbuf[strlen(outbuf)-1] )
    fputc('\n', fout);

  print_function_stack(fout);

  va_end(args);
}


/*

  PROCEDURE
  tf_leave - Deregister a function.
   
  DESCRIPTION
  This routine is called upon exit from a function.  It
  deregisters the last registered function.

*/
void
TFLEAVE(void)
{
  const char *name;
  NODEPTR tempPtr;

  if (is_stack_empty())
    TFEXIT(-2, "TraceFct error: The function stack is empty");

  tempPtr = stack_function_names;
  name = stack_function_names->function_name;
  stack_function_names = tempPtr->next_ptr;
  free(tempPtr);

  if (print_upon_enter_and_exit)
  {
    int i, height = height_function_stack();

    if ( NULL == output )
      output = stderr;
    
    print_prefix(output);
    
    for (i = 0 ; i < height ; i++)
      fputs(NUM_BLANK_SPACES, output);

    fputs("leaving ", output);
    fputs(name, output);
    fputc('\n', output);
  }
}



/*

  PROCEDURE
  tf_message - Print a formatted message to the @code{tracefct} output stream.

  DESCRIPTION
  This function prints a user supplied message to the @code{tracefct}
  output stream, prefixed by the standard @code{tracefct} style prefix
  string.  The message is passed in the same fashion as the arguments
  to @code{printf}, allowing formatted output.  Unlike @code{tf_exit},
  a newline character is @emph{not} appended to the message.

*/
void
#ifdef __cplusplus
TraceFct::message
#else
tf_message
#endif
(
  const char *format,		/* a printf style string */
  ...			/* objects to print */
)
{
  va_list args;

  va_start(args, format);

  if ( NULL == output )
    output = stderr;
  
  print_prefix(output);

  vsprintf(outbuf, format, args);
  fputs(outbuf, output);

  va_end(args);
}


/*

  PROCEDURE
  tf_vmessage - Print a message to @code{stderr} using a
  @code{stdargs} argument list.

  DESCRIPTION
  This function prints a user supplied message to @code{stderr},
  prefixed by the standard @code{tracefct} style prefix string.  The
  error message is passed in the same fashion as the arguments to
  @code{vprintf}.  Note that @code{vmessage} expects a @var{va_list}
  to be passed, rather than a variable argument list.  Unlike
  @code{tf_exit}, a newline character is @emph{not} appended to the
  message.

*/
void
#ifdef __cplusplus
TraceFct::vmessage
#else
tf_vmessage
#endif
(
  const char *format,		/* a printf style string */
  va_list args		/* objects to print */
)
{
  if ( NULL == output )
    output = stderr;

  print_prefix(output);

  vsprintf(outbuf, format, args);
  fputs(outbuf, output);
}


/*

  PROCEDURE
  tf_dump_stack - Dump the called function stack

  DESCRIPTION
  This function dumps the current function stack

*/   

void 
#ifdef __cplusplus
TraceFct::dump_stack
#else
tf_dump_stack
#endif
(void)
{
  if ( NULL == output )
    output = stderr;
  
  print_function_stack(output);
}

/*

  PROCEDURE
  tf_open - redirect the @code{tracefct} output stream

  DESCRIPTION
  @code{tf_open} serves to change the @code{tracefct} output stream to the
  specified file.  Normally @code{tracefct} writes to @code{stderr}.
  @code{tf_exit} @emph{always} writes to @code{stderr} (@pxref{tf_exit}).
  To reset the output stream, see @ref{tf_close}.  If the current output
  stream is not @code{stderr}, it is automatically closed.

  If the passed filename is the string @samp{stderr}, @code{tf_open}
  will use @code{stderr}.

  RETURNS
  It returns zero upon success, non-zero upon failure.

*/
int TFOPEN(
  const char *file	/* the file to which messages are to be written */
  )
{
  FILE *fout;

  if ( strcmp( file, "stderr" ) )
  {
    fout = fopen( file, "w" );

    if ( NULL == fout )
      return 1;
  }
  else
    fout = stderr;

  TFCLOSE( );
  output = fout;
    
  return 0;
}

/*

  PROCEDURE
  tf_close - rest the @code{tracefct} output stream to @code{stderr}.

  DESCRIPTION
  @code{tf_close} resets the output stream to @code{stderr}, closing
  the file previously opened by @code{tf_open}.

*/

void
TFCLOSE( void )
{
  if ( NULL != output && stderr != output )
  {
    fclose( output );
    output = stderr;
  }
}


/*
  allocate a node of the linked list
*/
static struct node *
get_node(void)
{
  return (NODEPTR) malloc(sizeof(NODE));
}

/* print out the standard prefix to the output stream, but only if
   the output stream is stderr */
static void
print_prefix(FILE *fp)
{
  if ( stderr == fp )
  {
    sprintf(prefix, "# %d: %s: ", getpid(), program_name ? program_name : "");
    fputs(prefix, fp);
  }
}

/*
  find the size of the stack
*/
static int
height_function_stack(void)
{
  int length = 0;
  NODEPTR tempPtr;

  if (is_stack_empty())
    return length;

  tempPtr = stack_function_names;
  while (tempPtr)
  {
    tempPtr = tempPtr->next_ptr;
    length++;
  }
  return length;
}


/*
  check if stack is empty or not
*/
static int
is_stack_empty(void)
{
  return !stack_function_names;
}


/*
  prints the current function stack (functions that have been called) to
  a stream;
*/
static void
print_function_stack(
  FILE *fp	/* the file stream to which to print the function stack */
)
{
  int counter = 0, max_level;
  NODEPTR tempPtr;

  tempPtr = stack_function_names;

  print_prefix(fp);

  fputs( "stack: ", fp );

  if (is_stack_empty())
  {
    fputs("<TOP>\n", fp);
    return;
  }

  max_level = stack_level_to_print > 0 ?
              stack_level_to_print : height_function_stack();

  while (tempPtr && counter < max_level)
  {
    if (counter)
      fputs("<-", fp);
    fputs(tempPtr->function_name, fp);
    tempPtr = tempPtr->next_ptr;
    counter++;
  }

  if ( max_level >= height_function_stack() )
    fputs("<-<TOP>\n", fp);
  else
    fputs("<-\n", fp);
    
}
