/*
   This file is part of MDic Dictionary

   Copyright (C) 2007-2010 Mehrdad Momeny <mehrdad.momeny@gmail.com
   Copyright (C) 2007-2010 Majid Ramezanpour <majid.ramezanpour@gmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License,
   (at your option) any later version

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details

   You should have received a copy of the GNU General Public License
   along with this program; if not, see http://www.gnu.org/licenses/

*/

#ifndef DICTCORE_H
#define DICTCORE_H

#include <QObject>

#include <QStringList>
#include <QPair>
#include <QHash>
#include <QFont>
#include "dicts/dict.h"

/**
 * The DictCore is a base dictionary class.
 */
class DictCore: public QObject
{
Q_OBJECT

public:
    enum BackendType {MDic =0, GoogleTranslate};

    /**
    * This class represents a dictionary.
    */
    class Dictionary
    {
    public:
        Dictionary(BackendType type, const QString &name)
            :m_name(name), m_isEnabled(true)
        {
            setBackendType(type);
        }

        Dictionary()
        { }

        bool isEnabled() const
        { return m_isEnabled; }

        void setEnabled(bool enabled = true)
        { m_isEnabled = enabled; }

        BackendType backendType() const
        { return m_backend; }

        QString typeName() const
        { return m_typeName; }

        const QString &name() const
        { return m_name; }

        void setBackendType(BackendType backend)
        {
            m_backend = backend;
            if(backend == MDic)
                m_typeName = "MDic";
            else
                m_typeName = "Google Translate";
        }

        void setName(const QString &name)
        { m_name = name; }

        bool operator == (const Dictionary &dict)
        { return m_name == dict.name() && m_backend == dict.backendType(); }

        QString code() const
        { return m_code; }

        void setCode( const QString &code )
        { m_code = code; }

    private:
        QString m_typeName;
        QString m_name;
        QString m_code;
        BackendType m_backend;
        bool m_isEnabled;
    };

    /**
        * Destructor.
        */
    ~DictCore();

    static DictCore * self();

    /**
        * Returns translation for word. If word not found, an empty list will return
        */
    QList<Translation> translate(const QString &word, bool justFirst = false);

    /**
        * Returns a list of loaded dictionaries.
        */
    const QList<Dictionary> &loadedDicts() const
    { return m_loadedDicts; }

    QList<Dictionary> loadedRemoteDicts() const;

    Dictionary dictionary( const QString &code );
    /**
    * Sets a loaded dictionaries.
    */
    void setLoadedDicts(const QList<Dictionary>& loadedDicts);

    static QString translationToHtml(const Translation &translation);
    static QString fontToHtml(QFont* inFont, const QString& text, const QString& color = QString());

private:
    /**
    * Construct dictionary.
    */
    DictCore();

    static DictCore * m_self;

    QList<Dictionary> m_loadedDicts;
};

#endif // DICTCORE_H

// vim: tabstop=4 softtabstop=4 shiftwidth=4 expandtab cindent textwidth=120 formatoptions=tc

