/*
  native-level.c -- native level module;

  Copyright (C) 2015, 2016, 2017 Bruno Félix Rezende Ribeiro
  <oitofelix@gnu.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "mininim.h"

struct level *
next_native_level (struct level *l, int n)
{
  n = validate_legacy_level_number (n);
  return load_native_level (l, n);
}

struct level *
load_native_level (struct level *l, int n)
{
  char *filename;
  xasprintf (&filename, "data/levels/%02d.mim", n);

  ALLEGRO_CONFIG *c =
    load_resource (filename, (load_resource_f) al_load_config_file);

  if (! c) {
    error (0, 0, "cannot read native level file %s", filename);
    return NULL;
  }

  al_free (filename);

  char *k;
  const char *v;
  int i;

  memset (l, 0, sizeof (*l));

  l->n = n;
  l->start = legacy_level_start;
  l->special_events = legacy_level_special_events;
  l->end = legacy_level_end;
  l->next_level = next_native_level;

  /* CUTSCENES: ok */
  switch (n) {
  default: break;
  case 1: l->cutscene = cutscene_01_05_11_anim; break;
  case 3: l->cutscene = cutscene_03_anim; break;
  case 5: l->cutscene = cutscene_01_05_11_anim; break;
  case 7: l->cutscene = cutscene_07_anim; break;
  case 8: l->cutscene = cutscene_08_anim; break;
  case 11: l->cutscene = cutscene_11_anim; break;
  case 14: l->cutscene = cutscene_14_anim; break;
  }

  /* NOMINAL NUMBER */
  /* N=n */
  l->nominal_n = n;
  v = al_get_config_value (c, NULL, "N");
  if (v) sscanf (v, "%i", &l->nominal_n);

  /* START POSITION AND DIRECTION */
  /* P=r f p d s */
  struct pos *sp = &l->start_pos;
  sp->room = 1, sp->floor = 0, sp->place = 0,
    l->start_dir = LEFT, l->has_sword = false;
  v = al_get_config_value (c, NULL, "P");
  if (v) sscanf (v, "%i %i %i %i %i", &sp->room, &sp->floor, &sp->place,
                 (int *) &l->start_dir, (int *) &l->has_sword);

  /* ENVIRONMENT AND HUE */
  /* S=e h */
  l->em = DUNGEON, l->hue = HUE_NONE;
  v = al_get_config_value (c, NULL, "S");
  if (v) sscanf (v, "%i %i", (int *) &l->em, (int *) &l->hue);

  /* GUARDS */
  for (i = 0;; i++) {
    struct guard *g = guard (l, i);

    /* GUARD TYPE AND STYLE */
    /* GiT=t s */
    xasprintf (&k, "G%iT", i);
    v = al_get_config_value (c, NULL, k);
    al_free (k);
    if (! v) break;
    g->type = NO_ANIM, g->style = 0;
    sscanf (v, "%i %i", (int *) &g->type, (int *) &g->style);

    /* GUARD START POSITION AND DIRECTION */
    /* GiP=r f p d */
    xasprintf (&k, "G%iP", i);
    v = al_get_config_value (c, NULL, k);
    al_free (k);
    if (! v) break;
    new_pos (&g->p, NULL, 0, 0, 0);
    g->dir = LEFT;
    sscanf (v, "%i %i %i %i", &g->p.room, &g->p.floor, &g->p.place,
            (int *) &g->dir);

    /* GUARD SKILLS AND TOTAL LIVES */
    /* GiK=a b d e a r f x l */
    xasprintf (&k, "G%iK", i);
    v = al_get_config_value (c, NULL, k);
    al_free (k);
    if (! v) break;
    sscanf (v, "%i %i %i %i %i %i %i %i %i",
            &g->skill.attack_prob, &g->skill.counter_attack_prob,
            &g->skill.defense_prob, &g->skill.counter_defense_prob,
            &g->skill.advance_prob, &g->skill.return_prob,
            &g->skill.refraction, &g->skill.extra_life,
            &g->total_lives);
  }

  /* LINKS */
  for (i = 1;; i++) {
    /* Li=l r a b */
    struct room_linking *r = llink (l, i);
    xasprintf (&k, "L%i", i);
    v = al_get_config_value (c, NULL, k);
    al_free (k);
    if (! v) break;
    sscanf (v, "%i %i %i %i", &r->l, &r->r, &r->a, &r->b);
  }

  /* EVENTS */
  for (i = 0;; i++) {
    /* Ei=r f p n*/
    struct level_event *e = event (l, i);
    xasprintf (&k, "E%i", i);
    v = al_get_config_value (c, NULL, k);
    al_free (k);
    if (! v) break;
    sscanf (v, "%i %i %i %i", &e->p.room, &e->p.floor, &e->p.place,
            (int *) &e->next);
  }

  /* CONSTRUCTIONS */
  struct pos p; new_pos (&p, l, -1, -1, -1);
  for (p.room = 1;; p.room++)
    for (p.floor = 0; p.floor < FLOORS; p.floor++)
      for (p.place = 0; p.place < PLACES; p.place++) {
        /* Cr f p=f b e ff */
        xasprintf (&k, "C%i %i %i", p.room, p.floor, p.place);
        v = al_get_config_value (c, NULL, k);
        al_free (k);
        if (! v) goto end_con_loop;
        int f = 0, b = 0, e = 0, ff = -1;
        sscanf (v, "%i %i %i %i", (int *) &f, &b, &e, &ff);
        set_con (&p, f, b, e, ff);
      }
 end_con_loop:

  al_destroy_config (c);

  return l;
}

bool
save_native_level (struct level *l, char *filename)
{
  ALLEGRO_CONFIG *c = create_config ();
  char *k, *v;
  int i;

  /* MININIM LEVEL FILE */
  al_add_config_comment (c, NULL, "MININIM LEVEL FILE");

  /* ENGINE VERSION */
  /* V=n */
  xasprintf (&v, "%s", VERSION);
  al_set_config_value (c, NULL, "V", v);
  al_free (v);

  /* NOMINAL NUMBER */
  /* N=n */
  xasprintf (&v, "%i", l->nominal_n);
  al_set_config_value (c, NULL, "N", v);
  al_free (v);

  /* START POSITION AND DIRECTION*/
  /* P=r f p d s */
  struct pos *sp = &l->start_pos;
  xasprintf (&v, "%i %i %i %i %i", sp->room, sp->floor, sp->place,
             l->start_dir, l->has_sword);
  al_set_config_value (c, NULL, "P", v);
  al_free (v);

  /* ENVIRONMENT AND HUE */
  /* S=e h */
  xasprintf (&v, "%i %i", l->em, l->hue);
  al_set_config_value (c, NULL, "S", v);
  al_free (v);

  /* GUARDS */
  for (i = 0; i < GUARDS; i++) {
    struct guard *g = guard (l, i);

    /* GUARD TYPE AND STYLE */
    /* GiT=t s */
    xasprintf (&k, "G%iT", i);
    xasprintf (&v, "%i %i", g->type, g->style);
    al_set_config_value (c, NULL, k, v);
    al_free (k);
    al_free (v);

    /* GUARD START POSITION AND DIRECTION */
    /* GiP=r f p d */
    xasprintf (&k, "G%iP", i);
    xasprintf (&v, "%i %i %i %i", g->p.room, g->p.floor, g->p.place, g->dir);
    al_set_config_value (c, NULL, k, v);
    al_free (k);
    al_free (v);

    /* GUARD SKILLS AND TOTAL LIVES */
    /* GiK=a b d e a r f x l */
    xasprintf (&k, "G%iK", i);
    xasprintf (&v, "%i %i %i %i %i %i %i %i %i",
               g->skill.attack_prob, g->skill.counter_attack_prob,
               g->skill.defense_prob, g->skill.counter_defense_prob,
               g->skill.advance_prob, g->skill.return_prob,
               g->skill.refraction, g->skill.extra_life,
               g->total_lives);
    al_set_config_value (c, NULL, k, v);
    al_free (k);
    al_free (v);
  }

  /* LINKS */
  for (i = 1; i < ROOMS; i++) {
    /* Li=l r a b */
    struct room_linking *r = llink (l, i);
    xasprintf (&k, "L%i", i);
    xasprintf (&v, "%i %i %i %i", r->l, r->r, r->a, r->b);
    al_set_config_value (c, NULL, k, v);
    al_free (k);
    al_free (v);
  }

  /* EVENTS */
  for (i = 0; i < EVENTS; i++) {
    /* Ei=r f p n */
    struct level_event *e = event (l, i);
    xasprintf (&k, "E%i", i);
    xasprintf (&v, "%i %i %i %i", e->p.room, e->p.floor, e->p.place, e->next);
    al_set_config_value (c, NULL, k, v);
    al_free (k);
    al_free (v);
  }

  /* CONSTRUCTIONS */
  struct pos p; new_pos (&p, l, -1, -1, -1);
  for (p.room = 1; p.room < ROOMS; p.room++)
    for (p.floor = 0; p.floor < FLOORS; p.floor++)
      for (p.place = 0; p.place < PLACES; p.place++) {
        /* Cr f p=f b e ff */
        xasprintf (&k, "C%i %i %i", p.room, p.floor, p.place);
        xasprintf (&v, "%i %i %i %i", fg (&p), bg (&p),
                   ext (&p), con (&p)->fake);
        al_set_config_value (c, NULL, k, v);
        al_free (k);
        al_free (v);
      }

  bool r = al_save_config_file (filename, c);
  al_destroy_config (c);
  return r;
}

char *
get_confg_str (struct pos *p)
{
  switch (fg (p)) {
  case NO_FLOOR: return "NO_FLOOR";
  case FLOOR: return "FLOOR";
  case BROKEN_FLOOR: return "BROKEN_FLOOR";
  case SKELETON_FLOOR: return "SKELETON_FLOOR";
  case LOOSE_FLOOR: return "LOOSE_FLOOR";
  case SPIKES_FLOOR: return "SPIKES_FLOOR";
  case OPENER_FLOOR: return "OPENER_FLOOR";
  case CLOSER_FLOOR: return "CLOSER_FLOOR";
  case STUCK_FLOOR: return "STUCK_FLOOR";
  case HIDDEN_FLOOR: return "HIDDEN_FLOOR";
  case PILLAR: return "PILLAR";
  case BIG_PILLAR_BOTTOM: return "BIG_PILLAR_BOTTOM";
  case BIG_PILLAR_TOP: return "BIG_PILLAR_TOP";
  case WALL: return "WALL";
  case DOOR: return "DOOR";
  case LEVEL_DOOR: return "LEVEL_DOOR";
  case CHOPPER: return "CHOPPER";
  case MIRROR: return "MIRROR";
  case CARPET: return "CARPET";
  case TCARPET: return "TCARPET";
  case ARCH_BOTTOM: return "ARCH_BOTTOM";
  case ARCH_TOP_LEFT: return "ARCH_TOP_LEFT";
  case ARCH_TOP_RIGHT: return "ARCH_TOP_RIGHT";
  case ARCH_TOP_MID: return "ARCH_TOP_MID";
  case ARCH_TOP_SMALL: return "ARCH_TOP_SMALL";
  default: return NULL;
  }
}

char *
get_conbg_str (struct pos *p)
{
  switch (bg (p)) {
  case NO_BG: return "NO_BG";
  case BRICKS_00: return "BRICKS_00";
  case BRICKS_01: return "BRICKS_01";
  case BRICKS_02: return "BRICKS_02";
  case BRICKS_03: return "BRICKS_03";
  case NO_BRICKS: return "NO_BRICKS";
  case TORCH: return "TORCH";
  case WINDOW: return "WINDOW";
  case BALCONY: return "BALCONY";
  default: return NULL;
  }
}

char *
get_conext_str (struct pos *p)
{
  char *s = NULL;
  int e = ext (p);

  switch (fg (p)) {
  case FLOOR:
    switch (e) {
    case NO_ITEM: xasprintf (&s, "NO_ITEM"); break;
    case EMPTY_POTION: xasprintf (&s, "EMPTY_POTION"); break;
    case SMALL_LIFE_POTION: xasprintf (&s, "SMALL_LIFE_POTION"); break;
    case BIG_LIFE_POTION: xasprintf (&s, "BIG_LIFE_POTION"); break;
    case SMALL_POISON_POTION: xasprintf (&s, "SMALL_POISON_POTION"); break;
    case BIG_POISON_POTION: xasprintf (&s, "BIG_POISON_POTION"); break;
    case FLOAT_POTION: xasprintf (&s, "FLOAT_POTION"); break;
    case FLIP_POTION: xasprintf (&s, "FLIP_POTION"); break;
    case ACTIVATION_POTION: xasprintf (&s, "ACTIVATION_POTION"); break;
    case SWORD: xasprintf (&s, "SWORD"); break;
    default: assert (false); break;
    }
    break;
  case LOOSE_FLOOR:
    xasprintf (&s, "%s", e ? "CANT_FALL" : "FALL");
    break;
  case SPIKES_FLOOR: case DOOR: case LEVEL_DOOR: case CHOPPER:
    xasprintf (&s, "%i", e);
    break;
  case OPENER_FLOOR: case CLOSER_FLOOR:
    xasprintf (&s, "%i", e);
    break;
  case CARPET:
    switch (e) {
    case CARPET_00: s = "CARPET_00"; break;
    case CARPET_01: s = "CARPET_01"; break;
    case ARCH_CARPET_RIGHT_00: s = "ARCH_CARPET_RIGHT_00"; break;
    case ARCH_CARPET_RIGHT_01: s = "ARCH_CARPET_RIGHT_01"; break;
    case ARCH_CARPET_LEFT_00: s = "ARCH_CARPET_LEFT_00"; break;
    case ARCH_CARPET_LEFT_01: s = "ARCH_CARPET_LEFT_01"; break;
    default: assert (false); break;
    }
    if (s) xasprintf (&s, "%s", s);
    break;
  case TCARPET:
    switch (e) {
    case CARPET_00: s = "CARPET_00"; break;
    case CARPET_01: s = "CARPET_01"; break;
    case ARCH_CARPET_RIGHT_00: s = "ARCH_CARPET_RIGHT_00"; break;
    case ARCH_CARPET_RIGHT_01: s = "ARCH_CARPET_RIGHT_01"; break;
    case ARCH_CARPET_LEFT_00: s = "ARCH_CARPET_LEFT_00"; break;
    case ARCH_CARPET_LEFT_01: s = "ARCH_CARPET_LEFT_01"; break;
    default: assert (false); break;
    }
    if (s) xasprintf (&s, "%s", s);
    break;
  default: break;
  }

  return s;
}
