"use strict";
// Copyright 2021-2024 The Connect Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
var __asyncValues = (this && this.__asyncValues) || function (o) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var m = o[Symbol.asyncIterator], i;
    return m ? m.call(o) : (o = typeof __values === "function" ? __values(o) : o[Symbol.iterator](), i = {}, verb("next"), verb("throw"), verb("return"), i[Symbol.asyncIterator] = function () { return this; }, i);
    function verb(n) { i[n] = o[n] && function (v) { return new Promise(function (resolve, reject) { v = o[n](v), settle(resolve, reject, v.done, v.value); }); }; }
    function settle(resolve, reject, d, v) { Promise.resolve(v).then(function(v) { resolve({ value: v, done: d }); }, reject); }
};
var __await = (this && this.__await) || function (v) { return this instanceof __await ? (this.v = v, this) : new __await(v); }
var __asyncGenerator = (this && this.__asyncGenerator) || function (thisArg, _arguments, generator) {
    if (!Symbol.asyncIterator) throw new TypeError("Symbol.asyncIterator is not defined.");
    var g = generator.apply(thisArg, _arguments || []), i, q = [];
    return i = {}, verb("next"), verb("throw"), verb("return", awaitReturn), i[Symbol.asyncIterator] = function () { return this; }, i;
    function awaitReturn(f) { return function (v) { return Promise.resolve(v).then(f, reject); }; }
    function verb(n, f) { if (g[n]) { i[n] = function (v) { return new Promise(function (a, b) { q.push([n, v, a, b]) > 1 || resume(n, v); }); }; if (f) i[n] = f(i[n]); } }
    function resume(n, v) { try { step(g[n](v)); } catch (e) { settle(q[0][3], e); } }
    function step(r) { r.value instanceof __await ? Promise.resolve(r.value.v).then(fulfill, reject) : settle(q[0][2], r); }
    function fulfill(value) { resume("next", value); }
    function reject(value) { resume("throw", value); }
    function settle(f, v) { if (f(v), q.shift(), q.length) resume(q[0][0], q[0][1]); }
};
Object.defineProperty(exports, "__esModule", { value: true });
const http2 = require("http2");
const http = require("http");
const connect_1 = require("@connectrpc/connect");
const protocol_1 = require("@connectrpc/connect/protocol");
const node_universal_client_js_1 = require("./node-universal-client.js");
const use_node_server_helper_spec_js_1 = require("./use-node-server-helper.spec.js");
// Polyfill the Headers API for Node versions < 18
require("./node-headers-polyfill.js");
describe("node http/2 client closing with RST_STREAM with code CANCEL", function () {
    let serverReceivedRstCode;
    const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer().on("stream", (stream) => {
        stream.on("close", () => (serverReceivedRstCode = stream.rstCode));
    }));
    it("should send RST_STREAM frame to the server", async function () {
        return new Promise((resolve) => {
            http2.connect(server.getUrl(), (session) => {
                const stream = session.request({
                    ":method": "POST",
                    ":path": "/",
                }, {});
                setTimeout(() => {
                    stream.close(http2.constants.NGHTTP2_CANCEL, () => {
                        // We are seeing a race condition in Node v16.20.0, where closing
                        // the session right after closing a stream with an RST code
                        // _sometimes_ sends an INTERNAL_ERROR code.
                        // Simply delaying the session close until the next tick like
                        // we do here seems to work around the issue.
                        // We do _not_ guard against this case in the universal client,
                        // since we were not able to reproduce the issue there.
                        setTimeout(() => session.close(resolve), 0);
                    });
                }, 0);
            });
        });
        while (serverReceivedRstCode === undefined) {
            // wait for the server to see the reset code
            await new Promise((resolve) => setTimeout(resolve, 1));
        }
        expect(serverReceivedRstCode).toBe(http2.constants.NGHTTP2_CANCEL);
    });
});
describe("universal node http client", function () {
    describe("against an unresolvable host", function () {
        for (const httpVersion of ["2", "1.1"]) {
            describe(`over http ${httpVersion}`, function () {
                it("should raise Code.Unavailable", async function () {
                    const client = (0, node_universal_client_js_1.createNodeHttpClient)({
                        httpVersion,
                    });
                    try {
                        await client({
                            url: "https://unresolvable-host.some.domain",
                            method: "POST",
                            header: new Headers(),
                        });
                    }
                    catch (e) {
                        expect(connect_1.ConnectError.from(e).message).toBe("[unavailable] getaddrinfo ENOTFOUND unresolvable-host.some.domain");
                    }
                });
            });
        }
    });
    describe("against a server that closes immediately", function () {
        describe("over http/2", function () {
            let serverReceivedRequest = false;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer((request, response) => {
                serverReceivedRequest = true;
                response.stream.close(http2.constants.NGHTTP2_CANCEL);
            }));
            it("should reject the response promise with Code.Canceled", async function () {
                const client = server.getClient();
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] http/2 stream closed with error code CANCEL (0x8)");
                }
                expect(serverReceivedRequest).toBeTrue();
            });
        });
        describe("over http/1.1", function () {
            let serverReceivedRequest = false;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer((req, res) => {
                serverReceivedRequest = true;
                res.destroy();
            }));
            it("should reject the response promise", async function () {
                const client = server.getClient();
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[aborted] socket hang up");
                }
                expect(serverReceivedRequest).toBeTrue();
            });
        });
    });
    describe("against a server that closes before the first response byte", function () {
        describe("over http/2", function () {
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer((req, res) => {
                res.writeHead(200);
                // Calling close in the same tick as writeHead appears to prevent
                // headers from being sent. The client response promise will reject,
                // instead of the response body.
                setTimeout(() => res.stream.close(http2.constants.NGHTTP2_CANCEL), 0);
            }));
            it("should reject the response promise with Code.Canceled", async function () {
                var _a, e_1, _b, _c;
                const client = server.getClient();
                const res = await client({
                    url: server.getUrl(),
                    method: "POST",
                    header: new Headers(),
                });
                try {
                    try {
                        for (var _d = true, _e = __asyncValues(res.body), _f; _f = await _e.next(), _a = _f.done, !_a; _d = true) {
                            _c = _f.value;
                            _d = false;
                            const chunk = _c;
                            expect(chunk)
                                .withContext("response body iterable should be empty")
                                .toBeUndefined();
                        }
                    }
                    catch (e_1_1) { e_1 = { error: e_1_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = _e.return)) await _b.call(_e);
                        }
                        finally { if (e_1) throw e_1.error; }
                    }
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] http/2 stream closed with error code CANCEL (0x8)");
                }
            });
        });
        describe("over http/1.1", function () {
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer((req, res) => {
                res.writeHead(200);
                res.flushHeaders();
                res.destroy();
            }));
            it("should reject the response promise", async function () {
                var _a, e_2, _b, _c;
                const client = server.getClient();
                const res = await client({
                    url: server.getUrl(),
                    method: "POST",
                    header: new Headers(),
                });
                try {
                    try {
                        for (var _d = true, _e = __asyncValues(res.body), _f; _f = await _e.next(), _a = _f.done, !_a; _d = true) {
                            _c = _f.value;
                            _d = false;
                            const chunk = _c;
                            expect(chunk)
                                .withContext("response body iterable should be empty")
                                .toBeUndefined();
                        }
                    }
                    catch (e_2_1) { e_2 = { error: e_2_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = _e.return)) await _b.call(_e);
                        }
                        finally { if (e_2) throw e_2.error; }
                    }
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[aborted] aborted");
                }
            });
        });
    });
    describe("against a server that closes mid request", function () {
        describe("over http/2", function () {
            let serverReceivedBytes = 0;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer((req, res) => {
                void (async () => {
                    var _a, e_3, _b, _c;
                    try {
                        for (var _d = true, req_1 = __asyncValues(req), req_1_1; req_1_1 = await req_1.next(), _a = req_1_1.done, !_a; _d = true) {
                            _c = req_1_1.value;
                            _d = false;
                            const chunk = _c;
                            serverReceivedBytes += chunk.byteLength;
                            res.stream.close(http2.constants.NGHTTP2_CANCEL);
                            break;
                        }
                    }
                    catch (e_3_1) { e_3 = { error: e_3_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = req_1.return)) await _b.call(req_1);
                        }
                        finally { if (e_3) throw e_3.error; }
                    }
                })();
            }));
            it("should reject the response promise with Code.Canceled", async function () {
                const client = server.getClient();
                function body() {
                    return __asyncGenerator(this, arguments, function* body_1() {
                        yield yield __await(new Uint8Array(32));
                        yield __await(new Promise(() => {
                            // never resolves
                        }));
                    });
                }
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        body: body(),
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] http/2 stream closed with error code CANCEL (0x8)");
                }
                expect(serverReceivedBytes).toBe(32);
            });
        });
        describe("over http/1.1", function () {
            let serverReceivedBytes = 0;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer((req, res) => {
                void (async () => {
                    var _a, e_4, _b, _c;
                    try {
                        for (var _d = true, req_2 = __asyncValues(req), req_2_1; req_2_1 = await req_2.next(), _a = req_2_1.done, !_a; _d = true) {
                            _c = req_2_1.value;
                            _d = false;
                            const chunk = _c;
                            serverReceivedBytes += chunk.byteLength;
                            res.destroy();
                            break;
                        }
                    }
                    catch (e_4_1) { e_4 = { error: e_4_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = req_2.return)) await _b.call(req_2);
                        }
                        finally { if (e_4) throw e_4.error; }
                    }
                })();
            }));
            it("should reject the response promise", async function () {
                const client = server.getClient();
                function body() {
                    return __asyncGenerator(this, arguments, function* body_2() {
                        yield yield __await(new Uint8Array(32));
                        yield __await(new Promise(() => {
                            // never resolves
                        }));
                    });
                }
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        body: body(),
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[aborted] socket hang up");
                }
                expect(serverReceivedBytes).toBe(32);
            });
        });
    });
    describe("against a server that closes mid response", function () {
        describe("over http/2", function () {
            let serverSentBytes = 0;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer((req, res) => {
                void (async () => {
                    res.writeHead(200);
                    await new Promise((resolve, reject) => res.write(new Uint8Array(64), (e) => e ? reject(e) : resolve()));
                    serverSentBytes += 64;
                    res.stream.close(http2.constants.NGHTTP2_CANCEL);
                })();
            }));
            it("should reject the response promise with Code.Canceled", async function () {
                var _a, e_5, _b, _c;
                const client = server.getClient();
                const res = await client({
                    url: server.getUrl(),
                    method: "POST",
                    header: new Headers(),
                });
                try {
                    try {
                        for (var _d = true, _e = __asyncValues(res.body), _f; _f = await _e.next(), _a = _f.done, !_a; _d = true) {
                            _c = _f.value;
                            _d = false;
                            const chunk = _c;
                            expect(chunk.byteLength).toBe(64);
                        }
                    }
                    catch (e_5_1) { e_5 = { error: e_5_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = _e.return)) await _b.call(_e);
                        }
                        finally { if (e_5) throw e_5.error; }
                    }
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] http/2 stream closed with error code CANCEL (0x8)");
                }
                expect(serverSentBytes).toBe(64);
            });
        });
        describe("over http/1.1", function () {
            let serverSentBytes = 0;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer((req, res) => {
                void (async () => {
                    res.writeHead(200);
                    await new Promise((resolve, reject) => res.write(new Uint8Array(64), (e) => (e ? reject(e) : resolve())));
                    serverSentBytes += 64;
                    res.destroy();
                })();
            }));
            it("should reject the response promise", async function () {
                var _a, e_6, _b, _c;
                const client = server.getClient();
                const res = await client({
                    url: server.getUrl(),
                    method: "POST",
                    header: new Headers(),
                });
                try {
                    try {
                        for (var _d = true, _e = __asyncValues(res.body), _f; _f = await _e.next(), _a = _f.done, !_a; _d = true) {
                            _c = _f.value;
                            _d = false;
                            const chunk = _c;
                            expect(chunk.byteLength).toBe(64);
                        }
                    }
                    catch (e_6_1) { e_6 = { error: e_6_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = _e.return)) await _b.call(_e);
                        }
                        finally { if (e_6) throw e_6.error; }
                    }
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toMatch(/\[aborted] (aborted|read ECONNRESET)/);
                }
                expect(serverSentBytes).toBe(64);
            });
        });
    });
    describe("with a signal that is already aborted", function () {
        describe("over http/2", function () {
            let serverReceivedRequest = false;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer(() => (serverReceivedRequest = true)));
            it("should raise error with Code.Canceled and never hit the server", async function () {
                const client = server.getClient();
                const signal = AbortSignal.abort();
                // client should raise error
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        signal,
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // request should never hit the server
                expect(serverReceivedRequest).toBeFalse();
            });
        });
        describe("over http/1.1", function () {
            let serverReceivedRequest = false;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer(() => (serverReceivedRequest = true)));
            it("should raise error with Code.Canceled and never hit the server", async function () {
                const client = server.getClient();
                const signal = AbortSignal.abort();
                // client should raise error
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        signal,
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // request should never hit the server
                expect(serverReceivedRequest).toBeFalse();
            });
        });
    });
    describe("with a signal aborting before first request byte", function () {
        describe("over http/2", function () {
            let serverReceivedRstCode;
            let serverReceivedBytes = 0;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer((req, res) => {
                res.stream.on("close", () => (serverReceivedRstCode = res.stream.rstCode));
                void (async () => {
                    var _a, e_7, _b, _c;
                    try {
                        for (var _d = true, req_3 = __asyncValues(req), req_3_1; req_3_1 = await req_3.next(), _a = req_3_1.done, !_a; _d = true) {
                            _c = req_3_1.value;
                            _d = false;
                            const chunk = _c;
                            serverReceivedBytes += chunk.byteLength;
                        }
                    }
                    catch (e_7_1) { e_7 = { error: e_7_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = req_3.return)) await _b.call(req_3);
                        }
                        finally { if (e_7) throw e_7.error; }
                    }
                })();
            }));
            it("should raise error with code canceled and send RST_STREAM with code CANCEL", async function () {
                // set up a client that aborts while still streaming the request body
                const client = server.getClient();
                const ac = new AbortController();
                function body() {
                    return __asyncGenerator(this, arguments, function* body_3() {
                        yield __await(new Promise((resolve) => setTimeout(resolve, 50)));
                        ac.abort();
                        yield yield __await(new Uint8Array(32));
                    });
                }
                // client should raise error
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        body: body(),
                        signal: ac.signal,
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // server should receive chunks until client cancelled
                while (serverReceivedRstCode === undefined) {
                    // wait for the server to see the reset code
                    await new Promise((resolve) => setTimeout(resolve, 1));
                }
                expect(serverReceivedRstCode).toBe(http2.constants.NGHTTP2_CANCEL);
                expect(serverReceivedBytes).toBe(0);
            });
        });
        describe("over http/1.1", function () {
            let serverReceivedRequest = false;
            let serverReceivedBytes = 0;
            let serverRequestClosed = false;
            let serverResponseClosed = false;
            let serverRequestEmittedAborted = false;
            let serverRequestEmittedError;
            let serverRequestIterableErrored;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer((req, res) => {
                serverReceivedRequest = true;
                req.on("aborted", () => (serverRequestEmittedAborted = true));
                req.on("error", (e) => (serverRequestEmittedError = e));
                req.on("close", () => (serverRequestClosed = true));
                res.on("close", () => (serverResponseClosed = true));
                void (async () => {
                    var _a, e_8, _b, _c;
                    try {
                        try {
                            for (var _d = true, req_4 = __asyncValues(req), req_4_1; req_4_1 = await req_4.next(), _a = req_4_1.done, !_a; _d = true) {
                                _c = req_4_1.value;
                                _d = false;
                                const chunk = _c;
                                serverReceivedBytes += chunk.byteLength;
                            }
                        }
                        catch (e_8_1) { e_8 = { error: e_8_1 }; }
                        finally {
                            try {
                                if (!_d && !_a && (_b = req_4.return)) await _b.call(req_4);
                            }
                            finally { if (e_8) throw e_8.error; }
                        }
                    }
                    catch (e) {
                        serverRequestIterableErrored = e;
                    }
                })();
            }));
            it("should raise error with code canceled", async function () {
                // set up a client that aborts while still streaming the request body
                const client = server.getClient();
                const ac = new AbortController();
                function body() {
                    return __asyncGenerator(this, arguments, function* body_4() {
                        yield __await(new Promise((resolve) => setTimeout(resolve, 50)));
                        ac.abort();
                        yield yield __await(new Uint8Array(32));
                    });
                }
                // client should raise error
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        body: body(),
                        signal: ac.signal,
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // server should receive chunks until client cancelled
                expect(serverReceivedRequest)
                    .withContext("serverReceivedRequest")
                    .toBeTrue();
                while (!serverResponseClosed) {
                    // wait for the server to see the response being closed
                    await new Promise((resolve) => setTimeout(resolve, 1));
                }
                expect(serverRequestClosed).toBeTrue();
                expect(serverResponseClosed).toBeTrue();
                expect(serverRequestEmittedAborted).toBeTrue();
                expect(serverRequestEmittedError === null || serverRequestEmittedError === void 0 ? void 0 : serverRequestEmittedError.code).toBe("ECONNRESET");
                expect(serverRequestIterableErrored === null || serverRequestIterableErrored === void 0 ? void 0 : serverRequestIterableErrored.code).toBe("ECONNRESET");
                expect(serverReceivedBytes).toBe(0);
            });
        });
    });
    describe("with a signal aborting mid request", function () {
        describe("over http/2", function () {
            let serverReceivedRstCode;
            let serverReceivedBytes = 0;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer((req, res) => {
                res.stream.on("close", () => (serverReceivedRstCode = res.stream.rstCode));
                void (async () => {
                    var _a, e_9, _b, _c;
                    try {
                        for (var _d = true, req_5 = __asyncValues(req), req_5_1; req_5_1 = await req_5.next(), _a = req_5_1.done, !_a; _d = true) {
                            _c = req_5_1.value;
                            _d = false;
                            const chunk = _c;
                            serverReceivedBytes += chunk.byteLength;
                        }
                    }
                    catch (e_9_1) { e_9 = { error: e_9_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = req_5.return)) await _b.call(req_5);
                        }
                        finally { if (e_9) throw e_9.error; }
                    }
                })();
            }));
            it("should raise error with code canceled and send RST_STREAM with code CANCEL", async function () {
                // set up a client that aborts while still streaming the request body
                const client = server.getClient();
                const ac = new AbortController();
                function body() {
                    return __asyncGenerator(this, arguments, function* body_5() {
                        yield yield __await(new Uint8Array(32));
                        yield __await(new Promise((resolve) => setTimeout(resolve, 50)));
                        ac.abort();
                        yield yield __await(new Uint8Array(32));
                    });
                }
                // client should raise error
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        body: body(),
                        signal: ac.signal,
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // server should receive chunks until client cancelled
                while (serverReceivedRstCode === undefined) {
                    // wait for the server to see the reset code
                    await new Promise((resolve) => setTimeout(resolve, 1));
                }
                expect(serverReceivedRstCode).toBe(http2.constants.NGHTTP2_CANCEL);
                expect(serverReceivedBytes).toBe(32);
            });
        });
        describe("over http/1.1", function () {
            let serverReceivedRequest = false;
            let serverReceivedBytes = 0;
            let serverRequestClosed = false;
            let serverResponseClosed = false;
            let serverRequestEmittedAborted = false;
            let serverRequestEmittedError;
            let serverRequestIterableErrored;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer((req, res) => {
                serverReceivedRequest = true;
                req.on("aborted", () => (serverRequestEmittedAborted = true));
                req.on("error", (e) => (serverRequestEmittedError = e));
                req.on("close", () => (serverRequestClosed = true));
                res.on("close", () => (serverResponseClosed = true));
                void (async () => {
                    var _a, e_10, _b, _c;
                    try {
                        try {
                            for (var _d = true, req_6 = __asyncValues(req), req_6_1; req_6_1 = await req_6.next(), _a = req_6_1.done, !_a; _d = true) {
                                _c = req_6_1.value;
                                _d = false;
                                const chunk = _c;
                                serverReceivedBytes += chunk.byteLength;
                            }
                        }
                        catch (e_10_1) { e_10 = { error: e_10_1 }; }
                        finally {
                            try {
                                if (!_d && !_a && (_b = req_6.return)) await _b.call(req_6);
                            }
                            finally { if (e_10) throw e_10.error; }
                        }
                    }
                    catch (e) {
                        serverRequestIterableErrored = e;
                    }
                })();
            }));
            it("should raise error with code canceled", async function () {
                // set up a client that aborts while still streaming the request body
                const client = server.getClient();
                const ac = new AbortController();
                function body() {
                    return __asyncGenerator(this, arguments, function* body_6() {
                        yield yield __await(new Uint8Array(32));
                        yield __await(new Promise((resolve) => setTimeout(resolve, 50)));
                        ac.abort();
                        yield yield __await(new Uint8Array(32));
                    });
                }
                // client should raise error
                try {
                    await client({
                        url: server.getUrl(),
                        method: "POST",
                        header: new Headers(),
                        body: body(),
                        signal: ac.signal,
                    });
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // server should receive chunks until client cancelled
                expect(serverReceivedRequest).toBeTrue();
                while (!serverResponseClosed) {
                    // wait for the server to see the response being closed
                    await new Promise((resolve) => setTimeout(resolve, 1));
                }
                expect(serverRequestClosed).toBeTrue();
                expect(serverResponseClosed).toBeTrue();
                expect(serverRequestEmittedAborted).toBeTrue();
                expect(serverRequestEmittedError === null || serverRequestEmittedError === void 0 ? void 0 : serverRequestEmittedError.code).toBe("ECONNRESET");
                expect(serverRequestIterableErrored === null || serverRequestIterableErrored === void 0 ? void 0 : serverRequestIterableErrored.code).toBe("ECONNRESET");
                expect(serverReceivedBytes).toBe(32);
            });
        });
    });
    describe("with a signal aborting mid response", function () {
        describe("over http/2", function () {
            let serverReceivedRstCode;
            let serverSentBytes = 0;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http2.createServer((req, res) => {
                res.stream.on("close", () => (serverReceivedRstCode = res.stream.rstCode));
                void (async () => {
                    res.writeHead(200);
                    res.write(new Uint8Array(64));
                    serverSentBytes += 64;
                    await new Promise(() => {
                        // never resolves
                    });
                })();
            }));
            it("should raise error with code canceled and send RST_STREAM with code CANCEL", async function () {
                var _a, e_11, _b, _c;
                // set up a client that aborts while still streaming the request body
                const client = server.getClient();
                const ac = new AbortController();
                const res = await client({
                    url: server.getUrl(),
                    method: "POST",
                    header: new Headers(),
                    body: (0, protocol_1.createAsyncIterable)([]),
                    signal: ac.signal,
                });
                // should raise error with code canceled
                try {
                    try {
                        for (var _d = true, _e = __asyncValues(res.body), _f; _f = await _e.next(), _a = _f.done, !_a; _d = true) {
                            _c = _f.value;
                            _d = false;
                            const chunk = _c;
                            expect(chunk.byteLength).toBe(64);
                            ac.abort();
                        }
                    }
                    catch (e_11_1) { e_11 = { error: e_11_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = _e.return)) await _b.call(_e);
                        }
                        finally { if (e_11) throw e_11.error; }
                    }
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // server should receive RST_STREAM with code CANCEL
                while (serverReceivedRstCode === undefined) {
                    // wait for the server to see the reset code
                    await new Promise((resolve) => setTimeout(resolve, 1));
                }
                expect(serverReceivedRstCode).toBe(http2.constants.NGHTTP2_CANCEL);
                expect(serverSentBytes).toBe(64);
            });
        });
        describe("over http/1.1", function () {
            let serverSentBytes = 0;
            let serverRequestClosed = false;
            let serverResponseClosed = false;
            const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => http.createServer((req, res) => {
                req.on("close", () => (serverRequestClosed = true));
                res.on("close", () => (serverResponseClosed = true));
                void (async () => {
                    res.writeHead(200);
                    res.write(new Uint8Array(64));
                    serverSentBytes += 64;
                    await new Promise(() => {
                        // never resolves
                    });
                })();
            }));
            it("should raise error with code canceled", async function () {
                var _a, e_12, _b, _c;
                // set up a client that aborts while still streaming the request body
                const client = server.getClient();
                const ac = new AbortController();
                const res = await client({
                    url: server.getUrl(),
                    method: "POST",
                    header: new Headers(),
                    body: (0, protocol_1.createAsyncIterable)([]),
                    signal: ac.signal,
                });
                // should raise error with code canceled
                try {
                    try {
                        for (var _d = true, _e = __asyncValues(res.body), _f; _f = await _e.next(), _a = _f.done, !_a; _d = true) {
                            _c = _f.value;
                            _d = false;
                            const chunk = _c;
                            expect(chunk.byteLength).toBe(64);
                            ac.abort();
                        }
                    }
                    catch (e_12_1) { e_12 = { error: e_12_1 }; }
                    finally {
                        try {
                            if (!_d && !_a && (_b = _e.return)) await _b.call(_e);
                        }
                        finally { if (e_12) throw e_12.error; }
                    }
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(connect_1.ConnectError);
                    expect(connect_1.ConnectError.from(e).message).toBe("[canceled] This operation was aborted");
                }
                // server should see request close
                while (!serverResponseClosed) {
                    // wait for the server to see the response being closed
                    await new Promise((resolve) => setTimeout(resolve, 1));
                }
                expect(serverRequestClosed).toBeTrue();
                expect(serverResponseClosed).toBeTrue();
                expect(serverSentBytes).toBe(64);
            });
        });
    });
});
