"use strict";
// Copyright 2021-2024 The Connect Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
const use_node_server_helper_spec_js_1 = require("./use-node-server-helper.spec.js");
const http2 = require("http2");
const http = require("http");
const node_universal_handler_js_1 = require("./node-universal-handler.js");
const connect_1 = require("@connectrpc/connect");
const node_error_js_1 = require("./node-error.js");
const protocol_1 = require("@connectrpc/connect/protocol");
// Polyfill the Headers API for Node versions < 18
require("./node-headers-polyfill.js");
describe("universalRequestFromNodeRequest()", function () {
    describe("with HTTP/2 stream closed with an RST code", function () {
        let serverRequest;
        const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => {
            serverRequest = undefined;
            return http2.createServer(function (request) {
                serverRequest = (0, node_universal_handler_js_1.universalRequestFromNodeRequest)(request, undefined, undefined);
            });
        });
        async function request(rstCode) {
            await new Promise((resolve) => {
                http2.connect(server.getUrl(), (session) => {
                    const stream = session.request({
                        ":method": "POST",
                        ":path": "/",
                    }, {});
                    setTimeout(() => {
                        stream.on("error", () => {
                            // Closing with _some_ codes raises an ERR_HTTP2_STREAM_ERROR
                            // error here.
                        });
                        stream.close(rstCode, () => {
                            // We are seeing a race condition in Node v16.20.0, where closing
                            // the session right after closing a stream with an RST code
                            // _sometimes_ sends an INTERNAL_ERROR code.
                            // Simply delaying the session close until the next tick like
                            // we do here seems to work around the issue.
                            setTimeout(() => session.close(resolve), 0);
                        });
                    }, 0);
                });
            });
        }
        it("should abort request signal with ConnectError and Code.Canceled for NO_ERROR", async function () {
            await request(http2.constants.NGHTTP2_NO_ERROR);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal).toBeInstanceOf(AbortSignal);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.aborted).toBeTrue();
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason).toBeInstanceOf(Error);
            if ((serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason) instanceof Error) {
                expect(serverRequest.signal.reason.name).toBe("AbortError");
                expect(serverRequest.signal.reason.message).toBe("This operation was aborted");
            }
        });
        it("should silently end request body stream for NO_ERROR", async function () {
            await request(http2.constants.NGHTTP2_NO_ERROR);
            expect(serverRequest).toBeDefined();
            if (serverRequest !== undefined) {
                (0, protocol_1.assertByteStreamRequest)(serverRequest);
                const it = serverRequest.body[Symbol.asyncIterator]();
                const r = await it.next();
                expect(r.done).toBeTrue();
            }
        });
        it("should abort request signal with ConnectError and Code.Canceled for CANCEL", async function () {
            await request(http2.constants.NGHTTP2_CANCEL);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal).toBeInstanceOf(AbortSignal);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.aborted).toBeTrue();
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason).toBeInstanceOf(connect_1.ConnectError);
            const ce = connect_1.ConnectError.from(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason);
            expect(ce.message).toBe("[canceled] http/2 stream closed with error code CANCEL (0x8)");
        });
        it("should silently end request body stream for CANCEL", async function () {
            await request(http2.constants.NGHTTP2_CANCEL);
            expect(serverRequest).toBeDefined();
            if (serverRequest !== undefined) {
                (0, protocol_1.assertByteStreamRequest)(serverRequest);
                const it = serverRequest.body[Symbol.asyncIterator]();
                const r = await it.next();
                expect(r.done).toBeTrue();
            }
        });
        it("should abort request signal with ConnectError and Code.ResourceExhausted for ENHANCE_YOUR_CALM", async function () {
            await request(http2.constants.NGHTTP2_ENHANCE_YOUR_CALM);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal).toBeInstanceOf(AbortSignal);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.aborted).toBeTrue();
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason).toBeInstanceOf(connect_1.ConnectError);
            const ce = connect_1.ConnectError.from(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason);
            expect(ce.message).toBe("[resource_exhausted] http/2 stream closed with error code ENHANCE_YOUR_CALM (0xb)");
        });
        it("should silently end request body stream for ENHANCE_YOUR_CALM", async function () {
            await request(http2.constants.NGHTTP2_ENHANCE_YOUR_CALM);
            expect(serverRequest).toBeDefined();
            if (serverRequest !== undefined) {
                (0, protocol_1.assertByteStreamRequest)(serverRequest);
                const it = serverRequest.body[Symbol.asyncIterator]();
                const r = await it.next();
                expect(r.done).toBeTrue();
            }
        });
        it("should abort request signal with ConnectError and Code.Internal for FRAME_SIZE_ERROR", async function () {
            await request(http2.constants.NGHTTP2_FRAME_SIZE_ERROR);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal).toBeInstanceOf(AbortSignal);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.aborted).toBeTrue();
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason).toBeInstanceOf(connect_1.ConnectError);
            const ce = connect_1.ConnectError.from(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason);
            expect(ce.message).toBe("[internal] http/2 stream closed with error code FRAME_SIZE_ERROR (0x6)");
        });
        it("should silently end request body stream for FRAME_SIZE_ERROR", async function () {
            await request(http2.constants.NGHTTP2_FRAME_SIZE_ERROR);
            expect(serverRequest).toBeDefined();
            if (serverRequest !== undefined) {
                (0, protocol_1.assertByteStreamRequest)(serverRequest);
                const it = serverRequest.body[Symbol.asyncIterator]();
                const r = await it.next();
                expect(r.done).toBeTrue();
            }
        });
    });
    describe("with HTTP/1.1 ECONNRESET", function () {
        let serverRequest;
        const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => {
            serverRequest = undefined;
            const s = http.createServer({
                // We want to test behavior when a connection is dropped, and we do
                // not want to wait for the default check interval of 30 seconds,
                // because it would make our test suite idle for 30 seconds, so we
                // set it to a very low interval.
                //
                // Node 18 also enabled two timeouts by default: headersTimeout and
                // requestTimeout, which are 60s and 5 mins respectively.
                // https://github.com/nodejs/node/pull/41263
                //
                // However, this change seems to be buggy:
                // https://github.com/nodejs/node/issues/44228
                // https://github.com/b3nsn0w/koa-easy-ws/issues/36
                //
                // And coupled with our low check interval, it seems to be causing
                // intermittent timeouts in our test server. So, we are disabling
                // them by default.
                //
                connectionsCheckingInterval: 1,
                requestTimeout: 0,
            }, function (request) {
                serverRequest = (0, node_universal_handler_js_1.universalRequestFromNodeRequest)(request, undefined, undefined);
            });
            // For some reason, the type definitions for ServerOptions do not include
            // headersTimeout:
            // https://github.com/DefinitelyTyped/DefinitelyTyped/blob/master/types/node/http.d.ts
            // So, it must be added to the server property after construction.
            s.headersTimeout = 0;
            return s;
        });
        async function request() {
            return new Promise((resolve) => {
                const request = http.request(server.getUrl(), {
                    method: "POST",
                });
                request.on("error", () => {
                    // we need this event lister so that Node.js does not raise the error
                    // we trigger by calling destroy()
                });
                request.flushHeaders();
                setTimeout(() => {
                    request.destroy();
                    resolve();
                }, 20);
            });
        }
        it("should abort request signal with ConnectError and Code.Aborted", async function () {
            await request();
            while ((serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason) === undefined) {
                await new Promise((r) => setTimeout(r, 1));
            }
            expect(serverRequest.signal.reason).toBeInstanceOf(Error);
            if (serverRequest.signal.reason instanceof Error) {
                expect(serverRequest.signal.reason).toBeInstanceOf(connect_1.ConnectError);
                const ce = connect_1.ConnectError.from(serverRequest.signal.reason);
                expect(ce.message).toBe("[aborted] aborted");
            }
        });
        it("should error request body stream with ECONNRESET", async function () {
            await request();
            expect(serverRequest).toBeDefined();
            if (serverRequest !== undefined) {
                (0, protocol_1.assertByteStreamRequest)(serverRequest);
                const it = serverRequest.body[Symbol.asyncIterator]();
                try {
                    await it.next();
                    fail("expected error");
                }
                catch (e) {
                    expect(e).toBeInstanceOf(Error);
                    expect(e).not.toBeInstanceOf(connect_1.ConnectError);
                    if (e instanceof Error) {
                        expect(e.message).toBe("aborted");
                        expect((0, node_error_js_1.getNodeErrorProps)(e)).toEqual({
                            code: "ECONNRESET",
                        });
                    }
                }
            }
        });
    });
    describe("with HTTP/1.1 request finishing without error", function () {
        let serverRequest;
        const server = (0, use_node_server_helper_spec_js_1.useNodeServer)(() => {
            const s = http.createServer({
                // We want to test behavior when a connection is dropped, and we do
                // not want to wait for the default check interval of 30 seconds,
                // because it would make our test suite idle for 30 seconds, so we
                // set it to a very low interval.
                //
                // Node 18 also enabled two timeouts by default: headersTimeout and
                // requestTimeout, which are 60s and 5 mins respectively.
                // https://github.com/nodejs/node/pull/41263
                //
                // However, this change seems to be buggy:
                // https://github.com/nodejs/node/issues/44228
                // https://github.com/b3nsn0w/koa-easy-ws/issues/36
                //
                // And coupled with our low check interval, it seems to be causing
                // intermittent timeouts in our test server. So, we are disabling
                // them by default.
                //
                connectionsCheckingInterval: 1,
                requestTimeout: 0,
            }, function (request, response) {
                serverRequest = (0, node_universal_handler_js_1.universalRequestFromNodeRequest)(request, undefined, undefined);
                response.writeHead(200);
                response.end();
            });
            // For some reason, the type definitions for ServerOptions do not include
            // headersTimeout:
            // https://github.com/DefinitelyTyped/DefinitelyTyped/blob/master/types/node/http.d.ts
            // So, it must be added to the server property after construction.
            s.headersTimeout = 0;
            return s;
        });
        async function request() {
            return new Promise((resolve) => {
                const request = http.request(server.getUrl(), {
                    method: "POST",
                    // close TCP connection after we're done so that the server shuts down cleanly
                    agent: new http.Agent({ keepAlive: false }),
                });
                request.flushHeaders();
                request.end();
                request.on("response", (response) => {
                    void (0, protocol_1.readAllBytes)(response, Number.MAX_SAFE_INTEGER).then(() => resolve());
                });
            });
        }
        it("should abort request signal with AbortError", async function () {
            await request();
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal).toBeInstanceOf(AbortSignal);
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.aborted).toBeTrue();
            expect(serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason).toBeInstanceOf(Error);
            if ((serverRequest === null || serverRequest === void 0 ? void 0 : serverRequest.signal.reason) instanceof Error) {
                expect(serverRequest.signal.reason.name).toBe("AbortError");
                expect(serverRequest.signal.reason.message).toBe("This operation was aborted");
            }
        });
        it("should silently end request body stream", async function () {
            await request();
            expect(serverRequest).toBeDefined();
            if (serverRequest !== undefined) {
                (0, protocol_1.assertByteStreamRequest)(serverRequest);
                const it = serverRequest.body[Symbol.asyncIterator]();
                const r = await it.next();
                expect(r.done).toBeTrue();
            }
        });
    });
});
