import { ReactiveEffect, isReactive, isRef, isProxy, ReactiveFlags, toRaw, effectScope, reactive, readonly } from '@vue/reactivity';
export * from '@vue/reactivity';
import { isPromise, isObject, isArray, isSet, isMap, isPlainObject, isFunction } from '@vue/shared';
import { isValidElement, useRef, useCallback, useMemo, useEffect, useState } from 'react';
import { useSyncExternalStore } from 'use-sync-external-store/shim/index.js';

/**
 * @internal
 */
const isServer = typeof window === "undefined";
/**
 * @internal
 */
var InternalNameSpace;
(function (InternalNameSpace) {
    InternalNameSpace["$$__ignore__$$"] = "$$__ignore__$$";
    InternalNameSpace["$$__persist__$$"] = "$$__persist__$$";
    InternalNameSpace["$$__subscribe__$$"] = "$$__subscribe__$$";
    InternalNameSpace["$$__redux_dev_tool__$$"] = "$$__redux_dev_tool__$$";
})(InternalNameSpace || (InternalNameSpace = {}));

const jobs = new Set();
const MAX_UPDATE = 20;
let process$1 = false;
let updateCount = 0;
const flushQueue = () => {
    const all = [...jobs.values()].slice(0);
    jobs.clear();
    for (const job of all) {
        job.notify();
    }
    process$1 = false;
    // fix miss update
    if (jobs.size) {
        updateCount++;
        if (updateCount > MAX_UPDATE) {
            throw new Error(`[reactivity-store] have a infinity update for current store, pendingJobs: ${new Set(jobs)}`);
        }
        flushQueue();
    }
};
/**
 * @internal
 */
const queueJob = (job) => {
    jobs.add(job);
    if (process$1)
        return;
    process$1 = true;
    updateCount = 0;
    Promise.resolve().then(flushQueue);
};

let currentController = null;
function getCurrentController() {
    return currentController;
}
class ControllerEffect extends ReactiveEffect {
    get _isControllerEffect() {
        return true;
    }
    constructor(getter) {
        super(getter);
        if (process.env.NODE_ENV === "development") {
            this._devVersion = "3.5.16";
        }
    }
}
const catchError = (cb, instance) => {
    return () => {
        if (!instance._isActive)
            return;
        if (process.env.NODE_ENV === "development") {
            instance._devRunCount = instance._devRunCount || 0;
            instance._devRunCount++;
        }
        try {
            const res = cb();
            if (isPromise(res)) {
                throw new Error(`[reactivity-store] selector should be a pure function, but current is a async function`);
            }
            return res;
        }
        catch (e) {
            if (process.env.NODE_ENV === "development") {
                console.error(`[reactivity-store] have an error for current selector, ${e === null || e === void 0 ? void 0 : e.message}, maybe you use the middleware with wrong usage, %o`, instance);
            }
            return null;
        }
    };
};
class Controller {
    constructor(_getState, _compare, _lifeCycle, _list, _namespace, _onUpdate) {
        this._getState = _getState;
        this._compare = _compare;
        this._lifeCycle = _lifeCycle;
        this._namespace = _namespace;
        this._onUpdate = _onUpdate;
        this._listeners = new Set();
        // make the state change and component update
        this._updateCount = 0;
        this._isActive = true;
        this.notify = () => {
            var _a;
            if (!this._isActive)
                return;
            // TODO implement server side initialState
            if (process.env.NODE_ENV === "development" && isServer) {
                console.error(`[reactivity-store] unexpected update for reactivity-store, should not update a state on the server`);
            }
            this._updateCount++;
            try {
                (_a = this._onUpdate) === null || _a === void 0 ? void 0 : _a.call(this);
            }
            catch (e) {
                if (process.env.NODE_ENV === "development") {
                    console.error(`[reactivity-store] have an error for current updater, ${e === null || e === void 0 ? void 0 : e.message}, please check your subscribe, %o`, this);
                }
                this._lifeCycle.canUpdateComponent = false;
            }
            this._listeners.forEach((f) => f());
        };
        this._scheduler = () => {
            const p = getCurrentController();
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            currentController = this;
            const newState = this._effect.run();
            currentController = p;
            if (!this._isActive)
                return;
            const isSame = this._compare(this._state, newState);
            this._state = newState;
            if (!isSame) {
                if (this._lifeCycle.canUpdateComponent) {
                    if (this._lifeCycle.syncUpdateComponent) {
                        this.notify();
                    }
                    else {
                        queueJob(this);
                    }
                }
            }
        };
        this.subscribe = (listener) => {
            this._listeners.add(listener);
            return () => this._listeners.delete(listener);
        };
        this.getState = () => {
            return this._updateCount;
        };
        this.getEffect = () => {
            return this._effect;
        };
        this.getSelectorState = () => {
            return this._getStateSafe();
        };
        this.getLifeCycle = () => {
            return this._lifeCycle;
        };
        this._getStateSafe = catchError(_getState, this);
        this._effect = new ControllerEffect(this._getStateSafe);
        this._effect.scheduler = this._scheduler;
        if (this._namespace !== InternalNameSpace.$$__persist__$$ &&
            this._namespace !== InternalNameSpace.$$__subscribe__$$ &&
            this._namespace !== InternalNameSpace.$$__redux_dev_tool__$$) {
            this._list = _list;
            this._list.add(this);
        }
        if (process.env.NODE_ENV === "development") {
            this._devVersion = "0.3.11";
        }
    }
    // TODO move into constructor function?
    run() {
        const p = getCurrentController();
        // eslint-disable-next-line @typescript-eslint/no-this-alias
        currentController = this;
        this._state = this._effect.run();
        currentController = p;
    }
    stop() {
        var _a, _b;
        this._effect.stop();
        this._listeners.clear();
        (_b = (_a = this._list) === null || _a === void 0 ? void 0 : _a.delete) === null || _b === void 0 ? void 0 : _b.call(_a, this);
        this._isActive = false;
    }
    setActive(d) {
        this._isActive = d;
    }
}

/******************************************************************************
Copyright (c) Microsoft Corporation.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
***************************************************************************** */
/* global Reflect, Promise, SuppressedError, Symbol, Iterator */


function __rest(s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
}

typeof SuppressedError === "function" ? SuppressedError : function (error, suppressed, message) {
    var e = new Error(message);
    return e.name = "SuppressedError", e.error = error, e.suppressed = suppressed, e;
};

/**
 * @internal
 */
const createLifeCycle = () => ({
    onBeforeMount: [],
    onBeforeUpdate: [],
    onBeforeUnmount: [],
    onMounted: [],
    onUpdated: [],
    onUnmounted: [],
    hasHookInstall: false,
    canUpdateComponent: true,
    syncUpdateComponent: false,
});

/* eslint-disable @typescript-eslint/no-unused-expressions */
function _traverse(value, seen) {
    if (!isObject(value) || value[ReactiveFlags.SKIP] || isValidElement(value)) {
        return value;
    }
    seen = seen || new Set();
    if (seen.has(value)) {
        return value;
    }
    seen.add(value);
    if (isRef(value)) {
        traverse(value.value, seen);
    }
    else if (isArray(value)) {
        for (let i = 0; i < value.length; i++) {
            traverse(value[i], seen);
        }
    }
    else if (isSet(value) || isMap(value)) {
        value.forEach((v) => {
            traverse(v, seen);
        });
    }
    else if (isPlainObject(value)) {
        for (const key in value) {
            traverse(value[key], seen);
        }
    }
    return value;
}
function traverseShallow(value) {
    if (!isObject(value) || value[ReactiveFlags.SKIP] || isValidElement(value)) {
        return value;
    }
    if (isRef(value)) {
        value.value;
    }
    else if (isArray(value)) {
        for (let i = 0; i < value.length; i++) {
            value[i];
        }
    }
    else if (isSet(value) || isMap(value)) {
        value.forEach((v) => {
        });
    }
    else if (isPlainObject(value)) {
        for (const key in value) {
            value[key];
        }
    }
    return value;
}
/**
 * @internal
 */
function traverse(value, seen) {
    if (process.env.NODE_ENV === "development") {
        const start = Date.now();
        const re = _traverse(value, seen);
        const end = Date.now();
        if (end - start > 5) {
            console.warn(`[reactivity-store] 'traverse' current data: %o take a lot of time`, re);
        }
        return re;
    }
    else {
        return _traverse(value, seen);
    }
}
/**
 * @internal
 */
function checkHasReactive(value) {
    let hasReactive = false;
    function traverse(value, seen) {
        if (!isObject(value))
            return;
        if (hasReactive)
            return;
        if (isReactive(value) || isRef(value) || isProxy(value)) {
            hasReactive = true;
            return;
        }
        seen = seen || new Set();
        if (seen.has(value)) {
            return;
        }
        seen.add(value);
        if (isArray(value)) {
            for (let i = 0; i < value.length; i++) {
                traverse(value[i], seen);
            }
        }
        else if (isSet(value) || isMap(value)) {
            value.forEach((v) => {
                traverse(v, seen);
            });
        }
        else if (isPlainObject(value)) {
            for (const key in value) {
                traverse(value[key], seen);
            }
        }
        return;
    }
    traverse(value);
    return hasReactive;
}
/**
 * @internal
 */
function checkHasFunction(value) {
    let hasFunction = false;
    function traverse(value, seen) {
        if (hasFunction)
            return;
        if (isFunction(value)) {
            hasFunction = true;
            return;
        }
        seen = seen || new Set();
        if (seen.has(value)) {
            return;
        }
        seen.add(value);
        if (isArray(value)) {
            for (let i = 0; i < value.length; i++) {
                traverse(value[i], seen);
            }
        }
        else if (isSet(value) || isMap(value)) {
            value.forEach((v) => {
                traverse(v, seen);
            });
        }
        else if (isPlainObject(value)) {
            for (const key in value) {
                traverse(value[key], seen);
            }
        }
        return;
    }
    traverse(value);
    return hasFunction;
}
/**
 * @internal
 */
function checkHasSameField(source, target) {
    return Object.keys(source).filter((key) => key in target);
}
/**
 * @internal
 */
function checkHasMiddleware(value) {
    if (value && (value === null || value === void 0 ? void 0 : value["$$__state__$$"]) && (value === null || value === void 0 ? void 0 : value["$$__middleware__$$"])) {
        return true;
    }
}

const namespaceMap = {};
const temp$2 = new Set();
const defaultCompare$2 = () => false;
/**
 * @internal
 */
const setNamespaceMap = (key, value) => {
    namespaceMap[key] = value;
};
/**
 * @internal
 */
const delNamespace = (key) => {
    delete namespaceMap[key];
};
/**
 * @internal
 */
const checkHasKey = (key) => {
    return key in namespaceMap;
};
// cache state which has connect to devtool
const devToolMap = {};
const devController = {};
const globalName = "__reactivity-store-redux-devtools__";
let globalDevTools = null;
/**
 * @internal
 */
const getDevToolInstance = () => globalDevTools || window.__REDUX_DEVTOOLS_EXTENSION__.connect({ name: globalName });
/**
 * @internal
 */
const sendToDevTools = (action) => {
    const { getUpdatedState } = action, rest = __rest(action, ["getUpdatedState"]);
    try {
        const state = getUpdatedState();
        getDevToolInstance().send(rest, state);
    }
    catch (e) {
        console.log(e);
    }
};
/**
 * @internal
 */
const connectDevTool = (name, actions, readonlyState, reactiveState, options) => {
    if (window && window.__REDUX_DEVTOOLS_EXTENSION__ && typeof window.__REDUX_DEVTOOLS_EXTENSION__.connect === "function") {
        try {
            const devTools = globalDevTools || window.__REDUX_DEVTOOLS_EXTENSION__.connect({ name: globalName });
            globalDevTools = devTools;
            const existController = devController[name];
            if (existController) {
                existController.stop();
            }
            devToolMap[name] = readonlyState;
            const lifeCycle = createLifeCycle();
            lifeCycle.syncUpdateComponent = true;
            let updateInActionCount = 0;
            const onUpdateWithoutAction = () => {
                if (updateInActionCount > 0)
                    return;
                sendToDevTools({
                    type: `subscribeAction-${name}`,
                    getUpdatedState: () => (Object.assign(Object.assign({}, devToolMap), { [name]: readonlyState })),
                });
            };
            const subscribe = () => {
                let re = reactiveState;
                if ((options === null || options === void 0 ? void 0 : options.listener) && typeof (options === null || options === void 0 ? void 0 : options.listener) === "function") {
                    re = options.listener(reactiveState);
                }
                if (options === null || options === void 0 ? void 0 : options.shallow) {
                    traverseShallow(re);
                }
                else {
                    traverse(re);
                }
            };
            // create a subscribe controller to listen to the state change, because some state change may not trigger by the `action`
            const controller = new Controller(subscribe, defaultCompare$2, lifeCycle, temp$2, InternalNameSpace.$$__redux_dev_tool__$$, onUpdateWithoutAction);
            controller._devReduxOptions = options;
            devController[name] = controller;
            controller.run();
            const obj = Object.assign({}, devToolMap);
            devTools.init(obj);
            return Object.keys(actions).reduce((p, c) => {
                p[c] = (...args) => {
                    updateInActionCount++;
                    const len = actions[c].length || 0;
                    const re = actions[c](...args);
                    const action = actions[c];
                    if (isPromise(re)) {
                        re.finally(() => {
                            sendToDevTools({
                                type: `asyncAction-${name}/${action.name || "anonymous"}`,
                                $payload: args.slice(0, len),
                                getUpdatedState: () => (Object.assign(Object.assign({}, devToolMap), { [name]: readonlyState })),
                            });
                            updateInActionCount--;
                        });
                    }
                    else {
                        sendToDevTools({
                            type: `syncAction-${name}/${action.name || "anonymous"}`,
                            $payload: args.slice(0, len),
                            getUpdatedState: () => (Object.assign(Object.assign({}, devToolMap), { [name]: readonlyState })),
                        });
                        updateInActionCount--;
                    }
                    return re;
                };
                return p;
            }, {});
        }
        catch (e) {
            if (process.env.NODE_ENV === "development") {
                console.warn(`[reactivity-store] connect to redux devtools failed, please check the redux devtools extension`, e);
            }
            return actions;
        }
    }
    else {
        return actions;
    }
};

/* eslint-disable @typescript-eslint/no-unsafe-function-type */
/**
 * @internal
 */
const useCallbackRef = (callback) => {
    const callbackRef = useRef(callback);
    callbackRef.current = callback;
    const memoCallback = useCallback((...args) => {
        var _a;
        return (_a = callbackRef.current) === null || _a === void 0 ? void 0 : _a.call(null, ...args);
    }, []);
    return memoCallback;
};
/**
 * @internal
 */
const useSubscribeCallbackRef = (callback, deepSelector) => {
    const callbackRef = useRef();
    callbackRef.current = typeof callback === "function" ? callback : null;
    const memoCallback = useCallbackRef((arg) => {
        if (callbackRef.current) {
            const re = callbackRef.current(arg);
            if (deepSelector) {
                traverse(re);
            }
            else {
                // fix useState(s => s) not subscribe reactive state update
                traverseShallow(re);
            }
            return re;
        }
        else {
            // !BREAKING CHANGE, will change the default behavior when the deepSelector is true
            if (deepSelector) {
                traverse(arg);
            }
            else {
                traverseShallow(arg);
            }
            return arg;
        }
    });
    return memoCallback;
};
/**
 * @internal
 */
const usePrevValue = (v) => {
    const vRef = useRef(v);
    useEffect(() => {
        vRef.current = v;
    }, [v]);
    return vRef.current;
};
const createHook = (reactiveState, readonlyState, initialState, lifeCycle, deepSelector = true, stableSelector = false, stableCompare = true, namespace, actions = undefined) => {
    const controllerList = new Set();
    // TODO
    if (process.env.NODE_ENV === "development" && !isServer && namespace) {
        setNamespaceMap(namespace, initialState);
    }
    let active = true;
    namespace = namespace || InternalNameSpace.$$__ignore__$$;
    // tool function to generate `useSelector` hook
    const generateUseHook = (type) => {
        const currentIsDeep = type === "default" ? deepSelector : type === "deep" || type === "deep-stable";
        const currentIsStable = type === "default" ? stableSelector : type === "deep-stable" || type === "shallow-stable";
        function useReactiveHookWithSelector(selector, compare) {
            const ref = useRef();
            const selectorRef = useSubscribeCallbackRef(selector, currentIsDeep);
            const getSelected = useCallbackRef(() => {
                // 0.1.9
                // make the returned value as a readonly value, so the only way to change the state is in the `actions` middleware
                if (selector) {
                    ref.current = selector(Object.assign(Object.assign({}, readonlyState), actions));
                }
                else {
                    ref.current = Object.assign(Object.assign({}, readonlyState), actions);
                }
            });
            const memoCompare = useCallbackRef((p, n) => {
                if (compare && typeof compare === "function") {
                    return compare(p, n);
                }
                return false;
            });
            // may not work will with hmr
            const prevSelector = currentIsStable ? selector : usePrevValue(selector);
            const prevCompare = stableCompare ? compare : usePrevValue(compare);
            const ControllerInstance = useMemo(() => new Controller(() => selectorRef(reactiveState), memoCompare, lifeCycle, controllerList, namespace, getSelected), []);
            useSyncExternalStore(ControllerInstance.subscribe, ControllerInstance.getState, ControllerInstance.getState);
            // initial
            useMemo(() => {
                ControllerInstance.run();
                getSelected();
            }, [ControllerInstance, getSelected]);
            // !TODO try to improve the performance
            // rerun when the 'selector' change
            useMemo(() => {
                if (prevSelector !== selector) {
                    ControllerInstance.run();
                    getSelected();
                }
            }, [ControllerInstance, prevSelector, selector]);
            useMemo(() => {
                if (prevCompare !== compare) {
                    ControllerInstance.run();
                    getSelected();
                }
            }, [ControllerInstance, prevCompare, compare]);
            if (process.env.NODE_ENV === "development") {
                ControllerInstance._devSelector = selector;
                ControllerInstance._devCompare = compare;
                ControllerInstance._devActions = actions;
                ControllerInstance._devWithDeep = currentIsDeep;
                ControllerInstance._devWithStable = currentIsStable;
                ControllerInstance._devType = type;
                ControllerInstance._devState = initialState;
                ControllerInstance._devResult = ref.current;
            }
            useEffect(() => {
                ControllerInstance.setActive(true);
                return () => {
                    // fix React strictMode issue
                    if (process.env.NODE_ENV === "development") {
                        ControllerInstance.setActive(false);
                    }
                    else {
                        ControllerInstance.stop();
                    }
                };
            }, [ControllerInstance]);
            return ref.current;
        }
        return useReactiveHookWithSelector;
    };
    const defaultHook = generateUseHook("default");
    const deepHook = generateUseHook("deep");
    const deepStableHook = generateUseHook("deep-stable");
    const shallowHook = generateUseHook("shallow");
    const shallowStableHook = generateUseHook("shallow-stable");
    function useSelector(selector, compare) {
        return defaultHook(selector, compare);
    }
    const typedUseSelector = useSelector;
    typedUseSelector.getState = () => toRaw(initialState);
    typedUseSelector.getLifeCycle = () => lifeCycle;
    typedUseSelector.getActions = () => actions;
    typedUseSelector.getReactiveState = () => reactiveState;
    typedUseSelector.getReadonlyState = () => readonlyState;
    typedUseSelector.useDeepSelector = deepHook;
    typedUseSelector.useDeepStableSelector = deepStableHook;
    typedUseSelector.useShallowSelector = shallowHook;
    typedUseSelector.useShallowStableSelector = shallowStableHook;
    typedUseSelector.subscribe = (selector, cb, shallow) => {
        const subscribeSelector = () => {
            const re = selector(reactiveState);
            if (process.env.NODE_ENV === "development" && isPromise(re)) {
                console.error(`[reactivity-store/subscribe] selector should return a plain object, but current is a promise`);
            }
            if (shallow) {
                traverseShallow(re);
            }
            else {
                traverse(re);
            }
            return re;
        };
        const controller = new Controller(subscribeSelector, Object.is, lifeCycle, controllerList, InternalNameSpace.$$__subscribe__$$, () => cb());
        controller.run();
        return () => controller.stop();
    };
    typedUseSelector.getIsActive = () => active;
    typedUseSelector.clear = () => {
        controllerList.forEach((i) => i.stop());
        if (process.env.NODE_ENV === "development" && !isServer && namespace) {
            delNamespace(namespace);
        }
        active = false;
    };
    return typedUseSelector;
};

/* eslint-disable @typescript-eslint/no-unsafe-function-type */
/**
 * @internal
 */
const persistKey = "reactivity-store/persist-";
/**
 * @internal
 */
const debounce = (cb, time) => {
    let id = null;
    return ((...args) => {
        clearTimeout(id);
        id = setTimeout(() => cb.call(null, ...args), time);
    });
};
/**
 * @internal
 */
const getFinalState = (state) => {
    if (state["$$__state__$$"])
        return state["$$__state__$$"];
    return state;
};
/**
 * @internal
 */
const getFinalMiddleware = (state) => {
    if (state["$$__state__$$"])
        return (state["$$__middleware__$$"] || {});
    return {};
};
/**
 * @internal
 */
const getFinalActions = (state) => {
    if (state["$$__state__$$"])
        return (state["$$__actions__$$"] || {});
    return {};
};
/**
 * @internal
 */
const getFinalNamespace = (state) => {
    if (state["$$__state__$$"])
        return (state["$$__namespace__$$"] || {});
    return {};
};
/**
 * @internal
 */
const getFinalSelectorOptions = (state) => {
    if (state["$$__state__$$"])
        return (state["$$__selectorOptions__$$"] || {});
    return {};
};
// function for help to build external middleware
/**
 * @internal
 */
function createMiddleware(setup, options) {
    return () => {
        const state = setup();
        const initialState = getFinalState(state);
        const middleware = getFinalMiddleware(state);
        const actions = getFinalActions(state);
        const namespace = getFinalNamespace(state);
        const selectorOptions = getFinalSelectorOptions(state);
        if (process.env.NODE_ENV === "development" && middleware[options.name]) {
            console.warn(`[reactivity-store/middleware] you are using multiple of the '${options.name}' middleware, this is a unexpected usage`);
        }
        middleware[options.name] = true;
        return {
            ["$$__state__$$"]: toRaw(initialState),
            ["$$__actions__$$"]: actions,
            // field to check duplicate middleware
            ["$$__middleware__$$"]: middleware,
            ["$$__namespace__$$"]: namespace,
            ["$$__selectorOptions__$$"]: selectorOptions,
        };
    };
}

/**
 * @internal
 */
const _internalCreateStore = (creator, name = "createStore", lifeCycle) => {
    const state = creator();
    if (process.env.NODE_ENV === "development" && isPromise(state)) {
        console.error(`[reactivity-store] '${name}' expect receive a reactive object but got a promise %o, this is a unexpected usage. should not return a promise in this 'creator' function`, state);
    }
    if (process.env.NODE_ENV === "development" && !isObject(state)) {
        console.error(`[reactivity-store] '${name}' expect receive a reactive object but got a ${state}, this is a unexpected usage. should return a reactive object in this 'creator' function`);
    }
    if (process.env.NODE_ENV === "development" && checkHasMiddleware(state)) {
        console.error(`[reactivity-store] '${name}' not support middleware usage, please change to use 'createState'`);
    }
    if (process.env.NODE_ENV === "development" && !checkHasReactive(state)) {
        console.error(`[reactivity-store] '${name}' expect receive a reactive object but got a plain object %o, this is a unexpected usage. should return a reactive object in this 'creator' function`, state);
    }
    const _state = getFinalState(state);
    const rawState = toRaw(_state);
    const reactiveState = reactive(_state);
    const readonlyState = readonly(_state);
    const lifeCycleInstance = lifeCycle || createLifeCycle();
    const useSelector = createHook(reactiveState, readonlyState, rawState, lifeCycleInstance);
    return useSelector;
};
/**
 * @internal
 */
const internalCreateStore = (creator, name = "createStore", lifeCycle) => {
    const scope = effectScope();
    const useSelector = scope.run(() => _internalCreateStore(creator, name, lifeCycle));
    useSelector.scope = scope;
    return useSelector;
};
/**
 * @internal
 */
let globalStoreLifeCycle = null;
/**
 * @internal
 */
const setGlobalStoreLifeCycle = (instance) => {
    globalStoreLifeCycle = instance;
};

// vue like lifeCycle for react app;
/**
 * @public
 */
const onMounted = (cb) => {
    if (!globalStoreLifeCycle)
        throw new Error("[reactivity-store] can not use 'onMounted' outside of the 'createStoreWithComponent' function");
    globalStoreLifeCycle.onMounted.push(cb);
    globalStoreLifeCycle.hasHookInstall = true;
};
/**
 * @public
 */
const onUpdated = (cb) => {
    if (!globalStoreLifeCycle)
        throw new Error("[reactivity-store] can not use 'onUpdated' outside of the 'createStoreWithComponent' function");
    globalStoreLifeCycle.onUpdated.push(cb);
    globalStoreLifeCycle.hasHookInstall = true;
};
/**
 * @public
 */
const onUnmounted = (cb) => {
    if (!globalStoreLifeCycle)
        throw new Error("[reactivity-store] can not use 'onUnmounted' outside of the 'createStoreWithComponent' function");
    globalStoreLifeCycle.onUnmounted.push(cb);
    globalStoreLifeCycle.hasHookInstall = true;
};
/**
 * @public
 */
const onBeforeMount = (cb) => {
    if (!globalStoreLifeCycle)
        throw new Error("[reactivity-store] can not use 'onBeforeMount' outside of the 'createStoreWithComponent' function");
    globalStoreLifeCycle.onBeforeMount.push(cb);
    globalStoreLifeCycle.hasHookInstall = true;
};
/**
 * @public
 */
const onBeforeUpdate = (cb) => {
    if (!globalStoreLifeCycle)
        throw new Error("[reactivity-store] can not use 'onBeforeUpdate' outside of the 'createStoreWithComponent' function");
    globalStoreLifeCycle.onBeforeUpdate.push(cb);
    globalStoreLifeCycle.hasHookInstall = true;
};
/**
 * @public
 */
const onBeforeUnmount = (cb) => {
    if (!globalStoreLifeCycle)
        throw new Error("[reactivity-store] can not use 'onBeforeUnmount' outside of the 'createStoreWithComponent' function");
    globalStoreLifeCycle.onBeforeUnmount.push(cb);
    globalStoreLifeCycle.hasHookInstall = true;
};

/* eslint-disable @typescript-eslint/no-unsafe-function-type */
/**
 * @public
 */
function withActions(setup, options) {
    return createMiddleware(() => {
        const _initialState = setup();
        const initialState = getFinalState(_initialState);
        const middleware = getFinalMiddleware(_initialState);
        const actions = getFinalActions(_initialState);
        const namespace = getFinalNamespace(_initialState);
        const selectorOptions = getFinalSelectorOptions(_initialState);
        const reactiveState = reactive(initialState);
        const pendingGenerate = options.generateActions;
        const allActions = pendingGenerate === null || pendingGenerate === void 0 ? void 0 : pendingGenerate(reactiveState);
        const batchActions = allActions;
        // check duplicate key
        if (process.env.NODE_ENV === "development") {
            Object.keys(initialState).forEach((key) => {
                if (allActions[key]) {
                    console.error(`[reactivity-store/actions] there are duplicate key: [${key}] in the 'setup' and 'generateAction' returned value, this is a unexpected behavior.`);
                }
            });
            Object.keys(allActions).forEach((key) => {
                if (typeof allActions[key] !== "function") {
                    console.error(`[reactivity-store/actions] the value[${key}] return from 'generateActions' should be a function, but current is ${allActions[key]} in %o`, allActions);
                }
            });
            Object.keys(actions).forEach((key) => {
                if (allActions[key]) {
                    console.error(`[reactivity-store/actions] there are duplicate key: [${key}] in the 'action' return from 'withActions', this is a unexpected behavior.`);
                }
            });
        }
        return {
            ["$$__state__$$"]: toRaw(reactiveState),
            ["$$__actions__$$"]: Object.assign(Object.assign({}, actions), batchActions),
            ["$$__middleware__$$"]: middleware,
            ["$$__namespace__$$"]: namespace,
            ["$$__selectorOptions__$$"]: selectorOptions,
        };
    }, { name: "withActions" });
}

/* eslint-disable @typescript-eslint/no-unsafe-function-type */
const temp$1 = new Set();
const defaultCompare$1 = () => false;
/**
 * @public
 */
function withPersist(setup, options) {
    return createMiddleware(() => {
        var _a, _b, _c, _d;
        const _initialState = setup();
        const initialState = getFinalState(_initialState);
        const middleware = getFinalMiddleware(_initialState);
        const auctions = getFinalActions(_initialState);
        const namespace = getFinalNamespace(_initialState);
        const selectorOptions = getFinalSelectorOptions(_initialState);
        let hasSet = false;
        if (middleware["withPersist"])
            hasSet = true;
        if (process.env.NODE_ENV === "development" && checkHasReactive(initialState)) {
            console.error(`[reactivity-store/persist] the 'setup' which from 'withPersist' should return a plain object, but current is a reactive object %o, you may use 'reactiveApi' in the 'setup' function`, initialState);
        }
        if (!isServer && !hasSet) {
            let re = initialState;
            const storageKey = persistKey + options.key;
            let storage = null;
            try {
                storage = ((_a = options === null || options === void 0 ? void 0 : options.getStorage) === null || _a === void 0 ? void 0 : _a.call(options)) || (window === null || window === void 0 ? void 0 : window.localStorage);
                if (!storage) {
                    if (process.env.NODE_ENV === "development") {
                        console.error(`[reactivity-store/persist] can't find storage, please check your environment`);
                    }
                    return {
                        ["$$__state__$$"]: toRaw(initialState),
                        ["$$__middleware__$$"]: middleware,
                        ["$$__actions__$$"]: auctions,
                        ["$$__namespace__$$"]: namespace,
                        ["$$__selectorOptions__$$"]: selectorOptions,
                    };
                }
                const storageStateString = storage.getItem(storageKey);
                const storageState = JSON.parse(storageStateString);
                if ((storageState === null || storageState === void 0 ? void 0 : storageState.version) === (options.version || options.key) && storageState.data) {
                    const cachedState = ((_b = options === null || options === void 0 ? void 0 : options.parse) === null || _b === void 0 ? void 0 : _b.call(options, storageState.data)) || JSON.parse(storageState.data);
                    re = ((_c = options === null || options === void 0 ? void 0 : options.merge) === null || _c === void 0 ? void 0 : _c.call(options, initialState, cachedState)) || Object.assign(initialState, cachedState);
                }
            }
            catch (e) {
                if (process.env.NODE_ENV === "development") {
                    console.error(`[reactivity-store/persist] middleware failed, error: ${e.message}`);
                }
                try {
                    (_d = storage.removeItem) === null || _d === void 0 ? void 0 : _d.call(storage, storageKey);
                }
                catch (_e) {
                }
            }
            re = reactive(re);
            const onUpdate = debounce(() => {
                var _a, _b, _c;
                try {
                    const stringifyState = ((_a = options === null || options === void 0 ? void 0 : options.stringify) === null || _a === void 0 ? void 0 : _a.call(options, re)) || JSON.stringify(re);
                    const cache = { data: stringifyState, version: options.version || options.key };
                    if (process.env.NODE_ENV === "development" && options.devLog) {
                        console.log(`[reactivity-store/persist] state changed, try to cache newState: %o`, cache);
                    }
                    (_b = storage.setItem) === null || _b === void 0 ? void 0 : _b.call(storage, storageKey, JSON.stringify(cache));
                }
                catch (e) {
                    if (process.env.NODE_ENV === "development") {
                        console.error(`[reactivity-store/persist] cache newState error, error: %o`, e);
                    }
                    try {
                        (_c = storage.removeItem) === null || _c === void 0 ? void 0 : _c.call(storage, storageKey);
                    }
                    catch (_d) {
                    }
                }
            }, options.debounceTime || 40);
            const subscribe = () => {
                let _re = re;
                if (typeof options.listener === "function") {
                    _re = options.listener(re);
                }
                if (process.env.NODE_ENV === "development" && isPromise(_re)) {
                    console.error(`[reactivity-store/persist] listener should return a plain object, but current is a promise`);
                    return;
                }
                if (options.shallow) {
                    traverseShallow(_re);
                }
                else {
                    traverse(_re);
                }
            };
            const ControllerInstance = new Controller(subscribe, defaultCompare$1, createLifeCycle(), temp$1, InternalNameSpace.$$__persist__$$, onUpdate);
            ControllerInstance.run();
            if (process.env.NODE_ENV === "development") {
                ControllerInstance._devPersistOptions = options;
            }
            return {
                ["$$__state__$$"]: toRaw(re),
                ["$$__middleware__$$"]: middleware,
                ["$$__actions__$$"]: auctions,
                ["$$__namespace__$$"]: namespace,
                ["$$__selectorOptions__$$"]: selectorOptions,
            };
        }
        else {
            return {
                ["$$__state__$$"]: toRaw(initialState),
                ["$$__middleware__$$"]: middleware,
                ["$$__actions__$$"]: auctions,
                ["$$__namespace__$$"]: namespace,
                ["$$__selectorOptions__$$"]: selectorOptions,
            };
        }
    }, { name: "withPersist" });
}

/* eslint-disable @typescript-eslint/no-unsafe-function-type */
/**
 * @public
 */
function withNamespace(setup, options) {
    return createMiddleware(() => {
        const _initialState = setup();
        const initialState = getFinalState(_initialState);
        const middleware = getFinalMiddleware(_initialState);
        const actions = getFinalActions(_initialState);
        const namespace = getFinalNamespace(_initialState);
        const selectorOptions = getFinalSelectorOptions(_initialState);
        if (process.env.NODE_ENV === "development" &&
            (options.namespace === InternalNameSpace.$$__ignore__$$ ||
                options.namespace === InternalNameSpace.$$__persist__$$ ||
                options.namespace === InternalNameSpace.$$__subscribe__$$ ||
                options.namespace === InternalNameSpace.$$__redux_dev_tool__$$)) {
            console.warn(`[reactivity-store/namespace] current namespace: '${options.namespace}' is a internal namespace, try to use another one`);
        }
        if (process.env.NODE_ENV === "development" &&
            !isServer &&
            options.namespace !== InternalNameSpace.$$__ignore__$$ &&
            options.namespace !== InternalNameSpace.$$__persist__$$ &&
            options.namespace !== InternalNameSpace.$$__subscribe__$$ &&
            options.namespace !== InternalNameSpace.$$__redux_dev_tool__$$) {
            const alreadyHasNameSpace = checkHasKey(options.namespace);
            if (alreadyHasNameSpace) {
                console.warn(`[reactivity-store/middleware] you have duplicate namespace '${options.namespace}' for current store, this is a unexpected usage`);
            }
            // setNamespaceMap(options.namespace, initialState);
        }
        return {
            ["$$__state__$$"]: toRaw(initialState),
            ["$$__actions__$$"]: actions,
            ["$$__middleware__$$"]: middleware,
            ["$$__namespace__$$"]: Object.assign(Object.assign({}, namespace), options),
            ["$$__selectorOptions__$$"]: selectorOptions,
        };
    }, { name: "withNamespace" });
}

/* eslint-disable @typescript-eslint/no-unsafe-function-type */
/**
 * @public
 */
function withSelectorOptions(setup, options) {
    return createMiddleware(() => {
        const _initialState = setup();
        const initialState = getFinalState(_initialState);
        const middleware = getFinalMiddleware(_initialState);
        const actions = getFinalActions(_initialState);
        const namespace = getFinalNamespace(_initialState);
        const selectorOptions = getFinalSelectorOptions(_initialState);
        return {
            ["$$__state__$$"]: toRaw(initialState),
            ["$$__actions__$$"]: actions,
            ["$$__middleware__$$"]: middleware,
            ["$$__namespace__$$"]: namespace,
            ["$$__selectorOptions__$$"]: Object.assign(Object.assign({}, selectorOptions), options),
        };
    }, { name: "withSelectorOptions" });
}
/**
 * @public
 * @deprecated
 * use `withSelectorOptions` instead
 */
const withDeepSelector = withSelectorOptions;

/**
 * @public
 * @deprecated
 * not recommend to use this function, use `createStore` instead
 */
function createStoreWithComponent(props) {
    const { setup, render } = props;
    const ComponentWithState = (props) => {
        const useSelector = useMemo(() => {
            const lifeCycleInstance = createLifeCycle();
            setGlobalStoreLifeCycle(lifeCycleInstance);
            const useSelector = internalCreateStore(setup, "createStoreWithComponent", lifeCycleInstance);
            setGlobalStoreLifeCycle(null);
            return useSelector;
        }, []);
        const [isMount, setIsMount] = useState(false);
        const state = useSelector.getReadonlyState();
        if (process.env.NODE_ENV === "development") {
            const sameField = checkHasSameField(state, props);
            sameField.forEach((key) => console.warn(`[reactivity-store] duplicate key: [${key}] in Component props and RStore state, this is a unexpected usage`));
        }
        const lifeCycleInstance = useSelector.getLifeCycle();
        const { children } = props, last = __rest(props, ["children"]);
        const _targetRender = render || props.children;
        const targetRender = _targetRender !== null && _targetRender !== void 0 ? _targetRender : (() => {
            lifeCycleInstance.canUpdateComponent = false;
            if (process.env.NODE_ENV === "development") {
                console.warn(`[reactivity-store] current reactive component not have a render function`);
            }
        });
        // subscribe reactivity-store update
        useSelector();
        useEffect(() => {
            if (lifeCycleInstance.hasHookInstall) {
                if (!isMount) {
                    lifeCycleInstance.onBeforeMount.forEach((f) => f());
                    lifeCycleInstance.onMounted.forEach((f) => f());
                    setIsMount(true);
                }
                else {
                    const lastSync = lifeCycleInstance.syncUpdateComponent;
                    lifeCycleInstance.syncUpdateComponent = true;
                    lifeCycleInstance.canUpdateComponent = false;
                    lifeCycleInstance.onBeforeUpdate.forEach((f) => f());
                    lifeCycleInstance.canUpdateComponent = true;
                    lifeCycleInstance.syncUpdateComponent = lastSync;
                    lifeCycleInstance.onUpdated.forEach((f) => f());
                }
            }
        });
        useEffect(() => {
            return () => {
                if (lifeCycleInstance.hasHookInstall) {
                    lifeCycleInstance.onBeforeUnmount.forEach((f) => f());
                    lifeCycleInstance.onUnmounted.forEach((f) => f());
                }
            };
        }, [lifeCycleInstance]);
        useEffect(() => () => { var _a; return (_a = useSelector.scope) === null || _a === void 0 ? void 0 : _a.stop(); }, [useSelector]);
        const renderedChildren = targetRender(Object.assign(Object.assign({}, last), state)) || null;
        return renderedChildren;
    };
    return ComponentWithState;
}

/**
 * @public
 *
 * @example
 * ```typescript
 * import { createStore, ref } from "r-store";
 *
 * const count = createStore(() => {
 *  const state = ref(0);
 *
 *  const increment = () => {
 *    state.value++;
 *  };
 *
 *  return { state, increment };
 * });
 * ```
 */
const createStore = (creator) => {
    return internalCreateStore(creator);
};

/**
 * @internal
 */
// eslint-disable-next-line @typescript-eslint/no-unsafe-function-type
function internalCreateState(setup, name, option) {
    var _a, _b, _c;
    let creator = setup;
    if (option === null || option === void 0 ? void 0 : option.withPersist) {
        creator = withPersist(creator, typeof option.withPersist === "string" ? { key: option.withPersist } : option.withPersist);
    }
    if (option === null || option === void 0 ? void 0 : option.withActions) {
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore
        creator = withActions(creator, { generateActions: option.withActions });
    }
    if (option === null || option === void 0 ? void 0 : option.withNamespace) {
        creator = withNamespace(creator, typeof option.withNamespace === "string" ? { namespace: option.withNamespace, reduxDevTool: true } : option.withNamespace);
    }
    if (typeof (option === null || option === void 0 ? void 0 : option.withDeepSelector) !== "undefined" || typeof (option === null || option === void 0 ? void 0 : option.withStableSelector) !== "undefined") {
        creator = withSelectorOptions(creator, { deepSelector: option.withDeepSelector, stableSelector: option.withStableSelector });
    }
    const lifeCycle = createLifeCycle();
    const state = creator();
    if (process.env.NODE_ENV === "development" && !isObject(state)) {
        console.error(`[reactivity-store] '${name}' expect receive a plain object but got a ${state}, this is a unexpected usage. should return a plain object in this 'setup' function`);
    }
    // handle withActions middleware;
    const initialState = getFinalState(state);
    if (process.env.NODE_ENV === "development" && isPromise(initialState)) {
        console.error(`[reactivity-store] '${name}' expect receive a plain object but got a promise %o, this is a unexpected usage. should not return a promise in this 'setup' function`, initialState);
    }
    let actions = getFinalActions(state);
    const namespaceOptions = getFinalNamespace(state);
    const selectorOptions = getFinalSelectorOptions(state);
    const rawState = toRaw(initialState);
    const reduxDevTool = namespaceOptions.reduxDevTool && !isServer;
    if (process.env.NODE_ENV === "development" && checkHasReactive(rawState)) {
        console.error(`[reactivity-store] '${name}' expect receive a plain object but got a reactive object/field %o, this is a unexpected usage. should not use 'reactiveApi' in this 'setup' function`, rawState);
    }
    if (process.env.NODE_ENV === "development" && checkHasFunction(rawState)) {
        console.error(`[reactivity-store] '${name}' has a function field in state %o, this is a unexpected usage. state should be only a plain object with data field`, rawState);
    }
    if (process.env.NODE_ENV === "development") {
        const sameField = checkHasSameField(rawState, actions);
        sameField.forEach((key) => console.warn(`[reactivity-store] duplicate key: [${key}] in 'state' and 'actions' from createState, this is a unexpected usage`));
    }
    const reactiveState = reactive(initialState);
    const readonlyState = readonly(initialState);
    const deepSelector = (_a = selectorOptions === null || selectorOptions === void 0 ? void 0 : selectorOptions.deepSelector) !== null && _a !== void 0 ? _a : true;
    const stableSelector = (_b = selectorOptions === null || selectorOptions === void 0 ? void 0 : selectorOptions.stableSelector) !== null && _b !== void 0 ? _b : false;
    const stableCompare = (_c = selectorOptions.stableCompare) !== null && _c !== void 0 ? _c : true;
    // TODO
    if (process.env.NODE_ENV === "development" && reduxDevTool) {
        actions = connectDevTool(namespaceOptions.namespace, actions, readonlyState, reactiveState, namespaceOptions);
    }
    const useSelector = createHook(reactiveState, readonlyState, rawState, lifeCycle, deepSelector, stableSelector, stableCompare, namespaceOptions.namespace, actions);
    return useSelector;
}

/* eslint-disable @typescript-eslint/no-empty-object-type */
/* eslint-disable @typescript-eslint/no-unsafe-function-type */
// `createState` provider
/**
 * @public
 *
 * @example
 * ```typescript
 * import { createState } from "r-store";
 *
 * const count = createState(() => ({state: 0}), {
 *  withPersist: "count",
 *  withActions: (state) => ({
 *   increment: () => {
 *    state.state++;
 *   },
 *  }),
 *  withNamespace: "count",
 *  withDeepSelector: true,
 *  withStableSelector: true,
 * });
 * ```
 */
function createState(setup, options) {
    return internalCreateState(setup, "createState", options);
}

const defaultBatch = (cb) => cb();
/**
 * @internal
 */
const batchObject = { current: defaultBatch };
/**
 * @public
 * @deprecated
 * no need to use this function
 */
const setBatch = (batch) => {
    batchObject.current = batch;
};
/**
 * @public
 * @deprecated
 * no need to use this function
 */
const getBatch = () => {
    return batchObject.current;
};
/**
 * @public
 * @deprecated
 * no need to use this function
 */
const resetBatch = () => {
    batchObject.current = defaultBatch;
};
/**
 * @public
 * @deprecated
 * no need to use this function
 */
// eslint-disable-next-line @typescript-eslint/no-unsafe-function-type
const wrapperBatchUpdate = (cb) => {
    return ((...args) => batchObject.current(() => (args.length ? cb.call(null, ...args) : cb.call(null))));
};

const temp = new Set();
const defaultCompare = () => false;
/**
 * @public
 */
const useReactiveEffect = (effectCallback) => {
    const memoCallback = useCallbackRef(effectCallback);
    useEffect(() => {
        let cleanCb = () => void 0;
        const subscribe = () => {
            const clean = memoCallback();
            if (typeof clean === "function") {
                cleanCb = clean;
            }
            else {
                cleanCb = () => void 0;
            }
        };
        const controller = new Controller(subscribe, defaultCompare, createLifeCycle(), temp, InternalNameSpace.$$__subscribe__$$, () => {
            // run the effect when the subscribed state change
            cleanCb();
        });
        // run the effect on the component mount
        controller.run();
        return () => {
            cleanCb();
            controller.stop();
        };
    }, []);
};

/**
 * @public
 */
const useReactiveState = (initialState) => {
    const [useSelector] = useState(() => {
        const setup = typeof initialState === "function" ? initialState : () => initialState;
        return internalCreateState(setup, "useReactiveState");
    });
    // subscribe reactive store update
    useSelector();
    const setState = useMemo(() => (payload) => {
        if (typeof payload === "function") {
            payload(useSelector.getReactiveState());
        }
        else {
            const reactiveObj = useSelector.getReactiveState();
            Object.keys(payload).forEach((key) => {
                // eslint-disable-next-line @typescript-eslint/ban-ts-comment
                // @ts-expect-error
                reactiveObj[key] = payload[key];
            });
        }
    }, [useSelector]);
    useEffect(() => () => !(process.env.NODE_ENV === "development") && useSelector.clear(), [useSelector]);
    // make the state can be used in the `useReactiveEffect` hook
    // use getReactiveState to make effect can track deps
    return [useSelector.getReactiveState(), setState];
};

/**
 * @public
 */
const version = "0.3.11";

export { Controller, createState, createStore, createStoreWithComponent, getBatch, getCurrentController, onBeforeMount, onBeforeUnmount, onBeforeUpdate, onMounted, onUnmounted, onUpdated, resetBatch, setBatch, useReactiveEffect, useReactiveState, version, withActions, withDeepSelector, withNamespace, withPersist, withSelectorOptions, wrapperBatchUpdate };
