"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnleashMetricClient = exports.MetricsAPI = void 0;
const stream_1 = require("stream");
const unleash_1 = require("../unleash");
const environment_resolver_1 = require("./environment-resolver");
class MetricsAPI extends stream_1.EventEmitter {
    constructor(metricRegistry, variantResolver, staticContext) {
        super();
        this.metricRegistry = metricRegistry;
        this.variantResolver = variantResolver;
        this.staticContext = staticContext;
    }
    defineCounter(name, help) {
        if (!name || !help) {
            this.emit(unleash_1.UnleashEvents.Warn, `Counter name or help cannot be empty: ${name}, ${help}.`);
            return;
        }
        const labelNames = ['featureName', 'appName', 'environment'];
        this.metricRegistry.counter({ name, help, labelNames });
    }
    defineGauge(name, help) {
        if (!name || !help) {
            this.emit(unleash_1.UnleashEvents.Warn, `Gauge name or help cannot be empty: ${name}, ${help}.`);
            return;
        }
        const labelNames = ['featureName', 'appName', 'environment'];
        this.metricRegistry.gauge({ name, help, labelNames });
    }
    getFlagLabels(flagContext) {
        let flagLabels = {};
        if (flagContext) {
            for (const flag of flagContext.flagNames) {
                const variant = this.variantResolver.forceGetVariant(flag, flagContext.context);
                if (variant.enabled) {
                    flagLabels[flag] = variant.name;
                }
                else if (variant.feature_enabled) {
                    flagLabels[flag] = 'enabled';
                }
                else {
                    flagLabels[flag] = 'disabled';
                }
            }
        }
        return flagLabels;
    }
    incrementCounter(name, value, flagContext) {
        const counter = this.metricRegistry.getCounter(name);
        if (!counter) {
            this.emit(unleash_1.UnleashEvents.Warn, `Counter ${name} not defined, this counter will not be incremented.`);
            return;
        }
        const flagLabels = this.getFlagLabels(flagContext);
        const labels = {
            ...flagLabels,
            ...this.staticContext,
        };
        counter.inc(value, labels);
    }
    updateGauge(name, value, flagContext) {
        const gauge = this.metricRegistry.getGauge(name);
        if (!gauge) {
            this.emit(unleash_1.UnleashEvents.Warn, `Gauge ${name} not defined, this gauge will not be updated.`);
            return;
        }
        const flagLabels = this.getFlagLabels(flagContext);
        const labels = {
            ...flagLabels,
            ...this.staticContext,
        };
        gauge.set(value, labels);
    }
}
exports.MetricsAPI = MetricsAPI;
class UnleashMetricClient extends unleash_1.Unleash {
    constructor(...args) {
        super(...args);
        const config = args[0];
        const metricsContext = { ...this.staticContext };
        if (config && config.customHeaders) {
            const environment = (0, environment_resolver_1.extractEnvironmentFromCustomHeaders)(config.customHeaders);
            if (environment) {
                metricsContext.environment = environment;
            }
        }
        this.impactMetrics = new MetricsAPI(this.metricRegistry, this.client, metricsContext);
    }
}
exports.UnleashMetricClient = UnleashMetricClient;//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/c595276fa83d83a7c3233d582e4120f92017171c/node_modules/unleash-client/lib/impact-metrics/metric-client.js.map