"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InMemoryMetricRegistry = void 0;
function getLabelKey(labels) {
    if (!labels)
        return '';
    return Object.keys(labels)
        .sort()
        .map((k) => `${k}=${labels[k]}`)
        .join(',');
}
function parseLabelKey(key) {
    const labels = {};
    if (!key)
        return labels;
    for (const pair of key.split(',')) {
        const [k, v] = pair.split('=');
        labels[k] = v;
    }
    return labels;
}
class CounterImpl {
    constructor(opts) {
        this.opts = opts;
        this.values = new Map();
    }
    inc(value, labels) {
        var _a;
        const delta = value !== null && value !== void 0 ? value : 1;
        const key = getLabelKey(labels);
        const current = (_a = this.values.get(key)) !== null && _a !== void 0 ? _a : 0;
        this.values.set(key, current + delta);
    }
    collect() {
        const samples = [...this.values.entries()].map(([key, value]) => ({
            labels: parseLabelKey(key),
            value,
        }));
        this.values.clear();
        return {
            name: this.opts.name,
            help: this.opts.help,
            type: 'counter',
            samples,
        };
    }
}
class GaugeImpl {
    constructor(opts) {
        this.opts = opts;
        this.values = new Map();
    }
    inc(value, labels) {
        var _a;
        const delta = value !== null && value !== void 0 ? value : 1;
        const key = getLabelKey(labels);
        const current = (_a = this.values.get(key)) !== null && _a !== void 0 ? _a : 0;
        this.values.set(key, current + delta);
    }
    dec(value, labels) {
        var _a;
        const delta = value !== null && value !== void 0 ? value : 1;
        const key = getLabelKey(labels);
        const current = (_a = this.values.get(key)) !== null && _a !== void 0 ? _a : 0;
        this.values.set(key, current - delta);
    }
    set(value, labels) {
        const key = getLabelKey(labels);
        this.values.set(key, value);
    }
    collect() {
        const samples = [...this.values.entries()].map(([key, value]) => ({
            labels: parseLabelKey(key),
            value,
        }));
        this.values.clear();
        return {
            name: this.opts.name,
            help: this.opts.help,
            type: 'gauge',
            samples,
        };
    }
}
class InMemoryMetricRegistry {
    constructor() {
        this.counters = new Map();
        this.gauges = new Map();
    }
    getCounter(counterName) {
        return this.counters.get(counterName);
    }
    getGauge(gaugeName) {
        return this.gauges.get(gaugeName);
    }
    counter(opts) {
        const key = opts.name;
        if (!this.counters.has(key)) {
            this.counters.set(key, new CounterImpl(opts));
        }
        return this.counters.get(key);
    }
    gauge(opts) {
        const key = opts.name;
        if (!this.gauges.has(key)) {
            this.gauges.set(key, new GaugeImpl(opts));
        }
        return this.gauges.get(key);
    }
    collect() {
        const allCounters = [...this.counters.values()].map((c) => c.collect());
        const allGauges = [...this.gauges.values()].map((g) => g.collect());
        const allMetrics = [...allCounters, ...allGauges];
        const nonEmpty = allMetrics.filter((metric) => metric.samples.length > 0);
        return nonEmpty.length > 0 ? nonEmpty : [];
    }
    restore(metrics) {
        for (const metric of metrics) {
            switch (metric.type) {
                case 'counter': {
                    const counter = this.counter({ name: metric.name, help: metric.help });
                    for (const sample of metric.samples) {
                        counter.inc(sample.value, sample.labels);
                    }
                    break;
                }
                case 'gauge': {
                    const gauge = this.gauge({ name: metric.name, help: metric.help });
                    for (const sample of metric.samples) {
                        gauge.set(sample.value, sample.labels);
                    }
                    break;
                }
            }
        }
    }
}
exports.InMemoryMetricRegistry = InMemoryMetricRegistry;//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/c595276fa83d83a7c3233d582e4120f92017171c/node_modules/unleash-client/lib/impact-metrics/metric-types.js.map