"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamingFetcher = void 0;
const events_1 = require("events");
const feature_1 = require("../feature");
const request_1 = require("../request");
const url_utils_1 = require("../url-utils");
const events_2 = require("../events");
const event_source_1 = require("../event-source");
class StreamingFetcher extends events_1.EventEmitter {
    constructor(options) {
        super();
        this.options = options;
        this.eventSource = options.eventSource;
    }
    setupEventSource() {
        if (this.eventSource) {
            this.eventSource.addEventListener('unleash-connected', async (event) => {
                await this.handleFlagsFromStream(event);
            });
            this.eventSource.addEventListener('unleash-updated', this.handleFlagsFromStream.bind(this));
            this.eventSource.addEventListener('error', (error) => {
                this.emit(events_2.UnleashEvents.Warn, error);
            });
            this.eventSource.addEventListener('end', (error) => {
                this.emit(events_2.UnleashEvents.Warn, error);
            });
            this.eventSource.addEventListener('fetch-mode', this.handleModeChange.bind(this));
        }
    }
    async handleFlagsFromStream(event) {
        try {
            const data = (0, feature_1.parseClientFeaturesDelta)(JSON.parse(event.data));
            await this.options.onSaveDelta(data);
        }
        catch (err) {
            this.emit(events_2.UnleashEvents.Error, err);
        }
    }
    async handleModeChange(event) {
        try {
            const newMode = event.data;
            if (this.options.onModeChange) {
                await this.options.onModeChange(newMode);
            }
        }
        catch (err) {
            this.emit(events_2.UnleashEvents.Error, new Error(`Failed to handle mode change: ${err}`));
        }
    }
    createEventSource() {
        return new event_source_1.EventSource((0, url_utils_1.resolveUrl)(this.options.url, './client/streaming'), {
            headers: (0, request_1.buildHeaders)({
                appName: this.options.appName,
                instanceId: this.options.instanceId,
                etag: undefined,
                contentType: undefined,
                custom: this.options.headers,
                specVersionSupported: '5.2.0',
                connectionId: this.options.connectionId,
            }),
            readTimeoutMillis: 60000,
            initialRetryDelayMillis: 2000,
            maxBackoffMillis: 30000,
            retryResetIntervalMillis: 60000,
            jitterRatio: 0.5,
            errorFilter: function () {
                return true;
            },
        });
    }
    async start() {
        if (!this.eventSource) {
            this.eventSource = this.createEventSource();
        }
        this.setupEventSource();
    }
    stop() {
        if (this.eventSource) {
            this.eventSource.close();
            this.eventSource = undefined;
        }
    }
}
exports.StreamingFetcher = StreamingFetcher;//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/c595276fa83d83a7c3233d582e4120f92017171c/node_modules/unleash-client/lib/repository/streaming-fetcher.js.map