#include <shared/system.h>
#include <shared/debug.h>
#include <6502/6502.h>
#include <beeb/adc.h>
#include <beeb/Trace.h>

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

static constexpr uint16_t CONVERSION_TIME_USEC_10_BIT = 10000;
static constexpr uint16_t CONVERSION_TIME_USEC_8_BIT = 4000;
static constexpr uint16_t DEFAULT_CHANNEL_VALUE = 1023;

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

ADC::ADC() {
    m_status.bits.not_eoc = 1;
    m_status.bits.not_busy = 1;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void ADC::Write0(void *adc_, M6502Word, uint8_t value) {
    auto adc = (ADC *)adc_;

    adc->m_status.value = value;

    if (adc->m_status.bits.prec_10_bit) {
        adc->m_timer = CONVERSION_TIME_USEC_10_BIT;
    } else {
        adc->m_timer = CONVERSION_TIME_USEC_8_BIT;
    }

    adc->m_status.bits.msb = adc->m_dvalue.b.h >> 6;
    adc->m_status.bits.not_busy = 0;
    adc->m_status.bits.not_eoc = 1;

    if (adc->m_handler_fn) {
        adc->m_avalue = (*adc->m_handler_fn)(adc->m_status.bits.channel, adc->m_handler_context);
    } else {
        adc->m_avalue = DEFAULT_CHANNEL_VALUE;
    }
    ASSERT(adc->m_avalue < 1024);

    TRACEF(adc->m_trace, "ADC: Conversion begin: ch=%d (avalue=$%04x) prec=%s (timer=%u ms)", adc->m_status.bits.channel, adc->m_avalue, adc->m_status.bits.prec_10_bit ? "10" : "8", adc->m_timer);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void ADC::Write1(void *, M6502Word, uint8_t) {
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void ADC::Write2(void *, M6502Word, uint8_t) {
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void ADC::Write3(void *, M6502Word, uint8_t) {
    // This is supposed to activate some kind of test mode. I didn't have
    // success getting anything interesting out of this on a Master 128 though.
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

uint8_t ADC::Read0(void *adc_, M6502Word) {
    auto adc = (ADC *)adc_;

    return adc->m_status.value;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

uint8_t ADC::Read1(void *adc_, M6502Word) {
    auto adc = (ADC *)adc_;

    adc->m_status.bits.not_eoc = 1;

    return adc->m_dvalue.b.h;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

uint8_t ADC::Read2(void *adc_, M6502Word) {
    auto adc = (ADC *)adc_;

    return adc->m_dvalue.b.l;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

uint8_t ADC::Read3(void *adc, M6502Word addr) {
    return Read2(adc, addr);
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

bool ADC::Update() {
    if (m_timer > 0) {
        --m_timer;

        if (m_timer == 0) {
            m_dvalue.w = m_avalue << 6;
            m_status.bits.msb = m_dvalue.b.h >> 6;

            m_status.bits.not_busy = 1;
            m_status.bits.not_eoc = 0;

            TRACEF(m_trace, "ADC: Conversion complete: dvalue=$%04x", m_dvalue.w);
        }
    }

    return m_status.bits.not_eoc;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

#if BBCMICRO_TRACE
void ADC::SetTrace(Trace *t) {
    m_trace = t;
}
#endif

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////

void ADC::SetHandler(HandlerFn fn, void *context) {
    m_handler_fn = fn;
    m_handler_context = context;
}

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
