#ifndef BITCOIN_QT_TRANSACTIONRECORD_H
#define BITCOIN_QT_TRANSACTIONRECORD_H

#include "uint256.h"
#include "wallet/generated_type.h"
#include "wallet/ismine.h"

#include <QList>

class CWallet;
class CWalletTx;

/** UI model for transaction status. The transaction status is the part of a transaction that will change over time.
 */
class TransactionStatus
{
public:
    TransactionStatus()
        : countsForBalance(false)
        , sortKey("")
        , matures_in(0)
        , status(Offline)
        , generated_type(MinedType::UNKNOWN)
        , depth(0)
        , open_for(0)
        , cur_num_blocks(-1)
    { }

    enum Status {
        Confirmed,          /**< Have 10 or more confirmations (normal tx) or fully mature (mined tx) **/
        /// Normal (sent/received) transactions
        OpenUntilDate,      /**< Transaction not yet final, waiting for date */
        OpenUntilBlock,     /**< Transaction not yet final, waiting for block */
        Offline,            /**< Not sent to any other nodes **/
        Unconfirmed,        /**< Not yet mined into a block **/
        Confirming,         /**< Confirmed, but waiting for the recommended number of confirmations **/
        Conflicted,         /**< Conflicts with other transaction or mempool **/
        /// Generated (mined) transactions
        Immature,           /**< Mined but waiting for maturity */
        MaturesWarning,     /**< Transaction will likely not mature because no nodes have confirmed */
        NotAccepted         /**< Mined but not accepted */
    };

    /// Transaction counts towards available balance
    bool countsForBalance;
    /// Sorting key based on status
    std::string sortKey;

    /** @name Generated (mined) transactions
       @{*/
    int matures_in;
    /**@}*/

    /** @name Reported status
       @{*/
    Status status;
    MinedType generated_type;
    int64_t depth;
    int64_t open_for; /**< Timestamp if status==OpenUntilDate, otherwise number
                       of additional blocks that need to be mined before
                       finalization */
    /**@}*/

    /** Current number of blocks (to know whether cached status is still valid) */
    int cur_num_blocks;
};

/** UI model for a transaction. A core transaction can be represented by multiple UI transactions if it has
    multiple outputs.
 */
class TransactionRecord
{
public:
    enum Type
    {
        Other,
        Generated,
        SendToAddress,
        SendToOther,
        RecvWithAddress,
        RecvFromOther,
        SendToSelf,
        BeaconAdvertisement,
        Poll,
        Vote,
        Message,
        MRC
    };

    static constexpr std::initializer_list<Type> TYPES {Other,
                                                       Generated,
                                                       SendToAddress,
                                                       RecvWithAddress,
                                                       RecvFromOther,
                                                       SendToSelf,
                                                       BeaconAdvertisement,
                                                       Poll,
                                                       Vote,
                                                       Message,
                                                       MRC};

    /** Number of confirmation recommended for accepting a transaction */
    static const int RecommendedNumConfirmations = 10;

    TransactionRecord():
            hash(), time(0), type(Other), address(""), debit(0), credit(0), vout(0), idx(0)
    {
    }

    TransactionRecord(uint256 hash, int64_t time):
            hash(hash), time(time), type(Other), address(""), debit(0),
            credit(0), vout(0), idx(0)
    {
    }

    TransactionRecord(uint256 hash, int64_t time,
                Type type, const std::string &address,
                int64_t debit, int64_t credit, unsigned int vout):
            hash(hash), time(time), type(type), address(address), debit(debit), credit(credit),
            vout(vout), idx(0)
    {
    }

    /** Decompose CWallet transaction to model transaction records.
     */
    static bool showTransaction(const CWalletTx &wtx, bool datetime_limit_flag = false, const int64_t &datetime_limit = 0);
    static QList<TransactionRecord> decomposeTransaction(const CWallet *wallet, const CWalletTx &wtx);

    QString TypeToString() const;
    static QString TypeToString(const Type& type, const bool& translated = true);

    /** @name Immutable transaction attributes
      @{*/
    uint256 hash;
    qint64 time;
    Type type;
    std::string address;
    qint64 debit;
    qint64 credit;

    // Side/Split Stake
    unsigned int vout;
    /**@}*/

    /** Subtransaction index, for sort key */
    int idx;

    /** Status: can change with block chain update */
    TransactionStatus status;

    /** Return the unique identifier for this transaction (part) */
    std::string getTxID();

    /** Update status from core wallet tx.
     */
    void updateStatus(const CWalletTx &wtx);

    /** Return whether a status update is needed.
     */
    bool statusUpdateNeeded();
};

#endif // BITCOIN_QT_TRANSACTIONRECORD_H
