#!/usr/bin/env bash
#
# Copyright (c) 2018-2020 The Bitcoin Core developers
# Distributed under the MIT software license, see the accompanying
# file COPYING or https://opensource.org/licenses/mit-license.php.
#
# Check for duplicate includes.
# Guard against accidental introduction of new Boost dependencies.
# Check includes: Check for duplicate includes. Enforce bracket syntax includes.

export LC_ALL=C
IGNORE_REGEXP="/(leveldb|secp256k1|univalue|crc32c|bdb53)/"

# cd to root folder of git repo for git ls-files to work properly
cd "$(dirname $0)/../.." || exit 1

filter_suffix() {
    git ls-files | grep -E "^src/.*\.${1}"'$' | grep -Ev "${IGNORE_REGEXP}"
}

EXIT_CODE=0

for HEADER_FILE in $(filter_suffix h); do
    DUPLICATE_INCLUDES_IN_HEADER_FILE=$(grep -E "^#include " < "${HEADER_FILE}" | sort | uniq -d)
    if [[ ${DUPLICATE_INCLUDES_IN_HEADER_FILE} != "" ]]; then
        echo "Duplicate include(s) in ${HEADER_FILE}:"
        echo "${DUPLICATE_INCLUDES_IN_HEADER_FILE}"
        echo
        EXIT_CODE=1
    fi
done

for CPP_FILE in $(filter_suffix cpp); do
    DUPLICATE_INCLUDES_IN_CPP_FILE=$(grep -E "^#include " < "${CPP_FILE}" | sort | uniq -d)
    if [[ ${DUPLICATE_INCLUDES_IN_CPP_FILE} != "" ]]; then
        echo "Duplicate include(s) in ${CPP_FILE}:"
        echo "${DUPLICATE_INCLUDES_IN_CPP_FILE}"
        echo
        EXIT_CODE=1
    fi
done

INCLUDED_CPP_FILES=$(git grep -E "^#include [<\"][^>\"]+\.cpp[>\"]" -- "*.cpp" "*.h")
if [[ ${INCLUDED_CPP_FILES} != "" ]]; then
    echo "The following files #include .cpp files:"
    echo "${INCLUDED_CPP_FILES}"
    echo
    EXIT_CODE=1
fi

EXPECTED_BOOST_INCLUDES=(
    boost/algorithm/hex.hpp
    boost/algorithm/string.hpp
    boost/algorithm/string/case_conv.hpp
    boost/algorithm/string/classification.hpp
    boost/algorithm/string/join.hpp
    boost/algorithm/string/predicate.hpp
    boost/algorithm/string/replace.hpp
    boost/algorithm/string/split.hpp
    boost/array.hpp
    boost/asio.hpp
    boost/asio/ip/udp.hpp
    boost/asio/ip/v6_only.hpp
    boost/asio/ssl.hpp
    boost/asio/system_timer.hpp
    boost/assert.hpp
    boost/assign/list_of.hpp
    boost/bind/bind.hpp
    boost/date_time.hpp
    boost/date_time/gregorian/greg_date.hpp
    boost/date_time/gregorian/gregorian.hpp
    boost/date_time/gregorian/gregorian_types.hpp
    boost/date_time/posix_time/posix_time.hpp
    boost/date_time/posix_time/posix_time_types.hpp
    boost/exception/diagnostic_information.hpp
    boost/exception/exception.hpp
    boost/filesystem.hpp
    boost/filesystem/fstream.hpp
    boost/interprocess/ipc/message_queue.hpp
    boost/iostreams/copy.hpp
    boost/iostreams/concepts.hpp
    boost/iostreams/device/array.hpp
    boost/iostreams/filter/gzip.hpp
    boost/iostreams/filter/newline.hpp
    boost/iostreams/filtering_stream.hpp
    boost/iostreams/stream.hpp
    boost/lexical_cast.hpp
    boost/range/adaptor/reversed.hpp
    boost/serialization/binary_object.hpp
    boost/shared_ptr.hpp
    boost/signals2/optional_last_value.hpp
    boost/signals2/signal.hpp
    boost/test/unit_test.hpp
    boost/thread.hpp
    boost/thread/condition_variable.hpp
    boost/tuple/tuple.hpp
    boost/version.hpp
)

for BOOST_INCLUDE in $(git grep '^#include <boost/' -- "*.cpp" "*.h" | cut -f2 -d: | cut -f2 -d'<' | cut -f1 -d'>' | sort -u); do
    IS_EXPECTED_INCLUDE=0
    for EXPECTED_BOOST_INCLUDE in "${EXPECTED_BOOST_INCLUDES[@]}"; do
        if [[ "${BOOST_INCLUDE}" == "${EXPECTED_BOOST_INCLUDE}" ]]; then
            IS_EXPECTED_INCLUDE=1
            break
        fi
    done
    if [[ ${IS_EXPECTED_INCLUDE} == 0 ]]; then
        EXIT_CODE=1
        echo "A new Boost dependency in the form of \"${BOOST_INCLUDE}\" appears to have been introduced:"
        git grep "${BOOST_INCLUDE}" -- "*.cpp" "*.h"
        echo
    fi
done

for EXPECTED_BOOST_INCLUDE in "${EXPECTED_BOOST_INCLUDES[@]}"; do
    if ! git grep -q "^#include <${EXPECTED_BOOST_INCLUDE}>" -- "*.cpp" "*.h"; then
        echo "Good job! The Boost dependency \"${EXPECTED_BOOST_INCLUDE}\" is no longer used."
        echo "Please remove it from EXPECTED_BOOST_INCLUDES in $0"
        echo "to make sure this dependency is not accidentally reintroduced."
        echo
        EXIT_CODE=1
    fi
done

exit ${EXIT_CODE}
