/*
 * Decompiled with CFR 0.152.
 */
package app.tuxguitar.io.musicxml;

import app.tuxguitar.gm.GMChannelRoute;
import app.tuxguitar.gm.GMChannelRouter;
import app.tuxguitar.gm.GMChannelRouterConfigurator;
import app.tuxguitar.io.base.TGFileFormatException;
import app.tuxguitar.io.musicxml.MusicXMLLyricWriter;
import app.tuxguitar.song.managers.TGSongManager;
import app.tuxguitar.song.managers.TGTrackManager;
import app.tuxguitar.song.models.TGBeat;
import app.tuxguitar.song.models.TGChannel;
import app.tuxguitar.song.models.TGDivisionType;
import app.tuxguitar.song.models.TGDuration;
import app.tuxguitar.song.models.TGMarker;
import app.tuxguitar.song.models.TGMeasure;
import app.tuxguitar.song.models.TGNote;
import app.tuxguitar.song.models.TGSong;
import app.tuxguitar.song.models.TGString;
import app.tuxguitar.song.models.TGTempo;
import app.tuxguitar.song.models.TGTimeSignature;
import app.tuxguitar.song.models.TGTrack;
import app.tuxguitar.song.models.TGVoice;
import app.tuxguitar.song.models.effects.TGEffectBend;
import app.tuxguitar.song.models.effects.TGEffectHarmonic;
import app.tuxguitar.util.TGMusicKeyUtils;
import app.tuxguitar.util.TGVersion;
import java.io.OutputStream;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class MusicXMLWriter {
    private static final String[] DURATION_NAMES = new String[]{"whole", "half", "quarter", "eighth", "16th", "32nd", "64th"};
    private Map<Integer, String> durations;
    private static final int DURATION_DIVISIONS = 960;
    private static final int[] DURATION_VALUES = new int[]{3840, 1920, 960, 480, 240, 120, 60};
    private TGSongManager manager;
    private OutputStream stream;
    private Document document;

    public MusicXMLWriter(OutputStream stream) {
        this.stream = stream;
        this.durations = new HashMap<Integer, String>();
        this.durations.put(1, DURATION_NAMES[0]);
        this.durations.put(2, DURATION_NAMES[1]);
        this.durations.put(4, DURATION_NAMES[2]);
        this.durations.put(8, DURATION_NAMES[3]);
        this.durations.put(16, DURATION_NAMES[4]);
    }

    public void writeSong(TGSong song) throws TGFileFormatException {
        try {
            this.manager = new TGSongManager();
            this.document = this.newDocument();
            Node node = this.addNode(this.document, "score-partwise");
            this.addAttribute(node, "version", "4.0");
            this.writeHeaders(song, node);
            this.writeSong(song, node);
            this.saveDocument();
            this.stream.flush();
            this.stream.close();
        }
        catch (Throwable throwable) {
            throw new TGFileFormatException("Could not write song!.", throwable);
        }
    }

    private void writeHeaders(TGSong song, Node parent) {
        this.writeWork(song, parent);
        this.writeIdentification(song, parent);
    }

    private void writeWork(TGSong song, Node parent) {
        this.addNode(this.addNode(parent, "work"), "work-title", song.getName());
    }

    private void writeIdentification(TGSong song, Node parent) {
        Node identification = this.addNode(parent, "identification");
        if (!song.getAuthor().equals("")) {
            this.addAttribute(this.addNode(identification, "creator", song.getAuthor()), "type", "composer");
        }
        if (!song.getCopyright().equals("")) {
            this.addNode(identification, "rights", song.getCopyright());
        }
        Node encoding = this.addNode(identification, "encoding");
        this.addNode(encoding, "encoding-date", LocalDate.now().toString());
        this.addNode(encoding, "software", "TuxGuitar " + TGVersion.CURRENT.getVersion());
    }

    private void writeSong(TGSong song, Node parent) {
        this.writePartList(song, parent);
        this.writeParts(song, parent);
    }

    private void writePartList(TGSong song, Node parent) {
        Node partList = this.addNode(parent, "part-list");
        GMChannelRouter gmChannelRouter = new GMChannelRouter();
        GMChannelRouterConfigurator gmChannelRouterConfigurator = new GMChannelRouterConfigurator(gmChannelRouter);
        gmChannelRouterConfigurator.configureRouter(song.getChannels());
        Iterator tracks = song.getTracks();
        while (tracks.hasNext()) {
            TGTrack track = (TGTrack)tracks.next();
            TGChannel channel = this.manager.getChannel(song, track.getChannelId());
            Node scoreParts = this.addNode(partList, "score-part");
            this.addAttribute(scoreParts, "id", "P" + track.getNumber());
            this.addNode(scoreParts, "part-name", track.getName());
            if (channel == null) continue;
            GMChannelRoute gmChannelRoute = gmChannelRouter.getRoute(channel.getChannelId());
            Node scoreInstrument = this.addAttribute(this.addNode(scoreParts, "score-instrument"), "id", "P" + track.getNumber() + "-I1");
            this.addNode(scoreInstrument, "instrument-name", channel.getName());
            Node midiInstrument = this.addAttribute(this.addNode(scoreParts, "midi-instrument"), "id", "P" + track.getNumber() + "-I1");
            this.addNode(midiInstrument, "midi-channel", Integer.toString(gmChannelRoute != null ? gmChannelRoute.getChannel1() + 1 : 16));
            this.addNode(midiInstrument, "midi-program", Integer.toString(channel.getProgram() + 1));
        }
    }

    private void writeParts(TGSong song, Node parent) {
        Iterator tracks = song.getTracks();
        while (tracks.hasNext()) {
            TGTrack track = (TGTrack)tracks.next();
            this.writeTrack(track, parent);
        }
    }

    private void writeTrack(TGTrack track, Node parent) {
        Node part = this.addAttribute(this.addNode(parent, "part"), "id", "P" + track.getNumber());
        TGMeasure previousMeasure = null;
        MusicXMLLyricWriter lyricWriter = new MusicXMLLyricWriter(track);
        Iterator measures = track.getMeasures();
        TGMeasure nextMeasure = null;
        while (measures.hasNext() || nextMeasure != null) {
            int nVoice;
            TGMeasure currentMeasure = nextMeasure != null ? nextMeasure : (TGMeasure)measures.next();
            nextMeasure = measures.hasNext() ? (TGMeasure)measures.next() : null;
            Node measureNode = this.addAttribute(this.addNode(part, "measure"), "number", Integer.toString(currentMeasure.getNumber()));
            this.writeMeasureAttributes(measureNode, currentMeasure, previousMeasure, track.isPercussion());
            this.writeDirection(measureNode, currentMeasure, previousMeasure);
            this.writeBarline(measureNode, currentMeasure, previousMeasure, nextMeasure);
            MusicXMLLyricWriter.MusicXMLMeasureLyric[] measureLyrics = lyricWriter.generateLyricList(currentMeasure);
            boolean currentMeasureIsEmpty = true;
            for (nVoice = 0; nVoice < 2; ++nVoice) {
                this.writeBeats(measureNode, currentMeasure, nVoice, currentMeasureIsEmpty, false, (MusicXMLLyricWriter.MusicXMLMeasureLyric[])(nVoice == 0 ? measureLyrics : null));
                currentMeasureIsEmpty = false;
            }
            if (!track.isPercussion()) {
                currentMeasureIsEmpty = true;
                this.backToMeasureStart(measureNode, currentMeasure);
                for (nVoice = 0; nVoice < 2; ++nVoice) {
                    this.writeBeats(measureNode, currentMeasure, nVoice, currentMeasureIsEmpty, true, null);
                    currentMeasureIsEmpty = false;
                }
            }
            previousMeasure = currentMeasure;
        }
    }

    private void backToMeasureStart(Node parent, TGMeasure measure) {
        Node backupNode = this.addNode(parent, "backup");
        TGTimeSignature ts = measure.getTimeSignature();
        this.addNode(backupNode, "duration", String.valueOf(3840 * ts.getNumerator() / ts.getDenominator().getValue()));
    }

    private void writeBarline(Node parent, TGMeasure currentMeasure, TGMeasure previousMeasure, TGMeasure nextMeasure) {
        Node repeat;
        Node barLine;
        Node startBarLine = null;
        Node endBarLine = null;
        String previousMeasureAlternateEndings = this.generateAlternateEndingString(previousMeasure);
        String currentMeasureAlternateEndings = this.generateAlternateEndingString(currentMeasure);
        String nextMeasureAlternateEndings = this.generateAlternateEndingString(nextMeasure);
        if (!currentMeasureAlternateEndings.equals("")) {
            Node ending;
            if (!currentMeasureAlternateEndings.equals(previousMeasureAlternateEndings)) {
                startBarLine = barLine = this.addNode(parent, "barline");
                ending = this.addNode(barLine, "ending", currentMeasureAlternateEndings);
                this.addAttribute(ending, "number", currentMeasureAlternateEndings);
                this.addAttribute(ending, "type", "start");
            }
            if (!currentMeasureAlternateEndings.equals(nextMeasureAlternateEndings)) {
                endBarLine = barLine = this.addNode(parent, "barline");
                ending = this.addNode(barLine, "ending", currentMeasureAlternateEndings);
                this.addAttribute(ending, "number", currentMeasureAlternateEndings);
                this.addAttribute(ending, "type", "stop");
            }
        }
        if (currentMeasure.isRepeatOpen()) {
            startBarLine = barLine = startBarLine != null ? startBarLine : this.addNode(parent, "barline");
            repeat = this.addNode(barLine, "repeat");
            this.addAttribute(repeat, "direction", "forward");
        }
        if (currentMeasure.getRepeatClose() > 0) {
            endBarLine = barLine = endBarLine != null ? endBarLine : this.addNode(parent, "barline");
            repeat = this.addNode(barLine, "repeat");
            this.addAttribute(repeat, "direction", "backward");
            this.addAttribute(repeat, "times", Integer.toString(currentMeasure.getRepeatClose() + 1));
        }
    }

    private String generateAlternateEndingString(TGMeasure measure) {
        StringBuilder alternateEndingNumbers = new StringBuilder();
        if (measure != null && measure.getHeader().getRepeatAlternative() != 0) {
            byte repeatAlternativeByte = (byte)measure.getHeader().getRepeatAlternative();
            BitSet repeatAlternateBitset = BitSet.valueOf(new byte[]{repeatAlternativeByte});
            int i = repeatAlternateBitset.nextSetBit(0);
            while (i >= 0) {
                alternateEndingNumbers.append(Integer.toString(i + 1) + ",");
                i = repeatAlternateBitset.nextSetBit(i + 1);
            }
            if (alternateEndingNumbers.length() > 0) {
                alternateEndingNumbers.deleteCharAt(alternateEndingNumbers.length() - 1);
            }
        }
        return alternateEndingNumbers.toString();
    }

    private void writeMeasureAttributes(Node parent, TGMeasure measure, TGMeasure previous, boolean isPercussion) {
        boolean timeSignatureChanges;
        boolean divisionChanges = previous == null;
        boolean keyChanges = previous == null || measure.getKeySignature() != previous.getKeySignature();
        boolean clefChanges = previous == null || measure.getClef() != previous.getClef();
        boolean bl = timeSignatureChanges = previous == null || !measure.getTimeSignature().isEqual(previous.getTimeSignature());
        if (divisionChanges || keyChanges || clefChanges || timeSignatureChanges) {
            Node measureAttributes = this.addNode(parent, "attributes");
            if (divisionChanges) {
                this.addNode(measureAttributes, "divisions", Integer.toString(960));
            }
            if (keyChanges) {
                this.writeKeySignature(measureAttributes, measure.getKeySignature());
            }
            if (timeSignatureChanges) {
                this.writeTimeSignature(measureAttributes, measure.getTimeSignature());
            }
            this.addNode(measureAttributes, "staves", "2");
            this.addNode(measureAttributes, "part-symbol", "none");
            if (clefChanges) {
                this.writeClef(measureAttributes, measure.getClef(), previous == null, isPercussion);
            }
            if (!(isPercussion || previous != null && measure.getNumber() != 1)) {
                this.writeTuning(measureAttributes, measure.getTrack(), measure.getKeySignature());
            }
        }
    }

    private void writeTuning(Node parent, TGTrack track, int keySignature) {
        Node staffDetailsNode = this.addNode(parent, "staff-details");
        this.addNode(staffDetailsNode, "staff-type", "alternate");
        this.addAttribute(staffDetailsNode, "number", "2");
        this.addNode(staffDetailsNode, "staff-lines", Integer.toString(track.stringCount()));
        for (int i = track.stringCount(); i > 0; --i) {
            TGString string = track.getString(i);
            Node stringNode = this.addNode(staffDetailsNode, "staff-tuning");
            this.addAttribute(stringNode, "line", Integer.toString(track.stringCount() - string.getNumber() + 1));
            this.writeNote(stringNode, "tuning-", string.getValue(), keySignature);
        }
        int trackOffset = track.getOffset();
        if (trackOffset > 0) {
            this.addNode(staffDetailsNode, "capo", Integer.toString(trackOffset));
        }
    }

    private void writeNote(Node parent, String prefix, int value, int keySignature) {
        this.addNode(parent, prefix + "step", TGMusicKeyUtils.noteShortName((int)value, (int)keySignature));
        int alteration = TGMusicKeyUtils.noteAlteration((int)value, (int)keySignature);
        if (alteration != 1) {
            this.addNode(parent, prefix + "alter", alteration == 2 ? "1" : "-1");
        }
        this.addNode(parent, prefix + "octave", String.valueOf(TGMusicKeyUtils.noteOctave((int)value, (int)keySignature)));
    }

    private void writeTimeSignature(Node parent, TGTimeSignature ts) {
        Node node = this.addNode(parent, "time");
        this.addNode(node, "beats", Integer.toString(ts.getNumerator()));
        this.addNode(node, "beat-type", Integer.toString(ts.getDenominator().getValue()));
    }

    private void writeKeySignature(Node parent, int ks) {
        int value = ks;
        if (value != 0) {
            value = ((ks - 1) % 7 + 1) * (ks > 7 ? -1 : 1);
        }
        Node key = this.addNode(parent, "key");
        this.addNode(key, "fifths", Integer.toString(value));
    }

    private void writeClef(Node parent, int clef, boolean isStart, boolean isPercussion) {
        Node node = this.addNode(parent, "clef");
        if (!isStart) {
            this.addAttribute(node, "after-barline", "yes");
        }
        if (!isPercussion) {
            this.addAttribute(node, "number", "1");
        }
        if (isPercussion) {
            this.addNode(node, "sign", "percussion");
        } else {
            if (clef == 1) {
                this.addNode(node, "sign", "G");
                this.addNode(node, "line", "2");
            } else if (clef == 2) {
                this.addNode(node, "sign", "F");
                this.addNode(node, "line", "4");
            } else if (clef == 3) {
                this.addNode(node, "sign", "C");
                this.addNode(node, "line", "4");
            } else if (clef == 4) {
                this.addNode(node, "sign", "C");
                this.addNode(node, "line", "3");
            }
            this.addNode(node, "clef-octave-change", String.valueOf(-1));
        }
        if (isStart && !isPercussion) {
            node = this.addNode(parent, "clef");
            this.addAttribute(node, "number", "2");
            this.addNode(node, "sign", "TAB");
        }
    }

    private String getDynamicNameFromVelocity(TGNote note) {
        int noteVelocity = note.getVelocity();
        switch (noteVelocity) {
            case 15: {
                return "ppp";
            }
            case 31: {
                return "pp";
            }
            case 47: {
                return "p";
            }
            case 63: {
                return "mp";
            }
            case 79: {
                return "mf";
            }
            case 95: {
                return "f";
            }
            case 111: {
                return "ff";
            }
            case 127: {
                return "fff";
            }
        }
        return "";
    }

    private void writeDynamics(Node parent, TGNote note) {
        Node direction = this.addAttribute(this.addNode(parent, "direction"), "placement", "above");
        Node directionType = this.addNode(direction, "direction-type");
        Node dynamics = this.addNode(directionType, "dynamics");
        String dynamicType = this.getDynamicNameFromVelocity(note);
        if (!dynamicType.equals("")) {
            this.addNode(dynamics, dynamicType);
        }
    }

    private void writeDirection(Node parent, TGMeasure measure, TGMeasure previous) {
        TGMarker measureMarker;
        boolean tempoChanges;
        boolean needsDirectionNode = false;
        Node direction = this.addAttribute(this.addNode(parent, "direction"), "placement", "above");
        boolean bl = tempoChanges = previous == null || measure.getTempo().getRawValue() != previous.getTempo().getRawValue() || measure.getTempo().getBase() != previous.getTempo().getBase();
        if (tempoChanges) {
            Node directionType = this.addNode(direction, "direction-type");
            Node metronome = this.addNode(directionType, "metronome");
            TGTempo tempo = measure.getTempo();
            this.addNode(metronome, "beat-unit", this.durations.get(tempo.getBase()));
            if (tempo.isDotted()) {
                this.addNode(metronome, "beat-unit-dot");
            }
            this.addNode(metronome, "per-minute", String.valueOf(measure.getTempo().getRawValue()));
            needsDirectionNode = true;
        }
        if ((measureMarker = measure.getHeader().getMarker()) != null) {
            Node directionType = this.addNode(direction, "direction-type");
            this.addNode(directionType, "rehearsal", measureMarker.getTitle());
            needsDirectionNode = true;
        }
        if (!needsDirectionNode) {
            this.removeNode(direction);
        }
    }

    private void writeBeats(Node parent, TGMeasure measure, int nVoice, boolean measureIsEmpty, boolean isTablature, MusicXMLLyricWriter.MusicXMLMeasureLyric[] lyrics) {
        TGTrackManager trackMgr = new TGSongManager().getTrackManager();
        int ks = measure.getKeySignature();
        int beatCount = measure.countBeats();
        int lyricIndex = 0;
        ArrayList<TGBeat> firstBeats = new ArrayList<TGBeat>();
        boolean wroteSomething = false;
        long lastWrittenNoteEnd = 0L;
        int lastVelocity = 14;
        HashMap<Integer, TGNote> previousNotesOnAllStrings = new HashMap<Integer, TGNote>();
        for (int b = 0; b < beatCount; ++b) {
            TGBeat beat = measure.getBeat(b);
            TGVoice voice = beat.getVoice(nVoice);
            if (voice.isRestVoice() && !wroteSomething) {
                firstBeats.add(beat);
                continue;
            }
            if (!measureIsEmpty && !wroteSomething) {
                this.backToMeasureStart(parent, measure);
            }
            if (!firstBeats.isEmpty()) {
                for (TGBeat restBeat : firstBeats) {
                    this.insertRest(parent, restBeat.getVoice(nVoice).getDuration(), nVoice, isTablature);
                }
                firstBeats.clear();
            }
            if (voice.isRestVoice()) {
                if (beat.getStart() < lastWrittenNoteEnd) continue;
                this.insertRest(parent, voice.getDuration(), nVoice, isTablature);
                continue;
            }
            int noteCount = voice.countNotes();
            for (int n = 0; n < noteCount; ++n) {
                Node slideNode;
                boolean isPreviousNoteSlide;
                Node noteheadNode;
                TGEffectHarmonic harmonicEffect;
                TGNote note = voice.getNote(n);
                TGNote previousNoteOnString = (TGNote)previousNotesOnAllStrings.get(note.getString());
                if (!isTablature && note.getEffect().isPalmMute()) {
                    Node direction = this.addAttribute(this.addNode(parent, "direction"), "placement", "above");
                    this.addAttribute(direction, "placement", "below");
                    Node directionType = this.addNode(direction, "direction-type");
                    Node node = this.addNode(directionType, "words", "P.M.");
                }
                Node noteNode = this.addNode(parent, "note");
                float noteVelocity = (float)note.getVelocity() / 95.0f * 100.0f;
                this.addAttribute(noteNode, "dynamics", Float.toString(noteVelocity));
                int stringValue = beat.getMeasure().getTrack().getString(note.getString()).getValue();
                int noteValue = note.getValue();
                int harmonicValue = 0;
                if (!isTablature && (harmonicEffect = note.getEffect().getHarmonic()) != null && harmonicEffect.isNatural()) {
                    harmonicValue = TGEffectHarmonic.NATURAL_FREQUENCIES[harmonicEffect.getData()][1];
                    harmonicValue -= noteValue;
                }
                int harmonicAdjustedValue = stringValue + noteValue + harmonicValue;
                if (n > 0) {
                    this.addNode(noteNode, "chord");
                }
                Node pitchNode = this.addNode(noteNode, "pitch");
                this.writeNote(pitchNode, "", harmonicAdjustedValue, ks);
                this.writeDurationAndVoice(noteNode, voice.getDuration(), nVoice, trackMgr.isAnyTiedTo(note), note.isTiedNote());
                if (isTablature) {
                    this.addNode(noteNode, "stem", "none");
                }
                if (note.getEffect().isGhostNote()) {
                    noteheadNode = this.addNode(noteNode, "notehead", "normal");
                    this.addAttribute(noteheadNode, "parentheses", "yes");
                } else if (!isTablature && note.getEffect().isDeadNote()) {
                    noteheadNode = this.addNode(noteNode, "notehead", "x");
                }
                this.addNode(noteNode, "staff", isTablature ? "2" : "1");
                Node notationsNode = this.addNode(noteNode, "notations");
                if (!isTablature && (trackMgr.isAnyTiedTo(note) || note.isTiedNote())) {
                    this.writeTiedNotations(notationsNode, trackMgr.isAnyTiedTo(note), note.isTiedNote());
                }
                this.writeArticulationNotations(notationsNode, note);
                this.writeTechnicalNotations(notationsNode, note, previousNoteOnString, isTablature, n > 0);
                this.writeOrnamentsNotations(notationsNode, note);
                boolean isNoteSlide = note.getEffect().isSlide();
                boolean bl = isPreviousNoteSlide = previousNoteOnString != null ? previousNoteOnString.getEffect().isSlide() : false;
                if (isPreviousNoteSlide) {
                    slideNode = this.addNode(notationsNode, "slide");
                    this.addAttribute(slideNode, "type", "stop");
                }
                if (isNoteSlide) {
                    slideNode = this.addNode(notationsNode, "slide");
                    this.addAttribute(slideNode, "type", "start");
                }
                this.removeNodeIfNoChildren(notationsNode);
                if (note.getEffect().isLetRing()) {
                    Node tiedNode = this.addNode(noteNode, "tied");
                    this.addAttribute(tiedNode, "type", "let-ring");
                }
                if (!isTablature && n == 0) {
                    try {
                        MusicXMLLyricWriter.MusicXMLMeasureLyric measureLyric = lyrics[lyricIndex++];
                        this.writeLyric(noteNode, measureLyric);
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
                this.writePlay(noteNode, note);
                lastWrittenNoteEnd = beat.getStart() + voice.getDuration().getTime();
                previousNotesOnAllStrings.put(note.getString(), note);
            }
            wroteSomething = true;
        }
        if (!wroteSomething && measureIsEmpty && !firstBeats.isEmpty()) {
            for (TGBeat restBeat : firstBeats) {
                this.insertRest(parent, restBeat.getVoice(nVoice).getDuration(), nVoice, isTablature);
            }
        }
    }

    private void writeTiedNotations(Node parent, boolean tieStart, boolean tieStop) {
        Node tiedNode = this.addNode(parent, "tied");
        this.addAttribute(tiedNode, "orientation", "over");
        if (tieStop) {
            this.addAttribute(tiedNode, "type", "stop");
        }
        if (tieStart) {
            this.addAttribute(tiedNode, "type", "start");
        }
    }

    private void writeArticulationNotations(Node parent, TGNote note) {
        Node articulationsNode = this.addNode(parent, "articulations");
        if (note.getEffect().isAccentuatedNote()) {
            this.addNode(articulationsNode, "accent");
        }
        if (note.getEffect().isHeavyAccentuatedNote()) {
            this.addNode(articulationsNode, "strong-accent");
        }
        if (note.getEffect().isStaccato()) {
            this.addNode(articulationsNode, "staccato");
        }
        this.removeNodeIfNoChildren(articulationsNode);
    }

    private void writeTechnicalNotations(Node parent, TGNote note, TGNote previousNoteOnString, boolean isTablature, boolean isChordNote) {
        TGEffectHarmonic harmonicEffect;
        TGEffectBend bendEffect;
        Node hammerNode;
        Node technicalNode = this.addNode(parent, "technical");
        if (isTablature) {
            this.addNode(technicalNode, "fret", Integer.toString(note.getValue()));
            this.addNode(technicalNode, "string", Integer.toString(note.getString()));
        }
        if (previousNoteOnString != null && previousNoteOnString.getEffect().isHammer()) {
            hammerNode = this.addNode(technicalNode, "hammer-on");
            this.addAttribute(hammerNode, "type", "stop");
        }
        if (note.getEffect().isHammer()) {
            hammerNode = this.addNode(technicalNode, "hammer-on", "H");
            this.addAttribute(hammerNode, "type", "start");
        }
        if (note.getEffect().isTapping()) {
            this.addNode(technicalNode, "tap");
        }
        if (!isChordNote) {
            TGBeat beat = note.getVoice().getBeat();
            if (beat.getPickStroke().getDirection() == 1) {
                this.addNode(technicalNode, "up-bow");
            } else if (beat.getPickStroke().getDirection() == -1) {
                this.addNode(technicalNode, "down-bow");
            }
        }
        if ((bendEffect = note.getEffect().getBend()) != null) {
            boolean isFirstBend = true;
            Integer previousBendValue = 0;
            for (TGEffectBend.BendPoint currentBendPoint : bendEffect.getPoints()) {
                if (previousBendValue.intValue() == currentBendPoint.getValue()) {
                    isFirstBend = false;
                    continue;
                }
                boolean isPreBend = false;
                boolean isRelease = false;
                if (isFirstBend && currentBendPoint.getValue() != 0) {
                    isPreBend = true;
                } else if (currentBendPoint.getValue() == 0) {
                    isRelease = true;
                }
                Node bendNode = this.addNode(technicalNode, "bend");
                this.addAttribute(bendNode, "shape", "curved");
                float pitchAlter = (float)currentBendPoint.getValue() / 2.0f;
                this.addNode(bendNode, "bend-alter", Float.toString(pitchAlter));
                if (isPreBend) {
                    this.addNode(bendNode, "pre-bend");
                } else if (isRelease) {
                    this.addNode(bendNode, "release");
                }
                previousBendValue = currentBendPoint.getValue();
                isFirstBend = false;
            }
        }
        if ((harmonicEffect = note.getEffect().getHarmonic()) != null) {
            Node harmonicNode = this.addNode(technicalNode, "harmonic");
            if (harmonicEffect.isNatural()) {
                this.addNode(harmonicNode, "natural");
            } else if (harmonicEffect.isArtificial() || harmonicEffect.isPinch()) {
                this.addNode(harmonicNode, "artificial");
            }
        }
        this.removeNodeIfNoChildren(technicalNode);
    }

    private void writeOrnamentsNotations(Node parent, TGNote note) {
        Node ornamentsNode = this.addNode(parent, "ornaments");
        if (note.getEffect().isTremoloPicking()) {
            int tremoloDuration = note.getEffect().getTremoloPicking().getDuration().getValue();
            int tremoloMarkings = 0;
            int loopDuration = 4;
            while (loopDuration < tremoloDuration) {
                loopDuration *= 2;
                ++tremoloMarkings;
            }
            this.addNode(ornamentsNode, "tremolo", Integer.toString(tremoloMarkings));
        }
        if (note.getEffect().isTrill()) {
            Node trillNode = this.addNode(ornamentsNode, "trill-mark");
            TGDuration noteDuration = note.getVoice().getDuration();
            int trillBeats = note.getEffect().getTrill().getDuration().getValue() / noteDuration.getValue();
            if (noteDuration.isDotted()) {
                trillBeats = trillBeats * 3 / 2;
            }
            if (noteDuration.isDoubleDotted()) {
                trillBeats = trillBeats * 7 / 4;
            }
            if (trillBeats > 1) {
                this.addAttribute(trillNode, "beats", Integer.toString(trillBeats));
            }
            String trillStep = "unison";
            int trillFret = note.getEffect().getTrill().getFret();
            int trillFretDifference = trillFret - note.getValue();
            if (trillFretDifference == 2) {
                trillStep = "whole";
            } else if (trillFretDifference == 1) {
                trillStep = "half";
            }
            this.addAttribute(trillNode, "trill-step", trillStep);
        }
        this.removeNodeIfNoChildren(ornamentsNode);
    }

    private void writePlay(Node parent, TGNote note) {
        Node playNode = this.addNode(parent, "play");
        if (note.getEffect().isDeadNote()) {
            this.addNode(playNode, "mute", "straight");
        } else if (note.getEffect().isPalmMute()) {
            this.addNode(playNode, "mute", "palm");
        }
        this.removeNodeIfNoChildren(playNode);
    }

    private void insertRest(Node parent, TGDuration duration, int nVoice, boolean isTablature) {
        Node noteRestNode = this.addNode(parent, "note");
        this.addNode(noteRestNode, "rest");
        this.writeDurationAndVoice(noteRestNode, duration, nVoice, false, false);
        this.addNode(noteRestNode, "staff", isTablature ? "2" : "1");
    }

    private void writeLyric(Node parent, MusicXMLLyricWriter.MusicXMLMeasureLyric measureLyric) {
        if (measureLyric.text.length() > 0) {
            Node lyricNode = this.addNode(parent, "lyric");
            this.addNode(lyricNode, "syllabic", measureLyric.syllabic.toString());
            this.addNode(lyricNode, "text", measureLyric.text);
        }
    }

    private void writeDurationAndVoice(Node parent, TGDuration duration, int nVoice, boolean tieStart, boolean tieStop) {
        int index = duration.getIndex();
        if (index >= 0 && index <= 6) {
            int value = DURATION_VALUES[index] * duration.getDivision().getTimes() / duration.getDivision().getEnters();
            if (duration.isDotted()) {
                value += value / 2;
            } else if (duration.isDoubleDotted()) {
                value += value / 4 * 3;
            }
            this.addNode(parent, "duration", Integer.toString(value));
            if (tieStop) {
                this.addAttribute(this.addNode(parent, "tie"), "type", "stop");
            }
            if (tieStart) {
                this.addAttribute(this.addNode(parent, "tie"), "type", "start");
            }
            this.addNode(parent, "voice", String.valueOf(nVoice + 1));
            this.addNode(parent, "type", DURATION_NAMES[index]);
            if (duration.isDotted()) {
                this.addNode(parent, "dot");
            } else if (duration.isDoubleDotted()) {
                this.addNode(parent, "dot");
                this.addNode(parent, "dot");
            }
            if (!duration.getDivision().isEqual(TGDivisionType.NORMAL)) {
                Node divisionType = this.addNode(parent, "time-modification");
                this.addNode(divisionType, "actual-notes", Integer.toString(duration.getDivision().getEnters()));
                this.addNode(divisionType, "normal-notes", Integer.toString(duration.getDivision().getTimes()));
            }
        }
    }

    private Node addAttribute(Node node, String name, String value) {
        Attr attribute = this.document.createAttribute(name);
        attribute.setNodeValue(value);
        node.getAttributes().setNamedItem(attribute);
        return node;
    }

    private Node addNode(Node parent, String name) {
        Element node = this.document.createElement(name);
        parent.appendChild(node);
        return node;
    }

    private Node addNode(Node parent, String name, String content) {
        Node node = this.addNode(parent, name);
        node.setTextContent(content);
        return node;
    }

    private void removeNode(Node nodeToRemove) {
        nodeToRemove.getParentNode().removeChild(nodeToRemove);
    }

    private void removeNodeIfNoChildren(Node nodeToRemove) {
        if (nodeToRemove.getChildNodes().getLength() != 0) {
            return;
        }
        this.removeNode(nodeToRemove);
    }

    private Document newDocument() {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document document = builder.newDocument();
            return document;
        }
        catch (Throwable throwable) {
            throwable.printStackTrace();
            return null;
        }
    }

    private void saveDocument() {
        try {
            TransformerFactory xformFactory = TransformerFactory.newInstance();
            Transformer idTransform = xformFactory.newTransformer();
            DOMSource input = new DOMSource(this.document);
            StreamResult output = new StreamResult(this.stream);
            idTransform.setOutputProperty("indent", "yes");
            DOMImplementation domImpl = this.document.getImplementation();
            DocumentType docType = domImpl.createDocumentType("doctype", "-//Recordare//DTD MusicXML 4.0 Partwise//EN", "http://www.musicxml.org/dtds/partwise.dtd");
            idTransform.setOutputProperty("doctype-public", docType.getPublicId());
            idTransform.setOutputProperty("doctype-system", docType.getSystemId());
            idTransform.transform(input, output);
        }
        catch (Throwable throwable) {
            throwable.printStackTrace();
        }
    }
}

