/**
 * "UltimateDailyWallpaper" Copyright (C) 2025 Patrice Coni
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include "photobrowser.h"
#include "ui_photobrowser.h"
#include "itemdelegate.h"

#include <QSplitter>
#include <QScrollArea>
#include <QSize>
#include <QFile>
#include <QDir>
#include <QSettings>
#include <QPixmap>
#include <QImage>
#include <QIcon>
#include <QStandardItem>
#include <QObject>
#include <QtConcurrent>
#include <QThreadPool>
#include <QCloseEvent>
#include <QFileInfoList>
#include <QSettings>
#include <QDesktopServices>
#include <QSqlDatabase>
#include <QSqlQuery>
#include <QSqlError>
#include <QDate>
#include <QMessageBox>
#if QT_VERSION >= 0x060000
    #include <QWidget>
#else
    #include <QDesktopWidget>
#endif
#include <QDebug>
#include <QScreen>

PhotoBrowser::PhotoBrowser(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::PhotoBrowser)
{
    ui->setupUi(this);

    move(screen()->geometry().center() - frameGeometry().center());

    _read_settings();

    _tempdirpath=QDir::homePath()+"/.UltimateDailyWallpaper/temp";

    _scaled_picture_width = 258;
    _scaled_picture_height = 145;

    QObject::connect(ui->pushButton_4, &QPushButton::clicked, [this]()
    {
        ui->calendarWidget->setSelectedDate(QDate::currentDate());
    });

    QObject::connect(ui->pushButton, &QPushButton::clicked, this, &PhotoBrowser::setWallpaperFile);
    QObject::connect(ui->pushButton_3, &QPushButton::clicked, this, &PhotoBrowser::show);
    QObject::connect(ui->pushButton_2, &QPushButton::clicked, this, &PhotoBrowser::download_and_set);

    QObject::connect(this, &PhotoBrowser::go_setWallpaper, [=](QString _wallpaperfile, int _Parameter, int _change_lockscreen_wallpaper){
        _setwall._set_wallpaper(_wallpaperfile, _Parameter, _change_lockscreen_wallpaper);
    });

    ui->widget_2->setVisible(false);
    ui->widget_3->setVisible(false);
    ui->frame_4->setVisible(false);

    _get_specific_values=false;
}

void PhotoBrowser::closeEvent(QCloseEvent * event)
{
    deltempdir();

    event->ignore();
    this->hide();
}

void PhotoBrowser::reject()
{
    this->hide();
}

PhotoBrowser::~PhotoBrowser()
{
    delete ui;
}

void PhotoBrowser::init(int mode)
{
    // Set window size
    this->resize(1220, 750);

    deltempdir();

    ui->widget_2->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    ui->widget_3->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    ui->widget->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    /* **********************
     * Initialize QSplitter *
     * ********************** */

    // QSplitter is child of ui->widget
    QSplitter* splitter = new QSplitter(Qt::Horizontal, ui->widget);

    // Remove widgets from the previous layout.
    if (ui->widget->layout()) {
        QLayoutItem* item;
        while ((item = ui->widget->layout()->takeAt(0)) != nullptr) {
            item->widget()->setParent(nullptr);
            delete item;
        }
        delete ui->widget->layout();
    }

    // Add widgets to splitter
    splitter->addWidget(ui->frame_4);
    splitter->addWidget(ui->listView);
    splitter->addWidget(ui->scrollArea);

    // Set splitter in "widget"
    QVBoxLayout* layout = new QVBoxLayout(ui->widget);
    layout->addWidget(splitter);
    layout->setContentsMargins(0,0,0,0);
    splitter->setSizes({300, 500, 300});
    splitter->setChildrenCollapsible(false);

    ui->scrollArea->setMinimumSize(ui->scrollArea->minimumSize());
    ui->frame_4->setMinimumSize(ui->frame_4->minimumSize());

    // Index 0 = frame_4, Index 1 = listView, Index 2 = scrollArea
    splitter->setSizes({
        ui->frame_4->minimumWidth(),  // far left
        500,                          // ListView (any starting value)
        ui->scrollArea->minimumWidth()// far right
    });

    splitter->setStretchFactor(0, 0); // frame_4 will not expand
    splitter->setStretchFactor(1, 1); // listView expands
    splitter->setStretchFactor(2, 0); // scrollArea will not expand

    ui->listView->setResizeMode(QListView::Adjust);
    ui->listView->setFlow(QListView::LeftToRight);
    ui->listView->setSpacing(15);
    ui->listView->setWrapping(true);
    ui->listView->setUniformItemSizes(true);
    ui->listView->setIconSize(QSize (_scaled_picture_width, _scaled_picture_height));
    ui->listView->setViewMode(QListView::IconMode);
    ui->listView->setMovement(QListView::Static);
    ui->listView->setSelectionMode(QAbstractItemView::SingleSelection);
    ui->listView->setSelectionBehavior(QAbstractItemView::SelectItems);
    ui->listView->setFocusPolicy(Qt::NoFocus);
    ui->listView->setUniformItemSizes(true);
    ItemDelegate *delegate = new ItemDelegate(this);
    delegate->displayRoleEnabled = false; // hide text
    ui->listView->setItemDelegate(delegate);


    /* *************************************************
     * mode variants
     * *************************************************
     *
     *  0: select stored picture
     *
     *  1: Download Wikimedia Commons picture of the day
     *     of past days within a given time range
     *
     * *************************************************
     */

    _mode=mode;

    switch(mode)
    {
    case 0:
        _thumbfiledir = QDir::homePath()+"/.UltimateDailyWallpaper/thumbnails";
        _databaseFilePath=QDir::homePath()+"/.UltimateDailyWallpaper/udw_database.sqlite";

        ui->widget_2->setVisible(true);

        QObject::connect(ui->listView, &QListView::doubleClicked, this, &PhotoBrowser::setWallpaperFile);

        break;
    case 1:
        _thumbfiledir = QDir::homePath()+"/.UltimateDailyWallpaper/temp/thumbnails";
        _databaseFilePath=QDir::homePath()+"/.UltimateDailyWallpaper/temp/temp_udw_database.sqlite";

        QObject::connect(ui->listView, &QListView::doubleClicked, this, &PhotoBrowser::download_and_set);

        if(!loadPlugin(_pluginsdir+"/"+_selected_plugin))
        {
            QMessageBox::critical(this, tr("Error"), tr("Error while loading plugin."));
            deltempdir();
            this->hide();
        }

        this->setWindowTitle(tr("Past pictures of the day (Wikimedia Commons)"));

        QDir _tempthumbfiledir(_thumbfiledir);
        if(!_tempthumbfiledir.exists())
        {
            _tempthumbfiledir.mkpath(_thumbfiledir);
        }

        ui->frame_4->setVisible(true);
        ui->widget_3->setVisible(true);
        ui->spinBox->setValue(1);

        if(create_temp_database()==true)
        {
            download_thumb(ui->spinBox->value());
        }

        ui->spinBox->setValue(3);

        break;
    }

    _get_values();

    QStandardItemModel *model = new QStandardItemModel(_thumbfilelist.size(), 1, this);

    for(int i=0;i<_thumbfilelist.size(); i++)
    {
        _thumbfilename = _thumbfilelist.at(i).toUtf8();
        QPixmap thumb_orig;
        thumb_orig.load(_thumbfiledir+"/"+_thumbfilename);
        QPixmap thumb=thumb_orig.scaled(QSize(_scaled_picture_width, _scaled_picture_height),  Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
        QIcon icon = thumb;
        QStandardItem *item = new QStandardItem(icon, _thumbfilelist.at(i).toUtf8());
        model->setItem(i, item);
    }

    ui->listView->setModel(model);

    wallchanged = false;

    _label_Details = new QLabel(this);
    _label_headline = new QLabel(this);
    _label_headline1 = new QLabel(this);
    _label_description_and_copyright1 = new QLabel(this);
    _label_description_and_copyright = new QLabel(this);
    _preview_text = new QLabel(this);
    _previewImageLabel = new QLabel(this);
    _getmoreInformations = new QPushButton(this);
    _getmoreInformations->setText(tr("Get more information"));
    _getmoreInformations->setVisible(false);
    connect(_getmoreInformations, &QPushButton::released, this, &PhotoBrowser::_getmoreInformationsButton_clicked);

    _label_image_size1 = new QLabel(this);
    _label_image_size = new QLabel(this);

    _layout_details = new QVBoxLayout();
    ui->scrollArea->setLayout(_layout_details);

    QWidget *scrollAreaWidgetContents = new QWidget();
    scrollAreaWidgetContents->setLayout(_layout_details);
    ui->scrollArea->setWidget(scrollAreaWidgetContents);

    _layout_details->addWidget(_label_Details);
    _layout_details->addSpacerItem(new QSpacerItem(0,20, QSizePolicy::Minimum,QSizePolicy::Minimum));

    _layout_details->addWidget(_label_headline);
    _layout_details->addWidget(_label_headline1);
    _layout_details->addSpacerItem(new QSpacerItem(0,20, QSizePolicy::Minimum,QSizePolicy::Minimum));

    _layout_details->addWidget(_label_description_and_copyright1);
    _layout_details->addWidget(_label_description_and_copyright);
    _layout_details->addSpacerItem(new QSpacerItem(0,20, QSizePolicy::Minimum,QSizePolicy::Minimum));

    _layout_details->addWidget(_label_image_size1);
    _layout_details->addWidget(_label_image_size);
    _layout_details->addSpacerItem(new QSpacerItem(0,20, QSizePolicy::Minimum,QSizePolicy::Minimum));

    _layout_details->addWidget(_preview_text);
    _layout_details->addWidget(_previewImageLabel);
    _layout_details->setAlignment(_previewImageLabel, Qt::AlignHCenter);
    _layout_details->addSpacerItem(new QSpacerItem(0,20, QSizePolicy::Minimum,QSizePolicy::Minimum));
    _layout_details->addWidget(_getmoreInformations);
    _layout_details->setAlignment(_getmoreInformations, Qt::AlignHCenter);
    _layout_details->addStretch();

    ui->scrollArea->setWidgetResizable(true);

    _first_run=true;
    _display_details();
}

void PhotoBrowser::_read_settings()
{
    QString _confFilePath = QDir::homePath()+"/.UltimateDailyWallpaper/settings.conf";

    QSettings settings(_confFilePath, QSettings::NativeFormat);

    settings.beginGroup("SETWALLPAPER");
    _Parameter = settings.value("used_desktop","").toInt();    
    _change_lockscreen_wallpaper = settings.value("change_lockscreen_wallpaper","").toInt();
    settings.endGroup();

    settings.beginGroup("SETTINGS");
    _pluginsdir = settings.value("pluginsdir","").toString();
    _WallpaperDir = settings.value("picturedir","").toString();
    settings.endGroup();

    settings.beginGroup("PROVIDER_SETTINGS");
    _lang = settings.value("lang","").toString();
    _selected_plugin = settings.value("selected_plugin","").toString();
    settings.endGroup();
}

void PhotoBrowser::setWallpaperFile()
{
    _get_specific_values=true;
    wallchanged = true;

    if(select_single_value("filename", _databaseFilePath)==true)
    {
        QModelIndex index = ui->listView->currentIndex();
        _thumb_filename = index.data(Qt::DisplayRole).toString();
        emit go_setWallpaper(_WallpaperDir+"/"+_output_value, _Parameter, _change_lockscreen_wallpaper);
        _photobrowser_specific_wallpaperfilename = _output_value;
    }

    _get_values();
    this->hide();
}

void PhotoBrowser::_get_values()
{
    QString connectionName = "PhotoBrowserConnection";

    // Scope sorgt dafür, dass db und query zerstört werden, bevor removeDatabase aufgerufen wird
    {
        QSqlDatabase udw_db;

        if(QSqlDatabase::contains(connectionName))
            udw_db = QSqlDatabase::database(connectionName);
        else
            udw_db = QSqlDatabase::addDatabase("QSQLITE", connectionName);

        udw_db.setDatabaseName(_databaseFilePath);

        if(!udw_db.open()) {
            qDebug() << "Database could not be opened:" << udw_db.lastError().text();
            return;
        }

        QSqlQuery udw_query(udw_db);

        if(!_get_specific_values)
        {
            // Load all thumbnail filenames into list
            if(!udw_query.exec("SELECT thumb_filename FROM udw_history")) {
                qDebug() << "Error querying thumbnails:" << udw_query.lastError().text();
                return;
            }

            while(udw_query.next())
            {
                QByteArray thumb = udw_query.value(0).toString().toUtf8();
                if(!_thumbfilelist.contains(thumb) && !thumb.isEmpty())
                {
                    _thumbfilelist.append(thumb);
                }
            }
        }
        else
        {
            // Load specific image details for the selected thumbnail
            QModelIndex index = ui->listView->currentIndex();
            _thumb_filename = index.data(Qt::DisplayRole).toString();

            udw_query.prepare("SELECT description, copyright, title, browser_url, size_width, size_height, filename "
                              "FROM udw_history WHERE thumb_filename = :thumb_filename");
            udw_query.bindValue(":thumb_filename", _thumb_filename);

            if(udw_query.exec() && udw_query.next())
            {
                _pb_copyright_description_photo = udw_query.value(0).toString();
                _pb_copyright = udw_query.value(1).toString();
                _pb_headline = udw_query.value(2).toString();
                _pb_copyright_link = udw_query.value(3).toString();
                _size_width = udw_query.value(4).toInt();
                _size_height = udw_query.value(5).toInt();
                _wallpaperfile = udw_query.value(6).toString();
            }
            else
            {
                qDebug() << "Error querying image details:" << udw_query.lastError().text();
            }

            _get_specific_values = false;
        }

        udw_query.finish();
    } // udw_query und udw_db gehen hier aus dem Scope -> Verbindung ist frei

    // Verbindung jetzt sicher entfernen, falls noch vorhanden
    if(QSqlDatabase::contains(connectionName)) {
        QSqlDatabase::removeDatabase(connectionName);
    }

    // Avoid crash if wallpaper filename is empty
    if (_wallpaperfile.isEmpty()) _wallpaperfile = "NULL";
}

void PhotoBrowser::on_listView_clicked()
{
    _display_details();
    ui->scrollArea->update();
}

void PhotoBrowser::_display_details()
{
    _label_Details->setText(tr("Details"));
    _label_Details->setStyleSheet("font: 14pt; font-weight: bold;");

    _label_headline->setStyleSheet("font-style: italic;");
    _label_headline->setText(tr("Please select a picture."));

    if(_first_run==false)
    {
        _get_specific_values=true;
        _get_values();

        _label_headline->setStyleSheet("font-weight: bold;");
        _label_headline->setText(tr("Title: "));

        _label_headline1->setText(_pb_headline);
        _label_headline1->setWordWrap(true);

        _label_description_and_copyright1->setStyleSheet("font-weight: bold;");
        _label_description_and_copyright1->setText(tr("Description: "));

        if(_pb_copyright.isEmpty())
        {
            _label_description_and_copyright->setText(_pb_copyright_description_photo);
        } else
        {
            _label_description_and_copyright->setText(_pb_copyright_description_photo+" - "+_pb_copyright);
        }

        _label_description_and_copyright->setWordWrap(true);

        _label_image_size1->setText(tr("Picture size:"));
        _label_image_size1->setStyleSheet("font-weight: bold;");
        _label_image_size->setText(QString::number(_size_width)+" x "+QString::number(_size_height));

        if(_mode==1)
        {
            _label_image_size1->setHidden(true);
            _label_image_size->setHidden(true);
        }

        _preview_text->setText(tr("Preview:"));
        _preview_text->setStyleSheet("font-weight: bold;");

        QImage _loadPrevImage;
        QImage _displayImage;

        if(_mode==0)
        {            
            _loadPrevImage.load(_WallpaperDir+"/"+_wallpaperfile);
        }
        else
        {
            _loadPrevImage.load(_thumbfiledir+"/"+_wallpaperfile);
        }

        _displayImage = _loadPrevImage.scaled(240,150, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
        _previewImageLabel->setPixmap(QPixmap::fromImage(_displayImage));

        QPixmap _gotoBing(":icons/Info.png");
        _getmoreInformations->setIcon(_gotoBing);
        _getmoreInformations->setVisible(true);
        _getmoreInformations->setMinimumWidth(240);
    }
    _first_run=false;
}

void PhotoBrowser::_getmoreInformationsButton_clicked()
{
    openURL();
}

void PhotoBrowser::openURL()
{
    QDesktopServices::openUrl(_pb_copyright_link);
}

void PhotoBrowser::download_thumb(int days)
{
    progressdialog = new QProgressDialog(tr("Download in progress..."), "", 0, 1, this);
    progressdialog->setWindowFlags(Qt::Dialog | Qt::CustomizeWindowHint | Qt::WindowTitleHint);
    progressdialog->setCancelButton(0);
    progressdialog->setWindowModality(Qt::WindowModal);
    progressdialog->adjustSize();
    progressdialog->move(screen()->geometry().center() - progressdialog->rect().center());
    progressdialog->show();
    progressdialog->setMaximum(days);

    QDate selected_date=ui->calendarWidget->selectedDate();

    QString year;
    QString month;
    QString day;

    _thumbfilelist.clear();

    for(int i=0; i<days; i++)
    {
        progressdialog->setValue(i);

        year=selected_date.toString("yyyy");
        month=selected_date.toString("MM");
        day=selected_date.toString("dd");

        basicinterface->get_picture(true, _thumbfiledir, _lang, year.toInt(), month.toInt(), day.toInt());

        selected_date=selected_date.addDays(-1);

        refresh();
    }

    refresh();
    progressdialog->setValue(days);
    progressdialog->close();
}

void PhotoBrowser::show()
{
    if(QDate::currentDate()<ui->calendarWidget->selectedDate())
    {
        QMessageBox MsgError;
        MsgError.setIcon(QMessageBox::Critical);
        MsgError.setText(tr("The entered date must not be in the future."));
        MsgError.exec();
    }

    if(QDate::currentDate()>=ui->calendarWidget->selectedDate())
    {
        if(QFile::exists(_databaseFilePath))
        {
            QFile::remove(_databaseFilePath);
        }

        QDir thumbfiledir(_thumbfiledir);
        if(thumbfiledir.exists())
        {
            thumbfiledir.removeRecursively();
            thumbfiledir.mkpath(_thumbfiledir);
        }

        if(create_temp_database()==true)
        {
            download_thumb(ui->spinBox->value());
        }
    }
}

bool PhotoBrowser::create_temp_database()
{
    QString connectionName = "PhotoBrowserCreateTempDBConnection";

    {
        QSqlDatabase udw_temp_db;

        if(QSqlDatabase::contains(connectionName))
            udw_temp_db = QSqlDatabase::database(connectionName);
        else
            udw_temp_db = QSqlDatabase::addDatabase("QSQLITE", connectionName);

        udw_temp_db.setDatabaseName(_databaseFilePath);

        if (!udw_temp_db.open())
        {
            qDebug() << "Database could not be opened:" << udw_temp_db.lastError().text();
            return false;
        }

        QSqlQuery udw_temp_query(udw_temp_db);

        if(!udw_temp_db.tables().contains(QLatin1String("udw_history")))
        {
            if(!udw_temp_query.exec("create table udw_history (id int primary key, "
                                    "date varchar(30), description varchar(900), copyright varchar(500),"
                                    "title varchar(500), provider varchar(100),"
                                    "filename varchar(150), browser_url varchar(500), size_width int, size_height int,"
                                    "thumb_filename varchar(150), pageid int, potd_date varchar(500))"))
            {
                qDebug() << "Failed to create table:" << udw_temp_query.lastError().text();
            }

            udw_temp_query.finish();
            udw_temp_query.clear();
        }
    } // udw_temp_db & udw_temp_query verlassen den Scope

    // Verbindung entfernen
    if(QSqlDatabase::contains(connectionName))
        QSqlDatabase::removeDatabase(connectionName);

    return true;
}

void PhotoBrowser::refresh()
{
    _get_values();

    QStandardItemModel *model = new QStandardItemModel(_thumbfilelist.size(), 1, this);

    for(int i=0;i<_thumbfilelist.size(); i++)
    {
        _thumbfilename = _thumbfilelist.at(i).toUtf8();
        QPixmap thumb_orig;
        thumb_orig.load(_thumbfiledir+"/"+_thumbfilename);
        QPixmap thumb=thumb_orig.scaled(QSize(_scaled_picture_width, _scaled_picture_height),  Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
        QIcon icon = thumb;
        QStandardItem *item = new QStandardItem(icon, _thumbfilelist.at(i).toUtf8());
        model->setItem(i, item);
    }
    ui->listView->setModel(model);
}

void PhotoBrowser::download_and_set()
{
    progressdialog = new QProgressDialog(tr("Working in progress..."), "", 0, 100);
    progressdialog->setWindowFlags(Qt::Dialog | Qt::CustomizeWindowHint | Qt::WindowTitleHint);
    progressdialog->setCancelButton(0);
    progressdialog->adjustSize();
    progressdialog->move(screen()->geometry().center() - progressdialog->rect().center());
    progressdialog->setWindowModality(Qt::WindowModal);
    progressdialog->show();

    wallchanged = true;

    QString year;
    QString month;
    QString day;

    if(select_single_value("potd_date", _databaseFilePath)==true)
    {
        progressdialog->setValue(50);
        QDate date=QDate::fromString(_output_value, "yyyyMMdd");

        year=date.toString("yyyy");
        month=date.toString("MM");
        day=date.toString("dd");

        basicinterface->get_picture(false, _WallpaperDir, _lang, year.toInt(), month.toInt(), day.toInt());

        // (temp)thumb_filename is the real filename.
        QModelIndex index = ui->listView->currentIndex();
        _thumb_filename = index.data(Qt::DisplayRole).toString();
        emit go_setWallpaper(_WallpaperDir+"/"+_thumb_filename, _Parameter, _change_lockscreen_wallpaper);

        if (progressdialog->wasCanceled())
        {
            progressdialog->cancel();
        }
        progressdialog->setValue(100);
    }

    deltempdir();

    progressdialog->close();

    this->hide();
}

bool PhotoBrowser::select_single_value(QString desired_column, QString db_filepath)
{
    // Only allow valid columns
    static const QStringList allowedColumns = {
        "filename", "description", "copyright", "title",
        "browser_url", "size_width", "size_height", "potd_date"
    };

    if(!allowedColumns.contains(desired_column)) {
        qDebug() << "Invalid column requested:" << desired_column;
        return false;
    }

    QString connectionName = "PhotoBrowserSelectConnection";

    // Scope sorgt dafür, dass db und query zerstört werden, bevor removeDatabase aufgerufen wird
    bool result = false; // Rückgabewert

    {
        QSqlDatabase udw_db;

        if(QSqlDatabase::contains(connectionName))
            udw_db = QSqlDatabase::database(connectionName);
        else
            udw_db = QSqlDatabase::addDatabase("QSQLITE", connectionName);

        udw_db.setDatabaseName(db_filepath);

        if(!udw_db.open()) {
            qDebug() << "Database could not be opened:" << udw_db.lastError().text();
            return false;
        }

        QSqlQuery udw_query(udw_db);
        udw_query.prepare(QString("SELECT %1 FROM udw_history WHERE thumb_filename = :thumb_filename").arg(desired_column));
        udw_query.bindValue(":thumb_filename", _thumb_filename);

        if(udw_query.exec() && udw_query.next())
        {
            _output_value = udw_query.value(0).toString();
            result = true;
        }
        else
        {
            qDebug() << "Error executing SELECT query:" << udw_query.lastError().text();
            result = false;
        }

        udw_query.finish();
    } // udw_query und udw_db gehen hier aus dem Scope -> Verbindung ist frei

    // Verbindung jetzt sicher entfernen, falls noch vorhanden
    if(QSqlDatabase::contains(connectionName)) {
        QSqlDatabase::removeDatabase(connectionName);
    }

    return result;
}

void PhotoBrowser::deltempdir()
{
    QDir tempdir(_tempdirpath);
    if(tempdir.exists())
    {
        tempdir.removeRecursively();
    }
}

bool PhotoBrowser::loadPlugin(QString _pluginfile)
{
    QPluginLoader pluginLoader;

    pluginLoader.setFileName(_pluginfile);

    QObject *plugin = pluginLoader.instance();
    if (plugin)
    {
        basicinterface = qobject_cast<BasicInterface *>(plugin);

        if (basicinterface)
        {
            connect(plugin, SIGNAL(download_successful(QString, QString, QString,
                                                      QString, QString,
                                                      QString, QString,
                                                      int, int, int,
                                                      bool, QString)),
                    this, SLOT(_add_record(QString, QString, QString,
                                     QString, QString,
                                     QString, QString,
                                     int, int, int,
                                     bool, QString)));

            return true;
        }
        pluginLoader.unload();
    }
    return false;
}

void PhotoBrowser::_add_record(QString _description,
                               QString _copyright,
                               QString _provider,
                               QString _title_headline,
                               QString _filename,
                               QString _browser_url,
                               QString _thumb_filename,
                               int _size_height,
                               int _size_width,
                               int pageid,
                               bool tempdatabase,
                               QString potd_date)
{
    QString connectionName = "PhotoBrowserAddRecordConnection";
    QString _tempdatabaseFilePath = QDir::homePath()+"/.UltimateDailyWallpaper/temp/temp_udw_database.sqlite";

    {
        QSqlDatabase udw_db;

        if(QSqlDatabase::contains(connectionName))
            udw_db = QSqlDatabase::database(connectionName);
        else
            udw_db = QSqlDatabase::addDatabase("QSQLITE", connectionName);

        if(tempdatabase)
            udw_db.setDatabaseName(_tempdatabaseFilePath);
        else
            udw_db.setDatabaseName(_databaseFilePath);

        if(!udw_db.open()) {
            qDebug() << "Database could not be opened:" << udw_db.lastError().text();
            return;
        }

        QSqlQuery udw_query(udw_db);

        if(tempdatabase)
        {
            udw_query.prepare("INSERT INTO udw_history (id, date, description, copyright, title, provider, filename, browser_url, size_width, size_height, thumb_filename, pageid, potd_date)"
                              "VALUES (:id, :date, :description, :copyright, :title, :provider, :filename, :browser_url, :size_width, :size_height, :thumb_filename, :pageid, :potd_date)");
        }
        else
        {
            udw_query.prepare("INSERT INTO udw_history (id, date, description, copyright, title, provider, filename, browser_url, size_width, size_height, thumb_filename, pageid)"
                              "VALUES (:id, :date, :description, :copyright, :title, :provider, :filename, :browser_url, :size_width, :size_height, :thumb_filename, :pageid)");
        }

        udw_query.bindValue(":id", QDateTime::currentDateTime().toString("yyyyMMddHHmmsszzz"));
        udw_query.bindValue(":date", QDate::currentDate().toString("yyyyMMdd"));
        udw_query.bindValue(":description", _description);
        udw_query.bindValue(":copyright", _copyright);
        udw_query.bindValue(":title", _title_headline);
        udw_query.bindValue(":provider", _provider);
        udw_query.bindValue(":filename", _filename);
        udw_query.bindValue(":browser_url", _browser_url);
        udw_query.bindValue(":size_width", _size_width);
        udw_query.bindValue(":size_height", _size_height);
        udw_query.bindValue(":thumb_filename", _thumb_filename);
        udw_query.bindValue(":pageid", pageid);

        if(tempdatabase)
            udw_query.bindValue(":potd_date", potd_date);

        if(!udw_query.exec())
            qDebug() << udw_query.lastError();

        udw_query.finish();
        udw_query.clear();
    } // udw_query & udw_db verlassen den Scope -> Verbindung frei

    // Verbindung entfernen
    if(QSqlDatabase::contains(connectionName))
    {
        QSqlDatabase::removeDatabase(connectionName);
    }
}
