/*
 * myGtkMenu - read a description file and generate a menu.
 * Copyright (C) 2004-2021 John Vorthman
 *
 * -------------------------------------------------------------------------
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, version 2, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59
 * Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * -------------------------------------------------------------------------
 *
 * This program requires GTK+ 3 libraries.
 *
 * main.c is used to generate myGtkMenu.
 *
 * gcc -Wall -o myGtkMenu main.c `pkg-config gtk+-3.0 --cflags --libs`
 *
 */


/* I would like to thank Jean-Pierre Demailly for showing me how to replace
 * the deprecated function gtk_menu_popup and how to easily change the menu
 * font size. I modified his code so if it looks wrong here, don't blame him.
 */


#include <stdio.h>
#include <gtk/gtk.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <ctype.h>
#include <time.h>
#include <fcntl.h>

#define MAX_LINE_LENGTH 200
#define MAX_MENU_ENTRIES 1000
#define MAX_SUBMENU_DEPTH 4
#define MAX_ICON_SIZE 200
#define MIN_ICON_SIZE 10
#define LOCK_NAME "myGtkMenu_lockfile"
#define BUFFER_LEN 2048
#define MAX_HOME_LEN 100

// Function prototypes
int ReadLine ();
static void RunItem (char *Cmd);
static void QuitMenu (char *Msg);
gboolean Get2Numbers (char *data);
int Already_running (void);
static void Expand_Tilde (char *txt, int maxLen);

// Global variables
char Line[MAX_LINE_LENGTH], data[BUFFER_LEN];
int depth, lineNum, menuX, menuY;
FILE *pFile;
GtkWidget *menu[MAX_SUBMENU_DEPTH];
char *HelpMsg =
    "\nUsage: myGtkMenu MenuDescriptionFilename"
    "\n"
    "\nPurpose: read a description file and display a menu."
    "\n"
    "\nTestMenu.txt is an example input file."
    "\nSee TestMenu.txt for help."
    "\n"
    "\nmyGtkMenu version 1.4, Copyright (C) 2004-2021 John Vorthman."
    "\nmyGtkMenu comes with ABSOLUTELY NO WARRANTY - see license file."
    "\n"
    "\n";


// ----------------------------------------------------------------------
int main (int argc, char *argv[]) {
// ----------------------------------------------------------------------

    char *Filename;
    int Mode;       // What kind of input are we looking for?
    int Kind;       // Type of input actually read
    int curDepth;   // Root menu is depth = 0
    int curItem;    // Count number of menu entries
    gint w, h;      // Size of menu icon
    gboolean bSetMenuPos = FALSE;
    int i;
    char Item[MAX_LINE_LENGTH], Cmd[MAX_MENU_ENTRIES][MAX_LINE_LENGTH];
    GtkWidget *menuItem, *SubmenuItem = NULL;
    GError *error = NULL;
    struct stat statbuf;
    GtkWidget *box;
    GtkWidget *icon;
    GtkWidget *spacer;
    GtkWidget *label;   
    GdkPixbuf *pixbuf;
    GdkWindow *Screen;
    GdkEvent  *event;
    GtkCssProvider  *css;
    GtkStyleContext *context;
    gint fontsize = -1;     // fontsize = -1 means use default
    GdkRectangle rect;


    // Some hot keys (keybindings) get carried away and start
    // many instances. Will try to use a lock file so that
    // only one instance of this program runs at a given time.
    // If this check causes you problems, comment it out.
    int i_Already_running = Already_running();
    if (i_Already_running == 1) {
        fprintf(stderr, "All ready running, will quit.\n");
        return EXIT_FAILURE;
    }
    else if (i_Already_running == 2) {
        fprintf(stderr, "%s: Error in routine Already_running(), "
                        "will quit.\n", argv[0]);
        return EXIT_FAILURE;
    }

    if (!gtk_init_check (&argc, &argv)) {
        g_print("Error, cannot initialize gtk.\n");
        exit (EXIT_FAILURE);
    }

    if ((argc > 1) && (argv[1][0] == '-')) {
        g_print (HelpMsg);
        exit (EXIT_SUCCESS);
    }

    if (argc < 2) {
        g_print (HelpMsg);
        g_print ("Missing the menu-description filename.\n");
        g_print ("Will try to open TestMenu.txt.\n");
        memset (data, 0, sizeof (data));
        strncpy (data, argv[0], sizeof (data) - 1); // Get myGtkMenu path
        i = strlen (data);
        while ((i > 0) && (data[i] != '/'))
            data[i--] = '\0';   // Remove filename
        if ((i > 0) && (i < sizeof (data) - 14)) {
            strcat (data, "TestMenu.txt");
            Filename = data;
        }
        else
            Filename = "TestMenu.txt";
    }
    else {
        strncpy(data, argv[1], sizeof (data) - 1);
        Expand_Tilde(data, sizeof (data) - 1);
        Filename = (char *) data;
    }

    g_print ("Reading the file: %s\n", Filename);

    pFile = fopen (Filename, "r");
    if (pFile == NULL) {
        g_print ("Cannot open the file.\n");
        exit (EXIT_FAILURE);
    }

    menu[0] = gtk_menu_new ();
    gtk_menu_set_reserve_toggle_size(GTK_MENU(menu[0]), 0);
    if (!gtk_icon_size_lookup (GTK_ICON_SIZE_BUTTON, &w, &h)) { // Default
        w = 30;
        h = 30;
    };
    g_print("Default icon size = %d pixels.\n", h);


    curItem = 0;
    Mode = 0;
    curDepth = 0;
    while ((Kind = ReadLine()) != 0) {  // Read next line and get 'Kind'
                                        // of keyword
        if (Kind == -1)
            Mode = -1;  // Error parsing file

        if (depth > curDepth) {
            g_print ("Keyword found at incorrect indentation.\n");
            Mode = -1;
        }
        else if (depth < curDepth) {    // Close submenu
            curDepth = depth;
        }

        if (Mode == 0) {    // Starting new sequence. Whats next?
            if (Kind == 1)
                Mode = 1;   // New item
            else if (Kind == 4)
                Mode = 4;   // New submenu
            else if (Kind == 5)
                Mode = 6;   // New separator
            else if (Kind == 6)
                Mode = 7;   // New icon size
            else if (Kind == 7)
                Mode = 8;   // Set menu position
            else if (Kind == 8)
                Mode = 9;   // Set font size
            else {          // Problems
                g_print ("Keyword out of order.\n");
                Mode = -1;
            }
        }

        switch (Mode) {

        case 1: // item=     ;starting new menu item
            if (curItem >= MAX_MENU_ENTRIES) {
                g_print ("Exceeded maximum number of menu items.\n");
                Mode = -1;
                break;
            }
            strcpy (Item, data);
            Mode = 2;
            break;

        case 2: // cmd=
            if (Kind != 2) {    // Problems if keyword 'cmd=' not found
                g_print ("Missing keyword 'cmd=' (after 'item=').\n");
                Mode = -1;
                break;
            }
            Expand_Tilde(data, sizeof (data) - 1);
            strcpy (Cmd[curItem], data);
            Mode = 3;
            break;

        case 3: // icon=
            if (Kind != 3) {    // Problems if keyword 'icon=' not found
                g_print ("Missing keyword 'icon=' (after 'cmd=').\n");
                Mode = -1;
                break;
            }
            // Create new menu item
            menuItem = gtk_menu_item_new ();
            box = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 6);
            gtk_widget_set_size_request(box, -1, h); /////
            label = gtk_label_new (NULL);
            gtk_label_set_markup_with_mnemonic(GTK_LABEL(label), Item);

            if ((strlen(data) == 0) || (strncasecmp (data, "NULL", 4) == 0)) {
                icon = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 0); // Fake a blank icon
                gtk_widget_set_size_request(icon, h, h);
                goto noicon3;
            }

            Expand_Tilde(data, sizeof (data) - 1);
            stat (data, &statbuf); // If data is a dir name, program can hang.
            if (!S_ISREG (statbuf.st_mode)) {
                g_print ("\nBad file name\n");
                Mode = -1;
                break;
            }

            pixbuf = gdk_pixbuf_new_from_file_at_scale (data, h, h, TRUE, &error);
            if (pixbuf == NULL) {
                g_print ("\n%s\n", error->message);
                g_error_free (error);
                error = NULL;
                Mode = -1;
                break;
            }           
            icon = gtk_image_new_from_pixbuf (pixbuf);
            
            noicon3:

            spacer = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 0); // Add to right side of menu
            gtk_widget_set_size_request(spacer, h, h);


            // Insert icon and label
            gtk_container_add (GTK_CONTAINER (box), icon);
            gtk_container_add (GTK_CONTAINER (box), label);
            gtk_container_add (GTK_CONTAINER (box), spacer);
            gtk_container_add (GTK_CONTAINER (menuItem), box);
            gtk_menu_shell_append (GTK_MENU_SHELL (menu[curDepth]), menuItem);

            g_signal_connect_swapped (menuItem, "activate",
                          G_CALLBACK (RunItem), Cmd[curItem]);
            curItem++;
            Mode = 0;
            break;

        case 4: // Start submenu
            if (curDepth >= MAX_SUBMENU_DEPTH) {
                g_print ("Maximum submenu depth exceeded.\n");
                Mode = -1;
                break;
            }
            SubmenuItem = gtk_menu_item_new();
            gtk_menu_shell_append(GTK_MENU_SHELL(menu[curDepth]), SubmenuItem);
            curDepth++;
            menu[curDepth] = gtk_menu_new ();
            box = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 6);
            gtk_widget_set_size_request(box, -1, h); /////
            label = gtk_label_new (NULL);
            gtk_label_set_markup_with_mnemonic(GTK_LABEL(label), data);
            gtk_menu_set_reserve_toggle_size(GTK_MENU(menu[curDepth]), 0);
            gtk_menu_item_set_submenu (GTK_MENU_ITEM (SubmenuItem), menu[curDepth]);
            Mode = 5;
            break;

        case 5: // Add image and label to new submenu
            if (Kind != 3) {    // Problems if keyword 'icon=' not found
                g_print ("Missing keyword 'icon=' (after 'submenu=').\n");
                Mode = -1;
                break;
            }
            Mode = 0;

            if ((strlen(data) == 0) || (strncasecmp (data, "NULL", 4) == 0)) {
                icon = gtk_box_new (GTK_ORIENTATION_HORIZONTAL, 0); // Fake a blank icon
                gtk_widget_set_size_request(icon, h, h);
                goto noicon5;
            }

            Expand_Tilde(data, sizeof (data) - 1);
            stat (data, &statbuf); // If data is a dir name, program can hang.
            if (!S_ISREG (statbuf.st_mode)) {
                g_print ("\nBad file name\n");
                Mode = -1;
                break;
            }
         
            pixbuf = gdk_pixbuf_new_from_file_at_scale (data, h, h, TRUE, &error);
            if (pixbuf == NULL) {
                g_print ("\n%s\n", error->message);
                g_error_free (error);
                error = NULL;
                Mode = -1;
                break;
            }
            icon = gtk_image_new_from_pixbuf (pixbuf);

            noicon5:

            // Insert icon and label
            gtk_container_add (GTK_CONTAINER (box), icon);
            gtk_container_add (GTK_CONTAINER (box), label);
            gtk_container_add (GTK_CONTAINER (SubmenuItem), box);
            break;

        case 6: // Insert separator into menu
            menuItem = gtk_separator_menu_item_new ();
            gtk_menu_shell_append (GTK_MENU_SHELL (menu[curDepth]), menuItem);
            Mode = 0;
            break;

        case 7: // Change menu icon size
            i = atoi (data);
            if ((i < MIN_ICON_SIZE) || (i > MAX_ICON_SIZE)) {
                g_print ("Illegal size for menu icon.\n");
                Mode = -1;
                break;
            }
            w = i;
            h = i;
            g_print ("New icon size = %d.\n", w);
            Mode = 0;
            break;

        case 8: // Set menu position
            if (Get2Numbers (data)) {
                bSetMenuPos = TRUE;
                g_print ("Menu position = %d, %d.\n", menuX, menuY);
                Mode = 0;
            }
            else {
                g_print ("Error reading menu position.\n");
                Mode = -1;
            }
            break;

        case 9: // Change font size
            i = atoi (data);
            if ((i != 0) && ((abs(i) < 5) || (abs(i) > 50))) {
                g_print ("Illegal font size.\n");
                Mode = -1;
                break;
            }
            fontsize = i;
            if (fontsize > 0)
                g_print ("Font size = %d points.\n", i);
            else
                g_print ("Font size = %d pixels.\n", -i);
            Mode = 0;
            break;

        default:
            Mode = -1;
        }   // switch

        if (Mode == -1) {   // Error parsing file
            // Placed here so that ReadLine is not called again
            g_print ("Error at line # %d\n", lineNum);
            g_print (">>>%s\n", Line);
            break;  // Quit reading file
        }
    }   // while

    fclose (pFile);

    if (curItem == 0) {
        g_print("Error, no menu items found!\n");
        exit (EXIT_FAILURE);
    }

    if (fontsize == 0) fontsize = (int)(-h * 17.0 / 24.0);
    
    if (fontsize != -1) {
        if (fontsize > 0)
            snprintf(data, sizeof(data), "menu { font-size: %dpt }", fontsize);
        else
            snprintf(data, sizeof(data), "menu { font-size: %dpx }", -fontsize);
        css = gtk_css_provider_new();
        gtk_css_provider_load_from_data(css, data, -1, NULL);
        context = gtk_widget_get_style_context(menu[0]);
        gtk_style_context_add_provider
          (context, GTK_STYLE_PROVIDER(css), GTK_STYLE_PROVIDER_PRIORITY_APPLICATION);
        g_object_unref (css);
    }


    g_signal_connect_swapped (menu[0], "deactivate",
                  G_CALLBACK (QuitMenu), NULL);

    gtk_widget_show_all (menu[0]);

    event = gdk_event_new(GDK_NOTHING); // create a fake event
    Screen = gdk_screen_get_root_window(gdk_screen_get_default());
    ((GdkEventAny*)event)->window = Screen; 

    rect.x = menuX;
    rect.y = menuY;         
  
    while (!gtk_widget_get_visible(menu[0])) { // Keep trying until startup
        usleep(100000);                        // button (or key) is released
        if (bSetMenuPos)
            gtk_menu_popup_at_rect (GTK_MENU(menu[0]), Screen, &rect,
                GDK_GRAVITY_NORTH_WEST, GDK_GRAVITY_NORTH_WEST, event);
        else
            gtk_menu_popup_at_pointer (GTK_MENU(menu[0]), event);

        gtk_main_iteration();
    }

    gtk_main ();

    return 0;

}   // int main

// ----------------------------------------------------------------------
gboolean Get2Numbers (char *data) {
// ----------------------------------------------------------------------
    int n, i;

    n = strlen (data);
    if ((n == 0) | !isdigit (data[0]))
        return FALSE;
    menuX = atoi (data);
    i = 0;

    // Skip over first number
    while (isdigit (data[i])) {
        i++;
        if (i == n)
            return FALSE;
    }

    // Find start of the next number
    while (!isdigit (data[i])) {
        i++;
        if (i == n) return FALSE;
    }

    menuY = atoi (&data[i]);
    return TRUE;
}   // gboolean Get2Numbers

// ----------------------------------------------------------------------
static void RunItem (char *Cmd) {
// ----------------------------------------------------------------------
    GError *error = NULL;

    g_print ("Run: %s\n", Cmd);
    if (!Cmd) return;

    if (!g_spawn_command_line_async(Cmd, &error)) {
        g_print ("Error running command.\n");
        g_print ("%s\n", error->message);
        g_error_free (error);
    }
    gtk_main_quit ();
}   // static void RunItem

// ----------------------------------------------------------------------
static void QuitMenu (char *Msg) {
// ----------------------------------------------------------------------
    g_print ("Menu was deactivated.\n");

    gtk_main_quit ();
}   // static void QuitMenu

// ----------------------------------------------------------------------
int ReadLine () {
// ----------------------------------------------------------------------
    // Return kind of line, menu depth, and stripped text
    // return(-1) = Error, return(0) = EOF, return(>0) = keyword
    char *chop;
    int i, len, count, Kind;
    char tmp[MAX_LINE_LENGTH];
    char *str1, *str2;

    len = 0;
    while (len == 0) {
        // Read one line.
        if (fgets (Line, MAX_LINE_LENGTH, pFile) == NULL)
            return (0);
        strcpy (tmp, Line);
        lineNum++;

        // Remove comments
        chop = strchr (tmp, '#');
        if (chop != 0)
            *chop = '\0';

        len = strlen (tmp);

        // Remove trailing spaces & CR
        if (len > 0) {
            chop = &tmp[len - 1];
            while ((chop >= tmp) && (isspace (*chop) != 0)) {
                *chop = '\0';
                chop--;
            }
            len = strlen (tmp);
        }
    };

    // Big error?
    if (len >= MAX_LINE_LENGTH) {
        strncpy (data, tmp, MAX_LINE_LENGTH);
        data[MAX_LINE_LENGTH] = '\0';
        return (-1);
    }

    count = 0;

    // Calculate menu depth
    for (i = 0; i < len; i++) {
        if (tmp[i] == ' ')
            count += 1;
        else if (tmp[i] == '\t')    // Tab character = 4 spaces
            count += 4;
        else
            break;
    };
    depth = count / 4;

    // Remove leading white space
    if (count > 0) {
        str1 = tmp;
        str2 = tmp;
        while ((*str2 == ' ') || (*str2 == '\t')) {
            str2++;
            len--;
        }
        for (i = 0; i <= len; i++)
            *str1++ = *str2++;
    }

    if (strncasecmp (tmp, "separator", 9) == 0) {       // Found 'separator'
        strcpy (data, tmp);
        return (5);
    }
    else if (strchr (tmp, '=') == NULL) {               // Its a bad line
        strcpy (data, tmp);
        return (-1);
    }
    else if (strncasecmp (tmp, "iconsize", 8) == 0) {   // Found 'iconsize'
        Kind = 6;
    }
    else if (strncasecmp (tmp, "item", 4) == 0) {       // Found 'item'
        Kind = 1;
    }
    else if (strncasecmp (tmp, "cmd", 3) == 0) {        // Found 'cmd'
        Kind = 2;
    }
    else if (strncasecmp (tmp, "icon", 4) == 0) {       // Found 'icon'
        Kind = 3;
    }
    else if (strncasecmp (tmp, "submenu", 7) == 0) {    // Found 'submenu'
        Kind = 4;
    }
    else if (strncasecmp (tmp, "menupos", 7) == 0) {    // Found 'menupos'
        Kind = 7;
    }
    else if (strncasecmp (tmp, "fontsize", 8) == 0) {   // Found 'fontsize'
        Kind = 8;
    }
    else {                                              // Its a bad line
        strcpy (data, tmp);
        return (-1);
    }

    // Remove keywords and white space
    str2 = strchr (tmp, '=') + 1;
    while ((*str2 == ' ') || (*str2 == '\t'))
        str2++;
    strcpy (data, str2);


    return (Kind);
}   // int ReadLine

// ---------------------------------------------------------------------
int Already_running (void)
// ---------------------------------------------------------------------
{
    struct flock fl;
    int fd;
    char *home;
    int n;
    int ret;
    char *Lock_path;

    // Use lock file 'LOCK_NAME' to see if program is already running.
    // 0 = No, 1 = Yes, 2 = Error

    // Place lock in the home directory of user
    // If user messes with "HOME", will probably have problems
    home = (char *) getenv("HOME");

    Lock_path = malloc(BUFFER_LEN + 1);
    n = snprintf(Lock_path, BUFFER_LEN, "%s/.%s", home, LOCK_NAME);

    if (n > BUFFER_LEN || Lock_path == NULL) {
        fprintf(stderr, "Error, path name too long: %s.\n", Lock_path);
        ret = 2;
        goto Done;
     }

    fd = open(Lock_path, O_RDWR | O_CREAT, 0600);
    if (fd < 0) {
        fprintf(stderr, "Error opening %s.\n", Lock_path);
        ret = 2;
        goto Done;
     }

    fl.l_start = 0;
    fl.l_len = 0;
    fl.l_type = F_WRLCK;
    fl.l_whence = SEEK_SET;

    if (fcntl(fd, F_SETLK, &fl) < 0)
        ret = 1;
    else
        ret = 0;

Done:
    free(Lock_path);

    return ret;
} // Already_running


// ----------------------------------------------------------------------
static void Expand_Tilde (char *txt, int maxLen) {
// ----------------------------------------------------------------------
    // Replace ~ in text

    static char home[MAX_HOME_LEN+1];
    static char buffer[BUFFER_LEN+1];
    static int  homeLen = 0; // Length of $HOME
    int len_txt;
    int i, j; // index to txt & buffer

    if (! strstr(txt, "~/")) return;

    if (maxLen > sizeof(buffer)) maxLen = sizeof(buffer);

    if (homeLen == 0) {
        strncpy (home, (char *) getenv("HOME"), MAX_HOME_LEN);
        homeLen = strlen(home);
        if (homeLen < strlen(getenv("HOME"))) {
            g_print("Error, length of $HOME > %d.\n", MAX_HOME_LEN);
            homeLen = 0;
            return;
        }
    }

    len_txt = strlen(txt);

    if (len_txt + homeLen >= maxLen) {
        g_print("Error, cannot expand '~', string is too long.\n");
        return;
    }

    i = j = 0;

    if ((txt[0] == '~') && (txt[1] == '/')) {
        strncpy(buffer, home, maxLen);
        i += 1;
        j += homeLen;
    } 
    else 
        buffer[j++] = txt[i++];
      
    while (i < len_txt) {
        if ((txt[i-1] == ' ') && (txt[i] == '~') && (txt[i+1] == '/')) {
            strncpy(&buffer[j], home, maxLen-j);
            i += 1;
            j += homeLen;
        }     
        else 
          buffer[j++] = txt[i++];

        if (j >= maxLen) {
            g_print("Error, cannot expand '~', string is too long.\n");
            return;
        }         
    }     

    strncpy(txt, buffer, maxLen);

return;
}


