#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/* */
/*  Copyright (C)  1999-2011 Edscott Wilson Garcia under GNU GPL
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; 
*/

static gint
update_combo_model(diff_t *diff_p, gboolean left, GList *alt_list);
static list_t *
make_list_record(const gchar *path);
static gint
make_list_from_directory(GList **list, const gchar *folder, gint *patchmake_count);
static void
clear_list(GList **list);

/* memcpy is necesary because patch file may contain binary data */
#ifdef __GNUC__
/* memcpy is a GNU extension.*/
# define MEMCPY memcpy
#else
/* memcpy is a GNU extension.*/
static void *
MEMCPY (void *dest, const void *src, size_t n) {
    char *destC,
     *srcC;
    size_t i;

    destC = (char *)dest;
    srcC = (char *)src;
    for(i = 0; i < n; i++)
        destC[i] = srcC[i];
    return dest;
}
#endif

static
gchar *
get_rcfile (void) {
    gchar *rcdir = g_build_filename (USER_RFM_DIR, NULL);
    gchar *rcfile = g_build_filename (USER_RFM_DIR, "rodent-diff-rc", NULL);
    g_mkdir_with_parents (rcdir, 0700);
    g_free(rcdir);
    return rcfile;
}
#if GTK_MAJOR_VERSION==3 && GTK_MINOR_VERSION>=4
static
void
get_defaults (diff_t *diff_p) {
    FILE *defaults;
    gchar *rcfile = get_rcfile ();
    //unlink(rcfile);
    defaults = fopen (rcfile, "r");
    g_free (rcfile);


#ifdef GN_PATCH
    verbose = 0;
#endif
    if(!defaults) {
	diff_p->diff_flags |= (FILLEDP|SHOW_LINEN|MINIMAL);
    } else {
        char line[256],
         *word,
         *value;
        while((!feof (defaults)) && (fgets (line, 255, defaults))) {
            line[255] = 0;
            if(line[0] == '#')
                continue;
            word = strtok (line, ":");
            if(!word) continue;
            value = word + strlen (word) + 1;
            TRACE("word=%s value=%s\n",word,value); 
            if(strstr (word, "diff_flags"))
                diff_p->diff_flags = atof (value);
        }
        fclose (defaults);
    }
    TRACE("read defaults: 0x%x\n", diff_p->patchmake_flags);

} 

void
save_defaults (diff_t *diff_p) {
    FILE *defaults;
    gchar *rcfile = get_rcfile ();

    defaults = fopen (rcfile, "w");
    if(!defaults) {
	DBG("cannot save options: %s (%s)\n",
		rcfile, strerror(errno));
	g_free(rcfile);
        return;
    }
    g_free(rcfile);

    fprintf (defaults, "# file created by rodent-diff, if removed rodent-diff returns to defaults.\n");
    fprintf (defaults, "diff_flags : %d\n", diff_p->diff_flags);

    fclose (defaults);
    TRACE("saved defaults: 0x%x\n", diff_p->patchmake_flags);
    return;
}
#else
static
void
get_defaults (diff_t *diff_p) {
    FILE *defaults;
    gchar *rcfile = get_rcfile ();
    //unlink(rcfile);
    defaults = fopen (rcfile, "r");
    g_free (rcfile);


#ifdef GN_PATCH
    verbose = 0;
#endif
    if(!defaults) {
	diff_p->diff_flags |= (FILLEDP|SHOW_LINEN|MINIMAL);
    } else {
        char line[256];
        const gchar *word;
        const gchar  *value;
        while((!feof (defaults)) && (fgets (line, 255, defaults))) {
            line[255] = 0;
            if(line[0] == '#') continue;
            word = strtok (line, ":");
            if(!word) continue;
            value = word + strlen (word) + 1;
            TRACE("word=%s value=%s\n",word,value); 

            if(strstr (word, "diff_flags"))
                diff_p->diff_flags = atoi (value);
        }
        fclose (defaults);
    }
    TRACE("read defaults: 0x%x\n", diff_p->patchmake_flags);

} 

void
save_defaults (diff_t *diff_p) {
    FILE *defaults;
    gchar *rcfile = get_rcfile ();

    defaults = fopen (rcfile, "w");
    if(!defaults) {
	DBG("cannot save options: %s (%s)\n",
		rcfile, strerror(errno));
	g_free(rcfile);
        return;
    }
    g_free(rcfile);

    fprintf (defaults, "# file created by rodent-diff, if removed rodent-diff returns to defaults.\n");
    fprintf (defaults, "diff_flags : %d\n", diff_p->diff_flags);

    fclose (defaults);
    TRACE("saved defaults: 0x%x\n", diff_p->patchmake_flags);
    return;
}

#endif

static void *
diff_init (void *data) {
    diff_t *diff_p=(diff_t *)malloc(sizeof(diff_t));
    memset(diff_p, 0, sizeof(diff_t));
    // initial sizes for text files, sizes in line numbers and average char length 
    get_defaults(diff_p);
    diff_p->sizeR = 30;
    diff_p->sizeL = 30;
    diff_p->sizeH = 15;
    return (void *)diff_p;
  
}



static gchar *
fileselect (gchar *title, gint action) {
    gchar *retval=NULL;
    GtkWidget *dialog = gtk_file_chooser_dialog_new (title,
                                NULL,
                                action,
				_("Cancel"),
                                GTK_RESPONSE_CANCEL,
                                _("Open"),
                                GTK_RESPONSE_ACCEPT,
				NULL);
    //gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER(dialog), TRUE);
    //gtk_file_chooser_set_show_hidden (GTK_FILE_CHOOSER(dialog), TRUE);

    if (action==GTK_FILE_CHOOSER_ACTION_SAVE) {
	gtk_file_chooser_set_do_overwrite_confirmation(GTK_FILE_CHOOSER(dialog), TRUE);
	gtk_file_chooser_set_create_folders (GTK_FILE_CHOOSER(dialog), TRUE);

    } else {
	gtk_file_chooser_set_create_folders (GTK_FILE_CHOOSER(dialog), FALSE);
	//g_object_set(G_OBJECT(dialog), "create-folders",FALSE,NULL);
    }


    gchar *wd=g_get_current_dir ();
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dialog), wd);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT){
	//GSList *gtk_file_chooser_get_uris (GtkFileChooser *chooser);
	retval=gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    } 
    g_free(wd);
    gtk_widget_destroy (dialog);
   return retval;
}

static gboolean
populate_hash (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data){
    GHashTable *hash = data;
    list_t *list_p;
    gtk_tree_model_get (model, iter, 
	    LIST_P_COLUMN, &list_p,
	    -1);
    if (!list_p){
	TRACE( "this is wrong\n");
	return FALSE;
    }
	TRACE( "hashing %s\n", list_p->name);
    list_p->treepath =  gtk_tree_model_get_path (model, iter);
    g_hash_table_insert(hash, g_strdup(list_p->name), list_p);

    return FALSE;

}

static
gint
update_combo_model(diff_t *diff_p, gboolean left, GList *alt_list){
    GtkWidget *combo=(left)?
	diff_p->combo_left:diff_p->combo_right;
    const gchar *file=(left)?
	diff_p->left_file:diff_p->right_file;
    GList **list=(left)?
	&(diff_p->left_list):&(diff_p->right_list);
    gint count=0; 
    if (!file && !alt_list) return 0;
    
    diff_p->diff_flags |= DIFF_UPDATING_MODEL;
    
    g_mutex_lock(diff_p->serial_mutex);
    if (alt_list) {
	clear_list(list);
	GList *tmp = alt_list;
	count = 0;
	for (; tmp && tmp->data; tmp=tmp->next){
	    gchar *file = tmp->data;
	    //if (!rfm_g_file_test(file, G_FILE_TEST_IS_REGULAR)) continue;
	    if (!rfm_g_file_test(file, G_FILE_TEST_EXISTS)) continue;
	    list_t *list_p=make_list_record(file);
	    g_free(file);
	    *list = g_list_append(*list, list_p);	
	    count++;    
	}
	g_list_free(alt_list);
    } else {
      if (rfm_g_file_test(file, G_FILE_TEST_IS_DIR)) {
	count = 
	    make_list_from_directory(list, file,
		   (left)?
		   &(diff_p->left_patchmake_count):&(diff_p->right_patchmake_count)
		   );     
      } else {
	clear_list(list);
	if (rfm_g_file_test(file, G_FILE_TEST_IS_REGULAR)){
	    list_t *list_p=make_list_record(file);
	    *list = g_list_append(*list, list_p);	
	    count=1;    
	}
      }
    }

    if (diff_p->hash[(left)?0:1]){
	g_hash_table_destroy(diff_p->hash[(left)?0:1]);
    }
    diff_p->hash[(left)?0:1] = 
	g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

    GtkListStore *list_store=
	GTK_LIST_STORE(gtk_combo_box_get_model(GTK_COMBO_BOX(combo)));
    gtk_list_store_clear(list_store);
    if (count==0 && *list==NULL) {
	DBG("This should not happen. If no files found or only binary files found, a message should be generated.\n");
	diff_p->serial++;
	g_mutex_unlock(diff_p->serial_mutex);
	return count;
    }


    GList *tmp;
    GtkTreeIter iter;
    for (tmp=*list; tmp && tmp->data; tmp=tmp->next) {
	list_t *list_p=tmp->data;
 	gtk_list_store_append (list_store, &iter);
 	gtk_list_store_set (list_store, &iter, 
		PIXBUF_COLUMN, list_p->pixbuf,
		BASENAME_COLUMN, list_p->name, 
		PATH_COLUMN, list_p->path, 
		DATE_COLUMN, list_p->date,
		LIST_P_COLUMN, list_p,
		-1);
    }
    // Populate hash table here, once treemodel is sorted...
    gtk_tree_model_foreach(GTK_TREE_MODEL(list_store),
	    populate_hash, diff_p->hash[(left)?0:1]);
    gtk_combo_box_set_active (GTK_COMBO_BOX(combo), 0);

    diff_p->serial++;
    diff_p->diff_flags &= (DIFF_UPDATING_MODEL ^ 0xffffffff);
    
    g_mutex_unlock(diff_p->serial_mutex);
    rfm_thread_create("do_quick_diff", do_quick_diff, diff_p, FALSE);

    return count;
}


static
void
clear_list(GList **list){
    // cleanup old list
    GList *tmp;
    for (tmp = *list; tmp && tmp->data; tmp=tmp->next){
	list_t *list_p=tmp->data;
	g_free(list_p->path);
	g_free(list_p->name);
	g_free(list_p->date);
	g_free(list_p->mimetype);
	gtk_tree_path_free(list_p->treepath);
	g_free(list_p);
    }
    g_list_free(*list);
    *list=NULL;
}

static list_t *
make_list_record(const gchar *path){
    if (!path || !strlen(path)) {
	g_warning("make_list_record(): path is NULL or empty.\n");
        return NULL;
    }
    struct stat st;
    list_t *list_p=(list_t *)malloc(sizeof(list_t));
    memset(list_p, 0, sizeof(list_t));
    if (stat(path, &st) < 0) {
        list_p->date = g_strdup(""); // binary files?
        list_p->pixbuf = rfm_get_pixbuf("xffm/emblem_no-read", SIZE_BUTTON);
        list_p->name=g_path_get_basename(path);
        list_p->path=g_strdup(path);
        list_p->mimetype=g_strdup("xffm/emblem_no-read");
        return list_p;
    }
    
    if (!rfm_void(RFM_MODULE_DIR, "mime", "module_active")){
        if (rfm_g_file_test(path, G_FILE_TEST_IS_DIR)){
            list_p->mimetype = g_strdup("xffm/stock_directory");
        } else {
            list_p->mimetype = g_strdup("xffm/stock_file");
        }
    } else {
        list_p->mimetype=MIME_type(path, NULL);
        if (!list_p->mimetype) {
            list_p->mimetype=MIME_magic(path);
        }
    }


#ifdef HAVE_LOCALTIME_R
    struct tm t_r;
#endif
    struct tm *t;
#ifdef HAVE_LOCALTIME_R
    t = localtime_r (&st.st_mtime, &t_r);
#else
    t = localtime (&st.st_mtime);
#endif
    list_p->date = g_strdup_printf("%04d/%02d/%02d  %02d:%02d",
            t->tm_year + 1900, t->tm_mon + 1, t->tm_mday, 
            t->tm_hour, t->tm_min);
    
    list_p->pixbuf = rfm_get_pixbuf(list_p->mimetype, SIZE_BUTTON);
    list_p->name=g_path_get_basename (path);
    list_p->path=g_strdup(path);
    list_p->size = st.st_size;
    return list_p;
}

static
gint make_list_from_directory(GList **list, const gchar *folder, gint *patchmake_count){
    clear_list(list);
    struct dirent *d;
    DIR *directory = opendir (folder);
    gint binary_count=0;
    while(directory && (d = readdir (directory)) != NULL) {
        //if(strcmp (d->d_name, ".") == 0) continue;
        //if(strcmp (d->d_name, "..") == 0) continue;
	gchar *path=g_build_filename(folder, d->d_name, NULL);
	gchar *encoding=MIME_encoding(path);
	// For missing module condition:
	if (!encoding) encoding = g_strdup("unknown");
	if (!rfm_g_file_test(path, G_FILE_TEST_IS_DIR) && strstr(encoding, "binary")){
	    g_free(path);
	    g_free(encoding);
	    binary_count++;
	    continue;
	}
	g_free(encoding);

	list_t *list_p=make_list_record(path);
	g_free(path);
	*list = g_list_append(*list, list_p);
    }
    if (directory) closedir (directory);
    *patchmake_count=binary_count;
    if (*list==NULL) {
	TRACE("if no files or only binary encoded files found, issue a message in combobox\n");
	gchar *message=NULL;
	if (!binary_count){
	    message = g_strdup ( _("No files found"));
	} else {
            gchar *plural_text=g_strdup_printf (ngettext ("%'u item", "%'u items",
                                       binary_count),binary_count);
            gchar *p = g_strdup_printf ("%s: %s", _("Binary comparison"), 
                    plural_text);
            g_free(plural_text);
            message=p;
	}

	list_t *list_p=make_list_record(message);
	g_free(message);
	*list = g_list_append(*list, list_p);
	return 0;
    }
    return g_list_length(*list);
}

// clean polygon
static void
cleanPolygon (diff_t *diff_p) {
    polygon_t *thisP;
    diff_p->polygon_ids = 0;
    if(!diff_p->polygon_head){
	return;
    }

    while(diff_p->polygon_head != NULL) {
        thisP = diff_p->polygon_head;
        diff_p->polygon_head = thisP->next;
        free (thisP);
    }
    diff_p->polygon_current = NULL;

    return;
}
polygon_t *
pushPolygon (diff_t *diff_p, int *top, int *bottom, int *current_line) {
    int d;
    polygon_t *tmp;
    if(!diff_p->polygon_head) {
        diff_p->polygon_head = (polygon_t *) malloc (sizeof (polygon_t));
        if(!diff_p->polygon_head){
	    g_error("diff_p->polygon_head==NULL");
	}
        tmp = diff_p->polygon_head;
        tmp->previous = NULL;
    } else {
        for(tmp = diff_p->polygon_head; tmp->next; tmp = tmp->next) ;

        tmp->next = (polygon_t *) malloc (sizeof (polygon_t));
        if(!tmp->next)
	    g_error("diff_p->polygon_head->next==NULL");
        (tmp->next)->previous = tmp;
        tmp = tmp->next;
    }
    tmp->next = NULL;
    tmp->topR = top[1] * diff_p->lineH;
    tmp->botR = bottom[1] * diff_p->lineH;
    tmp->topL = top[0] * diff_p->lineH;
    tmp->botL = bottom[0] * diff_p->lineH;

    d = bottom[1] - top[1];
    tmp->topLR = diff_p->current_line[1] - d + 1;
    tmp->botLR = diff_p->current_line[1];

    d = bottom[0] - top[0];
    tmp->topLL = diff_p->current_line[0] - d + 1;
    tmp->botLL = diff_p->current_line[0];
    diff_p->polygon_ids++;
    tmp->id = diff_p->polygon_ids;
    return tmp;
}


// clean drawing area
static void
cleanAreaDraw (diff_t *diff_p) {
    if(!diff_p->drawA)
        return;
    if(!diff_p->drawP)
        return;
    GdkRectangle allocation;
    gtk_widget_get_allocation (diff_p->drawA, &allocation);
    /*GdkRectangle update_rect;
    update_rect.x = 0;
    update_rect.y = gtk_adjustment_get_value(GTK_ADJUSTMENT (diff_p->adj));
    update_rect.width = allocation.width;
    update_rect.width = allocation.height;*/

    Display *display=gdk_x11_display_get_xdisplay(gdk_display_get_default());

    if(diff_p->drawP){
        XFreePixmap (display, diff_p->drawP);
    }
    gint root_d;
    rfm_get_drawable_geometry(gdk_x11_get_default_root_xwindow (),
	    NULL, NULL, NULL, NULL, 
	    &root_d);    
    diff_p->drawP = XCreatePixmap (display, GDK_ROOT_WINDOW (), 
	    allocation.width, allocation.height, root_d);
    Visual *visual = gdk_x11_visual_get_xvisual(gdk_visual_get_system());
    cairo_surface_t *pixmap_surface = 
	cairo_xlib_surface_create (
		display, // Display *
		diff_p->drawP, // Drawable 
		visual, // Visual *
		allocation.width, allocation.height);

    if(cairo_surface_status (pixmap_surface) != CAIRO_STATUS_SUCCESS) {
	g_error ("cairo_surface_status(surface) != CAIRO_STATUS_SUCCESS");
    }
    cairo_t *gdk_context = cairo_create (pixmap_surface);

    cairo_set_source_rgb(gdk_context, 0.96, 0.96, 0.96);
    cairo_paint(gdk_context);
    cairo_destroy(gdk_context);
}


static void
clear_text_buffer (GtkTextBuffer * buffer) {
    if(!buffer)
        return;
    gtk_text_buffer_set_text (buffer, "", -1);
}

// clean text buffers
static void
cleanTextAreas (diff_t *diff_p) {
    clear_text_buffer (diff_p->text_buffer[0]); //left
    clear_text_buffer (diff_p->text_buffer[1]); //right
}

#if 0
// check if path is really a patch file, by mimemagic
// this currently does not work. libmagic does not
// correctly identify patch files, neither in default
// patch format or unified patch format
static gboolean
is_patch_file(const gchar *path){
    gchar *mimetype=MIME_type(file, NULL);
    gchar *mimemagic=MIME_magic(file);
    gboolean OK = (mimetype && strstr(mimetype,"text/x-patch"))
		||(mimemagic && strstr(mimemagic, "text/x-patch"));
    if (!OK) {
    	gchar *g=g_strdup_printf("\n  %s\n\n  %s text/x-patch (%s)\n", 
			    _("Patch file can not be opened for read!"),
			    _("Input Required:"),
			    _("MIME Type")
			    );
        rfm_confirm (NULL, GTK_MESSAGE_ERROR, g, NULL, NULL);
        g_free(g);
    }
    g_free(mimetype);
    g_free(mimemagic);		
    return OK;
}
#endif


static gboolean
set_right_index (diff_t *diff_p, gint index) {

    // get the combo active basename
    if (!diff_p->left_file || !diff_p->right_file) {
	return FALSE;
    }
    //if (strcmp(diff_p->left_file, diff_p->right_file)==0) { return; }
    if (!rfm_g_file_test(diff_p->right_file, G_FILE_TEST_IS_DIR)){
	return FALSE;
    }
    if (gtk_combo_box_get_active  (GTK_COMBO_BOX(diff_p->combo_right)) < 0){
	return FALSE;
    }

    GtkTreeIter iter;
    gchar *active_left=NULL;	
    gchar *active_right=NULL;	
    gtk_combo_box_set_active (GTK_COMBO_BOX(diff_p->combo_left), index); 
    if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX(diff_p->combo_left), &iter)){
	return FALSE; 
    }
    gtk_tree_model_get (GTK_TREE_MODEL(diff_p->list_store_left), &iter,
	    BASENAME_COLUMN, &active_left,
	    -1);
    // search for the basename at the right combo
    if( !gtk_tree_model_get_iter_first(GTK_TREE_MODEL(diff_p->list_store_right), &iter)){
	g_free(active_left);  
	return FALSE;
    }
 
    do {
	gchar *path_right=NULL;	
	gtk_tree_model_get (GTK_TREE_MODEL(diff_p->list_store_right), &iter,
	    BASENAME_COLUMN, &active_right,
	    PATH_COLUMN, &path_right,
	    -1);


	if (strcmp(active_right, active_left)==0){
	    if (path_right && rfm_g_file_test(path_right, G_FILE_TEST_IS_DIR)){
		gchar *message = 
		    g_strdup_printf(_("\"%s\" is a directory, rather than a file."), 
			    path_right);
		widgets_t *widgets_p=&(diff_p->widgets);
		rfm_diagnostics (widgets_p, "xffm/stock_dialog-warning", NULL);
		rfm_diagnostics (widgets_p, "xffm_tag/stderr", " ", 
			message, "\n", NULL);
		g_free(message);
		g_free(active_right);
		g_free(active_left);  
		g_free(path_right);
		return FALSE;
	    }
	    // set the right combo if appropriate
	    gtk_combo_box_set_active_iter (GTK_COMBO_BOX(diff_p->combo_right), &iter); 
	    diff_p->right_index = 
		gtk_combo_box_get_active (GTK_COMBO_BOX(diff_p->combo_right)); 
	    g_free(active_left);  
	    g_free(active_right);
	    return TRUE;
	}
	g_free(path_right);
	g_free(active_right);
    } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(diff_p->list_store_right), &iter));
    g_free(active_left);  
    return TRUE;
}


static gchar *
get_combo_file(GtkWidget *combobox, GtkListStore *store){
    GtkTreeIter iter;
    gchar *path;
    if (!gtk_combo_box_get_active_iter (GTK_COMBO_BOX(combobox), &iter)){
	DBG("get_combo_file(): !gtk_combo_box_get_active_iter\n");
    }
    gchar *g;	
    gtk_tree_model_get (GTK_TREE_MODEL(store), &iter,
	PATH_COLUMN, &g,
	-1);
    path = rfm_esc_string(g);
    g_free(g);
    return path;
}

