"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TelemetrySession = void 0;
const crypto_1 = require("crypto");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const installation_id_1 = require("./installation-id");
const library_version_1 = require("./library-version");
const sanitation_1 = require("./sanitation");
const schema_1 = require("./schema");
const ci_systems_1 = require("../ci-systems");
const messages_1 = require("../telemetry/messages");
const ci_1 = require("../util/ci");
const version_1 = require("../version");
const ABORTED_ERROR_MESSAGE = '__CDK-Toolkit__Aborted';
class TelemetrySession {
    constructor(props) {
        this.props = props;
        this.count = 0;
        this.ioHost = props.ioHost;
        this.client = props.client;
    }
    async begin() {
        // sanitize the raw cli input
        const { path, parameters } = (0, sanitation_1.sanitizeCommandLineArguments)(this.props.arguments);
        this._sessionInfo = {
            identifiers: {
                installationId: await (0, installation_id_1.getOrCreateInstallationId)(this.ioHost.asIoHelper()),
                sessionId: (0, crypto_1.randomUUID)(),
                telemetryVersion: '1.0',
                cdkCliVersion: (0, version_1.versionNumber)(),
                cdkLibraryVersion: await (0, library_version_1.getLibraryVersion)(this.ioHost.asIoHelper()),
            },
            event: {
                command: {
                    path,
                    parameters,
                    config: {
                        context: (0, sanitation_1.sanitizeContext)(this.props.context),
                    },
                },
            },
            environment: {
                ci: (0, ci_1.isCI)() || Boolean((0, ci_systems_1.detectCiSystem)()),
                os: {
                    platform: process.platform,
                    release: process.release.name,
                },
                nodeVersion: process.version,
            },
            project: {},
        };
        // If SIGINT has a listener installed, its default behavior will be removed (Node.js will no longer exit).
        // This ensures that on SIGINT we process safely close the telemetry session before exiting.
        process.on('SIGINT', async () => {
            try {
                await this.end({
                    name: schema_1.ErrorName.TOOLKIT_ERROR,
                    message: ABORTED_ERROR_MESSAGE,
                });
            }
            catch (e) {
                await this.ioHost.defaults.trace(`Ending Telemetry failed: ${e.message}`);
            }
            process.exit(1);
        });
        // Begin the session span
        this.span = await this.ioHost.asIoHelper().span(messages_1.CLI_PRIVATE_SPAN.COMMAND).begin({});
    }
    async attachRegion(region) {
        this.sessionInfo.identifiers = {
            ...this.sessionInfo.identifiers,
            region,
        };
    }
    /**
     * When the command is complete, so is the CliIoHost. Ends the span of the entire CliIoHost
     * and notifies with an optional error message in the data.
     */
    async end(error) {
        await this.span?.end({ error });
        // Ideally span.end() should no-op if called twice, but that is not the case right now
        this.span = undefined;
        await this.client.flush();
    }
    async emit(event) {
        this.count += 1;
        return this.client.emit({
            event: {
                command: this.sessionInfo.event.command,
                state: getState(event.error),
                eventType: event.eventType,
            },
            identifiers: {
                ...this.sessionInfo.identifiers,
                eventId: `${this.sessionInfo.identifiers.sessionId}:${this.count}`,
                timestamp: new Date().toISOString(),
            },
            environment: this.sessionInfo.environment,
            project: this.sessionInfo.project,
            duration: {
                total: event.duration,
            },
            ...(event.error ? {
                error: {
                    name: event.error.name,
                },
            } : {}),
        });
    }
    get sessionInfo() {
        if (!this._sessionInfo) {
            throw new toolkit_lib_1.ToolkitError('Session Info not initialized. Call begin() first.');
        }
        return this._sessionInfo;
    }
}
exports.TelemetrySession = TelemetrySession;
function getState(error) {
    if (error) {
        return isAbortedError(error) ? 'ABORTED' : 'FAILED';
    }
    return 'SUCCEEDED';
}
function isAbortedError(error) {
    if (error?.name === 'ToolkitError' && error?.message?.includes(ABORTED_ERROR_MESSAGE)) {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,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