"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exec = exec;
exports.cli = cli;
/* eslint-disable @typescript-eslint/no-shadow */ // yargs
const cxapi = require("@aws-cdk/cx-api");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const cdk_toolkit_1 = require("./cdk-toolkit");
const ci_systems_1 = require("./ci-systems");
const display_version_1 = require("./display-version");
const io_host_1 = require("./io-host");
const parse_command_line_arguments_1 = require("./parse-command-line-arguments");
const platform_warnings_1 = require("./platform-warnings");
const pretty_print_error_1 = require("./pretty-print-error");
const singleton_plugin_host_1 = require("./singleton-plugin-host");
const user_configuration_1 = require("./user-configuration");
const api_private_1 = require("../../lib/api-private");
const api_1 = require("../api");
const aws_auth_1 = require("../api/aws-auth");
const bootstrap_1 = require("../api/bootstrap");
const deployments_1 = require("../api/deployments");
const hotswap_1 = require("../api/hotswap");
const context_1 = require("../commands/context");
const docs_1 = require("../commands/docs");
const doctor_1 = require("../commands/doctor");
const flags_1 = require("../commands/flags/flags");
const init_1 = require("../commands/init");
const migrate_1 = require("../commands/migrate");
const cxapp_1 = require("../cxapp");
const proxy_agent_1 = require("./proxy-agent");
const error_1 = require("./telemetry/error");
const ci_1 = require("./util/ci");
const version_1 = require("./version");
const language_1 = require("../commands/language");
if (!process.stdout.isTTY) {
    // Disable chalk color highlighting
    process.env.FORCE_COLOR = '0';
}
async function exec(args, synthesizer) {
    const argv = await (0, parse_command_line_arguments_1.parseCommandLineArguments)(args);
    argv.language = (0, language_1.getLanguageFromAlias)(argv.language) ?? argv.language;
    const cmd = argv._[0];
    // if one -v, log at a DEBUG level
    // if 2 -v, log at a TRACE level
    let ioMessageLevel = 'info';
    if (argv.verbose) {
        switch (argv.verbose) {
            case 1:
                ioMessageLevel = 'debug';
                break;
            case 2:
            default:
                ioMessageLevel = 'trace';
                break;
        }
    }
    const ioHost = io_host_1.CliIoHost.instance({
        logLevel: ioMessageLevel,
        isTTY: process.stdout.isTTY,
        isCI: Boolean(argv.ci),
        currentAction: cmd,
        stackProgress: argv.progress,
        autoRespond: argv.yes,
    }, true);
    const ioHelper = (0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction);
    // Debug should always imply tracing
    (0, aws_auth_1.setSdkTracing)(argv.debug || argv.verbose > 2);
    try {
        await (0, platform_warnings_1.checkForPlatformWarnings)(ioHelper);
    }
    catch (e) {
        await ioHost.defaults.debug(`Error while checking for platform warnings: ${e}`);
    }
    await ioHost.defaults.debug('CDK Toolkit CLI version:', (0, version_1.versionWithBuild)());
    await ioHost.defaults.debug('Command line arguments:', argv);
    const configuration = await user_configuration_1.Configuration.fromArgsAndFiles(ioHelper, {
        commandLineArguments: {
            ...argv,
            _: argv._, // TypeScript at its best
        },
    });
    // Always create and use ProxyAgent to support configuration via env vars
    const proxyAgent = await new proxy_agent_1.ProxyAgentProvider(ioHelper).create({
        proxyAddress: configuration.settings.get(['proxy']),
        caBundlePath: configuration.settings.get(['caBundlePath']),
    });
    if (argv['telemetry-file'] && !configuration.settings.get(['unstable']).includes('telemetry')) {
        throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'telemetry-file\' is unstable. It must be opted in via \'--unstable\', e.g. \'cdk deploy --unstable=telemetry --telemetry-file=my/file/path\'');
    }
    try {
        await ioHost.startTelemetry(argv, configuration.context);
    }
    catch (e) {
        await ioHost.asIoHelper().defaults.trace(`Telemetry instantiation failed: ${e.message}`);
    }
    /**
     * The default value for displaying (and refreshing) notices on all commands.
     *
     * If the user didn't supply either `--notices` or `--no-notices`, we do
     * autodetection. The autodetection currently is: do write notices if we are
     * not on CI, or are on a CI system where we know that writing to stderr is
     * safe. We fail "closed"; that is, we decide to NOT print for unknown CI
     * systems, even though technically we maybe could.
     */
    const isSafeToWriteNotices = !(0, ci_1.isCI)() || Boolean((0, ci_systems_1.ciSystemIsStdErrSafe)());
    // Determine if notices should be displayed based on CLI args and configuration
    let shouldDisplayNotices;
    if (argv.notices !== undefined) {
        // CLI argument takes precedence
        shouldDisplayNotices = argv.notices;
    }
    else {
        // Fall back to configuration file setting, then autodetection
        const configNotices = configuration.settings.get(['notices']);
        if (configNotices !== undefined) {
            // Consider string "false" to be falsy in this context
            shouldDisplayNotices = configNotices !== 'false' && Boolean(configNotices);
        }
        else {
            // Default autodetection behavior
            shouldDisplayNotices = isSafeToWriteNotices;
        }
    }
    // Notices either go to stderr, or nowhere
    ioHost.noticesDestination = shouldDisplayNotices ? 'stderr' : 'drop';
    const notices = api_1.Notices.create({
        ioHost,
        context: configuration.context,
        output: configuration.settings.get(['outdir']),
        httpOptions: { agent: proxyAgent },
        cliVersion: (0, version_1.versionNumber)(),
    });
    const refreshNotices = (async () => {
        // the cdk notices command has it's own refresh
        if (shouldDisplayNotices && cmd !== 'notices') {
            try {
                return await notices.refresh();
            }
            catch (e) {
                await ioHelper.defaults.debug(`Could not refresh notices: ${e}`);
            }
        }
    })();
    const sdkProvider = await aws_auth_1.SdkProvider.withAwsCliCompatibleDefaults({
        ioHelper,
        requestHandler: (0, aws_auth_1.sdkRequestHandler)(proxyAgent),
        logger: new aws_auth_1.IoHostSdkLogger((0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction)),
        pluginHost: singleton_plugin_host_1.GLOBAL_PLUGIN_HOST,
    }, configuration.settings.get(['profile']));
    try {
        await ioHost.telemetry?.attachRegion(sdkProvider.defaultRegion);
    }
    catch (e) {
        await ioHost.asIoHelper().defaults.trace(`Telemetry attach region failed: ${e.message}`);
    }
    let outDirLock;
    const cloudExecutable = new cxapp_1.CloudExecutable({
        configuration,
        sdkProvider,
        synthesizer: synthesizer ??
            (async (aws, config) => {
                // Invoke 'execProgram', and copy the lock for the directory in the global
                // variable here. It will be released when the CLI exits. Locks are not re-entrant
                // so release it if we have to synthesize more than once (because of context lookups).
                await outDirLock?.release();
                const { assembly, lock } = await (0, cxapp_1.execProgram)(aws, ioHost.asIoHelper(), config);
                outDirLock = lock;
                return assembly;
            }),
        ioHelper: ioHost.asIoHelper(),
    });
    /** Function to load plug-ins, using configurations additively. */
    async function loadPlugins(...settings) {
        for (const source of settings) {
            const plugins = source.get(['plugin']) || [];
            for (const plugin of plugins) {
                await singleton_plugin_host_1.GLOBAL_PLUGIN_HOST.load(plugin, ioHost);
            }
        }
    }
    await loadPlugins(configuration.settings);
    if ((typeof cmd) !== 'string') {
        throw new toolkit_lib_1.ToolkitError(`First argument should be a string. Got: ${cmd} (${typeof cmd})`);
    }
    try {
        return await main(cmd, argv);
    }
    finally {
        // If we locked the 'cdk.out' directory, release it here.
        await outDirLock?.release();
        // Do PSAs here
        await (0, display_version_1.displayVersionMessage)(ioHelper);
        await refreshNotices;
        if (cmd === 'notices') {
            await notices.refresh({ force: true });
            await notices.display({
                includeAcknowledged: !argv.unacknowledged,
                showTotal: argv.unacknowledged,
            });
        }
        else if (shouldDisplayNotices && cmd !== 'version') {
            await notices.display();
        }
    }
    async function main(command, args) {
        ioHost.currentAction = command;
        const toolkitStackName = api_1.ToolkitInfo.determineName(configuration.settings.get(['toolkitStackName']));
        await ioHost.defaults.debug(`Toolkit stack: ${chalk.bold(toolkitStackName)}`);
        const cloudFormation = new deployments_1.Deployments({
            sdkProvider,
            toolkitStackName,
            ioHelper: (0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction),
        });
        if (args.all && args.STACKS) {
            throw new toolkit_lib_1.ToolkitError('You must either specify a list of Stacks or the `--all` argument');
        }
        args.STACKS = args.STACKS ?? (args.STACK ? [args.STACK] : []);
        args.ENVIRONMENTS = args.ENVIRONMENTS ?? [];
        const selector = {
            allTopLevel: args.all,
            patterns: args.STACKS,
        };
        const cli = new cdk_toolkit_1.CdkToolkit({
            ioHost,
            cloudExecutable,
            toolkitStackName,
            deployments: cloudFormation,
            verbose: argv.trace || argv.verbose > 0,
            ignoreErrors: argv['ignore-errors'],
            strict: argv.strict,
            configuration,
            sdkProvider,
        });
        switch (command) {
            case 'context':
                ioHost.currentAction = 'context';
                return (0, context_1.contextHandler)({
                    ioHelper,
                    context: configuration.context,
                    clear: argv.clear,
                    json: argv.json,
                    force: argv.force,
                    reset: argv.reset,
                });
            case 'docs':
            case 'doc':
                ioHost.currentAction = 'docs';
                return (0, docs_1.docs)({
                    ioHelper,
                    browser: configuration.settings.get(['browser']),
                });
            case 'doctor':
                ioHost.currentAction = 'doctor';
                return (0, doctor_1.doctor)({
                    ioHelper,
                });
            case 'ls':
            case 'list':
                ioHost.currentAction = 'list';
                return cli.list(args.STACKS, {
                    long: args.long,
                    json: argv.json,
                    showDeps: args.showDependencies,
                });
            case 'diff':
                ioHost.currentAction = 'diff';
                const enableDiffNoFail = isFeatureEnabled(configuration, cxapi.ENABLE_DIFF_NO_FAIL_CONTEXT);
                return cli.diff({
                    stackNames: args.STACKS,
                    exclusively: args.exclusively,
                    templatePath: args.template,
                    strict: args.strict,
                    contextLines: args.contextLines,
                    securityOnly: args.securityOnly,
                    fail: args.fail != null ? args.fail : !enableDiffNoFail,
                    compareAgainstProcessedTemplate: args.processed,
                    quiet: args.quiet,
                    changeSet: args['change-set'],
                    toolkitStackName: toolkitStackName,
                    importExistingResources: args.importExistingResources,
                    includeMoves: args['include-moves'],
                });
            case 'drift':
                ioHost.currentAction = 'drift';
                return cli.drift({
                    selector,
                    fail: args.fail,
                });
            case 'refactor':
                if (!configuration.settings.get(['unstable']).includes('refactor')) {
                    throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'refactor\' is unstable. It must be opted in via \'--unstable\', e.g. \'cdk refactor --unstable=refactor\'');
                }
                ioHost.currentAction = 'refactor';
                return cli.refactor({
                    dryRun: args.dryRun,
                    overrideFile: args.overrideFile,
                    revert: args.revert,
                    stacks: selector,
                    additionalStackNames: arrayFromYargs(args.additionalStackName ?? []),
                    force: args.force ?? false,
                    roleArn: args.roleArn,
                });
            case 'bootstrap':
                ioHost.currentAction = 'bootstrap';
                const source = await determineBootstrapVersion(ioHost, args);
                if (args.showTemplate) {
                    const bootstrapper = new bootstrap_1.Bootstrapper(source, (0, api_private_1.asIoHelper)(ioHost, ioHost.currentAction));
                    return bootstrapper.showTemplate(args.json);
                }
                return cli.bootstrap(args.ENVIRONMENTS, {
                    source,
                    roleArn: args.roleArn,
                    forceDeployment: argv.force,
                    toolkitStackName: toolkitStackName,
                    execute: args.execute,
                    tags: configuration.settings.get(['tags']),
                    terminationProtection: args.terminationProtection,
                    usePreviousParameters: args['previous-parameters'],
                    parameters: {
                        bucketName: configuration.settings.get(['toolkitBucket', 'bucketName']),
                        kmsKeyId: configuration.settings.get(['toolkitBucket', 'kmsKeyId']),
                        createCustomerMasterKey: args.bootstrapCustomerKey,
                        qualifier: args.qualifier ?? configuration.context.get('@aws-cdk/core:bootstrapQualifier'),
                        publicAccessBlockConfiguration: args.publicAccessBlockConfiguration,
                        examplePermissionsBoundary: argv.examplePermissionsBoundary,
                        customPermissionsBoundary: argv.customPermissionsBoundary,
                        trustedAccounts: arrayFromYargs(args.trust),
                        trustedAccountsForLookup: arrayFromYargs(args.trustForLookup),
                        untrustedAccounts: arrayFromYargs(args.untrust),
                        cloudFormationExecutionPolicies: arrayFromYargs(args.cloudformationExecutionPolicies),
                        denyExternalId: args.denyExternalId,
                    },
                });
            case 'deploy':
                ioHost.currentAction = 'deploy';
                const parameterMap = {};
                for (const parameter of args.parameters) {
                    if (typeof parameter === 'string') {
                        const keyValue = parameter.split('=');
                        parameterMap[keyValue[0]] = keyValue.slice(1).join('=');
                    }
                }
                if (args.execute !== undefined && args.method !== undefined) {
                    throw new toolkit_lib_1.ToolkitError('Can not supply both --[no-]execute and --method at the same time');
                }
                return cli.deploy({
                    selector,
                    exclusively: args.exclusively,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    notificationArns: args.notificationArns,
                    requireApproval: configuration.settings.get(['requireApproval']),
                    reuseAssets: args['build-exclude'],
                    tags: configuration.settings.get(['tags']),
                    deploymentMethod: determineDeploymentMethod(args, configuration),
                    force: args.force,
                    parameters: parameterMap,
                    usePreviousParameters: args['previous-parameters'],
                    outputsFile: configuration.settings.get(['outputsFile']),
                    progress: configuration.settings.get(['progress']),
                    ci: args.ci,
                    rollback: configuration.settings.get(['rollback']),
                    watch: args.watch,
                    traceLogs: args.logs,
                    concurrency: args.concurrency,
                    assetParallelism: configuration.settings.get(['assetParallelism']),
                    assetBuildTime: configuration.settings.get(['assetPrebuild'])
                        ? cdk_toolkit_1.AssetBuildTime.ALL_BEFORE_DEPLOY
                        : cdk_toolkit_1.AssetBuildTime.JUST_IN_TIME,
                    ignoreNoStacks: args.ignoreNoStacks,
                });
            case 'rollback':
                ioHost.currentAction = 'rollback';
                return cli.rollback({
                    selector,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    force: args.force,
                    validateBootstrapStackVersion: args['validate-bootstrap-version'],
                    orphanLogicalIds: args.orphan,
                });
            case 'import':
                ioHost.currentAction = 'import';
                return cli.import({
                    selector,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    deploymentMethod: {
                        method: 'change-set',
                        execute: args.execute,
                        changeSetName: args.changeSetName,
                    },
                    progress: configuration.settings.get(['progress']),
                    rollback: configuration.settings.get(['rollback']),
                    recordResourceMapping: args['record-resource-mapping'],
                    resourceMappingFile: args['resource-mapping'],
                    force: args.force,
                });
            case 'watch':
                ioHost.currentAction = 'watch';
                await cli.watch({
                    selector,
                    exclusively: args.exclusively,
                    toolkitStackName,
                    roleArn: args.roleArn,
                    reuseAssets: args['build-exclude'],
                    deploymentMethod: determineDeploymentMethod(args, configuration, true),
                    force: args.force,
                    progress: configuration.settings.get(['progress']),
                    rollback: configuration.settings.get(['rollback']),
                    traceLogs: args.logs,
                    concurrency: args.concurrency,
                });
                return;
            case 'destroy':
                ioHost.currentAction = 'destroy';
                return cli.destroy({
                    selector,
                    exclusively: args.exclusively,
                    force: args.force,
                    roleArn: args.roleArn,
                });
            case 'gc':
                ioHost.currentAction = 'gc';
                if (!configuration.settings.get(['unstable']).includes('gc')) {
                    throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'gc\' is unstable. It must be opted in via \'--unstable\', e.g. \'cdk gc --unstable=gc\'');
                }
                if (args.bootstrapStackName) {
                    await ioHost.defaults.warn('--bootstrap-stack-name is deprecated and will be removed when gc is GA. Use --toolkit-stack-name.');
                }
                // roleArn is defined for when cloudformation is invoked
                // This conflicts with direct sdk calls existing in the gc command to s3 and ecr
                if (args.roleArn) {
                    await ioHost.defaults.warn('The --role-arn option is not supported for the gc command and will be ignored.');
                }
                return cli.garbageCollect(args.ENVIRONMENTS, {
                    action: args.action,
                    type: args.type,
                    rollbackBufferDays: args['rollback-buffer-days'],
                    createdBufferDays: args['created-buffer-days'],
                    bootstrapStackName: args.toolkitStackName ?? args.bootstrapStackName,
                    confirm: args.confirm,
                });
            case 'flags':
                ioHost.currentAction = 'flags';
                if (!configuration.settings.get(['unstable']).includes('flags')) {
                    throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'flags\' is unstable. It must be opted in via \'--unstable\', e.g. \'cdk flags --unstable=flags\'');
                }
                const toolkit = new toolkit_lib_1.Toolkit({
                    ioHost,
                    toolkitStackName,
                    unstableFeatures: configuration.settings.get(['unstable']),
                });
                const flagsData = await toolkit.flags(cloudExecutable);
                const handler = new flags_1.FlagCommandHandler(flagsData, ioHelper, args, toolkit);
                return handler.processFlagsCommand();
            case 'synthesize':
            case 'synth':
                ioHost.currentAction = 'synth';
                const quiet = configuration.settings.get(['quiet']) ?? args.quiet;
                if (args.exclusively) {
                    return cli.synth(args.STACKS, args.exclusively, quiet, args.validation, argv.json);
                }
                else {
                    return cli.synth(args.STACKS, true, quiet, args.validation, argv.json);
                }
            case 'notices':
                ioHost.currentAction = 'notices';
                // If the user explicitly asks for notices, they are now the primary output
                // of the command and they should go to stdout.
                ioHost.noticesDestination = 'stdout';
                // This is a valid command, but we're postponing its execution because displaying
                // notices automatically happens after every command.
                return;
            case 'metadata':
                ioHost.currentAction = 'metadata';
                return cli.metadata(args.STACK, argv.json);
            case 'acknowledge':
            case 'ack':
                ioHost.currentAction = 'notices';
                return cli.acknowledge(args.ID);
            case 'cli-telemetry':
                ioHost.currentAction = 'cli-telemetry';
                if (args.enable === undefined && args.disable === undefined && args.status === undefined) {
                    throw new toolkit_lib_1.ToolkitError('Must specify \'--enable\', \'--disable\', or \'--status\'');
                }
                if (args.status) {
                    return cli.cliTelemetryStatus(args);
                }
                else {
                    const enable = args.enable ?? !args.disable;
                    return cli.cliTelemetry(enable);
                }
            case 'init':
                ioHost.currentAction = 'init';
                const language = configuration.settings.get(['language']);
                if (args.list) {
                    return (0, init_1.printAvailableTemplates)(ioHelper, language);
                }
                else {
                    // Gate custom template support with unstable flag
                    if (args['from-path'] && !configuration.settings.get(['unstable']).includes('init')) {
                        throw new toolkit_lib_1.ToolkitError('Unstable feature use: \'init\' with custom templates is unstable. It must be opted in via \'--unstable\', e.g. \'cdk init --from-path=./my-template --unstable=init\'');
                    }
                    return (0, init_1.cliInit)({
                        ioHelper,
                        type: args.TEMPLATE,
                        language,
                        canUseNetwork: undefined,
                        generateOnly: args.generateOnly,
                        libVersion: args.libVersion,
                        fromPath: args['from-path'],
                        templatePath: args['template-path'],
                        packageManager: args['package-manager'],
                    });
                }
            case 'migrate':
                ioHost.currentAction = 'migrate';
                return cli.migrate({
                    stackName: args['stack-name'],
                    fromPath: args['from-path'],
                    fromStack: args['from-stack'],
                    language: args.language,
                    outputPath: args['output-path'],
                    fromScan: (0, migrate_1.getMigrateScanType)(args['from-scan']),
                    filter: args.filter,
                    account: args.account,
                    region: args.region,
                    compress: args.compress,
                });
            case 'version':
                ioHost.currentAction = 'version';
                return ioHost.defaults.result((0, version_1.versionWithBuild)());
            default:
                throw new toolkit_lib_1.ToolkitError('Unknown command: ' + command);
        }
    }
}
/**
 * Determine which version of bootstrapping
 */
async function determineBootstrapVersion(ioHost, args) {
    let source;
    if (args.template) {
        await ioHost.defaults.info(`Using bootstrapping template from ${args.template}`);
        source = { source: 'custom', templateFile: args.template };
    }
    else if (process.env.CDK_LEGACY_BOOTSTRAP) {
        await ioHost.defaults.info('CDK_LEGACY_BOOTSTRAP set, using legacy-style bootstrapping');
        source = { source: 'legacy' };
    }
    else {
        // in V2, the "new" bootstrapping is the default
        source = { source: 'default' };
    }
    return source;
}
function isFeatureEnabled(configuration, featureFlag) {
    return configuration.context.get(featureFlag) ?? cxapi.futureFlagDefault(featureFlag);
}
/**
 * Translate a Yargs input array to something that makes more sense in a programming language
 * model (telling the difference between absence and an empty array)
 *
 * - An empty array is the default case, meaning the user didn't pass any arguments. We return
 *   undefined.
 * - If the user passed a single empty string, they did something like `--array=`, which we'll
 *   take to mean they passed an empty array.
 */
function arrayFromYargs(xs) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.filter((x) => x !== '');
}
function determineDeploymentMethod(args, configuration, watch) {
    let deploymentMethod;
    switch (args.method) {
        case 'direct':
            if (args.changeSetName) {
                throw new toolkit_lib_1.ToolkitError('--change-set-name cannot be used with method=direct');
            }
            if (args.importExistingResources) {
                throw new toolkit_lib_1.ToolkitError('--import-existing-resources cannot be enabled with method=direct');
            }
            deploymentMethod = { method: 'direct' };
            break;
        case 'change-set':
            deploymentMethod = {
                method: 'change-set',
                execute: true,
                changeSetName: args.changeSetName,
                importExistingResources: args.importExistingResources,
            };
            break;
        case 'prepare-change-set':
            deploymentMethod = {
                method: 'change-set',
                execute: false,
                changeSetName: args.changeSetName,
                importExistingResources: args.importExistingResources,
            };
            break;
        case undefined:
        default:
            deploymentMethod = {
                method: 'change-set',
                execute: watch ? true : args.execute ?? true,
                changeSetName: args.changeSetName,
                importExistingResources: args.importExistingResources,
            };
            break;
    }
    const hotswapMode = determineHotswapMode(args.hotswap, args.hotswapFallback, watch);
    const hotswapProperties = configuration.settings.get(['hotswap']) || {};
    switch (hotswapMode) {
        case hotswap_1.HotswapMode.FALL_BACK:
            return {
                method: 'hotswap',
                properties: hotswapProperties,
                fallback: deploymentMethod,
            };
        case hotswap_1.HotswapMode.HOTSWAP_ONLY:
            return {
                method: 'hotswap',
                properties: hotswapProperties,
            };
        default:
        case hotswap_1.HotswapMode.FULL_DEPLOYMENT:
            return deploymentMethod;
    }
}
function determineHotswapMode(hotswap, hotswapFallback, watch) {
    if (hotswap && hotswapFallback) {
        throw new toolkit_lib_1.ToolkitError('Can not supply both --hotswap and --hotswap-fallback at the same time');
    }
    else if (!hotswap && !hotswapFallback) {
        if (hotswap === undefined && hotswapFallback === undefined) {
            return watch ? hotswap_1.HotswapMode.HOTSWAP_ONLY : hotswap_1.HotswapMode.FULL_DEPLOYMENT;
        }
        else if (hotswap === false || hotswapFallback === false) {
            return hotswap_1.HotswapMode.FULL_DEPLOYMENT;
        }
    }
    let hotswapMode;
    if (hotswap) {
        hotswapMode = hotswap_1.HotswapMode.HOTSWAP_ONLY;
        /* if (hotswapFallback)*/
    }
    else {
        hotswapMode = hotswap_1.HotswapMode.FALL_BACK;
    }
    return hotswapMode;
}
/* c8 ignore start */ // we never call this in unit tests
function cli(args = process.argv.slice(2)) {
    let error;
    exec(args)
        .then(async (value) => {
        if (typeof value === 'number') {
            process.exitCode = value;
        }
    })
        .catch(async (err) => {
        // Log the stack trace if we're on a developer workstation. Otherwise this will be into a minified
        // file and the printed code line and stack trace are huge and useless.
        (0, pretty_print_error_1.prettyPrintError)(err, (0, version_1.isDeveloperBuildVersion)());
        error = {
            name: (0, error_1.cdkCliErrorName)(err.name),
        };
        process.exitCode = 1;
    })
        .finally(async () => {
        try {
            await io_host_1.CliIoHost.get()?.telemetry?.end(error);
        }
        catch (e) {
            await io_host_1.CliIoHost.get()?.asIoHelper().defaults.trace(`Ending Telemetry failed: ${e.message}`);
        }
    });
}
/* c8 ignore stop */
//# sourceMappingURL=data:application/json;base64,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