"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CliIoHost = void 0;
const util = require("node:util");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const promptly = require("promptly");
const api_private_1 = require("../../../lib/api-private");
const deploy_1 = require("../../commands/deploy");
const collect_telemetry_1 = require("../telemetry/collect-telemetry");
const messages_1 = require("../telemetry/messages");
const session_1 = require("../telemetry/session");
const endpoint_sink_1 = require("../telemetry/sink/endpoint-sink");
const file_sink_1 = require("../telemetry/sink/file-sink");
const funnel_1 = require("../telemetry/sink/funnel");
const ci_1 = require("../util/ci");
/**
 * A simple IO host for the CLI that writes messages to the console.
 */
class CliIoHost {
    /**
     * Returns the singleton instance
     */
    static instance(props = {}, forceNew = false) {
        if (forceNew || !CliIoHost._instance) {
            CliIoHost._instance = new CliIoHost(props);
        }
        return CliIoHost._instance;
    }
    /**
     * Returns the singleton instance if it exists
     */
    static get() {
        return CliIoHost._instance;
    }
    constructor(props = {}) {
        /**
         * Configure the target stream for notices
         *
         * (Not a setter because there's no need for additional logic when this value
         * is changed yet)
         */
        this.noticesDestination = 'stderr';
        this._progress = deploy_1.StackActivityProgress.BAR;
        // Corked Logging
        this.corkedCounter = 0;
        this.corkedLoggingBuffer = [];
        this.currentAction = props.currentAction ?? 'none';
        this.isTTY = props.isTTY ?? process.stdout.isTTY ?? false;
        this.logLevel = props.logLevel ?? 'info';
        this.isCI = props.isCI ?? (0, ci_1.isCI)();
        this.requireDeployApproval = props.requireDeployApproval ?? cloud_assembly_schema_1.RequireApproval.BROADENING;
        this.stackProgress = props.stackProgress ?? deploy_1.StackActivityProgress.BAR;
        this.autoRespond = props.autoRespond ?? false;
    }
    async startTelemetry(args, context, proxyAgent) {
        let sinks = [];
        const telemetryFilePath = args['telemetry-file'];
        if (telemetryFilePath) {
            try {
                sinks.push(new file_sink_1.FileTelemetrySink({
                    ioHost: this,
                    logFilePath: telemetryFilePath,
                }));
                await this.asIoHelper().defaults.trace('File Telemetry connected');
            }
            catch (e) {
                await this.asIoHelper().defaults.trace(`File Telemetry instantiation failed: ${e.message}`);
            }
        }
        const telemetryEndpoint = process.env.TELEMETRY_ENDPOINT;
        if ((0, collect_telemetry_1.canCollectTelemetry)(args, context) && telemetryEndpoint) {
            try {
                sinks.push(new endpoint_sink_1.EndpointTelemetrySink({
                    ioHost: this,
                    agent: proxyAgent,
                    endpoint: telemetryEndpoint,
                }));
                await this.asIoHelper().defaults.trace('Endpoint Telemetry connected');
            }
            catch (e) {
                await this.asIoHelper().defaults.trace(`Endpoint Telemetry instantiation failed: ${e.message}`);
            }
        }
        else {
            await this.asIoHelper().defaults.trace('Endpoint Telemetry NOT connected');
        }
        if (sinks.length > 0) {
            this.telemetry = new session_1.TelemetrySession({
                ioHost: this,
                client: new funnel_1.Funnel({ sinks }),
                arguments: args,
                context: context,
            });
        }
        await this.telemetry?.begin();
    }
    /**
     * Update the stackProgress preference.
     */
    set stackProgress(type) {
        this._progress = type;
    }
    /**
     * Gets the stackProgress value.
     *
     * This takes into account other state of the ioHost,
     * like if isTTY and isCI.
     */
    get stackProgress() {
        // We can always use EVENTS
        if (this._progress === deploy_1.StackActivityProgress.EVENTS) {
            return this._progress;
        }
        // if a debug message (and thus any more verbose messages) are relevant to the current log level, we have verbose logging
        const verboseLogging = (0, api_private_1.isMessageRelevantForLevel)({ level: 'debug' }, this.logLevel);
        if (verboseLogging) {
            return deploy_1.StackActivityProgress.EVENTS;
        }
        // On Windows we cannot use fancy output
        const isWindows = process.platform === 'win32';
        if (isWindows) {
            return deploy_1.StackActivityProgress.EVENTS;
        }
        // On some CI systems (such as CircleCI) output still reports as a TTY so we also
        // need an individual check for whether we're running on CI.
        // see: https://discuss.circleci.com/t/circleci-terminal-is-a-tty-but-term-is-not-set/9965
        const fancyOutputAvailable = this.isTTY && !this.isCI;
        if (!fancyOutputAvailable) {
            return deploy_1.StackActivityProgress.EVENTS;
        }
        // Use the user preference
        return this._progress;
    }
    get defaults() {
        return this.asIoHelper().defaults;
    }
    asIoHelper() {
        return (0, api_private_1.asIoHelper)(this, this.currentAction);
    }
    /**
     * Executes a block of code with corked logging. All log messages during execution
     * are buffered and only written when all nested cork blocks complete (when CORK_COUNTER reaches 0).
     * The corking is bound to the specific instance of the CliIoHost.
     *
     * @param block - Async function to execute with corked logging
     * @returns Promise that resolves with the block's return value
     */
    async withCorkedLogging(block) {
        this.corkedCounter++;
        try {
            return await block();
        }
        finally {
            this.corkedCounter--;
            if (this.corkedCounter === 0) {
                // Process each buffered message through notify
                for (const ioMessage of this.corkedLoggingBuffer) {
                    await this.notify(ioMessage);
                }
                // remove all buffered messages in-place
                this.corkedLoggingBuffer.splice(0);
            }
        }
    }
    /**
     * Notifies the host of a message.
     * The caller waits until the notification completes.
     */
    async notify(msg) {
        await this.maybeEmitTelemetry(msg);
        if (this.isStackActivity(msg)) {
            if (!this.activityPrinter) {
                this.activityPrinter = this.makeActivityPrinter();
            }
            this.activityPrinter.notify(msg);
            return;
        }
        if (!(0, api_private_1.isMessageRelevantForLevel)(msg, this.logLevel)) {
            return;
        }
        if (this.corkedCounter > 0) {
            this.corkedLoggingBuffer.push(msg);
            return;
        }
        const output = this.formatMessage(msg);
        const stream = this.selectStream(msg);
        stream?.write(output);
    }
    async maybeEmitTelemetry(msg) {
        try {
            if (this.telemetry && isTelemetryMessage(msg)) {
                await this.telemetry.emit({
                    eventType: getEventType(msg),
                    duration: msg.data.duration,
                    error: msg.data.error,
                });
            }
        }
        catch (e) {
            await this.defaults.trace(`Emit Telemetry Failed ${e.message}`);
        }
    }
    /**
     * Detect stack activity messages so they can be send to the printer.
     */
    isStackActivity(msg) {
        return msg.code && [
            'CDK_TOOLKIT_I5501',
            'CDK_TOOLKIT_I5502',
            'CDK_TOOLKIT_I5503',
        ].includes(msg.code);
    }
    /**
     * Detect special messages encode information about whether or not
     * they require approval
     */
    skipApprovalStep(msg) {
        const approvalToolkitCodes = ['CDK_TOOLKIT_I5060'];
        if (!(msg.code && approvalToolkitCodes.includes(msg.code))) {
            false;
        }
        switch (this.requireDeployApproval) {
            // Never require approval
            case cloud_assembly_schema_1.RequireApproval.NEVER:
                return true;
            // Always require approval
            case cloud_assembly_schema_1.RequireApproval.ANYCHANGE:
                return false;
            // Require approval if changes include broadening permissions
            case cloud_assembly_schema_1.RequireApproval.BROADENING:
                return ['none', 'non-broadening'].includes(msg.data?.permissionChangeType);
        }
    }
    /**
     * Determines the output stream, based on message and configuration.
     */
    selectStream(msg) {
        if (isNoticesMessage(msg)) {
            return targetStreamObject(this.noticesDestination);
        }
        return this.selectStreamFromLevel(msg.level);
    }
    /**
     * Determines the output stream, based on message level and configuration.
     */
    selectStreamFromLevel(level) {
        // The stream selection policy for the CLI is the following:
        //
        //   (1) Messages of level `result` always go to `stdout`
        //   (2) Messages of level `error` always go to `stderr`.
        //   (3a) All remaining messages go to `stderr`.
        //   (3b) If we are in CI mode, all remaining messages go to `stdout`.
        //
        switch (level) {
            case 'error':
                return process.stderr;
            case 'result':
                return process.stdout;
            default:
                return this.isCI ? process.stdout : process.stderr;
        }
    }
    /**
     * Notifies the host of a message that requires a response.
     *
     * If the host does not return a response the suggested
     * default response from the input message will be used.
     */
    async requestResponse(msg) {
        // If the request cannot be prompted for by the CliIoHost, we just accept the default
        if (!isPromptableRequest(msg)) {
            await this.notify(msg);
            return msg.defaultResponse;
        }
        const response = await this.withCorkedLogging(async () => {
            // prepare prompt data
            // @todo this format is not defined anywhere, probably should be
            const data = msg.data ?? {};
            const motivation = data.motivation ?? 'User input is needed';
            const concurrency = data.concurrency ?? 0;
            const responseDescription = data.responseDescription;
            // Special approval prompt
            // Determine if the message needs approval. If it does, continue (it is a basic confirmation prompt)
            // If it does not, return success (true). We only check messages with codes that we are aware
            // are requires approval codes.
            if (this.skipApprovalStep(msg)) {
                return true;
            }
            // In --yes mode, respond for the user if we can
            if (this.autoRespond) {
                // respond with yes to all confirmations
                if (isConfirmationPrompt(msg)) {
                    await this.notify({
                        ...msg,
                        message: `${chalk.cyan(msg.message)} (auto-confirmed)`,
                    });
                    return true;
                }
                // respond with the default for all other messages
                if (msg.defaultResponse) {
                    await this.notify({
                        ...msg,
                        message: `${chalk.cyan(msg.message)} (auto-responded with default: ${util.format(msg.defaultResponse)})`,
                    });
                    return msg.defaultResponse;
                }
            }
            // only talk to user if STDIN is a terminal (otherwise, fail)
            if (!this.isTTY) {
                throw new toolkit_lib_1.ToolkitError(`${motivation}, but terminal (TTY) is not attached so we are unable to get a confirmation from the user`);
            }
            // only talk to user if concurrency is 1 (otherwise, fail)
            if (concurrency > 1) {
                throw new toolkit_lib_1.ToolkitError(`${motivation}, but concurrency is greater than 1 so we are unable to get a confirmation from the user`);
            }
            // Basic confirmation prompt
            // We treat all requests with a boolean response as confirmation prompts
            if (isConfirmationPrompt(msg)) {
                const confirmed = await promptly.confirm(`${chalk.cyan(msg.message)} (y/n)`);
                if (!confirmed) {
                    throw new toolkit_lib_1.ToolkitError('Aborted by user');
                }
                return confirmed;
            }
            // Asking for a specific value
            const prompt = extractPromptInfo(msg);
            const desc = responseDescription ?? prompt.default;
            const answer = await promptly.prompt(`${chalk.cyan(msg.message)}${desc ? ` (${desc})` : ''}`, {
                default: prompt.default,
                trim: true,
            });
            return prompt.convertAnswer(answer);
        });
        // We need to cast this because it is impossible to narrow the generic type
        // isPromptableRequest ensures that the response type is one we can prompt for
        // the remaining code ensure we are indeed returning the correct type
        return response;
    }
    /**
     * Formats a message for console output with optional color support
     */
    formatMessage(msg) {
        // apply provided style or a default style if we're in TTY mode
        let message_text = this.isTTY
            ? styleMap[msg.level](msg.message)
            : msg.message;
        // prepend timestamp if IoMessageLevel is DEBUG or TRACE. Postpend a newline.
        return ((msg.level === 'debug' || msg.level === 'trace')
            ? `[${this.formatTime(msg.time)}] ${message_text}`
            : message_text) + '\n';
    }
    /**
     * Formats date to HH:MM:SS
     */
    formatTime(d) {
        const pad = (n) => n.toString().padStart(2, '0');
        return `${pad(d.getHours())}:${pad(d.getMinutes())}:${pad(d.getSeconds())}`;
    }
    /**
     * Get an instance of the ActivityPrinter
     */
    makeActivityPrinter() {
        const props = {
            stream: this.selectStreamFromLevel('info'),
        };
        switch (this.stackProgress) {
            case deploy_1.StackActivityProgress.EVENTS:
                return new api_private_1.HistoryActivityPrinter(props);
            case deploy_1.StackActivityProgress.BAR:
                return new api_private_1.CurrentActivityPrinter(props);
        }
    }
}
exports.CliIoHost = CliIoHost;
/**
 * This IoHost implementation considers a request promptable, if:
 * - it's a yes/no confirmation
 * - asking for a string or number value
 */
function isPromptableRequest(msg) {
    return isConfirmationPrompt(msg)
        || typeof msg.defaultResponse === 'string'
        || typeof msg.defaultResponse === 'number';
}
/**
 * Check if the request is a confirmation prompt
 * We treat all requests with a boolean response as confirmation prompts
 */
function isConfirmationPrompt(msg) {
    return typeof msg.defaultResponse === 'boolean';
}
/**
 * Helper to extract information for promptly from the request
 */
function extractPromptInfo(msg) {
    const isNumber = (typeof msg.defaultResponse === 'number');
    const defaultResponse = util.format(msg.defaultResponse);
    return {
        default: defaultResponse,
        defaultDesc: 'defaultDescription' in msg && msg.defaultDescription ? util.format(msg.defaultDescription) : defaultResponse,
        convertAnswer: isNumber ? (v) => Number(v) : (v) => String(v),
    };
}
const styleMap = {
    error: chalk.red,
    warn: chalk.yellow,
    result: chalk.reset,
    info: chalk.reset,
    debug: chalk.gray,
    trace: chalk.gray,
};
function targetStreamObject(x) {
    switch (x) {
        case 'stderr':
            return process.stderr;
        case 'stdout':
            return process.stdout;
        case 'drop':
            return undefined;
    }
}
function isNoticesMessage(msg) {
    return api_private_1.IO.CDK_TOOLKIT_I0100.is(msg) || api_private_1.IO.CDK_TOOLKIT_W0101.is(msg) || api_private_1.IO.CDK_TOOLKIT_E0101.is(msg) || api_private_1.IO.CDK_TOOLKIT_I0101.is(msg);
}
function isTelemetryMessage(msg) {
    return messages_1.CLI_TELEMETRY_CODES.some((c) => c.is(msg));
}
function getEventType(msg) {
    switch (msg.code) {
        case messages_1.CLI_PRIVATE_IO.CDK_CLI_I1001.code:
            return 'SYNTH';
        case messages_1.CLI_PRIVATE_IO.CDK_CLI_I2001.code:
            return 'INVOKE';
        case messages_1.CLI_PRIVATE_IO.CDK_CLI_I3001.code:
            return 'DEPLOY';
        default:
            throw new toolkit_lib_1.ToolkitError(`Unrecognized Telemetry Message Code: ${msg.code}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xpLWlvLWhvc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjbGktaW8taG9zdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxrQ0FBa0M7QUFDbEMsMEVBQWlFO0FBQ2pFLHNEQUFvRDtBQUdwRCwrQkFBK0I7QUFDL0IscUNBQXFDO0FBRXJDLDBEQUFxSTtBQUNySSxrREFBOEQ7QUFDOUQsc0VBQXFFO0FBRXJFLG9EQUE0RTtBQUU1RSxrREFBd0Q7QUFDeEQsbUVBQXdFO0FBQ3hFLDJEQUFnRTtBQUNoRSxxREFBa0Q7QUFFbEQsbUNBQWtDO0FBb0ZsQzs7R0FFRztBQUNILE1BQWEsU0FBUztJQUNwQjs7T0FFRztJQUNILE1BQU0sQ0FBQyxRQUFRLENBQUMsUUFBd0IsRUFBRSxFQUFFLFFBQVEsR0FBRyxLQUFLO1FBQzFELElBQUksUUFBUSxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRSxDQUFDO1lBQ3JDLFNBQVMsQ0FBQyxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDN0MsQ0FBQztRQUNELE9BQU8sU0FBUyxDQUFDLFNBQVMsQ0FBQztJQUM3QixDQUFDO0lBRUQ7O09BRUc7SUFDSCxNQUFNLENBQUMsR0FBRztRQUNSLE9BQU8sU0FBUyxDQUFDLFNBQVMsQ0FBQztJQUM3QixDQUFDO0lBeURELFlBQW9CLFFBQXdCLEVBQUU7UUFyQjlDOzs7OztXQUtHO1FBQ0ksdUJBQWtCLEdBQWlCLFFBQVEsQ0FBQztRQUUzQyxjQUFTLEdBQTBCLDhCQUFxQixDQUFDLEdBQUcsQ0FBQztRQUtyRSxpQkFBaUI7UUFDVCxrQkFBYSxHQUFHLENBQUMsQ0FBQztRQUNULHdCQUFtQixHQUF5QixFQUFFLENBQUM7UUFPOUQsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxJQUFJLE1BQU0sQ0FBQztRQUNuRCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLElBQUksS0FBSyxDQUFDO1FBQzFELElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxNQUFNLENBQUM7UUFDekMsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxJQUFJLElBQUEsU0FBSSxHQUFFLENBQUM7UUFDakMsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxxQkFBcUIsSUFBSSx1Q0FBZSxDQUFDLFVBQVUsQ0FBQztRQUN2RixJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLElBQUksOEJBQXFCLENBQUMsR0FBRyxDQUFDO1FBQ3RFLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxLQUFLLENBQUM7SUFDaEQsQ0FBQztJQUVNLEtBQUssQ0FBQyxjQUFjLENBQUMsSUFBUyxFQUFFLE9BQWdCLEVBQUUsVUFBa0I7UUFDekUsSUFBSSxLQUFLLEdBQXFCLEVBQUUsQ0FBQztRQUNqQyxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2pELElBQUksaUJBQWlCLEVBQUUsQ0FBQztZQUN0QixJQUFJLENBQUM7Z0JBQ0gsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLDZCQUFpQixDQUFDO29CQUMvQixNQUFNLEVBQUUsSUFBSTtvQkFDWixXQUFXLEVBQUUsaUJBQWlCO2lCQUMvQixDQUFDLENBQUMsQ0FBQztnQkFDSixNQUFNLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLDBCQUEwQixDQUFDLENBQUM7WUFDckUsQ0FBQztZQUFDLE9BQU8sQ0FBTSxFQUFFLENBQUM7Z0JBQ2hCLE1BQU0sSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsd0NBQXdDLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1lBQzlGLENBQUM7UUFDSCxDQUFDO1FBRUQsTUFBTSxpQkFBaUIsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDO1FBQ3pELElBQUksSUFBQSx1Q0FBbUIsRUFBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUksaUJBQWlCLEVBQUUsQ0FBQztZQUM1RCxJQUFJLENBQUM7Z0JBQ0gsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLHFDQUFxQixDQUFDO29CQUNuQyxNQUFNLEVBQUUsSUFBSTtvQkFDWixLQUFLLEVBQUUsVUFBVTtvQkFDakIsUUFBUSxFQUFFLGlCQUFpQjtpQkFDNUIsQ0FBQyxDQUFDLENBQUM7Z0JBQ0osTUFBTSxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1lBQ3pFLENBQUM7WUFBQyxPQUFPLENBQU0sRUFBRSxDQUFDO2dCQUNoQixNQUFNLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLDRDQUE0QyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztZQUNsRyxDQUFDO1FBQ0gsQ0FBQzthQUFNLENBQUM7WUFDTixNQUFNLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0UsQ0FBQztRQUVELElBQUksS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUNyQixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksMEJBQWdCLENBQUM7Z0JBQ3BDLE1BQU0sRUFBRSxJQUFJO2dCQUNaLE1BQU0sRUFBRSxJQUFJLGVBQU0sQ0FBQyxFQUFFLEtBQUssRUFBRSxDQUFDO2dCQUM3QixTQUFTLEVBQUUsSUFBSTtnQkFDZixPQUFPLEVBQUUsT0FBTzthQUNqQixDQUFDLENBQUM7UUFDTCxDQUFDO1FBRUQsTUFBTSxJQUFJLENBQUMsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsYUFBYSxDQUFDLElBQTJCO1FBQ2xELElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDO0lBQ3hCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNILElBQVcsYUFBYTtRQUN0QiwyQkFBMkI7UUFDM0IsSUFBSSxJQUFJLENBQUMsU0FBUyxLQUFLLDhCQUFxQixDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ3BELE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQztRQUN4QixDQUFDO1FBRUQseUhBQXlIO1FBQ3pILE1BQU0sY0FBYyxHQUFHLElBQUEsdUNBQXlCLEVBQUMsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3BGLElBQUksY0FBYyxFQUFFLENBQUM7WUFDbkIsT0FBTyw4QkFBcUIsQ0FBQyxNQUFNLENBQUM7UUFDdEMsQ0FBQztRQUVELHdDQUF3QztRQUN4QyxNQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsUUFBUSxLQUFLLE9BQU8sQ0FBQztRQUMvQyxJQUFJLFNBQVMsRUFBRSxDQUFDO1lBQ2QsT0FBTyw4QkFBcUIsQ0FBQyxNQUFNLENBQUM7UUFDdEMsQ0FBQztRQUVELGlGQUFpRjtRQUNqRiw0REFBNEQ7UUFDNUQsMEZBQTBGO1FBQzFGLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSxDQUFDLEtBQUssSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7UUFDdEQsSUFBSSxDQUFDLG9CQUFvQixFQUFFLENBQUM7WUFDMUIsT0FBTyw4QkFBcUIsQ0FBQyxNQUFNLENBQUM7UUFDdEMsQ0FBQztRQUVELDBCQUEwQjtRQUMxQixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUM7SUFDeEIsQ0FBQztJQUVELElBQVcsUUFBUTtRQUNqQixPQUFPLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxRQUFRLENBQUM7SUFDcEMsQ0FBQztJQUVNLFVBQVU7UUFDZixPQUFPLElBQUEsd0JBQVUsRUFBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLGFBQW9CLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBSSxLQUF1QjtRQUN2RCxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7UUFDckIsSUFBSSxDQUFDO1lBQ0gsT0FBTyxNQUFNLEtBQUssRUFBRSxDQUFDO1FBQ3ZCLENBQUM7Z0JBQVMsQ0FBQztZQUNULElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUNyQixJQUFJLElBQUksQ0FBQyxhQUFhLEtBQUssQ0FBQyxFQUFFLENBQUM7Z0JBQzdCLCtDQUErQztnQkFDL0MsS0FBSyxNQUFNLFNBQVMsSUFBSSxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztvQkFDakQsTUFBTSxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUMvQixDQUFDO2dCQUNELHdDQUF3QztnQkFDeEMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNyQyxDQUFDO1FBQ0gsQ0FBQztJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSSxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQXVCO1FBQ3pDLE1BQU0sSUFBSSxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRW5DLElBQUksSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQzlCLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFLENBQUM7Z0JBQzFCLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUM7WUFDcEQsQ0FBQztZQUNELElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ2pDLE9BQU87UUFDVCxDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUEsdUNBQXlCLEVBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDO1lBQ25ELE9BQU87UUFDVCxDQUFDO1FBRUQsSUFBSSxJQUFJLENBQUMsYUFBYSxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQzNCLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDbkMsT0FBTztRQUNULENBQUM7UUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3ZDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDdEMsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRU8sS0FBSyxDQUFDLGtCQUFrQixDQUFDLEdBQXVCO1FBQ3RELElBQUksQ0FBQztZQUNILElBQUksSUFBSSxDQUFDLFNBQVMsSUFBSSxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDO2dCQUM5QyxNQUFNLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDO29CQUN4QixTQUFTLEVBQUUsWUFBWSxDQUFDLEdBQUcsQ0FBQztvQkFDNUIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUTtvQkFDM0IsS0FBSyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSztpQkFDdEIsQ0FBQyxDQUFDO1lBQ0wsQ0FBQztRQUNILENBQUM7UUFBQyxPQUFPLENBQU0sRUFBRSxDQUFDO1lBQ2hCLE1BQU0sSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMseUJBQXlCLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBQ2xFLENBQUM7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSyxlQUFlLENBQUMsR0FBdUI7UUFDN0MsT0FBTyxHQUFHLENBQUMsSUFBSSxJQUFJO1lBQ2pCLG1CQUFtQjtZQUNuQixtQkFBbUI7WUFDbkIsbUJBQW1CO1NBQ3BCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN2QixDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssZ0JBQWdCLENBQUMsR0FBd0I7UUFDL0MsTUFBTSxvQkFBb0IsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksSUFBSSxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQUUsQ0FBQztZQUMzRCxLQUFLLENBQUM7UUFDUixDQUFDO1FBRUQsUUFBUSxJQUFJLENBQUMscUJBQXFCLEVBQUUsQ0FBQztZQUNuQyx5QkFBeUI7WUFDekIsS0FBSyx1Q0FBZSxDQUFDLEtBQUs7Z0JBQ3hCLE9BQU8sSUFBSSxDQUFDO1lBQ2QsMEJBQTBCO1lBQzFCLEtBQUssdUNBQWUsQ0FBQyxTQUFTO2dCQUM1QixPQUFPLEtBQUssQ0FBQztZQUNmLDZEQUE2RDtZQUM3RCxLQUFLLHVDQUFlLENBQUMsVUFBVTtnQkFDN0IsT0FBTyxDQUFDLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLG9CQUFvQixDQUFDLENBQUM7UUFDL0UsQ0FBQztJQUNILENBQUM7SUFFRDs7T0FFRztJQUNLLFlBQVksQ0FBQyxHQUFtQjtRQUN0QyxJQUFJLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDMUIsT0FBTyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUNyRCxDQUFDO1FBRUQsT0FBTyxJQUFJLENBQUMscUJBQXFCLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7T0FFRztJQUNLLHFCQUFxQixDQUFDLEtBQXFCO1FBQ2pELDREQUE0RDtRQUM1RCxFQUFFO1FBQ0YseURBQXlEO1FBQ3pELHlEQUF5RDtRQUN6RCxnREFBZ0Q7UUFDaEQsc0VBQXNFO1FBQ3RFLEVBQUU7UUFDRixRQUFRLEtBQUssRUFBRSxDQUFDO1lBQ2QsS0FBSyxPQUFPO2dCQUNWLE9BQU8sT0FBTyxDQUFDLE1BQU0sQ0FBQztZQUN4QixLQUFLLFFBQVE7Z0JBQ1gsT0FBTyxPQUFPLENBQUMsTUFBTSxDQUFDO1lBQ3hCO2dCQUNFLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUN2RCxDQUFDO0lBQ0gsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksS0FBSyxDQUFDLGVBQWUsQ0FBeUIsR0FBc0M7UUFDekYscUZBQXFGO1FBQ3JGLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUN2QixPQUFPLEdBQUcsQ0FBQyxlQUFlLENBQUM7UUFDN0IsQ0FBQztRQUVELE1BQU0sUUFBUSxHQUFHLE1BQU0sSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssSUFBcUMsRUFBRTtZQUN4RixzQkFBc0I7WUFDdEIsZ0VBQWdFO1lBQ2hFLE1BQU0sSUFBSSxHQUlOLEdBQUcsQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDO1lBRW5CLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxVQUFVLElBQUksc0JBQXNCLENBQUM7WUFDN0QsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFdBQVcsSUFBSSxDQUFDLENBQUM7WUFDMUMsTUFBTSxtQkFBbUIsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUM7WUFFckQsMEJBQTBCO1lBQzFCLG9HQUFvRztZQUNwRyw2RkFBNkY7WUFDN0YsK0JBQStCO1lBQy9CLElBQUksSUFBSSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7Z0JBQy9CLE9BQU8sSUFBSSxDQUFDO1lBQ2QsQ0FBQztZQUVELGdEQUFnRDtZQUNoRCxJQUFJLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztnQkFDckIsd0NBQXdDO2dCQUN4QyxJQUFJLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7b0JBQzlCLE1BQU0sSUFBSSxDQUFDLE1BQU0sQ0FBQzt3QkFDaEIsR0FBRyxHQUFHO3dCQUNOLE9BQU8sRUFBRSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxtQkFBbUI7cUJBQ3ZELENBQUMsQ0FBQztvQkFDSCxPQUFPLElBQUksQ0FBQztnQkFDZCxDQUFDO2dCQUVELGtEQUFrRDtnQkFDbEQsSUFBSSxHQUFHLENBQUMsZUFBZSxFQUFFLENBQUM7b0JBQ3hCLE1BQU0sSUFBSSxDQUFDLE1BQU0sQ0FBQzt3QkFDaEIsR0FBRyxHQUFHO3dCQUNOLE9BQU8sRUFBRSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxrQ0FBa0MsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLEdBQUc7cUJBQ3pHLENBQUMsQ0FBQztvQkFDSCxPQUFPLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQzdCLENBQUM7WUFDSCxDQUFDO1lBRUQsNkRBQTZEO1lBQzdELElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQ2hCLE1BQU0sSUFBSSwwQkFBWSxDQUFDLEdBQUcsVUFBVSwyRkFBMkYsQ0FBQyxDQUFDO1lBQ25JLENBQUM7WUFFRCwwREFBMEQ7WUFDMUQsSUFBSSxXQUFXLEdBQUcsQ0FBQyxFQUFFLENBQUM7Z0JBQ3BCLE1BQU0sSUFBSSwwQkFBWSxDQUFDLEdBQUcsVUFBVSwwRkFBMEYsQ0FBQyxDQUFDO1lBQ2xJLENBQUM7WUFFRCw0QkFBNEI7WUFDNUIsd0VBQXdFO1lBQ3hFLElBQUksb0JBQW9CLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztnQkFDOUIsTUFBTSxTQUFTLEdBQUcsTUFBTSxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUM3RSxJQUFJLENBQUMsU0FBUyxFQUFFLENBQUM7b0JBQ2YsTUFBTSxJQUFJLDBCQUFZLENBQUMsaUJBQWlCLENBQUMsQ0FBQztnQkFDNUMsQ0FBQztnQkFDRCxPQUFPLFNBQVMsQ0FBQztZQUNuQixDQUFDO1lBRUQsOEJBQThCO1lBQzlCLE1BQU0sTUFBTSxHQUFHLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ3RDLE1BQU0sSUFBSSxHQUFHLG1CQUFtQixJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUM7WUFDbkQsTUFBTSxNQUFNLEdBQUcsTUFBTSxRQUFRLENBQUMsTUFBTSxDQUFDLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLElBQUksR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRTtnQkFDNUYsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPO2dCQUN2QixJQUFJLEVBQUUsSUFBSTthQUNYLENBQUMsQ0FBQztZQUNILE9BQU8sTUFBTSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN0QyxDQUFDLENBQUMsQ0FBQztRQUVILDJFQUEyRTtRQUMzRSw4RUFBOEU7UUFDOUUscUVBQXFFO1FBQ3JFLE9BQU8sUUFBd0IsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7O09BRUc7SUFDSyxhQUFhLENBQUMsR0FBdUI7UUFDM0MsK0RBQStEO1FBQy9ELElBQUksWUFBWSxHQUFHLElBQUksQ0FBQyxLQUFLO1lBQzNCLENBQUMsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUM7WUFDbEMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUM7UUFFaEIsNkVBQTZFO1FBQzdFLE9BQU8sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEtBQUssT0FBTyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEtBQUssT0FBTyxDQUFDO1lBQ3RELENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLFlBQVksRUFBRTtZQUNsRCxDQUFDLENBQUMsWUFBWSxDQUFDLEdBQUcsSUFBSSxDQUFDO0lBQzNCLENBQUM7SUFFRDs7T0FFRztJQUNLLFVBQVUsQ0FBQyxDQUFPO1FBQ3hCLE1BQU0sR0FBRyxHQUFHLENBQUMsQ0FBUyxFQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUNqRSxPQUFPLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDLENBQUMsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDLFVBQVUsRUFBRSxDQUFDLEVBQUUsQ0FBQztJQUM5RSxDQUFDO0lBRUQ7O09BRUc7SUFDSyxtQkFBbUI7UUFDekIsTUFBTSxLQUFLLEdBQXlCO1lBQ2xDLE1BQU0sRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsTUFBTSxDQUFDO1NBQzNDLENBQUM7UUFFRixRQUFRLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztZQUMzQixLQUFLLDhCQUFxQixDQUFDLE1BQU07Z0JBQy9CLE9BQU8sSUFBSSxvQ0FBc0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUMzQyxLQUFLLDhCQUFxQixDQUFDLEdBQUc7Z0JBQzVCLE9BQU8sSUFBSSxvQ0FBc0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM3QyxDQUFDO0lBQ0gsQ0FBQztDQUNGO0FBeGJELDhCQXdiQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLG1CQUFtQixDQUFDLEdBQXdCO0lBQ25ELE9BQU8sb0JBQW9CLENBQUMsR0FBRyxDQUFDO1dBQzNCLE9BQU8sR0FBRyxDQUFDLGVBQWUsS0FBSyxRQUFRO1dBQ3ZDLE9BQU8sR0FBRyxDQUFDLGVBQWUsS0FBSyxRQUFRLENBQUM7QUFDL0MsQ0FBQztBQUVEOzs7R0FHRztBQUNILFNBQVMsb0JBQW9CLENBQUMsR0FBd0I7SUFDcEQsT0FBTyxPQUFPLEdBQUcsQ0FBQyxlQUFlLEtBQUssU0FBUyxDQUFDO0FBQ2xELENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsaUJBQWlCLENBQUMsR0FBd0I7SUFLakQsTUFBTSxRQUFRLEdBQUcsQ0FBQyxPQUFPLEdBQUcsQ0FBQyxlQUFlLEtBQUssUUFBUSxDQUFDLENBQUM7SUFDM0QsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDekQsT0FBTztRQUNMLE9BQU8sRUFBRSxlQUFlO1FBQ3hCLFdBQVcsRUFBRSxvQkFBb0IsSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFlO1FBQzFILGFBQWEsRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO0tBQzlELENBQUM7QUFDSixDQUFDO0FBRUQsTUFBTSxRQUFRLEdBQW9EO0lBQ2hFLEtBQUssRUFBRSxLQUFLLENBQUMsR0FBRztJQUNoQixJQUFJLEVBQUUsS0FBSyxDQUFDLE1BQU07SUFDbEIsTUFBTSxFQUFFLEtBQUssQ0FBQyxLQUFLO0lBQ25CLElBQUksRUFBRSxLQUFLLENBQUMsS0FBSztJQUNqQixLQUFLLEVBQUUsS0FBSyxDQUFDLElBQUk7SUFDakIsS0FBSyxFQUFFLEtBQUssQ0FBQyxJQUFJO0NBQ2xCLENBQUM7QUFFRixTQUFTLGtCQUFrQixDQUFDLENBQWU7SUFDekMsUUFBUSxDQUFDLEVBQUUsQ0FBQztRQUNWLEtBQUssUUFBUTtZQUNYLE9BQU8sT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUN4QixLQUFLLFFBQVE7WUFDWCxPQUFPLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDeEIsS0FBSyxNQUFNO1lBQ1QsT0FBTyxTQUFTLENBQUM7SUFDckIsQ0FBQztBQUNILENBQUM7QUFFRCxTQUFTLGdCQUFnQixDQUFDLEdBQXVCO0lBQy9DLE9BQU8sZ0JBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksZ0JBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksZ0JBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLElBQUksZ0JBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7QUFDdEksQ0FBQztBQUVELFNBQVMsa0JBQWtCLENBQUMsR0FBdUI7SUFDakQsT0FBTyw4QkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztBQUNwRCxDQUFDO0FBRUQsU0FBUyxZQUFZLENBQUMsR0FBdUI7SUFDM0MsUUFBUSxHQUFHLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDakIsS0FBSyx5QkFBYyxDQUFDLGFBQWEsQ0FBQyxJQUFJO1lBQ3BDLE9BQU8sT0FBTyxDQUFDO1FBQ2pCLEtBQUsseUJBQWMsQ0FBQyxhQUFhLENBQUMsSUFBSTtZQUNwQyxPQUFPLFFBQVEsQ0FBQztRQUNsQixLQUFLLHlCQUFjLENBQUMsYUFBYSxDQUFDLElBQUk7WUFDcEMsT0FBTyxRQUFRLENBQUM7UUFDbEI7WUFDRSxNQUFNLElBQUksMEJBQVksQ0FBQyx3Q0FBd0MsR0FBRyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7SUFDL0UsQ0FBQztBQUNILENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgdHlwZSB7IEFnZW50IH0gZnJvbSAnbm9kZTpodHRwcyc7XG5pbXBvcnQgKiBhcyB1dGlsIGZyb20gJ25vZGU6dXRpbCc7XG5pbXBvcnQgeyBSZXF1aXJlQXBwcm92YWwgfSBmcm9tICdAYXdzLWNkay9jbG91ZC1hc3NlbWJseS1zY2hlbWEnO1xuaW1wb3J0IHsgVG9vbGtpdEVycm9yIH0gZnJvbSAnQGF3cy1jZGsvdG9vbGtpdC1saWInO1xuaW1wb3J0IHR5cGUgeyBJSW9Ib3N0LCBJb01lc3NhZ2UsIElvTWVzc2FnZUNvZGUsIElvTWVzc2FnZUxldmVsLCBJb1JlcXVlc3QsIFRvb2xraXRBY3Rpb24gfSBmcm9tICdAYXdzLWNkay90b29sa2l0LWxpYic7XG5pbXBvcnQgdHlwZSB7IENvbnRleHQgfSBmcm9tICdAYXdzLWNkay90b29sa2l0LWxpYi9saWIvYXBpJztcbmltcG9ydCAqIGFzIGNoYWxrIGZyb20gJ2NoYWxrJztcbmltcG9ydCAqIGFzIHByb21wdGx5IGZyb20gJ3Byb21wdGx5JztcbmltcG9ydCB0eXBlIHsgSW9IZWxwZXIsIEFjdGl2aXR5UHJpbnRlclByb3BzLCBJQWN0aXZpdHlQcmludGVyIH0gZnJvbSAnLi4vLi4vLi4vbGliL2FwaS1wcml2YXRlJztcbmltcG9ydCB7IGFzSW9IZWxwZXIsIElPLCBpc01lc3NhZ2VSZWxldmFudEZvckxldmVsLCBDdXJyZW50QWN0aXZpdHlQcmludGVyLCBIaXN0b3J5QWN0aXZpdHlQcmludGVyIH0gZnJvbSAnLi4vLi4vLi4vbGliL2FwaS1wcml2YXRlJztcbmltcG9ydCB7IFN0YWNrQWN0aXZpdHlQcm9ncmVzcyB9IGZyb20gJy4uLy4uL2NvbW1hbmRzL2RlcGxveSc7XG5pbXBvcnQgeyBjYW5Db2xsZWN0VGVsZW1ldHJ5IH0gZnJvbSAnLi4vdGVsZW1ldHJ5L2NvbGxlY3QtdGVsZW1ldHJ5JztcbmltcG9ydCB0eXBlIHsgRXZlbnRSZXN1bHQgfSBmcm9tICcuLi90ZWxlbWV0cnkvbWVzc2FnZXMnO1xuaW1wb3J0IHsgQ0xJX1BSSVZBVEVfSU8sIENMSV9URUxFTUVUUllfQ09ERVMgfSBmcm9tICcuLi90ZWxlbWV0cnkvbWVzc2FnZXMnO1xuaW1wb3J0IHR5cGUgeyBFdmVudFR5cGUgfSBmcm9tICcuLi90ZWxlbWV0cnkvc2NoZW1hJztcbmltcG9ydCB7IFRlbGVtZXRyeVNlc3Npb24gfSBmcm9tICcuLi90ZWxlbWV0cnkvc2Vzc2lvbic7XG5pbXBvcnQgeyBFbmRwb2ludFRlbGVtZXRyeVNpbmsgfSBmcm9tICcuLi90ZWxlbWV0cnkvc2luay9lbmRwb2ludC1zaW5rJztcbmltcG9ydCB7IEZpbGVUZWxlbWV0cnlTaW5rIH0gZnJvbSAnLi4vdGVsZW1ldHJ5L3NpbmsvZmlsZS1zaW5rJztcbmltcG9ydCB7IEZ1bm5lbCB9IGZyb20gJy4uL3RlbGVtZXRyeS9zaW5rL2Z1bm5lbCc7XG5pbXBvcnQgdHlwZSB7IElUZWxlbWV0cnlTaW5rIH0gZnJvbSAnLi4vdGVsZW1ldHJ5L3Npbmsvc2luay1pbnRlcmZhY2UnO1xuaW1wb3J0IHsgaXNDSSB9IGZyb20gJy4uL3V0aWwvY2knO1xuXG5leHBvcnQgdHlwZSB7IElJb0hvc3QsIElvTWVzc2FnZSwgSW9NZXNzYWdlQ29kZSwgSW9NZXNzYWdlTGV2ZWwsIElvUmVxdWVzdCB9O1xuXG4vKipcbiAqIFRoZSBjdXJyZW50IGFjdGlvbiBiZWluZyBwZXJmb3JtZWQgYnkgdGhlIENMSS4gJ25vbmUnIHJlcHJlc2VudHMgdGhlIGFic2VuY2Ugb2YgYW4gYWN0aW9uLlxuICovXG50eXBlIENsaUFjdGlvbiA9XG58IFRvb2xraXRBY3Rpb25cbnwgJ2NvbnRleHQnXG58ICdkb2NzJ1xufCAnZmxhZ3MnXG58ICdub3RpY2VzJ1xufCAndmVyc2lvbidcbnwgJ2NsaS10ZWxlbWV0cnknXG58ICdub25lJztcblxuZXhwb3J0IGludGVyZmFjZSBDbGlJb0hvc3RQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgaW5pdGlhbCBUb29sa2l0IGFjdGlvbiB0aGUgaG9zdHMgc3RhcnRzIHdpdGguXG4gICAqXG4gICAqIEBkZWZhdWx0ICdub25lJ1xuICAgKi9cbiAgcmVhZG9ubHkgY3VycmVudEFjdGlvbj86IENsaUFjdGlvbjtcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyB0aGUgdmVyYm9zaXR5IG9mIHRoZSBvdXRwdXQuXG4gICAqXG4gICAqIFRoZSBDbGlJb0hvc3Qgd2lsbCBzdGlsbCByZWNlaXZlIGFsbCBtZXNzYWdlcyBhbmQgcmVxdWVzdHMsXG4gICAqIGJ1dCBvbmx5IHRoZSBtZXNzYWdlcyBpbmNsdWRlZCBpbiB0aGlzIGxldmVsIHdpbGwgYmUgcHJpbnRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgJ2luZm8nXG4gICAqL1xuICByZWFkb25seSBsb2dMZXZlbD86IElvTWVzc2FnZUxldmVsO1xuXG4gIC8qKlxuICAgKiBPdmVycmlkZXMgdGhlIGF1dG9tYXRpYyBUVFkgZGV0ZWN0aW9uLlxuICAgKlxuICAgKiBXaGVuIFRUWSBpcyBkaXNhYmxlZCwgdGhlIENMSSB3aWxsIGhhdmUgbm8gaW50ZXJhY3Rpb25zIG9yIGNvbG9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRldGVybWluZWQgZnJvbSB0aGUgY3VycmVudCBwcm9jZXNzXG4gICAqL1xuICByZWFkb25seSBpc1RUWT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhlIENsaUlvSG9zdCBpcyBydW5uaW5nIGluIENJIG1vZGUuXG4gICAqXG4gICAqIEluIENJIG1vZGUsIGFsbCBub24tZXJyb3Igb3V0cHV0IGdvZXMgdG8gc3Rkb3V0IGluc3RlYWQgb2Ygc3RkZXJyLlxuICAgKiBTZXQgdG8gZmFsc2UgaW4gdGhlIENsaUlvSG9zdCBjb25zdHJ1Y3RvciBpdCB3aWxsIGJlIG92ZXJ3cml0dGVuIGlmIHRoZSBDTEkgQ0kgYXJndW1lbnQgaXMgcGFzc2VkXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZGV0ZXJtaW5lZCBmcm9tIHRoZSBlbnZpcm9ubWVudCwgc3BlY2lmaWNhbGx5IGJhc2VkIG9uIGBwcm9jZXNzLmVudi5DSWBcbiAgICovXG4gIHJlYWRvbmx5IGlzQ0k/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJbiB3aGF0IHNjZW5hcmlvcyBzaG91bGQgdGhlIENsaUlvSG9zdCBhc2sgZm9yIGFwcHJvdmFsXG4gICAqXG4gICAqIEBkZWZhdWx0IFJlcXVpcmVBcHByb3ZhbC5CUk9BREVOSU5HXG4gICAqL1xuICByZWFkb25seSByZXF1aXJlRGVwbG95QXBwcm92YWw/OiBSZXF1aXJlQXBwcm92YWw7XG5cbiAgLyoqXG4gICAqIFRoZSBpbml0aWFsIFRvb2xraXQgYWN0aW9uIHRoZSBob3N0cyBzdGFydHMgd2l0aC5cbiAgICpcbiAgICogQGRlZmF1bHQgU3RhY2tBY3Rpdml0eVByb2dyZXNzLkJBUlxuICAgKi9cbiAgcmVhZG9ubHkgc3RhY2tQcm9ncmVzcz86IFN0YWNrQWN0aXZpdHlQcm9ncmVzcztcblxuICAvKipcbiAgICogV2hldGhlciB0aGUgQ0xJIHNob3VsZCBhdHRlbXB0IHRvIGF1dG9tYXRpY2FsbHkgcmVzcG9uZCB0byBwcm9tcHRzLlxuICAgKlxuICAgKiBXaGVuIHRydWUsIG9wZXJhdGlvbiB3aWxsIHVzdWFsbHkgcHJvY2VlZCB3aXRob3V0IGludGVyYWN0aXZlIGNvbmZpcm1hdGlvbi5cbiAgICogQ29uZmlybWF0aW9ucyBhcmUgcmVzcG9uZGVkIHRvIHdpdGggeWVzLiBPdGhlciBwcm9tcHRzIHdpbGwgcmVzcG9uZCB3aXRoIHRoZSBkZWZhdWx0IHZhbHVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0b1Jlc3BvbmQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEEgdHlwZSBmb3IgY29uZmlndXJpbmcgYSB0YXJnZXQgc3RyZWFtXG4gKi9cbmV4cG9ydCB0eXBlIFRhcmdldFN0cmVhbSA9ICdzdGRvdXQnIHwgJ3N0ZGVycicgfCAnZHJvcCc7XG5cbi8qKlxuICogQSBzaW1wbGUgSU8gaG9zdCBmb3IgdGhlIENMSSB0aGF0IHdyaXRlcyBtZXNzYWdlcyB0byB0aGUgY29uc29sZS5cbiAqL1xuZXhwb3J0IGNsYXNzIENsaUlvSG9zdCBpbXBsZW1lbnRzIElJb0hvc3Qge1xuICAvKipcbiAgICogUmV0dXJucyB0aGUgc2luZ2xldG9uIGluc3RhbmNlXG4gICAqL1xuICBzdGF0aWMgaW5zdGFuY2UocHJvcHM6IENsaUlvSG9zdFByb3BzID0ge30sIGZvcmNlTmV3ID0gZmFsc2UpOiBDbGlJb0hvc3Qge1xuICAgIGlmIChmb3JjZU5ldyB8fCAhQ2xpSW9Ib3N0Ll9pbnN0YW5jZSkge1xuICAgICAgQ2xpSW9Ib3N0Ll9pbnN0YW5jZSA9IG5ldyBDbGlJb0hvc3QocHJvcHMpO1xuICAgIH1cbiAgICByZXR1cm4gQ2xpSW9Ib3N0Ll9pbnN0YW5jZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBzaW5nbGV0b24gaW5zdGFuY2UgaWYgaXQgZXhpc3RzXG4gICAqL1xuICBzdGF0aWMgZ2V0KCk6IENsaUlvSG9zdCB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIENsaUlvSG9zdC5faW5zdGFuY2U7XG4gIH1cblxuICAvKipcbiAgICogU2luZ2xldG9uIGluc3RhbmNlIG9mIHRoZSBDbGlJb0hvc3RcbiAgICovXG4gIHByaXZhdGUgc3RhdGljIF9pbnN0YW5jZTogQ2xpSW9Ib3N0IHwgdW5kZWZpbmVkO1xuXG4gIC8qKlxuICAgKiBUaGUgY3VycmVudCBhY3Rpb24gYmVpbmcgcGVyZm9ybWVkIGJ5IHRoZSBDTEkuXG4gICAqL1xuICBwdWJsaWMgY3VycmVudEFjdGlvbjogQ2xpQWN0aW9uO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBDbGlJb0hvc3QgaXMgcnVubmluZyBpbiBDSSBtb2RlLlxuICAgKlxuICAgKiBJbiBDSSBtb2RlLCBhbGwgbm9uLWVycm9yIG91dHB1dCBnb2VzIHRvIHN0ZG91dCBpbnN0ZWFkIG9mIHN0ZGVyci5cbiAgICovXG4gIHB1YmxpYyBpc0NJOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBob3N0IGNhbiB1c2UgaW50ZXJhY3Rpb25zIGFuZCBtZXNzYWdlIHN0eWxpbmcuXG4gICAqL1xuICBwdWJsaWMgaXNUVFk6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBjdXJyZW50IHRocmVzaG9sZC5cbiAgICpcbiAgICogTWVzc2FnZXMgd2l0aCBhIGxvd2VyIHByaW9yaXR5IGxldmVsIHdpbGwgYmUgaWdub3JlZC5cbiAgICovXG4gIHB1YmxpYyBsb2dMZXZlbDogSW9NZXNzYWdlTGV2ZWw7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25kaXRpb25zIGZvciByZXF1aXJpbmcgYXBwcm92YWwgaW4gdGhpcyBDbGlJb0hvc3QuXG4gICAqL1xuICBwdWJsaWMgcmVxdWlyZURlcGxveUFwcHJvdmFsOiBSZXF1aXJlQXBwcm92YWw7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZSB0aGUgdGFyZ2V0IHN0cmVhbSBmb3Igbm90aWNlc1xuICAgKlxuICAgKiAoTm90IGEgc2V0dGVyIGJlY2F1c2UgdGhlcmUncyBubyBuZWVkIGZvciBhZGRpdGlvbmFsIGxvZ2ljIHdoZW4gdGhpcyB2YWx1ZVxuICAgKiBpcyBjaGFuZ2VkIHlldClcbiAgICovXG4gIHB1YmxpYyBub3RpY2VzRGVzdGluYXRpb246IFRhcmdldFN0cmVhbSA9ICdzdGRlcnInO1xuXG4gIHByaXZhdGUgX3Byb2dyZXNzOiBTdGFja0FjdGl2aXR5UHJvZ3Jlc3MgPSBTdGFja0FjdGl2aXR5UHJvZ3Jlc3MuQkFSO1xuXG4gIC8vIFN0YWNrIEFjdGl2aXR5IFByaW50ZXJcbiAgcHJpdmF0ZSBhY3Rpdml0eVByaW50ZXI/OiBJQWN0aXZpdHlQcmludGVyO1xuXG4gIC8vIENvcmtlZCBMb2dnaW5nXG4gIHByaXZhdGUgY29ya2VkQ291bnRlciA9IDA7XG4gIHByaXZhdGUgcmVhZG9ubHkgY29ya2VkTG9nZ2luZ0J1ZmZlcjogSW9NZXNzYWdlPHVua25vd24+W10gPSBbXTtcblxuICBwcml2YXRlIHJlYWRvbmx5IGF1dG9SZXNwb25kOiBib29sZWFuO1xuXG4gIHB1YmxpYyB0ZWxlbWV0cnk/OiBUZWxlbWV0cnlTZXNzaW9uO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHJvcHM6IENsaUlvSG9zdFByb3BzID0ge30pIHtcbiAgICB0aGlzLmN1cnJlbnRBY3Rpb24gPSBwcm9wcy5jdXJyZW50QWN0aW9uID8/ICdub25lJztcbiAgICB0aGlzLmlzVFRZID0gcHJvcHMuaXNUVFkgPz8gcHJvY2Vzcy5zdGRvdXQuaXNUVFkgPz8gZmFsc2U7XG4gICAgdGhpcy5sb2dMZXZlbCA9IHByb3BzLmxvZ0xldmVsID8/ICdpbmZvJztcbiAgICB0aGlzLmlzQ0kgPSBwcm9wcy5pc0NJID8/IGlzQ0koKTtcbiAgICB0aGlzLnJlcXVpcmVEZXBsb3lBcHByb3ZhbCA9IHByb3BzLnJlcXVpcmVEZXBsb3lBcHByb3ZhbCA/PyBSZXF1aXJlQXBwcm92YWwuQlJPQURFTklORztcbiAgICB0aGlzLnN0YWNrUHJvZ3Jlc3MgPSBwcm9wcy5zdGFja1Byb2dyZXNzID8/IFN0YWNrQWN0aXZpdHlQcm9ncmVzcy5CQVI7XG4gICAgdGhpcy5hdXRvUmVzcG9uZCA9IHByb3BzLmF1dG9SZXNwb25kID8/IGZhbHNlO1xuICB9XG5cbiAgcHVibGljIGFzeW5jIHN0YXJ0VGVsZW1ldHJ5KGFyZ3M6IGFueSwgY29udGV4dDogQ29udGV4dCwgcHJveHlBZ2VudD86IEFnZW50KSB7XG4gICAgbGV0IHNpbmtzOiBJVGVsZW1ldHJ5U2lua1tdID0gW107XG4gICAgY29uc3QgdGVsZW1ldHJ5RmlsZVBhdGggPSBhcmdzWyd0ZWxlbWV0cnktZmlsZSddO1xuICAgIGlmICh0ZWxlbWV0cnlGaWxlUGF0aCkge1xuICAgICAgdHJ5IHtcbiAgICAgICAgc2lua3MucHVzaChuZXcgRmlsZVRlbGVtZXRyeVNpbmsoe1xuICAgICAgICAgIGlvSG9zdDogdGhpcyxcbiAgICAgICAgICBsb2dGaWxlUGF0aDogdGVsZW1ldHJ5RmlsZVBhdGgsXG4gICAgICAgIH0pKTtcbiAgICAgICAgYXdhaXQgdGhpcy5hc0lvSGVscGVyKCkuZGVmYXVsdHMudHJhY2UoJ0ZpbGUgVGVsZW1ldHJ5IGNvbm5lY3RlZCcpO1xuICAgICAgfSBjYXRjaCAoZTogYW55KSB7XG4gICAgICAgIGF3YWl0IHRoaXMuYXNJb0hlbHBlcigpLmRlZmF1bHRzLnRyYWNlKGBGaWxlIFRlbGVtZXRyeSBpbnN0YW50aWF0aW9uIGZhaWxlZDogJHtlLm1lc3NhZ2V9YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgdGVsZW1ldHJ5RW5kcG9pbnQgPSBwcm9jZXNzLmVudi5URUxFTUVUUllfRU5EUE9JTlQ7XG4gICAgaWYgKGNhbkNvbGxlY3RUZWxlbWV0cnkoYXJncywgY29udGV4dCkgJiYgdGVsZW1ldHJ5RW5kcG9pbnQpIHtcbiAgICAgIHRyeSB7XG4gICAgICAgIHNpbmtzLnB1c2gobmV3IEVuZHBvaW50VGVsZW1ldHJ5U2luayh7XG4gICAgICAgICAgaW9Ib3N0OiB0aGlzLFxuICAgICAgICAgIGFnZW50OiBwcm94eUFnZW50LFxuICAgICAgICAgIGVuZHBvaW50OiB0ZWxlbWV0cnlFbmRwb2ludCxcbiAgICAgICAgfSkpO1xuICAgICAgICBhd2FpdCB0aGlzLmFzSW9IZWxwZXIoKS5kZWZhdWx0cy50cmFjZSgnRW5kcG9pbnQgVGVsZW1ldHJ5IGNvbm5lY3RlZCcpO1xuICAgICAgfSBjYXRjaCAoZTogYW55KSB7XG4gICAgICAgIGF3YWl0IHRoaXMuYXNJb0hlbHBlcigpLmRlZmF1bHRzLnRyYWNlKGBFbmRwb2ludCBUZWxlbWV0cnkgaW5zdGFudGlhdGlvbiBmYWlsZWQ6ICR7ZS5tZXNzYWdlfWApO1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBhd2FpdCB0aGlzLmFzSW9IZWxwZXIoKS5kZWZhdWx0cy50cmFjZSgnRW5kcG9pbnQgVGVsZW1ldHJ5IE5PVCBjb25uZWN0ZWQnKTtcbiAgICB9XG5cbiAgICBpZiAoc2lua3MubGVuZ3RoID4gMCkge1xuICAgICAgdGhpcy50ZWxlbWV0cnkgPSBuZXcgVGVsZW1ldHJ5U2Vzc2lvbih7XG4gICAgICAgIGlvSG9zdDogdGhpcyxcbiAgICAgICAgY2xpZW50OiBuZXcgRnVubmVsKHsgc2lua3MgfSksXG4gICAgICAgIGFyZ3VtZW50czogYXJncyxcbiAgICAgICAgY29udGV4dDogY29udGV4dCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGF3YWl0IHRoaXMudGVsZW1ldHJ5Py5iZWdpbigpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSB0aGUgc3RhY2tQcm9ncmVzcyBwcmVmZXJlbmNlLlxuICAgKi9cbiAgcHVibGljIHNldCBzdGFja1Byb2dyZXNzKHR5cGU6IFN0YWNrQWN0aXZpdHlQcm9ncmVzcykge1xuICAgIHRoaXMuX3Byb2dyZXNzID0gdHlwZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIHRoZSBzdGFja1Byb2dyZXNzIHZhbHVlLlxuICAgKlxuICAgKiBUaGlzIHRha2VzIGludG8gYWNjb3VudCBvdGhlciBzdGF0ZSBvZiB0aGUgaW9Ib3N0LFxuICAgKiBsaWtlIGlmIGlzVFRZIGFuZCBpc0NJLlxuICAgKi9cbiAgcHVibGljIGdldCBzdGFja1Byb2dyZXNzKCk6IFN0YWNrQWN0aXZpdHlQcm9ncmVzcyB7XG4gICAgLy8gV2UgY2FuIGFsd2F5cyB1c2UgRVZFTlRTXG4gICAgaWYgKHRoaXMuX3Byb2dyZXNzID09PSBTdGFja0FjdGl2aXR5UHJvZ3Jlc3MuRVZFTlRTKSB7XG4gICAgICByZXR1cm4gdGhpcy5fcHJvZ3Jlc3M7XG4gICAgfVxuXG4gICAgLy8gaWYgYSBkZWJ1ZyBtZXNzYWdlIChhbmQgdGh1cyBhbnkgbW9yZSB2ZXJib3NlIG1lc3NhZ2VzKSBhcmUgcmVsZXZhbnQgdG8gdGhlIGN1cnJlbnQgbG9nIGxldmVsLCB3ZSBoYXZlIHZlcmJvc2UgbG9nZ2luZ1xuICAgIGNvbnN0IHZlcmJvc2VMb2dnaW5nID0gaXNNZXNzYWdlUmVsZXZhbnRGb3JMZXZlbCh7IGxldmVsOiAnZGVidWcnIH0sIHRoaXMubG9nTGV2ZWwpO1xuICAgIGlmICh2ZXJib3NlTG9nZ2luZykge1xuICAgICAgcmV0dXJuIFN0YWNrQWN0aXZpdHlQcm9ncmVzcy5FVkVOVFM7XG4gICAgfVxuXG4gICAgLy8gT24gV2luZG93cyB3ZSBjYW5ub3QgdXNlIGZhbmN5IG91dHB1dFxuICAgIGNvbnN0IGlzV2luZG93cyA9IHByb2Nlc3MucGxhdGZvcm0gPT09ICd3aW4zMic7XG4gICAgaWYgKGlzV2luZG93cykge1xuICAgICAgcmV0dXJuIFN0YWNrQWN0aXZpdHlQcm9ncmVzcy5FVkVOVFM7XG4gICAgfVxuXG4gICAgLy8gT24gc29tZSBDSSBzeXN0ZW1zIChzdWNoIGFzIENpcmNsZUNJKSBvdXRwdXQgc3RpbGwgcmVwb3J0cyBhcyBhIFRUWSBzbyB3ZSBhbHNvXG4gICAgLy8gbmVlZCBhbiBpbmRpdmlkdWFsIGNoZWNrIGZvciB3aGV0aGVyIHdlJ3JlIHJ1bm5pbmcgb24gQ0kuXG4gICAgLy8gc2VlOiBodHRwczovL2Rpc2N1c3MuY2lyY2xlY2kuY29tL3QvY2lyY2xlY2ktdGVybWluYWwtaXMtYS10dHktYnV0LXRlcm0taXMtbm90LXNldC85OTY1XG4gICAgY29uc3QgZmFuY3lPdXRwdXRBdmFpbGFibGUgPSB0aGlzLmlzVFRZICYmICF0aGlzLmlzQ0k7XG4gICAgaWYgKCFmYW5jeU91dHB1dEF2YWlsYWJsZSkge1xuICAgICAgcmV0dXJuIFN0YWNrQWN0aXZpdHlQcm9ncmVzcy5FVkVOVFM7XG4gICAgfVxuXG4gICAgLy8gVXNlIHRoZSB1c2VyIHByZWZlcmVuY2VcbiAgICByZXR1cm4gdGhpcy5fcHJvZ3Jlc3M7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGRlZmF1bHRzKCkge1xuICAgIHJldHVybiB0aGlzLmFzSW9IZWxwZXIoKS5kZWZhdWx0cztcbiAgfVxuXG4gIHB1YmxpYyBhc0lvSGVscGVyKCk6IElvSGVscGVyIHtcbiAgICByZXR1cm4gYXNJb0hlbHBlcih0aGlzLCB0aGlzLmN1cnJlbnRBY3Rpb24gYXMgYW55KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFeGVjdXRlcyBhIGJsb2NrIG9mIGNvZGUgd2l0aCBjb3JrZWQgbG9nZ2luZy4gQWxsIGxvZyBtZXNzYWdlcyBkdXJpbmcgZXhlY3V0aW9uXG4gICAqIGFyZSBidWZmZXJlZCBhbmQgb25seSB3cml0dGVuIHdoZW4gYWxsIG5lc3RlZCBjb3JrIGJsb2NrcyBjb21wbGV0ZSAod2hlbiBDT1JLX0NPVU5URVIgcmVhY2hlcyAwKS5cbiAgICogVGhlIGNvcmtpbmcgaXMgYm91bmQgdG8gdGhlIHNwZWNpZmljIGluc3RhbmNlIG9mIHRoZSBDbGlJb0hvc3QuXG4gICAqXG4gICAqIEBwYXJhbSBibG9jayAtIEFzeW5jIGZ1bmN0aW9uIHRvIGV4ZWN1dGUgd2l0aCBjb3JrZWQgbG9nZ2luZ1xuICAgKiBAcmV0dXJucyBQcm9taXNlIHRoYXQgcmVzb2x2ZXMgd2l0aCB0aGUgYmxvY2sncyByZXR1cm4gdmFsdWVcbiAgICovXG4gIHB1YmxpYyBhc3luYyB3aXRoQ29ya2VkTG9nZ2luZzxUPihibG9jazogKCkgPT4gUHJvbWlzZTxUPik6IFByb21pc2U8VD4ge1xuICAgIHRoaXMuY29ya2VkQ291bnRlcisrO1xuICAgIHRyeSB7XG4gICAgICByZXR1cm4gYXdhaXQgYmxvY2soKTtcbiAgICB9IGZpbmFsbHkge1xuICAgICAgdGhpcy5jb3JrZWRDb3VudGVyLS07XG4gICAgICBpZiAodGhpcy5jb3JrZWRDb3VudGVyID09PSAwKSB7XG4gICAgICAgIC8vIFByb2Nlc3MgZWFjaCBidWZmZXJlZCBtZXNzYWdlIHRocm91Z2ggbm90aWZ5XG4gICAgICAgIGZvciAoY29uc3QgaW9NZXNzYWdlIG9mIHRoaXMuY29ya2VkTG9nZ2luZ0J1ZmZlcikge1xuICAgICAgICAgIGF3YWl0IHRoaXMubm90aWZ5KGlvTWVzc2FnZSk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gcmVtb3ZlIGFsbCBidWZmZXJlZCBtZXNzYWdlcyBpbi1wbGFjZVxuICAgICAgICB0aGlzLmNvcmtlZExvZ2dpbmdCdWZmZXIuc3BsaWNlKDApO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBOb3RpZmllcyB0aGUgaG9zdCBvZiBhIG1lc3NhZ2UuXG4gICAqIFRoZSBjYWxsZXIgd2FpdHMgdW50aWwgdGhlIG5vdGlmaWNhdGlvbiBjb21wbGV0ZXMuXG4gICAqL1xuICBwdWJsaWMgYXN5bmMgbm90aWZ5KG1zZzogSW9NZXNzYWdlPHVua25vd24+KTogUHJvbWlzZTx2b2lkPiB7XG4gICAgYXdhaXQgdGhpcy5tYXliZUVtaXRUZWxlbWV0cnkobXNnKTtcblxuICAgIGlmICh0aGlzLmlzU3RhY2tBY3Rpdml0eShtc2cpKSB7XG4gICAgICBpZiAoIXRoaXMuYWN0aXZpdHlQcmludGVyKSB7XG4gICAgICAgIHRoaXMuYWN0aXZpdHlQcmludGVyID0gdGhpcy5tYWtlQWN0aXZpdHlQcmludGVyKCk7XG4gICAgICB9XG4gICAgICB0aGlzLmFjdGl2aXR5UHJpbnRlci5ub3RpZnkobXNnKTtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBpZiAoIWlzTWVzc2FnZVJlbGV2YW50Rm9yTGV2ZWwobXNnLCB0aGlzLmxvZ0xldmVsKSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmNvcmtlZENvdW50ZXIgPiAwKSB7XG4gICAgICB0aGlzLmNvcmtlZExvZ2dpbmdCdWZmZXIucHVzaChtc2cpO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IG91dHB1dCA9IHRoaXMuZm9ybWF0TWVzc2FnZShtc2cpO1xuICAgIGNvbnN0IHN0cmVhbSA9IHRoaXMuc2VsZWN0U3RyZWFtKG1zZyk7XG4gICAgc3RyZWFtPy53cml0ZShvdXRwdXQpO1xuICB9XG5cbiAgcHJpdmF0ZSBhc3luYyBtYXliZUVtaXRUZWxlbWV0cnkobXNnOiBJb01lc3NhZ2U8dW5rbm93bj4pIHtcbiAgICB0cnkge1xuICAgICAgaWYgKHRoaXMudGVsZW1ldHJ5ICYmIGlzVGVsZW1ldHJ5TWVzc2FnZShtc2cpKSB7XG4gICAgICAgIGF3YWl0IHRoaXMudGVsZW1ldHJ5LmVtaXQoe1xuICAgICAgICAgIGV2ZW50VHlwZTogZ2V0RXZlbnRUeXBlKG1zZyksXG4gICAgICAgICAgZHVyYXRpb246IG1zZy5kYXRhLmR1cmF0aW9uLFxuICAgICAgICAgIGVycm9yOiBtc2cuZGF0YS5lcnJvcixcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfSBjYXRjaCAoZTogYW55KSB7XG4gICAgICBhd2FpdCB0aGlzLmRlZmF1bHRzLnRyYWNlKGBFbWl0IFRlbGVtZXRyeSBGYWlsZWQgJHtlLm1lc3NhZ2V9YCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIERldGVjdCBzdGFjayBhY3Rpdml0eSBtZXNzYWdlcyBzbyB0aGV5IGNhbiBiZSBzZW5kIHRvIHRoZSBwcmludGVyLlxuICAgKi9cbiAgcHJpdmF0ZSBpc1N0YWNrQWN0aXZpdHkobXNnOiBJb01lc3NhZ2U8dW5rbm93bj4pIHtcbiAgICByZXR1cm4gbXNnLmNvZGUgJiYgW1xuICAgICAgJ0NES19UT09MS0lUX0k1NTAxJyxcbiAgICAgICdDREtfVE9PTEtJVF9JNTUwMicsXG4gICAgICAnQ0RLX1RPT0xLSVRfSTU1MDMnLFxuICAgIF0uaW5jbHVkZXMobXNnLmNvZGUpO1xuICB9XG5cbiAgLyoqXG4gICAqIERldGVjdCBzcGVjaWFsIG1lc3NhZ2VzIGVuY29kZSBpbmZvcm1hdGlvbiBhYm91dCB3aGV0aGVyIG9yIG5vdFxuICAgKiB0aGV5IHJlcXVpcmUgYXBwcm92YWxcbiAgICovXG4gIHByaXZhdGUgc2tpcEFwcHJvdmFsU3RlcChtc2c6IElvUmVxdWVzdDxhbnksIGFueT4pOiBib29sZWFuIHtcbiAgICBjb25zdCBhcHByb3ZhbFRvb2xraXRDb2RlcyA9IFsnQ0RLX1RPT0xLSVRfSTUwNjAnXTtcbiAgICBpZiAoIShtc2cuY29kZSAmJiBhcHByb3ZhbFRvb2xraXRDb2Rlcy5pbmNsdWRlcyhtc2cuY29kZSkpKSB7XG4gICAgICBmYWxzZTtcbiAgICB9XG5cbiAgICBzd2l0Y2ggKHRoaXMucmVxdWlyZURlcGxveUFwcHJvdmFsKSB7XG4gICAgICAvLyBOZXZlciByZXF1aXJlIGFwcHJvdmFsXG4gICAgICBjYXNlIFJlcXVpcmVBcHByb3ZhbC5ORVZFUjpcbiAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAvLyBBbHdheXMgcmVxdWlyZSBhcHByb3ZhbFxuICAgICAgY2FzZSBSZXF1aXJlQXBwcm92YWwuQU5ZQ0hBTkdFOlxuICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICAvLyBSZXF1aXJlIGFwcHJvdmFsIGlmIGNoYW5nZXMgaW5jbHVkZSBicm9hZGVuaW5nIHBlcm1pc3Npb25zXG4gICAgICBjYXNlIFJlcXVpcmVBcHByb3ZhbC5CUk9BREVOSU5HOlxuICAgICAgICByZXR1cm4gWydub25lJywgJ25vbi1icm9hZGVuaW5nJ10uaW5jbHVkZXMobXNnLmRhdGE/LnBlcm1pc3Npb25DaGFuZ2VUeXBlKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyB0aGUgb3V0cHV0IHN0cmVhbSwgYmFzZWQgb24gbWVzc2FnZSBhbmQgY29uZmlndXJhdGlvbi5cbiAgICovXG4gIHByaXZhdGUgc2VsZWN0U3RyZWFtKG1zZzogSW9NZXNzYWdlPGFueT4pOiBOb2RlSlMuV3JpdGVTdHJlYW0gfCB1bmRlZmluZWQge1xuICAgIGlmIChpc05vdGljZXNNZXNzYWdlKG1zZykpIHtcbiAgICAgIHJldHVybiB0YXJnZXRTdHJlYW1PYmplY3QodGhpcy5ub3RpY2VzRGVzdGluYXRpb24pO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLnNlbGVjdFN0cmVhbUZyb21MZXZlbChtc2cubGV2ZWwpO1xuICB9XG5cbiAgLyoqXG4gICAqIERldGVybWluZXMgdGhlIG91dHB1dCBzdHJlYW0sIGJhc2VkIG9uIG1lc3NhZ2UgbGV2ZWwgYW5kIGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICBwcml2YXRlIHNlbGVjdFN0cmVhbUZyb21MZXZlbChsZXZlbDogSW9NZXNzYWdlTGV2ZWwpOiBOb2RlSlMuV3JpdGVTdHJlYW0ge1xuICAgIC8vIFRoZSBzdHJlYW0gc2VsZWN0aW9uIHBvbGljeSBmb3IgdGhlIENMSSBpcyB0aGUgZm9sbG93aW5nOlxuICAgIC8vXG4gICAgLy8gICAoMSkgTWVzc2FnZXMgb2YgbGV2ZWwgYHJlc3VsdGAgYWx3YXlzIGdvIHRvIGBzdGRvdXRgXG4gICAgLy8gICAoMikgTWVzc2FnZXMgb2YgbGV2ZWwgYGVycm9yYCBhbHdheXMgZ28gdG8gYHN0ZGVycmAuXG4gICAgLy8gICAoM2EpIEFsbCByZW1haW5pbmcgbWVzc2FnZXMgZ28gdG8gYHN0ZGVycmAuXG4gICAgLy8gICAoM2IpIElmIHdlIGFyZSBpbiBDSSBtb2RlLCBhbGwgcmVtYWluaW5nIG1lc3NhZ2VzIGdvIHRvIGBzdGRvdXRgLlxuICAgIC8vXG4gICAgc3dpdGNoIChsZXZlbCkge1xuICAgICAgY2FzZSAnZXJyb3InOlxuICAgICAgICByZXR1cm4gcHJvY2Vzcy5zdGRlcnI7XG4gICAgICBjYXNlICdyZXN1bHQnOlxuICAgICAgICByZXR1cm4gcHJvY2Vzcy5zdGRvdXQ7XG4gICAgICBkZWZhdWx0OlxuICAgICAgICByZXR1cm4gdGhpcy5pc0NJID8gcHJvY2Vzcy5zdGRvdXQgOiBwcm9jZXNzLnN0ZGVycjtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogTm90aWZpZXMgdGhlIGhvc3Qgb2YgYSBtZXNzYWdlIHRoYXQgcmVxdWlyZXMgYSByZXNwb25zZS5cbiAgICpcbiAgICogSWYgdGhlIGhvc3QgZG9lcyBub3QgcmV0dXJuIGEgcmVzcG9uc2UgdGhlIHN1Z2dlc3RlZFxuICAgKiBkZWZhdWx0IHJlc3BvbnNlIGZyb20gdGhlIGlucHV0IG1lc3NhZ2Ugd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcHVibGljIGFzeW5jIHJlcXVlc3RSZXNwb25zZTxEYXRhVHlwZSwgUmVzcG9uc2VUeXBlPihtc2c6IElvUmVxdWVzdDxEYXRhVHlwZSwgUmVzcG9uc2VUeXBlPik6IFByb21pc2U8UmVzcG9uc2VUeXBlPiB7XG4gICAgLy8gSWYgdGhlIHJlcXVlc3QgY2Fubm90IGJlIHByb21wdGVkIGZvciBieSB0aGUgQ2xpSW9Ib3N0LCB3ZSBqdXN0IGFjY2VwdCB0aGUgZGVmYXVsdFxuICAgIGlmICghaXNQcm9tcHRhYmxlUmVxdWVzdChtc2cpKSB7XG4gICAgICBhd2FpdCB0aGlzLm5vdGlmeShtc2cpO1xuICAgICAgcmV0dXJuIG1zZy5kZWZhdWx0UmVzcG9uc2U7XG4gICAgfVxuXG4gICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCB0aGlzLndpdGhDb3JrZWRMb2dnaW5nKGFzeW5jICgpOiBQcm9taXNlPHN0cmluZyB8IG51bWJlciB8IHRydWU+ID0+IHtcbiAgICAgIC8vIHByZXBhcmUgcHJvbXB0IGRhdGFcbiAgICAgIC8vIEB0b2RvIHRoaXMgZm9ybWF0IGlzIG5vdCBkZWZpbmVkIGFueXdoZXJlLCBwcm9iYWJseSBzaG91bGQgYmVcbiAgICAgIGNvbnN0IGRhdGE6IHtcbiAgICAgICAgbW90aXZhdGlvbj86IHN0cmluZztcbiAgICAgICAgY29uY3VycmVuY3k/OiBudW1iZXI7XG4gICAgICAgIHJlc3BvbnNlRGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gICAgICB9ID0gbXNnLmRhdGEgPz8ge307XG5cbiAgICAgIGNvbnN0IG1vdGl2YXRpb24gPSBkYXRhLm1vdGl2YXRpb24gPz8gJ1VzZXIgaW5wdXQgaXMgbmVlZGVkJztcbiAgICAgIGNvbnN0IGNvbmN1cnJlbmN5ID0gZGF0YS5jb25jdXJyZW5jeSA/PyAwO1xuICAgICAgY29uc3QgcmVzcG9uc2VEZXNjcmlwdGlvbiA9IGRhdGEucmVzcG9uc2VEZXNjcmlwdGlvbjtcblxuICAgICAgLy8gU3BlY2lhbCBhcHByb3ZhbCBwcm9tcHRcbiAgICAgIC8vIERldGVybWluZSBpZiB0aGUgbWVzc2FnZSBuZWVkcyBhcHByb3ZhbC4gSWYgaXQgZG9lcywgY29udGludWUgKGl0IGlzIGEgYmFzaWMgY29uZmlybWF0aW9uIHByb21wdClcbiAgICAgIC8vIElmIGl0IGRvZXMgbm90LCByZXR1cm4gc3VjY2VzcyAodHJ1ZSkuIFdlIG9ubHkgY2hlY2sgbWVzc2FnZXMgd2l0aCBjb2RlcyB0aGF0IHdlIGFyZSBhd2FyZVxuICAgICAgLy8gYXJlIHJlcXVpcmVzIGFwcHJvdmFsIGNvZGVzLlxuICAgICAgaWYgKHRoaXMuc2tpcEFwcHJvdmFsU3RlcChtc2cpKSB7XG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfVxuXG4gICAgICAvLyBJbiAtLXllcyBtb2RlLCByZXNwb25kIGZvciB0aGUgdXNlciBpZiB3ZSBjYW5cbiAgICAgIGlmICh0aGlzLmF1dG9SZXNwb25kKSB7XG4gICAgICAgIC8vIHJlc3BvbmQgd2l0aCB5ZXMgdG8gYWxsIGNvbmZpcm1hdGlvbnNcbiAgICAgICAgaWYgKGlzQ29uZmlybWF0aW9uUHJvbXB0KG1zZykpIHtcbiAgICAgICAgICBhd2FpdCB0aGlzLm5vdGlmeSh7XG4gICAgICAgICAgICAuLi5tc2csXG4gICAgICAgICAgICBtZXNzYWdlOiBgJHtjaGFsay5jeWFuKG1zZy5tZXNzYWdlKX0gKGF1dG8tY29uZmlybWVkKWAsXG4gICAgICAgICAgfSk7XG4gICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgIH1cblxuICAgICAgICAvLyByZXNwb25kIHdpdGggdGhlIGRlZmF1bHQgZm9yIGFsbCBvdGhlciBtZXNzYWdlc1xuICAgICAgICBpZiAobXNnLmRlZmF1bHRSZXNwb25zZSkge1xuICAgICAgICAgIGF3YWl0IHRoaXMubm90aWZ5KHtcbiAgICAgICAgICAgIC4uLm1zZyxcbiAgICAgICAgICAgIG1lc3NhZ2U6IGAke2NoYWxrLmN5YW4obXNnLm1lc3NhZ2UpfSAoYXV0by1yZXNwb25kZWQgd2l0aCBkZWZhdWx0OiAke3V0aWwuZm9ybWF0KG1zZy5kZWZhdWx0UmVzcG9uc2UpfSlgLFxuICAgICAgICAgIH0pO1xuICAgICAgICAgIHJldHVybiBtc2cuZGVmYXVsdFJlc3BvbnNlO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIC8vIG9ubHkgdGFsayB0byB1c2VyIGlmIFNURElOIGlzIGEgdGVybWluYWwgKG90aGVyd2lzZSwgZmFpbClcbiAgICAgIGlmICghdGhpcy5pc1RUWSkge1xuICAgICAgICB0aHJvdyBuZXcgVG9vbGtpdEVycm9yKGAke21vdGl2YXRpb259LCBidXQgdGVybWluYWwgKFRUWSkgaXMgbm90IGF0dGFjaGVkIHNvIHdlIGFyZSB1bmFibGUgdG8gZ2V0IGEgY29uZmlybWF0aW9uIGZyb20gdGhlIHVzZXJgKTtcbiAgICAgIH1cblxuICAgICAgLy8gb25seSB0YWxrIHRvIHVzZXIgaWYgY29uY3VycmVuY3kgaXMgMSAob3RoZXJ3aXNlLCBmYWlsKVxuICAgICAgaWYgKGNvbmN1cnJlbmN5ID4gMSkge1xuICAgICAgICB0aHJvdyBuZXcgVG9vbGtpdEVycm9yKGAke21vdGl2YXRpb259LCBidXQgY29uY3VycmVuY3kgaXMgZ3JlYXRlciB0aGFuIDEgc28gd2UgYXJlIHVuYWJsZSB0byBnZXQgYSBjb25maXJtYXRpb24gZnJvbSB0aGUgdXNlcmApO1xuICAgICAgfVxuXG4gICAgICAvLyBCYXNpYyBjb25maXJtYXRpb24gcHJvbXB0XG4gICAgICAvLyBXZSB0cmVhdCBhbGwgcmVxdWVzdHMgd2l0aCBhIGJvb2xlYW4gcmVzcG9uc2UgYXMgY29uZmlybWF0aW9uIHByb21wdHNcbiAgICAgIGlmIChpc0NvbmZpcm1hdGlvblByb21wdChtc2cpKSB7XG4gICAgICAgIGNvbnN0IGNvbmZpcm1lZCA9IGF3YWl0IHByb21wdGx5LmNvbmZpcm0oYCR7Y2hhbGsuY3lhbihtc2cubWVzc2FnZSl9ICh5L24pYCk7XG4gICAgICAgIGlmICghY29uZmlybWVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFRvb2xraXRFcnJvcignQWJvcnRlZCBieSB1c2VyJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGNvbmZpcm1lZDtcbiAgICAgIH1cblxuICAgICAgLy8gQXNraW5nIGZvciBhIHNwZWNpZmljIHZhbHVlXG4gICAgICBjb25zdCBwcm9tcHQgPSBleHRyYWN0UHJvbXB0SW5mbyhtc2cpO1xuICAgICAgY29uc3QgZGVzYyA9IHJlc3BvbnNlRGVzY3JpcHRpb24gPz8gcHJvbXB0LmRlZmF1bHQ7XG4gICAgICBjb25zdCBhbnN3ZXIgPSBhd2FpdCBwcm9tcHRseS5wcm9tcHQoYCR7Y2hhbGsuY3lhbihtc2cubWVzc2FnZSl9JHtkZXNjID8gYCAoJHtkZXNjfSlgIDogJyd9YCwge1xuICAgICAgICBkZWZhdWx0OiBwcm9tcHQuZGVmYXVsdCxcbiAgICAgICAgdHJpbTogdHJ1ZSxcbiAgICAgIH0pO1xuICAgICAgcmV0dXJuIHByb21wdC5jb252ZXJ0QW5zd2VyKGFuc3dlcik7XG4gICAgfSk7XG5cbiAgICAvLyBXZSBuZWVkIHRvIGNhc3QgdGhpcyBiZWNhdXNlIGl0IGlzIGltcG9zc2libGUgdG8gbmFycm93IHRoZSBnZW5lcmljIHR5cGVcbiAgICAvLyBpc1Byb21wdGFibGVSZXF1ZXN0IGVuc3VyZXMgdGhhdCB0aGUgcmVzcG9uc2UgdHlwZSBpcyBvbmUgd2UgY2FuIHByb21wdCBmb3JcbiAgICAvLyB0aGUgcmVtYWluaW5nIGNvZGUgZW5zdXJlIHdlIGFyZSBpbmRlZWQgcmV0dXJuaW5nIHRoZSBjb3JyZWN0IHR5cGVcbiAgICByZXR1cm4gcmVzcG9uc2UgYXMgUmVzcG9uc2VUeXBlO1xuICB9XG5cbiAgLyoqXG4gICAqIEZvcm1hdHMgYSBtZXNzYWdlIGZvciBjb25zb2xlIG91dHB1dCB3aXRoIG9wdGlvbmFsIGNvbG9yIHN1cHBvcnRcbiAgICovXG4gIHByaXZhdGUgZm9ybWF0TWVzc2FnZShtc2c6IElvTWVzc2FnZTx1bmtub3duPik6IHN0cmluZyB7XG4gICAgLy8gYXBwbHkgcHJvdmlkZWQgc3R5bGUgb3IgYSBkZWZhdWx0IHN0eWxlIGlmIHdlJ3JlIGluIFRUWSBtb2RlXG4gICAgbGV0IG1lc3NhZ2VfdGV4dCA9IHRoaXMuaXNUVFlcbiAgICAgID8gc3R5bGVNYXBbbXNnLmxldmVsXShtc2cubWVzc2FnZSlcbiAgICAgIDogbXNnLm1lc3NhZ2U7XG5cbiAgICAvLyBwcmVwZW5kIHRpbWVzdGFtcCBpZiBJb01lc3NhZ2VMZXZlbCBpcyBERUJVRyBvciBUUkFDRS4gUG9zdHBlbmQgYSBuZXdsaW5lLlxuICAgIHJldHVybiAoKG1zZy5sZXZlbCA9PT0gJ2RlYnVnJyB8fCBtc2cubGV2ZWwgPT09ICd0cmFjZScpXG4gICAgICA/IGBbJHt0aGlzLmZvcm1hdFRpbWUobXNnLnRpbWUpfV0gJHttZXNzYWdlX3RleHR9YFxuICAgICAgOiBtZXNzYWdlX3RleHQpICsgJ1xcbic7XG4gIH1cblxuICAvKipcbiAgICogRm9ybWF0cyBkYXRlIHRvIEhIOk1NOlNTXG4gICAqL1xuICBwcml2YXRlIGZvcm1hdFRpbWUoZDogRGF0ZSk6IHN0cmluZyB7XG4gICAgY29uc3QgcGFkID0gKG46IG51bWJlcik6IHN0cmluZyA9PiBuLnRvU3RyaW5nKCkucGFkU3RhcnQoMiwgJzAnKTtcbiAgICByZXR1cm4gYCR7cGFkKGQuZ2V0SG91cnMoKSl9OiR7cGFkKGQuZ2V0TWludXRlcygpKX06JHtwYWQoZC5nZXRTZWNvbmRzKCkpfWA7XG4gIH1cblxuICAvKipcbiAgICogR2V0IGFuIGluc3RhbmNlIG9mIHRoZSBBY3Rpdml0eVByaW50ZXJcbiAgICovXG4gIHByaXZhdGUgbWFrZUFjdGl2aXR5UHJpbnRlcigpIHtcbiAgICBjb25zdCBwcm9wczogQWN0aXZpdHlQcmludGVyUHJvcHMgPSB7XG4gICAgICBzdHJlYW06IHRoaXMuc2VsZWN0U3RyZWFtRnJvbUxldmVsKCdpbmZvJyksXG4gICAgfTtcblxuICAgIHN3aXRjaCAodGhpcy5zdGFja1Byb2dyZXNzKSB7XG4gICAgICBjYXNlIFN0YWNrQWN0aXZpdHlQcm9ncmVzcy5FVkVOVFM6XG4gICAgICAgIHJldHVybiBuZXcgSGlzdG9yeUFjdGl2aXR5UHJpbnRlcihwcm9wcyk7XG4gICAgICBjYXNlIFN0YWNrQWN0aXZpdHlQcm9ncmVzcy5CQVI6XG4gICAgICAgIHJldHVybiBuZXcgQ3VycmVudEFjdGl2aXR5UHJpbnRlcihwcm9wcyk7XG4gICAgfVxuICB9XG59XG5cbi8qKlxuICogVGhpcyBJb0hvc3QgaW1wbGVtZW50YXRpb24gY29uc2lkZXJzIGEgcmVxdWVzdCBwcm9tcHRhYmxlLCBpZjpcbiAqIC0gaXQncyBhIHllcy9ubyBjb25maXJtYXRpb25cbiAqIC0gYXNraW5nIGZvciBhIHN0cmluZyBvciBudW1iZXIgdmFsdWVcbiAqL1xuZnVuY3Rpb24gaXNQcm9tcHRhYmxlUmVxdWVzdChtc2c6IElvUmVxdWVzdDxhbnksIGFueT4pOiBtc2cgaXMgSW9SZXF1ZXN0PGFueSwgc3RyaW5nIHwgbnVtYmVyIHwgYm9vbGVhbj4ge1xuICByZXR1cm4gaXNDb25maXJtYXRpb25Qcm9tcHQobXNnKVxuICAgIHx8IHR5cGVvZiBtc2cuZGVmYXVsdFJlc3BvbnNlID09PSAnc3RyaW5nJ1xuICAgIHx8IHR5cGVvZiBtc2cuZGVmYXVsdFJlc3BvbnNlID09PSAnbnVtYmVyJztcbn1cblxuLyoqXG4gKiBDaGVjayBpZiB0aGUgcmVxdWVzdCBpcyBhIGNvbmZpcm1hdGlvbiBwcm9tcHRcbiAqIFdlIHRyZWF0IGFsbCByZXF1ZXN0cyB3aXRoIGEgYm9vbGVhbiByZXNwb25zZSBhcyBjb25maXJtYXRpb24gcHJvbXB0c1xuICovXG5mdW5jdGlvbiBpc0NvbmZpcm1hdGlvblByb21wdChtc2c6IElvUmVxdWVzdDxhbnksIGFueT4pOiBtc2cgaXMgSW9SZXF1ZXN0PGFueSwgYm9vbGVhbj4ge1xuICByZXR1cm4gdHlwZW9mIG1zZy5kZWZhdWx0UmVzcG9uc2UgPT09ICdib29sZWFuJztcbn1cblxuLyoqXG4gKiBIZWxwZXIgdG8gZXh0cmFjdCBpbmZvcm1hdGlvbiBmb3IgcHJvbXB0bHkgZnJvbSB0aGUgcmVxdWVzdFxuICovXG5mdW5jdGlvbiBleHRyYWN0UHJvbXB0SW5mbyhtc2c6IElvUmVxdWVzdDxhbnksIGFueT4pOiB7XG4gIGRlZmF1bHQ6IHN0cmluZztcbiAgZGVmYXVsdERlc2M6IHN0cmluZztcbiAgY29udmVydEFuc3dlcjogKGlucHV0OiBzdHJpbmcpID0+IHN0cmluZyB8IG51bWJlcjtcbn0ge1xuICBjb25zdCBpc051bWJlciA9ICh0eXBlb2YgbXNnLmRlZmF1bHRSZXNwb25zZSA9PT0gJ251bWJlcicpO1xuICBjb25zdCBkZWZhdWx0UmVzcG9uc2UgPSB1dGlsLmZvcm1hdChtc2cuZGVmYXVsdFJlc3BvbnNlKTtcbiAgcmV0dXJuIHtcbiAgICBkZWZhdWx0OiBkZWZhdWx0UmVzcG9uc2UsXG4gICAgZGVmYXVsdERlc2M6ICdkZWZhdWx0RGVzY3JpcHRpb24nIGluIG1zZyAmJiBtc2cuZGVmYXVsdERlc2NyaXB0aW9uID8gdXRpbC5mb3JtYXQobXNnLmRlZmF1bHREZXNjcmlwdGlvbikgOiBkZWZhdWx0UmVzcG9uc2UsXG4gICAgY29udmVydEFuc3dlcjogaXNOdW1iZXIgPyAodikgPT4gTnVtYmVyKHYpIDogKHYpID0+IFN0cmluZyh2KSxcbiAgfTtcbn1cblxuY29uc3Qgc3R5bGVNYXA6IFJlY29yZDxJb01lc3NhZ2VMZXZlbCwgKHN0cjogc3RyaW5nKSA9PiBzdHJpbmc+ID0ge1xuICBlcnJvcjogY2hhbGsucmVkLFxuICB3YXJuOiBjaGFsay55ZWxsb3csXG4gIHJlc3VsdDogY2hhbGsucmVzZXQsXG4gIGluZm86IGNoYWxrLnJlc2V0LFxuICBkZWJ1ZzogY2hhbGsuZ3JheSxcbiAgdHJhY2U6IGNoYWxrLmdyYXksXG59O1xuXG5mdW5jdGlvbiB0YXJnZXRTdHJlYW1PYmplY3QoeDogVGFyZ2V0U3RyZWFtKTogTm9kZUpTLldyaXRlU3RyZWFtIHwgdW5kZWZpbmVkIHtcbiAgc3dpdGNoICh4KSB7XG4gICAgY2FzZSAnc3RkZXJyJzpcbiAgICAgIHJldHVybiBwcm9jZXNzLnN0ZGVycjtcbiAgICBjYXNlICdzdGRvdXQnOlxuICAgICAgcmV0dXJuIHByb2Nlc3Muc3Rkb3V0O1xuICAgIGNhc2UgJ2Ryb3AnOlxuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxufVxuXG5mdW5jdGlvbiBpc05vdGljZXNNZXNzYWdlKG1zZzogSW9NZXNzYWdlPHVua25vd24+KTogbXNnIGlzIElvTWVzc2FnZTx2b2lkPiB7XG4gIHJldHVybiBJTy5DREtfVE9PTEtJVF9JMDEwMC5pcyhtc2cpIHx8IElPLkNES19UT09MS0lUX1cwMTAxLmlzKG1zZykgfHwgSU8uQ0RLX1RPT0xLSVRfRTAxMDEuaXMobXNnKSB8fCBJTy5DREtfVE9PTEtJVF9JMDEwMS5pcyhtc2cpO1xufVxuXG5mdW5jdGlvbiBpc1RlbGVtZXRyeU1lc3NhZ2UobXNnOiBJb01lc3NhZ2U8dW5rbm93bj4pOiBtc2cgaXMgSW9NZXNzYWdlPEV2ZW50UmVzdWx0PiB7XG4gIHJldHVybiBDTElfVEVMRU1FVFJZX0NPREVTLnNvbWUoKGMpID0+IGMuaXMobXNnKSk7XG59XG5cbmZ1bmN0aW9uIGdldEV2ZW50VHlwZShtc2c6IElvTWVzc2FnZTx1bmtub3duPik6IEV2ZW50VHlwZSB7XG4gIHN3aXRjaCAobXNnLmNvZGUpIHtcbiAgICBjYXNlIENMSV9QUklWQVRFX0lPLkNES19DTElfSTEwMDEuY29kZTpcbiAgICAgIHJldHVybiAnU1lOVEgnO1xuICAgIGNhc2UgQ0xJX1BSSVZBVEVfSU8uQ0RLX0NMSV9JMjAwMS5jb2RlOlxuICAgICAgcmV0dXJuICdJTlZPS0UnO1xuICAgIGNhc2UgQ0xJX1BSSVZBVEVfSU8uQ0RLX0NMSV9JMzAwMS5jb2RlOlxuICAgICAgcmV0dXJuICdERVBMT1knO1xuICAgIGRlZmF1bHQ6XG4gICAgICB0aHJvdyBuZXcgVG9vbGtpdEVycm9yKGBVbnJlY29nbml6ZWQgVGVsZW1ldHJ5IE1lc3NhZ2UgQ29kZTogJHttc2cuY29kZX1gKTtcbiAgfVxufVxuIl19