"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitTemplate = void 0;
exports.cliInit = cliInit;
exports.expandPlaceholders = expandPlaceholders;
exports.availableInitTemplates = availableInitTemplates;
exports.availableInitLanguages = availableInitLanguages;
exports.printAvailableTemplates = printAvailableTemplates;
exports.currentlyRecommendedAwsCdkLibFlags = currentlyRecommendedAwsCdkLibFlags;
const childProcess = require("child_process");
const path = require("path");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const fs = require("fs-extra");
const init_hooks_1 = require("./init-hooks");
const root_dir_1 = require("../../cli/root-dir");
const version_1 = require("../../cli/version");
const util_1 = require("../../util");
const language_1 = require("../language");
/* eslint-disable @typescript-eslint/no-var-requires */ // Packages don't have @types module
// eslint-disable-next-line @typescript-eslint/no-require-imports
const camelCase = require('camelcase');
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const SUPPORTED_LANGUAGE_NAMES = language_1.SUPPORTED_LANGUAGES.map((l) => l.name);
/**
 * Initialize a CDK package in the current directory
 */
async function cliInit(options) {
    await ensureValidCliInitOptions(options, options.ioHelper);
    const ioHelper = options.ioHelper;
    const canUseNetwork = options.canUseNetwork ?? true;
    const generateOnly = options.generateOnly ?? false;
    const workDir = options.workDir ?? process.cwd();
    // Show available templates only if no fromPath, type, or language provided
    if (!options.fromPath && !options.type && !options.language) {
        await printAvailableTemplates(ioHelper);
        return;
    }
    // Step 1: Load template
    let template;
    if (options.fromPath) {
        template = await loadLocalTemplate(options.fromPath, options.templatePath);
    }
    else {
        template = await loadBuiltinTemplate(ioHelper, options.type, options.language);
    }
    // Step 2: Resolve language
    const language = await resolveLanguage(ioHelper, template, options.language, options.type);
    // Step 3: Initialize project following standard process
    await initializeProject(ioHelper, template, language, canUseNetwork, generateOnly, workDir, options.stackName, options.migrate, options.libVersion, options.packageManager);
}
/**
 * Validate CLI init options and handle invalid or incompatible option combinations
 */
async function ensureValidCliInitOptions(options, ioHelper) {
    if (options.packageManager && !['javascript', 'typescript'].includes(options.language ?? '')) {
        await ioHelper.defaults.warn(`--package-manager option is only applicable for JavaScript and TypeScript projects. Ignoring the provided value: ${options.packageManager}`);
    }
}
/**
 * Load a local custom template from file system path
 * @param fromPath - Path to the local template directory or multi-template repository
 * @param templatePath - Optional path to a specific template within a multi-template repository
 * @returns Promise resolving to the loaded InitTemplate
 */
async function loadLocalTemplate(fromPath, templatePath) {
    try {
        let actualTemplatePath = fromPath;
        // If templatePath is provided, it's a multi-template repository
        if (templatePath) {
            actualTemplatePath = path.join(fromPath, templatePath);
            if (!await fs.pathExists(actualTemplatePath)) {
                throw new toolkit_lib_1.ToolkitError(`Template path does not exist: ${actualTemplatePath}`);
            }
        }
        const template = await InitTemplate.fromPath(actualTemplatePath);
        if (template.languages.length === 0) {
            // Check if this might be a multi-template repository
            if (!templatePath) {
                const availableTemplates = await findPotentialTemplates(fromPath);
                if (availableTemplates.length > 0) {
                    throw new toolkit_lib_1.ToolkitError('Use --template-path to specify which template to use.');
                }
            }
            throw new toolkit_lib_1.ToolkitError('Custom template must contain at least one language directory');
        }
        return template;
    }
    catch (error) {
        const displayPath = templatePath ? `${fromPath}/${templatePath}` : fromPath;
        throw new toolkit_lib_1.ToolkitError(`Failed to load template from path: ${displayPath}. ${error.message}`);
    }
}
/**
 * Load a built-in template by name
 */
async function loadBuiltinTemplate(ioHelper, type, language) {
    const templateType = type || 'default'; // "default" is the default type (and maps to "app")
    const template = (await availableInitTemplates()).find((t) => t.hasName(templateType));
    if (!template) {
        await printAvailableTemplates(ioHelper, language);
        throw new toolkit_lib_1.ToolkitError(`Unknown init template: ${templateType}`);
    }
    return template;
}
/**
 * Resolve the programming language for the template
 * @param ioHelper - IO helper for user interaction
 * @param template - The template to resolve language for
 * @param requestedLanguage - User-requested language (optional)
 * @param type - The template type name for messages
 * @default undefined
 * @returns Promise resolving to the selected language
 */
async function resolveLanguage(ioHelper, template, requestedLanguage, type) {
    return (async () => {
        if (requestedLanguage) {
            return requestedLanguage;
        }
        if (template.languages.length === 1) {
            const templateLanguage = template.languages[0];
            // Only show auto-detection message for built-in templates
            if (template.templateType !== TemplateType.CUSTOM) {
                await ioHelper.defaults.warn(`No --language was provided, but '${type || template.name}' supports only '${templateLanguage}', so defaulting to --language=${templateLanguage}`);
            }
            return templateLanguage;
        }
        await ioHelper.defaults.info(`Available languages for ${chalk.green(type || template.name)}: ${template.languages.map((l) => chalk.blue(l)).join(', ')}`);
        throw new toolkit_lib_1.ToolkitError('No language was selected');
    })();
}
/**
 * Find potential template directories in a multi-template repository
 * @param repositoryPath - Path to the repository root
 * @returns Promise resolving to array of potential template directory names
 */
async function findPotentialTemplates(repositoryPath) {
    const entries = await fs.readdir(repositoryPath, { withFileTypes: true });
    const templateValidationPromises = entries
        .filter(entry => entry.isDirectory() && !entry.name.startsWith('.'))
        .map(async (entry) => {
        try {
            const templatePath = path.join(repositoryPath, entry.name);
            const { languages } = await getLanguageDirectories(templatePath);
            return languages.length > 0 ? entry.name : null;
        }
        catch (error) {
            // If we can't read a specific template directory, skip it but don't fail the entire operation
            return null;
        }
    });
    /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Limited to directory entries
    const validationResults = await Promise.all(templateValidationPromises);
    return validationResults.filter((templateName) => templateName !== null);
}
/**
 * Get valid CDK language directories from a template path
 * @param templatePath - Path to the template directory
 * @returns Promise resolving to array of supported language names
 */
/**
 * Get valid CDK language directories from a template path
 * @param templatePath - Path to the template directory
 * @returns Promise resolving to array of supported language names and directory entries
 * @throws ToolkitError if directory cannot be read or validated
 */
async function getLanguageDirectories(templatePath) {
    try {
        const entries = await fs.readdir(templatePath, { withFileTypes: true });
        const languageValidationPromises = entries
            .filter(directoryEntry => directoryEntry.isDirectory() && SUPPORTED_LANGUAGE_NAMES.includes(directoryEntry.name))
            .map(async (directoryEntry) => {
            const languageDirectoryPath = path.join(templatePath, directoryEntry.name);
            try {
                const hasValidLanguageFiles = await hasLanguageFiles(languageDirectoryPath, (0, language_1.getLanguageExtensions)(directoryEntry.name));
                return hasValidLanguageFiles ? directoryEntry.name : null;
            }
            catch (error) {
                throw new toolkit_lib_1.ToolkitError(`Cannot read language directory '${directoryEntry.name}': ${error.message}`);
            }
        });
        /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Limited to supported CDK languages (7 max)
        const validationResults = await Promise.all(languageValidationPromises);
        return {
            languages: validationResults.filter((languageName) => languageName !== null),
            entries,
        };
    }
    catch (error) {
        throw new toolkit_lib_1.ToolkitError(`Cannot read template directory '${templatePath}': ${error.message}`);
    }
}
/**
 * Iteratively check if a directory contains files with the specified extensions
 * @param directoryPath - Path to search for language files
 * @param extensions - Array of file extensions to look for
 * @returns Promise resolving to true if language files are found
 */
async function hasLanguageFiles(directoryPath, extensions) {
    const dirsToCheck = [directoryPath];
    while (dirsToCheck.length > 0) {
        const currentDir = dirsToCheck.pop();
        try {
            const entries = await fs.readdir(currentDir, { withFileTypes: true });
            for (const entry of entries) {
                if (entry.isFile() && extensions.some(ext => entry.name.endsWith(ext))) {
                    return true;
                }
                else if (entry.isDirectory()) {
                    dirsToCheck.push(path.join(currentDir, entry.name));
                }
            }
        }
        catch (error) {
            throw error;
        }
    }
    return false;
}
/**
 * Returns the name of the Python executable for this OS
 * @returns The Python executable name for the current platform
 */
function pythonExecutable() {
    let python = 'python3';
    if (process.platform === 'win32') {
        python = 'python';
    }
    return python;
}
const INFO_DOT_JSON = 'info.json';
var TemplateType;
(function (TemplateType) {
    TemplateType["BUILT_IN"] = "builtin";
    TemplateType["CUSTOM"] = "custom";
})(TemplateType || (TemplateType = {}));
class InitTemplate {
    static async fromName(templatesDir, name) {
        const basePath = path.join(templatesDir, name);
        const languages = await listDirectory(basePath);
        const initInfo = await fs.readJson(path.join(basePath, INFO_DOT_JSON));
        return new InitTemplate(basePath, name, languages, initInfo, TemplateType.BUILT_IN);
    }
    static async fromPath(templatePath) {
        const basePath = path.resolve(templatePath);
        if (!await fs.pathExists(basePath)) {
            throw new toolkit_lib_1.ToolkitError(`Template path does not exist: ${basePath}`);
        }
        let templateSourcePath = basePath;
        let { languages, entries } = await getLanguageDirectories(basePath);
        if (languages.length === 0) {
            const languageDirs = entries.filter(entry => entry.isDirectory() &&
                SUPPORTED_LANGUAGE_NAMES.includes(entry.name));
            if (languageDirs.length === 1) {
                // Validate that the language directory contains appropriate files
                const langDir = languageDirs[0].name;
                templateSourcePath = path.join(basePath, langDir);
                const hasValidFiles = await hasLanguageFiles(templateSourcePath, (0, language_1.getLanguageExtensions)(langDir));
                if (!hasValidFiles) {
                    // If we found a language directory but it doesn't contain valid files, we should inform the user
                    throw new toolkit_lib_1.ToolkitError(`Found '${langDir}' directory but it doesn't contain the expected language files. Ensure the template contains ${langDir} source files.`);
                }
            }
        }
        const name = path.basename(basePath);
        return new InitTemplate(templateSourcePath, name, languages, null, TemplateType.CUSTOM);
    }
    constructor(basePath, name, languages, initInfo, templateType) {
        this.basePath = basePath;
        this.name = name;
        this.languages = languages;
        this.aliases = new Set();
        this.templateType = templateType;
        // Only built-in templates have descriptions and aliases from info.json
        if (templateType === TemplateType.BUILT_IN && initInfo) {
            this.description = initInfo.description;
            for (const alias of initInfo.aliases || []) {
                this.aliases.add(alias);
            }
        }
    }
    /**
     * @param name - the name that is being checked
     * @returns ``true`` if ``name`` is the name of this template or an alias of it.
     */
    hasName(name) {
        return name === this.name || this.aliases.has(name);
    }
    /**
     * Creates a new instance of this ``InitTemplate`` for a given language to a specified folder.
     *
     * @param language - the language to instantiate this template with
     * @param targetDirectory - the directory where the template is to be instantiated into
     * @param stackName - the name of the stack to create
     * @default undefined
     * @param libVersion - the version of the CDK library to use
     * @default undefined
     */
    async install(ioHelper, language, targetDirectory, stackName, libVersion) {
        if (this.languages.indexOf(language) === -1) {
            await ioHelper.defaults.error(`The ${chalk.blue(language)} language is not supported for ${chalk.green(this.name)} ` +
                `(it supports: ${this.languages.map((l) => chalk.blue(l)).join(', ')})`);
            throw new toolkit_lib_1.ToolkitError(`Unsupported language: ${language}`);
        }
        const projectInfo = {
            name: decamelize(path.basename(path.resolve(targetDirectory))),
            stackName,
            versions: await loadInitVersions(),
        };
        if (libVersion) {
            projectInfo.versions['aws-cdk-lib'] = libVersion;
        }
        let sourceDirectory = path.join(this.basePath, language);
        // For auto-detected single language templates, use basePath directly
        if (this.templateType === TemplateType.CUSTOM && this.languages.length === 1 &&
            path.basename(this.basePath) === language) {
            sourceDirectory = this.basePath;
        }
        if (this.templateType === TemplateType.CUSTOM) {
            // For custom templates, copy files without processing placeholders
            await this.installFilesWithoutProcessing(sourceDirectory, targetDirectory);
        }
        else {
            // For built-in templates, process placeholders as usual
            await this.installFiles(sourceDirectory, targetDirectory, language, projectInfo);
            await this.applyFutureFlags(targetDirectory);
            await (0, init_hooks_1.invokeBuiltinHooks)(ioHelper, { targetDirectory, language, templateName: this.name }, {
                substitutePlaceholdersIn: async (...fileNames) => {
                    const fileProcessingPromises = fileNames.map(async (fileName) => {
                        const fullPath = path.join(targetDirectory, fileName);
                        const template = await fs.readFile(fullPath, { encoding: 'utf-8' });
                        await fs.writeFile(fullPath, expandPlaceholders(template, language, projectInfo));
                    });
                    /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Processing a small, known set of template files
                    await Promise.all(fileProcessingPromises);
                },
                placeholder: (ph) => expandPlaceholders(`%${ph}%`, language, projectInfo),
            });
        }
    }
    async installFiles(sourceDirectory, targetDirectory, language, project) {
        for (const file of await fs.readdir(sourceDirectory)) {
            const fromFile = path.join(sourceDirectory, file);
            const toFile = path.join(targetDirectory, expandPlaceholders(file, language, project));
            if ((await fs.stat(fromFile)).isDirectory()) {
                await fs.mkdir(toFile);
                await this.installFiles(fromFile, toFile, language, project);
                continue;
            }
            else if (file.match(/^.*\.template\.[^.]+$/)) {
                await this.installProcessed(fromFile, toFile.replace(/\.template(\.[^.]+)$/, '$1'), language, project);
                continue;
            }
            else if (file.match(/^.*\.hook\.(d.)?[^.]+$/)) {
                // Ignore
                continue;
            }
            else {
                await fs.copy(fromFile, toFile);
            }
        }
    }
    async installProcessed(templatePath, toFile, language, project) {
        const template = await fs.readFile(templatePath, { encoding: 'utf-8' });
        await fs.writeFile(toFile, expandPlaceholders(template, language, project));
    }
    /**
     * Copy template files without processing placeholders (for custom templates)
     */
    async installFilesWithoutProcessing(sourceDirectory, targetDirectory) {
        await fs.copy(sourceDirectory, targetDirectory, {
            filter: (src) => {
                const filename = path.basename(src);
                return !filename.match(/^.*\.hook\.(d.)?[^.]+$/);
            },
        });
    }
    /**
     * Adds context variables to `cdk.json` in the generated project directory to
     * enable future behavior for new projects.
     */
    async applyFutureFlags(projectDir) {
        const cdkJson = path.join(projectDir, 'cdk.json');
        if (!(await fs.pathExists(cdkJson))) {
            return;
        }
        const config = await fs.readJson(cdkJson);
        config.context = {
            ...config.context,
            ...await currentlyRecommendedAwsCdkLibFlags(),
        };
        await fs.writeJson(cdkJson, config, { spaces: 2 });
    }
    async addMigrateContext(projectDir) {
        const cdkJson = path.join(projectDir, 'cdk.json');
        if (!(await fs.pathExists(cdkJson))) {
            return;
        }
        const config = await fs.readJson(cdkJson);
        config.context = {
            ...config.context,
            'cdk-migrate': true,
        };
        await fs.writeJson(cdkJson, config, { spaces: 2 });
    }
}
exports.InitTemplate = InitTemplate;
function expandPlaceholders(template, language, project) {
    const cdkVersion = project.versions['aws-cdk-lib'];
    const cdkCliVersion = project.versions['aws-cdk'];
    let constructsVersion = project.versions.constructs;
    switch (language) {
        case 'java':
        case 'csharp':
        case 'fsharp':
            constructsVersion = (0, util_1.rangeFromSemver)(constructsVersion, 'bracket');
            break;
        case 'python':
            constructsVersion = (0, util_1.rangeFromSemver)(constructsVersion, 'pep');
            break;
    }
    return template
        .replace(/%name%/g, project.name)
        .replace(/%stackname%/, project.stackName ?? '%name.PascalCased%Stack')
        .replace(/%PascalNameSpace%/, project.stackName ? camelCase(project.stackName + 'Stack', { pascalCase: true }) : '%name.PascalCased%')
        .replace(/%PascalStackProps%/, project.stackName ? camelCase(project.stackName, { pascalCase: true }) + 'StackProps' : 'StackProps')
        .replace(/%name\.camelCased%/g, camelCase(project.name))
        .replace(/%name\.PascalCased%/g, camelCase(project.name, { pascalCase: true }))
        .replace(/%cdk-version%/g, cdkVersion)
        .replace(/%cdk-cli-version%/g, cdkCliVersion)
        .replace(/%constructs-version%/g, constructsVersion)
        .replace(/%cdk-home%/g, (0, util_1.cdkHomeDir)())
        .replace(/%name\.PythonModule%/g, project.name.replace(/-/g, '_'))
        .replace(/%python-executable%/g, pythonExecutable())
        .replace(/%name\.StackName%/g, project.name.replace(/[^A-Za-z0-9-]/g, '-'));
}
async function availableInitTemplates() {
    try {
        const templatesDir = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates');
        const templateNames = await listDirectory(templatesDir);
        const templatePromises = templateNames.map(templateName => InitTemplate.fromName(templatesDir, templateName));
        /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Built-in templates are limited in number
        return await Promise.all(templatePromises);
    }
    catch (error) {
        // Return empty array if templates directory doesn't exist or can't be read
        // This allows the CLI to gracefully handle missing built-in templates
        if (error.code === 'ENOENT' || error.code === 'EACCES') {
            return [];
        }
        throw error;
    }
}
async function availableInitLanguages() {
    const templates = await availableInitTemplates();
    const result = new Set();
    for (const template of templates) {
        for (const language of template.languages) {
            const alias = (0, language_1.getLanguageAlias)(language);
            result.add(language);
            alias && result.add(alias);
        }
    }
    return [...result];
}
/**
 * @param dirPath - is the directory to be listed.
 * @returns the list of file or directory names contained in ``dirPath``, excluding any dot-file, and sorted.
 */
async function listDirectory(dirPath) {
    return ((await fs.readdir(dirPath))
        .filter((p) => !p.startsWith('.'))
        .filter((p) => !(p === 'LICENSE'))
        // if, for some reason, the temp folder for the hook doesn't get deleted we don't want to display it in this list
        .filter((p) => !(p === INFO_DOT_JSON))
        .sort());
}
/**
 * Print available templates to the user
 * @param ioHelper - IO helper for user interaction
 * @param language - Programming language filter
 * @default undefined
 */
async function printAvailableTemplates(ioHelper, language) {
    await ioHelper.defaults.info('Available templates:');
    for (const template of await availableInitTemplates()) {
        if (language && template.languages.indexOf(language) === -1) {
            continue;
        }
        await ioHelper.defaults.info(`* ${chalk.green(template.name)}: ${template.description}`);
        const languageArg = language
            ? chalk.bold(language)
            : template.languages.length > 1
                ? `[${template.languages.map((t) => chalk.bold(t)).join('|')}]`
                : chalk.bold(template.languages[0]);
        await ioHelper.defaults.info(`   └─ ${chalk.blue(`cdk init ${chalk.bold(template.name)} --language=${languageArg}`)}`);
    }
}
async function initializeProject(ioHelper, template, language, canUseNetwork, generateOnly, workDir, stackName, migrate, cdkVersion, packageManager) {
    // Step 1: Ensure target directory is empty
    await assertIsEmptyDirectory(workDir);
    // Step 2: Copy template files
    await ioHelper.defaults.info(`Applying project template ${chalk.green(template.name)} for ${chalk.blue(language)}`);
    await template.install(ioHelper, language, workDir, stackName, cdkVersion);
    if (migrate) {
        await template.addMigrateContext(workDir);
    }
    if (await fs.pathExists(`${workDir}/README.md`)) {
        const readme = await fs.readFile(`${workDir}/README.md`, { encoding: 'utf-8' });
        await ioHelper.defaults.info(chalk.green(readme));
    }
    if (!generateOnly) {
        // Step 3: Initialize Git repository and create initial commit
        await initializeGitRepository(ioHelper, workDir);
        // Step 4: Post-install steps
        await postInstall(ioHelper, language, canUseNetwork, workDir, packageManager);
    }
    await ioHelper.defaults.info('✅ All done!');
}
/**
 * Validate that a directory exists and is empty (ignoring hidden files)
 * @param workDir - Directory path to validate
 * @throws ToolkitError if directory doesn't exist or is not empty
 */
async function assertIsEmptyDirectory(workDir) {
    try {
        const stats = await fs.stat(workDir);
        if (!stats.isDirectory()) {
            throw new toolkit_lib_1.ToolkitError(`Path exists but is not a directory: ${workDir}`);
        }
        const files = await fs.readdir(workDir);
        const visibleFiles = files.filter(f => !f.startsWith('.'));
        if (visibleFiles.length > 0) {
            throw new toolkit_lib_1.ToolkitError('`cdk init` cannot be run in a non-empty directory!\n' +
                `Found ${visibleFiles.length} visible files in ${workDir}:\n` +
                visibleFiles.map(f => `  - ${f}`).join('\n'));
        }
    }
    catch (e) {
        if (e.code === 'ENOENT') {
            throw new toolkit_lib_1.ToolkitError(`Directory does not exist: ${workDir}\n` +
                'Please create the directory first using: mkdir -p ' + workDir);
        }
        throw new toolkit_lib_1.ToolkitError(`Failed to validate directory ${workDir}: ${e.message}`);
    }
}
async function initializeGitRepository(ioHelper, workDir) {
    if (await isInGitRepository(workDir)) {
        return;
    }
    await ioHelper.defaults.info('Initializing a new git repository...');
    try {
        await execute(ioHelper, 'git', ['init'], { cwd: workDir });
        await execute(ioHelper, 'git', ['add', '.'], { cwd: workDir });
        await execute(ioHelper, 'git', ['commit', '--message="Initial commit"', '--no-gpg-sign'], { cwd: workDir });
    }
    catch {
        await ioHelper.defaults.warn('Unable to initialize git repository for your project.');
    }
}
async function postInstall(ioHelper, language, canUseNetwork, workDir, packageManager) {
    switch (language) {
        case 'javascript':
            return postInstallJavascript(ioHelper, canUseNetwork, workDir, packageManager);
        case 'typescript':
            return postInstallTypescript(ioHelper, canUseNetwork, workDir, packageManager);
        case 'java':
            return postInstallJava(ioHelper, canUseNetwork, workDir);
        case 'python':
            return postInstallPython(ioHelper, workDir);
        case 'go':
            return postInstallGo(ioHelper, canUseNetwork, workDir);
        case 'csharp':
            return postInstallCSharp(ioHelper, canUseNetwork, workDir);
        case 'fsharp':
            return postInstallFSharp(ioHelper, canUseNetwork, workDir);
    }
}
async function postInstallJavascript(ioHelper, canUseNetwork, cwd, packageManager) {
    return postInstallTypescript(ioHelper, canUseNetwork, cwd, packageManager);
}
async function postInstallTypescript(ioHelper, canUseNetwork, cwd, packageManager) {
    const command = packageManager ?? 'npm';
    if (!canUseNetwork) {
        await ioHelper.defaults.warn(`Please run '${command} install'!`);
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green(`${command} install`)}...`);
    try {
        await execute(ioHelper, command, ['install'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn(`${command} install failed: ` + (0, util_1.formatErrorMessage)(e));
    }
}
async function postInstallJava(ioHelper, canUseNetwork, cwd) {
    // Check if this is a Gradle or Maven project
    const hasGradleBuild = await fs.pathExists(path.join(cwd, 'build.gradle'));
    const hasMavenPom = await fs.pathExists(path.join(cwd, 'pom.xml'));
    if (hasGradleBuild) {
        // Gradle project
        const gradleWarning = "Please run './gradlew build'!";
        if (!canUseNetwork) {
            await ioHelper.defaults.warn(gradleWarning);
            return;
        }
        await ioHelper.defaults.info("Executing './gradlew build'");
        try {
            await execute(ioHelper, './gradlew', ['build'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to build Gradle project');
            await ioHelper.defaults.warn(gradleWarning);
        }
    }
    else if (hasMavenPom) {
        // Maven project
        const mvnPackageWarning = "Please run 'mvn package'!";
        if (!canUseNetwork) {
            await ioHelper.defaults.warn(mvnPackageWarning);
            return;
        }
        await ioHelper.defaults.info("Executing 'mvn package'");
        try {
            await execute(ioHelper, 'mvn', ['package'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to package compiled code as JAR');
            await ioHelper.defaults.warn(mvnPackageWarning);
        }
    }
    else {
        // No recognized build file
        await ioHelper.defaults.warn('No build.gradle or pom.xml found. Please set up your build system manually.');
    }
}
async function postInstallPython(ioHelper, cwd) {
    const python = pythonExecutable();
    // Check if requirements.txt exists
    const hasRequirements = await fs.pathExists(path.join(cwd, 'requirements.txt'));
    if (hasRequirements) {
        await ioHelper.defaults.info(`Executing ${chalk.green('Creating virtualenv...')}`);
        try {
            await execute(ioHelper, python, ['-m', 'venv', '.venv'], { cwd });
            await ioHelper.defaults.info(`Executing ${chalk.green('Installing dependencies...')}`);
            // Install dependencies in the virtual environment
            const pipPath = process.platform === 'win32' ? '.venv\\Scripts\\pip' : '.venv/bin/pip';
            await execute(ioHelper, pipPath, ['install', '-r', 'requirements.txt'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to create virtualenv or install dependencies automatically');
            await ioHelper.defaults.warn(`Please run '${python} -m venv .venv && .venv/bin/pip install -r requirements.txt'!`);
        }
    }
    else {
        await ioHelper.defaults.warn('No requirements.txt found. Please set up your Python environment manually.');
    }
}
async function postInstallGo(ioHelper, canUseNetwork, cwd) {
    if (!canUseNetwork) {
        await ioHelper.defaults.warn('Please run \'go mod tidy\'!');
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green('go mod tidy')}...`);
    try {
        await execute(ioHelper, 'go', ['mod', 'tidy'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn('\'go mod tidy\' failed: ' + (0, util_1.formatErrorMessage)(e));
    }
}
async function postInstallCSharp(ioHelper, canUseNetwork, cwd) {
    const dotnetWarning = "Please run 'dotnet restore && dotnet build'!";
    if (!canUseNetwork) {
        await ioHelper.defaults.warn(dotnetWarning);
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green('dotnet restore')}...`);
    try {
        await execute(ioHelper, 'dotnet', ['restore'], { cwd });
        await ioHelper.defaults.info(`Executing ${chalk.green('dotnet build')}...`);
        await execute(ioHelper, 'dotnet', ['build'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn('Unable to restore/build .NET project: ' + (0, util_1.formatErrorMessage)(e));
        await ioHelper.defaults.warn(dotnetWarning);
    }
}
async function postInstallFSharp(ioHelper, canUseNetwork, cwd) {
    // F# uses the same build system as C#
    return postInstallCSharp(ioHelper, canUseNetwork, cwd);
}
/**
 * @param dir - a directory to be checked
 * @returns true if ``dir`` is within a git repository.
 */
async function isInGitRepository(dir) {
    while (true) {
        if (await fs.pathExists(path.join(dir, '.git'))) {
            return true;
        }
        if (isRoot(dir)) {
            return false;
        }
        dir = path.dirname(dir);
    }
}
/**
 * @param dir - a directory to be checked.
 * @returns true if ``dir`` is the root of a filesystem.
 */
function isRoot(dir) {
    return path.dirname(dir) === dir;
}
/**
 * Executes `command`. STDERR is emitted in real-time.
 *
 * If command exits with non-zero exit code, an exception is thrown and includes
 * the contents of STDOUT.
 *
 * @returns STDOUT (if successful).
 */
async function execute(ioHelper, cmd, args, { cwd }) {
    const child = childProcess.spawn(cmd, args, {
        cwd,
        shell: true,
        stdio: ['ignore', 'pipe', 'inherit'],
    });
    let stdout = '';
    child.stdout.on('data', (chunk) => (stdout += chunk.toString()));
    return new Promise((ok, fail) => {
        child.once('error', (err) => fail(err));
        child.once('exit', (status) => {
            if (status === 0) {
                return ok(stdout);
            }
            else {
                return fail(new toolkit_lib_1.ToolkitError(`${cmd} exited with status ${status}`));
            }
        });
    }).catch(async (err) => {
        await ioHelper.defaults.error(stdout);
        throw err;
    });
}
/**
 * Return the 'aws-cdk-lib' version we will init
 *
 * This has been built into the CLI at build time.
 */
async function loadInitVersions() {
    const initVersionFile = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates', '.init-version.json');
    const contents = JSON.parse(await fs.readFile(initVersionFile, { encoding: 'utf-8' }));
    const ret = {
        'aws-cdk-lib': contents['aws-cdk-lib'],
        'constructs': contents.constructs,
        'aws-cdk': (0, version_1.versionNumber)(),
    };
    for (const [key, value] of Object.entries(ret)) {
        if (!value) {
            throw new toolkit_lib_1.ToolkitError(`Missing init version from ${initVersionFile}: ${key}`);
        }
    }
    return ret;
}
/**
 * Return the currently recommended flags for `aws-cdk-lib`.
 *
 * These have been built into the CLI at build time.
 */
async function currentlyRecommendedAwsCdkLibFlags() {
    const recommendedFlagsFile = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates', '.recommended-feature-flags.json');
    return JSON.parse(await fs.readFile(recommendedFlagsFile, { encoding: 'utf-8' }));
}
//# sourceMappingURL=data:application/json;base64,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