"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.doctor = doctor;
const process = require("process");
const cxapi = require("@aws-cdk/cx-api");
const chalk = require("chalk");
const display_version_1 = require("../cli/display-version");
const version_1 = require("../cli/version");
async function doctor({ ioHelper }) {
    let exitStatus = 0;
    for (const verification of verifications) {
        if (!await verification(ioHelper)) {
            exitStatus = -1;
        }
    }
    await (0, display_version_1.displayVersionMessage)(ioHelper);
    return exitStatus;
}
const verifications = [
    displayVersionInformation,
    displayAwsEnvironmentVariables,
    displayCdkEnvironmentVariables,
];
// ### Verifications ###
async function displayVersionInformation(ioHelper) {
    await ioHelper.defaults.info(`ℹ️ CDK Version: ${chalk.green((0, version_1.versionWithBuild)())}`);
    return true;
}
async function displayAwsEnvironmentVariables(ioHelper) {
    const keys = Object.keys(process.env).filter(s => s.startsWith('AWS_'));
    if (keys.length === 0) {
        await ioHelper.defaults.info('ℹ️ No AWS environment variables');
        return true;
    }
    await ioHelper.defaults.info('ℹ️ AWS environment variables:');
    for (const key of keys) {
        await ioHelper.defaults.info(`  - ${chalk.blue(key)} = ${chalk.green(anonymizeAwsVariable(key, process.env[key]))}`);
    }
    return true;
}
async function displayCdkEnvironmentVariables(ioHelper) {
    const keys = Object.keys(process.env).filter(s => s.startsWith('CDK_'));
    if (keys.length === 0) {
        await ioHelper.defaults.info('ℹ️ No CDK environment variables');
        return true;
    }
    await ioHelper.defaults.info('ℹ️ CDK environment variables:');
    let healthy = true;
    for (const key of keys.sort()) {
        if (key === cxapi.CONTEXT_ENV || key === cxapi.CONTEXT_OVERFLOW_LOCATION_ENV || key === cxapi.OUTDIR_ENV) {
            await ioHelper.defaults.info(`  - ${chalk.red(key)} = ${chalk.green(process.env[key])} (⚠️ reserved for use by the CDK toolkit)`);
            healthy = false;
        }
        else {
            await ioHelper.defaults.info(`  - ${chalk.blue(key)} = ${chalk.green(process.env[key])}`);
        }
    }
    return healthy;
}
function anonymizeAwsVariable(name, value) {
    if (name === 'AWS_ACCESS_KEY_ID') {
        return value.slice(0, 4) + '<redacted>';
    } // Show ASIA/AKIA key type, but hide identifier
    if (name === 'AWS_SECRET_ACCESS_KEY' || name === 'AWS_SESSION_TOKEN' || name === 'AWS_SECURITY_TOKEN') {
        return '<redacted>';
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,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