"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeBuiltinHooks = invokeBuiltinHooks;
const path = require("path");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const os_1 = require("./os");
const util_1 = require("../../util");
/**
 * Invoke hooks for the given init template
 *
 * Sometimes templates need more complex logic than just replacing tokens. A 'hook' can be
 * used to do additional processing other than copying files.
 *
 * Hooks used to be defined externally to the CLI, by running arbitrarily
 * substituted shell scripts in the target directory.
 *
 * In practice, they're all TypeScript files and all the same, and the dynamism
 * that the original solution allowed wasn't used at all. Worse, since the CLI
 * is now bundled the hooks can't even reuse code from the CLI libraries at all
 * anymore, so all shared code would have to be copy/pasted.
 *
 * Bundle hooks as built-ins into the CLI, so they get bundled and can take advantage
 * of all shared code.
 */
async function invokeBuiltinHooks(ioHelper, target, context) {
    switch (target.language) {
        case 'csharp':
            if (['app', 'sample-app'].includes(target.templateName)) {
                return dotnetAddProject(ioHelper, target.targetDirectory, context);
            }
            break;
        case 'fsharp':
            if (['app', 'sample-app'].includes(target.templateName)) {
                return dotnetAddProject(ioHelper, target.targetDirectory, context, 'fsproj');
            }
            break;
        case 'python':
            // We can't call this file 'requirements.template.txt' because Dependabot needs to be able to find it.
            // Therefore, keep the in-repo name but still substitute placeholders.
            await context.substitutePlaceholdersIn('requirements.txt');
            break;
        case 'java':
            // We can't call this file 'pom.template.xml'... for the same reason as Python above.
            await context.substitutePlaceholdersIn('pom.xml');
            break;
        case 'javascript':
        case 'typescript':
            // See above, but for 'package.json'.
            await context.substitutePlaceholdersIn('package.json', 'README.md');
    }
}
async function dotnetAddProject(ioHelper, targetDirectory, context, ext = 'csproj') {
    const pname = context.placeholder('name.PascalCased');
    const slnPath = path.join(targetDirectory, 'src', `${pname}.sln`);
    const csprojPath = path.join(targetDirectory, 'src', pname, `${pname}.${ext}`);
    // We retry this command a couple of times. It usually never fails, except on CI where
    // we sometimes see:
    //
    //   System.IO.IOException: The system cannot open the device or file specified. : 'NuGet-Migrations'
    //
    // This error can be caused by lack of permissions on a temporary directory,
    // but in our case it's intermittent so my guess is it is caused by multiple
    // invocations of the .NET CLI running in parallel, and trampling on each
    // other creating a Mutex. There is no fix, and it is annoyingly breaking our
    // CI regularly. Retry a couple of times to increase reliability.
    //
    // - https://github.com/dotnet/sdk/issues/43750
    // - https://github.com/dotnet/runtime/issues/80619
    // - https://github.com/dotnet/runtime/issues/91987
    const MAX_ATTEMPTS = 3;
    for (let attempt = 1;; attempt++) {
        try {
            await (0, os_1.shell)(ioHelper, ['dotnet', 'sln', slnPath, 'add', csprojPath]);
            return;
        }
        catch (e) {
            if (attempt === MAX_ATTEMPTS) {
                throw new toolkit_lib_1.ToolkitError(`Could not add project ${pname}.${ext} to solution ${pname}.sln. ${(0, util_1.formatErrorMessage)(e)}`);
            }
            // Sleep for a bit then try again
            await new Promise(resolve => setTimeout(resolve, 1000));
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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