"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitTemplate = void 0;
exports.cliInit = cliInit;
exports.expandPlaceholders = expandPlaceholders;
exports.availableInitTemplates = availableInitTemplates;
exports.availableInitLanguages = availableInitLanguages;
exports.printAvailableTemplates = printAvailableTemplates;
exports.currentlyRecommendedAwsCdkLibFlags = currentlyRecommendedAwsCdkLibFlags;
const childProcess = require("child_process");
const path = require("path");
const toolkit_lib_1 = require("@aws-cdk/toolkit-lib");
const chalk = require("chalk");
const fs = require("fs-extra");
const init_hooks_1 = require("./init-hooks");
const root_dir_1 = require("../../cli/root-dir");
const version_1 = require("../../cli/version");
const util_1 = require("../../util");
const language_1 = require("../language");
const package_manager_1 = require("./package-manager");
/* eslint-disable @typescript-eslint/no-var-requires */ // Packages don't have @types module
// eslint-disable-next-line @typescript-eslint/no-require-imports
const camelCase = require('camelcase');
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const SUPPORTED_LANGUAGE_NAMES = language_1.SUPPORTED_LANGUAGES.map((l) => l.name);
/**
 * Initialize a CDK package in the current directory
 */
async function cliInit(options) {
    await ensureValidCliInitOptions(options, options.ioHelper);
    const ioHelper = options.ioHelper;
    const canUseNetwork = options.canUseNetwork ?? true;
    const generateOnly = options.generateOnly ?? false;
    const workDir = options.workDir ?? process.cwd();
    // Show available templates only if no fromPath, type, or language provided
    if (!options.fromPath && !options.type && !options.language) {
        await printAvailableTemplates(ioHelper);
        return;
    }
    // Step 1: Load template
    let template;
    if (options.fromPath) {
        template = await loadLocalTemplate(options.fromPath, options.templatePath);
    }
    else {
        template = await loadBuiltinTemplate(ioHelper, options.type, options.language);
    }
    // Step 2: Resolve language
    const language = await resolveLanguage(ioHelper, template, options.language, options.type);
    // Step 3: Initialize project following standard process
    await initializeProject(ioHelper, template, language, canUseNetwork, generateOnly, workDir, options.stackName, options.migrate, options.libVersion, options.packageManager);
}
/**
 * Validate CLI init options and handle invalid or incompatible option combinations
 */
async function ensureValidCliInitOptions(options, ioHelper) {
    if (options.packageManager && !['javascript', 'typescript'].includes(options.language ?? '')) {
        await ioHelper.defaults.warn(`--package-manager option is only applicable for JavaScript and TypeScript projects. Ignoring the provided value: ${options.packageManager}`);
    }
}
/**
 * Load a local custom template from file system path
 * @param fromPath - Path to the local template directory or multi-template repository
 * @param templatePath - Optional path to a specific template within a multi-template repository
 * @returns Promise resolving to the loaded InitTemplate
 */
async function loadLocalTemplate(fromPath, templatePath) {
    try {
        let actualTemplatePath = fromPath;
        // If templatePath is provided, it's a multi-template repository
        if (templatePath) {
            actualTemplatePath = path.join(fromPath, templatePath);
            if (!await fs.pathExists(actualTemplatePath)) {
                throw new toolkit_lib_1.ToolkitError(`Template path does not exist: ${actualTemplatePath}`);
            }
        }
        const template = await InitTemplate.fromPath(actualTemplatePath);
        if (template.languages.length === 0) {
            // Check if this might be a multi-template repository
            if (!templatePath) {
                const availableTemplates = await findPotentialTemplates(fromPath);
                if (availableTemplates.length > 0) {
                    throw new toolkit_lib_1.ToolkitError('Use --template-path to specify which template to use.');
                }
            }
            throw new toolkit_lib_1.ToolkitError('Custom template must contain at least one language directory');
        }
        return template;
    }
    catch (error) {
        const displayPath = templatePath ? `${fromPath}/${templatePath}` : fromPath;
        throw new toolkit_lib_1.ToolkitError(`Failed to load template from path: ${displayPath}. ${error.message}`);
    }
}
/**
 * Load a built-in template by name
 */
async function loadBuiltinTemplate(ioHelper, type, language) {
    const templateType = type || 'default'; // "default" is the default type (and maps to "app")
    const template = (await availableInitTemplates()).find((t) => t.hasName(templateType));
    if (!template) {
        await printAvailableTemplates(ioHelper, language);
        throw new toolkit_lib_1.ToolkitError(`Unknown init template: ${templateType}`);
    }
    return template;
}
/**
 * Resolve the programming language for the template
 * @param ioHelper - IO helper for user interaction
 * @param template - The template to resolve language for
 * @param requestedLanguage - User-requested language (optional)
 * @param type - The template type name for messages
 * @default undefined
 * @returns Promise resolving to the selected language
 */
async function resolveLanguage(ioHelper, template, requestedLanguage, type) {
    return (async () => {
        if (requestedLanguage) {
            return requestedLanguage;
        }
        if (template.languages.length === 1) {
            const templateLanguage = template.languages[0];
            // Only show auto-detection message for built-in templates
            if (template.templateType !== TemplateType.CUSTOM) {
                await ioHelper.defaults.warn(`No --language was provided, but '${type || template.name}' supports only '${templateLanguage}', so defaulting to --language=${templateLanguage}`);
            }
            return templateLanguage;
        }
        await ioHelper.defaults.info(`Available languages for ${chalk.green(type || template.name)}: ${template.languages.map((l) => chalk.blue(l)).join(', ')}`);
        throw new toolkit_lib_1.ToolkitError('No language was selected');
    })();
}
/**
 * Find potential template directories in a multi-template repository
 * @param repositoryPath - Path to the repository root
 * @returns Promise resolving to array of potential template directory names
 */
async function findPotentialTemplates(repositoryPath) {
    const entries = await fs.readdir(repositoryPath, { withFileTypes: true });
    const templateValidationPromises = entries
        .filter(entry => entry.isDirectory() && !entry.name.startsWith('.'))
        .map(async (entry) => {
        try {
            const templatePath = path.join(repositoryPath, entry.name);
            const { languages } = await getLanguageDirectories(templatePath);
            return languages.length > 0 ? entry.name : null;
        }
        catch (error) {
            // If we can't read a specific template directory, skip it but don't fail the entire operation
            return null;
        }
    });
    /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Limited to directory entries
    const validationResults = await Promise.all(templateValidationPromises);
    return validationResults.filter((templateName) => templateName !== null);
}
/**
 * Get valid CDK language directories from a template path
 * @param templatePath - Path to the template directory
 * @returns Promise resolving to array of supported language names
 */
/**
 * Get valid CDK language directories from a template path
 * @param templatePath - Path to the template directory
 * @returns Promise resolving to array of supported language names and directory entries
 * @throws ToolkitError if directory cannot be read or validated
 */
async function getLanguageDirectories(templatePath) {
    try {
        const entries = await fs.readdir(templatePath, { withFileTypes: true });
        const languageValidationPromises = entries
            .filter(directoryEntry => directoryEntry.isDirectory() && SUPPORTED_LANGUAGE_NAMES.includes(directoryEntry.name))
            .map(async (directoryEntry) => {
            const languageDirectoryPath = path.join(templatePath, directoryEntry.name);
            try {
                const hasValidLanguageFiles = await hasLanguageFiles(languageDirectoryPath, (0, language_1.getLanguageExtensions)(directoryEntry.name));
                return hasValidLanguageFiles ? directoryEntry.name : null;
            }
            catch (error) {
                throw new toolkit_lib_1.ToolkitError(`Cannot read language directory '${directoryEntry.name}': ${error.message}`);
            }
        });
        /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Limited to supported CDK languages (7 max)
        const validationResults = await Promise.all(languageValidationPromises);
        return {
            languages: validationResults.filter((languageName) => languageName !== null),
            entries,
        };
    }
    catch (error) {
        throw new toolkit_lib_1.ToolkitError(`Cannot read template directory '${templatePath}': ${error.message}`);
    }
}
/**
 * Iteratively check if a directory contains files with the specified extensions
 * @param directoryPath - Path to search for language files
 * @param extensions - Array of file extensions to look for
 * @returns Promise resolving to true if language files are found
 */
async function hasLanguageFiles(directoryPath, extensions) {
    const dirsToCheck = [directoryPath];
    while (dirsToCheck.length > 0) {
        const currentDir = dirsToCheck.pop();
        try {
            const entries = await fs.readdir(currentDir, { withFileTypes: true });
            for (const entry of entries) {
                if (entry.isFile() && extensions.some(ext => entry.name.endsWith(ext))) {
                    return true;
                }
                else if (entry.isDirectory()) {
                    dirsToCheck.push(path.join(currentDir, entry.name));
                }
            }
        }
        catch (error) {
            throw error;
        }
    }
    return false;
}
/**
 * Returns the name of the Python executable for this OS
 * @returns The Python executable name for the current platform
 */
function pythonExecutable() {
    let python = 'python3';
    if (process.platform === 'win32') {
        python = 'python';
    }
    return python;
}
const INFO_DOT_JSON = 'info.json';
var TemplateType;
(function (TemplateType) {
    TemplateType["BUILT_IN"] = "builtin";
    TemplateType["CUSTOM"] = "custom";
})(TemplateType || (TemplateType = {}));
class InitTemplate {
    static async fromName(templatesDir, name) {
        const basePath = path.join(templatesDir, name);
        const languages = await listDirectory(basePath);
        const initInfo = await fs.readJson(path.join(basePath, INFO_DOT_JSON));
        return new InitTemplate(basePath, name, languages, initInfo, TemplateType.BUILT_IN);
    }
    static async fromPath(templatePath) {
        const basePath = path.resolve(templatePath);
        if (!await fs.pathExists(basePath)) {
            throw new toolkit_lib_1.ToolkitError(`Template path does not exist: ${basePath}`);
        }
        let templateSourcePath = basePath;
        let { languages, entries } = await getLanguageDirectories(basePath);
        if (languages.length === 0) {
            const languageDirs = entries.filter(entry => entry.isDirectory() &&
                SUPPORTED_LANGUAGE_NAMES.includes(entry.name));
            if (languageDirs.length === 1) {
                // Validate that the language directory contains appropriate files
                const langDir = languageDirs[0].name;
                templateSourcePath = path.join(basePath, langDir);
                const hasValidFiles = await hasLanguageFiles(templateSourcePath, (0, language_1.getLanguageExtensions)(langDir));
                if (!hasValidFiles) {
                    // If we found a language directory but it doesn't contain valid files, we should inform the user
                    throw new toolkit_lib_1.ToolkitError(`Found '${langDir}' directory but it doesn't contain the expected language files. Ensure the template contains ${langDir} source files.`);
                }
            }
        }
        const name = path.basename(basePath);
        return new InitTemplate(templateSourcePath, name, languages, null, TemplateType.CUSTOM);
    }
    constructor(basePath, name, languages, initInfo, templateType) {
        this.basePath = basePath;
        this.name = name;
        this.languages = languages;
        this.aliases = new Set();
        this.templateType = templateType;
        // Only built-in templates have descriptions and aliases from info.json
        if (templateType === TemplateType.BUILT_IN && initInfo) {
            this.description = initInfo.description;
            for (const alias of initInfo.aliases || []) {
                this.aliases.add(alias);
            }
        }
    }
    /**
     * @param name - the name that is being checked
     * @returns ``true`` if ``name`` is the name of this template or an alias of it.
     */
    hasName(name) {
        return name === this.name || this.aliases.has(name);
    }
    /**
     * Creates a new instance of this ``InitTemplate`` for a given language to a specified folder.
     *
     * @param language - the language to instantiate this template with
     * @param targetDirectory - the directory where the template is to be instantiated into
     * @param stackName - the name of the stack to create
     * @default undefined
     * @param libVersion - the version of the CDK library to use
     * @default undefined
     */
    async install(ioHelper, language, targetDirectory, stackName, libVersion, packageManager) {
        if (this.languages.indexOf(language) === -1) {
            await ioHelper.defaults.error(`The ${chalk.blue(language)} language is not supported for ${chalk.green(this.name)} ` +
                `(it supports: ${this.languages.map((l) => chalk.blue(l)).join(', ')})`);
            throw new toolkit_lib_1.ToolkitError(`Unsupported language: ${language}`);
        }
        const projectInfo = {
            name: decamelize(path.basename(path.resolve(targetDirectory))),
            stackName,
            versions: await loadInitVersions(),
        };
        if (libVersion) {
            projectInfo.versions['aws-cdk-lib'] = libVersion;
        }
        let sourceDirectory = path.join(this.basePath, language);
        // For auto-detected single language templates, use basePath directly
        if (this.templateType === TemplateType.CUSTOM && this.languages.length === 1 &&
            path.basename(this.basePath) === language) {
            sourceDirectory = this.basePath;
        }
        if (this.templateType === TemplateType.CUSTOM) {
            // For custom templates, copy files without processing placeholders
            await this.installFilesWithoutProcessing(sourceDirectory, targetDirectory);
        }
        else {
            // For built-in templates, process placeholders as usual
            await this.installFiles(sourceDirectory, targetDirectory, language, projectInfo, packageManager);
            await this.applyFutureFlags(targetDirectory);
            await (0, init_hooks_1.invokeBuiltinHooks)(ioHelper, { targetDirectory, language, templateName: this.name }, {
                substitutePlaceholdersIn: async (...fileNames) => {
                    const fileProcessingPromises = fileNames.map(async (fileName) => {
                        const fullPath = path.join(targetDirectory, fileName);
                        const template = await fs.readFile(fullPath, { encoding: 'utf-8' });
                        await fs.writeFile(fullPath, expandPlaceholders(template, language, projectInfo, packageManager));
                    });
                    /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Processing a small, known set of template files
                    await Promise.all(fileProcessingPromises);
                },
                placeholder: (ph) => expandPlaceholders(`%${ph}%`, language, projectInfo, packageManager),
            });
        }
    }
    async installFiles(sourceDirectory, targetDirectory, language, project, packageManager) {
        for (const file of await fs.readdir(sourceDirectory)) {
            const fromFile = path.join(sourceDirectory, file);
            const toFile = path.join(targetDirectory, expandPlaceholders(file, language, project, packageManager));
            if ((await fs.stat(fromFile)).isDirectory()) {
                await fs.mkdir(toFile);
                await this.installFiles(fromFile, toFile, language, project, packageManager);
                continue;
            }
            else if (file.match(/^.*\.template\.[^.]+$/)) {
                await this.installProcessed(fromFile, toFile.replace(/\.template(\.[^.]+)$/, '$1'), language, project, packageManager);
                continue;
            }
            else if (file.match(/^.*\.hook\.(d.)?[^.]+$/)) {
                // Ignore
                continue;
            }
            else {
                await fs.copy(fromFile, toFile);
            }
        }
    }
    async installProcessed(templatePath, toFile, language, project, packageManager) {
        const template = await fs.readFile(templatePath, { encoding: 'utf-8' });
        await fs.writeFile(toFile, expandPlaceholders(template, language, project, packageManager));
    }
    /**
     * Copy template files without processing placeholders (for custom templates)
     */
    async installFilesWithoutProcessing(sourceDirectory, targetDirectory) {
        await fs.copy(sourceDirectory, targetDirectory, {
            filter: (src) => {
                const filename = path.basename(src);
                return !filename.match(/^.*\.hook\.(d.)?[^.]+$/);
            },
        });
    }
    /**
     * Adds context variables to `cdk.json` in the generated project directory to
     * enable future behavior for new projects.
     */
    async applyFutureFlags(projectDir) {
        const cdkJson = path.join(projectDir, 'cdk.json');
        if (!(await fs.pathExists(cdkJson))) {
            return;
        }
        const config = await fs.readJson(cdkJson);
        config.context = {
            ...config.context,
            ...await currentlyRecommendedAwsCdkLibFlags(),
        };
        await fs.writeJson(cdkJson, config, { spaces: 2 });
    }
    async addMigrateContext(projectDir) {
        const cdkJson = path.join(projectDir, 'cdk.json');
        if (!(await fs.pathExists(cdkJson))) {
            return;
        }
        const config = await fs.readJson(cdkJson);
        config.context = {
            ...config.context,
            'cdk-migrate': true,
        };
        await fs.writeJson(cdkJson, config, { spaces: 2 });
    }
}
exports.InitTemplate = InitTemplate;
function expandPlaceholders(template, language, project, packageManager) {
    const cdkVersion = project.versions['aws-cdk-lib'];
    const cdkCliVersion = project.versions['aws-cdk'];
    let constructsVersion = project.versions.constructs;
    switch (language) {
        case 'java':
        case 'csharp':
        case 'fsharp':
            constructsVersion = (0, util_1.rangeFromSemver)(constructsVersion, 'bracket');
            break;
        case 'python':
            constructsVersion = (0, util_1.rangeFromSemver)(constructsVersion, 'pep');
            break;
    }
    return template
        .replace(/%name%/g, project.name)
        .replace(/%stackname%/, project.stackName ?? '%name.PascalCased%Stack')
        .replace(/%PascalNameSpace%/, project.stackName ? camelCase(project.stackName + 'Stack', { pascalCase: true }) : '%name.PascalCased%')
        .replace(/%PascalStackProps%/, project.stackName ? camelCase(project.stackName, { pascalCase: true }) + 'StackProps' : 'StackProps')
        .replace(/%name\.camelCased%/g, camelCase(project.name))
        .replace(/%name\.PascalCased%/g, camelCase(project.name, { pascalCase: true }))
        .replace(/%cdk-version%/g, cdkVersion)
        .replace(/%cdk-cli-version%/g, cdkCliVersion)
        .replace(/%constructs-version%/g, constructsVersion)
        .replace(/%cdk-home%/g, (0, util_1.cdkHomeDir)())
        .replace(/%name\.PythonModule%/g, project.name.replace(/-/g, '_'))
        .replace(/%python-executable%/g, pythonExecutable())
        .replace(/%name\.StackName%/g, project.name.replace(/[^A-Za-z0-9-]/g, '-'))
        .replace(/%pm-cmd%/g, (0, package_manager_1.getPmCmdPrefix)(packageManager ?? 'npm'));
}
async function availableInitTemplates() {
    try {
        const templatesDir = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates');
        const templateNames = await listDirectory(templatesDir);
        const templatePromises = templateNames.map(templateName => InitTemplate.fromName(templatesDir, templateName));
        /* eslint-disable-next-line @cdklabs/promiseall-no-unbounded-parallelism */ // Built-in templates are limited in number
        return await Promise.all(templatePromises);
    }
    catch (error) {
        // Return empty array if templates directory doesn't exist or can't be read
        // This allows the CLI to gracefully handle missing built-in templates
        if (error.code === 'ENOENT' || error.code === 'EACCES') {
            return [];
        }
        throw error;
    }
}
async function availableInitLanguages() {
    const templates = await availableInitTemplates();
    const result = new Set();
    for (const template of templates) {
        for (const language of template.languages) {
            const alias = (0, language_1.getLanguageAlias)(language);
            result.add(language);
            alias && result.add(alias);
        }
    }
    return [...result];
}
/**
 * @param dirPath - is the directory to be listed.
 * @returns the list of file or directory names contained in ``dirPath``, excluding any dot-file, and sorted.
 */
async function listDirectory(dirPath) {
    return ((await fs.readdir(dirPath))
        .filter((p) => !p.startsWith('.'))
        .filter((p) => !(p === 'LICENSE'))
        // if, for some reason, the temp folder for the hook doesn't get deleted we don't want to display it in this list
        .filter((p) => !(p === INFO_DOT_JSON))
        .sort());
}
/**
 * Print available templates to the user
 * @param ioHelper - IO helper for user interaction
 * @param language - Programming language filter
 * @default undefined
 */
async function printAvailableTemplates(ioHelper, language) {
    await ioHelper.defaults.info('Available templates:');
    for (const template of await availableInitTemplates()) {
        if (language && template.languages.indexOf(language) === -1) {
            continue;
        }
        await ioHelper.defaults.info(`* ${chalk.green(template.name)}: ${template.description}`);
        const languageArg = language
            ? chalk.bold(language)
            : template.languages.length > 1
                ? `[${template.languages.map((t) => chalk.bold(t)).join('|')}]`
                : chalk.bold(template.languages[0]);
        await ioHelper.defaults.info(`   └─ ${chalk.blue(`cdk init ${chalk.bold(template.name)} --language=${languageArg}`)}`);
    }
}
async function initializeProject(ioHelper, template, language, canUseNetwork, generateOnly, workDir, stackName, migrate, cdkVersion, packageManager) {
    // Step 1: Ensure target directory is empty
    await assertIsEmptyDirectory(workDir);
    // Step 2: Copy template files
    await ioHelper.defaults.info(`Applying project template ${chalk.green(template.name)} for ${chalk.blue(language)}`);
    await template.install(ioHelper, language, workDir, stackName, cdkVersion, packageManager);
    if (migrate) {
        await template.addMigrateContext(workDir);
    }
    if (await fs.pathExists(`${workDir}/README.md`)) {
        const readme = await fs.readFile(`${workDir}/README.md`, { encoding: 'utf-8' });
        await ioHelper.defaults.info(chalk.green(readme));
    }
    if (!generateOnly) {
        // Step 3: Initialize Git repository and create initial commit
        await initializeGitRepository(ioHelper, workDir);
        // Step 4: Post-install steps
        await postInstall(ioHelper, language, canUseNetwork, workDir, packageManager);
    }
    await ioHelper.defaults.info('✅ All done!');
}
/**
 * Validate that a directory exists and is empty (ignoring hidden files)
 * @param workDir - Directory path to validate
 * @throws ToolkitError if directory doesn't exist or is not empty
 */
async function assertIsEmptyDirectory(workDir) {
    try {
        const stats = await fs.stat(workDir);
        if (!stats.isDirectory()) {
            throw new toolkit_lib_1.ToolkitError(`Path exists but is not a directory: ${workDir}`);
        }
        const files = await fs.readdir(workDir);
        const visibleFiles = files.filter(f => !f.startsWith('.'));
        if (visibleFiles.length > 0) {
            throw new toolkit_lib_1.ToolkitError('`cdk init` cannot be run in a non-empty directory!\n' +
                `Found ${visibleFiles.length} visible files in ${workDir}:\n` +
                visibleFiles.map(f => `  - ${f}`).join('\n'));
        }
    }
    catch (e) {
        if (e.code === 'ENOENT') {
            throw new toolkit_lib_1.ToolkitError(`Directory does not exist: ${workDir}\n` +
                'Please create the directory first using: mkdir -p ' + workDir);
        }
        throw new toolkit_lib_1.ToolkitError(`Failed to validate directory ${workDir}: ${e.message}`);
    }
}
async function initializeGitRepository(ioHelper, workDir) {
    if (await isInGitRepository(workDir)) {
        return;
    }
    await ioHelper.defaults.info('Initializing a new git repository...');
    try {
        await execute(ioHelper, 'git', ['init'], { cwd: workDir });
        await execute(ioHelper, 'git', ['add', '.'], { cwd: workDir });
        await execute(ioHelper, 'git', ['commit', '--message="Initial commit"', '--no-gpg-sign'], { cwd: workDir });
    }
    catch {
        await ioHelper.defaults.warn('Unable to initialize git repository for your project.');
    }
}
async function postInstall(ioHelper, language, canUseNetwork, workDir, packageManager) {
    switch (language) {
        case 'javascript':
            return postInstallJavascript(ioHelper, canUseNetwork, workDir, packageManager);
        case 'typescript':
            return postInstallTypescript(ioHelper, canUseNetwork, workDir, packageManager);
        case 'java':
            return postInstallJava(ioHelper, canUseNetwork, workDir);
        case 'python':
            return postInstallPython(ioHelper, workDir);
        case 'go':
            return postInstallGo(ioHelper, canUseNetwork, workDir);
        case 'csharp':
            return postInstallCSharp(ioHelper, canUseNetwork, workDir);
        case 'fsharp':
            return postInstallFSharp(ioHelper, canUseNetwork, workDir);
    }
}
async function postInstallJavascript(ioHelper, canUseNetwork, cwd, packageManager) {
    return postInstallTypescript(ioHelper, canUseNetwork, cwd, packageManager);
}
async function postInstallTypescript(ioHelper, canUseNetwork, cwd, packageManager) {
    const command = packageManager ?? 'npm';
    if (!canUseNetwork) {
        await ioHelper.defaults.warn(`Please run '${command} install'!`);
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green(`${command} install`)}...`);
    try {
        await execute(ioHelper, command, ['install'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn(`${command} install failed: ` + (0, util_1.formatErrorMessage)(e));
    }
}
async function postInstallJava(ioHelper, canUseNetwork, cwd) {
    // Check if this is a Gradle or Maven project
    const hasGradleBuild = await fs.pathExists(path.join(cwd, 'build.gradle'));
    const hasMavenPom = await fs.pathExists(path.join(cwd, 'pom.xml'));
    if (hasGradleBuild) {
        // Gradle project
        const gradleWarning = "Please run './gradlew build'!";
        if (!canUseNetwork) {
            await ioHelper.defaults.warn(gradleWarning);
            return;
        }
        await ioHelper.defaults.info("Executing './gradlew build'");
        try {
            await execute(ioHelper, './gradlew', ['build'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to build Gradle project');
            await ioHelper.defaults.warn(gradleWarning);
        }
    }
    else if (hasMavenPom) {
        // Maven project
        const mvnPackageWarning = "Please run 'mvn package'!";
        if (!canUseNetwork) {
            await ioHelper.defaults.warn(mvnPackageWarning);
            return;
        }
        await ioHelper.defaults.info("Executing 'mvn package'");
        try {
            await execute(ioHelper, 'mvn', ['package'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to package compiled code as JAR');
            await ioHelper.defaults.warn(mvnPackageWarning);
        }
    }
    else {
        // No recognized build file
        await ioHelper.defaults.warn('No build.gradle or pom.xml found. Please set up your build system manually.');
    }
}
async function postInstallPython(ioHelper, cwd) {
    const python = pythonExecutable();
    // Check if requirements.txt exists
    const hasRequirements = await fs.pathExists(path.join(cwd, 'requirements.txt'));
    if (hasRequirements) {
        await ioHelper.defaults.info(`Executing ${chalk.green('Creating virtualenv...')}`);
        try {
            await execute(ioHelper, python, ['-m', 'venv', '.venv'], { cwd });
            await ioHelper.defaults.info(`Executing ${chalk.green('Installing dependencies...')}`);
            // Install dependencies in the virtual environment
            const pipPath = process.platform === 'win32' ? '.venv\\Scripts\\pip' : '.venv/bin/pip';
            await execute(ioHelper, pipPath, ['install', '-r', 'requirements.txt'], { cwd });
        }
        catch {
            await ioHelper.defaults.warn('Unable to create virtualenv or install dependencies automatically');
            await ioHelper.defaults.warn(`Please run '${python} -m venv .venv && .venv/bin/pip install -r requirements.txt'!`);
        }
    }
    else {
        await ioHelper.defaults.warn('No requirements.txt found. Please set up your Python environment manually.');
    }
}
async function postInstallGo(ioHelper, canUseNetwork, cwd) {
    if (!canUseNetwork) {
        await ioHelper.defaults.warn('Please run \'go mod tidy\'!');
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green('go mod tidy')}...`);
    try {
        await execute(ioHelper, 'go', ['mod', 'tidy'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn('\'go mod tidy\' failed: ' + (0, util_1.formatErrorMessage)(e));
    }
}
async function postInstallCSharp(ioHelper, canUseNetwork, cwd) {
    const dotnetWarning = "Please run 'dotnet restore && dotnet build'!";
    if (!canUseNetwork) {
        await ioHelper.defaults.warn(dotnetWarning);
        return;
    }
    await ioHelper.defaults.info(`Executing ${chalk.green('dotnet restore')}...`);
    try {
        await execute(ioHelper, 'dotnet', ['restore'], { cwd });
        await ioHelper.defaults.info(`Executing ${chalk.green('dotnet build')}...`);
        await execute(ioHelper, 'dotnet', ['build'], { cwd });
    }
    catch (e) {
        await ioHelper.defaults.warn('Unable to restore/build .NET project: ' + (0, util_1.formatErrorMessage)(e));
        await ioHelper.defaults.warn(dotnetWarning);
    }
}
async function postInstallFSharp(ioHelper, canUseNetwork, cwd) {
    // F# uses the same build system as C#
    return postInstallCSharp(ioHelper, canUseNetwork, cwd);
}
/**
 * @param dir - a directory to be checked
 * @returns true if ``dir`` is within a git repository.
 */
async function isInGitRepository(dir) {
    while (true) {
        if (await fs.pathExists(path.join(dir, '.git'))) {
            return true;
        }
        if (isRoot(dir)) {
            return false;
        }
        dir = path.dirname(dir);
    }
}
/**
 * @param dir - a directory to be checked.
 * @returns true if ``dir`` is the root of a filesystem.
 */
function isRoot(dir) {
    return path.dirname(dir) === dir;
}
/**
 * Executes `command`. STDERR is emitted in real-time.
 *
 * If command exits with non-zero exit code, an exception is thrown and includes
 * the contents of STDOUT.
 *
 * @returns STDOUT (if successful).
 */
async function execute(ioHelper, cmd, args, { cwd }) {
    const child = childProcess.spawn(cmd, args, {
        cwd,
        shell: true,
        stdio: ['ignore', 'pipe', 'inherit'],
    });
    let stdout = '';
    child.stdout.on('data', (chunk) => (stdout += chunk.toString()));
    return new Promise((ok, fail) => {
        child.once('error', (err) => fail(err));
        child.once('exit', (status) => {
            if (status === 0) {
                return ok(stdout);
            }
            else {
                return fail(new toolkit_lib_1.ToolkitError(`${cmd} exited with status ${status}`));
            }
        });
    }).catch(async (err) => {
        await ioHelper.defaults.error(stdout);
        throw err;
    });
}
/**
 * Return the 'aws-cdk-lib' version we will init
 *
 * This has been built into the CLI at build time.
 */
async function loadInitVersions() {
    const initVersionFile = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates', '.init-version.json');
    const contents = JSON.parse(await fs.readFile(initVersionFile, { encoding: 'utf-8' }));
    const ret = {
        'aws-cdk-lib': contents['aws-cdk-lib'],
        'constructs': contents.constructs,
        'aws-cdk': (0, version_1.versionNumber)(),
    };
    for (const [key, value] of Object.entries(ret)) {
        if (!value) {
            throw new toolkit_lib_1.ToolkitError(`Missing init version from ${initVersionFile}: ${key}`);
        }
    }
    return ret;
}
/**
 * Return the currently recommended flags for `aws-cdk-lib`.
 *
 * These have been built into the CLI at build time.
 */
async function currentlyRecommendedAwsCdkLibFlags() {
    const recommendedFlagsFile = path.join((0, root_dir_1.cliRootDir)(), 'lib', 'init-templates', '.recommended-feature-flags.json');
    return JSON.parse(await fs.readFile(recommendedFlagsFile, { encoding: 'utf-8' }));
}
//# sourceMappingURL=data:application/json;base64,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